/*
vim:ts=3:sw=3:
 * xmlstuff.c -- xml headers and parsing functions.
 * hotwayd is a POP3-HTTPMail gateway. It will allow you to
 * check your HOTMAIL account from your unix box.
 *
 * Created by: Espeleta Tomas <espeleta@libero.it>, 12-Apr-2001
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 */

#include <libxml/xmlversion.h>
#include <libxml/xmlmemory.h>
#include <libxml/parser.h>
#include <string.h>
#include <stdarg.h> /* for error fns */

#include "libghttp-1.0.9-mod/ghttp.h"
#include "httpmail.h"
#include "hotwayd.h"
#include "commands.h"
#include "xmlstuff.h"

const char *dir_names_request = "\
<?xml version=\"1.0\"?>\r\n\
<D:propfind xmlns:D=\"DAV:\" xmlns:h=\"http://schemas.microsoft.com/hotmail/\" xmlns:hm=\"urn:schemas:httpmail:\">\r\n\
	<D:prop>\r\n\
		<h:adbar/>\r\n\
		<hm:contacts/>\r\n\
		<hm:inbox/>\r\n\
		<hm:outbox/>\r\n\
		<hm:sendmsg/>\r\n\
		<hm:sentitems/>\r\n\
		<hm:deleteditems/>\r\n\
		<hm:drafts/>\r\n\
		<hm:msgfolderroot/>\r\n\
		<h:sig/>\r\n\
	</D:prop>\r\n\
</D:propfind>";

const char *dir_props_request = "\
<?xml version=\"1.0\"?>\r\n\
<D:propfind xmlns:D=\"DAV:\" xmlns:hm=\"urn:schemas:httpmail:\">\r\n\
	<D:prop>\r\n\
		<D:isfolder/>\r\n\
		<D:displayname/>\r\n\
		<hm:special/>\r\n\
		<D:hassubs/>\r\n\
		<D:nosubs/>\r\n\
		<hm:unreadcount/>\r\n\
		<D:visiblecount/>\r\n\
		<hm:special/>\r\n\
	</D:prop>\r\n\
</D:propfind>";

const char *msg_props_request = "\
<?xml version=\"1.0\"?>\r\n\
    <D:propfind xmlns:D=\"DAV:\" xmlns:hm=\"urn:schemas:httpmail:\" xmlns:m=\"urn:schemas:mailheader:\">\r\n\
    <D:prop>\r\n\
        <D:isfolder/>\r\n\
        <hm:read/>\r\n\
        <m:hasattachment/>\r\n\
        <m:to/>\r\n\
        <m:from/>\r\n\
        <m:subject/>\r\n\
        <m:date/>\r\n\
        <D:getcontentlength/>\r\n\
    </D:prop>\r\n\
</D:propfind>";

const char *propupdate_read_start = "\
<?xml version=\"1.0\"?>\r\n\
<D:propertyupdate xmlns:D=\"DAV:\" xmlns:hm=\"urn:schemas:httpmail:\" xmlns:m=\"urn:schemas:mailheader:\">\r\n\
  <D:set>\r\n\
    <D:prop>\r\n\
      <hm:read>";

const char *propupdate_read_end = "</hm:read>\r\n\
    </D:prop>\r\n\
  </D:set>\r\n\
</D:propertyupdate>";


xmlDocPtr doc = NULL;
xmlNsPtr  ns  = NULL;

static xmlNodePtr next_node(xmlNodePtr node);
static xmlNodePtr child_node(xmlNodePtr node);
static void cp_href(xmlNodePtr hrefptr,  FOLDER_STRUCT *folder_props);
static FOLDER_STRUCT *handle_response_node(xmlNodePtr node, char *foldername, FOLDER_STRUCT *folder_props);
static int count_folders(xmlNodePtr node);

extern int log_level;


/* Function: xmlParseDoc_error_handler
 * We just use this function to get rid of any warning messages produced by
 * our friend libxml2 when trying to parse the namespace DAV: which according
 * to them is an invalid URI, blah blah who cares just parse the thing..
 */
static void xmlParseDoc_error_handler(void *ctx, const char *msg, ...)
{
  /* throw away any warning messages unless logging level is 2 */
  va_list ap;
  char *buf;
  int len;
  
  if (log_level > 2) {
    buf =(char*) malloc(1001*sizeof(char));
    va_start(ap, msg);
    len = vsnprintf(buf, 1000, msg, ap);
    va_end(ap);
    buf[len]='\0';
    
    LOG(buf);
    free(buf);
  }
  return;
}


/* Function: initxml
 * In this function we parse the XML doc, check if we have the desired
 * namespaces and check for a multistatus node. Then we return the pointer to
 * the beginning of the parsed xml tree.
 */
xmlNodePtr initxml(char *xmldoc, char *namesp, xmlNodePtr cur) {

    static int warning_sent=0;

    
    /* Set Alias for Windows-1252 char encoding */
    if (xmlAddEncodingAlias("CP1252", "WINDOWS-1252") == -1) {
	  if (warning_sent++ < 1 && log_level > 0) 
		LOG("Warning: Unable to use CP1252 decoding, defaulting to ISO-8859-1, do you have libiconv?");
	  if (xmlAddEncodingAlias("ISO-8859-1","WINDOWS-1252") == -1) {
		/* woah, we can't setup encoding, let's exit */
	    if (log_level > 0) {
	      LOG("Fatal error: Unable to setup XML decoding, do you have libiconv?");
	    }
	    return NULL;
	  }
	}
	
    /* Build an XML tree from given pointer xmldoc */
    xmlSetGenericErrorFunc(NULL, xmlParseDoc_error_handler);
    doc = xmlParseDoc(xmldoc);
    if (doc == NULL) {
	xmlFreeDoc(doc);
    	return NULL;
    }

    /* Check the document is of the right kind */
    cur = xmlDocGetRootElement(doc);
    if (cur == 0) {
	xmlFreeDoc(doc);
	return NULL;
    }
    ns = xmlSearchNs (doc, cur, (const xmlChar *) namesp);
    if (ns == NULL) {
	xmlFreeDoc(doc);
	return NULL;
    }
    if (xmlStrcmp(cur->name, (const xmlChar *) "multistatus")) {
	xmlFreeDoc(doc);
	return NULL;
    }
    return cur;
}


/* This function is used when we send a dir_names_request to find the href of
 * the trash folder and the href of the msgfolderroot (which we then send a
 * dir_props_request to.
 */
char *findvalue(char *xmldoc, char *namesp, char *key) {
    xmlNodePtr cur=NULL;
	char* key_details=NULL;
	
    if ((cur = initxml(xmldoc, namesp, cur)) == NULL) 
    	return NULL;

    /* goto root node  */
    while ( cur && xmlIsBlankNode(cur))
      if (!(cur=next_node(cur))) return NULL;
	
	/* traverse xml tree! */
	if (!xmlStrcmp(cur->name, (const xmlChar *) "multistatus")) {
	  if (!(cur=child_node(cur))) return NULL; /* go to response node */
	  while (xmlStrcmp(cur->name, (const xmlChar *) "response"))
		if (!(cur=next_node(cur))) return NULL;	  
	  if (!(cur=child_node(cur))) return NULL; /* go to href/prostat node */
	  while (xmlStrcmp(cur->name,(const xmlChar *) "propstat")) 
		if (!(cur=next_node(cur))) return NULL;	  
	  if (!(cur=child_node(cur))) return NULL; /* go to prop node */
	  while(xmlStrcmp(cur->name, (const xmlChar *) "prop"))
		if (!(cur=next_node(cur))) return NULL;
	  cur=cur->children;    /* go to h:adbar node */
	  /* now search through sibling nodes from h:adbar (children of
	   * prop) until we find the key or we get to the end */
	  while (cur && (xmlStrcmp(cur->name, (const xmlChar *) key))) 
		cur=cur->next;
	  if (cur && !xmlStrcmp(cur->name, (const xmlChar *) key) && (cur->ns == ns)) {
		key_details=strdup(xmlNodeListGetString(doc, cur->xmlChildrenNode, 1));
		xmlFreeDoc(doc);
		return key_details;
	  }
	}
	xmlFreeDoc(doc);
	return NULL;  
}


FOLDER_STRUCT **folderlist(char *xmldoc, FOLDER_STRUCT **folder_list, int *folder_cnt) {
  xmlNodePtr cur=NULL;
  int i;

  *folder_cnt = 0;
  
  if (!(cur = initxml(xmldoc, "hm", cur)))
    return NULL;

  /* goto root node  */
  while ( cur && xmlIsBlankNode(cur))
    if (!(cur=next_node(cur)))
      return NULL;

  if (xmlStrcmp(cur->name, (const xmlChar *) "multistatus"))
    return NULL;

  /* get number of folders and allocate space in the list for each folder */
  *folder_cnt = count_folders(cur);
  if (!(cur=child_node(cur))) return folder_list; /* go down to href/propstat */
  if (!folder_list) folder_list = (FOLDER_STRUCT **) calloc (*folder_cnt, sizeof(FOLDER_STRUCT *));

  for (i=0;i<*folder_cnt;i++) {
    /* go to the first response node */
    while (xmlStrcmp(cur->name, (const xmlChar *) "response"))
      cur=next_node(cur);
    if (cur == NULL)
      return NULL;    
    folder_list[i] = handle_response_node(cur, NULL, NULL);
    if (!(cur=next_node(cur)))
      break;    
  }

#if 0
  if (i != (*folder_cnt)-1) {
    /* urgh.. let's cheat and call folder_cnt i+1 */
    *folder_cnt = i+1;
  }
#endif
  
  return folder_list;
}

/* Function handle_response_node()
 * This function is used to handle the response node of a parsed folder list
 * from the hotmail server. It allocates a folder_props structure if none is
 * passed to the function and fills it in unconditionally if foldername is
 * NULL otherwise it only fills in the details if the folder is called
 * foldername.
 */
static FOLDER_STRUCT *handle_response_node(xmlNodePtr node, char *foldername, FOLDER_STRUCT *folder_props) {
  int found_fld=0,found_unread=0,found_visible=0, found=0;
  xmlNodePtr href=NULL; /* temp store for the href */

  if (xmlStrcmp(node->name, (const xmlChar *) "response"))
    return NULL;
  if (!(node=child_node(node))) return NULL; /* go down to href/propstat */
  if (!folder_props)
    folder_props = calloc(1, sizeof(FOLDER_STRUCT));
  
  do {
    if (!xmlStrcmp(node->name, (const xmlChar *) "href")) {
      href=node;

    } else if (!xmlStrcmp(node->name,(const xmlChar *) "propstat")) {
      int ignore_special = 0;

      if (!(node=child_node(node))) break;

      while (xmlStrcmp(node->name, (const xmlChar *) "prop")) {
	if (!(node=next_node(node))) break;
      }
      
      if (!(node=child_node(node))) break; /* goto D:isfolder */

      do {

	/* it is possible for the node to be <hm:special> if it is a fixed
	 * hotmail folder or <D:displayname> if it is a folder added by a
	 * user to hotmail so we will now check for both */
	if (!xmlStrcmp(node->name, (const xmlChar *) "special")
	    && (!node->xmlChildrenNode 
		|| strlen(xmlNodeListGetString(doc, node->xmlChildrenNode, 1))) < 1)
	  ignore_special++;

	if ((!folder_props->name || !ignore_special)
	    && node->xmlChildrenNode /* catch null pointers */
	    && (!xmlStrcmp(node->name, (const xmlChar *) "special")
		|| !xmlStrcmp(node->name, (const xmlChar *) "displayname"))
	    && (!foldername || (!xmlStrcmp(foldername, xmlNodeListGetString(doc, node->xmlChildrenNode, 1))))) {

	  if (folder_props->name)
	    free(folder_props->name); /* special overrides displayname */
	  folder_props->name = strdup(xmlNodeListGetString(doc, node->xmlChildrenNode, 1));
	  found_fld++;
	} else if (!xmlStrcmp(node->name, (const xmlChar *) "unreadcount")) {
	  folder_props->unreadcount = atoi(xmlNodeListGetString(doc, node->xmlChildrenNode, 1));
	  found_unread++;
	} else if (!xmlStrcmp(node->name, (const xmlChar *) "visiblecount") ) {
	  folder_props->visiblecount = atoi(xmlNodeListGetString(doc, node->xmlChildrenNode, 1));
	  found_visible++;
	}
	if (node->next != NULL) node=node->next;
      } while (node->next != NULL);
      
      node=node->parent; /* go back to D:prop/D:status node */
      node=node->parent; /* go back to D:propstat/D:href */
    }

    if (found_fld && found_unread && found_visible)
      found=1;
    else 
      found_fld=found_unread=found_visible=0;
    
    if (node->next != NULL) node=node->next;

  } while (node->next != NULL);

  /* if we didn't find the correct folder then free any allocated space */
  if (!found) {
    if (folder_props->name) free(folder_props->name);
    if (folder_props->href) free(folder_props->href);
    folder_props=NULL;
  } else {
    cp_href(href, folder_props);
  }

  return folder_props;
}

/* We need to know how many folders we have before we can allocate space for
 * the folder properties. Hence the purpose of this function is to count the
 * number of folders so we can allocate space for their properties.
 */
static int count_folders(xmlNodePtr node) {
  int folder_cnt=0;

  if (node == NULL) return 0;

  /* traverse xml tree! */
  if (!xmlStrcmp(node->name, (const xmlChar *) "multistatus")) {
    /* go to the first response node */
    node=node->children;
    while (node) {
      if (!xmlStrcmp(node->name, (const xmlChar *) "response")) {
	if (!(node=child_node(node))) return folder_cnt; /* go down to href/propstat */
	do {
	  if (!xmlStrcmp(node->name,(const xmlChar *) "propstat")) {
	    if (!(node=child_node(node))) return folder_cnt;
	    while (xmlStrcmp(node->name, (const xmlChar *) "prop")) {
	      if (!(node=next_node(node))) return folder_cnt;
	    }
	    if (!(node=child_node(node))) return folder_cnt; /* goto D:isfolder */
	    /* check if it is a folder or not */
	    do {
	      if (!xmlStrcmp(node->name, (const xmlChar *) "isfolder")
		  && atoi(xmlNodeListGetString(doc, node->xmlChildrenNode, 1))) {
		folder_cnt++;
		break; /* because we know it is a folder now */
	      }
	      if (node->next != NULL) node=node->next;
	    } while (node->next != NULL);
	    node=node->parent; /* go back to D:prop/D:status node */
	    node=node->parent; /* go back to D:propstat/D:href */
	  }
	  if (node->next != NULL) node=node->next;
	} while (node->next != NULL);
	node=node->parent; /* go back to D:response node */
      }
      node=node->next; /* try to go to the next D:response node */
    } /* end of while loop */
  }
  return folder_cnt;
}

/**
 * Setup the folder_props data structure with folder_props->name - the name of
 * the folder, folder_props->unreadcount - how many messages are unread in the
 * folder, and folder_props->visiblecount - how many messages are in the
 * folder. Returns 1 on success, 0 on failure.
 **/
int folderprops(char *xmldoc, char *foldername, FOLDER_STRUCT *folder_props) {
  int found=0;
  xmlNodePtr cur=NULL;
	
  if ((cur = initxml(xmldoc, "hm", cur)) == NULL)
	return found;

  /* goto root node  */
  while ( cur && xmlIsBlankNode(cur))
    if (!(cur=next_node(cur))) return found;

  /* traverse xml tree! */
  if (!xmlStrcmp(cur->name, (const xmlChar *) "multistatus")) {
	/* go to the first response node */
	cur=cur->children;
	while (cur != NULL && !found) {
	  if (!xmlStrcmp(cur->name, (const xmlChar *) "response")) {
	    if (handle_response_node(cur, foldername, folder_props))
	      found = 1;
	  }
	  cur=cur->next; /* try to go to the next D:response node */
	} /* end of while loop */
  }

  xmlFreeDoc(doc);
  return found;
}


static xmlNodePtr next_node(xmlNodePtr node) {
  if ((node=node->next) == NULL)
    xmlFreeDoc(doc);
  return node;
}


static xmlNodePtr child_node(xmlNodePtr node) {
  if ((node=node->children) == NULL)
	xmlFreeDoc(doc);
  return node;
}


static void cp_href(xmlNodePtr hrefptr,  FOLDER_STRUCT *folder_props) {
  if (folder_props->href) { /* first check we don't have anything there */
	free(folder_props->href);
	folder_props->href=NULL;
  }
  folder_props->href=strdup(xmlNodeListGetString(doc, hrefptr->xmlChildrenNode, 1));
}


/* Builds a list of unread messages similar to the msglist. */
/* Returns the number of unread messages in the list. */
int buildunreadlist(MSG_STRUCT *msg[], int num_msgs, MSG_STRUCT *unreadmsg[])
{
  int unreadnum = 0, i;

  for (i = 0; i < num_msgs; i++)
  {
	if (!(msg[i])->readmark)
	{
	  unreadmsg[unreadnum++] = msg[i];
	}
  }

  /* Don't want to depend too much on the unreadcount from the XML
   * being accurate.. */

  return unreadnum;
}

/* STAT: Grab from xmldoc message lengths & hrefs */
/* FIXME: does not report errors to caller! */
/* RJVB 20020927: updates (expands) the msg_props list when necessary!
 \ This is done 1 entry at a time, which is hopelessly inefficient; it should be done
 \ before this function is called, OR the (new) total number of messages should be
 \ determined first thing after entering.
 */
void msglengths(char *xmldoc, FOLDER_STRUCT *folder_props )
{
    xmlNodePtr cur=NULL;
	 MSG_STRUCT **msg= folder_props->msg_props;

    if ((cur = initxml(xmldoc, "hm", cur)) == NULL)
    	return;

    /* goto root node  */
    while (cur && xmlIsBlankNode(cur))
	  cur=cur->next;
    if (!cur) { xmlFreeDoc(doc); return; }

    /* Traverse xml tree */
    if (!xmlStrcmp(cur->name, (const xmlChar *) "multistatus")) {
	  int i=0;
	  cur=cur->children;
	  while (cur != NULL) {
		if (!xmlStrcmp(cur->name, (const xmlChar *) "response")) {
		  xmlNodePtr cur_propstat=NULL;
		  cur=cur->children;
		  if (!cur) { xmlFreeDoc(doc); return; }
		  for(;;) {
			  /* RJVB 20020927: */
			if( i>= folder_props->visiblecount ){
			 int j;
				  /* need to expand the msg_props list: */
				if( (folder_props->msg_props=
						(MSG_STRUCT**) realloc( folder_props->msg_props, (i+1)* sizeof(MSG_STRUCT*)))
				){
					msg= folder_props->msg_props;
					for( j= folder_props->visiblecount; j< i+1; j++ ){
						msg[j]= NULL;
					}
					folder_props->visiblecount= i+1;
				}
				else{
					return;
				}
			}
			if( !msg[i] ){
				  /* need to initialise a still empty entry: */
				if( !(msg[i]= (MSG_STRUCT*) calloc(1, sizeof(MSG_STRUCT))) ){
					folder_props->visiblecount= i;
					return;
				}
			}
			if (!xmlStrcmp(cur->name, (const xmlChar *) "href"))
			  (msg[i])->href = strdup((const char *)xmlNodeListGetString(doc, cur->xmlChildrenNode, 1));
			if (!xmlStrcmp(cur->name, (const xmlChar *) "propstat"))
			  cur_propstat=cur; /* store it temporarily incase propstat occurs before href node */
			if (cur->next != NULL)
			  cur=cur->next;
		    else {
			  if((cur=cur_propstat) == NULL) {
				xmlFreeDoc(doc);
				return;
			  }
			  else break;
			}
		  }
		  cur=cur->children;
		  if (!cur) { xmlFreeDoc(doc); return; }
		  while (xmlStrcmp(cur->name, (const xmlChar *) "prop")) {
			if ((cur=cur->next) == NULL) {
			  xmlFreeDoc(doc);
			  return;
			}
		  }
		  cur=cur->children;
		  if (!cur) { xmlFreeDoc(doc); return; }
		  /* FIXME: should check if both read and getcontentlength have been
		   * found and if not report error */
		  if (msg[i]->date) {
		    free(msg[i]->date);
		    msg[i]->date = NULL;
		  }
		  for(;;) {
			if (!xmlStrcmp(cur->name, (const xmlChar *) "read"))
			  (msg[i])->readmark = atoi((const char *)xmlNodeListGetString(doc, cur->xmlChildrenNode, 1));
			else if (!xmlStrcmp(cur->name, (const xmlChar *) "getcontentlength"))
			  (msg[i])->length = atoi((const char *)xmlNodeListGetString(doc, cur->xmlChildrenNode, 1));
			else if (!xmlStrcmp(cur->name, (const xmlChar *) "date"))
			  (msg[i])->date = strdup(xmlNodeListGetString(doc, cur->xmlChildrenNode, 1));
			if (cur->next == NULL)
			  break;
			else
			  cur=cur->next;
		  }
		  cur = cur->parent;
		  cur = cur->parent;
		  cur = cur->parent;
		  i++;
		}
		cur = cur->next;
	  }
	}
	xmlFreeDoc(doc);
}


/* this function should be called before a stat or list to update visiblecount
 \ it essentially does the same as msglengths, traversing through the xml
 \ tree, but doesn't actually fill in any structures, it just counts the
 \ number of messages. Yes, this is inefficient, but it is better than before :-) */
void update_visiblecount(char *xmldoc, FOLDER_STRUCT *folder_props ) {
  xmlNodePtr cur=NULL;

  if ((cur = initxml(xmldoc, "hm", cur)) == NULL) { return; }
  /* goto root node  */
  while (cur && xmlIsBlankNode(cur)) { cur=cur->next; }
  if (!cur) { xmlFreeDoc(doc); return; }
  /* Traverse xml tree */
  if (!xmlStrcmp(cur->name, (const xmlChar *) "multistatus")) {
	int msg_cnt=0; /* count how many messages we find */
	cur=cur->children;
	while (cur != NULL) {
	  if (!xmlStrcmp(cur->name, (const xmlChar *) "response")) {
		xmlNodePtr cur_propstat=NULL;
		cur=cur->children;
		if (!cur) { xmlFreeDoc(doc); return; }

		for(;;) {
		  /*		  if (!xmlStrcmp(cur->name, (const xmlChar *) "href"))
					  (msg[i])->href = strdup((const char *)xmlNodeListGetString(doc, cur->xmlChildrenNode, 1));*/
		  if (!xmlStrcmp(cur->name, (const xmlChar *) "propstat"))
			cur_propstat=cur; /* store it temporarily incase propstat occurs before href node */
		  if (cur->next != NULL) { cur=cur->next; }
		  else {
			if ((cur=cur_propstat) == NULL) { xmlFreeDoc(doc); return; }
			else { break; }
		  }
		} /* end of for(;;) loop */

		cur=cur->children;
		if (!cur) { xmlFreeDoc(doc); return; }
		while (xmlStrcmp(cur->name, (const xmlChar *) "prop")) {
		  if ((cur=cur->next) == NULL) { xmlFreeDoc(doc); return; }
		}

		cur=cur->children;

		if (!cur) { xmlFreeDoc(doc); return; }
		else { msg_cnt++; } /* we found a message! woohoo! */
		/* PICKY FIXME: might want to actually check the node getcontentlength
		 * actually exists, but I can't see it being a major problem. */
		cur = cur->parent; /* return to the top of the xml tree */
		cur = cur->parent; /* return to the top of the xml tree */
		cur = cur->parent; /* return to the top of the xml tree */
	  }
	  folder_props->visiblecount=msg_cnt;
	  cur = cur->next;
	}
  }
  xmlFreeDoc(doc);
}


static void cleanup_folder_props(FOLDER_STRUCT *folder_props)
{
  int i;
  if (folder_props->msg_props) {
    for (i=0; i<folder_props->visiblecount; i++) {
      if (folder_props->msg_props[i]->href)
	free(folder_props->msg_props[i]->href);
      if (folder_props->msg_props[i]->message)
	free(folder_props->msg_props[i]->message);
      if (folder_props->msg_props[i]->header)
	free(folder_props->msg_props[i]->header);
      if (folder_props->msg_props[i]->date)
	free(folder_props->msg_props[i]->date);
      free(folder_props->msg_props[i]);
      folder_props->msg_props[i]=NULL;
    }
    free(folder_props->msg_props);
    folder_props->msg_props=NULL;
  }
  if (folder_props->unread_props) {
    free(folder_props->unread_props);
    folder_props->unread_props=NULL;
  }
}

int ParseXMLMsgList(ghttp_request *request, FOLDER_STRUCT *folder_props)
{
  int i, n;
  char *src;

  if (folder_props->msg_props || folder_props->unread_props) {
    /* urgh, let's clean it this up */
    cleanup_folder_props(folder_props);
  }

  src = ghttp_get_body(request);
  remove_to_from_subject (src);

  update_visiblecount(src, folder_props);
  
  if ((n = folder_props->visiblecount) > 0) {
	/* Creates array of MSG_STRUCTs */
	folder_props->msg_props = (MSG_STRUCT **) calloc(n, sizeof(MSG_STRUCT *));
	
	/* Allocate memory for each MSG_STRUCT */
	for (i = 0; i < n; i++){
	  /* 20030205 RJVB: I knew it -- one day MSG_STRUCT would expand and we'd want it to be all zeroes initially... 	*/
	  folder_props->msg_props[i] = (MSG_STRUCT *) calloc(1,sizeof(MSG_STRUCT));
	}
	
	/* Creates similarly-sized array for unread messages */
	folder_props->unread_props = (MSG_STRUCT **) calloc(n, sizeof(MSG_STRUCT *));
	
	/* Fill our MSG_STRUCTS */
	/* RJVB 20020927: necessary to update visiblecount and msg_props if the nr. of messages changed! */
	msglengths(src, folder_props);
	
	/* Build the list of unread messages */
	folder_props->unreadcount = buildunreadlist(folder_props->msg_props, n, folder_props->unread_props);
  }
  return 1;
}

/*#define DEBUG_HEAD 1*/

/* remove to, from and subject contents because they may contain UTF-16 characters and those cause parsing error */
remove_to_from_subject (char *in)
{
  char *p, *q, *r;
  int i;
  static const struct {
    char *start;
    char *end;
  } eliminate[] = {
    {"<m:subject>", "</m:subject>"},
    {"<m:from>", "</m:from>"},
    {"<m:to>", "</m:to>"},
    {NULL, NULL}
  };

#ifdef DEBUG_HEAD
  FILE *fp;

  fp = fopen ("/tmp/111.txt", "a");
  fprintf (fp, "1: %s\n", in);
#endif

  for (i=0; eliminate[i].start != NULL; i++)
    {
      for (r =in;; r = p+strlen(eliminate[i].end))
        {
          p = strstr (r, eliminate[i].start);
          q = strstr (r, eliminate[i].end);
          if (p != NULL && q != NULL)
            {
              p += strlen (eliminate[i].start);
              while (p < q)
                *p++ = ' ';
            }
          else
            {
              break;
            }
        }
    }

#ifdef DEBUG_HEAD
  fprintf (fp, "2: %s\n", in);
  fclose (fp);
#endif
}


/* Use this function to attempt to find the folder foldername and fill in the
 * folder_props->href, folder_props->visiblecount and
 * folder_props->unreadcount values
 */
int ParseXMLFolderProps(ghttp_request *request, char *foldername, FOLDER_STRUCT *folder_props) {
  return folderprops(ghttp_get_body(request), foldername, folder_props);
}

/* Use this function to find the root address of all folders in a particular
 * msn/hotmail account. It returns the address of the root folder. It also
 * fills in the trash and sendmsg folders href while it does its work - why not :-)
 */
char *ParseXMLFolderRoot(ghttp_request *request, URL_STRUCT *trash, URL_STRUCT *sendmsg) {
  trash->href = findvalue(ghttp_get_body(request), "hm", "deleteditems");
  sendmsg->href = findvalue(ghttp_get_body(request), "hm", "sendmsg");
  return findvalue(ghttp_get_body(request), "hm", "msgfolderroot");
}

FOLDER_STRUCT **BuildXMLFolderList(ghttp_request *request, FOLDER_STRUCT **folder_list, int *folder_cnt) {
  return folderlist(ghttp_get_body(request), folder_list, folder_cnt);
}

char *build_propertyupdate_str(char *readmark) {
  char *tmpxml;
  int len = strlen(propupdate_read_start) + strlen(propupdate_read_end) + 2;
  tmpxml = (char *) malloc(len);
  strlcpy(tmpxml, propupdate_read_start, len);
  strlcat(tmpxml, readmark, len);
  strlcat(tmpxml, propupdate_read_end, len);

  return tmpxml;
}
