/*
 * Hexxagon board game.
 * Copyright (C) 2001 Erik Jonsson.
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 * Email erik@nesqi.homeip.net
 * 
 */


#include "game.h"
#include "rendering.h"
#include "gtk-interface.h"
#include "gtk-navigation.h"

#include <libhexx/libhexx.h>

#include <gtk/gtk.h>

class HexxagonGame hex_game; /* The game. =) */

extern int comp_timeout;
extern int comp_level;

extern GtkWidget *darea;

int thinking = 0;
int mark_mode = 0;
int marked_cell;

void init_game()
{
	update_board();
}

void new_game()
{
	hex_game.reset();
	update_board();
}

extern BitBoard64 *clone_lookups;
extern BitBoard64 *jump_lookups;

extern int human_against_human;

void game_board_clicked(int x, int y)
{
	BitBoard64 board = hex_game.getBoard()->getBBBoard();
	BitBoard64 color = hex_game.getBoard()->getBBColor();

	if(mark_mode)
	{
		int cell = get_cell_from_pos(x, y);

		if(cell >= 0) /* only if a cell is hit */
		{
			BitBoard64 clone = clone_lookups[marked_cell] & ~board;
			BitBoard64 jump = jump_lookups[marked_cell] & ~board;
			
			if(clone.getBit(cell) || jump.getBit(cell))
			{
				if(clone.getBit(cell))
				{
					HexxagonMove move(cell);
					
					hex_game.applyMove(move);
				}
				else
				{
					HexxagonMove move(marked_cell, cell);
					
					hex_game.applyMove(move);
				}
				
				update_board();
				
				mark_mode = 0;

				if(!human_against_human)
					computer_move();

				return;
			}
		}

		game_board_abort();
		
		return;
	}
	else
	{
		int cell = get_cell_from_pos(x, y);

		if(cell >= 0)
		{
			if(board.getBit(cell) && color.getBit(cell))
			{
				BitBoard64 clone = clone_lookups[cell] & ~board;
				BitBoard64 jump = jump_lookups[cell] & ~board;

				if(clone || jump)
				{
					for(int i = 0; i < 61; i++)
					{
						if(clone.getBit(i))
							fill_cell(i, 1);
						if(jump.getBit(i))
							fill_cell(i, 2);
					}
					mark_mode = 1;
					marked_cell = cell;
					redraw_window(darea);

					return;
				}
			}
		}
	}
	
	gdk_beep();
}

void game_board_abort()
{
	if(mark_mode)
	{
		mark_mode = 0;

		update_board();
	}
	else
		gdk_beep();
}

void update_board()
{
	clear_board();

	BitBoard64 board = hex_game.getBoard()->getBBBoard();
	BitBoard64 color = hex_game.getBoard()->getBBColor();

	if(!hex_game.getTurn())
		color = ~color;

	for(int i = 0; i < 61; i++)
	{
		if(board.getBit(i))
			render_piece_to_board(i, !color.getBit(i));
	}

	/* Score */
	
	int score = hex_game.getBoard()->countBricks(2 - hex_game.getTurn());
	render_digit_to_board((int) (score / 10), 17, 20);
	render_digit_to_board(score % 10, 17 + 48, 20);

	score = hex_game.getBoard()->countBricks(hex_game.getTurn() + 1);
	render_digit_to_board((int) (score / 10), 460, 20);
	render_digit_to_board(score % 10, 460 + 48, 20);

	redraw_window(darea);
}

void computer_move()
{
	thinking = 1;
	mark_mode = 0;

	render_thinking();
	redraw_window(darea);

	hex_game.computerMove(comp_level, update_gui, comp_timeout * 1000);
	update_board();

	thinking = 0;
}

void game_load_file(gchar *selected_filename)
{
	int ret = hex_game.loadGame(selected_filename);
	
	if(ret == -1)
		quick_message("\n   Could not open file for loading!   \n");
	else if(ret == -3)
		quick_message("\n   Unknown file format or unknown version.   \n");	   
	else if(ret < 0)
		quick_message("\n   Error reading from load file!   \n");	   

	update_board();
}

void game_save_file(gchar *selected_filename)
{
	int ret = hex_game.saveGame(selected_filename);
	
	if(ret == -1)
		quick_message("\n   Could not open file for saving!   \n");
	else if(ret == -2)
		quick_message("\n   Error writing to save file!   \n");

	update_board();
}
