/* ***** BEGIN LICENSE BLOCK *****
 * Source last modified: $Id: pxjpgff.cpp,v 1.3.24.1 2004/07/09 01:53:35 hubbe Exp $
 * 
 * Portions Copyright (c) 1995-2004 RealNetworks, Inc. All Rights Reserved.
 * 
 * The contents of this file, and the files included with this file,
 * are subject to the current version of the RealNetworks Public
 * Source License (the "RPSL") available at
 * http://www.helixcommunity.org/content/rpsl unless you have licensed
 * the file under the current version of the RealNetworks Community
 * Source License (the "RCSL") available at
 * http://www.helixcommunity.org/content/rcsl, in which case the RCSL
 * will apply. You may also obtain the license terms directly from
 * RealNetworks.  You may not use this file except in compliance with
 * the RPSL or, if you have a valid RCSL with RealNetworks applicable
 * to this file, the RCSL.  Please see the applicable RPSL or RCSL for
 * the rights, obligations and limitations governing use of the
 * contents of the file.
 * 
 * Alternatively, the contents of this file may be used under the
 * terms of the GNU General Public License Version 2 or later (the
 * "GPL") in which case the provisions of the GPL are applicable
 * instead of those above. If you wish to allow use of your version of
 * this file only under the terms of the GPL, and not to allow others
 * to use your version of this file under the terms of either the RPSL
 * or RCSL, indicate your decision by deleting the provisions above
 * and replace them with the notice and other provisions required by
 * the GPL. If you do not delete the provisions above, a recipient may
 * use your version of this file under the terms of any one of the
 * RPSL, the RCSL or the GPL.
 * 
 * This file is part of the Helix DNA Technology. RealNetworks is the
 * developer of the Original Code and owns the copyrights in the
 * portions it created.
 * 
 * This file, and the files included with this file, is distributed
 * and made available on an 'AS IS' basis, WITHOUT WARRANTY OF ANY
 * KIND, EITHER EXPRESS OR IMPLIED, AND REALNETWORKS HEREBY DISCLAIMS
 * ALL SUCH WARRANTIES, INCLUDING WITHOUT LIMITATION, ANY WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, QUIET
 * ENJOYMENT OR NON-INFRINGEMENT.
 * 
 * Technology Compatibility Kit Test Suite(s) Location:
 *    http://www.helixcommunity.org/content/tck
 * 
 * Contributor(s):
 * 
 * ***** END LICENSE BLOCK ***** */

#include <stdio.h>
#include <string.h>
#include <stdlib.h> // for atol
#include <ctype.h>  // for toupper

// From include
#include "hxcom.h"
#include "hxtypes.h"
#include "hxcomm.h"
#include "ihxpckts.h"
#include "hxfiles.h"
#include "hxformt.h"
#include "hxplugn.h"
#include "hxpends.h"
#include "hxasm.h"
#include "hxerror.h"
#include "hxver.h"
#include "hxxres.h"
#include "hxxrsmg.h"
#include "hxvsrc.h"  /*IHXFileViewSource*/
#include "ihxfgbuf.h"
#include "hxengin.h"

// From pncont
#include "chxpckts.h"
#include "hxstring.h"
#include "hxslist.h"

// From pnmisc
#include "hxurl.h"
#include "unkimp.h"
#include "baseobj.h"

// From pxcomlib
#include "pxcolor.h"
#include "parseurl.h"
#include "pxutil.h"

// From coreres
#include "pixres.h"

#include "jpegvsrc.h"
#include "pxjpgff.h"
#include "jpgfformat.ver"

// pndebug
#include "hxheap.h"
#ifdef _DEBUG
#undef HX_THIS_FILE		
static char HX_THIS_FILE[] = __FILE__;
#endif

#define BASE_JPEGSTREAM_VERSION    HX_ENCODE_PROD_VERSION(0, 0, 0, 0)
#define OPACITY_JPEGSTREAM_VERSION HX_ENCODE_PROD_VERSION(0, 1, 0, 0)

const UINT32 CJPEGFileFormat::m_ulContentVersion     = HX_ENCODE_PROD_VERSION(0, 0, 0, 0);
const UINT32 CJPEGFileFormat::m_ulStreamVersion      = BASE_JPEGSTREAM_VERSION;
const char * const CJPEGFileFormat::m_pszDescription       = "Helix JPEG File Format Plugin";
const char * const CJPEGFileFormat::m_pszCopyright         = HXVER_COPYRIGHT;
const char * const CJPEGFileFormat::m_pszMoreInfoURL       = HXVER_MOREINFO;
const char * const CJPEGFileFormat::m_ppszFileMimeTypes[]  = {"image/jpeg", NULL};
const char * const CJPEGFileFormat::m_ppszFileExtensions[] = {"jpg", "jpeg", "jpe", "jfif", NULL};
const char * const CJPEGFileFormat::m_ppszFileOpenNames[]  = {"JPEG Images (*.jpg)", NULL};
const char * const CJPEGFileFormat::m_pszStreamMimeType    = "application/vnd.rn-jpegstream";

#ifdef _WINDOWS
extern HINSTANCE g_hInstance;
#endif

CJPEGFileFormat::CJPEGFileFormat()
{
    m_lRefCount                 = 0;
    m_pContext                  = NULL;
    m_pFileObject               = NULL;
    m_pFormatResponse           = NULL;
    m_pClassFactory             = NULL;
    m_ulState                   = kStateConstructed;
    m_pRequest                  = NULL;
    m_pError                    = NULL;
    m_pFileStat                 = NULL;
    m_ulFileSize                = 0;
    m_pFileBuffer               = NULL;
    m_pPacketInfoList           = NULL;
    m_pPacketInfoListItr        = NULL;
    m_ulRestartInterval         = 0;
    m_ulImageWidth              = 0;
    m_ulImageHeight             = 0;
    m_ulBitRate                 = 0;
    m_ulPreroll                 = 0;
    m_ulMaxPacketSize           = 0;
    m_ulMinPacketSize           = 0;
    m_ulAvgPacketSize           = 0;
    m_ulTotalBytesToSend        = 0;
    m_ulDuration                = 0;
    m_lCurrentTimeStamp         = 0;
    m_ulDisplayTime             = 0;
    m_pURL                      = NULL;
    m_ulTarget                  = kTargetBrowser;
    m_ulURLType                 = kURLTypeNormal;
    m_ulSeekTime                = 0;
    m_bReliable                 = FALSE;
    m_pMediaOpacityStr          = NULL;
    m_pMediaChromaKeyStr        = NULL;
    m_pMediaChromaKeyTolStr     = NULL;
    m_pMediaChromaKeyOpacityStr = NULL;
    m_ulBytesRead               = 0;
};

CJPEGFileFormat::~CJPEGFileFormat()
{
    Close();
};
    

STDMETHODIMP CJPEGFileFormat::QueryInterface(REFIID riid, void **ppvObj)
{
    if (IsEqualIID(riid, IID_IUnknown))
    {
        AddRef();
        *ppvObj = (IUnknown *) (IHXPlugin *) this;
        return HXR_OK;
    }
    else if (IsEqualIID(riid, IID_IHXPlugin))
    {
        AddRef();
        *ppvObj = (IHXPlugin *) this;
        return HXR_OK;
    }
    else if (IsEqualIID(riid, IID_IHXFileFormatObject))
    {
        AddRef();
        *ppvObj = (IHXFileFormatObject *) this;
        return HXR_OK;
    }
    else if (IsEqualIID(riid, IID_IHXFileResponse))
    {
        AddRef();
        *ppvObj = (IHXFileResponse *) this;
        return HXR_OK;
    }
    else if (IsEqualIID(riid, IID_IHXBackChannel))
    {
        AddRef();
        *ppvObj = (IHXBackChannel *) this;
        return HXR_OK;
    }
    else if (IsEqualIID(riid, IID_IHXASMSource))
    {
        AddRef();
        *ppvObj = (IHXASMSource *) this;
        return HXR_OK;
    }
    else if (IsEqualIID(riid, IID_IHXFileViewSource))
    {
	CJPEGViewSource* pVsrc = new CJPEGViewSource(m_pContext,
		(IUnknown*)(IHXPlugin*)this);
	if ( pVsrc == NULL )
	{
	    return HXR_FAIL;
	}
	return pVsrc->QueryInterface(riid, ppvObj);
    }
    else if (IsEqualIID(riid, IID_IHXThreadSafeMethods))
    {
	AddRef();
	*ppvObj = (IHXThreadSafeMethods*)this;
	return HXR_OK;
    }
    
    *ppvObj = NULL;

    return HXR_NOINTERFACE;
}

STDMETHODIMP_(UINT32) CJPEGFileFormat::AddRef()
{
    return InterlockedIncrement(&m_lRefCount);
}

STDMETHODIMP_(UINT32) CJPEGFileFormat::Release()
{
    if (InterlockedDecrement(&m_lRefCount) > 0)
    {
        return m_lRefCount;
    }

    delete this;

    return 0;
}

STDMETHODIMP CJPEGFileFormat::GetPluginInfo(REF(BOOL)         rbLoadMultiple,
                                            REF(const char *) rpszDescription,
                                            REF(const char *) rpszCopyright,
                                            REF(const char *) rpszMoreInfoURL,
                                            REF(UINT32)       rulVersionNumber)
{
    rbLoadMultiple   = TRUE;
    rpszDescription  = m_pszDescription;
    rpszCopyright    = m_pszCopyright;
    rpszMoreInfoURL  = m_pszMoreInfoURL;
    rulVersionNumber = TARVER_ULONG32_VERSION;

    return HXR_OK;
}

STDMETHODIMP CJPEGFileFormat::InitPlugin(IUnknown *pContext)
{
    // Check for input error
    if (!pContext)
    {
        return HXR_FAIL;
    }

    // Save a copy of the calling context
    m_pContext = pContext;
    m_pContext->AddRef();

    // Get an interface to the common class factory
    HX_RELEASE(m_pClassFactory);
    HX_RESULT retVal = m_pContext->QueryInterface(IID_IHXCommonClassFactory, (void **) &m_pClassFactory);
    if (retVal != HXR_OK)
    {
        HX_RELEASE(m_pContext);
        return retVal;
    }

    // Get the IHXErrorMessages interface - it's OK if TLC doesn't support it
    HX_RELEASE(m_pError);
    m_pContext->QueryInterface(IID_IHXErrorMessages, (void **) &m_pError);

    // Set the state
    m_ulState = kStatePluginInitialized;

    return HXR_OK;
}


STDMETHODIMP CJPEGFileFormat::GetFileFormatInfo(REF(const char **) rppszFileMimeTypes,
                                                REF(const char **) rppszFileExtensions,  
                                                REF(const char **) rppszFileOpenNames)
{
    rppszFileMimeTypes  = (const char **) m_ppszFileMimeTypes;
    rppszFileExtensions = (const char **) m_ppszFileExtensions;
    rppszFileOpenNames  = (const char **) m_ppszFileOpenNames;

    return HXR_OK;
}

STDMETHODIMP CJPEGFileFormat::InitFileFormat(IHXRequest        *pRequest, 
                                             IHXFormatResponse *pFormatResponse,
                                             IHXFileObject     *pFileObject)
{
    // Check for input error
    if (!pFormatResponse)
    {
        return HXR_FAIL;
    }
    if (!pRequest || !pFileObject)
    {
        return pFormatResponse->InitDone(HXR_FAIL);
    }

    // Save a copy of the format response
    m_pFormatResponse = pFormatResponse;
    m_pFormatResponse->AddRef();

     // Create an IHXValues object to hold the param name/value pairs
    IHXValues *pValues = NULL;
    HX_RESULT retVal = m_pClassFactory->CreateInstance(CLSID_IHXValues,
                                                       (void**) &pValues);
    if (FAILED(retVal))
    {
        return m_pFormatResponse->InitDone(retVal);
    }
    // Now add the following parameters to the IHXValues. These
    // parameters could have come as URL-encoded parameters or as
    // parameters in the request headers. These parameters will be
    // added as CString properties to the IHXValues.
    AddURLOrRequestParam(pRequest, "bitrate",     m_pContext, pValues);
    AddURLOrRequestParam(pRequest, "preroll",     m_pContext, pValues);
    AddURLOrRequestParam(pRequest, "duration",    m_pContext, pValues);
    AddURLOrRequestParam(pRequest, "displaytime", m_pContext, pValues);
    AddURLOrRequestParam(pRequest, "url",         m_pContext, pValues);
    AddURLOrRequestParam(pRequest, "target",      m_pContext, pValues);
    AddURLOrRequestParam(pRequest, "reliable",    m_pContext, pValues);
    AddURLOrRequestParam(pRequest, "mediaOpacity",       m_pContext, pValues);
    AddURLOrRequestParam(pRequest, "chromaKey",          m_pContext, pValues);
    AddURLOrRequestParam(pRequest, "chromaKeyTolerance", m_pContext, pValues);
    AddURLOrRequestParam(pRequest, "chromaKeyOpacity",   m_pContext, pValues);
    // Set the bitrate
    retVal = ExtractValueUINT32(pValues, "bitrate", kDefaultBitRate, m_ulBitRate);
    if (retVal != HXR_OK)
    {
        HX_RELEASE(pValues);
        ReportError(IDS_ERR_JPG_BADBITRATE);
        return m_pFormatResponse->InitDone(retVal);
    }
    // Set the preroll
    retVal = ExtractValueTime(pValues, "preroll", 0, m_ulPreroll);
    if (retVal != HXR_OK)
    {
        HX_RELEASE(pValues);
        ReportError(IDS_ERR_JPG_BADPREROLL);
        return m_pFormatResponse->InitDone(retVal);
    }
    // Set the duration
    retVal = ExtractValueTime(pValues, "duration", kDefaultDuration, m_ulDuration);
    if (retVal != HXR_OK)
    {
        HX_RELEASE(pValues);
        ReportError(IDS_ERR_JPG_BADDURATION);
        return m_pFormatResponse->InitDone(retVal);
    }
    // Set the display time
    retVal = ExtractValueTime(pValues, "displaytime", kDefaultDisplayTime, m_ulDisplayTime);
    if (retVal != HXR_OK)
    {
        HX_RELEASE(pValues);
        ReportError(IDS_ERR_JPG_BADDISPLAYTIME);
        return m_pFormatResponse->InitDone(retVal);
    }
    // Set the clickthru url
    CHXString cTmp;
    retVal = ExtractValueString(pValues, "url", NULL, cTmp);
    if (retVal != HXR_OK)
    {
        HX_RELEASE(pValues);
        ReportError(IDS_ERR_JPG_BADURL);
        return m_pFormatResponse->InitDone(retVal);
    }
    if (cTmp.GetLength() > 0)
    {
        HX_DELETE(m_pURL);
        m_pURL = new CHXString((const char*) cTmp);
    }
    // Set the target for the clickthru url
    CHXString cTarget;
    retVal = ExtractValueString(pValues, "target", "_browser", cTarget);
    if (retVal != HXR_OK)
    {
        HX_RELEASE(pValues);
        ReportError(IDS_ERR_JPG_BADTARGET);
        return m_pFormatResponse->InitDone(retVal);
    }
    // Set the reliable flag
    retVal = ExtractValueBOOL(pValues, "reliable", FALSE, m_bReliable);
    if (retVal != HXR_OK)
    {
        HX_RELEASE(pValues);
        ReportError(IDS_ERR_JPG_BADRELFLAG);
        return m_pFormatResponse->InitDone(retVal);
    }
    // Set the media opacity
    HX_RELEASE(m_pMediaOpacityStr);
    pValues->GetPropertyCString("mediaOpacity", m_pMediaOpacityStr);
    // Set the media chroma key
    HX_RELEASE(m_pMediaChromaKeyStr);
    pValues->GetPropertyCString("chromaKey", m_pMediaChromaKeyStr);
    // Set the media chroma key tolerance
    HX_RELEASE(m_pMediaChromaKeyTolStr);
    pValues->GetPropertyCString("chromaKeyTolerance", m_pMediaChromaKeyTolStr);
    // Set the media chroma key opacity
    HX_RELEASE(m_pMediaChromaKeyOpacityStr);
    pValues->GetPropertyCString("chromaKeyOpacity", m_pMediaChromaKeyOpacityStr);
    // Release the IHXValues object
    HX_RELEASE(pValues);

    // Check url encoding strings for validity
    if (!m_ulBitRate)
    {
        ReportError(IDS_ERR_JPG_BITRATEZERO);
        return m_pFormatResponse->InitDone(HXR_FAIL);
    }
    if (!m_ulDuration)
    {
        ReportError(IDS_ERR_JPG_DURATIONZERO);
        return m_pFormatResponse->InitDone(HXR_FAIL);
    }
    if (m_ulDisplayTime >= m_ulDuration)
    {
        ReportError(IDS_ERR_JPG_DISPTIMETOOBIG);
        return m_pFormatResponse->InitDone(HXR_FAIL);
    }
    if (cTarget == "_player")
    {
        m_ulTarget = kTargetPlayer;
    }
    else if (cTarget == "_browser")
    {
        m_ulTarget = kTargetBrowser;
    }
    else
    {
        ReportError(IDS_ERR_JPG_ILLEGALTARGET);
        return m_pFormatResponse->InitDone(HXR_FAIL);
    }

    // Assume default values
    m_ulURLType  = kURLTypeNormal;
    m_ulSeekTime = 0;

    // Check the URL if present
    if (m_pURL && m_pURL->GetLength() > 0)
    {
        // Check to see if it's a command
        if (m_pURL->Left(8) == "command:")
        {
            // Yes it is a command. Make sure it's either seek, pause, or play
            if (*m_pURL == "command:pause()")
            {
                m_ulURLType = kURLTypeCommandPause;
            }
            else if (*m_pURL == "command:play()")
            {
                m_ulURLType = kURLTypeCommandPlay;
            }
            else if (*m_pURL == "command:stop()")
            {
                m_ulURLType = kURLTypeCommandStop;
            }
            else if (m_pURL->Mid(8,5) == "seek(" &&
                     m_pURL->Right(1) == ")")
            {
                m_ulURLType = kURLTypeCommandSeek;

                // Make sure there's a valid seek time
                CHXString cSeekStr   = m_pURL->Mid(13, m_pURL->GetLength() - 14);
                BOOL bRet = ConvertTimeStringToULONG32((char*) (const char*) cSeekStr,
                                                       cSeekStr.GetLength(),
                                                       m_ulSeekTime);
                if (bRet == FALSE)
                {
                    ReportError(IDS_ERR_JPG_BADSEEKTIME);
                    return m_pFormatResponse->InitDone(HXR_FAIL);
                }
            }
            else
            {
                ReportError(IDS_ERR_JPG_UNKPLAYERCOMMAND);
                return m_pFormatResponse->InitDone(HXR_FAIL);
            }
        }
    }

    // Reconcile the URL and the target
    if (m_ulTarget == kTargetBrowser && m_ulURLType != kURLTypeNormal)
    {
        ReportError(IDS_ERR_JPG_NOTARGETBROWSER);
        return m_pFormatResponse->InitDone(HXR_FAIL);
    }

    // Save a copy of the IHXRequest
    m_pRequest = pRequest;
    m_pRequest->AddRef();

    // Save a copy of the file object
    m_pFileObject = pFileObject;
    m_pFileObject->AddRef();

    // Set the state
    m_ulState = kStateFileInitPending;

    return m_pFileObject->Init(HX_FILE_READ | HX_FILE_BINARY, this);
}

STDMETHODIMP CJPEGFileFormat::InitDone(HX_RESULT status)
{
    // Check the state
    if (m_ulState != kStateFileInitPending)
    {
        return HXR_UNEXPECTED;
    }

    // Get the IHXStat interface from the IHXFileObject
    HX_RELEASE(m_pFileStat);
    HX_RESULT retVal = m_pFileObject->QueryInterface(IID_IHXFileStat, (void **) &m_pFileStat);
    if (retVal != HXR_OK)
    {
        return m_pFormatResponse->InitDone(retVal);
    }

    // Set the state
    m_ulState = kStateFileStatPending;

    // Call IHXStat::Stat()
    return m_pFileStat->Stat(this);
}

STDMETHODIMP CJPEGFileFormat::StatDone(HX_RESULT status,
                                       UINT32    ulSize,
                                       UINT32    ulCreationTime,
                                       UINT32    ulAccessTime,
                                       UINT32    ulModificationTime,
                                       UINT32    ulMode)
{
    // Check the state
    if (m_ulState != kStateFileStatPending)
    {
        return HXR_UNEXPECTED;
    }

    // Check the status
    if (status != HXR_OK)
    {
        return m_pFormatResponse->InitDone(status);
    }

    // Save the file size
    m_ulFileSize = ulSize;

    // Check the size just to be sure
    if (!m_ulFileSize)
    {
        return m_pFormatResponse->InitDone(HXR_FAIL);
    }

    // Now we can release the IHXFileStat object
    HX_RELEASE(m_pFileStat);

    // Go ahead and allocate the file buffer
    HX_RELEASE(m_pFileBuffer);
    HX_RESULT retVal = m_pClassFactory->CreateInstance(CLSID_IHXBuffer,
                                                       (void**) &m_pFileBuffer);
    if (SUCCEEDED(retVal))
    {
        retVal = m_pFileBuffer->SetSize(m_ulFileSize);
    }
    if (FAILED(retVal))
    {
        return m_pFormatResponse->InitDone(HXR_FAIL);
    }

    // Clear the number of bytes read
    m_ulBytesRead = 0;

    // Set the state
    m_ulState = kStateFileReadPending;

    // Do a read of the entire file
    return m_pFileObject->Read(kReadSize);
}

STDMETHODIMP CJPEGFileFormat::ReadDone(HX_RESULT status, IHXBuffer *pBuffer)
{
    // Check the state
    if (m_ulState != kStateFileReadPending)
    {
        return HXR_UNEXPECTED;
    }

    // Check the status
    if (status != HXR_OK)
    {
        return m_pFormatResponse->InitDone(status);
    }

    // Check for input error
    if (!pBuffer)
    {
        return m_pFormatResponse->InitDone(HXR_FAIL);
    }

    // Make sure we don't try to copy past the end
    // of the file buffer
    UINT32 ulBytesToCopy = pBuffer->GetSize();
    if (m_ulBytesRead + ulBytesToCopy > m_ulFileSize)
    {
        ulBytesToCopy = m_ulFileSize - m_ulBytesRead;
    }
    // Copy the read buffer into the file buffer
    memcpy(m_pFileBuffer->GetBuffer() + m_ulBytesRead, /* Flawfinder: ignore */
           pBuffer->GetBuffer(),
           ulBytesToCopy);
    // Update the number of bytes read
    m_ulBytesRead += ulBytesToCopy;
    // Now if we have not read the whole file, then
    // read some more of the file
    if (m_ulBytesRead < m_ulFileSize)
    {
        // Don't have to change the state, since
        // we know it's already kStateFilePending
        return m_pFileObject->Read(kReadSize);
    }

    // Now we no longer need the file object
    if (m_pFileObject)
    {
	m_pFileObject->Close();
	HX_RELEASE(m_pFileObject);
    }

    // Here we call the packetization routine
    HX_RESULT retVal = ParseImageBuffer(m_pFileBuffer->GetBuffer(), m_ulFileSize);
    if (retVal != HXR_OK)
    {
        HX_RELEASE(m_pFileBuffer);
        return m_pFormatResponse->InitDone(retVal);
    }

    // Set the new state
    m_ulState = kStateFileFormatInitialized;

    // Now tell the format response we're ready
    return m_pFormatResponse->InitDone(HXR_OK);
}

STDMETHODIMP CJPEGFileFormat::GetFileHeader()
{
    // Check the state
    if (m_ulState != kStateFileFormatInitialized)
    {
        return HXR_UNEXPECTED;
    }

    // Get an IHXValues object
    IHXValues *pValues = NULL;
    HX_RESULT   retVal  = m_pClassFactory->CreateInstance(CLSID_IHXValues, (void **) &pValues);
    if (retVal != HXR_OK)
    {
        return m_pFormatResponse->FileHeaderReady(retVal, NULL);
    }

    // Set the stream count
    retVal = pValues->SetPropertyULONG32("StreamCount", 1);
    retVal = pValues->SetPropertyULONG32("IsRealDataType", 1);
    if (retVal != HXR_OK)
    {
        HX_RELEASE(pValues);
        return m_pFormatResponse->FileHeaderReady(retVal, NULL);
    }

    // Set the width and height
    pValues->SetPropertyULONG32("Width",  m_ulImageWidth);
    pValues->SetPropertyULONG32("Height", m_ulImageHeight);

    // Set the new state
    m_ulState = kStateFileHeaderSent;

    // Call the response interface
    retVal = m_pFormatResponse->FileHeaderReady(HXR_OK, pValues);

    // Release the reference on the IHXValues object
    pValues->Release();
    
    return retVal;
}

STDMETHODIMP CJPEGFileFormat::GetStreamHeader(UINT16 usStreamNum)
{
    // Check the state
    if (m_ulState != kStateFileHeaderSent)
    {
        return HXR_UNEXPECTED;
    }

    // Create an IHXValues object
    IHXValues *pHeader = NULL;
    HX_RESULT retVal = m_pClassFactory->CreateInstance(CLSID_IHXValues, (void **) &pHeader);
    if (retVal != HXR_OK)
    {
        return m_pFormatResponse->StreamHeaderReady(retVal, NULL);
    }

    // Create mime type IHXBuffer
    IHXBuffer *pMimeType = NULL;
    retVal = m_pClassFactory->CreateInstance(CLSID_IHXBuffer, (void **) &pMimeType);
    if (retVal != HXR_OK)
    {
        HX_RELEASE(pHeader);
        return m_pFormatResponse->StreamHeaderReady(retVal, NULL);
    }

    // Fill in the mime type
    retVal = pMimeType->Set((const BYTE*) m_pszStreamMimeType, strlen(m_pszStreamMimeType) + 1);
    if (retVal != HXR_OK)
    {
        HX_RELEASE(pHeader);
        HX_RELEASE(pMimeType);
        return m_pFormatResponse->StreamHeaderReady(retVal, NULL);
    }

    // /Create intrinsicDurationType IHXBuffer:
    IHXBuffer* pIntrinsicDurationType = NULL;
    retVal = m_pClassFactory->CreateInstance(
	    CLSID_IHXBuffer, (void **) &pIntrinsicDurationType);
    if (retVal != HXR_OK)
    {
        HX_RELEASE(pHeader);
        HX_RELEASE(pMimeType);
        return m_pFormatResponse->StreamHeaderReady(retVal, NULL);
    }
    // /This is a "discrete" media stream; it has no
    // intrinsic (native) duration; we want to continue
    // declaring 5000 for the duration, however, since
    // we may be serving this stream to an old, SMIL 1.0
    // presentation or as a stand-alone play in which the
    // author expects the old 5000-msec duration:
    retVal = pIntrinsicDurationType->Set((const BYTE*)"intrinsicDurationDiscrete",
                                         strlen("intrinsicDurationDiscrete") + 1);
    if (retVal != HXR_OK)
    {
        HX_RELEASE(pHeader);
        HX_RELEASE(pMimeType);
        HX_RELEASE(pIntrinsicDurationType);
        return m_pFormatResponse->StreamHeaderReady(retVal, NULL);
    }

    // Get the opaque data buffer from the JPEG object
    IHXBuffer *pOpaqueData = NULL;
    retVal = m_pClassFactory->CreateInstance(CLSID_IHXBuffer, (void **) &pOpaqueData);
    if (retVal != HXR_OK || pOpaqueData == NULL)
    {
        HX_RELEASE(pHeader);
        HX_RELEASE(pMimeType);
        HX_RELEASE(pIntrinsicDurationType);
        return m_pFormatResponse->StreamHeaderReady(retVal, NULL);
    }

    // Compute the size of the opaque buffer
    UINT32 ulOpaqueSize = 4 + // image width
                          4 + // image height
                          4 + // number of packets
                          4 + // display time
                          1 + // url target (either kTargetBrowser or kTargetPlayer)
                          1 + // url type
                          4 + // seek time
                          2 + // url string length
                          (m_pURL ? m_pURL->GetLength() : 0); // url string

    // Set the opaque data size
    retVal = pOpaqueData->SetSize(ulOpaqueSize);
    if (retVal != HXR_OK)
    {
        HX_RELEASE(pHeader);
        HX_RELEASE(pMimeType);
        HX_RELEASE(pIntrinsicDurationType);
        HX_RELEASE(pOpaqueData);
        return m_pFormatResponse->StreamHeaderReady(retVal, NULL);
    }

    // Set the values of the opaque data
    BYTE *pBuffer = pOpaqueData->GetBuffer();
    Pack32(pBuffer, m_ulImageWidth);           // image width
    Pack32(pBuffer, m_ulImageHeight);          // image height
    Pack32(pBuffer, (m_pPacketInfoList ? m_pPacketInfoList->GetCount() : 0)); // number of packets
    Pack32(pBuffer, m_ulDisplayTime);          // display time
    Pack8(pBuffer,  (BYTE) m_ulTarget);        // target (kTargetBrowser or kTargetPlayer)
    Pack8(pBuffer,  (BYTE) m_ulURLType);       // url type (normal, command-pause, command-play, command-seek)
    Pack32(pBuffer, m_ulSeekTime);             // seek time
    Pack16(pBuffer, (UINT16) (m_pURL ? m_pURL->GetLength() : 0)); // url string length
    if (m_pURL && m_pURL->GetLength() > 0)
    {
        memcpy(pBuffer, (BYTE*) (const char*) *m_pURL, m_pURL->GetLength()); /* Flawfinder: ignore */
    }

    // Initialize the time stamp
    INT32 lTime         = (INT32) TransmissionTime(m_ulTotalBytesToSend, m_ulBitRate);
    m_lCurrentTimeStamp = ((INT32) m_ulDisplayTime) - lTime;

    // Compute min preroll
    UINT32 ulMinPreRoll = (UINT32) (m_lCurrentTimeStamp < 0 ? -m_lCurrentTimeStamp : 0);
    if (m_ulPreroll < ulMinPreRoll)
    {
        m_ulPreroll = ulMinPreRoll;
    }

    // Compute predata bytes
    UINT32 ulPreDataBytes;
    if (m_ulDisplayTime > 0)
    {
        ulPreDataBytes = m_ulBitRate * m_ulPreroll / 8000;
        if (ulPreDataBytes > m_ulTotalBytesToSend)
        {
            ulPreDataBytes = m_ulTotalBytesToSend;
        }
    }
    else
    {
        ulPreDataBytes = m_ulTotalBytesToSend;
    }

     // Create buffer to hold ASM rules
    IHXBuffer *pASMRuleBook = NULL;
    retVal                   = m_pClassFactory->CreateInstance(CLSID_IHXBuffer, (void **) &pASMRuleBook);
    if (retVal != HXR_OK)
    {
        HX_RELEASE(pHeader);
        HX_RELEASE(pMimeType);
        HX_RELEASE(pIntrinsicDurationType);
        HX_RELEASE(pOpaqueData);
        return m_pFormatResponse->StreamHeaderReady(retVal, NULL);
    }
    // Generate an ASM rule book
    char szASMRuleBook[256]; /* Flawfinder: ignore */
    if (m_bReliable)
    {
        sprintf(szASMRuleBook, "AverageBandwidth=%ld,Priority=10;", m_ulBitRate); /* Flawfinder: ignore */
    }
    else
    {
        sprintf(szASMRuleBook, /* Flawfinder: ignore */
                "AverageBandwidth=%ld,Priority=5;AverageBandwidth=%ld,Priority=9;",
                m_ulBitRate, 0);
    }
    pASMRuleBook->Set((const BYTE *) szASMRuleBook, strlen(szASMRuleBook) + 1);

    // Now fill in all the standard properties for this stream in the stream header
    pHeader->SetPropertyBuffer ("OpaqueData",       pOpaqueData);
    pHeader->SetPropertyULONG32("StreamNumber",     0);
    pHeader->SetPropertyULONG32("MaxBitRate",       m_ulBitRate);
    pHeader->SetPropertyULONG32("AvgBitRate",       m_ulBitRate);
    pHeader->SetPropertyULONG32("MaxPacketSize",    m_ulMaxPacketSize);
    pHeader->SetPropertyULONG32("AvgPacketSize",    m_ulAvgPacketSize);
    pHeader->SetPropertyULONG32("StartTime",        0);
    pHeader->SetPropertyULONG32("PreDataAtStart",   1);
    pHeader->SetPropertyULONG32("PreRollAtStart",   0);
    pHeader->SetPropertyULONG32("PreDataAfterSeek", 0);
    pHeader->SetPropertyULONG32("PreRollAfterSeek", 1);
    pHeader->SetPropertyULONG32("PreData",          ulPreDataBytes);
    pHeader->SetPropertyULONG32("PreRoll",          m_ulPreroll);
    pHeader->SetPropertyULONG32("Duration",         m_ulDuration);
    pHeader->SetPropertyCString("MimeType",         pMimeType);
    pHeader->SetPropertyULONG32("ContentVersion",   m_ulContentVersion);
    pHeader->SetPropertyCString("ASMRuleBook",      pASMRuleBook);
    // We need to declare that this is a discrete media type so that
    // the SMIL 2.0 renderer can treat it as such.  We don't want to
    // just set the duration to 0 for this as that would break old SMIL 1.0
    // content when someone switched image servers to include a new build
    // of the jpg file format plug-in:
    pHeader->SetPropertyCString("intrinsicDurationType",
                                pIntrinsicDurationType);
    // If any one of the transparency-related parameters has been 
    // specified, then we will bump the stream version.
    UINT32 ulStreamVersion = m_ulStreamVersion;
    if (m_pMediaOpacityStr)
    {
        pHeader->SetPropertyCString("mediaOpacity", m_pMediaOpacityStr);
        ulStreamVersion = OPACITY_JPEGSTREAM_VERSION;
    }
    if (m_pMediaChromaKeyStr)
    {
        pHeader->SetPropertyCString("chromaKey", m_pMediaChromaKeyStr);
        ulStreamVersion = OPACITY_JPEGSTREAM_VERSION;
    }
    if (m_pMediaChromaKeyTolStr)
    {
        pHeader->SetPropertyCString("chromaKeyTolerance", m_pMediaChromaKeyTolStr);
        ulStreamVersion = OPACITY_JPEGSTREAM_VERSION;
    }
    if (m_pMediaChromaKeyOpacityStr)
    {
        pHeader->SetPropertyCString("chromaKeyOpacity", m_pMediaChromaKeyOpacityStr);
        ulStreamVersion = OPACITY_JPEGSTREAM_VERSION;
    }
    // Now set the stream version
    pHeader->SetPropertyULONG32("StreamVersion", ulStreamVersion);

    // Set the new state
    m_ulState = kStateStreamHeaderSent;

    // Pass the stream header back to the server
    retVal = m_pFormatResponse->StreamHeaderReady(HXR_OK, pHeader);

    // Release our references to everything we got from the common class factory above
    pOpaqueData->Release();
    pHeader->Release();
    pMimeType->Release();
    HX_RELEASE(pIntrinsicDurationType);
    pASMRuleBook->Release();

    return retVal;
}

STDMETHODIMP CJPEGFileFormat::GetPacket(UINT16 usStreamNum)
{
    // Check for input error
    if (usStreamNum != 0)
    {
        return HXR_UNEXPECTED;
    }

    // Check the state
    if (m_ulState != kStateStreamHeaderSent)
    {
        return HXR_UNEXPECTED;
    }

    // Are we finished sending packets?
    if (!m_pPacketInfoListItr)
    {
        HX_RELEASE(m_pFileBuffer);
        ClearPacketInfoList();
        return m_pFormatResponse->StreamDone(0);
    }

    // Get the current packet info
    PacketInfo* pInfo = NULL;
    if (m_pPacketInfoList)
    {
        pInfo = (PacketInfo*) m_pPacketInfoList->GetNext(m_pPacketInfoListItr);
    }
    if (!pInfo)
    {
        return m_pFormatResponse->PacketReady(HXR_FAIL, NULL);
    }

    // Create an IHXBuffer
    IHXBuffer *pBuffer = NULL;
    HX_RESULT   retVal  = m_pClassFactory->CreateInstance(CLSID_IHXBuffer, (void **) &pBuffer);
    if (retVal != HXR_OK)
    {
        return m_pFormatResponse->PacketReady(retVal, NULL);
    }

    // Create an IHXPacket
    IHXPacket *pPacket = NULL;
    retVal = m_pClassFactory->CreateInstance(CLSID_IHXPacket, (void **) &pPacket);
    if (retVal != HXR_OK)
    {
        HX_RELEASE(pBuffer);
        return m_pFormatResponse->PacketReady(retVal, NULL);
    }

    // Set the buffer size
    retVal = pBuffer->SetSize(pInfo->m_ulSize + kJPEGPacketOverhead);
    if (retVal != HXR_OK)
    {
        HX_RELEASE(pPacket);
        HX_RELEASE(pBuffer);
        return m_pFormatResponse->PacketReady(retVal, NULL);
    }

    // Set the opaque buffer's data
    BYTE *pData = pBuffer->GetBuffer();
    Pack32(pData, 0);                                 // dummy packet type
    Pack32(pData, 0);                                 // dummy image handle
    Pack32(pData, pInfo->m_ulSequenceNumber);         // sequence number
    Pack32(pData, 0);                                 // dummy flags
    Pack16(pData, (UINT16) pInfo->m_ulStartingBlock); // starting block
    Pack16(pData, (UINT16) pInfo->m_ulNumBlocks);     // number of blocks

    // Do the copy of the raw data
    memcpy(pData, pInfo->m_pBuffer, pInfo->m_ulSize); /* Flawfinder: ignore */

    // Get the timestamp
    UINT32 ulTimeStamp;
    if (m_ulDisplayTime > 0)
    {
        if (m_lCurrentTimeStamp >= 0)
        {
            ulTimeStamp = (UINT32) m_lCurrentTimeStamp;
        }
        else
        {
            ulTimeStamp = 0;
        }
    }
    else
    {
        ulTimeStamp = 0;
    }

    // Set the packet parameters
    pPacket->Set(pBuffer,           // opaque data
                 ulTimeStamp,       // time stamp
                 0,                 // stream number
                 HX_ASM_SWITCH_ON, // asm flags
                 0);                // asm rule number

    // Increment the time stamp
    INT32 lDataPacketTime = (INT32) TransmissionTime(pInfo->m_ulSize      +  // raw image data
                                                     kJPEGPacketOverhead,    // jpeg packet overhead
                                                     m_ulBitRate);
    m_lCurrentTimeStamp  += lDataPacketTime;

    retVal = m_pFormatResponse->PacketReady(HXR_OK, pPacket);

    // Release the buffer and packet
    HX_RELEASE(pBuffer);
    HX_RELEASE(pPacket);

    return HXR_OK;
}

STDMETHODIMP CJPEGFileFormat::SeekDone(HX_RESULT status)
{
    return HXR_UNEXPECTED;
}

STDMETHODIMP CJPEGFileFormat::Seek(UINT32 ulOffset)
{
    if (m_pFormatResponse)
    {
        return m_pFormatResponse->SeekDone(HXR_OK);
    }
    else
    {
        return HXR_UNEXPECTED;
    }
}

STDMETHODIMP CJPEGFileFormat::Close()
{
    HX_RELEASE(m_pContext);
    if (m_pFileObject)
    {
	m_pFileObject->Close();
	HX_RELEASE(m_pFileObject);
    }
    HX_RELEASE(m_pFormatResponse);
    HX_RELEASE(m_pClassFactory);
    HX_RELEASE(m_pRequest);
    HX_RELEASE(m_pError);
    HX_RELEASE(m_pFileStat);
    HX_RELEASE(m_pFileBuffer);
    ClearPacketInfoList();
    HX_DELETE(m_pPacketInfoList);
    HX_DELETE(m_pURL);
    HX_RELEASE(m_pMediaOpacityStr);
    HX_RELEASE(m_pMediaChromaKeyStr);
    HX_RELEASE(m_pMediaChromaKeyTolStr);
    HX_RELEASE(m_pMediaChromaKeyOpacityStr);

    return HXR_OK;
}

STDMETHODIMP CJPEGFileFormat::CloseDone(HX_RESULT status)
{
    return HXR_OK;
}

STDMETHODIMP CJPEGFileFormat::WriteDone(HX_RESULT status)
{
    // We don't ever write, so we don't expect to get this...
    return HXR_UNEXPECTED;
}

STDMETHODIMP CJPEGFileFormat::PacketReady(IHXPacket *pPacket)
{
    return HXR_OK;
}

STDMETHODIMP CJPEGFileFormat::Subscribe(UINT16 usStreamNumber, UINT16 usRuleNumber)
{
    return HXR_OK;
}

STDMETHODIMP CJPEGFileFormat::Unsubscribe(UINT16 usStreamNumber, UINT16 usRuleNumber)
{
    return HXR_OK;
}

HX_RESULT STDAPICALLTYPE CJPEGFileFormat::HXCreateInstance(IUnknown** ppIUnknown)
{
    HX_RESULT retVal = HXR_OK;

    if (ppIUnknown)
    {
        // Set default
        *ppIUnknown = NULL;
        // Create the object
        CJPEGFileFormat *pObj = new CJPEGFileFormat();
        if (pObj)
        {
            // QI for IUnknown
            retVal = pObj->QueryInterface(IID_IUnknown, (void**) ppIUnknown);
        }
        else
        {
            retVal = HXR_OUTOFMEMORY;
        }
        if (FAILED(retVal))
        {
            HX_DELETE(pObj);
        }
    }
    else
    {
        retVal = HXR_FAIL;
    }

    return HXR_OK;
}

void CJPEGFileFormat::ClearPacketInfoList()
{
    if (m_pPacketInfoList)
    {
        LISTPOSITION pos = m_pPacketInfoList->GetHeadPosition();
        while (pos)
        {
            PacketInfo* pInfo = (PacketInfo*) m_pPacketInfoList->GetNext(pos);
            HX_DELETE(pInfo);
        }
        m_pPacketInfoList->RemoveAll();
    }
}

HX_RESULT CJPEGFileFormat::ParseImageBuffer(BYTE *pFileBuffer, UINT32 ulFileSize)
{
    // Clear the packet info list
    ClearPacketInfoList();

    // Make reasonably sure this is a JPEG image. Most .jpg images should
    // be JFIF-compliant. However, a lot of images coming off digital cameras
    // are non-compliant: they don't have an APP0 marker, they have non-JFIF
    // APP0 markers, multiple APP0 markers, etc. Therefore, we will only
    // check the beginning of the file for the SOI marker
    if (pFileBuffer[0] != 0xFF || pFileBuffer[1] != kMarkerSOI)
    {
        return HXR_FAIL;
    }

    // Set defaults
    m_ulRestartInterval   = 0;

    // Now parse the file buffer
    BOOL   bHeaderComplete  = FALSE;
    BYTE  *pCurByte         = pFileBuffer;
    BYTE  *pBufLimit        = pFileBuffer + ulFileSize;
    UINT32 ulSequenceNumber = 0;
    while (bHeaderComplete == FALSE && pCurByte < pBufLimit)
    {
        // Look for a 0xFF, which would signify a marker
        if (*pCurByte++ == 0xFF)
        {
            // Now take different actions depending on what kind of
            // marker this is - some markers have data associated with
            // them (such as DRI) and others don't (such as SOI and EOI)
            BYTE ucMarker = *pCurByte++;

            if (!(ucMarker == kMarkerSOI ||
                  ucMarker == kMarkerEOI ||
                  ucMarker == kMarkerTEM ||
                  (ucMarker >= kMarkerRST0 && ucMarker <= kMarkerRST7)))
            {
                UINT32 ulSegLen = (pCurByte[0] << 8) | pCurByte[1];

                // Add SOF1 and SOF2 to support progressive
                if (ucMarker == kMarkerSOF0 || ucMarker == kMarkerSOF1 || ucMarker == kMarkerSOF2)
                {
                    // We look at this marker to extract the width and height
                    m_ulImageHeight     = (pCurByte[3] << 8) | pCurByte[4];
                    m_ulImageWidth      = (pCurByte[5] << 8) | pCurByte[6];
                }
                else if (ucMarker == kMarkerDRI)
                {
                    // We look at this marker to extract the restart interval
                    m_ulRestartInterval = (pCurByte[2] << 8) | pCurByte[3];
                }
                else if (ucMarker == kMarkerSOS)
                {
                    // This marker is the last marker in the header
                    bHeaderComplete = TRUE;
                }

                // Advance the pointer
                pCurByte += ulSegLen;
            }
        }
    }

    // Did we get a valid header?
    if (!bHeaderComplete)
    {
        return HXR_FAILED;
    }

    // Now we've reached the end of the header, make append
    // the first packet info struct
    PacketInfo *pInfo = new PacketInfo;
    if (!pInfo)
    {
        return HXR_OUTOFMEMORY;
    }
    pInfo->m_pBuffer          = pFileBuffer;
    pInfo->m_ulSize           = pCurByte - pFileBuffer;
    pInfo->m_ulSequenceNumber = ulSequenceNumber++;
    pInfo->m_ulStartingBlock  = 0;
    pInfo->m_ulNumBlocks      = 0;
    if (!m_pPacketInfoList)
    {
        m_pPacketInfoList = new CHXSimpleList();
    }
    if (m_pPacketInfoList)
    {
        m_pPacketInfoList->AddTail((void*) pInfo);
    }

    // If the restart interval is non-zero, then we need to look for
    // restart markers. This gives us some measure of loss tolerance.
    if (m_ulRestartInterval > 0)
    {
        BOOL   bImageComplete = FALSE;
        UINT32 ulCurBlock     = 0; 
        while (pCurByte < pBufLimit && bImageComplete == FALSE)
        {
            BOOL   bPacketComplete = FALSE;
            UINT32 ulNumRSTMarkers = 0;
            BYTE  *pCurStart       = pCurByte;
            while (bPacketComplete == FALSE && pCurByte < pBufLimit)
            {
                if (*pCurByte++ == 0xFF)
                {
                    BYTE ucMarker = *pCurByte++;
                    if (ucMarker >= kMarkerRST0 && ucMarker <= kMarkerRST7)
                    {
                        ulNumRSTMarkers++;
                        if (pCurByte - pCurStart >= kDefaultPacketSize)
                        {
                            bPacketComplete = TRUE;
                        }
                    }
                    else if (ucMarker == kMarkerEOI)
                    {
                        bPacketComplete = TRUE;
                        bImageComplete = TRUE;
                    }
                }
            }

            // Did we get a valid packet?
            if (bPacketComplete == FALSE)
            {
                return HXR_FAILED;
            }

            // Make a new packet info entry
            PacketInfo *pInfo = new PacketInfo;
            if (!pInfo)
            {
                return HXR_OUTOFMEMORY;
            }
            pInfo->m_pBuffer          = pCurStart;
            // We check how many bytes we have left after this
            UINT32 ulNumBytesLeft     = pBufLimit - pCurByte;
            if (ulNumBytesLeft < kMinimumPacketSize)
            {
                pCurByte       = pBufLimit;
                bImageComplete = TRUE;
            }
            pInfo->m_ulSize           = pCurByte - pCurStart;
            pInfo->m_ulSequenceNumber = ulSequenceNumber++;
            pInfo->m_ulStartingBlock  = ulCurBlock;
            pInfo->m_ulNumBlocks      = ulNumRSTMarkers;
            if (!m_pPacketInfoList)
            {
                m_pPacketInfoList = new CHXSimpleList();
            }
            if (m_pPacketInfoList)
            {
                m_pPacketInfoList->AddTail((void*) pInfo);
            }

            // Increment the current block
            ulCurBlock += ulNumRSTMarkers;
        }

        // Did we finish the image OK?
        if (bImageComplete == FALSE)
        {
            return HXR_FAILED;
        }
    }
    else
    {
        while (pCurByte < pBufLimit)
        {
            BYTE *pCurStart = pCurByte;
            pCurByte       += kDefaultPacketSize;
            if (pCurByte > pBufLimit)
            {
                pCurByte = pBufLimit;
            }

            // Make a new packet info entry
            PacketInfo *pInfo = new PacketInfo;
            if (!pInfo)
            {
                return HXR_OUTOFMEMORY;
            }
            pInfo->m_pBuffer          = pCurStart;
            pInfo->m_ulSize           = pCurByte - pCurStart;
            pInfo->m_ulSequenceNumber = ulSequenceNumber++;
            pInfo->m_ulStartingBlock  = 0;
            pInfo->m_ulNumBlocks      = 0;
            if (!m_pPacketInfoList)
            {
                m_pPacketInfoList = new CHXSimpleList();
            }
            if (m_pPacketInfoList)
            {
                m_pPacketInfoList->AddTail((void*) pInfo);
            }
        }
    }

    // Compute some statistics on the packets
    UINT32        ulMin = 0xFFFFFFFF;
    UINT32        ulMax = 0;
    UINT32        ulSum = 0;
    if (m_pPacketInfoList)
    {
        LISTPOSITION  pos = m_pPacketInfoList->GetHeadPosition();
        while (pos)
        {
            PacketInfo* pLInfo = (PacketInfo*) m_pPacketInfoList->GetNext(pos);
            if (pLInfo)
            {
                UINT32 ulSize = pLInfo->m_ulSize + kJPEGPacketOverhead;
                if (ulSize > ulMax)
                {
                    ulMax = ulSize;
                }
                if (ulSize < ulMin)
                {
                    ulMin = ulSize;
                }
                ulSum += ulSize;
            }
        }
    }

    // Set min and max packet size
    m_ulMinPacketSize    = ulMin;
    m_ulMaxPacketSize    = ulMax;

    // Set average packet size
    UINT32 ulNumPackets  = 0;
    if (m_pPacketInfoList)
    {
        ulNumPackets = m_pPacketInfoList->GetCount();
    }
    if (!ulNumPackets)
    {
        return HXR_FAILED;
    }
    m_ulAvgPacketSize    = (ulSum + (ulNumPackets >> 1)) / ulNumPackets;

    // Set total bytes to send
//    m_ulTotalBytesToSend = m_ulFileSize + ulNumPackets * (kIRMAPacketOverhead + kJPEGPacketOverhead);
    m_ulTotalBytesToSend = m_ulFileSize + ulNumPackets * kJPEGPacketOverhead; // Changed since this screwed predata calculations

    // Initialize the iterator
    if (m_pPacketInfoList)
    {
        m_pPacketInfoListItr = m_pPacketInfoList->GetHeadPosition();
    }

    return HXR_OK;
}

void CJPEGFileFormat::ReportError(UINT32 ulErrorID)
{
    // Try to get the string from the resource manager
    CHXString cErrStr;
    HX_RESULT retVal = GetResourceErrorString(ulErrorID, cErrStr);
    if (retVal != HXR_OK)
    {
        switch (ulErrorID)
        {
            case IDS_ERR_JPG_BADBITRATE:
                cErrStr = ERRSTR_JPG_BADBITRATE;
                break;
            case IDS_ERR_JPG_BADPREROLL:
                cErrStr = ERRSTR_JPG_BADPREROLL;
                break;
            case IDS_ERR_JPG_BADDURATION:
                cErrStr = ERRSTR_JPG_BADDURATION;
                break;
            case IDS_ERR_JPG_BADDISPLAYTIME:
                cErrStr = ERRSTR_JPG_BADDISPLAYTIME;
                break;
            case IDS_ERR_JPG_BADURL:
                cErrStr = ERRSTR_JPG_BADURL;
                break;
            case IDS_ERR_JPG_BADTARGET:
                cErrStr = ERRSTR_JPG_BADTARGET;
                break;
            case IDS_ERR_JPG_BADRELFLAG:
                cErrStr = ERRSTR_JPG_BADRELFLAG;
                break;
            case IDS_ERR_JPG_BITRATEZERO:
                cErrStr = ERRSTR_JPG_BITRATEZERO;
                break;
            case IDS_ERR_JPG_DURATIONZERO:
                cErrStr = ERRSTR_JPG_DURATIONZERO;
                break;
            case IDS_ERR_JPG_DISPTIMETOOBIG:
                cErrStr = ERRSTR_JPG_DISPTIMETOOBIG;
                break;
            case IDS_ERR_JPG_ILLEGALTARGET:
                cErrStr = ERRSTR_JPG_ILLEGALTARGET;
                break;
            case IDS_ERR_JPG_BADSEEKTIME:
                cErrStr = ERRSTR_JPG_BADSEEKTIME;
                break;
            case IDS_ERR_JPG_UNKPLAYERCOMMAND:
                cErrStr = ERRSTR_JPG_UNKPLAYERCOMMAND;
                break;
            case IDS_ERR_JPG_NOTARGETBROWSER:
                cErrStr = ERRSTR_JPG_NOTARGETBROWSER;
                break;
            case IDS_ERR_JPG_NOPROGRESSIVE:
                cErrStr = ERRSTR_JPG_NOPROGRESSIVE;
                break;
            default:
                cErrStr = ERRSTR_JPG_GENERALERROR;
                break;
        }
    }
 
    if (m_pError)
    {
        m_pError->Report(HXLOG_CRIT, HXR_FAIL, 0, (const char*) cErrStr,  NULL);
    }
}

HX_RESULT CJPEGFileFormat::GetResourceErrorString(UINT32 ulErrorID, CHXString& rErrorStr)
{
    IHXExternalResourceManager* pResMgr = NULL;
    HX_RESULT retVal = m_pContext->QueryInterface(IID_IHXExternalResourceManager, (void**) &pResMgr);
    if (retVal != HXR_OK)
    {
        return retVal;
    }

	IHXExternalResourceReader* pResRdr = NULL;
    retVal = pResMgr->CreateExternalResourceReader(CORE_RESOURCE_SHORT_NAME, pResRdr);
    if (retVal != HXR_OK)
    {
        HX_RELEASE(pResMgr);
        return retVal;
    }

#ifdef _WINDOWS
    char szDLLPath[1024]; /* Flawfinder: ignore */
    GetModuleFileName((HMODULE)g_hInstance, szDLLPath, sizeof(szDLLPath));
    pResRdr->SetDefaultResourceFile(szDLLPath);
#endif

    IHXXResource* pRes = pResRdr->GetResource(HX_RT_STRING, ulErrorID);
    if(!pRes)
    {
        HX_RELEASE(pResRdr);
        HX_RELEASE(pResMgr);
        return HXR_FAIL;
    }

    // Assign the error string to the out parameter
    rErrorStr = (const char*) pRes->ResourceData();

    // Release all references
    HX_RELEASE(pRes);
    HX_RELEASE(pResRdr);
    HX_RELEASE(pResMgr);

    return HXR_OK;
}

STDMETHODIMP_(UINT32) CJPEGFileFormat::IsThreadSafe()
{
    return HX_THREADSAFE_METHOD_FF_GETPACKET |
           HX_THREADSAFE_METHOD_FSR_READDONE;
}
