// sched.c
// -------
//		Copyright 1988 D.C. Lindsay at Carnegie Mellon University
//
// Implements the "scheduler" class: see  .h for specs.

#include "sched.h"
#include <stdlib.h>
#include <stream.h>


//extern schedulerclass scheduler;
//extern schedulerclass* scheduler;

#define SWAPF( field, pa, pb )	e.field = pa->field;	\
				pa->field = pb->field;	\
				pb->field = e.field

inline void SWAPEVENT( event* pa, event* pb ) {
  struct event e;
  SWAPF( tag, pa, pb );
  SWAPF( iact, pa, pb );
  SWAPF( pmact, pa, pb);
  int i = pa->arg.i;
  pa->arg.i = pb->arg.i;
  pb->arg.i = i;
  SMessagePtr spm = pa->arg.pm;
  pa->arg.pm = pb->arg.pm;
  pb->arg.pm = spm;
  SWAPF( time,pa, pb );
}

//==================================================
// Returns NULL if heap empty, or else pointer to front event.
//
struct event *nextevent( heap *ph )
{
	if( ph->numevents == 0 ) return NULL;
	else return &(ph->events[1]);
}
//==================================================
// Consume removes the front entry from the heap, and restructures
// the heap to fill the front.
//
void consume( heap *ph )
{
	int i,j,k;
	struct event *ip, *jp, *kp, *np;
	i = 1;				// heap is 1..numevents
	ip = &(ph-> events[i]);
	np = &(ph-> events[ ph->numevents ]);
	*ip = *np;			// copy rear event to fromt
	(ph->numevents)--;
	for(;;){			// now sift i towards rear
		j = i + i;
		if( j > ph-> numevents ) break;
		k = j + 1;
		jp = &(ph-> events[j]);
		kp = jp + 1;		// 1 event, that is

		if( k <= ph-> numevents ) 
			if( jp-> time > kp-> time ) { jp = kp; j = k; }
		// now ignore k; j is the one to compare to i
		if( ip-> time <= jp-> time ) break;	// invariant satisfied

		SWAPEVENT( ip, jp );
		i = j;
		ip = jp;
	}

}

//==================================================
schedulerclass::schedulerclass()
{
	this-> clock = 0;
	this-> eventq.numevents = 0;		// inits eventq
	this-> eventscount = 0;
	this-> highwater = 0;
}

//==================================================
void schedulerclass::print( char *unitsstring )
{
	cout << "Scheduler: Elapsed time " 
		<< form( "%lu", this-> clock )
		// (this-> clock) 			// came out n've!
		<< " "
		<< unitsstring << ".\n"
		<< " Events scheduled: " << (this-> eventscount) << ".\n"
		<< " High water - events queued at once: " 
		<< (this-> highwater)	<< ".\n";
}


//==================================================
void schedulerclass::run()
{
	cout << "Scheduler running\n";
	
	for(;;){
		struct event *p = nextevent( &(this->eventq));
		if( p == NULL ) break;
		if( p-> time > this->clock ) {	
			// time must advance 
			this->clock = p-> time;
			if( SCHEDTRACE ) {
			  cout << "Clock to " << this->clock << "\n";
			  cout.flush();
			}
		}
		if( p-> time == this->clock ) {
		  // no time needs to elapse, just do it
		  (this-> eventscount) ++;
//#ifdef CASTINT
		  switch (p->tag) {
		   case INT:
		    (*p->iact)(p->arg.i);
		    break;
		   case PM:
		    (*p->pmact)(p->arg.pm);
		    break;
		  }
/*
#else
			(*fn)(p->arg);
#endif
*/
			consume( &(this->eventq));
			continue;
		}
		cout << "ERROR: retrograde time " << p-> time << "\n";
		exit(1);
	}
}
//==================================================

void schedulerclass::post( PMAction act, const SMessagePtr arg, delta when ) {
	if( SCHEDTRACE ) {
	  cout << "\tthis->clock is " << this->clock << ".\n";
	  cout << "\tpost in " << when << " = at " <<
	    (when+this->clock) << "\n";
	  cout.flush();
	}
	
	// find the rear of the eventq, and stamp things in
	int i = ++(this-> eventq.numevents);	// note use 1.. not 0..
	if( i > HEAP_SIZE ) {
		cout << "ERROR: SCHEDULE QUEUE OVERFLOW, quitting.\n";
		exit(1);
	}
	if( highwater < i ) highwater = i;
	struct event *ip = &(this->eventq.events[i]);
	ip->tag = PM;
	ip->pmact = act;
	ip->arg.pm = arg;
	ip-> time = this->clock + when;
	if( ip-> time < this-> clock ) {
		cout << "ERROR: SCHEDULE CLOCK OVERFLOW, quitting.\n";
		exit(1);
	}

	// sift the rear of the eventq frontwards.
	// The invariant is that [j] has its time <= time at [2j], [2j+1]
	// where j is unit-origin.
	int r;
	struct event *rp;
	for(;;){
		r = i/2;
		if( r == i ) break;			// done, at root
		rp = &(this->eventq.events[r]);
		if( rp->time <= ip->time ) break; 	// done, invariant ok

		// not done, swap the two and refocus on r
		SWAPEVENT( ip, rp );
		i = r;
		ip = rp;
	}
}


void schedulerclass::post( IntAction act, const int arg, delta when ) {
	if( SCHEDTRACE ) {
	  cout << "\tthis->clock is " << this->clock << ".\n";
	  cout << "\tpost in " << when << " = at " <<
	    (when+this->clock) << "\n";
	  cout.flush();
	}
	
	// find the rear of the eventq, and stamp things in
	int i = ++(this-> eventq.numevents);	// note use 1.. not 0..
	if( i > HEAP_SIZE ) {
		cout << "ERROR: SCHEDULE QUEUE OVERFLOW, quitting.\n";
		exit(1);
	}
	if( highwater < i ) highwater = i;
	struct event *ip = &(this->eventq.events[i]);
	ip->tag = INT;
	ip->iact = act;
	ip->arg.i = arg;
	ip->time = this->clock + when;
	if( ip-> time < this-> clock ) {
		cout << "ERROR: SCHEDULE CLOCK OVERFLOW, quitting.\n";
		exit(1);
	}

	// sift the rear of the eventq frontwards.
	// The invariant is that [j] has its time <= time at [2j], [2j+1]
	// where j is unit-origin.
	int r;
	struct event *rp;
	for(;;){
		r = i/2;
		if( r == i ) break;			// done, at root
		rp = &(this->eventq.events[r]);
		if( rp->time <= ip->time ) break; 	// done, invariant ok

		// not done, swap the two and refocus on r
		SWAPEVENT( ip, rp );
		i = r;
		ip = rp;
	}
}



//==================================================

//event::event() { }

