import Maybe
import List		(sort,isPrefixOf,isSuffixOf,intersperse)
import Monad		(when)
import System
import Char		(isDigit,digitToInt,isUpper,toUpper,isAlphaNum,isSpace)
import IO		(hFlush,stdout,hSetBuffering,BufferMode(..))
import FFIExtensions	(withCString)

import Observe		(ObserveResult(..)
			,isFound,isNotFound,isInterrupted,fromFound
			,newObservation,newObservationSrc)
import Ident		(Ident(..),collateIdents,getAllIdents,sortIdents
			,showInfo)
import LowLevel		(FileNode(..),nil,openHatFile,getResult,hatVersionNumber
			,getSrcRef,srcRefFile,srcRefLine,srcRefCol
			,getDefnRef,defnFile,defnLine,defnCol)
import SExp		(SExp(..),Label,prettyEquation,prettyExpression
			,fileNode2SExp,funId,label,QName(..))
import HighlightStyle	(cursorUp,cleareol,highlight,Highlight(..),Colour(..)
			,getTerminalSize)
import Trie		(Trie,Search(..),emptyTrie,match)
import TExp		(linearise)
import Pattern		(topMatchPat,parsePat,lexPat)
import CmdLine		(initialize,cmdline)


spawnDetectCmd  = "xterm -e hat-detect "
spawnObserveCmd = "xterm -e hat-observe "
spawnTrailCmd   = "xterm -e hat-trail "

hatView :: FilePath -> Int -> Int -> String
hatView f y x  = "xterm -e hat-view "++f++" "++show y++" "++show x++" &"


shortHelpText  = ":? for help, :q to quit"

----
data InteractiveState =
    State
	{ lastObserved	:: [FileNode]
	, more		:: Bool
	, equationsPerPage :: Int
	, hatFile       :: String
	, currentPos	:: Int
	, cutoffdepth	:: Int
	, unevalMode	:: Bool
	, stringSugar	:: Bool
	, listSugar	:: Bool
	, recursiveMode	:: Bool
	, showRHS	:: Bool
	, equations	:: Bool
	, showQual	:: Bool
	, filterMode    :: Keep
	, screenWidth	:: Int
	, symbolTable   :: ([Ident],[Ident],[Ident])
	}
initialState :: InteractiveState
initialState = State
	{ lastObserved = []
	, more = False
	, equationsPerPage = 10
	, hatFile = ""
	, currentPos = 0
	, cutoffdepth = 10
	, unevalMode = False
	, stringSugar = True
	, listSugar = True
	, recursiveMode = True
	, showQual = False
	, showRHS = True
	, equations = True
	, filterMode = Unique
	, screenWidth = 80
	, symbolTable = ([],[],[])
	}
----

-----------------------------------------------------------------------
-- main entry point
-----------------------------------------------------------------------

main = do
    hSetBuffering stdout NoBuffering
    arguments <- System.getArgs
 -- putStrLn ("cmdline args: "++show arguments)
    let numArgs = length arguments
    when (numArgs < 1) (do putStrLn cmdlineHelp
                           exitWith (ExitFailure 1))
    putStrLn ("\n        hat-observe "++hatVersionNumber
              ++"    ("++shortHelpText++")\n")
    (width,_) <- getTerminalSize
    CmdLine.initialize	-- for readline functionality
    let hatFile = rectify (arguments!!0)
        state   = initialState { screenWidth=width, hatFile=hatFile }
    prog <- System.getProgName
    withCString prog (\p-> withCString hatFile (openHatFile p))
    putStrLn (highlight [Foreground Red] "Loading symbol table...")
    collateIdents
    syms <- getAllIdents
    let state' = state { symbolTable = syms }
    putStr (cursorUp++cleareol)	-- delayed until next output appears
    case numArgs of
      1 -> interactive state'
      2 -> doCommand (Pattern (arguments!!1)) state'
      4 -> doCommand (Location (tail arguments)) state'
      _ -> do putStrLn cmdlineHelp
              exitWith (ExitFailure 1)
  where
    rectify :: FilePath -> FilePath
    rectify f | ".hat" `isSuffixOf` f = f
              | otherwise = f ++ ".hat"

----
showObservation :: InteractiveState -> Int -> FileNode -> IO ()
showObservation state i node = do
  putStr (cursorUp ++ cleareol)	-- delay relies on line-buffered output
  putStrLn (display (screenWidth state) (cutoffdepth state)
                    (unevalMode state) (stringSugar state)
                    (listSugar state) (showQual state) node)
  where
  display | equations state && showRHS state = prettyEquation (show i++" ") ""
          | otherwise                        = prettyExpression (show i++" ")


-- arguments to showObservationList are:
--   i     :: Int		numbered equation
--   max   :: Int		how many equations to show at once
--   exprs :: [FileNode]	roots of equations
-- result is number of equations that have been shown
showObservationList :: InteractiveState -> Int -> Int -> [FileNode] -> IO Int
showObservationList state _ _ [] = do putStr (cursorUp ++ cleareol); return 0
showObservationList state i 0 _  = do putStr (cursorUp ++ cleareol); return 0
showObservationList state i max (e:r) = do
    showObservation state i e
    putStrLn (highlight [Foreground Red] "searching: (^C to interrupt)")
    count <- showObservationList state (i+1) (max-1) r
    return (count+1)

makeObserveSrc :: [String] -> ObserveResult
makeObserveSrc [mod,line,col] | all isDigit line && all isDigit col =
    newObservationSrc mod (fromJust (stringToInt line))
                          (fromJust (stringToInt col))
makeObserveSrc _  | otherwise = NotFound


makeObserve :: InteractiveState -> QName -> Maybe QName -> SExp () -> Int
               -> ObserveResult
makeObserve state ident1 ident2 pat arity =
    let observed = newObservation ident1 ident2 (recursiveMode state) arity in
    if not (isFound observed) then observed
    else ( Found
         . map fromEqn
         . filter (\o-> topMatchPat o pat)
         . (case filterMode state of
              All -> id
              _   -> uniqueify)
         . map toSEqn
         . fromFound)
         observed


-- This is a unique-ifier which builds a Trie to compare expressions.
-- Each expression is represented as a token stream, obtained by first
-- converting the FileNode to an expression tree, then linearising the
-- tree (by pre-order traversal).  The token stream is matched against
-- the Trie, and if found, the corresponding FileNode is discarded.  If
-- not found, the representation is stored in the Trie, and the expression
-- returned.
uniqueify :: [SExp Label] -> [SExp Label]
uniqueify eqns = trymatch emptyTrie eqns
  where trymatch trie [] = []
        trymatch trie (eqn:eqns) =
            case match (linearise eqn) trie of
              Exists    -> trymatch trie eqns
              New trie' -> eqn: trymatch trie' eqns

-- For unique-ification *and* for pattern matching, we first need an
-- expression-tree representation of the equation at the file node.
-- This representation has a *very* deep cutoff value for correctness,
-- and no sugaring of subexprs to allow all forms of matching.
toSEqn :: FileNode -> SExp Label
toSEqn node =
  SEquation ("",nil)
            (fileNode2SExp 2000 False False True ("l",node))
            (fileNode2SExp 2000 False False False ("r",getResult node True))

-- After unique-ification and pattern matching, we want the original
-- file node back.
fromEqn :: SExp Label -> FileNode
fromEqn (SEquation _ lhs rhs) = snd (label lhs)



showSomeMore :: InteractiveState -> IO (Int,Bool)
showSomeMore state =
  let showNowList = drop (currentPos state) (lastObserved state);
      hasMore = not (null (drop (equationsPerPage state) showNowList))
  in do
    putStrLn (highlight [Foreground Red] "searching: (^C to interrupt)")
    count <- showObservationList state (currentPos state + 1)
	                         (equationsPerPage state) showNowList
    return (count + currentPos state, hasMore)

interactive :: InteractiveState -> IO ()
interactive state
    | more state =
        do cmd <- getCommand "--more--> "
           case cmd of
             Pattern _ -> interactive state{more=False}
             _ -> doCommand cmd state
    | otherwise =
        do cmd <- getCommand "hat-observe> "
           doCommand cmd state


getEquationNumber :: Int -> [FileNode] -> IO (Maybe FileNode)
getEquationNumber n lastObserved =
  let nodes = drop (n-1) lastObserved in
  if n>0 then
       if null nodes then do -- This test may take a while!
	    putStrLn "No equation with this number"
	    return Nothing
       else return (Just (head nodes))
  else return Nothing


----
setState :: Mode -> InteractiveState -> InteractiveState
setState (Uneval b)     state = state {unevalMode=b}
setState (Strings b)    state = state {stringSugar=b}
setState (Lists b)      state = state {listSugar=b}
setState (Recursive b)  state = state {recursiveMode=b}
setState (Qualify b)    state = state {showQual=b}
setState (Equations b)  state = state {equations=b}
setState (Filter k)     state = state {filterMode=k}
setState (CutOff n)     state = state {cutoffdepth=max n 1}
setState (Deeper n)     state = state {cutoffdepth=cutoffdepth state + n}
setState (Shallower n) state = state {cutoffdepth=max (cutoffdepth state - n) 1}
setState (Group n)      state = state {equationsPerPage=max n 1}

showState (Uneval _)    state = "  unevaluated exprs shown in full ("
                                ++ highlight [Underscore] "uneval"
                                ++ ") "
                                ++ showOnOff (unevalMode state)
showState (Strings _)   state = "  string sugar ("
                                ++ highlight [Underscore] "strSugar"
                                ++ ") "
                                ++ showOnOff (stringSugar state)
showState (Lists _)     state = "  list sugar ("
                                ++ highlight [Underscore] "listSugar"
                                ++ ") "
                                ++ showOnOff (listSugar state)
showState (Recursive _) state = "  "++highlight [Underscore] "recursive"
                                ++ " calls are shown "
                                ++ showOnOff (recursiveMode state)
showState (Qualify _) state   = "  identifier names are shown "
                                ++ highlight [Underscore] "qualified"
                                ++ " "++showOnOff (showQual state)
showState (Equations _) state = "  right-hand-side of "
                                ++ highlight [Underscore] "equations"
                                ++ " are shown "
                                ++ showOnOff (equations state)
showState (Filter _)    state = "  show ("
				++ concat
				     (intersperse ","
                                       (map keep [All,Unique,MostGeneralSoFar
				                 ,MostGeneral]))
                                ++ ") equations"
  where
    keep k | k==filterMode state = highlight [Underscore] (show k)
           | otherwise           = show k
showState (CutOff _) state    = "  expression "
                                ++ highlight [Underscore] "cutoff"
                                ++ " depth = "++ show (cutoffdepth state)
showState (Deeper _) state    = showState (CutOff 0) state
showState (Shallower _) state = showState (CutOff 0) state
showState (Group _)  state    = "  equations per "
                                ++ highlight [Underscore] "group"
                                ++ " = "++ show (equationsPerPage state)

showOnOff :: Bool -> String
showOnOff True  = "[on]"
showOnOff False = "[off]"

----
doCommand :: Cmd -> InteractiveState -> IO()
doCommand (Help s) state = do interactiveHelp (dropWhile isSpace s)
                              interactive state
doCommand Quit state =
    if more state then interactive state{more=False}
    else return ()
doCommand More state =
    if more state then do
        putStr (cursorUp ++ cleareol)
        (newPos,newMore) <- showSomeMore state
	interactive (state {more=newMore,currentPos=newPos})
    else do
        when (currentPos state>0) (putStrLn "No more applications observed.")
        interactive state
doCommand (Info mod) state =
     do let (glob,_,_) = symbolTable state
        putStr (showInfo mod (sortIdents glob) "")
	interactive state
doCommand (InfoC mod) state =
     do let (_,_,constrs) = symbolTable state
        putStr (showInfo mod (sortIdents constrs) "")
	interactive state
doCommand Resize state =
     do (width,_) <- getTerminalSize
        interactive state {screenWidth=width}
doCommand Count state =
     do when (more state)
             (putStrLn "One moment, this may take a while...")
	putStrLn ("Number of (all,unique) matching applications: "
		  ++(show (length (lastObserved state))))
	interactive state

doCommand Status state =
     do mapM_ (\m-> putStrLn (showState m state))
              [ Uneval True, Strings True, Lists True, Recursive True
              , Qualify True, Equations True, Filter All, Group 0, CutOff 0 ]
        interactive state
doCommand (Set mode) state =
     do let state' = setState mode state
        putStrLn (showState mode state')
        interactive state'

doCommand (StartTool tool) state =
     do startExternalTool tool state
	interactive state
doCommand (Source n) state =
     do node <- getEquationNumber n (lastObserved state)
        let sr = getSrcRef (fromJust node)
	when (isJust node && sr /= LowLevel.nil)
             (do System.system (hatView (srcRefFile sr)
                                        (srcRefLine sr)
                                        (srcRefCol sr))
                 return ())
	interactive state
doCommand (Definition n) state =
     do node <- getEquationNumber n (lastObserved state)
        let sr = getDefnRef (fromJust node)
	when (isJust node)
             (do System.system (hatView (defnFile sr)
                                        (defnLine sr)
                                        (defnCol sr))
                 return ())
	interactive state

doCommand (Pattern s) state =
    let (pat, ctx) = parsePat (lexPat s) in
    case pat of
      Left str -> do putStrLn ("Error in pattern: "++str)
                     interactive state
      Right p  -> do
        putStrLn (highlight [Foreground Red] "searching: (^C to interrupt)")
        let fun = funId p
            arity p = case p of
                        SApp _ es -> length es - 1
                        SEquation _ e _ -> arity e
                        _ -> 0
            newObserved = makeObserve state fun ctx p (arity p)
        if isNotFound newObserved || null (fromFound newObserved)
          then do
            putStrLn (cursorUp ++ cleareol ++ "no match found")
            interactive state
	  else do
            putStr (cursorUp ++ cleareol)
            (newPos,newMore) <- showSomeMore
                                   (state { currentPos=0
                                          , showRHS=not (isCon fun)
	 			          , lastObserved=fromFound newObserved})
            interactive (state { lastObserved=fromFound newObserved
                               , more=newMore
                               , showRHS=not (isCon fun)
                               , currentPos=newPos})
doCommand (Location ss) state =
    do putStrLn (highlight [Foreground Red]
                           "searching (source reference): (^C to interrupt)")
       let newObserved = makeObserveSrc ss
       if isNotFound newObserved || null (fromFound newObserved)
         then do
           putStrLn (cursorUp ++ cleareol ++"no match found")
           interactive state
         else do
           putStr (cursorUp ++ cleareol)
           (newPos,newMore) <- showSomeMore
                                  (state { currentPos=0
                                         , showRHS=True
	 			         , lastObserved=fromFound newObserved})
           interactive (state {lastObserved=fromFound newObserved
                              ,more=newMore
                              ,showRHS=True
                              ,currentPos=newPos})
doCommand (Shell cmd) state =
     do err <- system cmd
        when (err/=ExitSuccess) (putStrLn "shell command exited abnormally")
        interactive state
doCommand Unknown state =
     do putStrLn ("Unknown command.  "++shortHelpText)
        interactive state


isCon :: QName -> Bool
isCon qn = con v
  where con (x:xs) = isUpper x || x `elem` ":[,"
        v = case qn of Plain v -> v; Qualified _ v -> v

-- Start up a new window with a new instance of a hat tool.
startExternalTool tool state =
  let file = hatFile state
      checkOK errcode s = when (errcode/=ExitSuccess)
                               (putStrLn ("ERROR: Unable to start "++s++"."))
      esc []        = []
      esc ('"':xs) = '\\': '"': esc xs
      esc ('$':xs) = '\\': '$': esc xs
      esc ('`':xs) = '\\': '`': esc xs
      esc (x:xs)    = x: esc xs
  in case tool of
      Choose -> do
          putStrLn "choose :observe, :detect, or :trail"
      Observe pat -> do
          errcode <- system (spawnObserveCmd++file
                             ++(if null pat then "&"
                                else " \""++esc pat++"\" &"))
          checkOK errcode "hat-observe"
      Trail num -> do
          node <- getEquationNumber num (lastObserved state)
          if isJust node then do
              let id = show (int (fromJust node))
              errcode <- system (spawnTrailCmd++file++" -remote "++id++" &")
              checkOK errcode "hat-trail"
            else putStrLn ("ERROR: Equation "++show num++" not available.")
      Detect num -> do
          node <- getEquationNumber num (lastObserved state)
          if isJust node then do
              let id = show (int (fromJust node))
              errcode <- system (spawnDetectCmd++file++" -remote "++id++" &")
              checkOK errcode "hat-detect"
            else putStrLn ("ERROR: Equation "++show num++" not available.")


interactiveHelp s = do
    putStrLn (if null s then basicHelptext
              else detailedHelp s)
  where
    basicHelptext = "\
\------------ : help <cmd> for more detail ---------------------------------\n\
\ <query>            observe named function (:help query for pattern syntax)\n\
\ <RETURN>           show more observations (if available)\n\
\ :info              see a list of all observable functions\n\
\ :Info              see a list of all observable data constructors\n\
\ :detect <n>        start hat-detect on equation <n>\n\
\ :trail <n>         start hat-trail browser on equation <n>\n\
\ :observe [query]   start hat-observe in a new window with the new query\n\
\ :source <n>        show the source application for equation <n>\n\
\ :Source <n>        show the source definition for identifier in eqn <n>\n\
\ :set               show all current mode settings\n\
\ :set <flag>        change one mode setting\n\
\ :+[n]              short-cut to increase cutoff depth by <n> (default 1)\n\
\ :-[n]              short-cut to decrease cutoff depth by <n> (default 1)\n\
\ :resize            detect new window size for pretty-printing\n\
\ :help <cmd>   :?   show this help text (:help <cmd> for more detail)\n\
\ :quit              quit\n\
\---------------------------------------------------------------------------"

    detailedHelp s
      | ':' == head s         = detailedHelp (tail s)
      | s `isPrefixOf` "query" = "\
\---------------------------------------------------------------------------\n\
\ * A simple function identifier finds all applications of that function.\n\
\     e.g. myfn\n\
\ * To restrict the number of equations, follow the function name with\n\
\   argument or result patterns.\n\
\     e.g. myfn ((:) 1 ((:) 2 _)) \n\
\          myfn \"Hello World!\" (1:(2:_)) = [1,_]\n\
\ * Another way of refining the search is to ask for calls only from a\n\
\   specific enclosing function.\n\
\     e.g. myfn _ (Con 2 _) (1 `Con` 3) in myOtherFn\n\
\\n\
\ The full query syntax is:\n\
\    identifier [pattern]*  ['=' pattern]?  ['in' identifier]?\n\
\ where\n\
\    pattern = '_'                              wildcard\n\
\            | num                              number\n\
\            | ''' char '''                     character\n\
\            | '\"' string '\"'                   string\n\
\            | '[' pattern [',' pattern]* ']'   literal list\n\
\            | Con                              nullary constructor\n\
\            | '(' Con [pattern]* ')'           constructor application\n\
\ Infix functions/constructors take the normal infix syntax.\n\
\---------------------------------------------------------------------------"
      | s `isPrefixOf` "info" || s `isPrefixOf` "Info" = "\
\ :info [module]     see a list of all observable functions\n\
\ :Info [module]     see a list of all observable data constructors\n\
\    Identifiers are listed for the named module, or if no module is named,\n\
\    then for all modules, sorted alphabetically, with occurrence counts.\n\
\    A blue count indicates completed calls, a red count indicates\n\
\    uncompleted calls (unevaluated call counts are not shown)."
      | s `isPrefixOf` "set"  = "\
\ :set               show all current mode settings\n\
\ :set <flag>        change one mode setting\n\
\   <flag> can be: uneval [on|off]      show unevaluated expressions in full\n\
\                  strSugar [on|off]    sugar character strings\n\
\                  listSugar [on|off]   sugar lists\n\
\                  recursive [on|off]   show recursive calls\n\
\                  qualified [on|off]   show all identifiers qualified\n\
\                  equations [on|off]   show rhs of equations\n\
\                  [all|unique]         show all equations or only unique\n\
\                  group <n>            number of equations listed per page\n\
\                  cutoff <n>           cut-off depth for deeply nested exprs"
      | s `isPrefixOf` "detect"  = "\
\ :detect <n>        start hat-detect on equation <n>\n\
\    hat-detect is not currently available.  It is a browser that asks\n\
\    questions about whether certain equations are correct or incorrect,\n\
\    using an automatic method to locate the source of a bug."
      | s `isPrefixOf` "trail"  = "\
\ :trail <n>         start hat-trail browser on equation <n>\n\
\    hat-trail is an interactive browser that permits exploration backwards\n\
\    from a value, expression, or error message, through the function calls\n\
\    that ultimately led to the production of that value.\n\
\    When invoked from within hat-observe, hat-trail begins with the\n\
\    expression on the left of the equation numbered <n> in the list of\n\
\    equations shown by hat-observe."
      | s `isPrefixOf` "observe"  = "\
\ :observe [query]   start hat-observe in a new window with the new query\n\
\    With no argument, a new interactive hat-observe window is started for\n\
\    the same traced program.  Given a query argument, the new window starts\n\
\    with an immediate search for the expression pattern before giving\n\
\    an interactive prompt."
      | otherwise = " topic '"++s++"' has no further help text"

cmdlineHelp = "\
\Usage: hat-observe prog[.hat]\n\
\         An interactive tool to show actual function applications within\n\
\         a traced run of a Haskell program."

{-
cmdlineHelp = "\
\Usage:   hat-observe [-v] [-r] [-xu] identifier [in topidentifier] filename\n\
\Description:\n\
\       prints a table of all applications and results of the given\n\
\       top-level identifier [within the application of topidentifier].\n\
\Options:\n\
\       v: verbose mode. Unevaluated expressions are shown in full.\n\
\       r: recursive mode.  Do not omit recursive function applications.\n\
\       xu: expert's mode for a very fast response. All applications\n\
\           of the identifier are shown, rather than only the most\n\
\           general ones."
-}


data Cmd  = Pattern String | Location [String]
          | More | Info String | InfoC String | Count
          | Help String | Quit | Unknown
          | StartTool Tool | Source Int | Definition Int
          | Status | Set Mode | Shell String | Resize
data Tool = Trail Int | Observe String | Detect Int | Choose
data Mode = Uneval Bool | Strings Bool | Lists Bool | Recursive Bool
          | CutOff Int  | Deeper Int   | Shallower Int | Qualify Bool
          | Group Int   | Filter Keep  | Equations Bool
data Keep = All | Unique | MostGeneralSoFar | MostGeneral deriving Eq

getCommand :: String -> IO Cmd
getCommand prompt = do
    s <- cmdline prompt
    if null s then return More
   -- else if all isDigit s then return (number (StartTool Choose) [s] 0)
      else if head s /= ':' then return (Pattern s)
      else case words (tail s) of
          [] -> return Unknown
          (cmd:ss)
              | cmd `isPrefixOf` "quit" -> return Quit
              | cmd `isPrefixOf` "help" -> return (Help (unwords ss))
              | cmd `isPrefixOf` "location" -> return (Location ss)
              | cmd `isPrefixOf` "detect"   ->
				return (number (StartTool . Detect) ss 0)
              | cmd `isPrefixOf` "trail"    ->
				return (number (StartTool . Trail) ss 0)
              | cmd `isPrefixOf` "observe"  ->
				return (StartTool (Observe (unwords ss)))
              | cmd `isPrefixOf` "source"   -> return (number Source ss 0)
              | cmd `isPrefixOf` "Source"   -> return (number Definition ss 0)
              | cmd `isPrefixOf` "info"     -> return (Info (unwords ss))
              | cmd `isPrefixOf` "Info"     -> return (InfoC (unwords ss))
              | cmd `isPrefixOf` "resize"   -> return Resize
              | cmd `isPrefixOf` "set"      ->
                  case ss of
                      [] -> return Status
                      (m:sss)
                          | m `isPrefixOf` "uneval" ->
					return (onOff Uneval sss)
                          | m `isPrefixOf` "strSugar" ->
					return (onOff Strings sss)
                          | m `isPrefixOf` "listSugar" ->
					return (onOff Lists sss)
                          | m `isPrefixOf` "recursive" ->
					return (onOff Recursive sss)
                          | m `isPrefixOf` "all" ->
					return (Set (Filter All))
                          | m `isPrefixOf` "unique" ->
					return (Set (Filter Unique))
                          | m `isPrefixOf` "generalise" ->
					return (Set (Filter MostGeneralSoFar))
                          | m `isPrefixOf` "mostgeneral" ->
					return (Set (Filter MostGeneral))
                          | m `isPrefixOf` "cutoff" ->
					return (Set (number CutOff sss 10))
                          | m `isPrefixOf` "group" ->
					return (Set (number Group sss 10))
                          | m `isPrefixOf` "qualified" ->
					return (onOff Qualify sss)
                          | m `isPrefixOf` "equations" ->
					return (onOff Equations sss)
                          | m `isPrefixOf` "eqns" ->
					return (onOff Equations sss)
                          | otherwise -> return Unknown
              | head cmd == '?' -> return (Help (unwords (tail cmd:ss)))
              | head cmd == '+' ->
                          return (Set (number Deeper (tail cmd:ss) 1))
              | head cmd == '-' ->
                          return (Set (number Shallower (tail cmd:ss) 1))
              | head cmd == '!' -> return (Shell (unwords (tail cmd:ss)))
              | otherwise  ->  return Unknown

number :: (Int->a) -> [String] -> Int -> a
number cons s def = (maybe (cons def) cons . stringToInt . unwords) s

onOff :: (Bool->Mode) -> [String] -> Cmd
onOff mode s | null s = Set (mode True)
             | otherwise = case head s of "on" -> Set (mode True)
                                          "active" -> Set (mode True)
                                          "off" -> Set (mode False)
                                          "no" -> Set (mode False)
                                          _ -> Unknown

stringToInt :: String -> Maybe Int
stringToInt s = stringToInt' True 0 s
 where
  stringToInt' True _ ('#':r) = stringToInt' True 0 r -- skip "#" at beginning
  stringToInt' True _ (' ':r) = stringToInt' True 0 r -- skip " " at beginning
--stringToInt' False i (' ':r) = Just i
  stringToInt' first i [] = if first then Nothing else  Just i
  stringToInt' _ i (c:r)
      | isDigit c = stringToInt' False (i*10+(digitToInt c)) r
      | otherwise = Nothing

instance Show Keep where
  showsPrec p All		= showString "all"
  showsPrec p Unique 		= showString "unique"
  showsPrec p MostGeneralSoFar	= showString "generalise"
  showsPrec p MostGeneral	= showString "mostgeneral"
