/*
 *  $Id: freq_split.c 22340 2019-07-25 10:23:59Z yeti-dn $
 *  Copyright (C) 2018-2019 David Necas (Yeti).
 *  E-mail: yeti@gwyddion.net.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 */

#include "config.h"
#include <string.h>
#include <gtk/gtk.h>
#include <libgwyddion/gwymacros.h>
#include <libgwyddion/gwymath.h>
#include <libgwyddion/gwythreads.h>
#include <libprocess/arithmetic.h>
#include <libprocess/inttrans.h>
#include <libprocess/stats.h>
#include <libgwydgets/gwycombobox.h>
#include <libgwydgets/gwycheckboxes.h>
#include <libgwydgets/gwyradiobuttons.h>
#include <libgwymodule/gwymodule-process.h>
#include <app/gwymoduleutils.h>
#include <app/gwyapp.h>
#include "preview.h"

#define FSPLIT_RUN_MODES (GWY_RUN_IMMEDIATE | GWY_RUN_INTERACTIVE)

typedef enum {
    FSPLIT_PREVIEW_ORIGINAL  = 0,
    FSPLIT_PREVIEW_LOW_PASS  = 1,
    FSPLIT_PREVIEW_HIGH_PASS = 2,
    FSPLIT_PREVIEW_NTYPES,
} FSplitPreviewType;

typedef enum {
    FSPLIT_BOUNDARY_NONE    = 0,
    FSPLIT_BOUNDARY_LAPLACE = 1,
    FSPLIT_BOUNDARY_SMCONN  = 2,
    FSPLIT_BOUNDARY_MIRROR  = 3,
    FSPLIT_BOUNDARY_NTYPES,
} FSplitBoundaryType;

typedef enum {
    FSPLIT_OUTPUT_LOW_PASS  = (1 << 0),
    FSPLIT_OUTPUT_HIGH_PASS = (1 << 1),
    FSPLIT_OUTPUT_MASK      = (1 << 2) - 1,
} FSplitOutputType;

typedef struct {
    gdouble cutoff;
    gdouble width;
    FSplitBoundaryType boundary;
    FSplitPreviewType preview;
    FSplitOutputType output;
    gboolean update;
} FSplitArgs;

typedef struct {
    FSplitArgs *args;
    GtkWidget *dialogue;
    GtkWidget *view;
    GwySIValueFormat *vf;
    GtkObject *cutoff;
    GtkWidget *cutoff_value;
    GtkObject *width;
    GtkWidget *boundary;
    GSList *preview;
    GtkWidget *update;
    GSList *output;

    GwyContainer *mydata;
    GwyDataField *dfield;
    GwyDataField *extfftre;
    GwyDataField *extfftim;
    gint leftext;
    gint topext;
    guint preview_id;
} FSplitControls;

static gboolean module_register      (void);
static void     freq_split           (GwyContainer *data,
                                      GwyRunType run);
static gboolean fsplit_dialogue      (FSplitArgs *args,
                                      GwyContainer *data,
                                      GwyDataField *dfield,
                                      gint id);
static void     update_real_frequency(GtkWidget *label,
                                      GwySIValueFormat *vf,
                                      gdouble v,
                                      GwyDataField *dfield);
static void     cutoff_changed       (GtkAdjustment *adj,
                                      FSplitControls *controls);
static void     width_changed        (GtkAdjustment *adj,
                                      FSplitControls *controls);
static void     preview_changed      (GtkToggleButton *toggle,
                                      FSplitControls *controls);
static void     boundary_changed     (GtkComboBox *combo,
                                      FSplitControls *controls);
static void     output_changed       (GtkToggleButton *toggle,
                                      FSplitControls *controls);
static void     update_changed       (FSplitControls *controls);
static void     update_controls      (FSplitControls *controls,
                                      FSplitArgs *args);
static void     fsplit_invalidate    (FSplitControls *controls);
static gboolean preview              (gpointer user_data);
static void     extend_and_fft       (GwyDataField *dfield,
                                      GwyDataField **extfftre,
                                      GwyDataField **extfftim,
                                      gint *leftext,
                                      gint *topext,
                                      FSplitBoundaryType boundary);
static void     fsplit_do            (GwyDataField *extfftre,
                                      GwyDataField *extfftim,
                                      gint leftext,
                                      gint topext,
                                      GwyDataField *hipass,
                                      const FSplitArgs *args);
static void     fsplit_load_args     (GwyContainer *container,
                                      FSplitArgs *args);
static void     fsplit_save_args     (GwyContainer *container,
                                      FSplitArgs *args);

static const FSplitArgs fsplit_defaults = {
    0.3, 0.03, FSPLIT_BOUNDARY_NONE,
    FSPLIT_PREVIEW_HIGH_PASS, FSPLIT_OUTPUT_MASK,
    TRUE,
};

static const GwyEnum output_flags[] = {
    { N_("Low-pass image"),  FSPLIT_OUTPUT_LOW_PASS  },
    { N_("High-pass image"), FSPLIT_OUTPUT_HIGH_PASS },
};

enum {
    OUTPUT_NFLAGS = G_N_ELEMENTS(output_flags)
};

static GwyModuleInfo module_info = {
    GWY_MODULE_ABI_VERSION,
    &module_register,
    N_("Splits image into low and high frequency components."),
    "Yeti <yeti@gwyddion.net>",
    "1.1",
    "David Nečas (Yeti)",
    "2018",
};

GWY_MODULE_QUERY2(module_info, freq_split)

static gboolean
module_register(void)
{
    gwy_process_func_register("freq_split",
                              (GwyProcessFunc)&freq_split,
                              N_("/_Level/_Frequency Split..."),
                              GWY_STOCK_FREQUENCY_SPLIT,
                              FSPLIT_RUN_MODES,
                              GWY_MENU_FLAG_DATA,
                              N_("Split into low and high frequencies"));

    return TRUE;
}

static void
freq_split(GwyContainer *data, GwyRunType run)
{
    FSplitArgs args;
    GwyDataField *dfield, *hipass, *lopass, *extfftre, *extfftim;
    gint id, newid, leftext, topext;
    gboolean ok = TRUE;

    g_return_if_fail(run & FSPLIT_RUN_MODES);
    fsplit_load_args(gwy_app_settings_get(), &args);
    gwy_app_data_browser_get_current(GWY_APP_DATA_FIELD, &dfield,
                                     GWY_APP_DATA_FIELD_ID, &id,
                                     0);
    g_return_if_fail(dfield);

    if (run == GWY_RUN_INTERACTIVE)
        ok = fsplit_dialogue(&args, data, dfield, id);

    fsplit_save_args(gwy_app_settings_get(), &args);
    if (!ok)
        return;

    extend_and_fft(dfield, &extfftre, &extfftim, &leftext, &topext,
                   args.boundary);
    hipass = gwy_data_field_new_alike(dfield, FALSE);
    lopass = NULL;
    fsplit_do(extfftre, extfftim, leftext, topext, hipass, &args);
    if (args.output == (FSPLIT_OUTPUT_LOW_PASS | FSPLIT_OUTPUT_HIGH_PASS)) {
        lopass = gwy_data_field_new_alike(dfield, FALSE);
        gwy_data_field_subtract_fields(lopass, dfield, hipass);
    }
    else if (args.output == FSPLIT_OUTPUT_LOW_PASS) {
        lopass = hipass;
        gwy_data_field_subtract_fields(lopass, dfield, hipass);
        hipass = NULL;
    }
    /* Hipass is the default, do not do anything. */

    if (lopass) {
        newid = gwy_app_data_browser_add_data_field(lopass, data, TRUE);
        gwy_app_sync_data_items(data, data, id, newid, FALSE,
                                GWY_DATA_ITEM_GRADIENT,
                                GWY_DATA_ITEM_REAL_SQUARE,
                                0);
        gwy_app_set_data_field_title(data, newid, _("Low-pass"));
        gwy_app_channel_log_add_proc(data, id, newid);
        g_object_unref(lopass);
    }

    if (hipass) {
        newid = gwy_app_data_browser_add_data_field(hipass, data, TRUE);
        gwy_app_sync_data_items(data, data, id, newid, FALSE,
                                GWY_DATA_ITEM_GRADIENT,
                                GWY_DATA_ITEM_REAL_SQUARE,
                                0);
        gwy_app_set_data_field_title(data, newid, _("High-pass"));
        gwy_app_channel_log_add_proc(data, id, newid);
        g_object_unref(hipass);
    }
}

static gboolean
fsplit_dialogue(FSplitArgs *args,
                GwyContainer *data,
                GwyDataField *dfield,
                gint id)
{
    GtkWidget *dialogue, *table, *hbox, *label, *spin, *align;
    FSplitControls controls;
    gint response, row;
    gboolean temp, ok = TRUE;

    gwy_clear(&controls, 1);
    controls.args = args;
    controls.dfield = dfield;

    dialogue = gtk_dialog_new_with_buttons(_("Frequency Split"), NULL, 0,
                                           NULL);
    gtk_dialog_add_action_widget(GTK_DIALOG(dialogue),
                                 gwy_stock_like_button_new(_("_Update"),
                                                           GTK_STOCK_EXECUTE),
                                 RESPONSE_PREVIEW);
    gtk_dialog_add_button(GTK_DIALOG(dialogue), _("_Reset"), RESPONSE_RESET);
    gtk_dialog_add_button(GTK_DIALOG(dialogue),
                          GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL);
    gtk_dialog_add_button(GTK_DIALOG(dialogue),
                          GTK_STOCK_OK, GTK_RESPONSE_OK);
    gtk_dialog_set_default_response(GTK_DIALOG(dialogue), GTK_RESPONSE_OK);
    gwy_help_add_to_proc_dialog(GTK_DIALOG(dialogue), GWY_HELP_DEFAULT);
    controls.dialogue = dialogue;

    hbox = gtk_hbox_new(FALSE, 2);
    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialogue)->vbox), hbox,
                       FALSE, FALSE, 4);

    controls.mydata = gwy_container_new();
    gwy_container_set_object_by_name(controls.mydata, "/0/data", dfield);
    gwy_app_sync_data_items(data, controls.mydata, id, 0, FALSE,
                            GWY_DATA_ITEM_PALETTE,
                            GWY_DATA_ITEM_RANGE,
                            GWY_DATA_ITEM_REAL_SQUARE,
                            0);
    controls.view = create_preview(controls.mydata, 0, PREVIEW_SIZE, FALSE);

    align = gtk_alignment_new(0.0, 0.0, 0.0, 0.0);
    gtk_container_add(GTK_CONTAINER(align), controls.view);
    gtk_box_pack_start(GTK_BOX(hbox), align, FALSE, FALSE, 4);

    table = gtk_table_new(11, 3, FALSE);
    gtk_table_set_row_spacings(GTK_TABLE(table), 2);
    gtk_table_set_col_spacings(GTK_TABLE(table), 6);
    gtk_container_set_border_width(GTK_CONTAINER(table), 4);
    gtk_box_pack_start(GTK_BOX(hbox), table, TRUE, TRUE, 4);
    row = 0;

    controls.vf = gwy_data_field_get_value_format_xy(dfield,
                                                     GWY_SI_UNIT_FORMAT_VFMARKUP,
                                                     NULL);

    /* Cut-off */
    controls.cutoff = gtk_adjustment_new(args->cutoff,
                                         0.0, 0.3, 0.001, 0.1, 0);
    spin = gwy_table_attach_adjbar(table, row, _("C_ut-off:"), NULL,
                                   controls.cutoff, GWY_HSCALE_DEFAULT);
    gtk_spin_button_set_digits(GTK_SPIN_BUTTON(spin), 4);
    g_signal_connect(controls.cutoff, "value-changed",
                     G_CALLBACK(cutoff_changed), &controls);
    row++;

    controls.cutoff_value = gtk_label_new(NULL);
    gtk_misc_set_alignment(GTK_MISC(controls.cutoff_value), 1.0, 0.5);
    gtk_table_attach(GTK_TABLE(table), controls.cutoff_value,
                     1, 2, row, row+1, GTK_FILL, 0, 0, 0);

    label = gtk_label_new(NULL);
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
    gtk_label_set_markup(GTK_LABEL(label), controls.vf->units);
    gtk_table_attach(GTK_TABLE(table), label,
                     2, 3, row, row+1, GTK_FILL, 0, 0, 0);

    gtk_table_set_row_spacing(GTK_TABLE(table), row, 8);
    row++;

    /* Width */
    controls.width = gtk_adjustment_new(args->width,
                                         0.0, 0.2, 0.001, 0.1, 0);
    spin = gwy_table_attach_adjbar(table, row, _("_Edge width:"), NULL,
                                   controls.width, GWY_HSCALE_SQRT);
    gtk_spin_button_set_digits(GTK_SPIN_BUTTON(spin), 4);
    g_signal_connect(controls.width, "value-changed",
                     G_CALLBACK(width_changed), &controls);
    row++;

    /* Boundary */
    gtk_table_set_row_spacing(GTK_TABLE(table), row-1, 8);
    controls.boundary
        = gwy_enum_combo_box_newl(G_CALLBACK(boundary_changed), &controls,
                                  args->boundary,
                                  gwy_sgettext("boundary-handling|None"),
                                  FSPLIT_BOUNDARY_NONE,
                                  _("Laplace"), FSPLIT_BOUNDARY_LAPLACE,
                                  _("Smooth connect"), FSPLIT_BOUNDARY_SMCONN,
                                  _("Mirror"), FSPLIT_BOUNDARY_MIRROR,
                                  NULL);
    gwy_table_attach_adjbar(table, row++, _("_Boundary treatment:"), NULL,
                            GTK_OBJECT(controls.boundary), GWY_HSCALE_WIDGET);

    /* Preview */
    gtk_table_set_row_spacing(GTK_TABLE(table), row-1, 8);
    label = gtk_label_new(_("Display:"));
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
    gtk_table_attach(GTK_TABLE(table), label,
                     0, 2, row, row+1, GTK_FILL, 0, 0, 0);
    row++;

    controls.preview
        = gwy_radio_buttons_createl(G_CALLBACK(preview_changed), &controls,
                                    args->preview,
                                    _("Data"), FSPLIT_PREVIEW_ORIGINAL,
                                    _("High-pass"), FSPLIT_PREVIEW_HIGH_PASS,
                                    _("Low-pass"), FSPLIT_PREVIEW_LOW_PASS,
                                    NULL);
    row = gwy_radio_buttons_attach_to_table(controls.preview, GTK_TABLE(table),
                                            3, row);

    controls.update = gtk_check_button_new_with_mnemonic(_("I_nstant updates"));
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(controls.update),
                                 args->update);
    gtk_dialog_set_response_sensitive(GTK_DIALOG(controls.dialogue),
                                      RESPONSE_PREVIEW, !args->update);
    gtk_table_attach(GTK_TABLE(table), controls.update,
                     0, 2, row, row+1, GTK_FILL, 0, 0, 0);
    g_signal_connect_swapped(controls.update, "toggled",
                             G_CALLBACK(update_changed), &controls);
    row++;

    /* Output */
    gtk_table_set_row_spacing(GTK_TABLE(table), row-1, 8);
    label = gtk_label_new(_("Output type:"));
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
    gtk_table_attach(GTK_TABLE(table), label,
                     0, 2, row, row+1, GTK_FILL, 0, 0, 0);
    row++;

    controls.output
        = gwy_check_boxes_create(output_flags, OUTPUT_NFLAGS,
                                 G_CALLBACK(output_changed), &controls,
                                 args->output);
    row = gwy_check_boxes_attach_to_table(controls.output,
                                          GTK_TABLE(table), 3, row);

    gtk_widget_show_all(dialogue);
    update_real_frequency(controls.cutoff_value, controls.vf, args->cutoff,
                          dfield);
    preview(&controls);

    do {
        response = gtk_dialog_run(GTK_DIALOG(dialogue));
        switch (response) {
            case GTK_RESPONSE_CANCEL:
            case GTK_RESPONSE_DELETE_EVENT:
            gtk_widget_destroy(dialogue);
            case GTK_RESPONSE_NONE:
            ok = FALSE;
            goto finalize;
            break;

            case GTK_RESPONSE_OK:
            break;

            case RESPONSE_RESET:
            temp = args->update;
            *args = fsplit_defaults;
            args->update = temp;
            update_controls(&controls, args);
            fsplit_invalidate(&controls);
            break;

            case RESPONSE_PREVIEW:
            preview(&controls);
            break;

            default:
            g_assert_not_reached();
            break;
        }
    } while (response != GTK_RESPONSE_OK);

    gtk_widget_destroy(dialogue);

finalize:
    if (controls.preview_id)
        g_source_remove(controls.preview_id);
    gwy_si_unit_value_format_free(controls.vf);
    g_object_unref(controls.mydata);
    GWY_OBJECT_UNREF(controls.extfftre);
    GWY_OBJECT_UNREF(controls.extfftim);

    return ok;
}

static void
update_real_frequency(GtkWidget *label, GwySIValueFormat *vf, gdouble v,
                      GwyDataField *dfield)
{
    gdouble dx;
    gchar *s;

    if (v > 0.0) {
        dx = gwy_data_field_get_dx(dfield);
        v = 2.0*dx/v;
        s = g_strdup_printf("%.*f", vf->precision, v/vf->magnitude);
        gtk_label_set_markup(GTK_LABEL(label), s);
        g_free(s);
    }
    else
        gtk_label_set_text(GTK_LABEL(label), "∞");
}

static void
cutoff_changed(GtkAdjustment *adj, FSplitControls *controls)
{
    FSplitArgs *args = controls->args;

    args->cutoff = gtk_adjustment_get_value(adj);
    update_real_frequency(controls->cutoff_value, controls->vf, args->cutoff,
                          controls->dfield);
    if (args->preview != FSPLIT_PREVIEW_ORIGINAL)
        fsplit_invalidate(controls);
}

static void
width_changed(GtkAdjustment *adj, FSplitControls *controls)
{
    FSplitArgs *args = controls->args;

    args->width = gtk_adjustment_get_value(adj);
    if (args->preview != FSPLIT_PREVIEW_ORIGINAL)
        fsplit_invalidate(controls);
}

static void
preview_changed(GtkToggleButton *toggle, FSplitControls *controls)
{
    FSplitArgs *args = controls->args;

    if (!gtk_toggle_button_get_active(toggle))
        return;

    args->preview = gwy_radio_buttons_get_current(controls->preview);
    /* FIXME: We do not need to recalculate anything, just change what we
     * display! */
    fsplit_invalidate(controls);
}

static void
boundary_changed(GtkComboBox *combo, FSplitControls *controls)
{
    FSplitArgs *args = controls->args;

    args->boundary = gwy_enum_combo_box_get_active(combo);
    GWY_OBJECT_UNREF(controls->extfftre);
    GWY_OBJECT_UNREF(controls->extfftim);
    if (args->preview != FSPLIT_PREVIEW_ORIGINAL)
        fsplit_invalidate(controls);
}

static void
output_changed(G_GNUC_UNUSED GtkToggleButton *toggle, FSplitControls *controls)
{
    FSplitArgs *args = controls->args;

    args->output = gwy_check_boxes_get_selected(controls->output);
    /* TODO: Update OK response sensitivity */
}

static void
update_changed(FSplitControls *controls)
{
    FSplitArgs *args = controls->args;

    args->update
        = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(controls->update));
    gtk_dialog_set_response_sensitive(GTK_DIALOG(controls->dialogue),
                                      RESPONSE_PREVIEW, !args->update);
    fsplit_invalidate(controls);
}

static void
update_controls(FSplitControls *controls, FSplitArgs *args)
{
    gtk_adjustment_set_value(GTK_ADJUSTMENT(controls->cutoff), args->cutoff);
    gtk_adjustment_set_value(GTK_ADJUSTMENT(controls->width), args->width);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(controls->update),
                                 args->update);
    gwy_enum_combo_box_set_active(GTK_COMBO_BOX(controls->boundary),
                                  args->boundary);
    gwy_radio_buttons_set_current(controls->preview, args->preview);
    gwy_check_boxes_set_selected(controls->output, args->output);
}

static void
fsplit_invalidate(FSplitControls *controls)
{
    if (controls->preview_id || !controls->args->update)
        return;

    controls->preview_id = g_idle_add(preview, controls);
}

static gboolean
preview(gpointer user_data)
{
    FSplitControls *controls = (FSplitControls*)user_data;
    FSplitArgs *args = controls->args;
    GwyDataField *dfield, *filtered;

    dfield = controls->dfield;
    if (args->preview == FSPLIT_PREVIEW_ORIGINAL) {
        gwy_container_set_object_by_name(controls->mydata, "/0/data", dfield);
        controls->preview_id = 0;
        return FALSE;
    }

    if (!controls->extfftre) {
        g_assert(!controls->extfftim);
        extend_and_fft(dfield, &controls->extfftre, &controls->extfftim,
                       &controls->leftext, &controls->topext, args->boundary);
    }

    filtered = gwy_data_field_new_alike(dfield, FALSE);
    fsplit_do(controls->extfftre, controls->extfftim,
              controls->leftext, controls->topext,
              filtered, args);
    if (args->preview == FSPLIT_PREVIEW_LOW_PASS)
        gwy_data_field_subtract_fields(filtered, dfield, filtered);
    gwy_container_set_object_by_name(controls->mydata, "/0/data", filtered);
    g_object_unref(filtered);

    controls->preview_id = 0;
    return FALSE;
}

static void
extend_one_row(const gdouble *row, guint n,
               gdouble *extrow, guint next)
{
    enum { SMEAR = 6 };
    gint k, i;
    gdouble der0, der1;

    g_return_if_fail(next < 3*n);
    gwy_assign(extrow, row, n);
    /* 0 and 1 in extension row coordinates, not primary row */
    der0 = (2*row[n-1] - row[n-2] - row[n-3])/3;
    der1 = (2*row[0] - row[1] - row[2])/3;
    k = next - n;
    for (i = 0; i < k; i++) {
        gdouble x, y, ww, w;

        y = w = 0.0;
        if (i < SMEAR) {
            ww = 2.0*(SMEAR-1 - i)/SMEAR;
            y += ww*(row[n-1] + der0*(i + 1));
            w += ww;
        }
        if (k-1 - i < SMEAR) {
            ww = 2.0*(i + SMEAR-1 - (k-1))/SMEAR;
            y += ww*(row[0] + der1*(k - i));
            w += ww;
        }
        if (i < n) {
            x = 1.0 - i/(k - 1.0);
            ww = x*x;
            y += ww*row[n-1 - i];
            w += ww;
        }
        if (k-1 - i < n) {
            x = 1.0 - (k-1 - i)/(k - 1.0);
            ww = x*x;
            y += ww*row[k-1 - i];
            w += ww;
        }
        extrow[n + i] = y/w;
    }
}

static GwyDataField*
extend_data_field_smconn(GwyDataField *dfield)
{
    GwyDataField *extfield, *flipped;
    gint xres, yres, extxres, extyres;
    const gdouble *data;
    gdouble *extdata, *buf;
    gint i, j;

    xres = gwy_data_field_get_xres(dfield);
    yres = gwy_data_field_get_yres(dfield);
    extxres = gwy_fft_find_nice_size(4*xres/3);
    extyres = gwy_fft_find_nice_size(4*yres/3);
    if (extxres >= 3*xres || extyres >= 3*extyres) {
        /* This is a silly case.  We just do not want to hit the assertion
         * in extend_one_row(). */
        return gwy_data_field_extend(dfield,
                                     0, 0, extxres - xres, extyres - yres,
                                     GWY_EXTERIOR_FIXED_VALUE,
                                     gwy_data_field_get_avg(dfield), FALSE);
    }

    extfield = gwy_data_field_new(extxres, extyres,
                                  extxres*gwy_data_field_get_dx(dfield),
                                  extyres*gwy_data_field_get_dy(dfield),
                                  FALSE);
    flipped = gwy_data_field_new(extyres, extxres,
                                 extyres*gwy_data_field_get_dy(dfield),
                                 extxres*gwy_data_field_get_dx(dfield),
                                 FALSE);
    data = gwy_data_field_get_data_const(dfield);

    /* Extend rows horizontally. */
    extdata = gwy_data_field_get_data(extfield);
    for (i = 0; i < yres; i++)
        extend_one_row(data + i*xres, xres, extdata + i*extxres, extxres);

    /* Extend columns, including the newly created ones. */
    gwy_data_field_flip_xy(extfield, flipped, FALSE);
    extdata = gwy_data_field_get_data(flipped);
    buf = g_new(gdouble, extyres);
    for (i = 0; i < extxres; i++) {
        extend_one_row(extdata + i*extyres, yres, buf, extyres);
        gwy_assign(extdata + i*extyres, buf, extyres);
    }

    /* Copy it back, extend the remaining rows and use the average to fill
     * the area unreachable by a single extension. */
    gwy_data_field_flip_xy(flipped, extfield, FALSE);
    g_object_unref(flipped);
    extdata = gwy_data_field_get_data(extfield);
    buf = g_renew(gdouble, buf, extxres);
    for (i = yres; i < extyres; i++) {
        extend_one_row(extdata + i*extxres, xres, buf, extxres);
        for (j = xres; j < extxres; j++)
            extdata[i*extxres + j] = 0.5*(extdata[i*extxres + j] + buf[j]);
    }

    return extfield;
}

static GwyDataField*
extend_data_field_mirror(GwyDataField *dfield)
{
    GwyDataField *extfield;
    gint extxres, extyres, xres, yres, i, j;
    const gdouble *data, *srow;
    gdouble *extdata, *trow;

    xres = gwy_data_field_get_xres(dfield);
    yres = gwy_data_field_get_yres(dfield);
    extxres = 2*xres;
    extyres = 2*yres;
    extfield = gwy_data_field_new(extxres, extyres,
                                  extxres*gwy_data_field_get_dx(dfield),
                                  extyres*gwy_data_field_get_dy(dfield),
                                  FALSE);
    data = gwy_data_field_get_data_const(dfield);
    extdata = gwy_data_field_get_data(extfield);

    for (i = 0; i < yres; i++) {
        srow = data + i*xres;
        trow = extdata + i*extxres;

        for (j = 0; j < xres; j++)
            trow[j] = trow[extxres-1 - j] = srow[j];

        srow = trow;
        trow = extdata + (extyres-1 - i)*extxres;
        gwy_assign(trow, srow, extxres);
    }

    return extfield;
}

static void
extend_and_fft(GwyDataField *dfield,
               GwyDataField **extfftre, GwyDataField **extfftim,
               gint *leftext, gint *topext,
               FSplitBoundaryType boundary)
{
    GwyDataField *extfield;
    gint xres, yres, xext, yext;

    xres = gwy_data_field_get_xres(dfield);
    yres = gwy_data_field_get_yres(dfield);

    *leftext = *topext = 0;
    if (boundary == FSPLIT_BOUNDARY_LAPLACE) {
        xext = gwy_fft_find_nice_size(5*xres/3);
        yext = gwy_fft_find_nice_size(5*yres/3);
        extfield = gwy_data_field_extend(dfield,
                                         xext/2, xext - xext/2,
                                         yext/2, yext - yext/2,
                                         GWY_EXTERIOR_LAPLACE, 0.0, FALSE);
        *leftext = xext/2;
        *topext = yext/2;
    }
    else if (boundary == FSPLIT_BOUNDARY_SMCONN) {
        /* The extension is asymmetrical, just to the right and bottom. */
        extfield = extend_data_field_smconn(dfield);
    }
    else if (boundary == FSPLIT_BOUNDARY_MIRROR) {
        /* The extension is asymmetrical, just to the right and bottom. */
        extfield = extend_data_field_mirror(dfield);
    }
    else {
        extfield = g_object_ref(dfield);
    }

    *extfftre = gwy_data_field_new_alike(extfield, FALSE);
    *extfftim = gwy_data_field_new_alike(extfield, FALSE);
    gwy_data_field_2dfft_raw(extfield, NULL, *extfftre, *extfftim,
                             GWY_TRANSFORM_DIRECTION_FORWARD);
    g_object_unref(extfield);
}

static void
filter_frequencies(GwyDataField *refield, GwyDataField *imfield,
                   gdouble cutoff, gdouble width)
{
    gint xres = gwy_data_field_get_xres(refield);
    gint yres = gwy_data_field_get_yres(refield);
    gdouble *re = gwy_data_field_get_data(refield);
    gdouble *im = gwy_data_field_get_data(imfield);
    gint i, j;

#ifdef _OPENMP
#pragma omp parallel for if (gwy_threads_are_enabled()) default(none) \
            private(i,j) \
            shared(re,im,xres,yres,cutoff,width)
#endif
    for (i = 0; i < yres; i++) {
        gdouble fy = 2.0*MIN(i, yres-i)/yres;
        for (j = 0; j < xres; j++) {
            gdouble fx = 2.0*MIN(j, xres-j)/xres;
            gdouble q, f = sqrt(fx*fx + fy*fy);

            if (width > 0.0)
                q = 0.5*(erf((f - cutoff)/width) + 1.0);
            else
                q = (f >= cutoff ? 1.0 : 0.0);

            re[i*xres + j] *= q;
            im[i*xres + j] *= q;
        }
    }
}

static void
fsplit_do(GwyDataField *extfftre, GwyDataField *extfftim,
          gint leftext, gint topext,
          GwyDataField *hipass,
          const FSplitArgs *args)
{
    GwyDataField *tmpre, *tmpim, *fre, *fim;
    gint xres, yres;

    xres = gwy_data_field_get_xres(hipass);
    yres = gwy_data_field_get_yres(hipass);
    tmpre = gwy_data_field_new_alike(extfftre, FALSE);
    tmpim = gwy_data_field_new_alike(extfftre, FALSE);
    fre = gwy_data_field_duplicate(extfftre);
    fim = gwy_data_field_duplicate(extfftim);
    filter_frequencies(fre, fim, args->cutoff, args->width);
    gwy_data_field_2dfft_raw(fre, fim, tmpre, tmpim,
                             GWY_TRANSFORM_DIRECTION_BACKWARD);
    g_object_unref(tmpim);
    g_object_unref(fre);
    g_object_unref(fim);

    gwy_data_field_area_copy(tmpre, hipass, leftext, topext, xres, yres, 0, 0);
    g_object_unref(tmpre);
}

static const gchar boundary_key[] = "/module/freq_split/boundary";
static const gchar cutoff_key[]   = "/module/freq_split/cutoff";
static const gchar output_key[]   = "/module/freq_split/output";
static const gchar preview_key[]  = "/module/freq_split/preview";
static const gchar update_key[]   = "/module/freq_split/update";
static const gchar width_key[]    = "/module/freq_split/width";

static void
fsplit_sanitize_args(FSplitArgs *args)
{
    args->update = !!args->update;
    args->cutoff = CLAMP(args->cutoff, 0.0, 0.3);
    args->width = CLAMP(args->width, 0.0, 0.1);
    args->preview = MIN(args->preview, FSPLIT_PREVIEW_NTYPES-1);
    args->boundary = MIN(args->boundary, FSPLIT_BOUNDARY_NTYPES-1);
    args->output &= FSPLIT_OUTPUT_MASK;
}

static void
fsplit_load_args(GwyContainer *container, FSplitArgs *args)
{
    *args = fsplit_defaults;

    gwy_container_gis_boolean_by_name(container, update_key, &args->update);
    gwy_container_gis_double_by_name(container, cutoff_key, &args->cutoff);
    gwy_container_gis_double_by_name(container, width_key, &args->width);
    gwy_container_gis_enum_by_name(container, preview_key, &args->preview);
    gwy_container_gis_enum_by_name(container, boundary_key, &args->boundary);
    gwy_container_gis_enum_by_name(container, output_key, &args->output);
    fsplit_sanitize_args(args);
}

static void
fsplit_save_args(GwyContainer *container, FSplitArgs *args)
{
    gwy_container_set_boolean_by_name(container, update_key, args->update);
    gwy_container_set_double_by_name(container, cutoff_key, args->cutoff);
    gwy_container_set_double_by_name(container, width_key, args->width);
    gwy_container_set_enum_by_name(container, preview_key, args->preview);
    gwy_container_set_enum_by_name(container, boundary_key, args->boundary);
    gwy_container_set_enum_by_name(container, output_key, args->output);
}

/* vim: set cin et ts=4 sw=4 cino=>1s,e0,n0,f0,{0,}0,^0,\:1s,=0,g1s,h0,t0,+1s,c3,(0,u0 : */
