/*
 *  $Id: linematch.c 21590 2018-11-03 21:08:30Z yeti-dn $
 *  Copyright (C) 2015-2018 David Necas (Yeti).
 *  E-mail: yeti@gwyddion.net.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 */

#include "config.h"
#include <string.h>
#include <gtk/gtk.h>
#include <libgwyddion/gwymacros.h>
#include <libgwyddion/gwymath.h>
#include <libprocess/gwyprocesstypes.h>
#include <libprocess/arithmetic.h>
#include <libprocess/level.h>
#include <libprocess/correct.h>
#include <libprocess/stats.h>
#include <libprocess/linestats.h>
#include <libprocess/gwyprocesstypes.h>
#include <libgwydgets/gwystock.h>
#include <libgwydgets/gwyradiobuttons.h>
#include <libgwydgets/gwycombobox.h>
#include <libgwymodule/gwymodule-process.h>
#include <app/gwymoduleutils.h>
#include <app/gwyapp.h>
#include "preview.h"

#define LINEMATCH_RUN_MODES (GWY_RUN_IMMEDIATE | GWY_RUN_INTERACTIVE)

/* Lower symmetric part indexing */
/* i MUST be greater or equal than j */
#define SLi(a, i, j) a[(i)*((i) + 1)/2 + (j)]

enum {
    MAX_DEGREE = 5,
};

typedef enum {
    LINE_LEVEL_POLY         = 0,
    LINE_LEVEL_MEDIAN       = 1,
    LINE_LEVEL_MEDIAN_DIFF  = 2,
    LINE_LEVEL_MODUS        = 3,
    LINE_LEVEL_MATCH        = 4,
    LINE_LEVEL_TRIMMED_MEAN = 5,
    LINE_LEVEL_TMEAN_DIFF   = 6,
    LINE_LEVEL_NMETHODS
} LineMatchMethod;

typedef struct {
    LineMatchMethod method;
    gint max_degree;
    gboolean do_extract;
    gboolean do_plot;
    gdouble trim_fraction;
    GwyMaskingType masking;
    GwyOrientation direction;
    GwyAppDataId target_graph;
    /* Runtime state */
    GwyDataField *result;
    GwyDataField *bg;
    GwyDataLine *shifts;
} LineMatchArgs;

typedef struct {
    LineMatchArgs *args;
    GtkWidget *dialog;
    GtkObject *max_degree;
    GtkObject *trim_fraction;
    GSList *masking_group;
    GSList *method_group;
    GtkWidget *do_extract;
    GtkWidget *do_plot;
    GtkWidget *target_graph;
    GtkWidget *dataview;
    GtkWidget *direction;
    GwyContainer *data;
    GwyDataField *dfield;
    gboolean in_update;
} LineMatchControls;

static gboolean module_register          (void);
static void     linematch                (GwyContainer *data,
                                          GwyRunType run);
static void     linematch_do             (GwyDataField *mask,
                                          LineMatchArgs *args);
static void     linematch_do_poly        (GwyDataField *dfield,
                                          GwyDataField *mask,
                                          const LineMatchArgs *args);
static void     linematch_do_trimmed_mean(GwyDataField *dfield,
                                          GwyDataField *mask,
                                          const LineMatchArgs *args,
                                          gdouble trim_fraction);
static void     linematch_do_trimmed_diff(GwyDataField *dfield,
                                          GwyDataField *mask,
                                          const LineMatchArgs *args,
                                          gdouble trim_fraction);
static void     linematch_do_modus       (GwyDataField *dfield,
                                          GwyDataField *mask,
                                          const LineMatchArgs *args);
static void     linematch_do_match       (GwyDataField *dfield,
                                          GwyDataField *mask,
                                          const LineMatchArgs *args);
static void     zero_level_row_shifts    (GwyDataLine *shifts);
static gboolean linematch_dialog         (LineMatchArgs *args,
                                          GwyContainer *data,
                                          GwyDataField *dfield,
                                          GwyDataField *mfield,
                                          gint id);
static void     linematch_dialog_update  (LineMatchControls *controls,
                                          LineMatchArgs *args);
static void     degree_changed           (LineMatchControls *controls,
                                          GtkAdjustment *adj);
static void     trim_fraction_changed    (LineMatchControls *controls,
                                          GtkAdjustment *adj);
static void     do_extract_changed       (LineMatchControls *controls,
                                          GtkToggleButton *check);
static void     do_plot_changed          (LineMatchControls *controls,
                                          GtkToggleButton *check);
static void     masking_changed          (GtkToggleButton *button,
                                          LineMatchControls *controls);
static void     method_changed           (GtkToggleButton *button,
                                          LineMatchControls *controls);
static void     direction_changed        (GtkWidget *combo,
                                          LineMatchControls *controls);
static gboolean filter_target_graphs     (GwyContainer *data,
                                          gint id,
                                          gpointer user_data);
static void     target_graph_changed     (LineMatchControls *controls);
static void     update_sensitivity       (LineMatchControls *controls);
static void     update_preview           (LineMatchControls *controls,
                                          LineMatchArgs *args);
static void     load_args                (GwyContainer *container,
                                          LineMatchArgs *args);
static void     save_args                (GwyContainer *container,
                                          LineMatchArgs *args);
static void     sanitize_args            (LineMatchArgs *args);

static const LineMatchArgs linematch_defaults = {
    LINE_LEVEL_MEDIAN,
    1,
    FALSE, FALSE, 0.05,
    GWY_MASK_IGNORE, GWY_ORIENTATION_HORIZONTAL,
    GWY_APP_DATA_ID_NONE,
    /* Runtime state */
    NULL, NULL, NULL,
};

static GwyAppDataId target_id = GWY_APP_DATA_ID_NONE;

static const GwyEnum methods[] = {
    /* Put polynomial last so that is it followed visally by the degree
     * controls. */
    { N_("Median"),                      LINE_LEVEL_MEDIAN,       },
    { N_("Median of differences"),       LINE_LEVEL_MEDIAN_DIFF,  },
    { N_("Modus"),                       LINE_LEVEL_MODUS,        },
    { N_("linematch|Matching"),          LINE_LEVEL_MATCH,        },
    { N_("linematch|Polynomial"),        LINE_LEVEL_POLY,         },
    { N_("Trimmed mean"),                LINE_LEVEL_TRIMMED_MEAN, },
    { N_("Trimmed mean of differences"), LINE_LEVEL_TMEAN_DIFF,   },
};

static GwyModuleInfo module_info = {
    GWY_MODULE_ABI_VERSION,
    &module_register,
    N_("Aligns rows by various methods."),
    "Yeti <yeti@gwyddion.net>",
    "1.8",
    "David Nečas (Yeti)",
    "2015",
};

GWY_MODULE_QUERY2(module_info, linematch)

static gboolean
module_register(void)
{
    gwy_process_func_register("align_rows",
                              (GwyProcessFunc)&linematch,
                              N_("/_Correct Data/_Align Rows..."),
                              GWY_STOCK_LINE_LEVEL,
                              LINEMATCH_RUN_MODES,
                              GWY_MENU_FLAG_DATA,
                              N_("Align rows using various methods"));

    return TRUE;
}

static void
linematch(GwyContainer *data, GwyRunType run)
{
    GwyDataField *dfield, *mfield;
    GQuark quark;
    LineMatchArgs args;
    gboolean ok;
    gint id, newid;
    const gchar *methodname;
    gchar *title;

    g_return_if_fail(run & LINEMATCH_RUN_MODES);
    gwy_app_data_browser_get_current(GWY_APP_DATA_FIELD_KEY, &quark,
                                     GWY_APP_DATA_FIELD, &dfield,
                                     GWY_APP_MASK_FIELD, &mfield,
                                     GWY_APP_DATA_FIELD_ID, &id,
                                     0);
    g_return_if_fail(dfield && quark);

    load_args(gwy_app_settings_get(), &args);
    args.bg = gwy_data_field_new_alike(dfield, FALSE);
    args.shifts = gwy_data_line_new(dfield->yres, dfield->yreal, FALSE);
    gwy_data_field_copy_units_to_data_line(dfield, args.shifts);

    if (run == GWY_RUN_INTERACTIVE) {
        ok = linematch_dialog(&args, data, dfield, mfield, id);
        save_args(gwy_app_settings_get(), &args);
        if (!ok)
            goto end;
        gwy_app_undo_qcheckpointv(data, 1, &quark);
        gwy_data_field_copy(args.result, dfield, FALSE);
    }
    else {
        gwy_app_undo_qcheckpointv(data, 1, &quark);
        args.result = g_object_ref(dfield);
        linematch_do(mfield, &args);
    }

    gwy_data_field_data_changed(dfield);
    gwy_app_channel_log_add(data, id, id, "proc::align_rows",
                            "settings-name", "linematch",
                            NULL);

    methodname = gwy_enum_to_string(args.method,
                                    methods, G_N_ELEMENTS(methods));
    methodname = gwy_sgettext(methodname);
    title = g_strdup_printf("%s (%s)", _("Row background"), methodname);

    if (args.do_extract) {
        newid = gwy_app_data_browser_add_data_field(args.bg, data, TRUE);
        gwy_app_sync_data_items(data, data, id, newid, FALSE,
                                GWY_DATA_ITEM_GRADIENT,
                                0);
        gwy_app_set_data_field_title(data, newid, title);
        gwy_app_channel_log_add(data, id, newid, "proc::align_rows",
                                "settings-name", "linematch",
                                NULL);
    }

    if (args.do_plot) {
        GwyGraphModel *gmodel = gwy_graph_model_new();
        GwyGraphCurveModel *gcmodel = gwy_graph_curve_model_new();

        gwy_graph_curve_model_set_data_from_dataline(gcmodel, args.shifts,
                                                     0, 0);
        g_object_set(gcmodel,
                     "description", title,
                     "mode", GWY_GRAPH_CURVE_LINE,
                     "color", gwy_graph_get_preset_color(0),
                     NULL);
        gwy_graph_model_add_curve(gmodel, gcmodel);
        g_object_unref(gcmodel);

        g_object_set(gmodel,
                     "title", _("Row background"),
                     "axis-label-bottom", _("Vertical position"),
                     "axis-label-left", _("Corrected offset"),
                     NULL);
        gwy_graph_model_set_units_from_data_line(gmodel, args.shifts);
        gwy_app_add_graph_or_curves(gmodel, data, &args.target_graph, 1);
        g_object_unref(gmodel);
    }

    g_free(title);

end:
    GWY_OBJECT_UNREF(args.result);
    GWY_OBJECT_UNREF(args.shifts);
    GWY_OBJECT_UNREF(args.bg);
}

static void
linematch_do(GwyDataField *mask,
             LineMatchArgs *args)
{
    GwyMaskingType masking = args->masking;
    GwyDataField *mymask, *myfield;

    gwy_data_field_copy(args->result, args->bg, TRUE);
    if (args->masking == GWY_MASK_IGNORE)
        mask = NULL;
    if (!mask)
        args->masking = GWY_MASK_IGNORE;

    /* Transpose the fields if necessary. */
    mymask = mask;
    myfield = args->result;
    if (args->direction == GWY_ORIENTATION_VERTICAL) {
        myfield = gwy_data_field_new_alike(args->result, FALSE);
        gwy_data_field_flip_xy(args->result, myfield, FALSE);
        if (mask) {
            mymask = gwy_data_field_new_alike(mask, FALSE);
            gwy_data_field_flip_xy(mask, mymask, FALSE);
        }
    }

    gwy_data_line_resample(args->shifts, gwy_data_field_get_yres(myfield),
                           GWY_INTERPOLATION_NONE);
    gwy_data_line_set_real(args->shifts, gwy_data_field_get_yreal(myfield));

    /* Perform the correction. */
    if (args->method == LINE_LEVEL_POLY) {
        if (args->max_degree == 0)
            linematch_do_trimmed_mean(myfield, mymask, args, 0.0);
        else
            linematch_do_poly(myfield, mymask, args);
    }
    else if (args->method == LINE_LEVEL_MEDIAN)
        linematch_do_trimmed_mean(myfield, mymask, args, 0.5);
    else if (args->method == LINE_LEVEL_MEDIAN_DIFF)
        linematch_do_trimmed_diff(myfield, mymask, args, 0.5);
    else if (args->method == LINE_LEVEL_MODUS)
        linematch_do_modus(myfield, mymask, args);
    else if (args->method == LINE_LEVEL_MATCH)
        linematch_do_match(myfield, mymask, args);
    else if (args->method == LINE_LEVEL_TRIMMED_MEAN)
        linematch_do_trimmed_mean(myfield, mymask, args, args->trim_fraction);
    else if (args->method == LINE_LEVEL_TMEAN_DIFF)
        linematch_do_trimmed_diff(myfield, mymask, args, args->trim_fraction);
    else {
        g_assert_not_reached();
    }

    /* Transpose back if necessary. */
    if (args->direction == GWY_ORIENTATION_VERTICAL) {
        GWY_OBJECT_UNREF(mymask);
        gwy_data_field_flip_xy(myfield, args->result, FALSE);
        g_object_unref(myfield);
    }
    gwy_data_field_subtract_fields(args->bg, args->bg, args->result);

    args->masking = masking;
}

static void
linematch_do_poly(GwyDataField *dfield, GwyDataField *mask,
                  const LineMatchArgs *args)
{
    GwyDataLine *means;
    GwyMaskingType masking;
    gdouble *xpowers, *zxpowers, *matrix;
    gint xres, yres, degree, i, j, k;
    gdouble xc, avg;
    const gdouble *m;
    gdouble *d;

    means = args->shifts;
    masking = args->masking;

    degree = args->max_degree;
    xres = gwy_data_field_get_xres(dfield);
    yres = gwy_data_field_get_yres(dfield);
    xc = 0.5*(xres - 1);
    avg = gwy_data_field_get_avg(dfield);
    d = gwy_data_field_get_data(dfield);

    m = mask ? gwy_data_field_get_data_const(mask) : NULL;

    xpowers = g_new(gdouble, 2*degree+1);
    zxpowers = g_new(gdouble, degree+1);
    matrix = g_new(gdouble, (degree+1)*(degree+2)/2);
    for (i = 0; i < yres; i++) {
        gwy_clear(xpowers, 2*degree+1);
        gwy_clear(zxpowers, degree+1);

        for (j = 0; j < xres; j++) {
            gdouble p = 1.0, x = j - xc;

            if ((masking == GWY_MASK_INCLUDE && m[j] <= 0.0)
                || (masking == GWY_MASK_EXCLUDE && m[j] >= 1.0))
                continue;

            for (k = 0; k <= degree; k++) {
                xpowers[k] += p;
                zxpowers[k] += p*d[j];
                p *= x;
            }
            for (k = degree+1; k <= 2*degree; k++) {
                xpowers[k] += p;
                p *= x;
            }
        }

        /* Solve polynomial coefficients. */
        if (xpowers[0] > degree) {
            for (j = 0; j <= degree; j++) {
                for (k = 0; k <= j; k++)
                    SLi(matrix, j, k) = xpowers[j + k];
            }
            gwy_math_choleski_decompose(degree+1, matrix);
            gwy_math_choleski_solve(degree+1, matrix, zxpowers);
        }
        else
            gwy_clear(zxpowers, degree+1);

        /* Subtract. */
        zxpowers[0] -= avg;
        gwy_data_line_set_val(means, i, zxpowers[0]);
        for (j = 0; j < xres; j++) {
            gdouble p = 1.0, x = j - xc, z = 0.0;

            for (k = 0; k <= degree; k++) {
                z += p*zxpowers[k];
                p *= x;
            }

            d[j] -= z;
        }

        d += xres;
        m = m ? m+xres : NULL;
    }

    g_free(matrix);
    g_free(zxpowers);
    g_free(xpowers);
}

static void
linematch_do_trimmed_mean(GwyDataField *dfield, GwyDataField *mask,
                          const LineMatchArgs *args,
                          gdouble trimfrac)
{
    GwyDataLine *shifts;

    shifts = gwy_data_field_find_row_shifts_trimmed_mean(dfield,
                                                         mask, args->masking,
                                                         trimfrac, 0);
    gwy_data_field_subtract_row_shifts(dfield, shifts);
    gwy_data_line_copy(shifts, args->shifts);
    g_object_unref(shifts);
}

static void
linematch_do_trimmed_diff(GwyDataField *dfield, GwyDataField *mask,
                          const LineMatchArgs *args,
                          gdouble trimfrac)
{
    GwyDataLine *shifts;

    shifts = gwy_data_field_find_row_shifts_trimmed_diff(dfield,
                                                         mask, args->masking,
                                                         trimfrac, 0);
    gwy_data_field_subtract_row_shifts(dfield, shifts);
    gwy_data_line_copy(shifts, args->shifts);
    g_object_unref(shifts);
}

static void
linematch_do_modus(GwyDataField *dfield, GwyDataField *mask,
                   const LineMatchArgs *args)
{
    GwyDataLine *modi, *line;
    GwyMaskingType masking;
    gint xres, yres, i;
    const gdouble *d, *m;
    gdouble modus, total_median;
    gdouble *buf;

    masking = args->masking;
    modi = args->shifts;

    xres = gwy_data_field_get_xres(dfield);
    yres = gwy_data_field_get_yres(dfield);
    total_median = gwy_data_field_area_get_median_mask(dfield, mask, masking,
                                                       0, 0, xres, yres);

    d = gwy_data_field_get_data(dfield);
    m = mask ? gwy_data_field_get_data_const(mask) : NULL;
    line = gwy_data_line_new(xres, 1.0, FALSE);
    buf = gwy_data_line_get_data(line);

    for (i = 0; i < yres; i++) {
        const gdouble *row = d + i*xres, *mrow = m + i*xres;
        gint count = 0, j;

        for (j = 0; j < xres; j++) {
            if ((masking == GWY_MASK_INCLUDE && mrow[j] <= 0.0)
                || (masking == GWY_MASK_EXCLUDE && mrow[j] >= 1.0))
                continue;

            buf[count++] = row[j];
        }

        if (!count)
            modus = total_median;
        else if (count < 9)
            modus = gwy_math_median(count, buf);
        else {
            gint seglen = GWY_ROUND(sqrt(count)), bestj = 0;
            gdouble diff, bestdiff = G_MAXDOUBLE;

            gwy_math_sort(count, buf);
            for (j = 0; j + seglen-1 < count; j++) {
                diff = buf[j + seglen-1] - buf[j];
                if (diff < bestdiff) {
                    bestdiff = diff;
                    bestj = j;
                }
            }
            modus = 0.0;
            count = 0;
            for (j = seglen/3; j < seglen - seglen/3; j++, count++)
                modus += buf[bestj + j];
            modus /= count;
        }

        gwy_data_line_set_val(modi, i, modus);
    }

    zero_level_row_shifts(modi);
    gwy_data_field_subtract_row_shifts(dfield, modi);

    g_object_unref(line);
}

static void
linematch_do_match(GwyDataField *dfield, GwyDataField *mask,
                   const LineMatchArgs *args)
{
    GwyDataLine *shifts;
    GwyMaskingType masking;
    gint xres, yres, i, j;
    gdouble q, wsum, lambda, x;
    const gdouble *d, *m, *a, *b, *ma, *mb;
    gdouble *s, *w;

    shifts = args->shifts;
    masking = args->masking;

    xres = gwy_data_field_get_xres(dfield);
    yres = gwy_data_field_get_yres(dfield);
    d = gwy_data_field_get_data(dfield);
    m = mask ? gwy_data_field_get_data_const(mask) : NULL;
    s = gwy_data_line_get_data(shifts);

    w = g_new(gdouble, xres-1);
    for (i = 1; i < yres; i++) {
        a = d + xres*(i - 1);
        b = d + xres*i;
        ma = m + xres*(i - 1);
        mb = m + xres*i;

        /* Diffnorm */
        wsum = 0.0;
        for (j = 0; j < xres-1; j++) {
            if ((masking == GWY_MASK_INCLUDE && (ma[j] <= 0.0
                                                 || mb[j] <= 0.0))
                || (masking == GWY_MASK_EXCLUDE && (ma[j] >= 1.0
                                                    || mb[j] >= 1.0)))
                continue;

            x = a[j+1] - a[j] - b[j+1] + b[j];
            wsum += fabs(x);
        }
        if (wsum == 0) {
            s[i] = s[i-1];
            continue;
        }
        q = wsum/(xres-1);

        /* Weights */
        wsum = 0.0;
        for (j = 0; j < xres-1; j++) {
            if ((masking == GWY_MASK_INCLUDE && (ma[j] <= 0.0
                                                 || mb[j] <= 0.0))
                || (masking == GWY_MASK_EXCLUDE && (ma[j] >= 1.0
                                                    || mb[j] >= 1.0)))
                continue;

            x = a[j+1] - a[j] - b[j+1] + b[j];
            w[j] = exp(-(x*x/(2.0*q)));
            wsum += w[j];
        }

        /* Correction */
        lambda = (a[0] - b[0])*w[0];
        for (j = 1; j < xres-1; j++) {
            if ((masking == GWY_MASK_INCLUDE && (ma[j] <= 0.0
                                                 || mb[j] <= 0.0))
                || (masking == GWY_MASK_EXCLUDE && (ma[j] >= 1.0
                                                    || mb[j] >= 1.0)))
                continue;

            lambda += (a[j] - b[j])*(w[j-1] + w[j]);
        }
        lambda += (a[xres-1] - b[xres-1])*w[xres-2];
        lambda /= 2.0*wsum;

        gwy_debug("%g %g %g", q, wsum, lambda);

        s[i] = -lambda + s[i-1];
    }
    zero_level_row_shifts(shifts);
    gwy_data_field_subtract_row_shifts(dfield, shifts);

    g_free(w);
}

static void
zero_level_row_shifts(GwyDataLine *shifts)
{
    gwy_data_line_add(shifts, -gwy_data_line_get_avg(shifts));
}

static gboolean
linematch_dialog(LineMatchArgs *args,
                 GwyContainer *data,
                 GwyDataField *dfield,
                 GwyDataField *mfield,
                 gint id)
{
    GtkWidget *dialog, *table, *label, *hbox, *alignment;
    GwyDataChooser *chooser;
    LineMatchControls controls;
    GSList *l;
    gint response, row;

    controls.args = args;
    controls.in_update = TRUE;
    controls.dfield = dfield;
    controls.data = gwy_container_new();

    args->result = gwy_data_field_duplicate(dfield);

    gwy_container_set_object_by_name(controls.data, "/0/data", args->result);
    gwy_app_sync_data_items(data, controls.data, id, 0, FALSE,
                            GWY_DATA_ITEM_REAL_SQUARE,
                            GWY_DATA_ITEM_GRADIENT,
                            GWY_DATA_ITEM_RANGE_TYPE,
                            0);
    if (mfield)
        gwy_container_set_object_by_name(controls.data, "/mask", mfield);

    dialog = gtk_dialog_new_with_buttons(_("Align Rows"),
                                         NULL, 0,
                                         _("_Reset"), RESPONSE_RESET,
                                         GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
                                         GTK_STOCK_OK, GTK_RESPONSE_OK,
                                         NULL);
    gtk_dialog_set_default_response(GTK_DIALOG(dialog), GTK_RESPONSE_OK);
    gwy_help_add_to_proc_dialog(GTK_DIALOG(dialog), GWY_HELP_DEFAULT);
    controls.dialog = dialog;

    hbox = gtk_hbox_new(FALSE, 0);
    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), hbox,
                       FALSE, FALSE, 0);

    controls.dataview = create_preview(controls.data, 0, PREVIEW_SIZE, FALSE);
    alignment = GTK_WIDGET(gtk_alignment_new(0.5, 0, 0, 0));
    gtk_container_add(GTK_CONTAINER(alignment), controls.dataview);
    gtk_box_pack_start(GTK_BOX(hbox), alignment, FALSE, FALSE, 4);

    table = gtk_table_new(9 + LINE_LEVEL_NMETHODS + (mfield ? 4 : 0), 3, FALSE);
    gtk_table_set_row_spacings(GTK_TABLE(table), 2);
    gtk_table_set_col_spacings(GTK_TABLE(table), 6);
    gtk_container_set_border_width(GTK_CONTAINER(table), 4);
    gtk_box_pack_start(GTK_BOX(hbox), table, TRUE, TRUE, 0);
    row = 0;

    label = gtk_label_new(_("Method:"));
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
    gtk_table_attach(GTK_TABLE(table), label,
                     0, 2, row, row+1, GTK_FILL, 0, 0, 0);
    row++;

    controls.method_group
        = gwy_radio_buttons_create(methods, G_N_ELEMENTS(methods),
                                   G_CALLBACK(method_changed),
                                   &controls, args->method);
    for (l = controls.method_group; l; l = g_slist_next(l)) {
        LineMatchMethod method;

        gtk_table_attach(GTK_TABLE(table), GTK_WIDGET(l->data),
                         0, 2, row, row+1, GTK_FILL, 0, 0, 0);
        row++;

        method = gwy_radio_button_get_value(GTK_WIDGET(l->data));
        if (method == LINE_LEVEL_POLY) {
            controls.max_degree = gtk_adjustment_new(args->max_degree,
                                                     0, MAX_DEGREE, 1, 1, 0);
            gwy_table_attach_adjbar(table, row++,
                                    _("_Polynomial degree:"), NULL,
                                    controls.max_degree,
                                    GWY_HSCALE_LINEAR | GWY_HSCALE_SNAP);
            g_signal_connect_swapped(controls.max_degree, "value-changed",
                                     G_CALLBACK(degree_changed), &controls);
        }
        else if (method == LINE_LEVEL_TMEAN_DIFF) {
            controls.trim_fraction = gtk_adjustment_new(args->trim_fraction,
                                                        0.0, 0.5, 0.001, 0.1,
                                                        0);
            gwy_table_attach_adjbar(table, row++, _("_Trim fraction:"), NULL,
                                    controls.trim_fraction, GWY_HSCALE_LINEAR);
            g_signal_connect_swapped(controls.trim_fraction, "value-changed",
                                     G_CALLBACK(trim_fraction_changed),
                                     &controls);
        }
    }

    gtk_table_set_row_spacing(GTK_TABLE(table), row-1, 8);
    label = gwy_label_new_header(_("Options"));
    gtk_table_attach(GTK_TABLE(table), label,
                     0, 2, row, row+1, GTK_FILL, 0, 0, 0);
    row++;

    controls.direction
        = gwy_enum_combo_box_new(gwy_orientation_get_enum(), -1,
                                 G_CALLBACK(direction_changed), &controls,
                                 args->direction, TRUE);
    gwy_table_attach_adjbar(table, row, _("_Direction:"), NULL,
                            GTK_OBJECT(controls.direction),
                            GWY_HSCALE_WIDGET_NO_EXPAND);
    row++;

    controls.do_extract
        = gtk_check_button_new_with_mnemonic(_("E_xtract background"));
    gtk_table_attach(GTK_TABLE(table), controls.do_extract,
                     0, 4, row, row+1, GTK_FILL, 0, 0, 0);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(controls.do_extract),
                                 args->do_extract);
    g_signal_connect_swapped(controls.do_extract, "toggled",
                             G_CALLBACK(do_extract_changed), &controls);
    row++;

    controls.do_plot
        = gtk_check_button_new_with_mnemonic(_("Plot background _graph"));
    gtk_table_attach(GTK_TABLE(table), controls.do_plot,
                     0, 4, row, row+1, GTK_FILL, 0, 0, 0);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(controls.do_plot),
                                 args->do_plot);
    g_signal_connect_swapped(controls.do_plot, "toggled",
                             G_CALLBACK(do_plot_changed), &controls);
    row++;

    controls.target_graph = gwy_data_chooser_new_graphs();
    chooser = GWY_DATA_CHOOSER(controls.target_graph);
    gwy_data_chooser_set_none(chooser, _("New graph"));
    gwy_data_chooser_set_active(chooser, NULL, -1);
    gwy_data_chooser_set_filter(chooser, filter_target_graphs, &controls, NULL);
    gwy_data_chooser_set_active_id(chooser, &args->target_graph);
    gwy_data_chooser_get_active_id(chooser, &args->target_graph);
    gwy_table_attach_adjbar(table, row, _("Target _graph:"), NULL,
                            GTK_OBJECT(controls.target_graph),
                            GWY_HSCALE_WIDGET_NO_EXPAND);
    g_signal_connect_swapped(controls.target_graph, "changed",
                             G_CALLBACK(target_graph_changed), &controls);
    row++;

    if (mfield) {
        gtk_table_set_row_spacing(GTK_TABLE(table), row-1, 8);
        label = gwy_label_new_header(_("Masking Mode"));
        gtk_table_attach(GTK_TABLE(table), label,
                        0, 3, row, row+1, GTK_FILL, 0, 0, 0);
        row++;

        controls.masking_group
            = gwy_radio_buttons_create(gwy_masking_type_get_enum(), -1,
                                       G_CALLBACK(masking_changed),
                                       &controls, args->masking);
        row = gwy_radio_buttons_attach_to_table(controls.masking_group,
                                                GTK_TABLE(table), 3, row);
    }
    else
        controls.masking_group = NULL;

    controls.in_update = FALSE;
    update_sensitivity(&controls);
    update_preview(&controls, args);

    gtk_widget_show_all(dialog);
    do {
        response = gtk_dialog_run(GTK_DIALOG(dialog));
        switch (response) {
            case GTK_RESPONSE_CANCEL:
            case GTK_RESPONSE_DELETE_EVENT:
            gtk_widget_destroy(dialog);
            case GTK_RESPONSE_NONE:
            return FALSE;
            break;

            case GTK_RESPONSE_OK:
            break;

            case RESPONSE_RESET:
            {
                GwyDataField *result = args->result, *bg = args->bg;
                GwyDataLine *shifts = args->shifts;
                *args = linematch_defaults;
                args->result = result;
                args->bg = bg;
                args->shifts = shifts;
                linematch_dialog_update(&controls, args);
            }
            break;

            default:
            g_assert_not_reached();
            break;
        }
    } while (response != GTK_RESPONSE_OK);

    gtk_widget_destroy(dialog);

    return TRUE;
}

static void
linematch_dialog_update(LineMatchControls *controls,
                       LineMatchArgs *args)
{
    controls->in_update = TRUE;
    gwy_radio_buttons_set_current(controls->method_group, args->method);
    gtk_adjustment_set_value(GTK_ADJUSTMENT(controls->max_degree),
                             args->max_degree);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(controls->do_extract),
                                 args->do_extract);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(controls->do_plot),
                                 args->do_plot);
    gwy_enum_combo_box_set_active(GTK_COMBO_BOX(controls->direction),
                                  args->direction);
    if (controls->masking_group)
        gwy_radio_buttons_set_current(controls->masking_group, args->masking);
    controls->in_update = FALSE;
    update_preview(controls, args);
}

static void
degree_changed(LineMatchControls *controls, GtkAdjustment *adj)
{
    LineMatchArgs *args = controls->args;

    args->max_degree = gwy_adjustment_get_int(adj);
    if (controls->in_update)
        return;

    update_preview(controls, controls->args);
}

static void
trim_fraction_changed(LineMatchControls *controls, GtkAdjustment *adj)
{
    LineMatchArgs *args = controls->args;

    args->trim_fraction = gtk_adjustment_get_value(adj);
    if (controls->in_update)
        return;

    update_preview(controls, controls->args);
}

static void
do_extract_changed(LineMatchControls *controls, GtkToggleButton *check)
{
    controls->args->do_extract = gtk_toggle_button_get_active(check);
}

static void
do_plot_changed(LineMatchControls *controls, GtkToggleButton *check)
{
    controls->args->do_plot = gtk_toggle_button_get_active(check);
    update_sensitivity(controls);
}

static void
masking_changed(GtkToggleButton *button, LineMatchControls *controls)
{
    LineMatchArgs *args = controls->args;

    if (!gtk_toggle_button_get_active(button))
        return;

    args->masking = gwy_radio_buttons_get_current(controls->masking_group);
    if (controls->in_update)
        return;

    update_preview(controls, args);
}

static void
method_changed(GtkToggleButton *button, LineMatchControls *controls)
{
    LineMatchArgs *args = controls->args;

    if (!gtk_toggle_button_get_active(button))
        return;

    args->method = gwy_radio_buttons_get_current(controls->method_group);
    update_sensitivity(controls);
    if (controls->in_update)
        return;

    update_preview(controls, args);
}

static void
direction_changed(GtkWidget *combo, LineMatchControls *controls)
{
    LineMatchArgs *args = controls->args;

    args->direction = gwy_enum_combo_box_get_active(GTK_COMBO_BOX(combo));
    if (controls->in_update)
        return;

    update_preview(controls, args);
}

static gboolean
filter_target_graphs(GwyContainer *data, gint id, gpointer user_data)
{
    LineMatchControls *controls = (LineMatchControls*)user_data;
    GwyGraphModel *targetgmodel;
    GQuark quark = gwy_app_get_graph_key_for_id(id);
    GwyDataField *dfield = controls->args->bg;
    GwySIUnit *unitx, *unity;
    gboolean ok;

    g_return_val_if_fail(GWY_IS_DATA_FIELD(dfield), FALSE);
    if (!gwy_container_gis_object(data, quark, (GObject**)&targetgmodel))
        return FALSE;

    g_object_get(targetgmodel, "si-unit-x", &unitx, "si-unit-y", &unity, NULL);
    ok = (gwy_si_unit_equal(unitx, gwy_data_field_get_si_unit_xy(dfield))
          && gwy_si_unit_equal(unity, gwy_data_field_get_si_unit_z(dfield)));
    g_object_unref(unitx);
    g_object_unref(unity);
    return ok;
}

static void
target_graph_changed(LineMatchControls *controls)
{
    GwyDataChooser *chooser = GWY_DATA_CHOOSER(controls->target_graph);
    gwy_data_chooser_get_active_id(chooser, &controls->args->target_graph);
}

static void
update_sensitivity(LineMatchControls *controls)
{
    LineMatchArgs *args = controls->args;

    gwy_table_hscale_set_sensitive(GTK_OBJECT(controls->max_degree),
                                   args->method == LINE_LEVEL_POLY);
    gwy_table_hscale_set_sensitive(GTK_OBJECT(controls->trim_fraction),
                                   args->method == LINE_LEVEL_TRIMMED_MEAN
                                   || args->method == LINE_LEVEL_TMEAN_DIFF);
    gwy_table_hscale_set_sensitive(GTK_OBJECT(controls->target_graph),
                                   args->do_plot);
}

static void
update_preview(LineMatchControls *controls, LineMatchArgs *args)
{
    GwyDataField *source, *mask = NULL;

    source = controls->dfield;
    gwy_container_gis_object_by_name(controls->data, "/mask", &mask);
    gwy_data_field_copy(source, args->result, FALSE);
    linematch_do(mask, args);
    gwy_data_field_data_changed(args->result);
}

static const gchar direction_key[]     = "/module/linematch/direction";
static const gchar do_extract_key[]    = "/module/linematch/do_extract";
static const gchar do_plot_key[]       = "/module/linematch/do_plot";
static const gchar masking_key[]       = "/module/linematch/masking";
static const gchar max_degree_key[]    = "/module/linematch/max_degree";
static const gchar method_key[]        = "/module/linematch/method";
static const gchar trim_fraction_key[] = "/module/linematch/trim_fraction";

static void
sanitize_args(LineMatchArgs *args)
{
    args->max_degree = CLAMP(args->max_degree, 0, MAX_DEGREE);
    args->masking = gwy_enum_sanitize_value(args->masking,
                                            GWY_TYPE_MASKING_TYPE);
    args->method = MIN(args->method, LINE_LEVEL_NMETHODS-1);
    args->direction = gwy_enum_sanitize_value(args->direction,
                                              GWY_TYPE_ORIENTATION);
    args->do_extract = !!args->do_extract;
    args->do_plot = !!args->do_plot;
    args->trim_fraction = CLAMP(args->trim_fraction, 0.0, 0.5);
    gwy_app_data_id_verify_graph(&args->target_graph);
}

static void
load_args(GwyContainer *container,
          LineMatchArgs *args)
{
    *args = linematch_defaults;

    gwy_container_gis_int32_by_name(container, max_degree_key,
                                    &args->max_degree);
    gwy_container_gis_enum_by_name(container, masking_key, &args->masking);
    gwy_container_gis_enum_by_name(container, method_key, &args->method);
    gwy_container_gis_enum_by_name(container, direction_key, &args->direction);
    gwy_container_gis_boolean_by_name(container, do_extract_key,
                                      &args->do_extract);
    gwy_container_gis_boolean_by_name(container, do_plot_key,
                                      &args->do_plot);
    gwy_container_gis_double_by_name(container, trim_fraction_key,
                                     &args->trim_fraction);
    args->target_graph = target_id;
    sanitize_args(args);
}

static void
save_args(GwyContainer *container,
          LineMatchArgs *args)
{
    target_id = args->target_graph;
    gwy_container_set_int32_by_name(container, max_degree_key,
                                    args->max_degree);
    gwy_container_set_enum_by_name(container, masking_key, args->masking);
    gwy_container_set_enum_by_name(container, method_key, args->method);
    gwy_container_set_enum_by_name(container, direction_key, args->direction);
    gwy_container_set_boolean_by_name(container, do_extract_key,
                                      args->do_extract);
    gwy_container_set_boolean_by_name(container, do_plot_key, args->do_plot);
    gwy_container_set_double_by_name(container, trim_fraction_key,
                                     args->trim_fraction);
}

/* vim: set cin et ts=4 sw=4 cino=>1s,e0,n0,f0,{0,}0,^0,\:1s,=0,g1s,h0,t0,+1s,c3,(0,u0 : */
