/*  -*-objc-*-
 *  IconsPanel.m: Implementation of the IconsPanel Class 
 *  of the GNUstep GWorkspace application
 *
 *  Copyright (c) 2001 Enrico Sersale <enrico@imago.ro>
 *  
 *  Author: Enrico Sersale
 *  Date: August 2001
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#import <Foundation/Foundation.h>
#import <AppKit/AppKit.h>
#include "IconsPanel.h"
#include "IconsViewerIcon.h"
#include "Functions.h"
#include "GNUstep.h"

#ifndef max
#define max(a,b) ((a) >= (b) ? (a):(b))
#endif

#ifndef min
#define min(a,b) ((a) <= (b) ? (a):(b))
#endif

extern NSString *GWIconsCellsWidthChangedNotification;

extern NSString *GWorkspaceRecycleOutOperation;

@implementation IconsPanel

- (void)dealloc
{
	[[NSNotificationCenter defaultCenter] removeObserver: self];
  RELEASE (icons);
  RELEASE (currentPath);
  TEST_RELEASE (keysBuff);
  [super dealloc];
}

- (id)initAtPath:(NSString *)path
        delegate:(id)adelegate
{
  self = [super initWithFrame: NSZeroRect];
  if (self) {
    fm = [NSFileManager defaultManager];

    ASSIGN (currentPath, path);
		[self setDelegate: adelegate];
		
		cellsWidth = [delegate getIconViewCellsWidth];
    [self setAutoresizingMask: (NSViewWidthSizable | NSViewHeightSizable)];            
    icons = [[NSMutableArray alloc] initWithCapacity: 1];  
		isDragTarget = NO;
    isShiftClick = NO;
    keysBuff = nil;
  	[self registerForDraggedTypes: [NSArray arrayWithObjects: NSFilenamesPboardType, nil]];

    [[NSNotificationCenter defaultCenter] addObserver: self 
                				  selector: @selector(cellsWidthChanged:) 
                					    name: GWIconsCellsWidthChangedNotification
                					  object: nil];

    [self makeFileIcons];
  }
  return self;
}

- (void)setPath:(NSString *)path
{
  ASSIGN (currentPath, path);
  [self makeFileIcons];
}

- (NSString *)currentPath
{
  return currentPath;
}

- (void)makeFileIcons
{
  NSArray *files, *hiddenFiles;
  NSMutableArray *paths;
	NSUserDefaults *defaults;
	BOOL gsFileBrowserHideDotFiles;
	NSString *h;
  int i, count;
  int stype;

  for (i = 0; i < [icons count]; i++) {
    IconsViewerIcon *icon = [icons objectAtIndex: i];
    [[icon myLabel] removeFromSuperview];
    [icon removeFromSuperview];
  }
  
  [icons removeAllObjects];
  
  files = [fm directoryContentsAtPath: currentPath];
	
	h = [currentPath stringByAppendingPathComponent: @".hidden"];
	h = [NSString stringWithContentsOfFile: h];
  hiddenFiles = [h componentsSeparatedByString: @"\n"];
	
	defaults = [NSUserDefaults standardUserDefaults];
	gsFileBrowserHideDotFiles = [defaults boolForKey: @"GSFileBrowserHideDotFiles"];
	
	if (hiddenFiles != nil  ||  gsFileBrowserHideDotFiles) {	
		NSMutableArray *mutableFiles = AUTORELEASE ([files mutableCopy]);
	
		if (hiddenFiles != nil) {
	    [mutableFiles removeObjectsInArray: hiddenFiles];
	  }
	
		if (gsFileBrowserHideDotFiles) {
	    int j = [mutableFiles count] - 1;
	    
	    while (j >= 0) {
				NSString *file = (NSString *)[mutableFiles objectAtIndex: j];

				if ([file hasPrefix: @"."]) {
		    	[mutableFiles removeObjectAtIndex: j];
		  	}
				j--;
			}
	  }
		
		files = mutableFiles;
	}
	
  count = [files count];
  if (count == 0) {
  	[self resizeWithOldSuperviewSize: [self frame].size]; 
		return;
	}

  paths = [NSMutableArray arrayWithCapacity: 1];

  for (i = 0; i < count; ++i) {
    NSString *s = [currentPath stringByAppendingPathComponent: [files objectAtIndex: i]];
    [paths addObject: s];
  }

	stype = [delegate sortTypeForDirectoryAtPath: currentPath];
  paths = AUTORELEASE ([[paths sortedArrayUsingFunction: (int (*)(id, id, void*))compPaths
                                      context: (void *)stype] mutableCopy]);

  for (i = 0; i < count; ++i) {
    NSString *s = [paths objectAtIndex: i];
    [self addIconWithPaths: [NSArray arrayWithObject: s]];
  }  

  [self resizeWithOldSuperviewSize: [self frame].size]; 
}

- (void)selectIconsWithPaths:(NSArray *)paths
{
  int i;
  
  isShiftClick = YES;

  for (i = 0; i < [icons count]; i++) {
    IconsViewerIcon *icon = [icons objectAtIndex: i];    
    NSString *ipath = [[icon paths] objectAtIndex: 0];

    if ([paths containsObject: ipath] == YES) {
      [icon select];
    }
  }

  isShiftClick = NO;
}

- (void)addIconsWithNames:(NSArray *)names
{
  NSMutableArray *paths;
  int i;

  paths = [NSMutableArray arrayWithCapacity: 1];
  
  for (i = 0; i < [names count]; i++) {
    NSString *s = [currentPath stringByAppendingPathComponent: [names objectAtIndex: i]];
    [paths addObject: s];
  }
  
  [self addIconWithPaths: paths];
  [self resizeWithOldSuperviewSize: [self frame].size];  
}

- (void)removeIconsWithNames:(NSArray *)names
{
  int i, count = [icons count];
  
  for (i = 0; i < count; i++) {
    IconsViewerIcon *icon = [icons objectAtIndex: i];
    NSString *name = [[[icon paths] objectAtIndex: 0] lastPathComponent];

    if ([names containsObject: name] == YES) {
      [[icon myLabel] removeFromSuperview];
      [icon removeFromSuperview];
      [icons removeObject: icon];
      count--;
      i--;
    }
  }

  [self resizeWithOldSuperviewSize: [self frame].size];  
}

- (void)cellsWidthChanged:(NSNotification *)notification
{
  int i;
  
	cellsWidth = [delegate getIconViewCellsWidth];
	  
  for (i = 0; i < [icons count]; i++) {
    [[icons objectAtIndex: i] setLabelWidth];
  }
  
  [self resizeWithOldSuperviewSize: [self frame].size];
}

- (void)selectAllIcons
{
	int i;

	isShiftClick = YES;
  
	for(i = 0; i < [icons count]; i++) {
		IconsViewerIcon *icon = [icons objectAtIndex: i];
    
		if([icon isSelect] == NO) {
			[icon select];
    }
	}
  
	isShiftClick = NO;
}

- (void)selectNextIcon
{
	int i, count = [icons count];
    
	for(i = 0; i < [icons count]; i++) {
		IconsViewerIcon *icon = [icons objectAtIndex: i];
    
		if([icon isSelect]) {
			if (i < (count - 1)) {
				[self unselectOtherIcons: icon];
				icon = [icons objectAtIndex: i + 1];
				[icon select];
				[self scrollRectToVisible: [[icon myLabel] frame]];		
				break;
			} else {
				break;
			}
		}
	} 
}

- (void)selectPrevIcon
{
	int i;
    
	for(i = 0; i < [icons count]; i++) {
		IconsViewerIcon *icon = [icons objectAtIndex: i];
    
		if([icon isSelect]) {
			if (i > 0) {
				[self unselectOtherIcons: icon];
				icon = [icons objectAtIndex: i - 1];
				[icon select];
				[self scrollRectToVisible: [[icon myLabel] frame]];		
				break;
			} else {
				break;
			}
		}
	}
}

- (void)selectIconInNextLine
{
	IconsViewerIcon *icon;
	int i, pos = [icons count];

	for (i = 0; i < [icons count]; i++) {
		icon = [icons objectAtIndex: i];
		if ([icon isSelect]) {
			pos = i + iconsperrow;
			break;
		}
	}
  
	if (pos <= ([icons count] -1)) {
		icon = [icons objectAtIndex: pos];
		[icon select];
		[self scrollRectToVisible: [[icon myLabel] frame]];		
	}
}

- (void)selectIconInPrevLine
{
	IconsViewerIcon *icon;
	int i, pos = -1;
  
	for (i = 0; i < [icons count]; i++) {
		icon = [icons objectAtIndex: i];
		if ([icon isSelect]) {
			pos = i - iconsperrow;
			break;
		}
	}
  
	if (pos >= 0) {
		icon = [icons objectAtIndex: pos];
		[icon select];
		[self scrollRectToVisible: [icon frame]];		
	}
}

- (void)selectIconWithPrefix:(NSString *)prefix
{
	int i;

	for (i = 0; i < [icons count]; i++) {
		IconsViewerIcon *icon = [icons objectAtIndex: i];
		if ([[icon name] hasPrefix: prefix]) {
			[icon select];
			[self scrollRectToVisible: [icon frame]];	
			[self scrollRectToVisible: [[icon myLabel] frame]];	
			break;
		}
	}
}

- (void)addCharToKeysBuff:(char)c
{
  NSTimer *timer;
  
  if (keysBuff == nil) {
    keysBuff = [[NSMutableString alloc] initWithFormat: @"%c", c];    
	  timer = [NSTimer scheduledTimerWithTimeInterval: 2.0 
                  target: self selector: @selector(clearKeysBuff:) 
                                              userInfo: nil repeats: NO];
  } else {
    [keysBuff appendString: [NSString stringWithFormat: @"%c", c]];
  }
  
  [self selectIconWithPrefix: keysBuff];
}

- (void)clearKeysBuff:(id)sender
{
  TEST_RELEASE (keysBuff);
  keysBuff = nil;
}

- (void)resizeWithOldSuperviewSize:(NSSize)oldFrameSize
{
  int wdt, hgt, suphgt, posx, posy, poscount, i;
  NSRect newFrame;
    
  wdt = (int)[[self superview] frame].size.width;
  suphgt = (int)[[self superview] frame].size.height;
  hgt = 55;
  posx = 32;
  
  poscount = 0;
  iconsperrow = 0; 
  
	for (i = 0; i < [icons count]; i++) {    
		if (i > 0) {
			posx += cellsWidth;      
    }
    if (posx >= wdt - 64) {
      posx = 32;      
      hgt += 75;  
      
      if (iconsperrow < poscount) {
        iconsperrow = poscount;
      }
      poscount = 0;    
    }
    poscount++;   
	}
  
  if (iconsperrow == 0) {
    iconsperrow = [icons count];
  }
 
  hgt += 30;
	
  if(hgt > suphgt) {
		posy = hgt - 55;
		newFrame = NSMakeRect(0, [self frame].origin.y, wdt, hgt);
	} else {
		posy = suphgt - 55;
		newFrame = NSMakeRect(0, [self frame].origin.y, wdt, suphgt);
	}
  
	[self setFrame: newFrame];
  
  posx = 32;

	for (i = 0; i < [icons count]; i++) { 
		IconsViewerIcon *icon = [icons objectAtIndex: i];
     
		if (i > 0) {
			posx += cellsWidth;      
    }
    if (posx >= wdt - 64) {
      posx = 32;
      posy -= 75;
    }
		[icon setFrame: NSMakeRect(posx, posy, 64, 52)];
    [icon setNeedsDisplay: YES];
		[self setLabelRectOfIcon: icon];
	}
  
  if ([icons count] > 0) {
    [self scrollRectToVisible: [[icons objectAtIndex: 0] frame]];
  }
  
	[self setNeedsDisplay: YES];
}

- (void)mouseDown:(NSEvent *)theEvent
{
  [[self window] makeFirstResponder: self];
  
	if([theEvent modifierFlags] != 2) {
		isShiftClick = NO;
		[self unselectOtherIcons: nil];
		[delegate setTheSelectedPaths: [NSArray arrayWithObject: currentPath]];
	}
}

- (void)mouseDragged:(NSEvent *)theEvent
{
  NSApplication	*app = [NSApplication sharedApplication];
  NSDate *farAway = [NSDate distantFuture];
  unsigned int eventMask = NSLeftMouseUpMask | NSLeftMouseDraggedMask;
  static NSRect	oldRect; 
  static BOOL	lit = NO;
  NSPoint p, oldp;
  NSEvent	*e;
  NSRect r;
  NSRect rects[[icons count]];
  float x, y, w, h;
  int i, count;
  
#define SETRECTS \
count = [icons count]; \
for (i = 0; i < count; i++) \
rects[i] = [[icons objectAtIndex: i] frame]
  
  oldp = [self convertPoint: [theEvent locationInWindow] fromView: nil];  
  oldRect = NSZeroRect;  
  SETRECTS;
  [self setShiftClick: YES];
  
  [self lockFocus];
  [[NSRunLoop currentRunLoop] limitDateForMode: NSEventTrackingRunLoopMode];

  e = [app nextEventMatchingMask: eventMask
		                   untilDate: farAway
			                    inMode: NSEventTrackingRunLoopMode
			                   dequeue: YES];

  while ([e type] != NSLeftMouseUp) {
    p = [self convertPoint: [e locationInWindow] fromView: nil];

    x = (p.x >= oldp.x) ? oldp.x : p.x;
    y = (p.y >= oldp.y) ? oldp.y : p.y;
    w = max(p.x, oldp.x) - min(p.x, oldp.x);
    w = (w == 0) ? 1 : w;
    h = max(p.y, oldp.y) - min(p.y, oldp.y);
    h = (h == 0) ? 1 : h;

    r = NSMakeRect(x, y, w, h);
       
    if (NSEqualRects(r, oldRect) == NO) {
	    if (lit == YES) {
	      [self setNeedsDisplayInRect: oldRect];
	      lit = NO;
	    }
      
      [[NSColor grayColor] set];
      NSFrameRect(r);	    
      lit = YES;    
	    oldRect = r;
      
      if ([self autoscroll: e] == YES) {
        SETRECTS;
      }
      
      [[self window] flushWindow];
	  }

    for (i = 0; i < count; i++) {
      IconsViewerIcon *icon = [icons objectAtIndex: i];
      
      if (NSContainsRect(r, rects[i]) == YES) {
        if([icon isSelect] == NO) {
          [icon select];
        }
      } else {
        if([icon isSelect] == YES) {
          [icon unselect];
        }
      }
    }
    
    e = [app nextEventMatchingMask: eventMask
		                     untilDate: farAway
			                      inMode: NSEventTrackingRunLoopMode
			                     dequeue: YES];
  }
  
  [self unlockFocus];
  [self setShiftClick: NO];
  [self setNeedsDisplay: YES];
}

- (void)keyDown:(NSEvent *)theEvent 
{
  NSString *characters = [theEvent characters];  
  unichar character;
	NSRect vRect, hiddRect;
	NSPoint p;
	float x, y, w, h;

	if ((characters == nil) || ([characters cStringLength] == 0)) {
		[super keyDown: theEvent];
		return;
	}

	character = [characters characterAtIndex: 0];
	
  switch (character) {

    case NSPageUpFunctionKey:
		  vRect = [self visibleRect];
		  p = vRect.origin;    
		  x = p.x;
		  y = p.y + vRect.size.height;
		  w = vRect.size.width;
		  h = vRect.size.height;
		  hiddRect = NSMakeRect(x, y, w, h);
		  [self scrollRectToVisible: hiddRect];
	    return;

    case NSPageDownFunctionKey:
		  vRect = [self visibleRect];
		  p = vRect.origin;
		  x = p.x;
		  y = p.y - vRect.size.height;
		  w = vRect.size.width;
		  h = vRect.size.height;
		  hiddRect = NSMakeRect(x, y, w, h);
		  [self scrollRectToVisible: hiddRect];
	    return;

    case NSUpArrowFunctionKey:
	    [self selectIconInPrevLine];
      return;

    case NSDownArrowFunctionKey:
	    [self selectIconInNextLine];
      return;

    case NSLeftArrowFunctionKey:
			{
				if ([theEvent modifierFlags] & NSControlKeyMask) {
	      	[super keyDown: theEvent];
	    	} else {
	    		[self selectPrevIcon];
				}
			}
      return;

    case NSRightArrowFunctionKey:
			{
				if ([theEvent modifierFlags] & NSControlKeyMask) {
	      	[super keyDown: theEvent];
	    	} else {
	    		[self selectNextIcon];
				}
			}
	  	return;
      
    case 13:
      [self openCurrentSelection: [self currentSelection] newViewer: NO];      
      break;

    default:    
      break;
  }
  
//  if (isalpha(character)) { 
//    [self addCharToKeysBuff: character]; 
//    return;
//  } 

  [super keyDown: theEvent];
}

- (BOOL)acceptsFirstResponder
{
  return YES;
}

- (BOOL)acceptsFirstMouse:(NSEvent *)theEvent
{
	return YES;
}

//
// <ViewersProtocol>
//
- (void)addIconWithPaths:(NSArray *)iconpaths
{
  int i;
  
  for (i = 0; i < [iconpaths count]; i++) {
    NSArray *arr = [NSArray arrayWithObject: [iconpaths objectAtIndex: i]];
    IconsViewerIcon *icon = [[IconsViewerIcon alloc] initForPaths: arr delegate: self];
    [icons addObject: icon];  
	  [self addSubview: icon];
	  [self addSubview: [icon myLabel]];
    RELEASE (icon);
  }
}

- (void)removeIcon:(id)anIcon
{
  IconsViewerIcon *icon = (IconsViewerIcon *)anIcon;
  [[icon myLabel] removeFromSuperview];
  [icon removeFromSuperview];
  [icons removeObject: icon];
	[self resizeWithOldSuperviewSize: [self frame].size];  
}

- (void)setLabelRectOfIcon:(id)anIcon
{
  IconsViewerIcon *icon;
	NSTextField *label;
	float objwidth, labwidth, labxpos;
  NSRect labelRect;
  
  icon = (IconsViewerIcon *)anIcon;	
	label = [icon myLabel];
  
	objwidth = [icon frame].size.width;
	labwidth = [label frame].size.width;

	if(objwidth > labwidth) 
		labxpos = [icon frame].origin.x + ((objwidth - labwidth) / 2);
	else
		labxpos = [icon frame].origin.x - ((labwidth - objwidth) / 2);

	labelRect = NSMakeRect(labxpos, [icon frame].origin.y - 14, labwidth, 14);
	[label setFrame: labelRect];
	[label setNeedsDisplay: YES];
}

- (void)unselectOtherIcons:(id)anIcon
{
  int i;
  
  if(isShiftClick == YES) {
    return;
  }
  
  for (i = 0; i < [icons count]; i++) {
    IconsViewerIcon *icon = [icons objectAtIndex: i];
    if (icon != anIcon) {  
      if ([icon isSelect] == YES) {  
        [icon unselect];
      }
    }
  }  
}

- (void)setShiftClick:(BOOL)value
{
  isShiftClick = value;
}

- (void)setCurrentSelection:(NSArray *)paths
{
	[delegate setTheSelectedPaths: [self currentSelection]];
}

- (void)openCurrentSelection:(NSArray *)paths newViewer:(BOOL)newv
{
	[delegate openTheCurrentSelection: [self currentSelection] newViewer: newv];
}

- (NSArray *)currentSelection
{
  NSMutableArray *allpaths = [NSMutableArray arrayWithCapacity: 1]; 
  int i;

  for (i = 0; i < [icons count]; i++) {
    IconsViewerIcon *icon = [icons objectAtIndex: i];    
    if ([icon isSelect] == YES) {
      [allpaths addObjectsFromArray: [icon paths]];
    }
  } 
  
  if ([allpaths count] == 0) {
    return nil;
  }
    
  return allpaths;
}

- (NSPoint)locationOfIconWithName:(NSString *)name
{
	int i;

  for (i = 0; i < [icons count]; i++) {
    IconsViewerIcon *icon = [icons objectAtIndex: i]; 
		
    if ([[icon name] isEqualToString: name]) {	
			NSPoint p = [icon frame].origin;
			NSSize s = [icon iconShift];
			return NSMakePoint(p.x + s.width, p.y + s.height);
    }
  } 

	return NSMakePoint(0, 0);
}

- (int)cellsWidth
{
  return cellsWidth;
}

- (void)setDelegate:(id)anObject
{
  delegate = anObject;
}

- (id)delegate
{
  return delegate;
}

@end

@implementation IconsPanel (DraggingDestination)

- (unsigned int)draggingEntered:(id <NSDraggingInfo>)sender
{
	NSPasteboard *pb;
  NSDragOperation sourceDragMask;
	NSData *pbData;
	NSArray *sourcePaths;
	NSString *fromPath;
  NSString *buff;
	int i, count;

	isDragTarget = NO;	
    
 	pb = [sender draggingPasteboard];
  if ([[pb types] indexOfObject: NSFilenamesPboardType] != NSNotFound) {
  
    pbData = [pb dataForType: NSFilenamesPboardType];
    sourcePaths = [NSUnarchiver unarchiveObjectWithData: pbData];  
  
	  count = [sourcePaths count];
	  fromPath = [[sourcePaths objectAtIndex: 0] stringByDeletingLastPathComponent];
  
	  if (count == 0) {
		  return NSDragOperationNone;
    } 
  
	  if ([fm isWritableFileAtPath: currentPath] == NO) {
		  return NSDragOperationNone;
	  }
  
	  if ([currentPath isEqualToString: fromPath]) {
		  return NSDragOperationNone;
    }  
  
	  for (i = 0; i < count; i++) {
		  if ([currentPath isEqualToString: [sourcePaths objectAtIndex: i]]) {
		    return NSDragOperationNone;
		  }
	  }
  
	  buff = [NSString stringWithString: currentPath];
	  while (1) {
		  for (i = 0; i < count; i++) {
			  if ([buff isEqualToString: [sourcePaths objectAtIndex: i]]) {
 		      return NSDragOperationNone;
			  }
		  }
      if ([buff isEqualToString: @"/"] == YES) {
        break;
      }            
		  buff = [buff stringByDeletingLastPathComponent];
	  }
  
    isDragTarget = YES;	

		sourceDragMask = [sender draggingSourceOperationMask];
	
		if (sourceDragMask == NSDragOperationCopy) {
			return NSDragOperationCopy;
		} else if (sourceDragMask == NSDragOperationLink) {
			return NSDragOperationLink;
		} else {
			return NSDragOperationAll;
		}		
    return NSDragOperationAll;
  }
  
	isDragTarget = NO;	
  return NSDragOperationNone;
}

- (unsigned int)draggingUpdated:(id <NSDraggingInfo>)sender
{
  NSDragOperation sourceDragMask;
	
	if (isDragTarget == NO) {
		return NSDragOperationNone;
	}

	sourceDragMask = [sender draggingSourceOperationMask];
	
	if (sourceDragMask == NSDragOperationCopy) {
		return NSDragOperationCopy;
	} else if (sourceDragMask == NSDragOperationLink) {
		return NSDragOperationLink;
	} else {
		return NSDragOperationAll;
	}

	return NSDragOperationNone;
}

- (void)draggingExited:(id <NSDraggingInfo>)sender
{
	isDragTarget = NO;
}

- (BOOL)prepareForDragOperation:(id <NSDraggingInfo>)sender
{
	return isDragTarget;
}

- (BOOL)performDragOperation:(id <NSDraggingInfo>)sender
{
	return YES;
}

- (void)concludeDragOperation:(id <NSDraggingInfo>)sender
{
	NSPasteboard *pb;
  NSDragOperation sourceDragMask;
	NSData *pbData;
	NSArray *sourcePaths;
  NSString *operation, *source;
  NSMutableArray *files;
	NSMutableDictionary *opDict;
	NSString *trashPath;
  int i;

  isDragTarget = NO;
	
	sourceDragMask = [sender draggingSourceOperationMask];  	
  pb = [sender draggingPasteboard];
  pbData = [pb dataForType: NSFilenamesPboardType];
  sourcePaths = [NSUnarchiver unarchiveObjectWithData: pbData];
  
  source = [[sourcePaths objectAtIndex: 0] stringByDeletingLastPathComponent];

	trashPath = [delegate getTrashPath];
	if ([source isEqualToString: trashPath]) {
		operation = GWorkspaceRecycleOutOperation;
	} else {
		if (sourceDragMask == NSDragOperationCopy) {
			operation = NSWorkspaceCopyOperation;
		} else if (sourceDragMask == NSDragOperationLink) {
			operation = NSWorkspaceLinkOperation;
		} else {
			operation = NSWorkspaceMoveOperation;
		}
  }
  
  files = [NSMutableArray arrayWithCapacity: 1];    
  for(i = 0; i < [sourcePaths count]; i++) {    
    [files addObject: [[sourcePaths objectAtIndex: i] lastPathComponent]];
  }  

	opDict = [NSMutableDictionary dictionaryWithCapacity: 4];
	[opDict setObject: operation forKey: @"operation"];
	[opDict setObject: source forKey: @"source"];
	[opDict setObject: currentPath forKey: @"destination"];
	[opDict setObject: files forKey: @"files"];
	
	[delegate performFileOperationWithDictionary: opDict];	
}

@end

//
// IconsViewerIcon Delegate Methods
//

@implementation IconsPanel (IconsViewerIconDelegateMethods)

- (int)getCellsWidth
{
	return cellsWidth;
}

- (void)setLabelFrameOfIcon:(id)aicon
{
	[self setLabelRectOfIcon: aicon];
}

- (void)unselectIconsDifferentFrom:(id)aicon
{
	[self unselectOtherIcons: aicon];
}

- (void)setShiftClickValue:(BOOL)value
{
	[self setShiftClick: value];
}

- (void)setTheCurrentSelection:(id)paths
{
	[self setCurrentSelection: paths];
}

- (NSArray *)getTheCurrentSelection
{
	return [self currentSelection];
}

- (void)openTheCurrentSelection:(id)paths newViewer:(BOOL)newv
{
	[self openCurrentSelection: paths newViewer: newv];
}

- (void)performFileOperationWithDictionary:(id)opdict
{
	[delegate performFileOperationWithDictionary: opdict];
}

- (NSString *)getTrashPath
{
	return [delegate getTrashPath];
}

@end
