/****************************************************************************
** Filename: zip.cpp
** Last updated [dd/mm/yyyy]: 23/10/2006
**
** pkzip 2.0 file compression.
**
** Some of the code has been inspired by other open source projects,
** (mainly Info-Zip).
** Compression and decompression actually uses the zlib library.
**
** Copyright (C) 2006 Angius Fabrizio. All rights reserved.
**
** This file is part of the OSDaB project (http://osdab.sourceforge.net/).
**
** This file may be distributed and/or modified under the terms of the
** GNU General Public License version 2 as published by the Free Software
** Foundation and appearing in the file LICENSE.GPL included in the
** packaging of this file.
**
** This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
** WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
**
** See the file LICENSE.GPL that came with this software distribution or
** visit http://www.gnu.org/copyleft/gpl.html for GPL licensing information.
**
**********************************************************************/

#include "zip.h"
#include "zip_p.h"

// we only use this to seed the random number generator
#include <time.h>

#include <QMap>
#include <QString>
#include <QStringList>
#include <QDir>
#include <QFile>
#include <QDateTime>

// You can remove this #include if you replace the qDebug() statements.
#include <QtDebug>

//! Max version handled by this API
#define ZIP_VERSION 0x14
//! Data descriptor size (signature + crc + compressed and uncompressed size)
#define ZIP_H_DD_SIZE 16
//! Central directory size (fixed fields only)
#define ZIP_CDIR_SIZE 46

//! This macro updates a one-char-only CRC; it's the Info-Zip macro re-adapted
#define CRC32(c, b) crcTable[((int)c^b) & 0xff] ^ (c >> 8)

/*!
	\class Zip zip.h
	\todo methods return a bool and set a lastError class member. same in UnZip class.

	\brief Zip file compression.
*/

/*! \enum Zip::ErrorCode The result of a compression operation.
	\value Zip::Ok No error occurred.
	\value Zip::ZlibInit Failed to init or load the zlib library.
	\value Zip::ZlibError The zlib library returned some error.
	\value Zip::FileExists The file already exists and will not be overwritten.
	\value Zip::OpenFailed Unable to create or open a device.
	\value Zip::NoOpenArchive CreateArchive() has not been called yet.
	\value Zip::FileNotFound File or directory does not exist.
	\value Zip::ReadFailed Reading of a file failed.
	\value Zip::WriteFailed Writing of a file failed.
	\value Zip::SeekFailed Seek failed.
*/

/*! \enum Zip::CompressionLevel Returns the result of a decompression operation.
	\value Zip::Store No compression.
	\value Zip::Deflate1 Deflate compression level 1(lowest compression).
	\value Zip::Deflate1 Deflate compression level 2.
	\value Zip::Deflate1 Deflate compression level 3.
	\value Zip::Deflate1 Deflate compression level 4.
	\value Zip::Deflate1 Deflate compression level 5.
	\value Zip::Deflate1 Deflate compression level 6.
	\value Zip::Deflate1 Deflate compression level 7.
	\value Zip::Deflate1 Deflate compression level 8.
	\value Zip::Deflate1 Deflate compression level 9 (maximum compression).
	\value Zip::AutoCPU Adapt compression level to CPU speed (faster CPU => better compression).
	\value Zip::AutoMIME Adapt compression level to MIME type of the file being compressed.
	\value Zip::AutoFull Use both CPU and MIME type detection.
*/


/************************************************************************
 Public interface
*************************************************************************/

/*!
	Creates a new compressor.
*/
Zip::Zip(QWidget* parent)
: QObject((QObject*)parent)
{
	d = new ZipPrivate(parent);
}

/*!
	Closes any open archive and releases used resources.
*/
Zip::~Zip()
{
	delete d;
}

/*!
	Returns true if there is an open archive.
*/
bool Zip::isOpen() const
{
	return d->device != 0;
}

/*!
	Attempts to create a new Zip archive. If \p overwrite is true and the file 
	already exist it will be overwritten.
	Any open archive will be closed.
 */
Zip::ErrorCode Zip::createArchive(const QString& filename, bool overwrite)
{
	QFile* file = new QFile(filename);

	if (file->exists() && !overwrite) {
		delete file;
		return Zip::FileExists;
	}

	if (!file->open(QIODevice::WriteOnly)) {
		delete file;
		return Zip::OpenFailed;
	}

	Zip::ErrorCode ec = createArchive(file);
	if (ec != Zip::Ok) {
		file->remove();
	}

	return ec;
}

/*!
	Attempts to create a new Zip archive. If there is another open archive this will be closed.
	\warning The class takes ownership of the device!
 */
Zip::ErrorCode Zip::createArchive(QIODevice* device)
{
	if (device == 0)
	{
		qDebug() << tr("Invalid device.");
		return Zip::OpenFailed;
	}

	return d->createArchive(device);
}

/*!
	Recursively adds files contained in \p dir using $root as name for the root folder.
	Stops adding files if some error occurs.
 */
Zip::ErrorCode Zip::addDirectory(const QString& dir, const QString& root, const QString* password, CompressionLevel level)
{
	// this should only happen if we didn't call createArchive() yet
	if (d->device == 0)
		return Zip::NoOpenArchive;

	QDir _dir(dir);
	if (!_dir.exists())
		return Zip::FileNotFound;

	// create directory entry
	ErrorCode ec = d->createEntry(_dir.absolutePath(), root, password, level);
	if (ec != Zip::Ok)
		return ec;

	// remove any trailing separator
	QString root2 = (root.endsWith("/") || root.endsWith("\\")) ? root.left(root.length()-2) : root;

	QFileInfo current(dir);
	if (root2.isEmpty())
		root2 = current.fileName();
	else
	{
		root2.append("/");
		root2.append(current.fileName());
	}

	// create entries for directory contents

	QFileInfoList list = _dir.entryInfoList();

	for (QFileInfoList::Iterator itr=list.begin(); itr!=list.end(); ++itr)
	{
		if (itr->fileName() == "." || itr->fileName() == "..")
			continue;

		if (itr->isDir())
			ec = addDirectory(itr->absoluteFilePath(), root2, password, level);
		else ec = d->createEntry(itr->absoluteFilePath(), root2, password, level);

		if (ec != Zip::Ok)
			return ec;
	}

	return Zip::Ok;
}

/*!
	Closes the archive and writes any pending data.
*/
Zip::ErrorCode Zip::closeArchive()
{
	return d->closeArchive();
}

/*!
	Returns a locale translated error string for a given error code.
*/
QString Zip::formatError(Zip::ErrorCode c)
{
	switch (c)
	{
	case Ok: return tr("ZIP operation completed successfully."); break;
	case ZlibInit: return tr("Failed to initialize or load zlib library."); break;
	case ZlibError: return tr("zlib library error."); break;
	case OpenFailed: return tr("Unable to create or open file."); break;
	case NoOpenArchive: return tr("No archive has been created yet."); break;
	case FileNotFound: return tr("File or directory does not exist."); break;
	case ReadFailed: return tr("File read error."); break;
	case WriteFailed: return tr("File write error."); break;
	case SeekFailed: return tr("File seek error."); break;
	default: ;
	}

	return tr("Unknown error.");
}


/************************************************************************
 Private interface
*************************************************************************/

//! \internal
ZipPrivate::ZipPrivate(QWidget* p)
: QObject((QObject*)p), parent(p), headers(0), device(0), crcTable(0)
{
	// keep an unsigned pointer so we avoid to over bloat the code with casts
	uBuffer = (unsigned char*) buffer1;
	crcTable = (quint32*) get_crc_table();

}

//! \internal
ZipPrivate::~ZipPrivate()
{
	if (device != 0)
		closeArchive();
}

//! \internal
Zip::ErrorCode ZipPrivate::createArchive(QIODevice* dev)
{
	Q_ASSERT(dev != 0);

	if (device != 0)
		closeArchive();

	device = dev;

	if (!device->isOpen())
	{
		if (!device->open(QIODevice::ReadOnly)) {
			delete device;
			device = 0;
			qDebug() << tr("Unable to open device for writing.");
			return Zip::OpenFailed;
		}
	}

	headers = new QMap<QString,ZipEntry>();
	return Zip::Ok;
}

//! \internal Writes a new entry in the zip file.
Zip::ErrorCode ZipPrivate::createEntry(const QString& filename, const QString& root, const QString* password, Zip::CompressionLevel level)
{
	QFileInfo fi(filename);

	if (!fi.exists())
		return Zip::FileNotFound;

	QString lowExt = "";

	//! \todo automatic level detection (cpu, extension & file size)
	if (fi.isDir())
	{
		level = Zip::Store;
	}
	else if (fi.size() < 512)
	{
		level = Zip::Store;
	}
	else
	{
		lowExt = fi.completeSuffix().toLower();

		switch (level)
		{
		case Zip::AutoCPU:
			level = Zip::Deflate5;
			break;
		case Zip::AutoMIME:
			level = detectCompressionByMime(lowExt);
			break;
		case Zip::AutoFull:
			level = detectCompressionByMime(lowExt);
			break;
		default:
			;
		}
	}

	QString entryName = (root.endsWith("/") || root.endsWith("\\")) ? root.left(root.length()-2) : root;
	entryName = root.isEmpty() ? fi.fileName() : QString("%1/%2").arg(entryName).arg(fi.fileName());


	if (fi.isDir())
		entryName.append("/");

	// create header and store it to write a central directory later
	ZipEntry h;

	h.compMethod = (level == Zip::Store) ? 0 : 0x0008;

	h.extraLen[0] = 0;
	h.extraLen[1] = 0;

	h.gpFlag[0] = 0;
	h.gpFlag[1] = 0;

	// set encryption bit and set the data descriptor bit
	// so we can use mod time instead of crc for password check
	if (password != 0)
		h.gpFlag[0] |= 9;

	QDateTime dt = fi.lastModified();
	QDate d = dt.date();
	h.modDate[1] = ((d.year()-1980)<<1)&254;
	h.modDate[1] |= ((d.month()>>3)&1);
	h.modDate[0] = ((d.month()&7)<<5)&224;
	h.modDate[0] |= d.day();

	QTime t = dt.time();
	h.modTime[1] = (t.hour()<<3)&248;
	h.modTime[1] |= ((t.minute()>>3)&7);
	h.modTime[0] = ((t.minute()&7)<<5)&224;
	h.modTime[0] |= t.second()/2;

	h.szUncomp = fi.isDir() ? 0 : fi.size();

	// **** write local file header ****

	// signature
	buffer1[0] = 'P'; buffer1[1] = 'K';
	buffer1[2] = 0x3; buffer1[3] = 0x4;
	// version needed to extract
	buffer1[4] = ZIP_VERSION; buffer1[5] = 0;
	// general purpose flag
	buffer1[6] = h.gpFlag[0]; buffer1[7] = h.gpFlag[1];
	// compression method
	buffer1[8] = h.compMethod&0xFF; buffer1[9] = (h.compMethod>>8)&0xFF;
	// last mod file time
	buffer1[10] = h.modTime[0]; buffer1[11] = h.modTime[1];
	// last mod file date
	buffer1[12] = h.modDate[0]; buffer1[13] = h.modDate[1];
	// skip crc (4bytes) [14,15,16,17]
	// skip compressed size but include evtl. encryption header (4bytes: [18,19,20,21])
	buffer1[18] = buffer1[19] = buffer1[20] = buffer1[21] = 0;
	h.szComp = (password == 0) ? 0 : 12;
	// uncompressed size [22,23,24,25]
	setULong(h.szUncomp, buffer1, 22);
	// filename length
	QByteArray entryNameBytes = entryName.toAscii();
	int sz = entryNameBytes.size();

	buffer1[26] = sz&0xFF; buffer1[27] = (sz>>8)&0xFF;
	// extra field length
	buffer1[28] = buffer1[29] = 0;

	// store offset to write crc and compressed size
	h.zoffset = device->pos();
	quint32 crcOffset = h.zoffset+14;

	if (device->write(buffer1, 30) != 30)
	{
		qDebug() << tr("An error occurred while writing to device");
		return Zip::WriteFailed;
	}

	// write out filename
	if (device->write(entryNameBytes) != sz)
	{
		qDebug() << tr("An error occurred while writing to device");
		return Zip::WriteFailed;
	}

	// encryption keys
	quint32 keys[3] = {0,0,0};

	if (password != 0)
	{
		// **** encryption header ****

		srand(time(NULL) ^ 3141592654UL);
		int randByte;

		// use ^PI and encrypt bytes to ensure better random numbers
		// with poorly implemented rand() --> suggested by Info-Zip :)
		initKeys(*password, keys);
		for (int i=0; i<10; ++i)
		{
			randByte = (rand() >> 7) & 0xff;
			buffer1[i] = decryptByte(keys[2])^randByte;
			updateKeys(keys, randByte);
		}

		// encrypt encryption header
		initKeys(*password, keys);
		for (int i=0; i<10; ++i)
		{
			randByte = decryptByte(keys[2]);
			updateKeys(keys, buffer1[i]);
			buffer1[i] ^= randByte;
		}

		// we don't know the CRC at this time, so we use the modification time
		// as the last two bytes
		randByte = decryptByte(keys[2]);
		updateKeys(keys, h.modTime[0]);
		buffer1[10] ^= randByte;

		randByte = decryptByte(keys[2]);
		updateKeys(keys, h.modTime[1]);
		buffer1[11] ^= randByte;

		// write out encryption header
		if (device->write(buffer1, 12) != 12)
		{
			qDebug() << tr("An error occurred while writing to device");
			return Zip::WriteFailed;
		}
	}

	qint64 written = 0;
	quint32 crc = crc32(0L, Z_NULL, 0);

	if (!fi.isDir())
	{
		QFile file(filename);
		if (!file.open(QIODevice::ReadOnly))
		{
			qDebug() << tr("An error occurred while opening %1").arg(fi.fileName());
			return Zip::OpenFailed;
		}

		// write evtl. compressed and/or encrypted file
		qint64 read = 0;
		qint64 totRead = 0;
		qint64 toRead = file.size();

		if (level == Zip::Store)
		{
			while ( (read = file.read(buffer1, ZIP_READ_BUFFER)) > 0 )
			{
				crc = crc32(crc, uBuffer, read);

				if (password != 0)
					encryptBytes(keys, buffer1, read);

				if ( (written = device->write(buffer1, read)) != read )
				{
					file.close();
					qDebug() << tr("An error occurred while writing to device");
					return Zip::WriteFailed;
				}
			}
		}
		else
		{
			z_stream zstr;

			// initialize zalloc, zfree and opaque before calling the init function
			zstr.zalloc = Z_NULL;
			zstr.zfree = Z_NULL;
			zstr.opaque = Z_NULL;

			int zret;

			// use deflateInit2 with negative windowBits to get raw compression
			if ((zret = deflateInit2_(
					&zstr,
					(int)level,
					Z_DEFLATED,
					-MAX_WBITS,
					8,
					(lowExt == "png") ? Z_RLE : Z_DEFAULT_STRATEGY,
					ZLIB_VERSION,
					sizeof(z_stream)
				)) != Z_OK )
			{
				file.close();
				qDebug() << tr("Could not initialize zlib for compression");
				return Zip::ZlibError;
			}

			qint64 compressed;

			int flush = Z_NO_FLUSH;

			do
			{
				read = file.read(buffer1, ZIP_READ_BUFFER);
				totRead += read;

				if (read == 0)
					break;
				if (read < 0)
				{
					file.close();
					deflateEnd(&zstr);
					qDebug() << tr("Error while reading %1").arg(filename);
					return Zip::ReadFailed;
				}

				crc = crc32(crc, uBuffer, read);

				zstr.next_in = (Bytef*) buffer1;
				zstr.avail_in = (uInt)read;

				// tell zlib if this is the last chunk we want to encode
				// by setting the flush parameter to Z_FINISH
				flush = (totRead == toRead) ? Z_FINISH : Z_NO_FLUSH;

				// run deflate() on input until output buffer not full
				// finish compression if all of source has been read in
        		do
        		{
					zstr.next_out = (Bytef*) buffer2;
					zstr.avail_out = ZIP_READ_BUFFER;

					zret = deflate(&zstr, flush);
					Q_ASSERT(zret != Z_STREAM_ERROR); /* state not clobbered */

					// write compressed data to file and empty buffer
					compressed = ZIP_READ_BUFFER - zstr.avail_out;

					if (password != 0)
						encryptBytes(keys, buffer2, compressed);

					if (device->write(buffer2, compressed) != compressed)
					{
						deflateEnd(&zstr);
						file.close();
						qDebug() << tr("Error while writing %1").arg(filename);
						return Zip::WriteFailed;
					}

					written += compressed;

				} while (zstr.avail_out == 0);

				Q_ASSERT(zstr.avail_in == 0); /* all input will be used */

			} while (flush != Z_FINISH);

			Q_ASSERT(zret == Z_STREAM_END); /* stream will be complete */

			deflateEnd(&zstr);

		} // if (level != STORE)

		file.close();
	}

	// store end of entry offset
	quint32 current = device->pos();

	// update crc and compressed size in local header
	if (!device->seek(crcOffset))
		return Zip::SeekFailed;

	h.crc = fi.isDir() ? 0 : crc;
	h.szComp += written;

	setULong(h.crc, buffer1, 0);
	setULong(h.szComp, buffer1, 4);
	if ( device->write(buffer1, 8) != 8)
	{
		qDebug() << tr("An error occurred while writing to device");
		return Zip::WriteFailed;
	}

	// seek to end of entry
	if (!device->seek(current))
		return Zip::SeekFailed;

	if ((h.gpFlag[0] & 8) == 8)
	{
		// write data descriptor

		// signature: PK\7\8
		buffer1[0] = 'P'; buffer1[1] = 'K';
		buffer1[2] = 0x7; buffer1[3] = 0x8;
		// crc
		setULong(h.crc, buffer1, 4);
		// compressed size
		setULong(h.szComp, buffer1, 8);
		// uncompressed size
		setULong(h.szUncomp, buffer1, 12);

		if (device->write(buffer1, ZIP_H_DD_SIZE) != ZIP_H_DD_SIZE)
		{
			qDebug() << tr("An error occurred while writing to device");
			return Zip::WriteFailed;
		}
	}

	headers->insert(entryName, h);
	return Zip::Ok;
}

//! \internal
int ZipPrivate::decryptByte(quint32 key2) const
{
	quint16 temp = ((quint16)(key2) & 0xffff) | 2;
	return (int)(((temp * (temp ^ 1)) >> 8) & 0xff);
}

//! \internal Writes an quint32 (4 bytes) to a byte array at given offset.
void ZipPrivate::setULong(quint32 v, char* buffer, unsigned int offset)
{
	buffer[offset+3] = ((v >> 24) & 0xFF);
	buffer[offset+2] = ((v >> 16) & 0xFF);
	buffer[offset+1] = ((v >> 8) & 0xFF);
	buffer[offset] = (v & 0xFF);
}

//! \internal Initializes decryption keys using a password.
void ZipPrivate::initKeys(const QString& pwd, quint32* keys) const
{
	keys[0] = 305419896L;
	keys[1] = 591751049L;
	keys[2] = 878082192L;

	QByteArray pwdBytes = pwd.toAscii();
	int sz = pwdBytes.size();
	const char* ascii = pwdBytes.data();

	for (int i=0; i<sz; ++i)
		updateKeys(keys, (int)ascii[i]);
}

//! \internal Updates encryption keys.
void ZipPrivate::updateKeys(quint32* keys, int c) const
{
	keys[0] = CRC32(keys[0], c);
	keys[1] += keys[0] & 0xff;
	keys[1] = keys[1] * 134775813L + 1;
	keys[2] = CRC32(keys[2], ((int)keys[1]) >> 24);
}

//! \internal Encrypts a byte array.
void ZipPrivate::encryptBytes(quint32* keys, char* buffer, qint64 read)
{
	char t;

	for (int i=0; i<(int)read; ++i)
	{
		t = buffer[i];
		buffer[i] ^= decryptByte(keys[2]);
		updateKeys(keys, t);
	}
}

//! \internal Detects the best compression level for a given file extension.
Zip::CompressionLevel ZipPrivate::detectCompressionByMime(const QString& ext)
{
	// files really hard to compress
	if ((ext == "png") ||
		(ext == "jpg") ||
		(ext == "jpeg") ||
		(ext == "mp3") ||
		(ext == "ogg") ||
		(ext == "ogm") ||
		(ext == "avi") ||
		(ext == "mov") ||
		(ext == "rm") ||
		(ext == "ra") ||
		(ext == "zip") ||
		(ext == "rar") ||
		(ext == "bz2") ||
		(ext == "gz") ||
		(ext == "7z") ||
		(ext == "z") ||
		(ext == "jar")
	) return Zip::Store;

	// files slow to compress
	if ((ext == "exe") ||
		(ext == "bin")
		) return Zip::Deflate3;

	return Zip::Deflate9;
}

/*!
	Closes the current archive and writes out pending data.
*/
Zip::ErrorCode ZipPrivate::closeArchive()
{
	// close current archive by writing out central directory
	// and free up resources

	if (device == 0)
		return Zip::Ok;

	const ZipEntry* h;

	unsigned int sz;
	quint32 szCentralDir = 0;
	quint32 offCentralDir = device->pos();

	for (QMap<QString,ZipEntry>::Iterator itr = headers->begin(); itr != headers->end(); ++itr)
	{
		h = &itr.value();

		// signature
		buffer1[0] = 'P'; buffer1[1] = 'K';
		buffer1[2] = 0x1; buffer1[3] = 0x2;
		// version made by  (currently only MS-DOS/FAT - no symlinks or other stuff supported)
		buffer1[4] = buffer1[5] = 0;
		// version needed to extract
		buffer1[6] = ZIP_VERSION; buffer1[7] = 0;
		// general purpose flag
		buffer1[8] = h->gpFlag[0]; buffer1[9] = h->gpFlag[1];
		// compression method
		buffer1[10] = h->compMethod&0xFF; buffer1[11] = (h->compMethod>>8)&0xFF;
		// last mod file time
		buffer1[12] = h->modTime[0]; buffer1[13] = h->modTime[1];
		// last mod file date
		buffer1[14] = h->modDate[0]; buffer1[15] = h->modDate[1];
		// crc (4bytes) [16,17,18,19]
		setULong(h->crc, buffer1, 16);
		// compressed size (4bytes: [20,21,22,23])
		setULong(h->szComp, buffer1, 20);
		// uncompressed size [24,25,26,27]
		setULong(h->szUncomp, buffer1, 24);

		// filename
		QByteArray fileNameBytes = itr.key().toAscii();
		sz = fileNameBytes.size();

		buffer1[28] = sz&0xFF; buffer1[29] = (sz>>8)&0xFF;
		// extra field length
		buffer1[30] = buffer1[31] = 0;
		// file comment length
		buffer1[32] = buffer1[33] = 0;
		// disk number start
		buffer1[34] = buffer1[35] = 0;
		// internal file attributes
		buffer1[36] = buffer1[37] = 0;
		// external file attributes
		buffer1[38] = buffer1[39] = buffer1[40] = buffer1[41] = 0;
		// relative offset of local header [42->45]
		setULong(h->zoffset, buffer1, 42);

		if (device->write(buffer1, 46) != 46)
		{
			device->close();
			//! \todo See if we can detect QFile objects using the Qt Meta Object System
			/*
			if (!device->remove())
				qDebug() << tr("Unable to delete corrupted archive: %1").arg(device->fileName());
			*/
			delete device; device = 0;
			delete headers; headers = 0;
			return Zip::WriteFailed;
		}

		// write out filename
		if ((unsigned int)device->write(fileNameBytes) != sz)
		{
			device->close();
			//! \todo SAME AS ABOVE: See if we can detect QFile objects using the Qt Meta Object System
			/*
			if (!device->remove())
				qDebug() << tr("Unable to delete corrupted archive: %1").arg(device->fileName());
				*/
			delete device; device = 0;
			delete headers; headers = 0;
			return Zip::WriteFailed;
		}

		szCentralDir += (ZIP_CDIR_SIZE + sz);

	} // central dir headers loop


	// write end of central directory

	// signature
	buffer1[0] = 'P'; buffer1[1] = 'K';
	buffer1[2] = 0x5; buffer1[3] = 0x6;
	// number of this disk
	buffer1[4] = buffer1[5] = 0;
	// number of disk with central directory
	buffer1[6] = buffer1[7] = 0;
	// number of entries in this disk
	sz = headers->count();
    buffer1[8] = sz&0xFF; buffer1[9] = (sz>>8)&0xFF;
	// total number of entries
	buffer1[10] = buffer1[8]; buffer1[11] = buffer1[9];
	// size of central directory [12->15]
	setULong(szCentralDir, buffer1, 12);
	// central dir offset [16->19]
	setULong(offCentralDir, buffer1, 16);
	// ZIP file comment length
	buffer1[20] = buffer1[21] = 0;

	if (device->write(buffer1, 22) != 22)
	{
		device->close();
		//! \todo SAME AS ABOVE: See if we can detect QFile objects using the Qt Meta Object System
		/*
		if (!device->remove())
			qDebug() << tr("Unable to delete corrupted archive: %1").arg(device->fileName());
			*/
		delete device; device = 0;
		delete headers; headers = 0;
		return Zip::WriteFailed;
	}

	device->close();

	delete device; device = 0;
	delete headers; headers = 0;

	return Zip::Ok;
}
