#ifndef NETSENDER_H
#define NETSENDER_H

/*
 * Thread to inject packets to a network
 *
 * Copyright (C) 2003  Enrico Zini <enrico@debian.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "Exception.h"
#include "Buffer.h"
#include "nettypes.h"
#include <string>

extern "C" {
	#include <libnet.h>
}

class libnetException : public ConsistencyCheckException
{
protected:
	std::string _libnet_errmsg;

public:
	libnetException(const libnet_t* ln_context, const std::string& context) throw ()
		: ConsistencyCheckException(context)
	{
		const char* msg = libnet_geterror(const_cast<libnet_t*>(ln_context));
		_libnet_errmsg = msg ? msg : "(can't provide an error message: libnet returned null from libnet_geterror!)";
	}
	libnetException(const char* ln_errbuf, const std::string& context) throw ()
		: ConsistencyCheckException(context), _libnet_errmsg(ln_errbuf) {}
	libnetException(const std::string& context) throw ()
		: ConsistencyCheckException(context), _libnet_errmsg() {}
	~libnetException() throw () {}

	virtual const char* type() const throw ()
	{
		return "libnetException";
	}

	virtual std::string desc() const throw ()
	{
		if (_libnet_errmsg.size())
			return _libnet_errmsg + " " + _context;
		else
			return "Unknown libnet error " + _context;
	}
};

/*
 * Injects ethernet packets to a given interface.
 *
 * Every method is thread-safe
 */
class NetSenderImpl;

class NetSender
{
protected:
	NetSenderImpl* impl;
	
public:
	NetSender(const std::string& iface) throw (SystemException, libnetException);
	NetSender(const NetSender& f) throw ();
	~NetSender() throw (SystemException);
	NetSender& operator=(const NetSender& f) throw (SystemException);

	struct libnet_ether_addr* getMACAddress() throw (libnetException);
	libnet_t *getLibnetContext() throw ();

	// Send packet once
	void post(Buffer packet) throw ();
	
	// Send packet at regular intervals.
	// msinterval: interval between sends (in milliseconds)
	// mstimeout: keep sending for mstimeout time (in millisecond)
	void post(Buffer packet, int msinterval, int mstimeout) throw ();
};

// vim:set ts=4 sw=4:
#endif
