/* ----------------------------------------------------------------------------
 * layout_gtk.c
 * functions to create the layout for the GTK theme.
 *
 * Copyright 2002 Matthias Grimm
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version
 * 2 of the License, or (at your option) any later version.
 * ----------------------------------------------------------------------------*/
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <stdarg.h>
#include <gtk/gtk.h>

#include "gettext_macros.h"
#include "themes.h"

/* private prototypes */
int layout_classic_get_popup_type(int popup);

struct layout_classic {
	GtkWidget	*brightness_pixmap1;
	GtkWidget	*brightness_pixmap2;
	GtkWidget	*volume_pixmap1;
	GtkWidget	*volume_pixmap2;
	GtkWidget	*mute_pixmap1;
	GtkWidget	*mute_pixmap2;
	GtkWidget	*mouse_pixmap1;
	GtkWidget	*mouse_pixmap2;
	GtkWidget	*battery_pixmap1;
	GtkWidget	*battery_pixmap2;
	GtkWidget	*sleep_pixmap1;
	GtkWidget	*sleep_pixmap2;
	GtkWidget	*noaudio_pixmap1;
	GtkWidget	*noaudio_pixmap2;
	GtkWidget	*kbdillu_pixmap1;
	GtkWidget	*kbdillu_pixmap2;
	GtkWidget	*cdrom_pixmap1;
	GtkWidget	*cdrom_pixmap2;
	GtkWidget	*mirror_pixmap1;
	GtkWidget	*mirror_pixmap2;

	int		current_popup;    /* popup currently prepared for */
	GtkWidget	*hbox;
	GtkWidget	*left_pixmap;
	GtkWidget	*right_pixmap;
	GtkWidget	*vbox;
	GtkWidget	*title;
	GtkWidget	*progressbar;
	GtkWidget	*message;
};

int
layout_classic_init (struct theme *td)
{
	int err, n;
	static struct layout_classic te;
	struct themeoption themedef[] = {
		{BRIGHTNESS, "image1", theme_image, &te.brightness_pixmap1},
		{BRIGHTNESS, "image2", theme_image, &te.brightness_pixmap2},
		{VOLUME,     "image1", theme_image, &te.volume_pixmap1},
		{VOLUME,     "image2", theme_image, &te.volume_pixmap2},
		{MUTE,       "image1", theme_image, &te.mute_pixmap1},
		{MUTE,       "image2", theme_image, &te.mute_pixmap2},
		{MOUSE,      "image1", theme_image, &te.mouse_pixmap1},
		{MOUSE,      "image2", theme_image, &te.mouse_pixmap2},
		{BATTERY,    "image1", theme_image, &te.battery_pixmap1},
		{BATTERY,    "image2", theme_image, &te.battery_pixmap2},
		{SLEEP,      "image1", theme_image, &te.sleep_pixmap1},
		{SLEEP,      "image2", theme_image, &te.sleep_pixmap2},
		{NOAUDIO,    "image1", theme_image, &te.noaudio_pixmap1},
		{NOAUDIO,    "image2", theme_image, &te.noaudio_pixmap2},
		{KBDILLU,    "image1", theme_image, &te.kbdillu_pixmap1},
		{KBDILLU,    "image2", theme_image, &te.kbdillu_pixmap2},
		{CDROM,      "image1", theme_image, &te.cdrom_pixmap1},
		{CDROM,      "image2", theme_image, &te.cdrom_pixmap2},
		{CRTMIRROR,  "image1", theme_image, &te.mirror_pixmap1},
		{CRTMIRROR,  "image2", theme_image, &te.mirror_pixmap2}
	};

	for (n=0; n < sizeof(themedef) / sizeof(struct themeoption); n++)
		* (GtkWidget**) themedef[n].data = NULL;

	te.progressbar	= NULL;
	te.message	= NULL;

	err = theme_load_data(td, themedef, sizeof(themedef) / sizeof(struct themeoption));
	if (err != 0) {
		layout_classic_exit (td);
		return -1;
	} else {
		td->user_data = (void*) &te;
		return 0;
	}
}

void
layout_classic_exit (struct theme *td)
{
	struct layout_classic *te = (struct layout_classic*) td->user_data;
	if (te != NULL) {
		if (te->brightness_pixmap1 != NULL)
			g_object_unref(te->brightness_pixmap1);
		if (te->brightness_pixmap2 != NULL)
			g_object_unref(te->brightness_pixmap2);
		if (te->volume_pixmap1 != NULL)
			g_object_unref(te->volume_pixmap1);
		if (te->volume_pixmap2 != NULL)
			g_object_unref(te->volume_pixmap2);
		if (te->mute_pixmap1 != NULL)
			g_object_unref(te->mute_pixmap1);
		if (te->mute_pixmap2 != NULL)
			g_object_unref(te->mute_pixmap2);
		if (te->mouse_pixmap1 != NULL)
			g_object_unref(te->mouse_pixmap1);
		if (te->mouse_pixmap2 != NULL)
			g_object_unref(te->mouse_pixmap2);
		if (te->battery_pixmap1 != NULL)
			g_object_unref(te->battery_pixmap1);
		if (te->battery_pixmap2 != NULL)
			g_object_unref(te->battery_pixmap2);
		if (te->sleep_pixmap1 != NULL)
			g_object_unref(te->sleep_pixmap1);
		if (te->sleep_pixmap2 != NULL)
			g_object_unref(te->sleep_pixmap2);
		if (te->noaudio_pixmap1 != NULL)
			g_object_unref(te->noaudio_pixmap1);
		if (te->noaudio_pixmap2 != NULL)
			g_object_unref(te->noaudio_pixmap2);
		if (te->kbdillu_pixmap1 != NULL)
			g_object_unref(te->kbdillu_pixmap1);
		if (te->kbdillu_pixmap2 != NULL)
			g_object_unref(te->kbdillu_pixmap2);
		if (te->cdrom_pixmap1 != NULL)
			g_object_unref(te->cdrom_pixmap1);
		if (te->cdrom_pixmap2 != NULL)
			g_object_unref(te->cdrom_pixmap2);
		if (te->mirror_pixmap1 != NULL)
			g_object_unref(te->mirror_pixmap1);
		if (te->mirror_pixmap2 != NULL)
			g_object_unref(te->mirror_pixmap2);	
		if (te->progressbar != NULL)
			g_object_unref(te->progressbar);
		if (te->message != NULL)
			g_object_unref(te->message);
	}
}

int
layout_classic_attach (struct theme *td, GtkWidget *window)
{
	struct layout_classic *te = (struct layout_classic*) td->user_data;
	GtkWidget *outerframe;
	GtkWidget *innerframe;
	GtkWidget *hbox;
	GtkWidget *vbox;
	GtkWidget *title;
	GtkWidget *progressbar;
	GtkWidget *message;

	outerframe = gtk_frame_new (NULL);
	gtk_container_add (GTK_CONTAINER (window), outerframe);

	innerframe = gtk_frame_new (NULL);
	gtk_container_add (GTK_CONTAINER (outerframe), innerframe);
	gtk_container_set_border_width (GTK_CONTAINER (innerframe), 4);
	gtk_frame_set_shadow_type (GTK_FRAME (innerframe), GTK_SHADOW_IN);

	hbox = gtk_hbox_new (FALSE, 0);
	gtk_container_add (GTK_CONTAINER (innerframe), hbox);
	gtk_container_set_border_width (GTK_CONTAINER (hbox), 10);

	if (te->brightness_pixmap1)
		gtk_box_pack_start (GTK_BOX (hbox), te->brightness_pixmap1, FALSE, FALSE, 0);

	vbox = gtk_vbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (hbox), vbox, TRUE, FALSE, 5);

	title = gtk_label_new (_("Brightness"));
	gtk_box_pack_start (GTK_BOX (vbox), title, FALSE, FALSE, 0);
	gtk_misc_set_padding (GTK_MISC (title), 0, 1);

	progressbar = gtk_progress_bar_new ();
	gtk_widget_ref(progressbar);
	gtk_progress_set_format_string (GTK_PROGRESS (progressbar), "%v");
	gtk_box_pack_start (GTK_BOX (vbox), progressbar, FALSE, FALSE, 0);

	if (te->brightness_pixmap2)
		gtk_box_pack_start (GTK_BOX (hbox), te->brightness_pixmap2, FALSE, FALSE, 0);

	message = gtk_label_new ("dummy");
	gtk_widget_ref(message);
	gtk_label_set_justify (GTK_LABEL(message), GTK_JUSTIFY_CENTER);

	te->current_popup	= POPUP_BRIGHTNESS;
	te->hbox		= hbox;
	te->left_pixmap		= te->brightness_pixmap1;
	te->right_pixmap	= te->brightness_pixmap2;
	te->vbox		= vbox;
	te->title		= title;
	te->progressbar		= progressbar;
	te->message		= message;
	return 0;
}

void
layout_classic_update (struct theme *td, GtkWidget *window, int popup, ...)
{
	struct layout_classic *te = (struct layout_classic*) td->user_data;
	GtkWidget	*pixmap_left = NULL;
	GtkWidget	*pixmap_right = NULL;
	char *title = NULL, message[150];
	int arg0, arg1;
	gdouble frac = 0.0;
	va_list list;
	char *tpmodes[] = {_("notap"), _("tap"), _("drag"), _("lock")};

	va_start(list, popup);

	switch (popup) {
	case POPUP_BRIGHTNESS:
		arg0	     = va_arg(list, int);
		if (arg0 < 0) arg0 = 0;		
		pixmap_left  = te->brightness_pixmap1;
		pixmap_right = te->brightness_pixmap2;
		title	     = _("Brightness");
		frac	     = (gdouble) arg0 / BACKLIGHT_MAX;
		sprintf (message, "%d", arg0);
		break;
	case POPUP_NOBACKLIGHT:
		title		 = _("NOTICE");
		pixmap_left  = te->brightness_pixmap1;
		pixmap_right = te->brightness_pixmap2;
		sprintf(message, _("Sorry, but backlight controller\n is disabled or not installed."));
		break;
	case POPUP_VOLUME:
		arg0		 = va_arg(list, int);
		title		 = _("Volume");
		pixmap_left  = te->volume_pixmap1;
		pixmap_right = te->volume_pixmap2;
		frac = (gdouble) arg0 / VOLUME_MAX;
		sprintf (message, "%d", arg0);
		break;
	case POPUP_MUTE:
		title		 = _("Mute");
		pixmap_left  = te->mute_pixmap1;
		pixmap_right = te->mute_pixmap2;
		frac = 0.0;
		sprintf (message, "0");
		break;
	case POPUP_NOAUDIO:
		title		 = _("NOTICE");
		pixmap_left  = te->noaudio_pixmap1;
		pixmap_right = te->noaudio_pixmap2;
		sprintf(message, _("Sorry, but the audio system is\ndisabled or not installed."));
		break;
	case POPUP_MOUSE:
		pixmap_left  = te->mouse_pixmap1;
		pixmap_right = te->mouse_pixmap2;
		title		 = _("Trackpad");
		sprintf(message, _("Operating mode set to:\n%s."),
			tpmodes[va_arg(list, int) & 3]);
		break;
	case POPUP_BATTERY:
		pixmap_left  = te->battery_pixmap1;
		pixmap_right = te->battery_pixmap2;
		title		 = _("WARNING");

		arg0 = va_arg(list, int);
		arg1 = va_arg(list, int);
		if (arg0 == 4)
			sprintf(message, _("Battery critically low!\nThe system is going down now."));
		else if (arg1 == -1)
			sprintf(message, _("Battery critically low!\nUnknown time till shutdown."));
		else
			sprintf(message, dngettext (PACKAGE,
				"Battery critically low!\nOnly one minute left until shutdown.",
				"Battery low at warn level %d\nOnly %d minutes left until shutdown.",
				arg1), arg0, arg1);
		break;
	case POPUP_SLEEP:
		pixmap_left  = te->sleep_pixmap1;
		pixmap_right = te->sleep_pixmap2;
		title		 = _("WARNING");

		arg0 = va_arg(list, int);
		if (arg0 == 0)
			sprintf(message, _("The Computer will sleep now."));
		else {
			sprintf(message, dngettext (PACKAGE,
				"The computer is going to sleep in %d second.",
				"The computer is going to sleep in %d seconds.",
				arg0), arg0);
		}
		break;
	case POPUP_KBDILLU:
		arg0		= va_arg(list, int);
		pixmap_left  	= te->kbdillu_pixmap1;
		pixmap_right	= te->kbdillu_pixmap2;
		title		= _("Keyboard Illumination");
		frac		= (gdouble) arg0 / KBDILLU_MAX;
		sprintf (message, "%d", arg0);
		break;
	case POPUP_CDROM:
		pixmap_left  = te->cdrom_pixmap1;
		pixmap_right = te->cdrom_pixmap2;
		title		 = _("CDROM");

		arg0 = va_arg(list, int);
		switch (arg0) {
		case 0:
		default:
			sprintf(message, _("CDROM is busy."));
			break;
		case 1:
			sprintf(message, _("Ejecting CDROM ..."));
			break;
		case 2:
			sprintf(message, _("Closing Tray."));
			break;
		}
		break;
	case POPUP_CRTMIRROR:
		pixmap_left  = te->mirror_pixmap1;
		pixmap_right = te->mirror_pixmap2;
		title		 = _("External Video");
		
		arg0 = va_arg(list, int) & 1;
		if (arg0 == 0)
			sprintf(message, _("External Video Output is OFF"));
		else
			sprintf(message, _("External Video Output is ON"));
		break;
	}

	arg0 = layout_classic_get_popup_type(popup);
	arg1 = layout_classic_get_popup_type(te->current_popup);

	if (te->current_popup != popup) { /* popup changed? */
		gtk_label_set_text(GTK_LABEL(te->title), title);

		if (te->left_pixmap)
			gtk_container_remove (GTK_CONTAINER(te->hbox), te->left_pixmap);
		if (pixmap_left) {
			gtk_box_pack_start (GTK_BOX (te->hbox), pixmap_left, FALSE, FALSE, 0);
			gtk_box_reorder_child (GTK_BOX (te->hbox), pixmap_left, 0);
		}
		te->left_pixmap = pixmap_left;

		if (te->right_pixmap)
			gtk_container_remove (GTK_CONTAINER(te->hbox), te->right_pixmap);
		if (pixmap_right)
			gtk_box_pack_start (GTK_BOX (te->hbox), pixmap_right, FALSE, FALSE, 0);
		te->right_pixmap = pixmap_right;

		if (arg0 != arg1) { /* popup type changed? */
			if (arg0 == POPUP_TYPE_PROGRESS) { /* what type is needed now? */
				gtk_container_remove (GTK_CONTAINER(te->vbox), te->message);
				gtk_box_pack_start (GTK_BOX (te->vbox), te->progressbar, FALSE, FALSE, 0);
			} else {
				gtk_container_remove (GTK_CONTAINER(te->vbox), te->progressbar);
				gtk_box_pack_start (GTK_BOX (te->vbox), te->message, FALSE, FALSE, 0);
			}
		}
		te->current_popup = popup;
	}

	switch (arg0) {  /* popup type of current popup */
	case 0:	gtk_progress_bar_set_fraction (GTK_PROGRESS_BAR(te->progressbar), frac);
			gtk_progress_bar_set_text (GTK_PROGRESS_BAR(te->progressbar), message);
			break;
	case 1:	gtk_label_set_text(GTK_LABEL(te->message), message);
			break;
	}
	va_end(list);
}

int
layout_classic_get_popup_type(int popup)
{
	int	popup_types[POPUPS];

	popup_types[POPUP_BRIGHTNESS]	= POPUP_TYPE_PROGRESS;
	popup_types[POPUP_NOBACKLIGHT]	= POPUP_TYPE_MESSAGE;
	popup_types[POPUP_VOLUME]	= POPUP_TYPE_PROGRESS;
	popup_types[POPUP_MUTE]		= POPUP_TYPE_PROGRESS;
	popup_types[POPUP_MOUSE]	= POPUP_TYPE_MESSAGE;
	popup_types[POPUP_BATTERY]	= POPUP_TYPE_MESSAGE;
	popup_types[POPUP_SLEEP]	= POPUP_TYPE_MESSAGE;
	popup_types[POPUP_NOAUDIO]	= POPUP_TYPE_MESSAGE;
	popup_types[POPUP_KBDILLU]	= POPUP_TYPE_PROGRESS;
	popup_types[POPUP_CDROM]	= POPUP_TYPE_MESSAGE;
	popup_types[POPUP_CRTMIRROR] = POPUP_TYPE_MESSAGE;

	return popup_types[popup];
}
