#     G-Tablix, graphical user interface for Tablix
#     Copyright (C) 2004,2005 Bostjan Spetic
# 
#     This program is free software; you can redistribute it and/or modify
#     it under the terms of the GNU General Public License as published by
#     the Free Software Foundation; either version 2 of the License, or
#     (at your option) any later version.
# 
#     This program is distributed in the hope that it will be useful,
#     but WITHOUT ANY WARRANTY; without even the implied warranty of
#     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#     GNU General Public License for more details.
# 
#     You should have received a copy of the GNU General Public License
#     along with this program; if not, write to the Free Software
#     Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# author: Bostjan Spetic, igzebedze@cyberpipe.org
# module for initialization of treeviews
# ran only at the startup...
# 19.3.2005

use strict;

my @checks = ( 'convert','gnuplot','iconv','gzip','killall',
					'gunzip','file','xpvm','html2ps','lp','clusterconf');
my $tablix = "";
my $form;
my %day_names;
my %month_names;

sub get_checks { return \@checks; }
sub get_tablix { return $tablix; }
sub get_form { return $form; }

# ----- init ---------------

sub init_stuff {
	 my $splash = splash();

	 &next_splash($splash,'i18n');
		 &load_settings;
		 &i18n;
		 #&get_locale_days;

	 &next_splash($splash,gettext('Building GUI'));
		 my $libdir = &get_libs;
		 $form = Gtk2::GladeXML->new("$libdir/gtablix/gtablix.glade",undef,'gtablix');
		 $form->signal_autoconnect_from_package('main');	
		 $splash->set_keep_above($form->get_widget('window_main'));
		 &prep_tvs;					# create simplelists
		 &prep_pics;
	 	 &init_legend;

	 &next_splash($splash,gettext('Loading settings'));
	 	&prep_checks;	# detects tablix version and modules location
	 	&prep_settings;
		&set_tablix_command_line;

	 &next_splash($splash,gettext('Checking dependencies'));
	 	&check_deps;					# check if all internally used programs are present
	 	&load_modules;				# check tablix's modules and fillup %modules

	 &next_splash($splash,gettext('Finalizing'));
	 	my $white = Gtk2::Gdk::Color->new(65535,65535,65535);
	 	$form->get_widget('eventbox_tt_table_explain')->modify_bg('normal',$white);
	 	$form->get_widget('eventbox_module_help')->modify_bg('normal',$white);
	 	$form->get_widget('radiobutton_tt_both')->set_active(1);
	 	&load_combos;

# setup the days and periods combos	
		&on_entry_times_changed(undef,undef,'periods');
		&on_entry_times_changed(undef,undef,'days');

# tablix2 uses different double periods scheduling...
	 	&write_to_statusbar(gettext("Welcome to G-Tablix."));
	 	&check_opts(shift(@ARGV),$splash);	 # check command line options

	$splash->destroy;
}

# make treeviews
sub prep_tvs { 			
	&draw_rests_editor;
	&prep_tt;
	&prep_lists;
	&prep_ttviews;
	&prep_teachers;
	&prep_combinations;
	&prep_modules;
#	&prep_combos;
}

# -----------------------------------------------------------------

sub get_locale_days {
	
	my @days = map { &langinfo } qw( DAY_1 DAY_2 DAY_3 DAY_4 DAY_5 DAY_6 DAY_7 );
	
	for (my $i = 0; $i < 7; $i++) {
		$day_names{$i} = $days[$i];
		print $day_names{$i}."\n";
	}
	
	#my ($abday_1, $yesstr, $nostr) = map { langinfo } qw(ABDAY_1 YESSTR NOSTR);

}

sub i18n {
	my $lang = ${&get_settings}{'gtablix'}{'lang'};	# users setting is default
	if (&get_optl) { $lang = &get_optl; }		# command line overrides..
	if (!defined $lang) { $lang = $ENV{LANG} ; }	# fallback to environment setting
	$lang = getFullLang ($lang);			# fallback to C
	$ENV{LANG} = $lang;
	$ENV{LANGUAGE} = $lang;
	$ENV{LC_ALL} = $lang;
	print setlocale(LC_ALL, $lang);
	textdomain('gtablix');
}

sub load_combos {
	my $form = &get_form;
# load possible timetable export functions
	my @exports = ( gettext("Result XML file"),
			gettext("Individual"),
			gettext("All Individuals"),
			gettext("Combined"),
			gettext("Styled Combined"),
			gettext("CVS Combined")
			);
	$form->get_widget("combo_tt_exports")->set_popdown_strings(@exports);

	my @plots;
	if ($tablix eq 'tablix2') {
		@plots = (gettext('conv-fitness'),
					 gettext('fit-fitness'),
					 gettext('functions')
					);
	} elsif ($tablix eq 'tablix') {
		@plots = (gettext('conv-grade'),
					 gettext('conv-mandatory'),
					 gettext('fit-grade'),
					 gettext('modules')
					);
	}
	$form->get_widget("combo_plot_style")->set_popdown_strings(@plots);
	
}

sub prep_combos {
# display day names instead of numbers
#	my %days = (1 => 'Monday', 2 => 'Tuesday', 3 => 'Wednesday', 4 => 'Thursday', 5 => 'Friday',
#		6 => 'Saturday', 7 => 'Sunday');
	my $form = &get_form;
	my $entry = $form->get_widget('combo_tt_days');
# 	my $model = $entry->get_model;
	my $model = new Gtk2::ListStore('Glib::String');
		$entry->list = $model;
   	my $renderer = new Gtk2::CellRendererText();
 	my $col = Gtk2::TreeViewColumn->new_with_attributes('',$renderer,'text' => 0);
 	$col->set_cell_data_func ($renderer, sub {
          my ($column, $cell, $model, $iter) = @_;
          my $i = $model->get($iter,0);
          my $text = gettext("Day").($i + 1);
          $cell->set (text => $text);
 	});
 	$model->append_column($col);
}
# -----------------------------------------------------------------
sub prep_pics {
	my $form = &get_form;
# setup shared pixmaps...
	$form->get_widget('image_hedge_about')->set_from_file(&get_picdir."/jezek-16.png");
	$form->get_widget('image_welcome_decor')->set_from_file(&get_picdir."/jezek-64_L.png");
	$form->get_widget('image_welcome_decor')->set_alignment(1,1);
	$form->get_widget('image_wizard_decor')->set_from_file(&get_picdir."/jezek-64_L.png");
	$form->get_widget('image_wizard_decor')->set_alignment(1,1);
	$form->get_widget('window_main')->set_default_icon_from_file(&get_picdir."/jezek-64_L.png");
	$form->get_widget('window_main')->set_icon_from_file(&get_picdir."/jezek-64_L.png");
}
# -----------------------------------------------------------------
sub prep_tt {
# prepare the tuples tree like list...
	my $form = &get_form;
	my $fontscale = $form->get_widget('spinbutton_set_fontscale')->get_value;
 	my $drevo2 = new Gtk2::TreeStore ('Glib::String','Glib::String',
													'Glib::String','Glib::String',
													'Glib::String','Glib::String',
													'Glib::String','Glib::String',
													'Glib::String');
 	$form->get_widget('treeview_tt')->set_model($drevo2);
	$form->get_widget('treeview_tt')->set('headers-clickable',1);
 	$form->get_widget('treeview_tt')->set('rules-hint',1);
 	$form->get_widget('treeview_tt')->get_selection->set_mode('multiple');
	my @columns = (gettext("Class"), 
					gettext("Subject"), 
					gettext("Teacher"), 
					gettext("Perweek"), 
					gettext("Room"), 
					gettext("Period"), 
					gettext("Day"), 
					gettext("Restrictions"), 
					gettext("Notes"));
	for (my $c = 0; $c <= $#columns; $c++) {
 		my $renderer = new Gtk2::CellRendererText();
		my $count = $c;
			$renderer->set('editable' => TRUE);
			$renderer->set('mode' => 'editable'); 

			$renderer->set_fixed_height_from_font(1);
			$renderer->set('scale' => $fontscale);

  			$renderer->signal_connect ('edited' => sub {
	           my ($cell,$text_path,$new_text) = @_;
				  my $tree = $form->get_widget('treeview_tt')->get_model;
				  my $iter = $tree->get_iter_from_string($text_path);
				  if (defined $iter) { 
					  my $old_text = $tree->get($iter,$count);
						$tree->set($iter,$count,$new_text); 
# if the user has renamed us, update other tvs
						if ($count == 0) { &rename_entity('classs',$old_text,$new_text); }
						elsif ($count == 1) { &rename_entity('subjects',$old_text,$new_text); }
						elsif ($count == 2) { &rename_entity('teachers',$old_text,$new_text); }
						elsif ($count == 4) { &rename_entity('classrooms',$old_text,$new_text); }
				  }
 			 	});	 
		my $col = Gtk2::TreeViewColumn->new_with_attributes($columns[$c],$renderer,'text' => $c);
		$col->set_resizable(1);
		$col->set_sort_column_id($c);
 		$form->get_widget('treeview_tt')->append_column($col);
 	}

}
# -----------------------------------------------------------------
sub prep_teachers {
# prepare the teachers tree like list...
	my $form = &get_form;
	my $fontscale = $form->get_widget('spinbutton_set_fontscale')->get_value;
 	my $tvt = $form->get_widget('treeview_teachers');
	my $drevo = new Gtk2::TreeStore ('Glib::String','Glib::String','Glib::String','Glib::String');
 	$tvt->set_model($drevo);
	$tvt->set('headers-clickable',1);
 	$tvt->set('rules-hint',1);
#	$tvt->set('fixed-height-mode',1);
	$tvt->get_selection->set_mode('multiple');
#	$tvt->get_selection->signal_connect(changed => sub { &select_row('teacher');});
	my @columns = (gettext("Teacher"),
						gettext("Subject"),
						gettext("Restrictions"),
						gettext("Notes"));
 	for (my $c = 0; $c <= $#columns; $c++) {
 		my $renderer = new Gtk2::CellRendererText();
		my $col = Gtk2::TreeViewColumn->new_with_attributes($columns[$c],$renderer,'text' => $c);
			$col->set_sort_column_id($c);
 			$tvt->append_column($col);
			$col->set_resizable(1);
		my $count = $c;
			$renderer->set('editable' => TRUE);
			$renderer->set('mode' => 'editable'); 
			
			$renderer->set_fixed_height_from_font(1);
			$renderer->set('scale' => $fontscale);
  			
			$renderer->signal_connect ('edited' => sub {
	           my ($cell,$text_path,$new_text) = @_;
				  my $tree = $tvt->get_model;
				  my $iter = $tree->get_iter_from_string($text_path);
				  if (defined $iter) { 
					  my $old_text = $tree->get($iter,$count);
						$tree->set($iter,$count,$new_text); 
# if the user has renamed us, update other tvs
						if ($count == 0) { &rename_entity('teachers',$old_text,$new_text); }
						elsif ($count == 1) { &rename_entity('subjects',$old_text,$new_text); }
				  }
 			 	});	 
 	}

}
# -----------------------------------------------------------------
sub prep_ttviews {
	my $form = &get_form;
	my %clists;
	$clists{"ttview_all"} = Gtk2::SimpleList->new_from_treeview(
									$form->get_widget('treeview_ttview_all'),
									gettext("Entity")=>"text");
	$clists{"ttview_files"} = Gtk2::SimpleList->new_from_treeview(
									$form->get_widget('treeview_ttview_files'),
									gettext("File")=>"text");
	foreach my $tv (keys(%clists)) {	# set the columns sort ids...
		my @columns = $form->get_widget("treeview_$tv")->get_columns;
		for (my $i = 0 ; $i <= $#columns ; $i++) { 
			$columns[$i]->set_sort_column_id ($i); 
			$columns[$i]->set_resizable(1);
		}
	}	
}
# -----------------------------------------------------------------
sub prep_modules {	
	my $form = &get_form;
		my $modules = Gtk2::SimpleList->new_from_treeview(
									$form->get_widget('treeview_modules'),
									gettext("Name")=>"text",
									gettext("Weight")=>"int",
									gettext("Mandatory")=>"bool",
									gettext("Active")=>"bool",
									gettext("Options")=>"text",
									gettext("Notes")=>"text");  
		$modules->set_column_editable("1","1");
		my @columns = $form->get_widget("treeview_modules")->get_columns;
		$form->get_widget("treeview_modules")->get_selection->set_mode('multiple');
		for (my $i = 0 ; $i <= $#columns ; $i++) { 
			$columns[$i]->set_sort_column_id ($i); 
			$columns[$i]->set_resizable(1);
		}
}
# -----------------------------------------------------------------
sub prep_lists {
	my $form = &get_form;
	my $fontscale = $form->get_widget('spinbutton_set_fontscale')->get_value;
#	my @clists = ('subjects', 'classrooms', 'classs');
	my %clists;
	$clists{"subjects"} = $form->get_widget('treeview_subjects'); 
	$clists{"classrooms"} = $form->get_widget('treeview_classrooms'); 
	$clists{"classs"} = $form->get_widget('treeview_classs'); 

# and other simple lists....
	foreach my $tv (keys(%clists)) {	# set the columns sort ids...
	  	my $drevo = new Gtk2::ListStore ('Glib::String','Glib::String','Glib::String');
	  	$clists{$tv}->set_model($drevo);
	 	$clists{$tv}->set('headers-clickable',1);
	   $clists{$tv}->set('rules-hint',1);
	 #	$clists{$tv}->set('fixed-height-mode',1);
	 	$clists{$tv}->get_selection->set_mode('multiple');
	
		my @columns = (gettext("Name"),
							gettext("Restrictions"),
							gettext("Notes"));
		for (my $i = 0 ; $i <= $#columns ; $i++) { 
			my $renderer = new Gtk2::CellRendererText();
			my $col = Gtk2::TreeViewColumn->new_with_attributes($columns[$i],$renderer,'text' => $i);
				$col->set_sort_column_id ($i); 
				$col->set_resizable(1);
			$clists{$tv}->append_column($col);
			my $count = $i;
			$renderer->set('editable' => TRUE);
			$renderer->set('mode' => 'editable'); 

			$renderer->set_fixed_height_from_font(1);
			$renderer->set('scale' => $fontscale);

			$renderer->signal_connect('edited' => sub {
	 	 		my ($renderer,$text_path,$new_text) = @_;
	 	 		my $tree = $form->get_widget('treeview_'.$tv)->get_model;
	 	 		my $iter = $tree->get_iter_from_string($text_path);
	 	 		if (defined $iter) { 
		 	 		my $old_text = $tree->get($iter,$count);
	 	 			$tree->set($iter,$count,$new_text); 
	 	 # if the user has renamed us, update other tvs
	 	 			if ($count == 0) { rename_entity($tv,$old_text,$new_text); }
	 	 		}
			});
		}
	}	
}
# -----------------------------------------------------------------
sub prep_combinations {
# prepare the combinations lists...
	my $form = &get_form;
	my $fontscale = $form->get_widget('spinbutton_set_fontscale')->get_value;
	my @combination = ('teachers','subjects','classs','classrooms');
	for (my $c = 0; $c <= $#combination; $c++) {
	  	my $drevo = new Gtk2::ListStore ('Glib::String','Glib::String');
	  	my $tree = 	$form->get_widget('treeview_combinations_'.$combination[$c]);
		$tree->set_model($drevo);
	 	$tree->set('headers-clickable',1);
	  	$tree->set('rules-hint',1); 
		$drevo->set_sort_func(0,\&sort_by_color);
	  	$drevo->set_sort_func(1,\&sort_by_number);

	# column with names
	  	my $renderer1 = new Gtk2::CellRendererText();
			$renderer1->set('editable' => TRUE);
			$renderer1->set('mode' => 'editable'); 
			
			$renderer1->set_fixed_height_from_font(1);
			$renderer1->set('scale' => $fontscale);
						
  			$renderer1->signal_connect ('edited' => sub {
	           my ($cell,$text_path,$new_text) = @_;
				  #my $tree = $form->get_widget('treeview_combinations_'.$combination[$c])->get_model;
				  my $iter = $tree->get_model->get_iter_from_string($text_path);
				  if (defined $iter) { 
					  my $old_text = $tree->get_model->get($iter,0);
						$tree->get_model->set($iter,0,$new_text); 
# if the user has renamed us, update other tvs
						&rename_entity(undef,$old_text,$new_text); 
				  }
 			 	});	 
	# ugly...
		my $title = $combination[$c];
		if ($combination[$c] eq 'classs') { $title = 'classes'; }
		$title = ucfirst($title);

		my $col1 = Gtk2::TreeViewColumn->new_with_attributes(gettext($title),$renderer1,'text' => 0);
			$col1->set_cell_data_func ($renderer1, sub {
	         my ($column, $cell, $model, $iter) = @_;
	         my ($text, $color, $fg) = do_something_cool ($model, $iter);
	         $cell->set (text => $text, background => $color, foreground => $fg);
    		});
	  	$col1->set_sort_column_id(0);
	   $col1->set_resizable(1);
		$tree->append_column($col1);

	# column with frequencies
	  	my $renderer2 = new Gtk2::CellRendererText();

			$renderer2->set_fixed_height_from_font(1);
			$renderer2->set('scale' => $fontscale);

		my $col2 = Gtk2::TreeViewColumn->new_with_attributes('#',$renderer2,'text' => 1);
			$col2->set_cell_data_func ($renderer2, sub {
	         my ($column, $cell, $model, $iter) = @_;
	         my ($text, $color, $fg) = do_something_cool_perweek ($model, $iter);
	         $cell->set (text => $text, background => $color, foreground => $fg);
    		});
		$col2->set_sort_column_id(1);
	   $col2->set_resizable(1);
		$tree->append_column($col2);
	}

}

# -----------------------------------------------------------------

sub check_opts {
	my ($open_file,$splash) = @_;
	my ($form,undef,$defined_tt_entries,$uid_tt_entries) = &get_vars;
	if ((defined $open_file) and !($open_file =~ /^-/)) { 
	 	&next_splash($splash,gettext('Loading file'));
		&parse_xml($open_file,$form,$defined_tt_entries,$uid_tt_entries); 
	}
}

sub prep_checks {
	my %tablix = (
						'tablix' => [
											'tablix', 'tablix_plot', 'tablix_modinfo', 'tablix_output'
										 ],	
						'tablix2' => [	
											'tablix2', 'tablix2_plot', 'tablix2_test', 'tablix2_output'
											]
					);
	foreach my $tablx (keys(%tablix)) {
		if (&try_exec($tablx)) {
			@checks = (@checks,@{$tablix{$tablx}});
			$tablix = $tablx;
		}
	}
}

sub check_deps {
	my ($quiet) = @_;
	my $errors = "";
# check for external executables
	foreach my $a (@checks) {
	 	my $path = &try_exec($a);
		if (!$path) {
 			if ($a ne 'clusterconf') { 
 				$errors = $errors.gettext("Couldn't start")." $a;\n"; 
 			}
 			&disable ($a);
 			&write_setting($a,gettext('not found'));
 		} else {
 			&write_setting($a,$path);
 		}
	}
# set modules path
	$errors = $errors.&set_modules_path; 
# report errors	
	if ($errors ne "" and !$quiet) {
		$errors = $errors."-----\n".gettext("Some external programs couldn't be started. Check that they are installed and in the path, or some functions will not work.");
		if (!$form->get_widget('checkbutton_hide_missing')->get_active) { &check($errors,'error'); }
	}
}
sub try_exec {
	my ($a) = @_;
	my $path = readpipe('echo $PATH');
	chop $path;
	$path =~ s/\:/ /g;
	my $b = readpipe("whereis -B $path -f $a");
	chop($b);
	$b = (split/ /,$b)[1];
	if (!defined $b or !-x $b) { 
		return 0;
	} else {
		return $b;
	}
}
sub disable {
	my ($what) = @_;
	my $form = &get_form;
	if ($what eq 'tablix') {
		$form->get_widget('frame_tablix')->set('sensitive' => 0);
		$form->get_widget('run1')->set('sensitive' => 0);
		&disable_modules;
	} elsif ($what eq 'tablix_plot') {
		$form->get_widget('frame_plot')->set('sensitive' => 0);
	} elsif ($what eq 'tablix_modinfo') {
		&disable_modules;
	} elsif ($what eq 'gnuplot') {
		$form->get_widget('frame_plot')->set('sensitive' => 0);
	} elsif ($what eq 'clusterconf') {
		$form->get_widget('clusterconf1')->set('visible' => 0);
	} elsif ($what eq 'xpvm') {
		$form->get_widget('xpvm1')->set('sensitive' => 0);
	} elsif (($what eq 'html2ps') or ($what eq 'lp')) {
		$form->get_widget('button_tt_print')->set('sensitive' => 0);
	}
}
sub disable_modules {
	my $form = &get_form;
	$form->get_widget('frame_modules')->set('sensitive' => 0);
	$form->get_widget('button_teacher_rests')->set('sensitive' => 0);
	$form->get_widget('button_subjects_rests')->set('sensitive' => 0);
	$form->get_widget('button_rooms_rests')->set('sensitive' => 0);
	$form->get_widget('button_classes_rests')->set('sensitive' => 0);
	$form->get_widget('button_tt_rests')->set('sensitive' => 0);
	$form->get_widget('toggletoolbutton_options')->set('sensitive' => 0);
	$form->get_widget('restrictions1')->set('sensitive' => 0);	
	$form->get_widget('checkbutton_double_period')->set('sensitive' => 0);	
}

sub set_modules_path {	# Extract module path from "tablix -v" output and set "entry_modpath"
# If "tablix -v" does not contain this information, use a default value.
	my $form = &get_form;
	my $tablix = &get_tablix;
	my $temp = readpipe("LC_ALL=C ; $tablix -v");

	if ($tablix eq 'tablix') {
		if (($temp =~ /Searching for modules in (.*)/) && ($temp !~ /default dir/)) {
			$temp = $1;
		} elsif (-d "/usr/lib/$tablix") {
			$temp = "/usr/lib/$tablix";
		} elsif (-d "/usr/local/lib/$tablix") {
			$temp = "/usr/local/lib/$tablix";
		} else {
			&disable_modules;
			return gettext("Couldn't find directory with ").$tablix.gettext(" modules");
		}	
	} elsif ($tablix eq 'tablix2') {
		if ($temp =~ /Module documentation available in (.*)/) {
			$temp = $1;
			if (!-d $temp) {
				&disable_modules;
				return gettext("Couldn't find directory with ").$tablix.gettext(" modules");
			}
		} else { $temp = "/usr/share/tablix2"; }
	}	
	$form->get_widget('entry_modpath')->set_text($temp);

	return "";
}

sub splash {
	 my $picdir = &get_picdir;
	 
	 my $splash = new Gtk2::Window('toplevel');
	 $splash->realize();	 
	 $splash->set_position('center');
#	 $splash->set_decorated(0);
	 
	 my $vbox = new Gtk2::VBox(0,0);
	 $splash->add($vbox);
	 
	 my $tablix = Gtk2::Image->new_from_file($picdir."/tablix-chrome.png");
	 $vbox->pack_start($tablix,0,0,0);
	 
	 my $hedge = Gtk2::Image->new_from_file($picdir."/jezek-64_L.png");
	 $vbox->pack_start($hedge,0,0,0);
	 $hedge->set_alignment(1,1);

	 my $label = new Gtk2::Label;
	 $vbox->pack_start($label,0,0,0);
	
	 my $progress = new Gtk2::ProgressBar;
	 $vbox->pack_start($progress,0,0,0);
	
	 $splash->{'label'} = $label;
	 $splash->{'progress'} = $progress;
	
	 $splash->set_icon_from_file($picdir."/jezek-64_L.png");

	 $splash->show_all();
	 return $splash;
}
sub next_splash {
	my ($splash,$text) = @_;
	 $splash->{'label'}->set_text($text);
	 $splash->{'progress'}->pulse;
	 while (Gtk2->events_pending) {
	 	Gtk2->main_iteration
	 }
}

sub check_locale_dirs {
	my $libdir = &get_libs;
	my @locales = split/\n/,readpipe("ls -1 $libdir/po/*.mo");
	foreach my $l (@locales) {
		my @lang = split/\//,$l;
		my $lang = (split/\./,$lang[$#lang])[1];
		if (!-e "$libdir/po/$lang") {
			print "Directory for language: $lang doesn't exists. Creating...\n";
			system("mkdir $libdir/po/$lang");
		}
		if (!-e "$libdir/po/$lang/LC_MESSAGES") {
			print "Directory for language messages: $lang doesn't exists. Creating...\n";
			system("mkdir $libdir/po/$lang/LC_MESSAGES");
		}
		if (!-e "$libdir/po/$lang/LC_MESSAGES/gtablix.mo") {
			print "File for language: $lang doesn't exists. Creating...\n";
			system("cp $l $libdir/po/$lang/LC_MESSAGES/gtablix.mo");
		}
	}
}

sub getFullLang {
	my ($l) = @_;
	$l =~ s/\..*//;
	foreach my $locale (readpipe("locale -a")) {
		my $loc = $locale;
		$loc =~ s/\..*//;
		if ($locale =~ /$l/i) { return $locale; }
	}
	return "C";
}

return 1;
