//*****************************************************************************
//                             PnlNgSpiceDC.cpp                               *
//                            ------------------                              *
// Started     : 08/05/2004                                                   *
// Last Update : 21/08/2009                                                   *
// Copyright   : (C) 2004 by MSWaters                                         *
// Email       : M.Waters@bom.gov.au                                          *
//*****************************************************************************

//*****************************************************************************
//                                                                            *
//    This program is free software; you can redistribute it and/or modify    *
//    it under the terms of the GNU General Public License as published by    *
//    the Free Software Foundation; either version 2 of the License, or       *
//    (at your option) any later version.                                     *
//                                                                            *
//*****************************************************************************

#include "ngspice/panels/PnlNgSpiceDC.hpp"

//*****************************************************************************
// Implement an event table.

BEGIN_EVENT_TABLE( PnlNgSpiceDC, PnlAnaBase )

  EVT_CHOICE  ( ID_CHO_SRCNAME, PnlNgSpiceDC::OnSrcName )
  EVT_RADIOBOX( ID_RBX_SWPTYPE, PnlNgSpiceDC::OnSwpType )

END_EVENT_TABLE( )

//*****************************************************************************
// Constructor.

PnlNgSpiceDC::PnlNgSpiceDC( wxWindow * poWin ) : PnlAnaBase( poWin )
{
  bSetSimrEng( eSIMR_NGSPICE );
  bSetAnaType( eCMD_DC );

  Create( );        // Create the analysis panel
  InitSwpUnits( );  // Initialize the sweep units
  bClear( );        // Clear all object attributes
}

//*****************************************************************************
// Destructor.

PnlNgSpiceDC::~PnlNgSpiceDC( )
{
}

//*****************************************************************************
// Create the display objects.

void  PnlNgSpiceDC::Create( void )
{
  // Disable the checkboxes for the parameters NG-Spice cannot calculate
  m_oCbxCurrent.Disable( );
  m_oCbxPower  .Disable( );
  m_oCbxResist .Disable( );

  // Set the sweep parameter labels
  m_oSbxSwpPars.SetLabel( wxT(" DC Sweep ") );
  m_oPnlStart  .bSetName( wxT("Start Value") );
  m_oPnlStop   .bSetName( wxT("Stop Value") );
  m_oPnlStep   .bSetName( wxT("Step Increment") );

  // Create and add input source controls
  m_oLblSrcCpnt.Create( this, ID_UNUSED, wxT("Signal Source"),
                        wxPoint(  18, 176 ) );
  m_oChoSrcName.Create( this, ID_CHO_SRCNAME,
                        wxPoint( 144, 171 ), wxSize( 110, 29 ) );

  // Create and add the sweep type radio buttons
  wxString  osSwpType[ 2 ] = { wxT("Source"), wxT("Temp.C ") };
  m_oRbxSwpType.Create( this, ID_RBX_SWPTYPE, wxT(" Sweep Type "),
                        wxPoint( 13, 123 ), wxDefaultSize, 2, osSwpType, 2 );
  m_oRbxSwpType.SetSelection( eSWP_CPNT );
  // Note : The following controls must have been created prior to call the next
  //        line : m_oLblSrcCpnt, m_oChoSrcCpnt
  InitSwpType( );

  PnlAnaBase::CreateTemp( );  // Create the analysis temperature controls
}

//*****************************************************************************
// Initialize the sweep type.

void  PnlNgSpiceDC::InitSwpType( void )
{
  int  ix, iy;

  m_oSbxSwpPars.GetSize( &ix, &iy );

  switch( (eSweepType) m_oRbxSwpType.GetSelection( ) )
  {
    case eSWP_CPNT :  // Component sweep
      m_oSbxSwpPars.SetSize( ix, 201 );
      m_oLblSrcCpnt.Show( TRUE );
      m_oChoSrcName.Show( TRUE );
      m_oSbxTemp   .Show( TRUE );
      m_oPnlTemp   .Show( TRUE );
      InitSwpUnits( );
      break;

    case eSWP_TEMP :    // Temperature sweep
      m_oSbxSwpPars.SetSize( ix, 167 );
      m_oLblSrcCpnt.Show( FALSE );
      m_oChoSrcName.Show( FALSE );
      m_oSbxTemp   .Show( FALSE );
      m_oPnlTemp   .Show( FALSE );
      InitSwpUnits( );
      break;

    default :
      break;
  }
}

//*****************************************************************************
// Initialize the sweep parameter units.

void  PnlNgSpiceDC::InitSwpUnits( void )
{
  wxString    os1;
  eUnitsType  eUnits;

  if( m_oRbxSwpType.GetSelection( ) == 0 )
  {
    os1 = m_oChoSrcName.GetStringSelection( );
    if( os1.IsEmpty( ) ) os1 = wxT('?');
    switch( wxToupper( os1.GetChar( 0 ) ) )
    {
      case wxT('R') :  // Units of resistance
        os1 = wxT("Resistance");
        eUnits = eUNITS_RES;
        break;

      case wxT('I') :  // Units of current
        os1 = wxT("Current");
        eUnits = eUNITS_CURR;
        break;

      case wxT('V') :  // Units of voltage
        os1 = wxT("Voltage");
        eUnits = eUNITS_VOLT;
        break;

      default :        // No units
        os1 = wxT("Value");
        eUnits = eUNITS_NONE;
    }
  }
  else
  {
    os1 = wxT("Temperature");
    eUnits = eUNITS_TEMP;
  }

  m_oPnlStart.bSetName( wxString( wxT("Start ") ) + os1 );
  m_oPnlStop .bSetName( wxString( wxT("Stop ")  ) + os1 );
  m_oPnlStart.bSetUnitsType( eUnits );
  m_oPnlStop .bSetUnitsType( eUnits );
  m_oPnlStep .bSetUnitsType( eUnits );
}

//*****************************************************************************
// Clear the object attributes.
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  PnlNgSpiceDC::bClear( void )
{
  bool  bRtn=TRUE;

  // Clear the base class
  if( ! PnlAnaBase::bClear( ) ) bRtn = FALSE;

  // Set the sweep parameters to their defaults
  m_oPnlStart.bSetValue( (float)   0.0 );
  m_oPnlStop .bSetValue( (float) 100.0 );
  m_oPnlStep .bSetValue( (float)  10.0 );

  // Set the default sweep type
  m_oRbxSwpType.SetSelection( eSWP_CPNT );
  InitSwpType( );

  // Set sweep default values
  m_oChoSrcName.Clear( );
  m_oChoSrcName.Append( wxT("None") );
  m_oChoSrcName.SetSelection( 0 );

  return( bRtn );
}

//*****************************************************************************
// Load information from a simulation object.
//
// Argument List :
//   roSim - A simulation object
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  PnlNgSpiceDC::bLoad( SimnNgSpice & roSimn )
{
  bool  bRtn=TRUE;

  // Load components into the signal source choice box
  PnlAnaBase::LoadSrcNames( roSimn.m_oaCpnts, wxT("VIR") );

  // Set the source component
  if( ! PnlAnaBase::bSetSrcCpnt( roSimn.m_oCpntSwpSrc ) )    bRtn = FALSE;
  m_oChoSrcName.SetStringSelection( roSimn.m_oCmdDC.m_osSource );
  InitSwpUnits( );  // Set the sweep parameter units

  // Set the sweep values
  if( ! m_oPnlStart.bSetValue( roSimn.m_oCmdDC.m_osStart ) ) bRtn = FALSE;
  if( ! m_oPnlStop .bSetValue( roSimn.m_oCmdDC.m_osStop  ) ) bRtn = FALSE;
  if( ! m_oPnlStep .bSetValue( roSimn.m_oCmdDC.m_osStep  ) ) bRtn = FALSE;
  if( m_oPnlStart.dfGetValue( ) == 0.0 )
        m_oPnlStart.bSetUnits( m_oPnlStop.rosGetUnits( ) );

  // Set the sweep type and source component if required
  if( roSimn.m_oCmdDC.m_osSource == wxT("TEMP") )
       m_oRbxSwpType.SetSelection( eSWP_TEMP );
  else m_oRbxSwpType.SetSelection( eSWP_CPNT );
  InitSwpType( );

  // Set the analysis temperature
  if( ! m_oPnlTemp.bSetValue( roSimn.m_oCmdOPT.m_osTEMP ) )  bRtn = FALSE;

  return( bRtn );
}

//*****************************************************************************
// Save information to a simulation object.
// (Prior to it being passed to a simulator object.)
//
// Argument List :
//   roSim - A simulation object
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  PnlNgSpiceDC::bSave( SimnNgSpice & roSimn )
{
  m_osErrMsg.Empty( );

  // Set the sweep values
  roSimn.m_oCmdDC.m_osStart = m_oPnlStart.rosGetValue( );
  roSimn.m_oCmdDC.m_osStop  = m_oPnlStop .rosGetValue( );
  roSimn.m_oCmdDC.m_osStep  = m_oPnlStep .rosGetValue( );

  // Set the sweep source
  switch( m_oRbxSwpType.GetSelection( ) )
  {
    case eSWP_CPNT :
      if( m_oChoSrcName.GetStringSelection( ) != wxT("None") )
        roSimn.m_oCmdDC.m_osSource = m_oChoSrcName.GetStringSelection( );
      break;
    case eSWP_TEMP :
      roSimn.m_oCmdDC.m_osSource = wxT("TEMP");
      break;
    default :
      roSimn.m_oCmdDC.m_osSource.Empty( );
  }
  roSimn.m_oCpntSwpSrc.bClear( );

  // Set the analysis type
  roSimn.m_oCmdPR.m_eAnaType = eCMD_DC;

  // Store the parameters to derive
  roSimn.m_oCmdPR.m_bParmtrs[ ePAR_VLT ] = m_oCbxVoltage.GetValue( );
  roSimn.m_oCmdPR.m_bParmtrs[ ePAR_CUR ] = m_oCbxCurrent.GetValue( );
  roSimn.m_oCmdPR.m_bParmtrs[ ePAR_PWR ] = m_oCbxPower  .GetValue( );
  roSimn.m_oCmdPR.m_bParmtrs[ ePAR_RES ] = m_oCbxResist .GetValue( );

  // Set the analysis temperature
  roSimn.m_oCmdOPT.m_osTEMP = m_oPnlTemp.rosGetValue( );

  // Create the command strings
  roSimn.m_oCmdDC.bFormat( );
  roSimn.m_oCmdPR.bFormat( );

  // Check for errors
  if( ! roSimn.m_oCmdDC.bIsValid( ) )
    SetErrMsg( roSimn.m_oCmdDC.rosGetErrMsg( ) );
  if( ! roSimn.m_oCmdPR.bIsValid( ) )
    SetErrMsg( roSimn.m_oCmdPR.rosGetErrMsg( ) );

  return( bIsOk( ) );
}

//*****************************************************************************
//                                                                            *
//                             Event Handlers                                 *
//                                                                            *
//*****************************************************************************
// Sweep type radio box event handler.
//
// Argument List:
//   roEvtCmd - An object holding information about the event

void  PnlNgSpiceDC::OnSwpType( wxCommandEvent & roEvtCmd )
{
  InitSwpType( );
}

//*****************************************************************************
// Source component choice box event handler.
//
// Argument List:
//   roEvtCmd - An object holding information about the event

void  PnlNgSpiceDC::OnSrcName( wxCommandEvent & roEvtCmd )
{
  InitSwpUnits( );
}

//*****************************************************************************
