#include <AppKit/AppKit.h>
#include <AppKit/NSWindowController.h>
#include <GSDictParameters.h>

@implementation GSDictParameters

+ (id)parametersWithDictionary:(NSDictionary*)dico
{
	// With initWithUserDefaults, I'm sure that the GSDictParameters will
	// be corrrectly initialised even if dico is missing some values
	id result = [[[GSDictParameters alloc] initWithUserDefaults] autorelease];
	NSString* word = [dico objectForKey:@"word"];
	NSString* hostName = [dico objectForKey:@"hostName"];
	int port = [[dico objectForKey:@"port"] intValue];
	NSString* database = [dico objectForKey:@"database"];
	NSString* strategy = [dico objectForKey:@"strategy"];
	if (word != nil)
	{
		[result setWord:word];
	}
	if (hostName != nil)
	{
		[result setHostName:hostName];
	}
	if ( port > 0 )
	{
		[result setPort:port];
	}
	if (database != nil)
	{
		[result setDatabase:database];
	}
	[result setMatch:[[dico objectForKey:@"match"] boolValue]];
	if (strategy != nil)
	{
		[result setStrategy:strategy];
	}
	return result;
}

+ (id)defaultParameters;
{
	return [[[GSDictParameters alloc] initWithUserDefaults] autorelease];
}

- (id) init
{
  [super init];

	_word = nil;
	_hostName = nil;
	_port = 0;
	_database = nil;
	_match = 0;
	_strategy = nil;

  return self;
}

- (id)initWithUserDefaults
{
	NSUserDefaults *defaults = nil;
	id result = nil;

	[self init];
	defaults = [NSUserDefaults standardUserDefaults];
	NSDebugLog( @"GSDict", @"[GSDictParameters -initFromUserDefaults]" );
	NSAssert( defaults, @"No user defaults found" );
	
	// hostname
	result = [defaults stringForKey:@"hostName"];
	if ( result == nil )
	{
		_hostName = [[NSString alloc] initWithString:@"dict.org"];
		[defaults setObject:_hostName forKey:@"hostName"];
	}
	else
	{
		_hostName = [[NSString alloc] initWithString:result];
	}
	NSAssert( _hostName, @"No hostName setting found" );

	// port
	_port = [defaults integerForKey:@"port"];
	if ( _port == 0 )
	{
		_port = 2628;
		[defaults setInteger:_port forKey:@"port"];
	}
	NSAssert( (_port>0), @"No port setting found" );

	// database
	result = [defaults stringForKey:@"database"];
	if ( result == nil )
	{
		_database = [[NSString alloc] initWithString:@"All"];
		[defaults setObject:_database forKey:@"database"];
	}
	else
	{
		_database = [[NSString alloc] initWithString:result];
	}
	NSAssert( _database, @"No database setting found" );

	// match
	_match = [defaults boolForKey:@"match"];

	// strategy
	result = [defaults stringForKey:@"strategy"];
	if ( result == nil )
	{
		_strategy = [[NSString alloc] initWithString:@"exact"];
		[defaults setObject:_strategy forKey:@"strategy"];
	}
	else
	{
		_strategy = [[NSString alloc] initWithString:result];
	}
	NSAssert( _strategy, @"No strategy setting found" );
	
	// update defaults
	[defaults synchronize];

  return self;
}

- (id)initWithParameters:(GSDictParameters*)parameters
{
	ASSIGN( _word, [[parameters word] copy] );
	ASSIGN( _hostName, [[parameters hostName] copy] );
	_port = [parameters port];
	ASSIGN( _database, [[parameters database] copy] );
	_match = [parameters match];
	ASSIGN( _strategy, [[parameters strategy] copy] );
	return self;
}

- (void) dealloc
{
	RELEASE(_word);
	RELEASE(_hostName);
	RELEASE(_database);
	RELEASE(_strategy);
	[super dealloc];
}

- (BOOL)saveAsDefaults
{
	NSUserDefaults *defaults = [NSUserDefaults standardUserDefaults];
	[defaults setObject:_hostName forKey:@"hostName"];
	[defaults setInteger:_port forKey:@"port"];
	[defaults setObject:_database forKey:@"database"];
	[defaults setBool:_match forKey:@"match"];
	[defaults setObject:_strategy forKey:@"strategy"];
	[defaults synchronize];
	return YES;
}

- (void)setWord:(NSString*)word
{
	NSAssert( word, @"invalid word" );
	ASSIGN( _word, word );
}

- (NSString*)word
{
	return _word;
}

- (void)setHostName:(NSString*)hostName
{
	NSAssert( hostName, @"invalid hostName" );
	ASSIGN( _hostName, hostName );
}

- (NSString*)hostName
{
	return _hostName;
}

- (void)setPort:(int)port
{
	NSAssert( port, @"invalid port" );
	_port = port;
}

- (int)port
{
	return _port;
}

- (void)setDatabase:(NSString*)database
{
	NSAssert( database, @"invalid database" );
	ASSIGN( _database, database );
}

- (NSString*)database
{
	return _database;
}

- (NSArray*)databases
{
	id databases = nil;
	id provider = [[NSApplication sharedApplication] servicesProvider];
	if ( provider != nil )
	{
		databases = [provider databases];
	}
	if ( databases == nil )
	{
		databases = [[[NSArray alloc] init] autorelease];
	}
	return databases;
}

- (void)setMatch:(BOOL)match
{
	_match = match;
}

- (BOOL)match
{
	return _match;
}

- (void)setStrategy:(NSString*)strategy
{
	NSAssert( strategy, @"invalid strategy" );
	ASSIGN( _strategy, strategy );
}

- (NSString*)strategy
{
	return _strategy;
}

- (NSArray*)strategies
{
	id strategies = nil;
	id provider = [[NSApplication sharedApplication] servicesProvider];
	if ( provider != nil )
	{
		strategies = [provider strategies];
	}
	if ( strategies == nil )
	{
		strategies = [[[NSArray alloc] init] autorelease];
	}
	return strategies;
}

- (id) copyWithZone: (NSZone*)zone
{
	id result = nil;
	result = [[self class] allocWithZone:zone];
	[result initWithParameters:self];
	return result;
}

-(NSMutableDictionary*)dictionaryRepresentation
{
	id result = nil;
	result = [NSMutableDictionary dictionary];
	if ( _word )
	{
		[result setObject:_word forKey:@"word"];
	}
	[result setObject:_hostName forKey:@"hostName"];
	[result setObject:[NSNumber numberWithInt:_port] forKey:@"port"];
	[result setObject:_database forKey:@"database"];
	[result setObject:[NSNumber numberWithBool:_match] forKey:@"match"];
	[result setObject:_strategy forKey:@"strategy"];
	return result;
}

-(NSString*)description
{
	return [[self dictionaryRepresentation] description];
}

- (BOOL)isEqualToString:(NSString*)string
{
	NSAssert( _word, @"_word==nil" );
	return ([_word isEqualToString:string]);
}

- (NSSize) sizeWithAttributes:(NSDictionary *)attrs
{
	NSAssert( _word, @"_word==nil" );
	return [_word sizeWithAttributes:attrs];
}

- (void) drawInRect: (NSRect)rect withAttributes: (NSDictionary *)attrs
{
	NSAssert( _word, @"_word==nil" );
	[_word drawInRect:rect withAttributes:attrs];
}

- (unsigned int) length
{
	NSAssert( _word, @"_word==nil" );
	return [_word length];
}

- (BOOL) canBeConvertedToEncoding:(NSStringEncoding)enc
{		
	NSAssert( _word, @"_word==nil" );
	return [_word canBeConvertedToEncoding:enc];
}

- (void) getCString: (char*)buffer maxLength: (unsigned)maxLength
{
	NSAssert( _word, @"_word==nil" );
	[_word getCString:buffer maxLength:maxLength];
}

- (unichar) characterAtIndex: (unsigned int)index
{
	NSAssert( _word, @"_word==nil" );
	return [_word characterAtIndex:index];
}

@end
