/*
 * Groach default 'theme' module
 * Refer to theme-default.h about details.
 *
 * Copyright INOUE Seiichiro <inoue@ainet.or.jp>, licensed under the GPL.
 */
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#include <math.h>
#include <gnome.h>

#include "groach.h"


/* Constant numbers */
enum {
	/* These two can't be overridden */
	/*   GRO_STAT_DEAD = -1,
		 GRO_STAT_WAKE = 0, */
	GRO_STAT_SQUISH = 1	/* I don't refer to this directly now. */
	/* ...
	   'default-theme' takes care of any number of states. */
};

/* Data structure definitions */	
/* GroMove instance has an arbitrary pointer, named move_env.
   GroMove itself doesn't care about its content.
   It is up to an external module, i.e. it is up to this module now.
   This module sets DefaultMoveEnv data to it, and use it in callbacks. */
typedef struct _DefaultMoveEnv DefaultMoveEnv;
typedef struct _AngleInfo AngleInfo;

struct _DefaultMoveEnv {
	gboolean turn_left;
	guint steps;	/* number of steps until recomputation */
	AngleInfo *angle_infos;/* array of AngleInfo for all directions */
};

struct _AngleInfo {
	gdouble sine;
	gdouble cosine;
};

/* Private function declarations */
static void move_move_cb(GroMove *gmove, const GroController *controller, const GroVector *vec, gpointer data);
static void move_push_cb(GroMove *gmove, const GroController *controller, GdkEventButton *event, gpointer data);
static void turn_gmove(GroMove *gmove);

#if 0
/* The following two callback functions are called once for one module.
   Default module doesn't use them. */
gint
theme_init(const GroController *controller)
{
	return 1;
}

gint
theme_finalize(const GroController *controller)
{
	return 1;
}
#endif

/**
 * move_init:
 * Called for each GroMove instance for the initialization.
 * Create this module's specific data structure, DefaultMoveEnv,
 * and set it to GroMove's move_env variable for later use.
 **/
gint
move_init(const GroController *controller, GroMove *gmove)
{
	GroVector init_pos;
	DefaultMoveEnv *move_env;
	AngleInfo *angle_infos;
	int nd;

	move_env = g_new(DefaultMoveEnv, 1);
	gmove->move_env = move_env;/* Set it for later use */
	move_env->turn_left = RAND_INT(100) >= 50;
	move_env->steps = RAND_INT(200);
	angle_infos = g_new(AngleInfo, gmove->num_direct);
	move_env->angle_infos = angle_infos;
	for (nd = 0; nd < gmove->num_direct; nd++) {
		gdouble angle;

		angle = 2 * M_PI * nd / gmove->num_direct;
		angle_infos[nd].sine = sin(angle);
		angle_infos[nd].cosine = cos(angle);
	}

	/* Initialize */
	gro_move_change_gstat(gmove, GRO_STAT_WAKE);
	gro_move_change_direct(gmove, RAND_INT(gmove->num_direct));

	/* Initial position is decided with randum number */
	init_pos.ix = RAND_INT(gcontroller_window_width(controller) - gmove_width(gmove));
	init_pos.iy = RAND_INT(gcontroller_window_height(controller) - gmove_height(gmove));
	gro_move_move(gmove, controller, &init_pos);
	
	gtk_signal_connect(GTK_OBJECT(gmove), "move",
					   GTK_SIGNAL_FUNC(move_move_cb),
					   NULL);
	gtk_signal_connect(GTK_OBJECT(gmove), "push",
					   GTK_SIGNAL_FUNC(move_push_cb),
					   NULL);

	return 1;
}

/**
 * move_compute:
 * Called for each GroMove instance for the motion.
 * The default motion logic is as follows,
 * If it's not in the last stat, just move it
 * (the frame is implicitly advanced, which is taken care of by GroMove).
 * If it's in the last stat but not in the last frame,
 * don't move but just advance the frame.
 * If it's in the last stat and in the last frame, kill it.
 * Output:
 * GroVector *ret_vec; Vector to move.
 * Retrun value;
 **/
GroMoveRet
move_compute(const GroController *controller, GroMove *gmove, const GdkRegion *vis_region, GroVector *ret_vec)
{
	DefaultMoveEnv *move_env = gmove->move_env;
	const AngleInfo *angle_infos = move_env->angle_infos;
	GdkRectangle tmp_gmove_rect = gmove->cur_rect;/* local copy */
	gint ix;
	gint iy;

	g_return_val_if_fail(gmove->cur_gstat != GRO_STAT_DEAD, GRO_RET_DONT_MOVE);
	
	/* Every theme is recommended to call this at first */
	DONTCARE_hidden_gmove(gmove, vis_region);
	
	/* If it's in the last stat and in the last frame, kill it. */
	/* if (gmove->cur_gstat == GRO_STAT_SQUISH) { */
	if (gmove->cur_gstat == (gmove->num_gstat - 1)) {
		if (gro_move_is_eof(gmove) == TRUE) {
			gro_move_change_gstat(gmove, GRO_STAT_DEAD);
			return GRO_RET_DONT_MOVE;
		}
		/* It's in the last stat but not the last frame.
		   Don't move but just advances the frame. */
		ret_vec->ix = ret_vec->iy = 0;
		return GRO_RET_MOVE;
	}
	
	/* Compute a new position temporarily to check whether it is
	   within the window and to check collisions with other gmoves. */
	ix = gcontroller_step_pixel(controller) * angle_infos[gmove->cur_direct].cosine;
	iy = -(gcontroller_step_pixel(controller) * angle_infos[gmove->cur_direct].sine);
	tmp_gmove_rect.x += ix;
	tmp_gmove_rect.y += iy;

	if (is_rect_in_gcontroller_window(controller, &tmp_gmove_rect) == TRUE) {
		const GList *other_gmoves = controller->gmove_list;
		const GList *node;

		if (move_env->steps-- <= 0) {
			turn_gmove(gmove);
			move_env->steps = RAND_INT(200);
		}

		/* Detect collision and avoid it.
		   This is intended to follow xroach's method. */
		for (node = other_gmoves; node; node = node->next) {
			GroMove *other = node->data;
			if (gmove == other)
				continue;
			if (is_rect_intersect(&tmp_gmove_rect, &other->cur_rect)) {
				turn_gmove(gmove);
				break;
			}
		}

		/* Recompute the move vector after turn */
		ix = gcontroller_step_pixel(controller) * angle_infos[gmove->cur_direct].cosine;
		iy = -(gcontroller_step_pixel(controller) * angle_infos[gmove->cur_direct].sine);
	} else {
		/* If a new position is out of the window, turn it */
		turn_gmove(gmove);
		ix = iy = 0;
    }
	
	ret_vec->ix = ix;
	ret_vec->iy = iy;

	return GRO_RET_MOVE;
}

/**
 * move_finalize:
 * Called for each GroMove instance for the finalization.
 **/
gint
move_finalize(const GroController *controller, GroMove *gmove)
{
	DefaultMoveEnv *move_env;
	AngleInfo *angle_infos;

	gtk_signal_disconnect_by_func(GTK_OBJECT(gmove),
								  GTK_SIGNAL_FUNC(move_push_cb),
								  NULL);
	gtk_signal_disconnect_by_func(GTK_OBJECT(gmove),
								  GTK_SIGNAL_FUNC(move_move_cb),
								  NULL);
	move_env = gmove->move_env;
	angle_infos = move_env->angle_infos;
	g_free(angle_infos);
	g_free(move_env);

	return 1;
}



/* ---The followings are private functions--- */
/**
 * move_move_cb:
 * "move" signal handler.
 **/
static void
move_move_cb(GroMove *gmove, const GroController *controller, const GroVector *vec, gpointer data)
{
	/* Do nothing currently */
}

/**
 * move_push_cb:
 * "push" signal handler.
 * Just advance its stat.
 * Caution: don't increment cur_gstat directly.
 * Instead, to call gro_move_change_gstat().
 * Note: the button press event isn't invoked on GroMove itself,
 * but on controller->gro_win window.
 **/
static void
move_push_cb(GroMove *gmove, const GroController *controller, GdkEventButton *event, gpointer data)
{
	gdk_beep();/* I want a nifty sound... */
	/*if (gmove->cur_gstat == GRO_STAT_SQUISH)
	   gro_move_change_gstat(gmove, GRO_STAT_DEAD);*/
	if (gmove->cur_gstat != GRO_STAT_DEAD) {
		guint next_gstat;
		next_gstat = gmove->cur_gstat + 1;
		if (next_gstat == gmove->num_gstat)
			gro_move_change_gstat(gmove, GRO_STAT_DEAD);
		else
			gro_move_change_gstat(gmove, next_gstat);
	} else {
		g_warning("Dead is pushed, I don't like it.\n");
	}
}

static void
turn_gmove(GroMove *gmove)
{
	const DefaultMoveEnv *move_env = gmove->move_env;
	gint cur_direct = gmove->cur_direct;/* not guint */
	int turn_base;

	turn_base = gmove->num_direct / 8;	/* heuristic */
    if (move_env->turn_left == TRUE) {
		cur_direct += RAND_INT(turn_base) + 1; /* between 1 and turn_base */
		if (cur_direct >= gmove->num_direct)
			cur_direct -= gmove->num_direct;
    } else {
		cur_direct -= RAND_INT(turn_base) + 1; /* between 1 and turn_base */
		if (cur_direct < 0)
			cur_direct += gmove->num_direct;
    }
	gro_move_change_direct(gmove, cur_direct);
}
