# -*- python -*-

### Copyright (C) 2005 Peter Williams <pwil3058@bigpond.net.au>

### This program is free software; you can redistribute it and/or modify
### it under the terms of the GNU General Public License as published by
### the Free Software Foundation; version 2 of the License only.

### This program is distributed in the hope that it will be useful,
### but WITHOUT ANY WARRANTY; without even the implied warranty of
### MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
### GNU General Public License for more details.

### You should have received a copy of the GNU General Public License
### along with this program; if not, write to the Free Software
### Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

import popen2, os, fcntl, select, signal, errno

def make_non_blocking(fd):
    fl = fcntl.fcntl(fd, fcntl.F_GETFL)
    try:
        fcntl.fcntl(fd, fcntl.F_SETFL, fl | os.O_NDELAY)
    except AttributeError:
        fcntl.fcntl(fd, fcntl.F_SETFL, fl | os.FNDELAY)

# Take advice from "Python Cookbook" on avoiding potential dead locks
# reading stdout and stderr from sub process
def run_cmd(cmd=None, input=None):
    if cmd == None:
        return [ 0, None, None ]
    savedsh = signal.getsignal(signal.SIGPIPE)
    signal.signal(signal.SIGPIPE, signal.SIG_DFL)
    sub = popen2.Popen3(cmd, True)
    if not input is None:
        sub.tochild.write(input)
    sub.tochild.close()
    outfd = sub.fromchild.fileno()
    errfd = sub.childerr.fileno()
    make_non_blocking(outfd)
    make_non_blocking(errfd)
    outd, errd = [], []
    outeof = erreof = False
    while True:
        to_check = [outfd] * (not outeof) + [errfd] * (not erreof)
        ready = select.select(to_check, [], [])
        if outfd in ready[0]:
            outchunk = sub.fromchild.read()
            if outchunk == '':
                outeof = True
            else:
                outd.append(outchunk)
        if errfd in ready[0]:
            outchunk = sub.childerr.read()
            if outchunk == '':
                erreof = True
            else:
                errd.append(outchunk)
        if outeof and erreof:
            break
        try:
            select.select([], [], [], 0.05)
        except select.error, data:
            if data[0] is errno.EINTR:
                pass
            else:
                return [ data[0], "", data[1] ]
    res = sub.wait()
    signal.signal(signal.SIGPIPE, savedsh)
    return [ res, ''.join(outd), ''.join(errd) ]

def is_hidden_file(f):
    return f[0] == '.'

def path_relative_to_dir(dir, path):
    if not os.path.isdir(dir):
        return None
    if dir == path:
        return "."
    lcwd = len(dir)
    if len(path) <= lcwd + 1 or dir != path[0:lcwd] or path[lcwd] != "/":
        return None
    return path[lcwd + 1:]

def dir_contents(dir):
    entries = os.listdir(dir)
    hfiles = []
    files = []
    hdirs = []
    dirs = []
    for e in entries:
        if os.path.isdir(dir + os.sep + e):
            if is_hidden_file(e):
                hdirs.append(e)
            else:
                dirs.append(e)
        elif is_hidden_file(e):
            hfiles.append(e)
        else:
            files.append(e)
    return (hdirs, dirs, hfiles, files)

def which(cmd):
    for d in os.environ['PATH'].split(':'):
        potential_path = d + "/" + cmd
        if os.path.isfile(potential_path) and os.access(potential_path, os.X_OK):
            return potential_path
    return None

class cmd_list:
    def __init__(self):
        self.cmds = self._get_commands()
    def _get_commands(self):
        assert 0, '_get_commands() must be provided in child class!!!'
    def has_cmd(self, cmd):
        try:
            self.cmds.index(cmd)
            return True
        except:
            return False
    def get_list(self):
        return self.cmds

editors_that_need_a_terminal = ["vi", "joe"]
default_editor = "gedit"
default_terminal = "gnome-terminal"

def _edit_files(filelist, editor_env_vars):
    edstr = default_editor
    for e in editor_env_vars:
        try:
            ed = os.environ[e]
            if ed is not "":
                edstr = ed
                break
        except KeyError:
            pass
    edlist = edstr.split()
    editor = edlist[0]
    options = edlist[1:]
    if not editor in editors_that_need_a_terminal:
        return apply(os.spawnlp, tuple([os.P_NOWAIT, editor, editor] + options + filelist))
    try:
        term  = os.environ['COLORTERM']
    except KeyError:
        try:
            term = os.environ['TERM']
        except KeyError:
            term = default_terminal
    return apply(os.spawnlp, tuple([os.P_NOWAIT, term, term, "-e", " ".join(edlist + filelist)]))

def edit_files(filelist):
    return _edit_files(filelist, ['GQUILT_EDITOR', 'EDITOR'])

def peruse_files(filelist):
    return _edit_files(filelist, ['GQUILT_PERUSER', 'GQUILT_EDITOR', 'EDITOR'])
