/*
 * GQmpeg
 * (C) 2002 John Ellis
 *
 * Author: John Ellis
 *
 * This software is released under the GNU General Public License (GNU GPL).
 * Please read the included file COPYING for more information.
 * This software comes with no warranty of any kind, use at your own risk!
 */


#include "gqmpeg.h"
#include "utilops.h"

#include "filelist.h"
#include "playlist.h"
#include "ui_fileops.h"
#include "ui_utildlg.h"

enum {
	DIALOG_NEW_DIR,
	DIALOG_COPY,
	DIALOG_MOVE,
	DIALOG_DELETE,
	DIALOG_RENAME
};

typedef struct _FileDataMult FileDataMult;
struct _FileDataMult
{
	gint confirm_all;
	gint confirmed;
	gint skip;
	GList *source_list;
	GList *source_next;
	gchar *dest_base;
	gchar *source;
	gchar *dest;
	gint copy;
};

typedef struct _FileDataSingle FileDataSingle;
struct _FileDataSingle
{
	gint confirmed;
	gchar *source;
	gchar *dest;
	gint copy;
};

static gint confirm_delete = TRUE; /* hard coded for now... */

/*
 *--------------------------------------------------------------------------
 * Move and Copy routines
 *--------------------------------------------------------------------------
 */

/*
 * Multi file move
 */

static FileDataMult *file_data_multiple_new(GList *source_list, gchar *dest, gint copy)
{
	FileDataMult *fdm = g_new0(FileDataMult, 1);
	fdm->confirm_all = FALSE;
	fdm->confirmed = FALSE;
	fdm->skip = FALSE;
	fdm->source_list = source_list;
	fdm->source_next = fdm->source_list;
	fdm->dest_base = g_strdup(dest);
	fdm->source = NULL;
	fdm->dest = NULL;
	fdm->copy = copy;
	return fdm;
}

static void file_data_multiple_free(FileDataMult *fdm)
{
	path_list_free(fdm->source_list);
	g_free(fdm->dest_base);
	g_free(fdm->dest);
	g_free(fdm);
}

static void file_util_move_multiple(FileDataMult *fdm);

static void file_util_move_multiple_ok_cb(GenericDialog *gd, gpointer data)
{
	FileDataMult *fdm = data;
	fdm->confirmed = TRUE;
	file_util_move_multiple(fdm);
}

static void file_util_move_multiple_all_cb(GenericDialog *gd, gpointer data)
{
	FileDataMult *fdm = data;
	fdm->confirm_all = TRUE;
	file_util_move_multiple(fdm);
}

static void file_util_move_multiple_skip_cb(GenericDialog *gd, gpointer data)
{
	FileDataMult *fdm = data;
	fdm->skip = TRUE;
	file_util_move_multiple(fdm);
}

static void file_util_move_multiple_cancel_cb(GenericDialog *gd, gpointer data)
{
	FileDataMult *fdm = data;
	file_data_multiple_free(fdm);
}

static void file_util_move_multiple(FileDataMult *fdm)
{
	while (fdm->dest || fdm->source_next)
		{
		if (!fdm->dest)
			{
			GList *work = fdm->source_next;
			fdm->source = work->data;
			fdm->dest = g_strconcat(fdm->dest_base, "/", filename_from_path(fdm->source), NULL);
			fdm->source_next = work->next;
			}

		if (isfile(fdm->dest) && !fdm->confirmed && !fdm->confirm_all && !fdm->skip)
			{
			GenericDialog *gd;
			gchar *text;

			text = g_strdup_printf(_("Overwrite file:\n %s\n with:\n %s"), fdm->dest, fdm->source);
			gd = generic_dialog_new(_("Overwrite file - GQmpeg"), text, "GQmpeg", "overwrite", TRUE,
						file_util_move_multiple_cancel_cb, fdm);
			g_free(text);

			generic_dialog_add(gd, _("Yes"), file_util_move_multiple_ok_cb, TRUE);
			generic_dialog_add(gd, _("Yes to all"), file_util_move_multiple_all_cb, FALSE);
			generic_dialog_add(gd, _("Skip"), file_util_move_multiple_skip_cb, FALSE);

			gtk_widget_show(gd->dialog);
			return;
			}
		else
			{
			gint success = FALSE;
			if (fdm->skip)
				{
				success = TRUE;
				fdm->skip = FALSE;
				}
			else
				{
				if (fdm->copy)
					{
					success = copy_file(fdm->source, fdm->dest);
					}
				else
					{
					if (move_file(fdm->source, fdm->dest))
						{
						success = TRUE;
/* --------- */					playlist_replace(fdm->source, fdm->dest);
						filelist_refresh();
						}
					}
				}
			if (!success)
				{
				GenericDialog *gd;
				gchar *title;
				gchar *text;
				if (fdm->copy)
					{
					title = _("Error copying file");
					text = g_strdup_printf(_("Unable to copy file:\n%sto:\n%s\n during multiple file copy."), fdm->source, fdm->dest);
					}
				else
					{
					title = _("Error moving file");
					text = g_strdup_printf(_("Unable to move file:\n%sto:\n%s\n during multiple file move."), fdm->source, fdm->dest);
					}
				gd = generic_dialog_new(title, text, "GQmpeg", "confirm", TRUE,
							file_util_move_multiple_cancel_cb, fdm);
				g_free(text);

				generic_dialog_add(gd, _("Continue"), file_util_move_multiple_skip_cb, TRUE);

				gtk_widget_show(gd->dialog);
				return;
				}
			fdm->confirmed = FALSE;
			g_free(fdm->dest);
			fdm->dest = NULL;
			}
		}

	file_data_multiple_free(fdm);
}

/*
 *-----------------
 * Single file move
 *-----------------
 */

static FileDataSingle *file_data_single_new(gchar *source, gchar *dest, gint copy)
{
	FileDataSingle *fds = g_new0(FileDataSingle, 1);
	fds->confirmed = FALSE;
	fds->source = g_strdup(source);
	fds->dest = g_strdup(dest);
	fds->copy = copy;
	return fds;
}

static void file_data_single_free(FileDataSingle *fds)
{
	g_free(fds->source);
	g_free(fds->dest);
	g_free(fds);
}

static void file_util_move_single(FileDataSingle *fds);

static void file_util_move_single_ok_cb(GenericDialog *gd, gpointer data)
{
	FileDataSingle *fds = data;
	fds->confirmed = TRUE;
	file_util_move_single(fds);
}

static void file_util_move_single_cancel_cb(GenericDialog *gd, gpointer data)
{
	FileDataSingle *fds = data;
	file_data_single_free(fds);
}

static void file_util_move_single(FileDataSingle *fds)
{
	if (isfile(fds->dest) && !fds->confirmed)
		{
		GenericDialog *gd;
		gchar *text;

		text = g_strdup_printf(_("Overwrite file:\n%s\n with:\n%s"), fds->dest, fds->source);
		gd = generic_dialog_new(_("Overwrite file - GQmpeg"), text, "GQmpeg", "overwrite", TRUE,
					file_util_move_single_cancel_cb, fds);
		g_free(text);
		generic_dialog_add(gd, _("Overwrite"), file_util_move_single_ok_cb, TRUE);
		return;
		}
	else
		{
		gint success = FALSE;
		if (fds->copy)
			{
			success = copy_file(fds->source, fds->dest);
			}
		else
			{
			if (move_file(fds->source, fds->dest))
				{
				success = TRUE;
/* --------- */			playlist_replace(fds->source, fds->dest);
				filelist_refresh();
				}
			}
		if (!success)
			{
			gchar *title;
			gchar *text;
			if (fds->copy)
				{
				title = _("Error copying file");
				text = g_strdup_printf(_("Unable to copy file:\n%s\nto:\n%s"), fds->source, fds->dest);
				}
			else
				{
				title = _("Error moving file");
				text = g_strdup_printf(_("Unable to move file:\n%s\nto:\n%s"), fds->source, fds->dest);
				}
			warning_dialog(title, text);
			g_free(text);
			}
		file_data_single_free(fds);
		}
}

/*
 *-----------------
 * file move dialog
 *-----------------
 */

static void file_util_move_do(FileDialog *fd)
{
	file_dialog_sync_history(fd, FALSE);

	if (fd->multiple_files)
		{
		file_util_move_multiple(file_data_multiple_new(fd->source_list, fd->dest_path, fd->type));
		fd->source_list = NULL;
		}
	else
		{
		if (isdir(fd->dest_path))
			{
			gchar *buf = g_strconcat(fd->dest_path, "/", filename_from_path(fd->source_path), NULL);
			g_free(fd->dest_path);
			fd->dest_path = buf;
			}
		file_util_move_single(file_data_single_new(fd->source_path, fd->dest_path, fd->type));
		}

	file_dialog_close(fd);
}

static void file_util_move_check(FileDialog *fd)
{
	if (fd->multiple_files && !isdir(fd->dest_path))
		{
		if (isfile(fd->dest_path))
			warning_dialog(_("Invalid destination"), _("When operating with multiple files, please select\n a directory, not file."));
		else
			warning_dialog(_("Invalid directory"), _("Please select an existing directory"));
		return;
		}

	file_util_move_do(fd);
}

static void file_util_cancel_cb(FileDialog *fd, gpointer data)
{
	file_dialog_close(fd);
}

static void file_util_move_cb(FileDialog *fd, gpointer data)
{
	file_util_move_check(fd);
}

static void real_file_util_move(const gchar *source_path, GList *source_list, const gchar *dest_path, gint copy)
{
	FileDialog *fd;
	gchar *path = NULL;
	const gchar *title;
	gchar *text;
	const gchar *op_text;

	if (!source_path && !source_list) return;

	if (source_path)
		{
		path = g_strdup(source_path);
		}
	else if (!source_list->next)
		{
		path = g_strdup(source_list->data);
		path_list_free(source_list);
		source_list = NULL;
		}

	if (copy)
		{
		title = _("Copy - GQmpeg");
		op_text = _("Copy");
		if (path)
			text = g_strdup_printf(_("Copy file:\n%s\nto:"), path);
		else
			text = g_strdup(_("Copy multiple files to:"));
		}
	else
		{
		title = _("Move - GQmpeg");
		op_text = _("Move");
		if (path)
			text = g_strdup_printf(_("Move file:\n%s\nto:"), path);
		else
			text = g_strdup(_("Move multiple files to:"));
		}

	fd = file_dialog_new(title, text, "GQmpeg", "copy_move",
			     file_util_cancel_cb, NULL);
	g_free(text);

	file_dialog_add_button(fd, op_text, file_util_move_cb, TRUE);

	fd->type = copy;
	fd->source_path = path;
	fd->source_list = source_list;
	fd->multiple_files = (path == NULL);

	file_dialog_add_path_widgets(fd, NULL, dest_path, "move_copy", NULL, NULL);

	gtk_widget_show(GENERIC_DIALOG(fd)->dialog);
}

void file_util_move(gchar *source_path, GList *source_list, gchar *dest_path)
{
	real_file_util_move(source_path, source_list, dest_path, FALSE);
}

void file_util_copy(gchar *source_path, GList *source_list, gchar *dest_path)
{
	real_file_util_move(source_path, source_list, dest_path, TRUE);
}

/*
 *--------------------------------------------------------------------------
 * Delete routines
 *--------------------------------------------------------------------------
 */

/*
 *----------------------
 * delete multiple files
 *----------------------
 */

static void file_util_delete_multiple_cancel_cb(GenericDialog *gd, gpointer data)
{
	GList *source_list = data;
	path_list_free(source_list);
}

static void file_util_delete_multiple_ok_cb(GenericDialog *gd, gpointer data)
{
	GList *source_list = data;

	while(source_list)
		{
		gchar *path = source_list->data;
		source_list = g_list_remove(source_list, path);
		if (unlink (path) < 0)
			{
			GenericDialog *gd;
			gchar *text;
			if (source_list)
				{
				text = g_strdup_printf(_("Unable to delete file:\n %s\n Continue multiple delete operation?"), path);
				gd = generic_dialog_new(_("Delete failed"), text,
							"GQmpeg", "delete_confirm", TRUE,
							file_util_delete_multiple_cancel_cb, source_list);
				generic_dialog_add(gd, _("Continue"), file_util_delete_multiple_ok_cb, TRUE);
				gtk_widget_show(gd->dialog);
				}
			else
				{
				text = g_strdup_printf(_("Unable to delete file:\n%s"), path);
				warning_dialog(_("Delete failed"), text);
				}
			g_free(text);
			g_free(path);
			return;
			}
		else
			{
/* --------- */		playlist_remove(path, 0, TRUE);
			filelist_refresh();
			}
		g_free(path);
		}
}

static void file_util_delete_multiple(GList *source_list)
{
	if (!confirm_delete)
		{
		file_util_delete_multiple_ok_cb(NULL, source_list);
		}
	else
		{
		GenericDialog *gd;
		gd = generic_dialog_new(_("Delete files"), _("About to delete multiple files..."),
					"GQmpeg", "delete_confirm", TRUE,
					file_util_delete_multiple_cancel_cb, source_list);
		generic_dialog_add(gd, _("Delete"), file_util_delete_multiple_ok_cb, TRUE);

		gtk_widget_show(gd->dialog);
		}
}

/*
 *-------------------
 * delete single file
 *-------------------
 */

static void file_util_delete_cancel_cb(GenericDialog *gd, gpointer data)
{
	gchar *path = data;
	g_free(path);
}

static void file_util_delete_ok_cb(GenericDialog *gd, gpointer data)
{
	gchar *path = data;

	if (unlink(path) < 0)
		{
		gchar *text = g_strdup_printf(_("Unable to delete file:\n%s"), path);
		warning_dialog(_("File deletion failed"), text);
		g_free(text);
		}
	else
		{
/* --------- */	playlist_remove(path, 0, TRUE);
		filelist_refresh();
		}

	file_util_delete_cancel_cb(gd, data);
}

static void file_util_delete_single(gchar *path)
{
	gchar *buf = g_strdup(path);

	if (!confirm_delete)
		{
		file_util_delete_ok_cb(NULL, buf);
		}
	else
		{
		GenericDialog *gd;
		gchar *text;

		text = g_strdup_printf(_("About to delete the file:\n %s"), buf);
		gd = generic_dialog_new(_("Delete file - GQmpeg"), text,
					"GQmpeg", "delete_confirm", TRUE,
					file_util_delete_cancel_cb, buf);
		generic_dialog_add(gd, _("Delete"), file_util_delete_ok_cb, TRUE);
		g_free(text);

		gtk_widget_show(gd->dialog);
		}
}

void file_util_delete(gchar *source_path, GList *source_list)
{
	if (!source_path && !source_list) return;

	if (source_path)
		{
		file_util_delete_single(source_path);
		}
	else if (!source_list->next)
		{
		file_util_delete_single(source_list->data);
		path_list_free(source_list);
		}
	else
		{
		file_util_delete_multiple(source_list);
		}
}

/*
 *--------------------------------------------------------------------------
 * Rename routines
 *--------------------------------------------------------------------------
 */

/*
 *-------------------
 * rename multiple files
 *-------------------
 */

static void file_util_rename_multiple(FileDialog *fd);

static void file_util_rename_multiple_close_cb(FileDialog *fd, gpointer data)
{
	file_dialog_close(fd);
}

static void file_util_rename_multiple_cancel_cb(GenericDialog *gd, gpointer data)
{
	FileDialog *fd = data;

	if (!GTK_WIDGET_VISIBLE(GENERIC_DIALOG(fd)->dialog)) gtk_widget_show(GENERIC_DIALOG(fd)->dialog);
}

static void file_util_rename_multiple_ok_cb(GenericDialog *gd, gpointer data)
{
	FileDialog *fd = data;

	file_util_rename_multiple_cancel_cb(gd, data);

	fd->type = TRUE;
	file_util_rename_multiple(fd);
}

static void file_util_rename_multiple(FileDialog *fd)
{
	if (isfile(fd->dest_path) && !fd->type)
		{
		GenericDialog *gd;
		gchar *text;

		text = g_strdup_printf(_("Overwrite file:\n%s\nby renaming:\n%s"), fd->dest_path, fd->source_path);
		gd = generic_dialog_new(_("Overwrite file - GQmpeg"), text,
					"GQmpeg", "overwrite", TRUE,
					file_util_rename_multiple_cancel_cb, fd);
		g_free(text);

		generic_dialog_add(gd, _("Overwrite"), file_util_rename_multiple_ok_cb, TRUE);
		gtk_widget_hide(GENERIC_DIALOG(fd)->dialog);
		return;
		}
	else
		{
		if (rename(fd->source_path, fd->dest_path) < 0)
			{
			gchar *text = g_strdup_printf(_("Unable to rename file:\n%s\n to:\n%s"),
						      filename_from_path(fd->source_path),
						      filename_from_path(fd->dest_path));
			warning_dialog(_("Error renaming file"), text);
			g_free(text);
			}
		else
			{
			gint row;
			gint n;
			GtkWidget *clist;
			gchar *path;

/* --------- */		playlist_replace(fd->source_path, fd->dest_path);
			filelist_refresh();

			clist = gtk_object_get_user_data(GTK_OBJECT(fd->entry));
			path = GENERIC_DIALOG(fd)->data;
			row = gtk_clist_find_row_from_data(GTK_CLIST(clist), path);

			n = g_list_length(GTK_CLIST(clist)->row_list);
			if (debug_mode) printf("r=%d n=%d\n", row, n);
			if (n - 1 > row)
				n = row;
			else if (n > 1)
				n = row - 1;
			else
				n = -1;

			if (n >= 0)
				{
				GENERIC_DIALOG(fd)->data = NULL;
				gtk_clist_remove(GTK_CLIST(clist), row);
				gtk_clist_select_row(GTK_CLIST(clist), n, -1);
				}
			else
				{
				if (debug_mode) printf("closed by #%d\n", n);
				file_dialog_close(fd);
				}
			}
		}
}

static void file_util_rename_multiple_rename_cb(FileDialog *fd, gpointer data)
{
	gchar *base;
	const gchar *name;

	name = gtk_entry_get_text(GTK_ENTRY(fd->entry));

	base = remove_level_from_path(fd->source_path);
	g_free(fd->dest_path);
	fd->dest_path = g_strconcat(base, "/", name, NULL);
	g_free(base);

	if (strlen(name) == 0 || strcmp(fd->source_path, fd->dest_path) == 0) return;

	fd->type = FALSE;
	file_util_rename_multiple(fd);
}

static void file_util_rename_multiple_select_cb(GtkWidget *clist, gint row, gint column,
						GdkEventButton *bevent, gpointer data)
{
	FileDialog *fd = data;
	GtkWidget *label;
	const gchar *name;
	gchar *path;

	label = gtk_object_get_user_data(GTK_OBJECT(GENERIC_DIALOG(fd)->dialog));
	path = gtk_clist_get_row_data(GTK_CLIST(clist), row);

	g_free(fd->source_path);
	fd->source_path = g_strdup(path);

	GENERIC_DIALOG(fd)->data = path;

	name = filename_from_path(fd->source_path);
	gtk_label_set(GTK_LABEL(label), name);
	gtk_entry_set_text(GTK_ENTRY(fd->entry), name);

	gtk_widget_grab_focus(fd->entry);
}

static void file_util_rename_multiple_do(GList *source_list)
{
	FileDialog *fd;
	GtkWidget *scrolled;
	GtkWidget *clist;
	GtkWidget *label;
	GList *work;

	fd = file_dialog_new(_("GQmpeg - rename"), _("Rename multiple files:"),
			     "GQmpeg", "rename_multiple",
			     file_util_rename_multiple_close_cb, NULL);

	file_dialog_add_button(fd, _("Rename"), file_util_rename_multiple_rename_cb, TRUE);

	fd->source_path = g_strdup(source_list->data);
	fd->dest_path = NULL;

	scrolled = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled),
				GTK_POLICY_AUTOMATIC, GTK_POLICY_ALWAYS);
	gtk_box_pack_start(GTK_BOX(GENERIC_DIALOG(fd)->vbox), scrolled, TRUE, TRUE, 0);
	gtk_widget_show(scrolled);

	clist=gtk_clist_new (1);
	gtk_clist_set_column_auto_resize(GTK_CLIST(clist), 0, TRUE);
	gtk_signal_connect(GTK_OBJECT(clist), "select_row",
			   GTK_SIGNAL_FUNC(file_util_rename_multiple_select_cb), fd);
	gtk_widget_set_usize(clist, 250, 150);
	gtk_container_add(GTK_CONTAINER(scrolled), clist);
	gtk_widget_show(clist);

	GENERIC_DIALOG(fd)->data = source_list->data;

	work = source_list;
	while(work)
		{
		gint row;
		gchar *buf[2];
		buf[0] = (gchar *)filename_from_path(work->data);
		buf[1] = NULL;
		row = gtk_clist_append(GTK_CLIST(clist), buf);
		gtk_clist_set_row_data_full(GTK_CLIST(clist), row,
					    work->data, (GtkDestroyNotify)g_free);
		work = work->next;
		}

	g_list_free(source_list);

	label = gtk_label_new(_("Rename:"));
	gtk_box_pack_start(GTK_BOX(GENERIC_DIALOG(fd)->vbox), label, FALSE, FALSE, 0);
	gtk_widget_show(label);

	label = gtk_label_new(filename_from_path(fd->source_path));
	gtk_box_pack_start(GTK_BOX(GENERIC_DIALOG(fd)->vbox), label, FALSE, FALSE, 0);
	gtk_widget_show(label);
	gtk_object_set_user_data(GTK_OBJECT(GENERIC_DIALOG(fd)->dialog), label);

	label = gtk_label_new(_("to:"));
	gtk_box_pack_start(GTK_BOX(GENERIC_DIALOG(fd)->vbox), label, FALSE, FALSE, 0);
	gtk_widget_show(label);

	fd->entry = gtk_entry_new();
	gtk_entry_set_text(GTK_ENTRY(fd->entry), filename_from_path(fd->source_path));
	gtk_box_pack_start(GTK_BOX(GENERIC_DIALOG(fd)->vbox), fd->entry, FALSE, FALSE, 0);
	generic_dialog_attach_default(GENERIC_DIALOG(fd), fd->entry);
	gtk_widget_grab_focus(fd->entry);
	gtk_widget_show(fd->entry);

	gtk_object_set_user_data(GTK_OBJECT(fd->entry), clist);

	gtk_widget_show(GENERIC_DIALOG(fd)->dialog);
}

/*
 *-------------------
 * rename single file
 *-------------------
 */

static void file_util_rename_single(FileDataSingle *fds);

static void file_util_rename_single_ok_cb(GenericDialog *gd, gpointer data)
{
	FileDataSingle *fds = data;
	fds->confirmed = TRUE;
	file_util_rename_single(fds);
}

static void file_util_rename_single_close_cb(GenericDialog *gd, gpointer data)
{
	FileDataSingle *fds = data;
	file_data_single_free(fds);
}

static void file_util_rename_single(FileDataSingle *fds)
{
	if (isfile(fds->dest) && !fds->confirmed)
		{
		GenericDialog *gd;
		gchar *text = g_strdup_printf(_("Overwrite file:\n%s\nwith:\n%s"), fds->dest, fds->source);
		gd = generic_dialog_new(_("Overwrite file - GQmpeg"), text,
					"GQmpeg", "overwrite", TRUE,
					file_util_rename_single_close_cb, fds);
		g_free(text);
		generic_dialog_add(gd, _("Overwrite"), file_util_rename_single_ok_cb, TRUE);
		gtk_widget_show(gd->dialog);
		return;
		}
	else
		{
		if (rename(fds->source, fds->dest) < 0)
			{
			gchar *text = g_strdup_printf(_("Unable to rename file:\n%s\nto:\n%s"), filename_from_path(fds->source), filename_from_path(fds->dest));
			warning_dialog(_("Error renaming file"), text);
			g_free(text);
			}
		else
			{
/* --------- */		playlist_replace(fds->source, fds->dest);
			filelist_refresh();
			}
		}
	file_data_single_free(fds);
}

static void file_util_rename_close_cb(FileDialog *fd, gpointer data)
{
	file_dialog_close(fd);
}

static void file_util_rename_single_cb(FileDialog *fd, gpointer data)
{
	const gchar *name;
	gchar *dest;

	name = gtk_entry_get_text(GTK_ENTRY(fd->entry));
	dest = g_strconcat(fd->dest_path, "/", name, NULL);

	if (strlen(name) == 0 || strcmp(fd->source_path, dest) == 0)
		{
		g_free(dest);
		return;
		}

	fd->type = FALSE;
	file_util_rename_single(file_data_single_new(fd->source_path, dest, fd->type));
	g_free(dest);
	file_util_rename_close_cb(fd, data);
}

static void file_util_rename_single_do(gchar *source_path)
{
	FileDialog *fd;
	gchar *text;
	const gchar *name = filename_from_path(source_path);

	text = g_strdup_printf(_("Rename file:\n%s\nto:"), name);
	fd = file_dialog_new(_("Rename - GQmpeg"), text,
			     "GQmpeg", "rename",
			     file_util_rename_close_cb, NULL);
	g_free(text);

	file_dialog_add_button(fd, _("Rename"), file_util_rename_single_cb, TRUE);

	fd->source_path = g_strdup(source_path);
	fd->dest_path = remove_level_from_path(source_path);

	fd->entry = gtk_entry_new();
	gtk_entry_set_text(GTK_ENTRY(fd->entry), name);
	gtk_box_pack_start(GTK_BOX(GENERIC_DIALOG(fd)->vbox), fd->entry, FALSE, FALSE, 0);
	generic_dialog_attach_default(GENERIC_DIALOG(fd), fd->entry);
	gtk_widget_grab_focus(fd->entry);
	gtk_widget_show(fd->entry);

	gtk_widget_show(GENERIC_DIALOG(fd)->dialog);
}

void file_util_rename(gchar *source_path, GList *source_list)
{
	if (!source_path && !source_list) return;

	if (source_path)
		{
		file_util_rename_single_do(source_path);
		}
	else if (!source_list->next)
		{
		file_util_rename_single_do(source_list->data);
		path_list_free(source_list);
		}
	else
		{
		file_util_rename_multiple_do(source_list);
		}
}

/*
 *--------------------------------------------------------------------------
 * Create directory routines
 *--------------------------------------------------------------------------
 */

static void file_util_create_dir_do(gchar *source, gchar *dest)
{
	if (isfile(dest))
		{
		gchar *text = g_strdup_printf(_("The path:\n%s\nalready exists as a file."), filename_from_path(dest));
		warning_dialog(_("Could not create directory"), text);
		g_free(text);
		}
	else if (isdir(dest))
		{
		gchar *text = g_strdup_printf(_("The directory:\n%s\nalready exists."), filename_from_path(dest));
		warning_dialog(_("Directory exists"), text);
		g_free(text);
		}
	else
		{
		if (mkdir(dest, 0755) < 0)
			{
			gchar *text = g_strdup_printf(_("Unable to create directory:\n%s"), filename_from_path(dest));
			warning_dialog(_("Error creating directory"), text);
			g_free(text);
			}
		else
			{
			if (strcmp(source, current_path) == 0)
				{
				filelist_refresh();
				}
			}
		}
}

static void file_util_create_dir_close_cb(FileDialog *fd, gpointer data)
{
	g_free(data);
	file_dialog_close(fd);
}

static void file_util_create_dir_create_cb(FileDialog *fd, gpointer data)
{
	gchar *source = data;
	gchar *dest;
	const gchar *name;

	name = gtk_entry_get_text(GTK_ENTRY(fd->entry));

	if (strlen(name) == 0) return;

	dest = g_strconcat(source, "/", name, NULL);
	file_util_create_dir_do(source, dest);
	g_free(dest);

	file_util_create_dir_close_cb(fd, data);
}

void file_util_create_dir(gchar *path)
{
	FileDialog *fd;
	gchar *text;
	const gchar *name;

	if (!isdir(path)) return;
	name = filename_from_path(path);

	text = g_strdup_printf(_("Create directory in:\n%s\nnamed:"), path);
	fd = file_dialog_new(_("New directory - GQmpeg"), text,
			     "GQmpeg", "new_directory",
			     file_util_create_dir_close_cb, g_strdup(path));
	g_free(text);
	file_dialog_add_button(fd, _("Create"), file_util_create_dir_create_cb, TRUE);

	fd->entry = gtk_entry_new();
	gtk_box_pack_start(GTK_BOX(GENERIC_DIALOG(fd)->vbox), fd->entry, FALSE, FALSE, 0);
	generic_dialog_attach_default(GENERIC_DIALOG(fd), fd->entry);
	gtk_widget_grab_focus(fd->entry);
	gtk_widget_show(fd->entry);

	gtk_widget_show(GENERIC_DIALOG(fd)->dialog);
}


