/*
 * (SLIK) SimpLIstic sKin functions
 * (C) 2002 John Ellis
 *
 * Author: John Ellis
 *
 * This software is released under the GNU General Public License (GNU GPL).
 * Please read the included file COPYING for more information.
 * This software comes with no warranty of any kind, use at your own risk!
 */

#include <pwd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>
#include <sys/param.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <sys/types.h>
#include <dirent.h>

#include <glib.h>

#include "ui_fileops.h"

/*
 *-----------------------------------------------------------------------------
 * generic file information and manipulation routines (public)
 *-----------------------------------------------------------------------------
 */ 

/* first we try the HOME environment var, if that doesn't work, we try getpwuid(). */
const gchar *homedir(void)
{
	gchar *home = getenv("HOME");
	if(home)
		return home;
	else
		{
		struct passwd *pw = getpwuid(getuid());
		if (pw)
			return pw->pw_dir;
		else
			return NULL ; /* now we've got a problem */
		}
}

gint isname(const gchar *s)
{
   struct stat st;

   if ((!s)||(!*s)) return 0;
   if (stat(s,&st)<0) return 0;
   return 1;
}

gint isfile(const gchar *s)
{
   struct stat st;
   
   if ((!s)||(!*s)) return 0;
   if (stat(s,&st)<0) return 0;
   if (S_ISREG(st.st_mode)) return 1;
   return 0;
}

gint isdir(const gchar *s)
{
   struct stat st;
   
   if ((!s)||(!*s)) return 0;
   if (stat(s,&st)<0) return 0;
   if (S_ISDIR(st.st_mode)) return 1;
   return 0;
}

gint filesize(const gchar *s)
{
   struct stat st;
   
   if ((!s)||(!*s)) return 0;
   if (stat(s,&st)<0) return 0;
   return (int)st.st_size;
}

time_t filetime(const gchar *s)
{
        struct stat st;

        if ((!s)||(!*s)) return 0;
        if (stat(s,&st)<0) return 0;
        return st.st_mtime;
}

gint copy_file(const gchar *s, const gchar *t)
{
	FILE *fi, *fo;
	gchar buf[4096];
	gint b;

	fi = fopen(s, "rb");
	if (!fi)
		{
		return FALSE;
		}

	fo = fopen(t, "wb");
	if (!fo)
		{
		fclose(fi);
		return FALSE;
		}

	while((b = fread(buf, sizeof(char), 4096, fi)) && b != 0)
		{
		if (fwrite(buf, sizeof(char), b, fo) != b)
			{
			fclose(fi);
			fclose(fo);
			return FALSE;
			}
		}

	fclose(fi);
	fclose(fo);
	return TRUE;
}

gint move_file(const gchar *s, const gchar *t)
{
	if (rename (s, t) < 0)
		{
		/* this may have failed because moving a file across filesystems
		was attempted, so try copy and delete instead */
		if (copy_file(s, t))
			{
			if (unlink (s) < 0)
				{
				/* err, now we can't delete the source file so return FALSE */
				return FALSE;
				}
			}
		else
			return FALSE;
		}

	return TRUE;
}

gchar *get_current_dir(void)
{
	return g_get_current_dir();
}

gint path_list(const gchar *path, GList **files, GList **dirs)
{
	DIR *dp;
	struct dirent *dir;
	struct stat ent_sbuf;
	GList *f_list = NULL;
	GList *d_list = NULL;

	if (!path) return FALSE;

	if((dp = opendir(path))==NULL)
		{
		/* dir not found */
		return FALSE;
		}

	/* root dir fix */
	if (path[0] == '/' && path[1] == '\0') path = "";

	while ((dir = readdir(dp)) != NULL)
		{
		/* skip removed files */
		if (dir->d_ino > 0)
			{
			gchar *name = dir->d_name;
			gchar *filepath = g_strconcat(path, "/", name, NULL);
			if (stat(filepath, &ent_sbuf) >= 0)
				{
				if (dirs && S_ISDIR(ent_sbuf.st_mode) &&
				    !(name[0] == '.' && (name[1] == '\0' || (name[1] == '.' && name[2] == '\0'))) )
					{
					d_list = g_list_prepend(d_list, filepath);
					filepath = NULL;
                       	                }
                               	else if (files && S_ISREG(ent_sbuf.st_mode))
					{
					f_list = g_list_prepend(f_list, filepath);
					filepath = NULL;
					}
				g_free(filepath);
				}
			}
		}
	closedir(dp);

	if (dirs) *dirs = g_list_reverse(d_list);
	if (files) *files = g_list_reverse(f_list);

	return TRUE;
}

void path_list_free(GList *list)
{
	g_list_foreach(list, (GFunc)g_free, NULL);
	g_list_free(list);
}

long checksum_simple(const gchar *path)
{
	FILE *f;
	long sum = 0;
	gint c;

	f = fopen(path, "r");
	if (!f) return -1;

	while((c = fgetc(f)) != EOF)
		{
		sum += c;
		}

	fclose(f);

	return sum;
}

gchar *unique_filename(const gchar *path, const gchar *ext, const gchar *divider, gint pad)
{
	gchar *unique;
	gint n = 1;

	if (!ext) ext = "";
	if (!divider) divider = "";

	unique = g_strconcat(path, ext, NULL);
	while (isname(unique))
		{
		g_free(unique);
		if (pad)
			{
			unique = g_strdup_printf("%s%s%03d%s", path, divider, n, ext);
			}
		else
			{
			unique = g_strdup_printf("%s%s%d%s", path, divider, n, ext);
			}
		n++;
		if (n > 999)
			{
			/* well, we tried */
			g_free(unique);
			return NULL;
			}
		}

	return unique;
}

const gchar *filename_from_path(const gchar *path)
{
	/* g_base_name warns about NULL, but we accept it */
	if (!path) return NULL;

	return g_basename(path);
}

gchar *remove_level_from_path(const gchar *path)
{
	gchar *new_path;
	const gchar *ptr = path;
	gint p;

	if (!path) return NULL;

	p = strlen(path) - 1;
	if (p < 0) return NULL;
	while(ptr[p] != '/' && p > 0) p--;
	if (p == 0 && ptr[p] == '/') p++;
	new_path = g_strndup(path, (guint)p);
	return new_path;
}

gchar *concat_dir_and_file(const gchar *base, const gchar *name)
{
	if (!base || !name) return NULL;

	if (strcmp(base, "/") == 0) return g_strconcat(base, name, NULL);

	return g_strconcat(base, "/", name, NULL);
}

gchar *remove_extension_from_path(const gchar *path)
{
	gchar *new_path;
	const gchar *ptr = path;
	gint p;

	if (!path) return NULL;
	if (strlen(path) < 2) return g_strdup(path);

	p = strlen(path) - 1;
	while(ptr[p] != '.' && p > 0) p--;
	if (p == 0) p = strlen(path) - 1;
	new_path = g_strndup(path, (guint)p);
	return new_path;
}

void parse_out_relatives(gchar *path)
{
	gint s, t;

	if (!path) return;

	s = t = 0;

	while (path[s] != '\0')
		{
		if (path[s] == '/' && path[s+1] == '.' && (path[s+2] == '/' || path[s+2] == '\0') )
			{
			s += 2;
			}
		else if (path[s] == '/' && path[s+1] == '.' && path[s+2] == '.' && (path[s+3] == '/' || path[s+3] == '\0') )
			{
			s += 3;
			if (t > 0) t--;
			while (path[t] != '/' && t > 0) t--;
			}
		else
			{
			if (s != t) path[t] = path[s];
			t++;
			s++;
			}
		}
	if (t == 0 && path[t] == '/') t++;
	if (t > 1 && path[t-1] == '/') t--;
	path[t] = '\0';
}

gint file_in_path(const gchar *name)
{
	gchar *path = g_strdup(getenv("PATH"));
	gint p, l;

	if (!path) return FALSE;

	p = 0;
	l = strlen(path);
	while (p < l)
		{
		gchar *f;
		gint e = p;
		while (path[e] != ':' && path[e] != '\0') e++;
		path[e] = '\0';
		e++;
		f = g_strconcat(path + p, "/", name, NULL);
		if (isfile(f))
			{
			g_free(f);
			g_free(path);
			return TRUE;
			}
		g_free(f);
		p = e;
		}
	g_free(path);
	return FALSE;
}

