/*
    GQ -- a GTK-based LDAP client
    Copyright (C) 1998-2001 Bert Vermeulen

    This program is released under the Gnu General Public License with
    the additional exemption that compiling, linking, and/or using
    OpenSSL is allowed.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/


#include <config.h>

#ifdef HAVE_LDAP_STR2OBJECTCLASS

#include <lber.h>
#include <ldap.h>
#include <ldap_schema.h>

#include <string.h>

#include <glib.h>
#include <gtk/gtk.h>

#include "mainwin.h"
#include "configfile.h"
#include "common.h"
#include "util.h"
#include "schemabrowse.h"
#include "schema.h"
#include "debug.h"


extern GtkWidget *statusbar, *mainwin;
extern struct gq_config config;



void new_schemamode(GHashTable *hash)
{
     GtkWidget *schemamode_vbox, *rightpane_vbox, *spacer;
     GtkWidget *mainpane, *treeroot;
     GtkWidget *leftpane_scrwin, *rightpane_scrwin;

     schemamode_vbox = gtk_vbox_new(FALSE, 0);

     spacer = gtk_hbox_new(FALSE, 0);
     gtk_widget_show(spacer);
     gtk_box_pack_start(GTK_BOX(schemamode_vbox), spacer, FALSE, FALSE, 3);

     mainpane = gtk_hpaned_new();
     gtk_container_border_width(GTK_CONTAINER(mainpane), 2);
     gtk_widget_show(mainpane);
     gtk_box_pack_start(GTK_BOX(schemamode_vbox), mainpane, TRUE, TRUE, 0);

     treeroot = gtk_tree_new();
     gtk_widget_show(treeroot);
     add_schema_servers(hash, treeroot);

     leftpane_scrwin = gtk_scrolled_window_new(NULL, NULL);
     gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(leftpane_scrwin),
                                    GTK_POLICY_AUTOMATIC,
                                    GTK_POLICY_AUTOMATIC);
     gtk_widget_show(leftpane_scrwin);
     gtk_paned_set_position(GTK_PANED(mainpane), 300);

     gtk_paned_add1(GTK_PANED(mainpane), leftpane_scrwin);
     gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(leftpane_scrwin),
                                           treeroot);

     rightpane_scrwin = gtk_scrolled_window_new(NULL, NULL);
     gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(rightpane_scrwin),
                                    GTK_POLICY_AUTOMATIC,
                                    GTK_POLICY_AUTOMATIC);
     gtk_widget_show(rightpane_scrwin);

     rightpane_vbox = gtk_vbox_new(FALSE, 5);
     gtk_widget_show(rightpane_vbox);
     g_hash_table_insert(hash, "rightpane_vbox", rightpane_vbox);
     gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(rightpane_scrwin),
                                           rightpane_vbox);
     gtk_paned_add2(GTK_PANED(mainpane), rightpane_scrwin);

     gtk_object_set_data(GTK_OBJECT(schemamode_vbox), "focus", rightpane_scrwin);
     gtk_widget_show(schemamode_vbox);

     g_hash_table_insert(hash, "vbox", schemamode_vbox);
}


void add_schema_servers(GHashTable *hash, GtkWidget *treeroot)
{
     GtkWidget *new_item, *new_subtree;
     struct ldapserver *server;
     int server_cnt;
     char message[128];

     server_cnt = 0;
     server = config.ldapservers;
     while(server) {
	  new_item = gtk_tree_item_new_with_label(server->name);
	  GTK_WIDGET_UNSET_FLAGS(new_item, GTK_CAN_FOCUS);
	  gtk_signal_connect(GTK_OBJECT(new_item), "button_press_event",
			     GTK_SIGNAL_FUNC(schema_button_tree), NULL);
	  gtk_object_set_data(GTK_OBJECT(new_item), "server", (gpointer) server);

	  gtk_tree_append(GTK_TREE(treeroot), new_item);
	  new_subtree = gtk_tree_new();
	  gtk_widget_show(new_subtree);
	  gtk_tree_item_set_subtree(GTK_TREE_ITEM(new_item), new_subtree);
	  gtk_widget_show(new_item);
	  gtk_signal_connect(GTK_OBJECT(new_item), "expand",
			     GTK_SIGNAL_FUNC(attach_server_schema), hash);

	  server = server->next;
	  server_cnt++;
     }

     make_message(message, server_cnt, "server", "servers", "found");
     statusbar_msg(message);

}


void attach_server_schema(GtkWidget *item, GHashTable *hash)
{
     LDAPObjectClass *oc;
     LDAPAttributeType *at;
     LDAPMatchingRule *mr;
     LDAPSyntax *s;
     GList *tmp;
     GtkWidget *new_item, *server_subtree, *new_subtree;
     struct ldapserver *server;
     struct server_schema *ss;

     gtk_signal_disconnect_by_func(GTK_OBJECT(item),
				   GTK_SIGNAL_FUNC(attach_server_schema), hash);

     if( (server = gtk_object_get_data(GTK_OBJECT(item), "server")) == NULL)
	  return;

     set_busycursor();

     ss = get_server_schema(server);
     if(ss) {

	  server_subtree = GTK_TREE_ITEM_SUBTREE(GTK_TREE_ITEM(item));
	  if(ss->oc) {
	       new_item = gtk_tree_item_new_with_label("objectClasses");
	       GTK_WIDGET_UNSET_FLAGS(new_item, GTK_CAN_FOCUS);
	       gtk_signal_connect(GTK_OBJECT(new_item), "button_press_event",
				  GTK_SIGNAL_FUNC(schema_button_tree), NULL);
	       gtk_widget_show(new_item);
	       gtk_tree_append(GTK_TREE(server_subtree), new_item);

	       new_subtree = gtk_tree_new();
	       GTK_WIDGET_UNSET_FLAGS(new_subtree, GTK_CAN_FOCUS);
	       gtk_object_set_data(GTK_OBJECT(new_subtree), "server", server);
	       gtk_widget_show(new_subtree);
	       gtk_tree_item_set_subtree(GTK_TREE_ITEM(new_item), new_subtree);
	       gtk_signal_connect(GTK_OBJECT(new_subtree), "select_child",
				  GTK_SIGNAL_FUNC(fill_oc_detail_rightpane), hash);

	       tmp = ss->oc;
	       while(tmp) {
		    oc = (LDAPObjectClass *) tmp->data;
		    if(oc->oc_names && oc->oc_names[0])
			 new_item = gtk_tree_item_new_with_label(oc->oc_names[0]);
		    else
			 new_item = gtk_tree_item_new_with_label(oc->oc_oid);
		    GTK_WIDGET_UNSET_FLAGS(new_item, GTK_CAN_FOCUS);
		    gtk_signal_connect(GTK_OBJECT(new_item), "button_press_event",
				       GTK_SIGNAL_FUNC(schema_button_tree), NULL);
		    gtk_object_set_data(GTK_OBJECT(new_item), "oc", oc);
		    gtk_widget_show(new_item);
		    gtk_tree_append(GTK_TREE(new_subtree), new_item);
		    tmp = tmp->next;
	       }
	  }

	  if(ss->at) {
	       new_item = gtk_tree_item_new_with_label("attributeTypes");
	       GTK_WIDGET_UNSET_FLAGS(new_item, GTK_CAN_FOCUS);
	       gtk_signal_connect(GTK_OBJECT(new_item), "button_press_event",
				  GTK_SIGNAL_FUNC(schema_button_tree), NULL);
	       gtk_tree_append(GTK_TREE(server_subtree), new_item);
	       gtk_widget_show(new_item);

	       new_subtree = gtk_tree_new();
	       GTK_WIDGET_UNSET_FLAGS(new_subtree, GTK_CAN_FOCUS);
	       gtk_object_set_data(GTK_OBJECT(new_subtree), "server", server);
	       gtk_widget_show(new_subtree);
	       gtk_tree_item_set_subtree(GTK_TREE_ITEM(new_item), new_subtree);
	       gtk_signal_connect(GTK_OBJECT(new_subtree), "select_child",
				  GTK_SIGNAL_FUNC(fill_at_detail_rightpane), hash);

	       tmp = ss->at;
	       while(tmp) {
		    at = (LDAPAttributeType *) tmp->data;
		    if(at->at_names && at->at_names[0])
			 new_item = gtk_tree_item_new_with_label(at->at_names[0]);
		    else
			 new_item = gtk_tree_item_new_with_label(at->at_oid);
		    GTK_WIDGET_UNSET_FLAGS(new_item, GTK_CAN_FOCUS);
		    gtk_signal_connect(GTK_OBJECT(new_item), "button_press_event",
				       GTK_SIGNAL_FUNC(schema_button_tree), NULL);
		    gtk_object_set_data(GTK_OBJECT(new_item), "at", at);
		    gtk_widget_show(new_item);
		    gtk_tree_append(GTK_TREE(new_subtree), new_item);
		    tmp = tmp->next;
	       }
	  }

	  if(ss->mr) {
	       new_item = gtk_tree_item_new_with_label("matchingRules");
	       GTK_WIDGET_UNSET_FLAGS(new_item, GTK_CAN_FOCUS);
	       gtk_signal_connect(GTK_OBJECT(new_item), "button_press_event",
				  GTK_SIGNAL_FUNC(schema_button_tree), NULL);
	       gtk_tree_append(GTK_TREE(server_subtree), new_item);
	       gtk_widget_show(new_item);

	       new_subtree = gtk_tree_new();
	       GTK_WIDGET_UNSET_FLAGS(new_subtree, GTK_CAN_FOCUS);
	       gtk_object_set_data(GTK_OBJECT(new_subtree), "server", server);
	       gtk_widget_show(new_subtree);
	       gtk_tree_item_set_subtree(GTK_TREE_ITEM(new_item), new_subtree);
	       gtk_signal_connect(GTK_OBJECT(new_subtree), "select_child",
				  GTK_SIGNAL_FUNC(fill_mr_detail_rightpane), hash);

	       tmp = ss->mr;
	       while(tmp) {
		    mr = (LDAPMatchingRule *) tmp->data;
		    if(mr->mr_names && mr->mr_names[0])
			 new_item = gtk_tree_item_new_with_label(mr->mr_names[0]);
		    else
			 new_item = gtk_tree_item_new_with_label(mr->mr_oid);
		    GTK_WIDGET_UNSET_FLAGS(new_item, GTK_CAN_FOCUS);
		    gtk_signal_connect(GTK_OBJECT(new_item), "button_press_event",
				       GTK_SIGNAL_FUNC(schema_button_tree), NULL);
		    gtk_object_set_data(GTK_OBJECT(new_item), "mr", mr);
		    gtk_widget_show(new_item);
		    gtk_tree_append(GTK_TREE(new_subtree), new_item);
		    tmp = tmp->next;
	       }

	  }

	  if(ss->s) {
	       new_item = gtk_tree_item_new_with_label("ldapSyntaxes");
	       GTK_WIDGET_UNSET_FLAGS(new_item, GTK_CAN_FOCUS);
	       gtk_signal_connect(GTK_OBJECT(new_item), "button_press_event",
				  GTK_SIGNAL_FUNC(schema_button_tree), NULL);
	       gtk_tree_append(GTK_TREE(server_subtree), new_item);
	       gtk_widget_show(new_item);

	       new_subtree = gtk_tree_new();
	       GTK_WIDGET_UNSET_FLAGS(new_subtree, GTK_CAN_FOCUS);
	       gtk_object_set_data(GTK_OBJECT(new_subtree), "server", server);
	       gtk_widget_show(new_subtree);
	       gtk_tree_item_set_subtree(GTK_TREE_ITEM(new_item), new_subtree);
	       gtk_signal_connect(GTK_OBJECT(new_subtree), "select_child",
				  GTK_SIGNAL_FUNC(fill_s_detail_rightpane), hash);

	       tmp = ss->s;
	       while(tmp) {
		    s = (LDAPSyntax *) tmp->data;
		    new_item = gtk_tree_item_new_with_label(s->syn_oid);
		    GTK_WIDGET_UNSET_FLAGS(new_item, GTK_CAN_FOCUS);
		    gtk_signal_connect(GTK_OBJECT(new_item), "button_press_event",
				       GTK_SIGNAL_FUNC(schema_button_tree), NULL);
		    gtk_widget_show(new_item);
		    gtk_object_set_data(GTK_OBJECT(new_item), "s", s);
		    gtk_tree_append(GTK_TREE(new_subtree), new_item);
		    tmp = tmp->next;
	       }
	  }

     }

     set_normalcursor();

}


gboolean schema_button_tree(GtkWidget *tree_item, GdkEventButton *event)
{
     GtkWidget *root_menu, *menu, *menu_item;

     if(event->type == GDK_BUTTON_PRESS && event->button == 3) {
	  gtk_signal_emit_stop_by_name(GTK_OBJECT(tree_item), "button_press_event");

	  root_menu = gtk_menu_item_new_with_label("Root");
	  gtk_widget_show(root_menu);
	  menu = gtk_menu_new();
	  gtk_menu_item_set_submenu(GTK_MENU_ITEM(root_menu), menu);

	  menu_item = gtk_tearoff_menu_item_new();
	  gtk_menu_append(GTK_MENU(menu), menu_item);
	  gtk_widget_set_sensitive(menu_item, FALSE);
	  gtk_widget_show(menu_item);

	  /* Refresh */
	  menu_item = gtk_menu_item_new_with_label("Refresh");
	  if(gtk_object_get_data(GTK_OBJECT(tree_item), "server") == NULL)
	       gtk_widget_set_sensitive(menu_item, FALSE);
	  gtk_menu_append(GTK_MENU(menu), menu_item);
	  gtk_widget_show(menu_item);
	  gtk_signal_connect(GTK_OBJECT(menu_item), "activate",
			     GTK_SIGNAL_FUNC(schema_refresh_server), (gpointer) tree_item);

	  /* Open in new window */
	  menu_item = gtk_menu_item_new_with_label("Open in new window");
	  /* only leaf nodes can have a detail popup */
	  if(GTK_TREE_ITEM(tree_item)->subtree)
	       gtk_widget_set_sensitive(menu_item, FALSE);
	  gtk_menu_append(GTK_MENU(menu), menu_item);
	  gtk_widget_show(menu_item);
	  gtk_signal_connect(GTK_OBJECT(menu_item), "activate",
			     GTK_SIGNAL_FUNC(popup_detail_callback), (gpointer) tree_item);


	  gtk_menu_popup(GTK_MENU(menu), NULL, NULL, NULL, NULL,
			 event->button, event->time);

	  return(TRUE);
     }
     else if(event->type == GDK_BUTTON_PRESS && event->button == 2) {
	  popup_detail_callback(NULL, tree_item);
	  return(TRUE);
     }

     return(FALSE);
}


void schema_refresh_server(GtkWidget *widget, GtkWidget *item)
{
     GList *sel;
     GtkTree *tree;
     GtkWidget *new_subtree;
     struct ldapserver *server;

     if( (server = gtk_object_get_data(GTK_OBJECT(item), "server")) == NULL)
	  return;

     close_connection(server, TRUE);

     tree = (GtkTree *) GTK_TREE_ITEM(item)->subtree;
     if(tree) {

	  /* this is a workaround -- lots of GTK warnings if I don't do this :-( */
	  sel = GTK_TREE_SELECTION(tree);
	  while(sel) {
	       if(sel->data)
		    gtk_tree_unselect_child(GTK_TREE(tree), sel->data);
	       sel = sel->next;
	  }

	  gtk_tree_item_remove_subtree(GTK_TREE_ITEM(item));

	  new_subtree = gtk_tree_new();
	  gtk_tree_item_set_subtree(GTK_TREE_ITEM(item), new_subtree);
	  gtk_signal_connect(GTK_OBJECT(item), "expand",
			     GTK_SIGNAL_FUNC(attach_server_schema), NULL);
     }

}


void make_detail_notebook(GHashTable *hash)
{
     GtkWidget *rightpane_notebook, *rightpane_vbox;
     GtkWidget *oc_vbox, *at_vbox, *mr_vbox, *s_vbox;
     GtkWidget *dummy_vbox, *label;

     oc_vbox = gtk_vbox_new(FALSE, 0);
     at_vbox = gtk_vbox_new(FALSE, 0);
     mr_vbox = gtk_vbox_new(FALSE, 0);
     s_vbox = gtk_vbox_new(FALSE, 0);
     gtk_widget_show(oc_vbox);
     gtk_widget_show(at_vbox);
     gtk_widget_show(mr_vbox);
     gtk_widget_show(s_vbox);
     g_hash_table_insert(hash, "oc_vbox", oc_vbox);
     g_hash_table_insert(hash, "at_vbox", at_vbox);
     g_hash_table_insert(hash, "mr_vbox", mr_vbox);
     g_hash_table_insert(hash, "s_vbox", s_vbox);

     rightpane_vbox = g_hash_table_lookup(hash, "rightpane_vbox");
     rightpane_notebook = gtk_notebook_new();
     gtk_box_pack_start(GTK_BOX(rightpane_vbox), rightpane_notebook, TRUE, TRUE, 0);
     GTK_WIDGET_UNSET_FLAGS(GTK_NOTEBOOK(rightpane_notebook), GTK_CAN_FOCUS);
     gtk_widget_show(rightpane_notebook);
     g_hash_table_insert(hash, "rightpane_notebook", rightpane_notebook);

     /* Objectclasses tab */
     dummy_vbox = gtk_vbox_new(FALSE, 0);
     gtk_widget_show(dummy_vbox);
     gtk_box_pack_start(GTK_BOX(dummy_vbox), oc_vbox, TRUE, TRUE, 0);
     make_oc_detail(oc_vbox);
     label = gtk_label_new("Objectclasses");
     gtk_widget_show(label);
     gtk_notebook_append_page(GTK_NOTEBOOK(rightpane_notebook), dummy_vbox, label);

     /* Attribute types tab */
     dummy_vbox = gtk_vbox_new(FALSE, 0);
     gtk_widget_show(dummy_vbox);
     gtk_box_pack_start(GTK_BOX(dummy_vbox), at_vbox, TRUE, TRUE, 0);
     make_at_detail(at_vbox);
     label = gtk_label_new("Attribute types");
     gtk_widget_show(label);
     gtk_notebook_append_page(GTK_NOTEBOOK(rightpane_notebook), dummy_vbox, label);

     /* Matching rules tab */
     dummy_vbox = gtk_vbox_new(FALSE, 0);
     gtk_widget_show(dummy_vbox);
     gtk_box_pack_start(GTK_BOX(dummy_vbox), mr_vbox, TRUE, TRUE, 0);
     make_mr_detail(mr_vbox);
     label = gtk_label_new("Matching rules");
     gtk_widget_show(label);
     gtk_notebook_append_page(GTK_NOTEBOOK(rightpane_notebook), dummy_vbox, label);

     /* Syntaxes tab */
     dummy_vbox = gtk_vbox_new(FALSE, 0);
     gtk_widget_show(dummy_vbox);
     gtk_box_pack_start(GTK_BOX(dummy_vbox), s_vbox, TRUE, TRUE, 0);
     make_s_detail(s_vbox);
     label = gtk_label_new("Syntaxes");
     gtk_widget_show(label);
     gtk_notebook_append_page(GTK_NOTEBOOK(rightpane_notebook), dummy_vbox, label);

}


void popup_detail_callback(GtkWidget *dummy, GtkWidget *widget)
{
     LDAPObjectClass *oc;
     LDAPAttributeType *at;
     LDAPMatchingRule *mr;
     LDAPSyntax *s;
     struct ldapserver *server;

     if( (server = gtk_object_get_data(GTK_OBJECT(widget), "server")) == NULL
	  && ( (server = gtk_object_get_data(GTK_OBJECT(widget->parent), "server")) == NULL))
	  return;

     oc = gtk_object_get_data(GTK_OBJECT(widget), "oc");
     at = gtk_object_get_data(GTK_OBJECT(widget), "at");
     mr = gtk_object_get_data(GTK_OBJECT(widget), "mr");
     s = gtk_object_get_data(GTK_OBJECT(widget), "s");

     if(!oc && !at && !mr && !s)
	  /* how did we get here? */
	  return;

     if(oc)
	  popup_detail(SCHEMA_TYPE_OC, server, oc);
     else if(at)
	  popup_detail(SCHEMA_TYPE_AT, server, at);
     else if(mr)
	  popup_detail(SCHEMA_TYPE_MR, server, mr);
     else if(s)
	  popup_detail(SCHEMA_TYPE_S, server, s);

}


void popup_detail(enum schema_detail_type type, struct ldapserver *server, void *detail)
{
     GtkWidget *window, *vbox;

     window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
     gtk_window_set_title(GTK_WINDOW(window), "GQ");
     gtk_window_set_default_size(GTK_WINDOW(window), 446, 340);
     gtk_signal_connect_object(GTK_OBJECT(window), "key_press_event",
			       (GtkSignalFunc) close_on_esc, (gpointer) window);
     vbox = gtk_vbox_new(FALSE, 0);
     gtk_widget_show(vbox);
     gtk_container_add(GTK_CONTAINER(window), vbox);

     switch(type) {
     case SCHEMA_TYPE_OC:
	  gtk_window_set_title(GTK_WINDOW(window), "GQ: objectclass");
	  make_oc_detail(vbox);
	  fill_oc_detail(vbox, server, detail);
	  break;
     case SCHEMA_TYPE_AT:
	  gtk_window_set_title(GTK_WINDOW(window), "GQ: attribute type");
	  make_at_detail(vbox);
	  fill_at_detail(vbox, server, detail);
	  break;
     case SCHEMA_TYPE_MR:
	  gtk_window_set_title(GTK_WINDOW(window), "GQ: matching rule");
	  make_mr_detail(vbox);
	  fill_mr_detail(vbox, server, detail);
	  break;
     case SCHEMA_TYPE_S:
	  gtk_window_set_title(GTK_WINDOW(window), "GQ: syntax");
	  make_s_detail(vbox);
	  fill_s_detail(vbox, server, detail);
	  break;
     }

     gtk_widget_show(window);

}


void make_oc_detail(GtkWidget *target_oc_vbox)
{
     GtkWidget *label, *entry, *combo, *check, *clist;
     GtkWidget *hbox1, *hbox2, *vbox1, *table1, *scrwin, *paned;
     char *rtitle[] = { "Required attributes", 0 };
     char *atitle[] = { "Allowed attributes", 0 };

     /* homogeneous hbox to split pane down the middle */
     hbox1 = gtk_hbox_new(TRUE, 0);
     gtk_widget_show(hbox1);
     gtk_box_pack_start(GTK_BOX(target_oc_vbox), hbox1, TRUE, TRUE, 10);

     /* left half */
     vbox1 = gtk_vbox_new(FALSE, 0);
     gtk_widget_show(vbox1);
     gtk_box_pack_start(GTK_BOX(hbox1), vbox1, TRUE, TRUE, 10);

     table1 = gtk_table_new(11, 1, FALSE);
     gtk_table_set_row_spacings(GTK_TABLE(table1), 2);
     gtk_widget_show(table1);
     gtk_box_pack_start(GTK_BOX(vbox1), table1, FALSE, FALSE, 0);

     /* Name */
     label = gtk_label_new("Name");
     gtk_misc_set_alignment(GTK_MISC(label), .0, .0);
     gtk_widget_show(label);
     gtk_table_attach(GTK_TABLE(table1), label, 0, 1, 0, 1,
		      GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);

     combo = gtk_combo_new();
     gtk_object_set_data(GTK_OBJECT(target_oc_vbox), "name", combo);
     gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(combo)->entry), FALSE);
     GTK_WIDGET_UNSET_FLAGS(GTK_ENTRY(GTK_COMBO(combo)->entry), GTK_CAN_FOCUS);
     gtk_widget_show(combo);
     gtk_table_attach(GTK_TABLE(table1), combo, 0, 1, 1, 2,
		      GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);

     gtk_table_set_row_spacing(GTK_TABLE(table1), 1, 10);

     /* Description */
     label = gtk_label_new("Description");
     gtk_misc_set_alignment(GTK_MISC(label), .0, .0);
     gtk_widget_show(label);
     gtk_table_attach(GTK_TABLE(table1), label, 0, 1, 2, 3,
		      GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);

     entry = gtk_entry_new();
     gtk_object_set_data(GTK_OBJECT(target_oc_vbox), "description", entry);
     GTK_WIDGET_UNSET_FLAGS(GTK_ENTRY(entry), GTK_CAN_FOCUS);
     gtk_widget_show(entry);
     gtk_table_attach(GTK_TABLE(table1), entry, 0, 1, 3, 4,
		      GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);

     gtk_table_set_row_spacing(GTK_TABLE(table1), 3, 10);

     /* OID */
     label = gtk_label_new("OID");
     gtk_misc_set_alignment(GTK_MISC(label), .0, .0);
     gtk_widget_show(label);
     gtk_table_attach(GTK_TABLE(table1), label, 0, 1, 4, 5,
		      GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);

     entry = gtk_entry_new();
     gtk_object_set_data(GTK_OBJECT(target_oc_vbox), "oid", entry);
     GTK_WIDGET_UNSET_FLAGS(GTK_ENTRY(entry), GTK_CAN_FOCUS);
     gtk_widget_show(entry);
     gtk_table_attach(GTK_TABLE(table1), entry, 0, 1, 5, 6,
		      GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);

     gtk_table_set_row_spacing(GTK_TABLE(table1), 5, 10);

     /* Superior */
     label = gtk_label_new("Superior");
     gtk_misc_set_alignment(GTK_MISC(label), .0, .0);
     gtk_widget_show(label);
     gtk_table_attach(GTK_TABLE(table1), label, 0, 1, 6, 7,
		      GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);

     combo = gtk_combo_new();
     gtk_object_set_data(GTK_OBJECT(target_oc_vbox), "superior", combo);
     gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(combo)->entry), FALSE);
     GTK_WIDGET_UNSET_FLAGS(GTK_ENTRY(GTK_COMBO(combo)->entry), GTK_CAN_FOCUS);
     gtk_widget_show(combo);
     gtk_table_attach(GTK_TABLE(table1), combo, 0, 1, 7, 8,
		      GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);

     gtk_table_set_row_spacing(GTK_TABLE(table1), 7, 10);

     /* Kind */
     label = gtk_label_new("Kind");
     gtk_misc_set_alignment(GTK_MISC(label), .0, .0);
     gtk_widget_show(label);
     gtk_table_attach(GTK_TABLE(table1), label, 0, 1, 8, 9,
		      GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);

     entry = gtk_entry_new();
     gtk_object_set_data(GTK_OBJECT(target_oc_vbox), "kind", entry);
     GTK_WIDGET_UNSET_FLAGS(GTK_ENTRY(entry), GTK_CAN_FOCUS);
     gtk_widget_show(entry);
     gtk_table_attach(GTK_TABLE(table1), entry, 0, 1, 9, 10,
		      GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);

     gtk_table_set_row_spacing(GTK_TABLE(table1), 9, 10);

     /* Obsolete */
     hbox2 = gtk_hbox_new(FALSE, 0);
     gtk_widget_show(hbox2);
     check = gtk_check_button_new();
     gtk_object_set_data(GTK_OBJECT(target_oc_vbox), "obsolete", check);
     gtk_widget_set_sensitive(check, FALSE);
     GTK_WIDGET_UNSET_FLAGS(check, GTK_CAN_FOCUS);
     gtk_widget_show(check);
     gtk_box_pack_start(GTK_BOX(hbox2), check, FALSE, FALSE, 0);
     label = gtk_label_new("Obsolete");
     gtk_widget_show(label);
     gtk_box_pack_start(GTK_BOX(hbox2), label, FALSE, FALSE, 0);
     gtk_table_attach(GTK_TABLE(table1), hbox2, 0, 1, 10, 11,
		      GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);


     /* right half */

     paned = gtk_vpaned_new();
     gtk_paned_set_gutter_size(GTK_PANED(paned), 0);
     gtk_paned_set_handle_size(GTK_PANED(paned), 0);
     gtk_widget_show(paned);
     gtk_box_pack_start(GTK_BOX(hbox1), paned, TRUE, TRUE, 10);

     /* Required attributes */
     scrwin = gtk_scrolled_window_new(NULL, NULL);
     gtk_widget_show(scrwin);
     gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrwin),
                                    GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);

     /* just about right for four lines in the top clist */
     gtk_widget_set_usize(scrwin, -1, 93);

     clist = gtk_clist_new_with_titles(1, rtitle);  
     gtk_widget_show(clist);
     GTK_CLIST(clist)->button_actions[1] = GTK_BUTTON_SELECTS;
     gtk_signal_connect(GTK_OBJECT(clist), "select_row",
			(GtkSignalFunc) select_at_from_clist, NULL);
     gtk_object_set_data(GTK_OBJECT(target_oc_vbox), "required", clist);
     gtk_clist_column_titles_passive(GTK_CLIST(clist));
     GTK_WIDGET_UNSET_FLAGS(clist, GTK_CAN_FOCUS);
     gtk_container_add(GTK_CONTAINER(scrwin), clist);
     gtk_paned_pack1(GTK_PANED(paned), scrwin, FALSE, FALSE);

     /* Allowed attributes */
     scrwin = gtk_scrolled_window_new(NULL, NULL);
     gtk_widget_show(scrwin);
     gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrwin),
                                    GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
     clist = gtk_clist_new_with_titles(1, atitle);
     gtk_widget_show(clist);
     GTK_CLIST(clist)->button_actions[1] = GTK_BUTTON_SELECTS;
     gtk_signal_connect(GTK_OBJECT(clist), "select_row",
			(GtkSignalFunc) select_at_from_clist, NULL);
     gtk_object_set_data(GTK_OBJECT(target_oc_vbox), "allowed", clist);
     gtk_clist_column_titles_passive(GTK_CLIST(clist));
     GTK_WIDGET_UNSET_FLAGS(clist, GTK_CAN_FOCUS);
     gtk_container_add(GTK_CONTAINER(scrwin), clist);
     gtk_paned_pack2(GTK_PANED(paned), scrwin, TRUE, FALSE);

     /* and again, for good measure */
     gtk_paned_set_position(GTK_PANED(paned), 93);

}


void fill_oc_detail_rightpane(GtkWidget *treeroot, GtkWidget *tree_item, GHashTable *hash)
{
     GtkWidget *rightpane_notebook, *oc_vbox;
     LDAPObjectClass *oc;
     struct ldapserver *server;

     if( (server = gtk_object_get_data(GTK_OBJECT(treeroot), "server")) == NULL)
	  return;

     if( (oc = gtk_object_get_data(GTK_OBJECT(tree_item), "oc")) == NULL)
	  return;

     if(g_hash_table_lookup(hash, "cur_oc") == oc)
	  return;

     g_hash_table_insert(hash, "cur_oc", oc);

     oc_vbox = g_hash_table_lookup(hash, "oc_vbox");
     if(oc_vbox == NULL) {
	  make_detail_notebook(hash);
	  oc_vbox = g_hash_table_lookup(hash, "oc_vbox");
     }

     fill_oc_detail(oc_vbox, server, oc);

     /* switch to objectClasses tab */
     rightpane_notebook = g_hash_table_lookup(hash, "rightpane_notebook");
     gtk_notebook_set_page(GTK_NOTEBOOK(rightpane_notebook), 0);

}


void fill_oc_detail(GtkWidget *target_oc_vbox, struct ldapserver *server,
		    LDAPObjectClass *oc)
{
     GList *list;
     GtkWidget *entry, *combo, *check, *clist;
     int i;
     char *kind, *dummy[2];

     /* left half */

     /* Name */
     combo = (GtkWidget *) gtk_object_get_data(GTK_OBJECT(target_oc_vbox), "name");
     list = ar2glist(oc->oc_names);
     if(list) {
	  gtk_combo_set_popdown_strings(GTK_COMBO(combo), list);
	  g_list_free(list);
     }

     /* Description */
     entry = (GtkWidget *) gtk_object_get_data(GTK_OBJECT(target_oc_vbox), "description");
     gtk_entry_set_text(GTK_ENTRY(entry), oc->oc_desc ? oc->oc_desc : "");

     /* OID */
     entry = (GtkWidget *) gtk_object_get_data(GTK_OBJECT(target_oc_vbox), "oid");
     gtk_entry_set_text(GTK_ENTRY(entry), oc->oc_oid ? oc->oc_oid : "");

     /* Superior OIDs */
     combo = (GtkWidget *) gtk_object_get_data(GTK_OBJECT(target_oc_vbox), "superior");
     list = ar2glist(oc->oc_sup_oids);
     if(list) {
	  gtk_combo_set_popdown_strings(GTK_COMBO(combo), list);
	  g_list_free(list);
     }

     /* Kind */
     entry = (GtkWidget *) gtk_object_get_data(GTK_OBJECT(target_oc_vbox), "kind");
     switch(oc->oc_kind) {
     case LDAP_SCHEMA_ABSTRACT:
	  kind = "Abstract";
	  break;
     case LDAP_SCHEMA_STRUCTURAL:
	  kind = "Structural";
	  break;
     case LDAP_SCHEMA_AUXILIARY:
	  kind = "Auxiliary";
	  break;
     default:
	  kind = "Unknown";
	  break;
     }
     gtk_entry_set_text(GTK_ENTRY(entry), kind);

     /* Obsolete */
     check = (GtkWidget *) gtk_object_get_data(GTK_OBJECT(target_oc_vbox), "obsolete");
     gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(check), oc->oc_obsolete);

     /* right half */

     /* Required attributes */
     clist = (GtkWidget *) gtk_object_get_data(GTK_OBJECT(target_oc_vbox), "required");
     gtk_object_set_data(GTK_OBJECT(clist), "server", server);
     gtk_clist_freeze(GTK_CLIST(clist));
     gtk_clist_clear(GTK_CLIST(clist));
     i = 0;
     dummy[1] = NULL;
     while(oc->oc_at_oids_must && oc->oc_at_oids_must[i]) {
	  dummy[0] = oc->oc_at_oids_must[i];
	  gtk_clist_insert(GTK_CLIST(clist), i, dummy);
	  i++;
     }
     gtk_clist_thaw(GTK_CLIST(clist));

     /* Allowed attributes */
     clist = (GtkWidget *) gtk_object_get_data(GTK_OBJECT(target_oc_vbox), "allowed");
     gtk_object_set_data(GTK_OBJECT(clist), "server", server);
     gtk_clist_freeze(GTK_CLIST(clist));
     gtk_clist_clear(GTK_CLIST(clist));
     i = 0;
     while(oc->oc_at_oids_may && oc->oc_at_oids_may[i]) {
	  dummy[0] = oc->oc_at_oids_may[i];
	  gtk_clist_insert(GTK_CLIST(clist), i, dummy);
	  i++;
     }
     gtk_clist_thaw(GTK_CLIST(clist));

}


void make_at_detail(GtkWidget *target_vbox)
{
     GtkWidget *hbox1, *hbox2, *vbox1, *vbox2, *table1, *table2, *scrwin;
     GtkWidget *label, *entry, *combo, *check, *clist;
     char *otitle[] = { "Used in objectclasses", 0 };

     /* homogeneous hbox to split the pane down the middle */
     hbox1 = gtk_hbox_new(TRUE, 0);
     gtk_widget_show(hbox1);
     gtk_box_pack_start(GTK_BOX(target_vbox), hbox1, TRUE, TRUE, 10);

     /* left half */

     vbox1 = gtk_vbox_new(FALSE, 0);
     gtk_widget_show(vbox1);
     gtk_box_pack_start(GTK_BOX(hbox1), vbox1, TRUE, TRUE, 10);

     table1 = gtk_table_new(14, 1, FALSE);
     gtk_table_set_row_spacings(GTK_TABLE(table1), 1);
     gtk_widget_show(table1);
     gtk_box_pack_start(GTK_BOX(vbox1), table1, FALSE, FALSE, 0);

     /* Name */
     label = gtk_label_new("Name");
     gtk_misc_set_alignment(GTK_MISC(label), .0, .0);
     gtk_widget_show(label);
     gtk_table_attach(GTK_TABLE(table1), label, 0, 1, 0, 1,
		      GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);

     combo = gtk_combo_new();
     gtk_object_set_data(GTK_OBJECT(target_vbox), "name", combo);
     gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(combo)->entry), FALSE);
     GTK_WIDGET_UNSET_FLAGS(GTK_ENTRY(GTK_COMBO(combo)->entry), GTK_CAN_FOCUS);
     gtk_widget_show(combo);
     gtk_table_attach(GTK_TABLE(table1), combo, 0, 1, 1, 2,
		      GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);

     gtk_table_set_row_spacing(GTK_TABLE(table1), 1, 10);

     /* Description */
     label = gtk_label_new("Description");
     gtk_misc_set_alignment(GTK_MISC(label), .0, .0);
     gtk_widget_show(label);
     gtk_table_attach(GTK_TABLE(table1), label, 0, 1, 2, 3,
		      GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);

     entry = gtk_entry_new();
     gtk_object_set_data(GTK_OBJECT(target_vbox), "description", entry);
     GTK_WIDGET_UNSET_FLAGS(GTK_ENTRY(entry), GTK_CAN_FOCUS);
     gtk_widget_show(entry);
     gtk_table_attach(GTK_TABLE(table1), entry, 0, 1, 3, 4,
		      GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);

     gtk_table_set_row_spacing(GTK_TABLE(table1), 3, 10);

     /* OID */
     label = gtk_label_new("OID");
     gtk_misc_set_alignment(GTK_MISC(label), .0, .0);
     gtk_widget_show(label);
     gtk_table_attach(GTK_TABLE(table1), label, 0, 1, 4, 5,
		      GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);

     entry = gtk_entry_new();
     gtk_object_set_data(GTK_OBJECT(target_vbox), "oid", entry);
     GTK_WIDGET_UNSET_FLAGS(GTK_ENTRY(entry), GTK_CAN_FOCUS);
     gtk_widget_show(entry);
     gtk_table_attach(GTK_TABLE(table1), entry, 0, 1, 5, 6,
		      GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);

     gtk_table_set_row_spacing(GTK_TABLE(table1), 5, 10);

     /* Superior */
     label = gtk_label_new("Superior");
     gtk_misc_set_alignment(GTK_MISC(label), .0, .0);
     gtk_widget_show(label);
     gtk_table_attach(GTK_TABLE(table1), label, 0, 1, 6, 7,
		      GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);

     entry = gtk_entry_new();
     gtk_object_set_data(GTK_OBJECT(target_vbox), "superior", entry);
     GTK_WIDGET_UNSET_FLAGS(GTK_ENTRY(entry), GTK_CAN_FOCUS);
     gtk_widget_show(entry);
     gtk_table_attach(GTK_TABLE(table1), entry, 0, 1, 7, 8,
		      GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);

     gtk_table_set_row_spacing(GTK_TABLE(table1), 7, 10);

     /* Usage */
     label = gtk_label_new("Usage");
     gtk_misc_set_alignment(GTK_MISC(label), .0, .0);
     gtk_widget_show(label);
     gtk_table_attach(GTK_TABLE(table1), label, 0, 1, 8, 9,
		      GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);

     entry = gtk_entry_new();
     gtk_object_set_data(GTK_OBJECT(target_vbox), "usage", entry);
     GTK_WIDGET_UNSET_FLAGS(GTK_ENTRY(entry), GTK_CAN_FOCUS);
     gtk_widget_show(entry);
     gtk_table_attach(GTK_TABLE(table1), entry, 0, 1, 9, 10,
		      GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);

     gtk_table_set_row_spacing(GTK_TABLE(table1), 9, 10);

     /* Obsolete */
     hbox2 = gtk_hbox_new(FALSE, 0);
     gtk_widget_show(hbox2);
     check = gtk_check_button_new();
     gtk_object_set_data(GTK_OBJECT(target_vbox), "obsolete", check);
     gtk_widget_set_sensitive(check, FALSE);
     GTK_WIDGET_UNSET_FLAGS(check, GTK_CAN_FOCUS);
     gtk_widget_show(check);
     gtk_box_pack_start(GTK_BOX(hbox2), check, FALSE, FALSE, 0);
     label = gtk_label_new("Obsolete");
     gtk_widget_show(label);
     gtk_box_pack_start(GTK_BOX(hbox2), label, FALSE, FALSE, 0);
     gtk_table_attach(GTK_TABLE(table1), hbox2, 0, 1, 10, 11,
		      GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);

     /* Single value */
     hbox2 = gtk_hbox_new(FALSE, 0);
     gtk_widget_show(hbox2);
     check = gtk_check_button_new();
     gtk_object_set_data(GTK_OBJECT(target_vbox), "singlevalue", check);
     gtk_widget_set_sensitive(check, FALSE);
     GTK_WIDGET_UNSET_FLAGS(check, GTK_CAN_FOCUS);
     gtk_widget_show(check);
     gtk_box_pack_start(GTK_BOX(hbox2), check, FALSE, FALSE, 0);
     label = gtk_label_new("Single value");
     gtk_widget_show(label);
     gtk_box_pack_start(GTK_BOX(hbox2), label, FALSE, FALSE, 0);
     gtk_table_attach(GTK_TABLE(table1), hbox2, 0, 1, 11, 12,
		      GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);

     /* Collective */
     hbox2 = gtk_hbox_new(FALSE, 0);
     gtk_widget_show(hbox2);
     check = gtk_check_button_new();
     gtk_object_set_data(GTK_OBJECT(target_vbox), "collective", check);
     gtk_widget_set_sensitive(check, FALSE);
     GTK_WIDGET_UNSET_FLAGS(check, GTK_CAN_FOCUS);
     gtk_widget_show(check);
     gtk_box_pack_start(GTK_BOX(hbox2), check, FALSE, FALSE, 0);
     label = gtk_label_new("Collective");
     gtk_widget_show(label);
     gtk_box_pack_start(GTK_BOX(hbox2), label, FALSE, FALSE, 0);
     gtk_table_attach(GTK_TABLE(table1), hbox2, 0, 1, 12, 13,
		      GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);

     /* No user modification */

     hbox2 = gtk_hbox_new(FALSE, 0);
     gtk_widget_show(hbox2);
     check = gtk_check_button_new();
     gtk_object_set_data(GTK_OBJECT(target_vbox), "nousermod", check);
     gtk_widget_set_sensitive(check, FALSE);
     GTK_WIDGET_UNSET_FLAGS(check, GTK_CAN_FOCUS);
     gtk_widget_show(check);
     gtk_box_pack_start(GTK_BOX(hbox2), check, FALSE, FALSE, 0);
     label = gtk_label_new("No user modification");
     gtk_widget_show(label);
     gtk_box_pack_start(GTK_BOX(hbox2), label, FALSE, FALSE, 0);
     gtk_table_attach(GTK_TABLE(table1), hbox2, 0, 1, 13, 14,
		      GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);


     /* right half */

     vbox2 = gtk_vbox_new(FALSE, 0);
     gtk_widget_show(vbox2);
     gtk_box_pack_start(GTK_BOX(hbox1), vbox2, TRUE, TRUE, 10);

     table2 = gtk_table_new(9, 1, FALSE);
     gtk_table_set_row_spacings(GTK_TABLE(table2), 1);
     gtk_widget_show(table2);
     gtk_box_pack_start(GTK_BOX(vbox2), table2, FALSE, FALSE, 0);

     /* Equality */
     label = gtk_label_new("Equality");
     gtk_misc_set_alignment(GTK_MISC(label), .0, .0);
     gtk_widget_show(label);
     gtk_table_attach(GTK_TABLE(table2), label, 0, 1, 0, 1,
		      GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);

     entry = gtk_entry_new();
     gtk_object_set_data(GTK_OBJECT(target_vbox), "equality", entry);
     GTK_WIDGET_UNSET_FLAGS(GTK_ENTRY(entry), GTK_CAN_FOCUS);
     gtk_widget_show(entry);
     gtk_table_attach(GTK_TABLE(table2), entry, 0, 1, 1, 2,
		      GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);

     gtk_table_set_row_spacing(GTK_TABLE(table2), 1, 10);

     /* Ordering */
     label = gtk_label_new("Ordering");
     gtk_misc_set_alignment(GTK_MISC(label), .0, .0);
     gtk_widget_show(label);
     gtk_table_attach(GTK_TABLE(table2), label, 0, 1, 2, 3,
		      GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);

     entry = gtk_entry_new();
     gtk_object_set_data(GTK_OBJECT(target_vbox), "ordering", entry);
     GTK_WIDGET_UNSET_FLAGS(GTK_ENTRY(entry), GTK_CAN_FOCUS);
     gtk_widget_show(entry);
     gtk_table_attach(GTK_TABLE(table2), entry, 0, 1, 3, 4,
		      GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);

     gtk_table_set_row_spacing(GTK_TABLE(table2), 3, 10);

     /* Substrings */
     label = gtk_label_new("Substrings");
     gtk_misc_set_alignment(GTK_MISC(label), .0, .0);
     gtk_widget_show(label);
     gtk_table_attach(GTK_TABLE(table2), label, 0, 1, 4, 5,
		      GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);

     entry = gtk_entry_new();
     gtk_object_set_data(GTK_OBJECT(target_vbox), "substrings", entry);
     GTK_WIDGET_UNSET_FLAGS(GTK_ENTRY(entry), GTK_CAN_FOCUS);
     gtk_widget_show(entry);
     gtk_table_attach(GTK_TABLE(table2), entry, 0, 1, 5, 6,
		      GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);

     gtk_table_set_row_spacing(GTK_TABLE(table2), 5, 10);

     /* Syntax */
     label = gtk_label_new("Syntax { length }");
     gtk_misc_set_alignment(GTK_MISC(label), .0, .0);
     gtk_widget_show(label);
     gtk_table_attach(GTK_TABLE(table2), label, 0, 1, 6, 7,
		      GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);

     entry = gtk_entry_new();
     gtk_object_set_data(GTK_OBJECT(target_vbox), "syntax", entry);
     GTK_WIDGET_UNSET_FLAGS(GTK_ENTRY(entry), GTK_CAN_FOCUS);
     gtk_widget_show(entry);
     gtk_table_attach(GTK_TABLE(table2), entry, 0, 1, 7, 8,
		      GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);

     gtk_table_set_row_spacing(GTK_TABLE(table2), 7, 10);

     /* Used in objectclasses */
     scrwin = gtk_scrolled_window_new(NULL, NULL);
     gtk_widget_show(scrwin);
     gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrwin),
                                    GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);

     clist = gtk_clist_new_with_titles(1, otitle);
     gtk_widget_show(clist);
     GTK_CLIST(clist)->button_actions[1] = GTK_BUTTON_SELECTS;
     gtk_signal_connect(GTK_OBJECT(clist), "select_row",
			(GtkSignalFunc) select_oc_from_clist, NULL);
     gtk_object_set_data(GTK_OBJECT(target_vbox), "usedoc", clist);
     gtk_clist_column_titles_passive(GTK_CLIST(clist));
     GTK_WIDGET_UNSET_FLAGS(clist, GTK_CAN_FOCUS);
     gtk_container_add(GTK_CONTAINER(scrwin), clist);
     gtk_box_pack_start(GTK_BOX(vbox2), scrwin, TRUE, TRUE, 0);

}


void fill_at_detail_rightpane(GtkWidget *treeroot, GtkWidget *tree_item, GHashTable *hash)
{
     GtkWidget *rightpane_notebook, *at_vbox;
     LDAPAttributeType *at;
     struct ldapserver *server;

     if( (server = gtk_object_get_data(GTK_OBJECT(treeroot), "server")) == NULL)
	  return;

     if( (at = gtk_object_get_data(GTK_OBJECT(tree_item), "at")) == NULL)
	  return;

     if(g_hash_table_lookup(hash, "cur_at") == at)
	  return;

     g_hash_table_insert(hash, "cur_at", at);

     at_vbox = g_hash_table_lookup(hash, "at_vbox");
     if(at_vbox == NULL) {
	  make_detail_notebook(hash);
	  at_vbox = g_hash_table_lookup(hash, "at_vbox");
     }

     fill_at_detail(at_vbox, server, at);

     /* switch to Attribute types tab */
     rightpane_notebook = g_hash_table_lookup(hash, "rightpane_notebook");
     gtk_notebook_set_page(GTK_NOTEBOOK(rightpane_notebook), 1);

}


void fill_at_detail(GtkWidget *target_vbox, struct ldapserver *server,
		    LDAPAttributeType *at)
{
     GList *list;
     GString *syntax;
     GtkWidget *combo, *entry, *check, *clist;
     int i;
     char *usage, *dummy[2];

     /* left half */

     /* Name */
     combo = (GtkWidget *) gtk_object_get_data(GTK_OBJECT(target_vbox), "name");
     list = ar2glist(at->at_names);
     if(list) {
	  gtk_combo_set_popdown_strings(GTK_COMBO(combo), list);
	  g_list_free(list);
     }

     /* Description */
     entry = (GtkWidget *) gtk_object_get_data(GTK_OBJECT(target_vbox), "description");
     gtk_entry_set_text(GTK_ENTRY(entry), at->at_desc ? at->at_desc : "");

     /* OID */
     entry = (GtkWidget *) gtk_object_get_data(GTK_OBJECT(target_vbox), "oid");
     gtk_entry_set_text(GTK_ENTRY(entry), at->at_oid ? at->at_oid : "");

     /* Superior */
     entry = (GtkWidget *) gtk_object_get_data(GTK_OBJECT(target_vbox), "superior");
     gtk_entry_set_text(GTK_ENTRY(entry), at->at_sup_oid ? at->at_sup_oid : "");

     /* Usage */
     entry = (GtkWidget *) gtk_object_get_data(GTK_OBJECT(target_vbox), "usage");
     switch(at->at_usage) {
     case LDAP_SCHEMA_USER_APPLICATIONS:
	  usage = "User applications";
	  break;
     case LDAP_SCHEMA_DIRECTORY_OPERATION:
	  usage = "Directory operation";
	  break;
     case LDAP_SCHEMA_DISTRIBUTED_OPERATION:
	  usage = "Distributed operation";
	  break;
     case LDAP_SCHEMA_DSA_OPERATION:
	  usage = "DSA operation";
	  break;
     default:
	  usage = "Unknown";
	  break;
     }
     gtk_entry_set_text(GTK_ENTRY(entry), usage);

     /* Obsolete */
     check = (GtkWidget *) gtk_object_get_data(GTK_OBJECT(target_vbox), "obsolete");
     gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(check), at->at_obsolete);

     /* Single value */
     check = (GtkWidget *) gtk_object_get_data(GTK_OBJECT(target_vbox), "singlevalue");
     gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(check), at->at_single_value);

     /* Collective */
     check = (GtkWidget *) gtk_object_get_data(GTK_OBJECT(target_vbox), "collective");
     gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(check), at->at_collective);

     /* No user modification */
     check = (GtkWidget *) gtk_object_get_data(GTK_OBJECT(target_vbox), "nousermod");
     gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(check), at->at_no_user_mod);

     /* right half */

     /* Equality */
     entry = (GtkWidget *) gtk_object_get_data(GTK_OBJECT(target_vbox), "equality");
     gtk_entry_set_text(GTK_ENTRY(entry), at->at_equality_oid ? at->at_equality_oid : "");

     /* Ordering */
     entry = (GtkWidget *) gtk_object_get_data(GTK_OBJECT(target_vbox), "ordering");
     gtk_entry_set_text(GTK_ENTRY(entry), at->at_ordering_oid ? at->at_ordering_oid : "");

     /* Substrings */
     entry = (GtkWidget *) gtk_object_get_data(GTK_OBJECT(target_vbox), "substrings");
     gtk_entry_set_text(GTK_ENTRY(entry), at->at_substr_oid ? at->at_substr_oid : "");

     /* Syntax */
     syntax = g_string_new(at->at_syntax_oid ? at->at_syntax_oid : "");
     if(at->at_syntax_len)
	  g_string_sprintfa(syntax, "{%d}", at->at_syntax_len);
     entry = (GtkWidget *) gtk_object_get_data(GTK_OBJECT(target_vbox), "syntax");
     gtk_entry_set_text(GTK_ENTRY(entry), syntax->str);

     clist = (GtkWidget *) gtk_object_get_data(GTK_OBJECT(target_vbox), "usedoc");
     gtk_object_set_data(GTK_OBJECT(clist), "server", server);
     gtk_clist_freeze(GTK_CLIST(clist));
     gtk_clist_clear(GTK_CLIST(clist));

     if(at->at_names && at->at_names[0]) {
	  list = find_oc_by_at(server, at->at_names[0]);
	  if(list) {
	       i = 0;
	       dummy[1] = NULL;
	       while(list) {
		    dummy[0] = list->data;
		    gtk_clist_insert(GTK_CLIST(clist), i, dummy);
		    list = list->next;
		    i++;
	       }
	       g_list_free(list);
	  }
     }
     gtk_clist_thaw(GTK_CLIST(clist));

}


void make_mr_detail(GtkWidget *target_vbox)
{
     GtkWidget *hbox1, *vbox1, *table1, *hbox2, *vbox2, *scrwin;
     GtkWidget *label, *combo, *entry, *check, *clist;
     char *atitle[] = { "Used in attribute types", 0 };

     /* homogeneous hbox to split the pane down the middle */
     hbox1 = gtk_hbox_new(TRUE, 0);
     gtk_widget_show(hbox1);
     gtk_box_pack_start(GTK_BOX(target_vbox), hbox1, TRUE, TRUE, 10);

     /* left half */

     vbox1 = gtk_vbox_new(FALSE, 0);
     gtk_widget_show(vbox1);
     gtk_box_pack_start(GTK_BOX(hbox1), vbox1, TRUE, TRUE, 10);

     table1 = gtk_table_new(9, 1, FALSE);
     gtk_table_set_row_spacings(GTK_TABLE(table1), 1);
     gtk_widget_show(table1);
     gtk_box_pack_start(GTK_BOX(vbox1), table1, FALSE, FALSE, 0);

     /* Name */
     label = gtk_label_new("Name");
     gtk_misc_set_alignment(GTK_MISC(label), .0, .0);
     gtk_widget_show(label);
     gtk_table_attach(GTK_TABLE(table1), label, 0, 1, 0, 1,
		      GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);

     combo = gtk_combo_new();
     gtk_object_set_data(GTK_OBJECT(target_vbox), "name", combo);
     gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(combo)->entry), FALSE);
     GTK_WIDGET_UNSET_FLAGS(GTK_ENTRY(GTK_COMBO(combo)->entry), GTK_CAN_FOCUS);
     gtk_widget_show(combo);
     gtk_table_attach(GTK_TABLE(table1), combo, 0, 1, 1, 2,
		      GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);

     gtk_table_set_row_spacing(GTK_TABLE(table1), 1, 10);

     /* Description */
     label = gtk_label_new("Description");
     gtk_misc_set_alignment(GTK_MISC(label), .0, .0);
     gtk_widget_show(label);
     gtk_table_attach(GTK_TABLE(table1), label, 0, 1, 2, 3,
		      GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);

     entry = gtk_entry_new();
     gtk_object_set_data(GTK_OBJECT(target_vbox), "description", entry);
     GTK_WIDGET_UNSET_FLAGS(GTK_ENTRY(entry), GTK_CAN_FOCUS);
     gtk_widget_show(entry);
     gtk_table_attach(GTK_TABLE(table1), entry, 0, 1, 3, 4,
		      GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);

     gtk_table_set_row_spacing(GTK_TABLE(table1), 3, 10);

     /* OID */
     label = gtk_label_new("OID");
     gtk_misc_set_alignment(GTK_MISC(label), .0, .0);
     gtk_widget_show(label);
     gtk_table_attach(GTK_TABLE(table1), label, 0, 1, 4, 5,
		      GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);

     entry = gtk_entry_new();
     gtk_object_set_data(GTK_OBJECT(target_vbox), "oid", entry);
     GTK_WIDGET_UNSET_FLAGS(GTK_ENTRY(entry), GTK_CAN_FOCUS);
     gtk_widget_show(entry);
     gtk_table_attach(GTK_TABLE(table1), entry, 0, 1, 5, 6,
		      GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);

     gtk_table_set_row_spacing(GTK_TABLE(table1), 5, 10);

     /* Syntax */
     label = gtk_label_new("Syntax");
     gtk_misc_set_alignment(GTK_MISC(label), .0, .0);
     gtk_widget_show(label);
     gtk_table_attach(GTK_TABLE(table1), label, 0, 1, 6, 7,
		      GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);

     entry = gtk_entry_new();
     gtk_object_set_data(GTK_OBJECT(target_vbox), "syntax", entry);
     GTK_WIDGET_UNSET_FLAGS(GTK_ENTRY(entry), GTK_CAN_FOCUS);
     gtk_widget_show(entry);
     gtk_table_attach(GTK_TABLE(table1), entry, 0, 1, 7, 8,
		      GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);

     gtk_table_set_row_spacing(GTK_TABLE(table1), 7, 10);

     /* Obsolete */
     hbox2 = gtk_hbox_new(FALSE, 0);
     gtk_widget_show(hbox2);
     check = gtk_check_button_new();
     gtk_object_set_data(GTK_OBJECT(target_vbox), "obsolete", check);
     gtk_widget_set_sensitive(check, FALSE);
     GTK_WIDGET_UNSET_FLAGS(check, GTK_CAN_FOCUS);
     gtk_widget_show(check);
     gtk_box_pack_start(GTK_BOX(hbox2), check, FALSE, FALSE, 0);
     label = gtk_label_new("Obsolete");
     gtk_widget_show(label);
     gtk_box_pack_start(GTK_BOX(hbox2), label, FALSE, FALSE, 0);
     gtk_table_attach(GTK_TABLE(table1), hbox2, 0, 1, 8, 9,
		      GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);

     /* right half */

     vbox2 = gtk_vbox_new(FALSE, 0);
     gtk_widget_show(vbox2);
     gtk_box_pack_start(GTK_BOX(hbox1), vbox2, TRUE, TRUE, 10);

     /* Used in attribute types */
     scrwin = gtk_scrolled_window_new(NULL, NULL);
     gtk_widget_show(scrwin);
     gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrwin),
                                    GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);

     clist = gtk_clist_new_with_titles(1, atitle);
     gtk_widget_show(clist);
     GTK_CLIST(clist)->button_actions[1] = GTK_BUTTON_SELECTS;
     gtk_signal_connect(GTK_OBJECT(clist), "select_row",
			(GtkSignalFunc) select_at_from_clist, NULL);
     gtk_object_set_data(GTK_OBJECT(target_vbox), "usedin", clist);
     gtk_clist_column_titles_passive(GTK_CLIST(clist));
     GTK_WIDGET_UNSET_FLAGS(clist, GTK_CAN_FOCUS);
     gtk_container_add(GTK_CONTAINER(scrwin), clist);
     gtk_box_pack_start(GTK_BOX(vbox2), scrwin, TRUE, TRUE, 0);

}


void fill_mr_detail_rightpane(GtkWidget *treeroot, GtkWidget *tree_item, GHashTable *hash)
{
     GtkWidget *rightpane_notebook, *mr_vbox;
     LDAPMatchingRule *mr;
     struct ldapserver *server;

     if( (server = gtk_object_get_data(GTK_OBJECT(treeroot), "server")) == NULL)
	  return;

     if( (mr = gtk_object_get_data(GTK_OBJECT(tree_item), "mr")) == NULL)
	  return;

     if(g_hash_table_lookup(hash, "cur_mr") == mr)
	  return;

     g_hash_table_insert(hash, "cur_mr", mr);

     mr_vbox = g_hash_table_lookup(hash, "mr_vbox");
     if(mr_vbox == NULL) {
	  make_detail_notebook(hash);
	  mr_vbox = g_hash_table_lookup(hash, "mr_vbox");
     }

     fill_mr_detail(mr_vbox, server, mr);

     /* switch to Matching rules tab */
     rightpane_notebook = g_hash_table_lookup(hash, "rightpane_notebook");
     gtk_notebook_set_page(GTK_NOTEBOOK(rightpane_notebook), 2);

}


void fill_mr_detail(GtkWidget *target_vbox, struct ldapserver *server,
		    LDAPMatchingRule *mr)
{
     GList *list;
     GtkWidget *combo, *entry, *check, *clist;
     int i;
     char *dummy[2];

     /* left half */

     /* Name */
     combo = (GtkWidget *) gtk_object_get_data(GTK_OBJECT(target_vbox), "name");
     list = ar2glist(mr->mr_names);
     if(list) {
	  gtk_combo_set_popdown_strings(GTK_COMBO(combo), list);
	  g_list_free(list);
     }

     /* Description */
     entry = (GtkWidget *) gtk_object_get_data(GTK_OBJECT(target_vbox), "description");
     gtk_entry_set_text(GTK_ENTRY(entry), mr->mr_desc ? mr->mr_desc : "");

     /* OID */
     entry = (GtkWidget *) gtk_object_get_data(GTK_OBJECT(target_vbox), "oid");
     gtk_entry_set_text(GTK_ENTRY(entry), mr->mr_oid ? mr->mr_oid : "");

     /* Syntax */
     entry = (GtkWidget *) gtk_object_get_data(GTK_OBJECT(target_vbox), "syntax");
     gtk_entry_set_text(GTK_ENTRY(entry), mr->mr_syntax_oid ? mr->mr_syntax_oid : "");

     /* Obsolete */
     check = (GtkWidget *) gtk_object_get_data(GTK_OBJECT(target_vbox), "obsolete");
     gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(check), mr->mr_obsolete);

     /* right half */

     clist = (GtkWidget *) gtk_object_get_data(GTK_OBJECT(target_vbox), "usedin");
     gtk_object_set_data(GTK_OBJECT(clist), "server", server);
     gtk_clist_freeze(GTK_CLIST(clist));
     gtk_clist_clear(GTK_CLIST(clist));

     list = find_at_by_mr_oid(server, mr->mr_oid);
     if(list) {
	  i = 0;
	  dummy[1] = NULL;
	  while(list) {
	       dummy[0] = list->data;
	       gtk_clist_insert(GTK_CLIST(clist), i, dummy);
	       list = list->next;
	       i++;
	  }
	  g_list_free(list);
     }

     gtk_clist_thaw(GTK_CLIST(clist));

}


void make_s_detail(GtkWidget *target_vbox)
{
     GtkWidget *hbox1, *vbox1, *table1, *scrwin, *paned;
     GtkWidget *label, *entry, *clist;
     char *atitle[] = { "Used in attribute types", 0 };
     char *mtitle[] = { "Used in matching rules", 0 };

     /* homogeneous hbox to split the pane down the middle */
     hbox1 = gtk_hbox_new(TRUE, 0);
     gtk_widget_show(hbox1);
     gtk_box_pack_start(GTK_BOX(target_vbox), hbox1, TRUE, TRUE, 10);

     /* left half */

     vbox1 = gtk_vbox_new(FALSE, 0);
     gtk_widget_show(vbox1);
     gtk_box_pack_start(GTK_BOX(hbox1), vbox1, TRUE, TRUE, 10);

     table1 = gtk_table_new(5, 1, FALSE);
     gtk_table_set_row_spacings(GTK_TABLE(table1), 1);
     gtk_widget_show(table1);
     gtk_box_pack_start(GTK_BOX(vbox1), table1, FALSE, FALSE, 0);

     /* OID */
     label = gtk_label_new("OID");
     gtk_misc_set_alignment(GTK_MISC(label), .0, .0);
     gtk_widget_show(label);
     gtk_table_attach(GTK_TABLE(table1), label, 0, 1, 0, 1,
		      GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);

     entry = gtk_entry_new();
     gtk_object_set_data(GTK_OBJECT(target_vbox), "oid", entry);
     GTK_WIDGET_UNSET_FLAGS(GTK_ENTRY(entry), GTK_CAN_FOCUS);
     gtk_widget_show(entry);
     gtk_table_attach(GTK_TABLE(table1), entry, 0, 1, 1, 2,
		      GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);

     gtk_table_set_row_spacing(GTK_TABLE(table1), 1, 10);

     /* Description */
     label = gtk_label_new("Description");
     gtk_misc_set_alignment(GTK_MISC(label), .0, .0);
     gtk_widget_show(label);
     gtk_table_attach(GTK_TABLE(table1), label, 0, 1, 2, 3,
		      GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);

     entry = gtk_entry_new();
     gtk_object_set_data(GTK_OBJECT(target_vbox), "description", entry);
     GTK_WIDGET_UNSET_FLAGS(GTK_ENTRY(entry), GTK_CAN_FOCUS);
     gtk_widget_show(entry);
     gtk_table_attach(GTK_TABLE(table1), entry, 0, 1, 3, 4,
		      GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);

     gtk_table_set_row_spacing(GTK_TABLE(table1), 3, 10);


     /* right half */

     paned = gtk_vpaned_new();
     gtk_paned_set_gutter_size(GTK_PANED(paned), 0);
     gtk_paned_set_handle_size(GTK_PANED(paned), 0);
     gtk_paned_set_position(GTK_PANED(paned), 157);
     gtk_widget_show(paned);
     gtk_box_pack_start(GTK_BOX(hbox1), paned, TRUE, TRUE, 10);

     /* Used in attribute types */
     scrwin = gtk_scrolled_window_new(NULL, NULL);
     gtk_widget_show(scrwin);
     gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrwin),
                                    GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);

     /* just about right for four lines in the top clist */
     gtk_widget_set_usize(scrwin, -1, 157);


     clist = gtk_clist_new_with_titles(1, atitle);
     gtk_widget_show(clist);
     GTK_CLIST(clist)->button_actions[1] = GTK_BUTTON_SELECTS;
     gtk_signal_connect(GTK_OBJECT(clist), "select_row",
			(GtkSignalFunc) select_at_from_clist, NULL);
     gtk_object_set_data(GTK_OBJECT(target_vbox), "usedat", clist);
     gtk_clist_column_titles_passive(GTK_CLIST(clist));
     GTK_WIDGET_UNSET_FLAGS(clist, GTK_CAN_FOCUS);
     gtk_container_add(GTK_CONTAINER(scrwin), clist);
     gtk_paned_pack1(GTK_PANED(paned), scrwin, FALSE, FALSE);

     /* Used in matching rules */
     scrwin = gtk_scrolled_window_new(NULL, NULL);
     gtk_widget_show(scrwin);
     gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrwin),
                                    GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);

     clist = gtk_clist_new_with_titles(1, mtitle);
     gtk_widget_show(clist);
     GTK_CLIST(clist)->button_actions[1] = GTK_BUTTON_SELECTS;
     gtk_signal_connect(GTK_OBJECT(clist), "select_row",
			(GtkSignalFunc) select_mr_from_clist, NULL);
     gtk_object_set_data(GTK_OBJECT(target_vbox), "usedmr", clist);
     gtk_clist_column_titles_passive(GTK_CLIST(clist));
     GTK_WIDGET_UNSET_FLAGS(clist, GTK_CAN_FOCUS);
     gtk_container_add(GTK_CONTAINER(scrwin), clist);
     gtk_paned_pack2(GTK_PANED(paned), scrwin, FALSE, FALSE);

}


void fill_s_detail_rightpane(GtkWidget *treeroot, GtkWidget *tree_item, GHashTable *hash)
{
     GtkWidget *rightpane_notebook, *s_vbox;
     LDAPSyntax *s;
     struct ldapserver *server;

     if( (server = gtk_object_get_data(GTK_OBJECT(treeroot), "server")) == NULL)
	  return;

     if( (s = gtk_object_get_data(GTK_OBJECT(tree_item), "s")) == NULL)
	  return;

     if(g_hash_table_lookup(hash, "cur_s") == s)
	  return;

     g_hash_table_insert(hash, "cur_s", s);

     s_vbox = g_hash_table_lookup(hash, "s_vbox");
     if(s_vbox == NULL) {
	  make_detail_notebook(hash);
	  s_vbox = g_hash_table_lookup(hash, "s_vbox");
     }

     fill_s_detail(s_vbox, server, s);

     /* switch to Syntaxes tab */
     rightpane_notebook = g_hash_table_lookup(hash, "rightpane_notebook");
     gtk_notebook_set_page(GTK_NOTEBOOK(rightpane_notebook), 3);

}


void fill_s_detail(GtkWidget *target_vbox, struct ldapserver *server,
		   LDAPSyntax *s)
{
     GList *list;
     GtkWidget *entry, *clist;
     int i;
     char *dummy[2];

     /* OID */
     entry = (GtkWidget *) gtk_object_get_data(GTK_OBJECT(target_vbox), "oid");
     gtk_entry_set_text(GTK_ENTRY(entry), s->syn_oid ? s->syn_oid : "");

     /* Description */
     entry = (GtkWidget *) gtk_object_get_data(GTK_OBJECT(target_vbox), "description");
     gtk_entry_set_text(GTK_ENTRY(entry), s->syn_desc ? s->syn_desc : "");

     /* Used in attribute types */
     clist = (GtkWidget *) gtk_object_get_data(GTK_OBJECT(target_vbox), "usedat");
     gtk_object_set_data(GTK_OBJECT(clist), "server", server);
     gtk_clist_freeze(GTK_CLIST(clist));
     gtk_clist_clear(GTK_CLIST(clist));

     list = find_at_by_s_oid(server, s->syn_oid);
     i = 0;
     dummy[1] = NULL;
     if(list) {
	  while(list) {
	       dummy[0] = list->data;
	       gtk_clist_insert(GTK_CLIST(clist), i, dummy);
	       i++;
	       list = list->next;
	  }
	  g_list_free(list);
     }
     gtk_clist_thaw(GTK_CLIST(clist));

     /* Used in matching rules */
     clist = (GtkWidget *) gtk_object_get_data(GTK_OBJECT(target_vbox), "usedmr");
     gtk_object_set_data(GTK_OBJECT(clist), "server", server);
     gtk_clist_freeze(GTK_CLIST(clist));
     gtk_clist_clear(GTK_CLIST(clist));

     list = find_mr_by_s_oid(server, s->syn_oid);
     i = 0;
     dummy[1] = NULL;
     if(list) {
	  while(list) {
	       dummy[0] = list->data;
	       gtk_clist_insert(GTK_CLIST(clist), i, dummy);
	       i++;
	       list = list->next;
	  }
	  g_list_free(list);
     }
     gtk_clist_thaw(GTK_CLIST(clist));

}


void select_oc_from_clist(GtkWidget *clist, gint row, gint column,
			  GdkEventButton *event, gpointer data)
{
     GList *list;
     LDAPObjectClass *oc;
     struct ldapserver *server;
     char *ocname;

     oc = NULL;

     /* double click or single middle button click */
     if( (event->type == GDK_BUTTON_RELEASE && event->button == 2) ||
	 (event->type == GDK_2BUTTON_PRESS && event->button == 1)) {

	  if( (server = gtk_object_get_data(GTK_OBJECT(clist), "server")) == NULL)
	       return;

	  if(server->ss == NULL || server->ss->oc == NULL)
	       return;

	  gtk_clist_get_text(GTK_CLIST(clist), row, column, &ocname);
	  list = server->ss->oc;
	  while(list) {
	       oc = list->data;
	       if(oc && oc->oc_names && oc->oc_names[0] &&
		  !strcasecmp(oc->oc_names[0], ocname))
		    break;
	       list = list->next;
	  }

	  if(list && oc)
	       popup_detail(SCHEMA_TYPE_OC, server, oc);

     }

}


void select_at_from_clist(GtkWidget *clist, gint row, gint column,
			  GdkEventButton *event, gpointer data)
{
     GList *list;
     LDAPAttributeType *at;
     struct ldapserver *server;
     char *attrname;

     at = NULL;

     /* double click or single middle button click */
     if( (event->type == GDK_BUTTON_RELEASE && event->button == 2) ||
	 (event->type == GDK_2BUTTON_PRESS && event->button == 1)) {

	  if( (server = gtk_object_get_data(GTK_OBJECT(clist), "server")) == NULL)
	       return;

	  if(server->ss == NULL || server->ss->at == NULL)
	       return;

	  gtk_clist_get_text(GTK_CLIST(clist), row, column, &attrname);
	  list = server->ss->at;
	  while(list) {
	       at = list->data;
	       if(at && at->at_names && at->at_names[0] &&
		  !strcasecmp(at->at_names[0], attrname))
		    break;
	       list = list->next;
	  }

	  if(list && at)
	       popup_detail(SCHEMA_TYPE_AT, server, at);

     }

}


void select_mr_from_clist(GtkWidget *clist, gint row, gint column,
			  GdkEventButton *event, gpointer data)
{
     GList *list;
     LDAPMatchingRule *mr;
     struct ldapserver *server;
     char *mrname;

     mr = NULL;

     /* double click or single middle button click */
     if( (event->type == GDK_BUTTON_RELEASE && event->button == 2) ||
	 (event->type == GDK_2BUTTON_PRESS && event->button == 1)) {

	  if( (server = gtk_object_get_data(GTK_OBJECT(clist), "server")) == NULL)
	       return;

	  if(server->ss == NULL || server->ss->mr == NULL)
	       return;

	  gtk_clist_get_text(GTK_CLIST(clist), row, column, &mrname);
	  list = server->ss->mr;
	  while(list) {
	       mr = list->data;
	       if(mr && mr->mr_names && mr->mr_names[0] &&
		  !strcasecmp(mr->mr_names[0], mrname))
		    break;
	       list = list->next;
	  }

	  if(list && mr)
	       popup_detail(SCHEMA_TYPE_MR, server, mr);

     }

}

#endif /* HAVE_LDAP_STR2OBJECTCLASS */

void cleanup_schema_mode(void)
{

     /* the whole thing is stateless! */

}


