/*
 * GooCanvas. Copyright (C) 2005 Damon Chaplin.
 * Released under the GNU LGPL license. See COPYING for details.
 *
 * goocanvasrectview.c - view for rect item.
 */

/**
 * SECTION:goocanvasrectview
 * @Title: GooCanvasRectView
 * @Short_Description: a view for a #GooCanvasRect item.
 *
 * #GooCanvasRectView represents a view of a #GooCanvasRect item.
 *
 * It implements the #GooCanvasItemView interface, so you can use the
 * #GooCanvasItemView functions such as goo_canvas_item_view_get_item()
 * and goo_canvas_item_view_get_bounds().
 *
 * Applications do not normally need to create item views themselves, as
 * they are created automatically by #GooCanvasView when needed.
 *
 * To respond to events such as mouse clicks in the ellipse view you can
 * connect to one of the #GooCanvasItemView signals such as
 * #GooCanvasItemView::button-press-event. You can connect to these signals
 * when the view is created. (See goo_canvas_view_get_item_view() and
 * #GooCanvasView::item-view-created.)
 */
#include <config.h>
#include <math.h>
#include <string.h>
#include <gtk/gtk.h>
#include "goocanvasrectview.h"
#include "goocanvasrect.h"


static void goo_canvas_rect_view_create_path (GooCanvasItemSimple *simple,
					      cairo_t             *cr);

G_DEFINE_TYPE_WITH_CODE (GooCanvasRectView, goo_canvas_rect_view,
			 GOO_TYPE_CANVAS_ITEM_VIEW_SIMPLE,
			 G_IMPLEMENT_INTERFACE (GOO_TYPE_CANVAS_ITEM_VIEW,
						NULL))


static void
goo_canvas_rect_view_class_init (GooCanvasRectViewClass *klass)
{
  GooCanvasItemViewSimpleClass *simple_view_class = (GooCanvasItemViewSimpleClass*) klass;

  simple_view_class->create_path = goo_canvas_rect_view_create_path;
}


static void
goo_canvas_rect_view_init (GooCanvasRectView *rect_view)
{

}


/**
 * goo_canvas_rect_view_new:
 * @canvas_view: the canvas view.
 * @parent_view: the parent view.
 * @rect: the rect item.
 * 
 * Creates a new #GooCanvasRectView for the given #GooCanvasRect item.
 *
 * This is not normally used by application code, as the views are created
 * automatically by #GooCanvasView.
 * 
 * Returns: a new #GooCanvasRectView.
 **/
GooCanvasItemView*
goo_canvas_rect_view_new (GooCanvasView     *canvas_view,
			  GooCanvasItemView *parent_view,
			  GooCanvasRect     *rect)
{
  GooCanvasItemViewSimple *view;

  view = g_object_new (GOO_TYPE_CANVAS_RECT_VIEW, NULL);
  view->canvas_view = canvas_view;
  view->parent_view = parent_view;
  view->item = g_object_ref (rect);

  goo_canvas_item_view_simple_setup_accessibility (view);

  g_signal_connect (rect, "changed",
		    G_CALLBACK (goo_canvas_item_view_simple_item_changed),
		    view);

  return (GooCanvasItemView*) view;
}


static void
goo_canvas_rect_view_create_path (GooCanvasItemSimple *simple,
				  cairo_t             *cr)
{
  GooCanvasRect *rect = (GooCanvasRect*) simple;

  cairo_new_path (cr);

  /* Check if we need to draw rounded corners. */
  if (rect->radius_x > 0 && rect->radius_y > 0)
    {
      /* The radii can't be more than half the size of the rect. */
      double rx = MIN (rect->radius_x, rect->width / 2);
      double ry = MIN (rect->radius_y, rect->height / 2);

      /* Draw the top-right arc. */
      cairo_save (cr);
      cairo_translate (cr, rect->x + rect->width - rx, rect->y + ry);
      cairo_scale (cr, rx, ry);
      cairo_arc (cr, 0.0, 0.0, 1.0, 1.5 * M_PI, 2.0 * M_PI);
      cairo_restore (cr);

      /* Draw the line down the right side. */
      cairo_line_to (cr, rect->x + rect->width, rect->y + rect->height - ry);

      /* Draw the bottom-right arc. */
      cairo_save (cr);
      cairo_translate (cr, rect->x + rect->width - rx,
		       rect->y + rect->height - ry);
      cairo_scale (cr, rx, ry);
      cairo_arc (cr, 0.0, 0.0, 1.0, 0.0, 0.5 * M_PI);
      cairo_restore (cr);

      /* Draw the line left across the bottom. */
      cairo_line_to (cr, rect->x + rx, rect->y + rect->height);

      /* Draw the bottom-left arc. */
      cairo_save (cr);
      cairo_translate (cr, rect->x + rx, rect->y + rect->height - ry);
      cairo_scale (cr, rx, ry);
      cairo_arc (cr, 0.0, 0.0, 1.0, 0.5 * M_PI, M_PI);
      cairo_restore (cr);

      /* Draw the line up the left side. */
      cairo_line_to (cr, rect->x, rect->y + ry);

      /* Draw the top-left arc. */
      cairo_save (cr);
      cairo_translate (cr, rect->x + rx, rect->y + ry);
      cairo_scale (cr, rx, ry);
      cairo_arc (cr, 0.0, 0.0, 1.0, M_PI, 1.5 * M_PI);
      cairo_restore (cr);

      /* Close the path across the top. */
      cairo_close_path (cr);
    }
  else
    {
      /* Draw the plain rectangle. */
      cairo_move_to (cr, rect->x, rect->y);
      cairo_line_to (cr, rect->x, rect->y + rect->height);
      cairo_line_to (cr, rect->x + rect->width, rect->y + rect->height);
      cairo_line_to (cr, rect->x + rect->width, rect->y);
      cairo_close_path (cr);
    }
}
