/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
  sattabs.c: Notebook tabs for main window.

  Copyright (C)  2001-2003  Alexandru Csete.

  Comments, questions and bugreports should be submitted via
  http://sourceforge.net/projects/groundstation/
  More details can be found at http://groundstation.sourceforge.net/
 
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the
          Free Software Foundation, Inc.,
	  59 Temple Place, Suite 330,
	  Boston, MA  02111-1307
	  USA
*/

#include <gnome.h>
#include <gconf/gconf.h>
#include <gconf/gconf-client.h>
#include "satlog.h"
#include "sattabs.h"

#ifdef HAVE_CONFIG_H
#  include "../config.h"
#endif


/* Main notebook widget */
static GtkWidget *notebook;

/* Determines whether tabs should be show even
   if there is only one page in the notebook.
   Can be overrided in config box.
*/
static gboolean alwaysshowtabs; 
static gint tabscnxn;

extern GConfClient *client;  /* main.c */


/* Private function prototypes */
static void sattabs_config_notify_cb (GConfClient *client, guint cnxn, GConfEntry *entry, gpointer data);


GtkWidget *sattabs_create ()
{
	/* This function creates a new notebook widget.
	   Should only be called once.
	*/

	notebook = gtk_notebook_new ();
	gtk_notebook_set_scrollable (GTK_NOTEBOOK (notebook), TRUE);
	gtk_notebook_popup_enable (GTK_NOTEBOOK (notebook));
	gtk_notebook_set_homogeneous_tabs (GTK_NOTEBOOK (notebook), TRUE);

	/* connect the destroy signal */

	/* get configuration values */
	alwaysshowtabs = gconf_client_get_bool (client, SATTABS_CONFIG_SHOW_TABS_PATH, NULL);
	if (alwaysshowtabs)
		gtk_notebook_set_show_tabs (GTK_NOTEBOOK (notebook), TRUE);
	else
		gtk_notebook_set_show_tabs (GTK_NOTEBOOK (notebook), FALSE);

	/* add a GConf notifier */
	tabscnxn = gconf_client_notify_add (client, SATTABS_CONFIG_PATH,
					    sattabs_config_notify_cb,
					    NULL, NULL, NULL);

	gtk_container_set_resize_mode (GTK_CONTAINER (notebook), GTK_RESIZE_QUEUE);
	
	return notebook;
}


void sattabs_add_tab (GtkWidget *widget, gchar *label)
{
	/* This function adds a new widget to the notebook.
	   The boolean parameter specifies whether the new tab should
	   have an optional close button.
	*/

	if (!label)
		satlog_log (SAT_LOG_CRITICAL, _("sattabs: Tryed to create an empty page (bug!)"));
	else {
		if (widget) {
			gchar *msg;

			/* Make a log message */
			msg = g_strdup_printf (_("SATTABS: Adding new page: %s"), label);
			satlog_log (SAT_LOG_DEBUG, msg);
			g_free (msg);

			/* Add page */
			gtk_notebook_append_page (GTK_NOTEBOOK (notebook), widget,
						  gtk_label_new (label));
			gtk_notebook_set_page (GTK_NOTEBOOK (notebook),
					       sattabs_lookup_tab (widget));

			if (!alwaysshowtabs) {
				gint cp;

				/* disable tabs if we only have one page */
				cp = g_list_length (GTK_NOTEBOOK (notebook)->children);
				if (cp < 2)
					gtk_notebook_set_show_tabs (GTK_NOTEBOOK (notebook), FALSE);
				else
					gtk_notebook_set_show_tabs (GTK_NOTEBOOK (notebook), TRUE);
			}
			else
				gtk_notebook_set_show_tabs (GTK_NOTEBOOK (notebook), TRUE);
		}
		else {
			satlog_log (SAT_LOG_CRITICAL,
				    _("SATTABS: Trying to insert an empty page! This is a bug!"));
		}
	}
}


void sattabs_add_tab_menu (GtkWidget *widget, gchar *label, gchar *menulabel)
{
	/* The same as above, but menulabel will appear in the notebook
	   popup menu instead of label. For example a map module can use
	   "name" as label and "name (map)" as menu label, thus giving a
	   better overview of which modules are in the notebook.
	*/

	if (!label)
		satlog_log (SAT_LOG_CRITICAL, _("sattabs: Tryed to create an empty page (bug!)"));
	else {
		if (!menulabel)
			menulabel = g_strdup (label);
		if (widget) {
			gchar *msg;

			/* Make a log message */
			msg = g_strdup_printf (_("SATTABS: Adding new page: %s"), menulabel);
			satlog_log (SAT_LOG_DEBUG, msg);
			g_free (msg);

			/* Add page */
			gtk_notebook_append_page_menu (GTK_NOTEBOOK (notebook), widget,
						       gtk_label_new (label),
						       gtk_label_new (menulabel));
			gtk_notebook_set_page (GTK_NOTEBOOK (notebook),
					       sattabs_lookup_tab (widget));

			if (!alwaysshowtabs) {
				gint cp;

				/* disable tabs if we only have one page */
				cp = g_list_length (GTK_NOTEBOOK (notebook)->children);
				if (cp < 2)
					gtk_notebook_set_show_tabs (GTK_NOTEBOOK (notebook), FALSE);
				else
					gtk_notebook_set_show_tabs (GTK_NOTEBOOK (notebook), TRUE);
			}
			else
				gtk_notebook_set_show_tabs (GTK_NOTEBOOK (notebook), TRUE);
		}
		else { /* wohooo... what's this? */
			satlog_log (SAT_LOG_CRITICAL,
				    _("SATTABS: Trying to insert an empty page! This is a bug!"));
		}
	}
}


gint sattabs_lookup_tab (GtkWidget *widget)
{
	/* This function returns the tab page number where
	   "widget" is located or -1 if "widget" is not in the
	   notebook.
	*/
	return gtk_notebook_page_num (GTK_NOTEBOOK (notebook), widget);
}


GtkWidget *sattabs_get_tab (guint pagenum)
{
	/* This function returns the module widget contained
	   in the notebook on page "pagenum".
	*/
	return gtk_notebook_get_nth_page (GTK_NOTEBOOK (notebook),
					  pagenum);
}

void sattabs_remove_tab_cb (GtkWidget *widget, gpointer pagenum)
{
	sattabs_remove_tab (GPOINTER_TO_INT (pagenum));
}


void sattabs_remove_tab (gint pagenum)
{
	/* This function removes the specified page
	   from the notebook.
	 */
	gchar *msg;

	msg = g_strdup_printf (_("SATTABS: Removing page number %d"), pagenum);
	satlog_log (SAT_LOG_DEBUG, msg);
	g_free (msg);

	gtk_notebook_remove_page (GTK_NOTEBOOK (notebook), pagenum);

	if (!alwaysshowtabs) {
		/* disable tabs if we only have one page left */
		pagenum = g_list_length (GTK_NOTEBOOK (notebook)->children);
		if (pagenum < 2)
			gtk_notebook_set_show_tabs (GTK_NOTEBOOK (notebook), FALSE);
		else
			gtk_notebook_set_show_tabs (GTK_NOTEBOOK (notebook), TRUE);
	}
}


void sattabs_remove_current_tab_cb (GtkWidget *widget, gpointer data)
{
	sattabs_remove_current_tab ();
}

void sattabs_remove_current_tab ()
{
	/* This function removes the currently
	   visible page from the notebook.
	*/

	gint pagenum;
	gchar *msg;

	pagenum = gtk_notebook_get_current_page (GTK_NOTEBOOK (notebook));
	gtk_notebook_remove_page (GTK_NOTEBOOK (notebook), pagenum);

	msg = g_strdup_printf (_("SATTABS: Removing current page (number %d)"), pagenum);
	satlog_log (SAT_LOG_DEBUG, msg);
	g_free (msg);
	
	if (!alwaysshowtabs) {
		/* disable tabs if we only have one page left */
		pagenum = g_list_length (GTK_NOTEBOOK (notebook)->children);
		if (pagenum < 2)
			gtk_notebook_set_show_tabs (GTK_NOTEBOOK (notebook), FALSE);
		else
			gtk_notebook_set_show_tabs (GTK_NOTEBOOK (notebook), TRUE);
	}
}

void sattabs_destroy ()
{
	/* this function cleans up the notebook widget
	   related stuff.
	 */

	gconf_client_notify_remove (client, tabscnxn);
}


static void 
sattabs_config_notify_cb (GConfClient *client, guint cnxn,
			  GConfEntry *entry, gpointer data)
{
	/* This function is called automatically by the GConf engine
	   whenever any changes in the sattabs configurations occur.
	*/

	satlog_log (SAT_LOG_DEBUG, _("SATTABS: Received notification from GConf engine."));

	/* always show tabs? */
	alwaysshowtabs = gconf_client_get_bool (client, SATTABS_CONFIG_SHOW_TABS_PATH, NULL);
	if (!alwaysshowtabs) {
		gint cp;

		cp = g_list_length (GTK_NOTEBOOK (notebook)->children);
		if (cp < 2)
			gtk_notebook_set_show_tabs (GTK_NOTEBOOK (notebook), FALSE);
		else
			gtk_notebook_set_show_tabs (GTK_NOTEBOOK (notebook), TRUE);
	}
	else
		gtk_notebook_set_show_tabs (GTK_NOTEBOOK (notebook), TRUE);
}
