/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*

  satpref-satmap.c: Map preferences (widgets and funcs).

  Copyright (C)  2001-2003  Alexandru Csete.

  Authors:  Alexandru Csete <csete@users.sourceforge.net>

  Comments, questions and bugreports should be submitted via
  http://sourceforge.net/projects/groundstation/
  More details can be found at http://groundstation.sourceforge.net/
 
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the
          Free Software Foundation, Inc.,
	  59 Temple Place, Suite 330,
	  Boston, MA  02111-1307
	  USA
*/

#include <gnome.h>
#include <gconf/gconf-client.h>
#include "defaults.h"
#include "satmap.h"
#include "satpref.h"
#include "satpref-satmap.h"

#ifdef HAVE_CONFIG_H
#include "../config.h"
#endif

extern GConfClient *client; /* main.c */

/* spin/adjustment pairs */
static GtkObject *mapadj,*grvadj,*grhadj;
static GtkWidget *mapspin,*grvspin,*grhspin;
static gchar *mapfile;

/* check boxes */
static GtkWidget *showgrid,*gridcp,*gridleg;


/* Private function prototypes */
static void       satpref_satmap_selector_cb (GtkButton *button, gpointer defmap);
static GtkWidget *satpref_satmap_grid_stuff  (void);
static GtkWidget *satpref_satmap_sun_stuff   (void);
static GtkWidget *satpref_satmap_moon_stuff  (void);
static GtkWidget *satpref_satmap_shade_stuff (void);


GtkWidget *satpref_satmap (void)
{
	/* This function creates the widgets for the Maps preferences page.
	 */
	GtkWidget *vbox,*table,*label,*mapbut,*defmap;
	gint delay;

	/* create table */
	table = gtk_table_new (4, 3, FALSE);
	gtk_table_set_col_spacings (GTK_TABLE (table), 2);
	gtk_table_set_row_spacings (GTK_TABLE (table), 2);

	/* satlist timer */
	label = gtk_label_new (_("Update the maps every"));
	gtk_misc_set_alignment (GTK_MISC (label), 0.0f, 0.5f);
	gtk_table_attach_defaults (GTK_TABLE (table), label, 1, 2, 0, 1);

	/* get timer value */
	delay = gconf_client_get_int (client, SATMAP_TIMER_PATH, NULL);
	if (!delay)
		delay = SATMAP_DEF_TIMER;

	mapadj = gtk_adjustment_new (delay, 100, 60000, 10, 100, 5);
	mapspin = gtk_spin_button_new (GTK_ADJUSTMENT (mapadj), 0.5, 0);
	gtk_signal_connect (GTK_OBJECT (mapspin), "changed",
			    GTK_SIGNAL_FUNC (satpref_changed), NULL);
	gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (mapspin), TRUE);
	gtk_table_attach_defaults (GTK_TABLE (table), mapspin, 2, 3, 0, 1);
	label = gtk_label_new (_(" [msec]") );
	gtk_misc_set_alignment (GTK_MISC (label), 0.0f, 0.5f);
	gtk_table_attach_defaults (GTK_TABLE (table), label, 3, 4, 0, 1);

	gtk_table_attach_defaults (GTK_TABLE (table), gtk_hseparator_new (), 0, 4, 1, 2);

	/* Default map label */
	label = gtk_label_new (_("Defualt Map"));
	gtk_misc_set_alignment (GTK_MISC (label), 0.0f, 0.5f);
	gtk_table_attach_defaults (GTK_TABLE (table), label, 1, 2, 2, 3);

	/* default map preview */
	mapfile = gconf_client_get_string (client, SATMAP_DEFMAP_PATH, NULL);
	if (!mapfile) {
		mapfile = g_strconcat (PACKAGE_PIXMAPS_DIR, G_DIR_SEPARATOR_S, "maps",
				       G_DIR_SEPARATOR_S, "earth_800.jpg", NULL);
	}
	defmap = gnome_pixmap_new_from_file_at_size (mapfile, PREVIEW_SIZE_X, PREVIEW_SIZE_Y);
	gtk_table_attach_defaults (GTK_TABLE (table), defmap, 2, 3, 2, 3);

	/* defulat map selector button */
	mapbut = gtk_button_new_with_label (_("Select..."));
	gtk_signal_connect (GTK_OBJECT (mapbut), "clicked",
			    GTK_SIGNAL_FUNC (satpref_satmap_selector_cb), defmap);
	gtk_table_attach (GTK_TABLE (table), mapbut, 3, 4, 2, 3, GTK_EXPAND | GTK_FILL, GTK_SHRINK, 0, 0);

	/* create vertical box */
	vbox = gtk_vbox_new (FALSE, 0);
	gtk_container_set_border_width (GTK_CONTAINER (vbox), 10);
	gtk_box_pack_start_defaults (GTK_BOX (vbox), table);
/* 	gtk_box_pack_start_defaults (GTK_BOX (vbox), satpref_satmap_grid_stuff ()); */
/* 	gtk_box_pack_start_defaults (GTK_BOX (vbox), satpref_satmap_sun_stuff ()); */
/* 	gtk_box_pack_start_defaults (GTK_BOX (vbox), satpref_satmap_moon_stuff ()); */
/* 	gtk_box_pack_start_defaults (GTK_BOX (vbox), satpref_satmap_shade_stuff ()); */
	
	return vbox;
}


static void
satpref_satmap_selector_cb (GtkButton *button, gpointer defmap)
{
	/* This function is called when the user clicks on the default map
	   selector button. It pops up a dialog showing a list of icons in
	   the standard gpredict map directory and in $HOME/.gpredict/maps/
	*/
	GtkWidget *dialog,*selector;
	gchar *dp1,*dp2;
	gint yesno;

	/* create directory entries where maps may be located */
	dp1 = g_strconcat (PACKAGE_PIXMAPS_DIR, G_DIR_SEPARATOR_S, "maps", NULL);
	dp2 = g_strconcat (g_get_home_dir (), G_DIR_SEPARATOR_S, ".gpredict",
			   G_DIR_SEPARATOR_S, "maps", NULL);

	/* Create the selector */
	selector = gnome_icon_selection_new ();
	gnome_icon_selection_add_directory (GNOME_ICON_SELECTION (selector), dp1);
	/* The user might not have installed extra maps */
	if (g_file_test (dp2, G_FILE_TEST_ISDIR)) {
		gnome_icon_selection_add_directory (GNOME_ICON_SELECTION (selector), dp2);
	}
	gnome_icon_selection_show_icons (GNOME_ICON_SELECTION (selector));

	/* Create dialog box */
	dialog = gnome_dialog_new (_("Select a new map"),
				   GNOME_STOCK_BUTTON_OK,
				   GNOME_STOCK_BUTTON_CANCEL,
				   NULL);
	/* we need to keep the dialog so that we can access the file name of the
	   selection after the dialog is closed.
	*/
	gnome_dialog_close_hides (GNOME_DIALOG (dialog), TRUE);

	gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (dialog)->vbox), selector, TRUE, TRUE, 0);
	gtk_widget_show_all (dialog);

	/* run the dialog */
	yesno = gnome_dialog_run_and_close (GNOME_DIALOG (dialog));

	if (!yesno) {
		if (gnome_icon_selection_get_icon (GNOME_ICON_SELECTION (selector), TRUE)) {
			/* we have a selection */
			g_free (mapfile);
			mapfile = g_strdup (gnome_icon_selection_get_icon (GNOME_ICON_SELECTION (selector), TRUE));

			/* update the preview widget */
			gnome_pixmap_load_file_at_size (GNOME_PIXMAP (defmap), mapfile,
							PREVIEW_SIZE_X, PREVIEW_SIZE_Y);

			satpref_changed ();
		}
	}
	
	/* we are done with the widgets in the dialog. close it */
	gnome_dialog_close_hides (GNOME_DIALOG (dialog), FALSE);
	gnome_dialog_close (GNOME_DIALOG (dialog));

	g_free (dp1);
	g_free (dp2);
}



void 
satpref_satmap_apply (void)
{
	/* timer value */
	gconf_client_set_int (client, SATMAP_TIMER_PATH,
			     gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (mapspin)),
			     NULL);
	/* default map */
	if (mapfile) {
		gconf_client_set_string (client, SATMAP_DEFMAP_PATH, mapfile, NULL);
	}
}




static GtkWidget *
satpref_satmap_grid_stuff  (void)
{
	/* This function creates the widgets used to edit
	   the grid related preferences.
	*/
	GtkWidget *frame,*table;
	GtkTooltips *kooltips;

	kooltips = gtk_tooltips_new ();

	/* show-grid check box */
	showgrid = gtk_check_button_new_with_label (_("Show grid"));
	gtk_tooltips_set_tip (kooltips, showgrid,
			      _("Select whether a grid with the specified color "\
				"and resolution should be shown on the maps."),
			      NULL);

	/* show grid legend check box */
	gridleg = gtk_check_button_new_with_label (_("Show grid annotations"));
	gtk_tooltips_set_tip (kooltips, gridleg,
			      _("Select whether the grid annotations should be "\
				"shown or not."),
			      _("Select whether the grid annotations should be "\
				"shown or not."));

	/* color picker */
	gridcp = gnome_color_picker_new ();
	gtk_tooltips_set_tip (kooltips, gridcp,
			      _("Click to select the grid color."),
			      _("Click to select the grid color."));

	/* horizontal resolution */
	grhadj = gtk_adjustment_new (40, 5, 80, 5, 20, 0);
	grhspin = gtk_spin_button_new (GTK_ADJUSTMENT (grhadj), 5, 0);
	gtk_tooltips_set_tip (kooltips, grhspin,
			      _("Specifies the grid resolution in the horizontal direction."),
			      _("Specifies the grid resolution in the horizontal direction."));

	/* vertical resolution */
	grvadj = gtk_adjustment_new (15, 5, 45, 5, 15, 0);
	grvspin = gtk_spin_button_new (GTK_ADJUSTMENT (grvadj), 5, 0);
	gtk_tooltips_set_tip (kooltips, grvspin,
			      _("Specifies the grid resolution in the vertical direction."),
			      _("Specifies the grid resolution in the vertical direction."));

	/* Table */
	table = gtk_table_new (2, 5, FALSE);
	gtk_table_attach_defaults (GTK_TABLE (table), showgrid,
				   0, 1, 0, 1);
	gtk_table_attach_defaults (GTK_TABLE (table), gridcp,
				   1, 2, 0, 1);
	gtk_table_attach_defaults (GTK_TABLE (table), gridleg,
				   0, 2, 1, 2);
	gtk_table_attach_defaults (GTK_TABLE (table), gtk_label_new (_("Horizontal resolution:")),
				   2, 3, 0, 1);
	gtk_table_attach_defaults (GTK_TABLE (table), grhspin,
				   3, 4, 0, 1);
	gtk_table_attach_defaults (GTK_TABLE (table), gtk_label_new (_("[deg]")),
				   4, 5, 0, 1);
	gtk_table_attach_defaults (GTK_TABLE (table), gtk_label_new (_("Vertical resolution:")),
				   2, 3, 1, 2);
	gtk_table_attach_defaults (GTK_TABLE (table), grvspin,
				   3, 4, 1, 2);
	gtk_table_attach_defaults (GTK_TABLE (table), gtk_label_new (_("[deg]")),
				   4, 5, 1, 2);

	/* create a nice frame */
	frame = gtk_frame_new (_("Grid"));
	gtk_container_add (GTK_CONTAINER (frame), table);

	gtk_widget_set_sensitive (frame, FALSE);

	return frame;
}

static GtkWidget *
satpref_satmap_sun_stuff   (void)
{
	/* This function creates the widgets used to edit
	   Sun related preferences.
	*/
	GtkWidget *frame;

	frame = gtk_frame_new (_("Sun"));

	return frame;
}


static GtkWidget *
satpref_satmap_moon_stuff  (void)
{
	/* This function creates the widgets used to edit
	   Moon related preferences.
	*/
	GtkWidget *frame;

	frame = gtk_frame_new (_("Moon"));

	return frame;
}


static GtkWidget *
satpref_satmap_shade_stuff (void)
{
	/* This function creates the widgets used to edit
	   Shading related preferences.
	*/
	GtkWidget *frame;

	frame = gtk_frame_new (_("Shading"));

	return frame;
}

