/***************************************************************************
                          gpc_print.c  -  description
                             -------------------
    begin                : Sun Jul 01 2001
    copyright            : (C) 2000,2001 by Thierry Florac
    email                : tflorac@free.fr
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/


#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <gnome.h>
#include <iconv.h>

#include "gpc_app.h"

#include "gpc_print.h"
#include "gpc_film.h"
#include "gpc_image.h"
#include "gpc_query.h"


#ifndef HAVE_GNOME_PRINT

void gpc_print_index (GpcFilmInfos *infos)
{
  gnome_dialog_run_and_close (GNOME_DIALOG (gnome_error_dialog (_("Printing support not supported !"))));
  return;
}


void gpc_print_image (GpcImageInfos *infos)
{
  gnome_dialog_run_and_close (GNOME_DIALOG (gnome_error_dialog (_("Printing support not supported !"))));
  return;
}

#else

#include <libgnomeprint/gnome-print.h>
#include <libgnomeprint/gnome-print-dialog.h>
#include <libgnomeprint/gnome-print-master.h>
#include <libgnomeprint/gnome-print-master-preview.h>


typedef struct _PrintInfo {

  // Gnome Print infos
  const GnomePaper   *paper;
  GnomePrintMaster   *master;
  GnomePrintContext  *context;

  // GPC Print infos
  gint     current_page;
  gfloat   page_width;
  gfloat   page_height;
  gfloat   margin_top;
  gfloat   margin_bottom;
  gfloat   margin_left;
  gfloat   margin_right;
  gfloat   printable_width;
  gfloat   printable_height;

  // GPC Print fonts
  GnomeFont *header_font;
  GnomeFont *image_font;
  gchar     *header_font_name;
  gint       header_font_size;
  gchar     *image_font_name;
  gint       image_font_size;

  // Charset conversion data
  iconv_t    conv_data;

} PrintInfo;


static PrintInfo *print_info_new (GnomePrintDialog *dlg)
{
    PrintInfo  *pi = g_new0 (PrintInfo, 1);

    pi->paper = gnome_paper_with_name (gnome_config_get_string ("/gphotocoll/printing/papersize=A4"));
    pi->master = gnome_print_master_new_from_dialog (dlg);
    pi->context = gnome_print_master_get_context (pi->master);

    pi->page_width = gnome_paper_pswidth (pi->paper);
    pi->page_height = gnome_paper_psheight (pi->paper);

    /* Margins are given in inches, and converted to 72 dpi */
    pi->margin_top = 0.6 * 72;
    pi->margin_bottom = 0.4 * 72;
    pi->margin_left = 1.0 * 72;
    pi->margin_right = 0.4 * 72;
    pi->printable_width =	pi->page_width - pi->margin_left - pi->margin_right;
    pi->printable_height = pi->page_height - pi->margin_top - pi->margin_bottom;

    /* Font sizes */
    pi->header_font_name = gnome_config_get_string ("/gphotocoll/printing/header_font_name=Helvetica");
    pi->header_font_size = gnome_config_get_int ("/gphotocoll/printing/header_font_size=10");
    pi->image_font_name = gnome_config_get_string ("/gphotocoll/printing/image_font_name=Helvetica");
    pi->image_font_size = gnome_config_get_int ("/gphotocoll/print/image_font_size=8");
    pi->header_font = gnome_font_new (pi->header_font_name, pi->header_font_size);
    pi->image_font = gnome_font_new (pi->image_font_name, pi->image_font_size);

    /* Conversion data */
    pi->conv_data = iconv_open ("utf8", "iso-8859-1");

    return pi;
}


static void 
print_info_destroy (PrintInfo *pi)
{
  if (pi->conv_data != (iconv_t) (-1))
    iconv_close (pi->conv_data);
  gtk_object_unref (GTK_OBJECT (pi->image_font));
  gtk_object_unref (GTK_OBJECT (pi->header_font));
  g_free (pi->image_font_name);
  g_free (pi->header_font_name);
  g_free (pi);
}


/* Utility function (extracted from Balsa mail user agent )
   Try to print with a correct charset */
static void
gnome_print_show_with_charset (PrintInfo *pi, const char *text)
{
  /* if we can not convert to utf8, try to print "raw" (which might fail) */
  if (pi->conv_data == (iconv_t)(-1))
    gnome_print_show(pi->context, text);
  else {
    gchar *conv_ibuf, *conv_ibufp, *conv_obuf, *conv_obufp;
    size_t ibuflen, obuflen;
    
    conv_ibuf = conv_ibufp = g_strdup (text);
    ibuflen = strlen(conv_ibuf) + 1;
    obuflen = ibuflen << 1;
    conv_obuf = conv_obufp = g_malloc(obuflen);
    /* the prototype of iconv() changed with glibc 2.2 */
#if defined __GLIBC__ && __GLIBC__ && __GLIBC_MINOR__ <= 1
    iconv(pi->conv_data, (const char **)&conv_ibuf, &ibuflen, &conv_obuf, &obuflen);
#else
    iconv(pi->conv_data, &conv_ibuf, &ibuflen, &conv_obuf, &obuflen);
#endif
    gnome_print_show(pi->context, conv_obufp);
    g_free (conv_ibufp);
    g_free (conv_obufp);
  }
}


static void start_new_page (PrintInfo *pi, gchar *header1, gchar *header2, gchar *header3)
{
  gchar      *page;
  gint        text_width;

  if (pi->current_page)
    gnome_print_showpage (pi->context);
  pi->current_page++;
  page = g_strdup_printf (_("Page %d"), pi->current_page);
  gnome_print_beginpage (pi->context, page);
  gnome_print_setfont (pi->context, pi->header_font);
  gnome_print_moveto (pi->context, pi->margin_left, pi->page_height - pi->margin_top);
  gnome_print_show_with_charset (pi, header1);
  text_width = gnome_font_get_width_string (pi->header_font, header2);
  gnome_print_moveto (pi->context, pi->margin_left + pi->printable_width - text_width, pi->page_height - pi->margin_top);
  gnome_print_show_with_charset (pi, header2);
  gnome_print_setfont (pi->context, pi->image_font);
  gnome_print_moveto (pi->context, pi->margin_left, pi->page_height - pi->margin_top - pi->header_font_size);
  gnome_print_show_with_charset (pi, header3);
  text_width = gnome_font_get_width_string (pi->image_font, page);
  gnome_print_moveto (pi->context, pi->margin_left + pi->printable_width - text_width, pi->page_height - pi->margin_top - pi->header_font_size);
  gnome_print_show_with_charset (pi, page);
  g_free (page);
}


void internal_print_index (GpcFilmInfos *film_infos, PrintInfo *pi)
{
  double matrix[6] = {0.0, 0.0, 0.0, 0.0, 0.0, 0.0};
  pgQuery       *gpc_images;
  GpcImageInfos *image_infos;
  GdkPixbuf     *image_pixbuf;
  GdkPixbuf     *index_pixbuf;
  gint    ID;
  gint    cols;
  gint    rows;
  gint    x;
  gint    y;
  gint    image_width;
  gint    image_height;
  gfloat  index_image_width;
  gfloat  index_image_height;
  gfloat  index_width;
  gfloat  index_height;
  gfloat  coeff_x;
  gfloat  coeff_y;
  gfloat  coeff;
  gfloat  text_width;
  gfloat  header_height;
  gchar  *description;
  gchar  *number;
  gchar  *pathname;
  gchar  *filename;
  gchar  *fullname;

  if (!film_infos)
    return;
  header_height = pi->header_font_size;
  description = gpc_film_get_description (film_infos);
  number = gpc_film_get_number (film_infos);
  pathname = gpc_film_get_full_pathname (film_infos);
  start_new_page (pi, description, number, pathname);
  if (gpc_query_mode)
    gpc_images = gpc_query_get_images (gpc_connection, gpc_film_get_id (film_infos));
  else
    gpc_images = gpc_image_get_list_from_film (gpc_connection, gpc_film_get_id (film_infos));
  if (gpc_images) {
    index_image_width = 72;
    index_image_height = 72;
    index_width = index_image_width + (0.1 * 72);
    index_height = index_image_height + (0.3 * 72);
    cols = (gint) (pi->printable_width / index_width);
    if (cols == 0)
      cols = 1;
    rows = (gint) pg_query_get_recordcount (gpc_images) == 1 ? 1 : ((pg_query_get_recordcount (gpc_images)-1) / cols)+1;
    x = 0;
    y = 1;
    pg_query_move_first (gpc_images);
    while (!pg_query_at_EOF (gpc_images)) {
      ID = atoi (pg_query_get_fieldvalue_byname (gpc_images, "ID"));
      image_infos = gpc_image_get_infos (gpc_connection, ID);
      filename = gpc_image_get_filename (image_infos);
      fullname = g_strconcat (pathname, filename, NULL);
      image_pixbuf = gdk_pixbuf_new_from_file (fullname);
      if (image_pixbuf) {
        image_width = gdk_pixbuf_get_width (image_pixbuf);
        image_height = gdk_pixbuf_get_height (image_pixbuf);
        coeff_x = (gfloat) index_image_width / image_width;
        coeff_y = (gfloat) index_image_height / image_height;
        (coeff_x > coeff_y) ? (coeff = coeff_y) : (coeff = coeff_x);
        if (coeff > 1)
          coeff = 1.0;
        /* Index images are *3 to obtain minimum resolution... */
        index_pixbuf = gdk_pixbuf_scale_simple (image_pixbuf, image_width * coeff * 3, image_height * coeff * 3, GDK_INTERP_NEAREST);
        if (index_pixbuf) {
          gdk_pixbuf_unref (image_pixbuf);
          gnome_print_gsave (pi->context);
          matrix[0] = image_width * coeff;
          matrix[3] = image_height * coeff;
          matrix[4] = pi->margin_left + (x * index_width) + (index_width - matrix[0]) / 2;
          matrix[5] = (pi->margin_bottom + pi->printable_height) + ((index_height - matrix[3]) / 2) - (y * index_height) - header_height;
          gnome_print_concat (pi->context, matrix);
          gnome_print_pixbuf (pi->context, index_pixbuf);
          gnome_print_grestore (pi->context);
          gdk_pixbuf_unref (index_pixbuf);
          text_width = gnome_font_get_width_string (pi->image_font, filename);
          gnome_print_moveto (pi->context, 
                              pi->margin_left + (x * index_width) + (index_width - text_width) / 2,
                              pi->margin_bottom + pi->printable_height - (y * index_height) - header_height);
          gnome_print_show_with_charset (pi, filename);
        }
        if (++x == cols) {
          x = 0; 
          y++;
          if (pi->printable_height - header_height - (y * index_height) < 0) {
            start_new_page (pi, description, number, pathname);
            y = 1;
          }
        }
        while (gtk_events_pending ())
          gtk_main_iteration ();
      }
      g_free (fullname);
      pg_query_move_next (gpc_images);
    }
    pg_query_free (gpc_images);
  }
  g_free (pathname);
  gnome_print_showpage (pi->context);
}


void gpc_print_index (GpcFilmInfos *infos)
{
  GtkWidget    *dialog;
  PrintInfo    *pi;
  gboolean      preview = FALSE;

  if (!infos)
    return;
  dialog = gnome_print_dialog_new (_("Print index"), GNOME_PRINT_DIALOG_COPIES);
  gnome_dialog_set_parent (GNOME_DIALOG(dialog),
                           GTK_WINDOW (gpc_app));
  gtk_window_set_wmclass (GTK_WINDOW (dialog), "gpc_print_index", "gphotocoll");
  switch (gnome_dialog_run (GNOME_DIALOG (dialog))) {
    case GNOME_PRINT_PRINT:
      break;
    case GNOME_PRINT_PREVIEW:
      preview = TRUE;
      break;
    case GNOME_PRINT_CANCEL:
      gnome_dialog_close (GNOME_DIALOG (dialog));
    default:
      return;
  }
  pi = print_info_new (GNOME_PRINT_DIALOG (dialog));
  gnome_dialog_close (GNOME_DIALOG (dialog));
  internal_print_index (infos, pi);
  gnome_print_master_close (pi->master);
  if (preview) {
    GnomePrintMasterPreview *preview_widget = gnome_print_master_preview_new (pi->master, _("GPC: index preview"));
    gtk_widget_show (GTK_WIDGET (preview_widget));
  } 
  else {
    gnome_print_master_print (pi->master);
    gtk_object_unref (GTK_OBJECT (pi->master));
  }
  print_info_destroy (pi);
}


void internal_print_image (GpcImageInfos *image_infos, PrintInfo *pi)
{
  double matrix[6] = {0.0, 0.0, 0.0, 0.0, 0.0, 0.0};
  GpcFilmInfos  *film_infos;
  GdkPixbuf     *image_pixbuf;
  gint           image_width;
  gint           image_height;
  gfloat         header_height;
  gfloat         coeff_x;
  gfloat         coeff_y;
  gfloat         coeff;
  gchar         *description;
  gchar         *image_number;
  gchar         *film_number;
  gchar         *number;
  gchar         *pathname;
  gchar         *filename;
  gchar         *fullname;

  if (!image_infos)
    return;
  film_infos = gpc_film_get_infos (gpc_connection, gpc_image_get_film_id (image_infos));
  header_height = pi->header_font_size;
  description = gpc_image_get_description (image_infos);
  image_number = gpc_image_get_number (image_infos);
  film_number = gpc_film_get_number (film_infos);
  number = g_strconcat (film_number, " / ", image_number, NULL);
  pathname = gpc_film_get_full_pathname (film_infos);
  filename = gpc_image_get_filename (image_infos);
  fullname = g_strconcat (pathname, filename, NULL);
  start_new_page (pi, description, number, fullname);
  image_pixbuf = gdk_pixbuf_new_from_file (fullname);
  if (image_pixbuf) {
    image_width = gdk_pixbuf_get_width (image_pixbuf);
    image_height = gdk_pixbuf_get_height (image_pixbuf);
    coeff_x = (gfloat) pi->printable_width / image_width;
    coeff_y = (gfloat) (pi->printable_height - pi->header_font_size) / image_height;
    (coeff_x > coeff_y) ? (coeff = coeff_y) : (coeff = coeff_x);
    if (coeff > 1)
      coeff = 1.0;
    gnome_print_gsave (pi->context);
    matrix[0] = image_width * coeff;
    matrix[3] = image_height * coeff;
    matrix[4] = pi->margin_left + (pi->printable_width / 2) - (matrix[0] / 2);
    matrix[5] = pi->margin_bottom + (pi->printable_height / 2) - (matrix[3] / 2) - header_height;
    gnome_print_concat (pi->context, matrix);
    gnome_print_pixbuf (pi->context, image_pixbuf);
    gnome_print_grestore (pi->context);
    gdk_pixbuf_unref (image_pixbuf);
  }
  g_free (fullname);
  g_free (pathname);
  g_free (number);
  gnome_print_showpage (pi->context);
}


void gpc_print_image (GpcImageInfos *infos)
{
  GtkWidget  *dialog;
  PrintInfo  *pi;
  gboolean    preview = FALSE;

  if (!infos)
    return;
  dialog = gnome_print_dialog_new (_("Print image"), GNOME_PRINT_DIALOG_COPIES);
  gnome_dialog_set_parent (GNOME_DIALOG(dialog),
                           GTK_WINDOW (gpc_app));
  gtk_window_set_wmclass (GTK_WINDOW (dialog), "gpc_print_image", "gphotocoll");
  switch (gnome_dialog_run (GNOME_DIALOG (dialog))) {
    case GNOME_PRINT_PRINT:
      break;
    case GNOME_PRINT_PREVIEW:
      preview = TRUE;
      break;
    case GNOME_PRINT_CANCEL:
      gnome_dialog_close (GNOME_DIALOG (dialog));
    default:
      return;
  }
  pi = print_info_new (GNOME_PRINT_DIALOG (dialog));
  gnome_dialog_close (GNOME_DIALOG (dialog));
  internal_print_image (infos, pi);
  gnome_print_master_close (pi->master);
  if (preview) {
    GnomePrintMasterPreview *preview_widget = gnome_print_master_preview_new (pi->master, _("GPC: index preview"));
    gtk_widget_show (GTK_WIDGET (preview_widget));
  } 
  else {
    gnome_print_master_print (pi->master);
    gtk_object_unref (GTK_OBJECT (pi->master));
  }
  print_info_destroy (pi);
}


#endif

