/***************************************************************************
 *            qof-expenses.h
 *
 *  Thu Oct 21 08:18:44 2004
 *  Copyright  2004,2005,2006,2009  Neil Williams  <linux@codehelp.co.uk>
 ****************************************************************************/
/*
    This package is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/** @addtogroup QOF
	@{ */
/** @addtogroup Exp Expenses

	Like ::DB, Expenses uses UTC time within QOF and sets localtime when
	calling pilot-link code.

Gradually becoming a standard QOF method, whilst retaining compatibility
with the simple pilot-link currency handling instead of going the whole
hog to GnuCash currencies. The old Expense_t has been removed for easier
sharing between applications.
 
The typed enumerators ::ExpenseDistance, ::ExpenseType and ::ExpensePayment 
may look strange initially because macros are used to ensure
that the enum value and a string version of the enum are always in sync.
This allows the QOF backends to write human-readable data stores by specifying
an ExpenseType of "Parking" as:
 \verbatim
<string type="type_of_expense">Parking</string>
 \endverbatim
instead of:
 \verbatim
<string type="type_of_expense">5</string>
 \endverbatim

In particular, do not let the macro syntax <tt>_(name, value)</tt> fool you
into thinking it is related to <tt>_(String)</tt> from gettext! Note how the
enumerator is handled by doxygen:
 \verbatim
enum ExpenseType {
  Airfare = 0, Breakfast = 1, Bus = 2, BusinessMeals, ...
 \endverbatim

Translation must only be used to represent the string version of the enum
value in the user interface and all internal handling of the enum value
must remain untranslated in the QofEntity itself
as this would lock the data into a specific locale.

 \note Translation support in the user interface requires a little
adjustment of the display routines. Do \b NOT use translated
strings in the ::QofEntity - only in the user interface. The problem
is that although only the UI cares, the translated strings themselves
are only accessible via the library. See ::ExpenseType and
::ExpensePayment.

	@{ */
/** @file  qof-expenses.h
	@brief QOF expense definitions
	@author Copyright (c) 2004-2009 Neil Williams <linux@codehelp.co.uk>
*/

#ifndef _QOF_EXPENSES_H
#define _QOF_EXPENSES_H

#include <glib.h>
#include <qof.h>

/** @bug need to support qof-expenses and gpe-expenses */
#define GPE_MOD_EXP "gpe-expenses-module"
#define GPE_QOF_EXPENSES "gpe_expenses"
#define QOF_EXPENSES_DESC "GPE QOF expenses"
#define EXP_DATE      "expense_date"
#define EXP_TYPE      "type_of_expense"
#define EXP_PAYMENT   "form_of_payment"
#define EXP_CURRENCY  "currency_code"
#define EXP_AMOUNT    "expense_amount"
#define EXP_VENDOR    "expense_vendor"
#define EXP_CITY      "expense_city"
#define EXP_ATTENDEES "expense_attendees"
#define EXP_NOTE      "expense_note"
#define EXP_CATEGORY  "category"
#define EXP_DISTANCE  "distance_unit"
#define EXP_KVP       "currency_lookup"

/** type:  KVP_TYPE_STRING - inherited from pilot-link via pilot-qof. */
#define PQ_CURRENCY_MNEMONIC "expense/currency/mnemonic"  
/** type:  KVP_TYPE_STRING - inherited from pilot-link via pilot-qof. */
#define PQ_CURRENCY_SYMBOL   "expense/currency/symbol"
/** type:  KVP_TYPE_GINT64 - inherited from pilot-link via pilot-qof. */
#define PQ_CURRENCY_FRACTION "expense/currency/fraction"


/** \brief Wraps QOF around pilot-link expenses */
typedef struct QofExp_s QofExp;

typedef void (*QofCurrencyCB) (gpointer key, gpointer value, gpointer user_data);

void qof_currency_foreach(QofCurrencyCB cb, gpointer user_data);

/** \brief qof-expenses currency data wrapper.

Uses the same format as the Palm Default Currency Table:

http://pilot-qof.sourceforge.net/manual.html#currency
*/
typedef struct 
{
	/** pilot-link integer ID */
	gint            pq_code;
	/** currency fraction, pennies/cents etc. */
	gint            fraction;
	/** locale symbol. */
	const gchar* 	symbol;
	/** gnucash/cashutil mnemonic. */
	const gchar*	mnemonic;
	/** TRUE if symbol is not UTF-8. (Currently,
	this is only useful if the symbol can still be shown
	as iso-8859-1.) */	
	gboolean        non_utf8;
}QofCurrency;

/** \brief Set details of a QofCurrency from a currency_code

Lookup details of a QofCurrency from a short identifier. The
QofInstance will have a KvpFrame set (replacing any existing
values for the same keys) containing:
 -::PQ_CURRENCY_MNEMONIC
 -::PQ_CURRENCY_FRACTION
 -::PQ_CURRENCY_SYMBOL

\param inst The QofInstance to hold the KvpFrame containing the
	details of the currency.
\param currency_code The identifier of this currency.

\return a pointer to the QofCurrency identified by the
	currency_code or NULL on error.

*/
QofCurrency *
qof_currency_lookup (QofInstance* inst, gint currency_code);

/** \brief Lookup a currency by name

Slower than ::gpe_currency_lookup because the entire currency
hash must be searched. Otherwise, equivalent.

*/
QofCurrency *
qof_currency_lookup_name (QofInstance* inst, gchar * mnemonic);

#define DISTANCE_LIST(_) \
 _(Miles, = 0) \
 _(Kilometers,)

#define PAYMENT_TYPE_LIST(_) \
 _(AmericanExpress, = 0) \
 _(Cash, = 1 ) \
 _(Cheque, ) \
 _(CreditCard, ) \
 _(MasterCard, ) \
 _(Prepaid, ) \
 _(VISA, ) \
 _(Unfiled, )

#define EXPENSE_TYPE_LIST(_) \
 _(Airfare, = 0) \
 _(Breakfast, = 1) \
 _(Bus, = 2) \
 _(BusinessMeals, ) \
 _(CarRental, ) \
 _(Dinner, ) \
 _(Entertainment, ) \
 _(Fax, ) \
 _(Gas, ) \
 _(Gifts, ) \
 _(Hotel, ) \
 _(Incidentals, ) \
 _(Laundry, ) \
 _(Limo, ) \
 _(Lodging, ) \
 _(Lunch, ) \
 _(Mileage, ) \
 _(Other, ) \
 _(Parking, ) \
 _(Postage, ) \
 _(Snack, ) \
 _(Subway, ) \
 _(Supplies, ) \
 _(Taxi, ) \
 _(Telephone, ) \
 _(Tips, ) \
 _(Tolls, ) \
 _(Train, )

DEFINE_ENUM(ExpenseDistance, DISTANCE_LIST)
/** \brief typedef for expense payment types.

 \note These \b can be translated but it requires a little
adjustment of the display routines. Do \b NOT use translated
strings in the ::QofEntity - only in the user interface. The problem
is that although only the UI cares, the translated strings themselves
are only accessible via the library. This requires the use of
<tt>dgettext</tt> not the ordinary <tt>gettext</tt> or the
<tt>_(String)</tt> macro.

In your UI, to retrieve the translated string for an ExpensePayment,
wrap ExpenseTypeasString in a call to the library translation support:

 \code
ExpensePayment i = Cash;
GtkComboBox * my_combo_box;
// set i to the enum value to display as a translated string
gtk_combo_box_append_text (my_combo_box, 
	dgettext(LIBRARY_GETTEXT_PACKAGE, ExpensePaymentasString(i)));
// the combo_box has a translation of "Cash" added to the list.
 \endcode

You may choose to define your own macro for this.

Other translatable strings in your application code must continue
to use <tt>gettext</tt> or the <tt>_(String)</tt> macro as normal.

In accordance with the gettext manual, the library code does \b not 
call <tt>setlocale (LC_ALL, "")</tt>. It is the responsibility of the 
main program to set the locale. 

*/
DEFINE_ENUM(ExpensePayment, PAYMENT_TYPE_LIST)

/** \brief typedef for expense types

 \note These \b can be translated but it requires a little
adjustment of the display routines. Do \b NOT use translated
strings in the ::QofEntity - only in the user interface. The problem
is that although only the UI cares, the translated strings themselves
are only accessible via the library. This requires the use of
<tt>dgettext</tt> not the ordinary <tt>gettext</tt> or the
<tt>_(String)</tt> macro.

In your UI, to retrieve the translated string for an ExpenseType,
wrap ExpenseTypeasString in a call to the library translation support:

 \code
GtkComboBox * my_combo_box;
ExpenseType i = Telephone;
// set i to the enum value to display as a translated string
gtk_combo_box_append_text (my_combo_box, 
	dgettext(LIBRARY_GETTEXT_PACKAGE, ExpenseTypeasString(i)));
// the combo_box has a translation of "Telephone" added to the list.
 \endcode

You may choose to define your own macro for this.

Other translatable strings in your application code must continue
to use <tt>gettext</tt> or the <tt>_(String)</tt> macro as normal.

In accordance with the gettext manual, the library code does \b not 
call <tt>setlocale (LC_ALL, "")</tt>. It is the responsibility of the 
main program to set the locale. 

*/
DEFINE_ENUM(ExpenseType, EXPENSE_TYPE_LIST)

AS_STRING_DEC  (ExpenseDistance, DISTANCE_LIST)
FROM_STRING_DEC(ExpenseDistance, DISTANCE_LIST)
AS_STRING_DEC  (ExpensePayment, PAYMENT_TYPE_LIST)
FROM_STRING_DEC(ExpensePayment, PAYMENT_TYPE_LIST)
AS_STRING_DEC  (ExpenseType, EXPENSE_TYPE_LIST)
FROM_STRING_DEC(ExpenseType, EXPENSE_TYPE_LIST)

/** \brief Register this object with QOF

In accordance with the gettext manual, the library code does \b not 
call <tt>setlocale (LC_ALL, "")</tt>. It is the responsibility of the 
main program to set the locale. The library just calls bindtextdomain
with the domain for the library, LIBRARY_GETTEXT_PACKAGE which is
exported via the libqofexpensesobjects API.

 \code
bindtextdomain (LIBRARY_GETTEXT_PACKAGE, LOCALE_DIR);
 \endcode

*/
gboolean ExpensesRegister (void);

/** @} */
/** @} */
#endif				/* _QOF_EXPENSES_H */
