/* Josh Pieper, (c) 2000 */

/* This file is distributed under the GPL, see file COPYING for details. */

#ifndef PROTOCOL_H
#define PROTOCOL_H

/* 0.4.28.c17 */
#include "sh_sys_types.h"

#include "lib.h"
#include "list.h"


/* First off lets define the critical structures used in the protocol.
  IMPORTANT: The fields are all given as arrays of bytes because that's
  the only way to guarantee proper 'network order' of bytes cross-platform.
  When actually reading or writing one of these structures you have to
  convert for endianness.  */

/* 0.4.28.c08 In order to make sure I didn't miss any places where
   endian conversion has to be done, I renamed all the littleendian fields
   and used the compiler errors to find out where I needed to look to make
   changes. I also renamed some local variables and function parameter names
   for the same reason. I actually found one place where endianness wasn't
   being converted properly (0.4.28.c10) */

/* ------- byte layout for header ------------------------------------------ */

#define PROT_SIZE_HDR (16+1+1+1+4)
typedef uint8 gnutella_hdr[PROT_SIZE_HDR];
#define PROT_HDR_GUID  0    /* array of 16 bytes */
#define PROT_HDR_FUNC 16    /* 8-bit unsigned integer */
#define PROT_HDR_TTL  17    /* 8-bit unsigned integer */
#define PROT_HDR_HOPS 18    /* 8-bit unsigned integer */
#define PROT_HDR_DLEN 19    /* little-endian 32-bit unsigned integer */

/* ------- byte layout for 0x00 PING --------------------------------------- */

/* PING packets have no payload data */

/* ------- byte layout for 0x01 PONG --------------------------------------- */

#define PROT_SIZE_PONG (2+4+4+4)
typedef uint8 gnutella_pong[PROT_SIZE_PONG];
#define PROT_PONG_PORT    0    /* little-endian 16-bit unsigned integer */
#define PROT_PONG_IP      2    /* array of four 8-bit unsigned integers */
#define PROT_PONG_FILES   6    /* little-endian 32-bit unsigned integer */
#define PROT_PONG_KBYTES 10    /* little-endian 32-bit unsigned integer */
/* typedef struct { uint8 port_le[2]; uint8 ip[4]; uint8 files_le[4];
   uint8 kbytes_le[4]; } gnutella_servent; */

/* ------- byte layout for 0x40 PUSH --------------------------------------- */

#define PROT_SIZE_PUSH (16+4+4+2)
typedef uint8 gnutella_push_a[PROT_SIZE_PUSH];
#define PROT_PUSH_GUID  0    /* array of 16 bytes */
#define PROT_PUSH_REF  16    /* little-endian 32-bit unsigned integer */
#define PROT_PUSH_IP   20    /* array of four 8-bit unsigned integers */
#define PROT_PUSH_PORT 24    /* little-endian 16-bit unsigned integer */

/* ------- byte layout for 0x80 QUERY -------------------------------------- */

/* gp_request_make does not use a data structure to make QUERY packets */

/* ------- byte layout for 0x81 QREPLY ------------------------------------- */

#define PROT_SIZE_QRHDR (1+2+4+4)
typedef uint8 gnutella_qrhdr[PROT_SIZE_QRHDR];
#define PROT_QRHDR_NUM   0    /* 8-bit unsigned integer */
#define PROT_QRHDR_PORT  1    /* little-endian 16-bit unsigned integer */
#define PROT_QRHDR_IP    3    /* array of four 8-bit unsigned integers */
#define PROT_QRHDR_SPEED 7    /* little-endian 32-bit unsigned integer */

#define PROT_SIZE_QRNAME (4+4+1)
typedef uint8 gnutella_qrname[PROT_SIZE_QRNAME];
#define PROT_QRNAME_REF  0    /* little-endian 32-bit unsigned integer */
#define PROT_QRNAME_SIZE 4    /* little-endian 32-bit unsigned integer */
#define PROT_QRNAME_NAME 8    /* C string plus an extra ending null byte */

#define PROT_SIZE_QRSUF (4+1+1+1+16)
typedef uint8 gnutella_qrsuf[PROT_SIZE_QRSUF];
#define PROT_QRSUF_VENDOR 0    /* array of 4 characters */
#define PROT_QRSUF_ODLEN  4    /* 8-bit unsigned integer */
#define PROT_QRSUF_FLAG1  5    /* 8-bit unsigned integer */
#define PROT_QRSUF_FLAG2  6    /* 8-bit unsigned integer */
#define PROT_QRSUF_GUID   7    /* array of 16 bytes */

/* ------- end of alignment-critical structures ---------------------------- */

typedef struct {
  gnutella_hdr gh;
  void * data;
} gnutella_packet;

typedef struct {
  uint8 guid[16];
  uint8 ref_le[4];
} guid_ref;

#ifdef __cplusplus
extern "C" {
#endif

  /* Now lets define the functions used to manipulate these structures. */

  void fre_gpa(gnutella_packet **x, int bugnum);
  gnutella_packet * gp_reply_make(char *guid, Gnut_List *results,
				  uint8 *ip, uint16 port, uint32 speed,
				  char * mguid, int ttl);

  gnutella_packet * gp_pong_make(char *guid, int files,
				 int bytes, uint8 *ip, uint16 port,
				 int ttl);

  gnutella_packet * gp_ping_make(char *mac, int ttl);

  gnutella_packet * gp_request_make(char *mac, char *name, int ttl);

  gnutella_packet * gp_push_make(char *mac, int ttl, char *guid, 
				 uint32 ref, uint8 *ip, uint16 port);

  gnutella_packet * gp_dup(gnutella_packet *gp);

  int gp_print(gnutella_packet *);

#ifdef __cplusplus
}
#endif

#endif
