/*
**  Utilities.m
**
**  Copyright (c) 2001, 2002
**
**  Jonathan B. Leffert <jonathan@leffert.net>
**  Ludovic Marcotte <ludovic@Sophos.ca>
**
**  This program is free software; you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation; either version 2 of the License, or
**  (at your option) any later version.
**
**  This program is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with this program; if not, write to the Free Software
**  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#import "Utilities.h"

#import "Address.h"
#import "ExtendedAttachmentCell.h"
#import "GNUMail.h"
#import "GNUMailConstants.h"
#import "MailWindowController.h"
#import "MimeType.h"
#import "MimeTypeManager.h"
#import "TextEnrichedConsumer.h"

#import <Pantomime/Constants.h>
#import <Pantomime/Folder.h>
#import <Pantomime/Message.h>
#import <Pantomime/MimeBodyPart.h>
#import <Pantomime/MimeMultipart.h>
#import <Pantomime/MimeUtility.h>

#import <limits.h>

char ent(char **ref);
char *striphtml(char *s);

@implementation Utilities

//
//
//
+ (void) appendAddress: (Address *) theAddress
	   toTextField: (NSTextField *) theTextField
{
  NSRange aRange;

  aRange = [[theTextField stringValue] rangeOfString:[theAddress formattedAddress]
				       options: NSCaseInsensitiveSearch];
  
  if ( aRange.location != NSNotFound )
    {
      return; 
    }

  if ( [[theTextField stringValue] length] )
    {
      [theTextField setStringValue: 
		      [NSString stringWithFormat: @"%@, %@",
				[theTextField stringValue],
				[theAddress formattedAddress]]];
    }
  else
    {
      [theTextField setStringValue: [theAddress formattedAddress]];
    }
}

//
// This method returns a NSAttributedString object that has been built
// from the content of the message.
//
+ (NSAttributedString *) attributedStringFromContentForPart: (Part *) thePart
					    mimeTypeManager: (MimeTypeManager *) theMimeTypeManager
{
  NSMutableAttributedString *maStr;
  NSMutableDictionary *tAttr;       // text attributes

  tAttr = [[NSMutableDictionary alloc] init];
  if ( [[NSUserDefaults standardUserDefaults] objectForKey: @"MESSAGE_FONT_NAME"] )
    {
      [tAttr setObject: [NSFont fontWithName: [[NSUserDefaults standardUserDefaults] stringForKey: @"MESSAGE_FONT_NAME"]
                                size: [[NSUserDefaults standardUserDefaults] floatForKey: @"MESSAGE_FONT_SIZE"]]
             forKey: NSFontAttributeName];
    }
  else
    {
      [tAttr setObject: [NSFont userFixedPitchFontOfSize: 0] forKey:NSFontAttributeName];
    }

  maStr = [[NSMutableAttributedString alloc] init];
  
  if ( [[thePart content] isKindOfClass: [MimeMultipart class]] )
    {
      // We first verify if our multipart object is a multipart alternative.
      // If yes, we represent the best representation of the part.
      if ( [thePart isMimeType: @"multipart" : @"alternative"] )
	{
	  // We append our \n to separate our body part from the headers
	  [maStr appendAttributedString: [Utilities attributedStringWithString: @"\n" 
						    attributes: nil] ];
	  
	  // We then append the best representation from this multipart/alternative object
	  [maStr appendAttributedString: [Utilities bestRepresentationFromMultipartAlternative:
						      (MimeMultipart *)[thePart content]] ];
	}
      // Then we verify if our multipart object is a multipart appledouble.
      else if ( [thePart isMimeType: @"multipart" : @"appledouble"] )
	{
	  // We append our \n to separate our body part from the headers
	  [maStr appendAttributedString: [Utilities attributedStringWithString: @"\n" 
						    attributes: nil] ];
	  
	  // We then append the best representation from this multipart/appledouble object
	  [maStr appendAttributedString: [Utilities bestRepresentationFromMultipartAppleDouble:
						      (MimeMultipart *)[thePart content]
						    mimeTypeManager: theMimeTypeManager] ];
	}
      // We have a multipart/mixed or multipart/related (or an other, unknown multipart/* object)
      else
	{
	  MimeMultipart *aMimeMultipart;
	  MimeBodyPart *aMimeBodyPart;
	  int i;
      
	  aMimeMultipart = (MimeMultipart*)[thePart content];
      
	  for (i = 0; i < [aMimeMultipart count]; i++)
	    {
	      // We get our part
	      aMimeBodyPart = [aMimeMultipart bodyPartAtIndex: i];
	      
	      // We recursevely call our method to show parts
	      [maStr appendAttributedString: [Utilities attributedStringFromContentForPart: aMimeBodyPart
							mimeTypeManager: theMimeTypeManager]];
	    }
	}
    }
  // We have a message with Content-Type: application/* OR audio/* OR image/* OR video/*
  // We can also have a text/* part that was base64 encoded, but, we skip it. It'll be
  // treated as a NSString object. (See below).
  else if ( [[thePart content] isKindOfClass:[NSData class]] )
    {
      NSTextAttachment *aTextAttachment;
      ExtendedAttachmentCell *cell;
      NSFileWrapper *aFileWrapper;
      NSImage *anImage;
      
      NSRect rectOfTextView;
      NSSize imageSize;
      
      
      aFileWrapper = [[NSFileWrapper alloc] initRegularFileWithContents: (NSData *)[thePart content]];
      
      if (! [thePart filename] )
	{
	  [aFileWrapper setPreferredFilename: @"unknown"];
	}
      else
	{
	  [aFileWrapper setPreferredFilename: [thePart filename]];
	}
      
      // If the image is a tiff, we create it directly from a NSData
      if ( [thePart isMimeType: @"image": @"tiff"] )
	{
	  anImage = [[NSImage alloc] initWithData:(NSData *)[thePart content]];
	}
      // If not, we temporary write it to a file and we init everything from
      // the content of the file. Libwraster will be used here to load the image
      // from the file. We simply remove the file when we're done.
      //
      // NOTE: Currently, GNUstep doesn't allow creating images other than TIFF from a NSData object.
      //       This is a limitation that will be fixed in the future.
      //       We currently write the data object to a file and initialize an image from
      //       the content of this file. Libwraster's functions will be used by the xgps
      //       backend to decode formats other than TIFF.
      //       - Ludo
      else if ( [thePart isMimeType: @"image": @"*"] )
	{
	  NSString *aString;

	  aString = [NSString stringWithFormat:@"%@/%d_%@", NSTemporaryDirectory(), 
			      [[NSProcessInfo processInfo] processIdentifier],
			      [[thePart filename] lowercaseString]];

	  [(NSData *)[thePart content] writeToFile: aString
		     atomically: YES];
	  
	  anImage = [[NSImage alloc] initWithContentsOfFile: aString];
	  
	  [[NSFileManager defaultManager] removeFileAtPath: aString
					  handler: nil];
	}
      //
      // It's not an image, let's query our MIME type manager in order if we can
      // get an icon from that attachment. If not, we just set a default one.
      //
      else
	{
	  NSString *aString;

	  aString = [[aFileWrapper preferredFilename] pathExtension];

	  if ( aString && [aString length] )
	    {
	      MimeType *aMimeType;

	      aMimeType = [theMimeTypeManager mimeTypeForFileExtension: aString];

	      if (aMimeType && [aMimeType icon])
		{
		  anImage = [aMimeType icon];
		}
	      else
		{
		  anImage = [NSImage imageNamed: @"common_Unknown.tiff"];
		}
	    }
	  else
	    {
	      anImage = [NSImage imageNamed: @"common_Unknown.tiff"];
	    }
	}
      
      // If the image has been loaded sucessfully, it become our icon for our filewrapper
      if ( anImage )
	{
	  [aFileWrapper setIcon: anImage];
	  RELEASE(anImage);
	} 	
      
      // We now rescale the attachment if it doesn't fit in the text view. That could happen
      // very often for images.
      rectOfTextView = [[[[GNUMail lastMailWindowOnTop] windowController] textView] frame];
      imageSize = [[aFileWrapper icon] size];
      
      if ( imageSize.width > rectOfTextView.size.width )
	{
	  double delta =  1.0 / ( imageSize.width / rectOfTextView.size.width );
	  
	  [[aFileWrapper icon] setScalesWhenResized: YES];
	  [[aFileWrapper icon] setSize: NSMakeSize( (imageSize.width * delta), imageSize.height * delta)];
	}
      
      // We now create our text attachment with our file wrapper
      aTextAttachment = [[NSTextAttachment alloc] initWithFileWrapper: aFileWrapper];
      
      // We add this attachment to our 'Save Attachment' menu
      [(GNUMail *)[NSApp delegate] addItemToMenuFromTextAttachment: aTextAttachment];
        
      cell = [[ExtendedAttachmentCell alloc] initWithFilename: [aFileWrapper preferredFilename]
					     size: [(NSData *)[thePart content] length] ];
      [cell setPart: thePart];
      
      [aTextAttachment setAttachmentCell: cell];
      
      // Cocoa bug
#ifdef MACOSX
      [cell setAttachment: aTextAttachment];
      [cell setImage: [aFileWrapper icon]];
#endif
      RELEASE(cell);
      RELEASE(aFileWrapper);
      
      // We separate the text attachment from any other line previously shown
      [maStr appendAttributedString: [Utilities attributedStringWithString: @"\n" 
						attributes: nil] ];
      
      [maStr appendAttributedString: [NSAttributedString attributedStringWithAttachment: aTextAttachment]];
      RELEASE(aTextAttachment);
    }
  //
  // We have a message/rfc822 has the Content-Type.
  //
  else if ( [[thePart content] isKindOfClass:[Message class]] )
    {
      Message *aMessage;
      
      aMessage = (Message *)[thePart content];
      
      [maStr appendAttributedString: [Utilities attributedStringWithString: @"\n" 
						attributes: nil] ];
      
      // We must represent this message/rfc822 part as an attachment
      if ( [thePart contentDisposition] &&
	   ([[thePart contentDisposition] caseInsensitiveCompare: @"attachment"] == NSOrderedSame) )
	{
	  NSTextAttachment *aTextAttachment;
	  ExtendedAttachmentCell *cell;
	  NSFileWrapper *aFileWrapper;
	  NSData *aData;

	  aData = [aMessage rawSource];
	  
	  aFileWrapper = [[NSFileWrapper alloc] initRegularFileWithContents: aData];
	  
	  [aFileWrapper setPreferredFilename: @"message/rfc822 attachment"];
	  
	  aTextAttachment = [[NSTextAttachment alloc] initWithFileWrapper: aFileWrapper];
	  
	  // We add this attachment to our 'Save Attachment' menu
	  [(GNUMail *)[NSApp delegate] addItemToMenuFromTextAttachment: aTextAttachment];
          
	  cell = [[ExtendedAttachmentCell alloc] initWithFilename: [aFileWrapper preferredFilename]
						 size: [aData length] ];
	  [cell setPart: thePart];
	  
	  [aTextAttachment setAttachmentCell: cell];
          
          // Cocoa bug
#ifdef MACOSX
          [cell setAttachment: aTextAttachment];
          [cell setImage: [aFileWrapper icon]];
#endif
	  RELEASE(cell); 
	  RELEASE(aFileWrapper);
	  
	  [maStr appendAttributedString: [NSAttributedString attributedStringWithAttachment:
							       aTextAttachment]];
	  RELEASE(aTextAttachment);
	}
      // Its inline..
      else
	{	      
	  [maStr appendAttributedString: [Utilities attributedStringFromHeadersForMessage: aMessage
						    showAllHeaders: NO] ];
	  [maStr appendAttributedString: [Utilities attributedStringFromContentForPart: aMessage
						    mimeTypeManager: theMimeTypeManager] ];
	}
    }
  //
  // We have a message with Content-Type: text/*
  // or at least, a part that we can represent directly in our textView.
  // 
  else if ( [[thePart content] isKindOfClass:[NSString class]] )
    { 
      NSString *aString;
      
      aString = (NSString *)[thePart content];

      // We must represent our NSString as an text attachment in our textView.
      if ( [thePart contentDisposition] &&
	   ([[thePart contentDisposition] caseInsensitiveCompare: @"attachment"] == NSOrderedSame) )
	{
	  NSTextAttachment *aTextAttachment;
	  ExtendedAttachmentCell *cell;
	  NSFileWrapper *aFileWrapper;
	  NSData *aData;
	  
	  aData = [aString dataUsingEncoding: NSUTF8StringEncoding];
	  aFileWrapper = [[NSFileWrapper alloc] initRegularFileWithContents: aData];
	  
	  if (! [thePart filename] )
	    {
	      [aFileWrapper setPreferredFilename: @"unknown"];
	    }
	  else
	    {
	      [aFileWrapper setPreferredFilename: [thePart filename]];
	    }
	  
	  aTextAttachment = [[NSTextAttachment alloc] initWithFileWrapper: aFileWrapper];
	  
	  // We add this attachment to our 'Save Attachment' menu
	  [(GNUMail *)[NSApp delegate] addItemToMenuFromTextAttachment: aTextAttachment];

	  cell = [[ExtendedAttachmentCell alloc] initWithFilename: [aFileWrapper preferredFilename]
						 size: [aData length] ];
	  [cell setPart: thePart];
	  
	  [aTextAttachment setAttachmentCell: cell];
          
          // Cocoa bug
#ifdef MACOSX
          [cell setAttachment: aTextAttachment];
          [cell setImage: [aFileWrapper icon]];
#endif
	  RELEASE(cell);
	  RELEASE(aFileWrapper);
	  
	  // We separate the text attachment from any other line previously shown
	  [maStr appendAttributedString: [Utilities attributedStringWithString: @"\n" 
						    attributes: nil] ];
		      
	  [maStr appendAttributedString: [NSAttributedString attributedStringWithAttachment:
							       aTextAttachment]];
	  RELEASE(aTextAttachment);
	}
      // It's inline...
      else
	{
	  //
	  //
	  if ( [thePart isMimeType: @"text": @"html"] )
	    {
#ifdef MACOSX
	      NSAttributedString *anAttributedString;
	      NSData *aData;
	      
	      aData = [aString dataUsingEncoding: NSASCIIStringEncoding];
	      
	      anAttributedString = [[NSAttributedString alloc] initWithHTML: aData
							       documentAttributes: nil];
	      
	      [maStr appendAttributedString: anAttributedString];
	      RELEASE(anAttributedString);
#else
	      char *cString;

	      NSLog(@"Showing a text/html part. No formatting will be done for now.");

	      cString = striphtml( (char *)[aString cString] );
	      [maStr appendAttributedString: [Utilities attributedStringWithString: [NSString stringWithCString: cString]
							attributes: tAttr] ];
#endif
	    }
	  else if ( [thePart isMimeType: @"text": @"enriched"] )
	    {
	      [maStr appendAttributedString: [TextEnrichedConsumer attributedStringFromTextEnrichedString: aString]];
	    }
	  else if ( [thePart isMimeType: @"text": @"rtf"] )
	    {
	      NSAttributedString *anAttributedString;
	      NSData *aData;
	      
	      aData = [aString dataUsingEncoding: NSASCIIStringEncoding];
	      
	      anAttributedString = [[NSAttributedString alloc] initWithRTF: aData
							       documentAttributes: NULL];
	      [maStr appendAttributedString: anAttributedString];
	      RELEASE(anAttributedString); 
	    }
	  else
	    {
	      [maStr appendAttributedString: [Utilities attributedStringWithString: aString
							attributes: tAttr] ];

	    }
	}
    }
  //
  // We have something that we probably can't display.
  // Let's inform the user about this situation.
  // 
  else
    {
      NSRunAlertPanel(_(@"Error!"),
		      _(@"A part of this E-Mail can't be represented. Please report this as a bug."),
		      _(@"OK"), // default
		      NULL,     // alternate
		      NULL);
    }
  
  RELEASE(tAttr);

  return AUTORELEASE(maStr);
}

//
// This method returns a NSAttributedString object that has been built
// from the headers of the message.
//
+ (NSAttributedString *) attributedStringFromHeadersForMessage: (Message *) theMessage
						showAllHeaders: (BOOL) aBOOL
{
  NSMutableAttributedString *maStr; 
  // header attributes
  NSMutableDictionary *tagAttr, *hdrAttr;

  NSArray *headersToShow;
  NSDictionary *allHeaders;
  int i;

  maStr = [[NSMutableAttributedString alloc] init];
  
  tagAttr = [[NSMutableDictionary alloc] init];
  if ( [[NSUserDefaults standardUserDefaults] objectForKey: @"TAG_FONT_NAME"] )
    {
      [tagAttr setObject: [NSFont fontWithName: [[NSUserDefaults standardUserDefaults] stringForKey: @"TAG_FONT_NAME"]
                                  size: [[NSUserDefaults standardUserDefaults] floatForKey: @"TAG_FONT_SIZE"]]
               forKey: NSFontAttributeName];
    }
  else
    {
      [tagAttr setObject: [NSFont boldSystemFontOfSize: 0]
               forKey: NSFontAttributeName];
    }

  hdrAttr = [[NSMutableDictionary alloc] init];
  if ( [[NSUserDefaults standardUserDefaults] objectForKey: @"HEADER_FONT_NAME"] )
    {
      [hdrAttr setObject: [NSFont fontWithName: [[NSUserDefaults standardUserDefaults] stringForKey: @"HEADER_FONT_NAME"]
                                  size: [[NSUserDefaults standardUserDefaults] floatForKey: @"HEADER_FONT_SIZE"]]
               forKey: NSFontAttributeName];
    }
  else
    {
      [hdrAttr setObject: [NSFont systemFontOfSize: 0]
               forKey: NSFontAttributeName];
    }


  allHeaders = [theMessage allHeaders];
  
  // We verify which headers of the message we show show.
  if ( aBOOL )
    {
      headersToShow = [allHeaders allKeys];
    }
  else
    {
      headersToShow = [[NSUserDefaults standardUserDefaults] objectForKey:@"SHOWNHEADERS"];
    }
  
  if ( headersToShow )
    {  
      for (i = 0; i < [headersToShow count]; i++)
	{
	  NSString *anHeader = [headersToShow objectAtIndex: i];
	  
	  if ( [anHeader caseInsensitiveCompare: @"Date"] == NSOrderedSame )
	    {
	      if ( [theMessage receivedDate] )
		{
		  [maStr appendAttributedString: [Utilities attributedStringWithString: @"Date: "
							    attributes: tagAttr] ];
		  [maStr appendAttributedString: [Utilities attributedStringWithString: [[theMessage receivedDate] description]
							    attributes: hdrAttr] ];
		  [maStr appendAttributedString: [Utilities attributedStringWithString: @"\n"
							    attributes: nil] ];
		}
	    }
	  else if ( [anHeader caseInsensitiveCompare: @"From"] == NSOrderedSame )
	    {
	      [maStr appendAttributedString: [Utilities attributedStringWithString: @"From: "
							attributes: tagAttr] ];
	      [maStr appendAttributedString: [Utilities attributedStringWithString: [[theMessage from] unicodeStringValue]
							attributes: hdrAttr] ];
	      [maStr appendAttributedString: [Utilities attributedStringWithString: @"\n"
							attributes: nil] ];
	    }
	  else if ( [anHeader caseInsensitiveCompare: @"Bcc"] == NSOrderedSame )
	    {
	      NSString *bccStr = [Utilities stringFromRecipients: [theMessage recipients]
					    type: BCC];
	      
	      if ( [bccStr length] > 0 )
		{
		  [maStr appendAttributedString: [Utilities attributedStringWithString: @"Bcc: "
							    attributes: tagAttr] ];
		  [maStr appendAttributedString: [Utilities attributedStringWithString: [bccStr substringToIndex:([bccStr length] - 2)]
							    attributes: hdrAttr] ];
		  [maStr appendAttributedString: [Utilities attributedStringWithString: @"\n"
							    attributes: nil] ];
		}
	    }
	  else if ( [anHeader caseInsensitiveCompare: @"Cc"] == NSOrderedSame )
	    {
	      NSString *ccStr= [Utilities stringFromRecipients: [theMessage recipients]
					  type: CC];
	      
	      if ( [ccStr length] > 0 )
		{
		  [maStr appendAttributedString: [Utilities attributedStringWithString: @"Cc: "
							    attributes: tagAttr] ];
		  [maStr appendAttributedString: [Utilities attributedStringWithString: [ccStr substringToIndex:([ccStr length] - 2)]
							    attributes: hdrAttr] ];
		  [maStr appendAttributedString: [Utilities attributedStringWithString: @"\n"
							    attributes: nil] ];
		}
	    }
	  else if ( [anHeader caseInsensitiveCompare: @"Reply-To"] == NSOrderedSame )
	    { 
	      InternetAddress *anInternetAddress = [theMessage replyTo];
	      
	      if ( anInternetAddress )
		{
		  [maStr appendAttributedString: [Utilities attributedStringWithString: @"Reply-To: "
							    attributes: tagAttr] ];
		  [maStr appendAttributedString: [Utilities attributedStringWithString: [[theMessage replyTo] unicodeStringValue]
							    attributes: hdrAttr] ];
		  [maStr appendAttributedString: [Utilities attributedStringWithString: @"\n"
							    attributes: nil] ];
		}	
	    }
	  else if ( [anHeader caseInsensitiveCompare: @"To"] == NSOrderedSame )
	    {
	      NSString *toStr = [Utilities stringFromRecipients: [theMessage recipients]
					   type: TO];
	      
	      if ( [toStr length] > 0 )
		{
		  [maStr appendAttributedString: [Utilities attributedStringWithString: @"To: "
							    attributes: tagAttr] ];
		  [maStr appendAttributedString: [Utilities attributedStringWithString: [toStr substringToIndex:([toStr length] - 2)]
							    attributes: hdrAttr] ];
		  [maStr appendAttributedString: [Utilities attributedStringWithString: @"\n"
							    attributes: nil] ];
		}
	    }								     
	  else
	    {
	      NSArray *allKeys = [allHeaders allKeys];
	      NSString *valueString = nil;
	      int i;
	      
	      for (i = 0; i < [allKeys count]; i++)
		{
		  if ( [[[allKeys objectAtIndex: i] uppercaseString] isEqualToString: [anHeader uppercaseString]] )
		    {
		      valueString = [allHeaders objectForKey: [allKeys objectAtIndex: i]];
		      break;
		    }
		}
		  
	      if ( valueString )
		{
		  [maStr appendAttributedString: [Utilities attributedStringWithString: [NSString stringWithFormat:@"%@: ", anHeader]
							    attributes: tagAttr] ];
		  [maStr appendAttributedString: [Utilities attributedStringWithString: valueString
							    attributes: hdrAttr] ];
		  [maStr appendAttributedString: [Utilities attributedStringWithString: @"\n"
							    attributes: nil] ];
		}
	    }
	} /* for (..) */
      
    } /* if ( headersToShow ) */
  
  [maStr appendAttributedString: [Utilities attributedStringWithString: @"\n"
					    attributes: nil] ];

  RELEASE(tagAttr);
  RELEASE(hdrAttr);

  return AUTORELEASE(maStr);
}

//
//
//
+ (NSAttributedString *) attributedStringWithString: (NSString *) theString
					 attributes: (NSDictionary *) theAttributes
{
  if (! theAttributes )
    {
      NSMutableDictionary *attributes;
      NSMutableDictionary *aMutableAttributedString;

      attributes = [[NSMutableDictionary alloc] init];
      [attributes setObject: [NSFont systemFontOfSize: 0]
		  forKey: NSFontAttributeName];
      
      aMutableAttributedString = [[NSAttributedString alloc] initWithString: theString
							     attributes: attributes];
      RELEASE(attributes);
      
      return AUTORELEASE( aMutableAttributedString );
    }
  else 
    {
      return AUTORELEASE( [[NSAttributedString alloc] initWithString: theString
						      attributes: theAttributes] );
    }
}


//
// FIXME: Currently, this method only search for the text/plain representation of a
// multipart/alternative object.
//
// In the future, we should try to represent the representation the user wants or, the
// more complex ones first (like text/html or text/enriched).
//
+ (NSAttributedString *) bestRepresentationFromMultipartAlternative: (MimeMultipart *) theMimeMultipart
{
  MimeBodyPart *aMimeBodyPart;
  int i;
  
  for (i = 0; i < [theMimeMultipart count]; i++)
    {
      aMimeBodyPart = [theMimeMultipart bodyPartAtIndex: i];
      
      if ( [aMimeBodyPart isMimeType: @"text": @"plain"] )
	{
	  NSMutableDictionary *tAttr;       // text attributes
	  NSString *aString;
	  
	  tAttr = [[NSMutableDictionary alloc] init];
	  AUTORELEASE(tAttr);
	  
	  if ( [[NSUserDefaults standardUserDefaults] objectForKey: @"MESSAGE_FONT_NAME"] )
	    {
	      [tAttr setObject: [NSFont fontWithName: [[NSUserDefaults standardUserDefaults] 
							stringForKey: @"MESSAGE_FONT_NAME"]
					size: [[NSUserDefaults standardUserDefaults] 
						floatForKey: @"MESSAGE_FONT_SIZE"]]
		     forKey: NSFontAttributeName];
	    }
	  else
	    {
	      [tAttr setObject: [NSFont userFixedPitchFontOfSize: 0] forKey:NSFontAttributeName];
	    }

          aString = (NSString *)[aMimeBodyPart content];
	  
	  return [Utilities attributedStringWithString: aString
			    attributes: tAttr];
	}
    }
  
  // We haven't found a text/plain part. Report this as a bug in GNUMail.app for not supporting
  // the other kind of parts.
  return [Utilities attributedStringWithString: @"No text/plain part found. Please report this bug since GNUMail.app doesn't support to other kind of parts."
		    attributes: nil];
}


//
//
//
+ (NSAttributedString *) bestRepresentationFromMultipartAppleDouble: (MimeMultipart *) theMimeMultipart
						    mimeTypeManager: (MimeTypeManager *) theMimeTypeManager
{
  NSMutableAttributedString *aMutableAttributedString;
  NSMutableDictionary *attributes;
  
  MimeBodyPart *aMimeBodyPart;
  int i;
  
  // We create a set of attributes (base font, color red)
  attributes = [[NSMutableDictionary alloc] init];
  [attributes setObject: [NSColor redColor]
	      forKey: NSForegroundColorAttributeName];

  aMutableAttributedString = [[NSMutableAttributedString alloc] init];

  for (i = 0; i < [theMimeMultipart count]; i++)
    {
      aMimeBodyPart = [theMimeMultipart bodyPartAtIndex: i];
    
      if ( [aMimeBodyPart isMimeType: @"application": @"applefile"] )
	{
	  [aMutableAttributedString appendAttributedString:
				      [Utilities attributedStringWithString: @"(Decoded Apple file follow...)"
						 attributes: attributes]];
	}
      else
	{
	  // We first add a \n between our applefile description and the 'representaiton' of
	  // the attachment
	  [aMutableAttributedString appendAttributedString:
			      [Utilities attributedStringWithString: @"\n" attributes: nil]];
	  
	  // We add the representation of our attachment
	  [aMutableAttributedString appendAttributedString:
				      [Utilities attributedStringFromContentForPart: aMimeBodyPart
						 mimeTypeManager: theMimeTypeManager] ];
	} 
      
    }
  
  // We add a \n to separate everything.
  [aMutableAttributedString appendAttributedString:
			      [Utilities attributedStringWithString: @"\n" attributes: nil]];

  RELEASE(attributes);

  return AUTORELEASE(aMutableAttributedString);
}


//
//
//
+ (NSAttributedString *) formattedAttributedStringFromAttributedString: (NSAttributedString *) theAttributedString
{
  NSMutableAttributedString *aMutableAttributedString = nil;
  NSRange aRange;
  
  id attribute;
  int index;
  
  // If we only have one char (or less), we don't do anything.
  // This could append for example when we only have a image/tiff as the Content-Type of an E-Mail.
  if ( [theAttributedString length] <= 1 )
    {
      return theAttributedString;
    }
  
  index = 0;
  attribute = [theAttributedString attribute: NSAttachmentAttributeName
				   atIndex: index
				   effectiveRange: &aRange];
  
  // We if have NO text attachments at all... we just return our param
  if ( !attribute && 
       (aRange.length == [theAttributedString length]) )
    {
      return theAttributedString;
    }
  
  
  // We initialize our mutable string
  aMutableAttributedString = [[NSMutableAttributedString alloc] initWithAttributedString: theAttributedString];
 
  while ( index < [aMutableAttributedString length] )
    {
      // We found a text attachment...
      if ( attribute )
	{
	  // We search for <<filename>> OR <filename> and if we got one of them,
	  // we simply replace that string with the actual attachment.
	  ExtendedAttachmentCell *cell;
	  NSString *aString;
	  NSRange r;
	  
	  cell = [attribute attachmentCell];
	  aString = [aMutableAttributedString string];
 
	  // We first search for <<filename>>
	  r = [aString rangeOfString: [NSString stringWithFormat: @"<<%@>>", [[cell part] filename]]];
	  
	  // If not found, we search for <filename>
	  if ( r.length == 0 )
	    {
	      r = [aString rangeOfString: [NSString stringWithFormat: @"<%@>", [[cell part] filename]]];
	    }
	  
	  // We found something, let's replace <filename> (or <<filename>>) by the actual
	  // attachment cell.
	  if ( r.length )
	    {
	      NSAttributedString *attachment;
	      
	      // We get the attributed that contains our attachment
	      attachment = [aMutableAttributedString attributedSubstringFromRange: aRange];
	      RETAIN(attachment);
	      
	      // FIXME - we should verify if the range of that attachment > of the range of
	      // the <filename in case the disposition of our E-Mail is 'wrong'.

	      // We remove it from our string
	      [aMutableAttributedString deleteCharactersInRange: aRange];
	      
	      // We replace
	      [aMutableAttributedString replaceCharactersInRange: r
					withAttributedString: attachment];
	      
	      RELEASE(attachment);
	      
	      // We adjust our index for the search
	      index = r.location + 1;
	    }
	  else
	    {
	      index = NSMaxRange(r);
	    }
	}
      else // of if ( attribute ) 
	{
	  index = NSMaxRange(aRange);
	}
      
      if (index >= [aMutableAttributedString length])
      	{
      	  break;
      	}

      attribute = [aMutableAttributedString attribute: NSAttachmentAttributeName
					    atIndex: index
					    effectiveRange: &aRange];
    }

  return aMutableAttributedString;
}


//
//
//
+ (NSString *) stringFromRecipients: (NSArray *) allRecipients
			       type: (int) recipientType
{
  InternetAddress *anInternetAddress;
  NSMutableString *aMutableString;
  int i;
  
  aMutableString = [[NSMutableString alloc] init];

  for (i = 0; i < [allRecipients count]; i++)
    {
      anInternetAddress = [allRecipients objectAtIndex: i];
      
      if ([anInternetAddress type] == recipientType)
	{
	  [aMutableString appendFormat: @"%@, ", [anInternetAddress unicodeStringValue]];
	}
      
    }
  
  return AUTORELEASE(aMutableString); 
}


//
//
//
+ (int) xCoordOfViewAfter: (NSView *) theView
		withSpace: (int) pixels
{
  if ( theView )
    {
      return [theView frame].origin.x +
	[theView frame].size.width +
	pixels;
    }
  else
    {
      return 0;
    }
}


//
//
//
+ (int) yCoordOfViewBelow: (NSView *) theView
		withSpace: (int) pixels
	       viewHeight: (int) height
{
  if ( theView )
    {
      return [theView frame].origin.y -
	pixels -
	height;
    }
  else
    {
      return 0;
    }
}


//
//
//
+ (NSRect) rectForEditWindow
{
  NSDictionary *aDictionary;
  NSRect aRect;

  aDictionary = [[NSUserDefaults standardUserDefaults] objectForKey: @"RECT_EDITWINDOW"];
  
  if ( aDictionary )
    {
      aRect.origin.x = [[aDictionary objectForKey: @"x"] intValue];
      aRect.origin.y = [[aDictionary objectForKey: @"y"] intValue];
      aRect.size.width = [[aDictionary objectForKey: @"width"] intValue];
      aRect.size.height = [[aDictionary objectForKey: @"height"] intValue]; 
    }
  else
    {
      aRect = NSMakeRect(50,75,650,495);
    }

  return aRect;
}


//
//
//
+ (NSRect) rectForMailWindow
{
  NSDictionary *aDictionary;
  NSRect aRect;

  aDictionary = [[NSUserDefaults standardUserDefaults] objectForKey: @"RECT_MAILWINDOW"];
  
  if ( aDictionary )
    {
      aRect.origin.x = [[aDictionary objectForKey: @"x"] intValue];
      aRect.origin.y = [[aDictionary objectForKey: @"y"] intValue];
      aRect.size.width = [[aDictionary objectForKey: @"width"] intValue];
      aRect.size.height = [[aDictionary objectForKey: @"height"] intValue]; 
    }
  else
    {
      aRect = NSMakeRect(150,100,750,595);
    }

  return aRect;
}


//
//
//
+ (void) saveRect: (NSRect) theRect
    forWindowName: (NSString *) theName
{
  NSMutableDictionary *aMutableDictionary;

  aMutableDictionary = [[NSMutableDictionary alloc] initWithCapacity: 4];
  
  [aMutableDictionary setObject: [NSNumber numberWithInt: theRect.origin.x] forKey: @"x"];
  [aMutableDictionary setObject: [NSNumber numberWithInt: theRect.origin.y] forKey: @"y"];
  [aMutableDictionary setObject: [NSNumber numberWithInt: theRect.size.width] forKey: @"width"];
  [aMutableDictionary setObject: [NSNumber numberWithInt: theRect.size.height] forKey: @"height"];

  [[NSUserDefaults standardUserDefaults] setObject:aMutableDictionary forKey: theName];
  [[NSUserDefaults standardUserDefaults] synchronize];
}


//
//
//
+ (NSString *) encryptPassword: (NSString *) thePassword
                       withKey: (NSString *) theKey
{
  NSMutableString *key;
  NSMutableData *encryptedPassword;
  NSString *result;
  int i;
  unichar p, k, e;

  // The length of the key must be greater (or equal) than
  // the length of the password
  key = [[NSMutableString alloc] init];
  
  while ( [key length] < [thePassword length] )
    {
      [key appendString: theKey];
    }

  encryptedPassword = [[NSMutableData alloc] init];
 
  for ( i = 0; i < [thePassword length]; i++ )
    {
      p = [thePassword characterAtIndex: i];
      k = [key characterAtIndex: i];

      e = p ^ k;
      
      [encryptedPassword appendBytes: (void *)&e length: 2];
    }

  result = AUTORELEASE([[NSString alloc] initWithData: [MimeUtility encodeBase64: encryptedPassword lineLength: 0]
                             encoding: NSASCIIStringEncoding]);

  RELEASE(encryptedPassword);
  RELEASE(key);

  return result;
}


//
//
//
+ (NSString *) decryptPassword: (NSString *) thePassword
                       withKey: (NSString *) theKey
{
  NSMutableString *key;
  NSMutableString *password;
  NSData *dec;
  unsigned char *decryptedPassword;
  NSString *result;
  int i;
  unichar p, k, d;

  if ( nil == thePassword || nil == theKey )
    {
      return nil;
    }

  // We 'verify' if the password is not encoded in base64
  // We should not rely on this method but it's currently the best guess we could make
  if ( [thePassword length] == 0 || 
       ([thePassword length] & 0x03) || 
       [theKey length] == 0)
    {
      return thePassword;
    }

  // The length of the key must be greater (or equal) than
  // the length of the password
  key = [[NSMutableString alloc] init];
  
  while ( [key length] < [thePassword length] )
    {
      [key appendString: theKey];
    }

  password = [[NSMutableString alloc] init];

  dec = [MimeUtility decodeBase64: [thePassword dataUsingEncoding: NSASCIIStringEncoding]];
  decryptedPassword = (unsigned char *)[dec bytes];
  
  for ( i = 0; i < [dec length]; i += 2 )
    {
      d = decryptedPassword[i] | decryptedPassword[i+1];
      k = [key characterAtIndex: i/2];

      p = d ^ k;
      [password appendString: [NSString stringWithCharacters: &p length: 1]];
    }

  result = [[NSString alloc] initWithString: password];

  RELEASE(password);
  RELEASE(key);

  return AUTORELEASE(result);
}


//
//
//
+ (void) loadPersonalProfilesInPopUpButton: (NSPopUpButton *) thePopUpButton
{
  NSDictionary *allPersonalProfiles;
  NSArray *allKeys;
  int i, index;
  
  NSString *defaultProfile;

  defaultProfile = [[NSUserDefaults standardUserDefaults] objectForKey: @"DEFAULT_PERSONAL"];
  index = 0;
  
  allPersonalProfiles = [[NSUserDefaults standardUserDefaults] objectForKey: @"PERSONAL"];
  
  allKeys = [[allPersonalProfiles allKeys] sortedArrayUsingSelector: @selector(caseInsensitiveCompare:)];
  
  // We initialize our popup button
  [thePopUpButton removeAllItems];
  
  for (i = 0; i < [allKeys count]; i++)
    {
      NSDictionary *aDictionary;
      
      aDictionary = [allPersonalProfiles objectForKey: [allKeys objectAtIndex: i]];
      
      if ( defaultProfile && 
	   [[allKeys objectAtIndex: i] isEqualToString: defaultProfile] ) 
	{
	  index = i;
	}

      [thePopUpButton insertItemWithTitle: [NSString stringWithFormat: @"%@ (%@)", 
						     [aDictionary objectForKey: @"EMAILADDR"],
						     [allKeys objectAtIndex: i]]
		      atIndex: i];
    }
  
  [thePopUpButton selectItemAtIndex: index];

  [thePopUpButton synchronizeTitleAndSelectedItem];
}


//
//
//
+ (void) loadTransportMethodsInPopUpButton: (NSPopUpButton *) thePopUpButton
{
  NSArray *anArray;
  int i;

  // We initialize our popup button used to select the transport methods
  [thePopUpButton removeAllItems];

  anArray = [[NSUserDefaults standardUserDefaults] objectForKey: @"SENDING"];
  
  for (i = 0; i < [anArray count]; i++)
    {
      NSDictionary *aDictionary;
      NSString *aString;
      
      aDictionary = [anArray objectAtIndex: i];
      
      if ( [[aDictionary objectForKey: @"TRANSPORT_METHOD"] intValue] == TRANSPORT_SMTP)
	{
	  aString = [NSString stringWithFormat: @"SMTP (%@)",
			      [aDictionary objectForKey: @"SMTP_HOST"]];
	}
      else
	{
	  aString = [NSString stringWithFormat: @"Mailer (%@)",
			      [aDictionary objectForKey: @"MAILER_PATH"]];
	}
      
      
      [thePopUpButton insertItemWithTitle: aString
		      atIndex: i];
    }
}


//
//
//
+ (NSString *) profileNameForItemTitle: (NSString *) theString
{
  NSString *aProfileName;
  NSRange aRange;
  
  // We first extract the profile name from the popup title.
  aRange = [theString rangeOfString: @"("
		      options: NSBackwardsSearch];
  
  aProfileName = [theString substringWithRange: NSMakeRange(aRange.location + 1,
							    [theString length] - aRange.location - 2)];
  
  return aProfileName;
}

//
//
//
+ (id) windowForFolderName: (NSString *) theName
{
  NSArray *allWindows;
	      
  // We get all opened windows
  allWindows = [GNUMail allMailWindows];
  
  if ( allWindows )
    {
      Folder *aFolder;
      id aWindow;
      int i;
      
      for (i = 0; i < [allWindows count]; i++)
	{
	  aWindow = [allWindows objectAtIndex: i];
	  aFolder = [(MailWindowController *)[aWindow windowController] folder];
	  
	  // If we found our opened folder
	  if ( [[aFolder name] isEqualToString: theName] )
	    {
	      return aWindow;
	    }
	}
    }
  
  return nil;
}

@end


//
//
//
NSString *GNUMailUserLibraryPath()
{
  NSString *aString;

  aString = [NSString stringWithFormat: @"%@/GNUMail", 
		      [NSSearchPathForDirectoriesInDomains(NSLibraryDirectory, NSUserDomainMask, YES) objectAtIndex:0] ];

  return aString;
}


//
// This C function has been written by Abhijit Menon-Sen <ams@wiw.org>
// This code is in the public domain.
//
char *striphtml(char *s)
{
    int sgml = 0, tag = 0;
    char c, last = '\0', quote = '\0', *t, *text;

    if ((t = text = malloc(strlen(s)+1)) == NULL)
        return NULL;

    while ((c = *s++)) {
        if (c == quote) {
            if (c == '-' && last != '-')
                goto next;
            else
                last = '\0';
            quote = '\0';
        }
        else if (!quote) {
            switch (c) {
            case '<':
                tag = 1;
                if (*s++ == '!')
                    sgml = 1;
                break;
            case '>':
                if (tag)
                    sgml = tag = 0;
                break;
            case '-':
                if (sgml && last == '-')
                    quote = '-';
                break;
            /* case '"':
            case '\'':
                if (tag)
                    quote = c;
                break; */
            case '&':
                *t++ = ent(&s);
                break;
            default:
                if (!tag)
                    *t++ = c;
                break;
            }
        }

    next:
        last = c;
    }

    return text;
}

//
// This C function has been written by Abhijit Menon-Sen <ams@wiw.org>
// This code is in the public domain.
//
char ent(char **ref)
{
    int i;
    char c = ' ', *s = *ref, *t = s;

    struct {
        char *name;
        char chr;
    } refs[] = {
        { "lt"    , '<'       },
        { "gt"    , '>'       },
        { "amp"   , '&'       },
        { "quot"  , '"'       },
        { "nbsp"  , (char)160 },
        { "iexcl" , (char)161 },
        { "cent"  , (char)162 },
        { "pound" , (char)163 },
        { "curren", (char)164 },
        { "yen"   , (char)165 },
        { "brvbar", (char)166 },
        { "sect"  , (char)167 },
        { "uml"   , (char)168 },
        { "copy"  , (char)169 },
        { "ordf"  , (char)170 },
        { "laquo" , (char)171 },
        { "not"   , (char)172 },
        { "shy"   , (char)173 },
        { "reg"   , (char)174 },
        { "macr"  , (char)175 },
        { "deg"   , (char)176 },
        { "plusmn", (char)177 },
        { "sup2"  , (char)178 },
        { "sup3"  , (char)179 },
        { "acute" , (char)180 },
        { "micro" , (char)181 },
        { "para"  , (char)182 },
        { "middot", (char)183 },
        { "cedil" , (char)184 },
        { "sup1"  , (char)185 },
        { "ordm"  , (char)186 },
        { "raquo" , (char)187 },
        { "frac14", (char)188 },
        { "frac12", (char)189 },
        { "frac34", (char)190 },
        { "iquest", (char)191 },
        { "Agrave", (char)192 },
        { "Aacute", (char)193 },
        { "Acirc" , (char)194 },
        { "Atilde", (char)195 },
        { "Auml"  , (char)196 },
        { "Aring" , (char)197 },
        { "AElig" , (char)198 },
        { "Ccedil", (char)199 },
        { "Egrave", (char)200 },
        { "Eacute", (char)201 },
        { "Ecirc" , (char)202 },
        { "Euml"  , (char)203 },
        { "Igrave", (char)204 },
        { "Iacute", (char)205 },
        { "Icirc" , (char)206 },
        { "Iuml"  , (char)207 },
        { "ETH"   , (char)208 },
        { "Ntilde", (char)209 },
        { "Ograve", (char)210 },
        { "Oacute", (char)211 },
        { "Ocirc" , (char)212 },
        { "Otilde", (char)213 },
        { "Ouml"  , (char)214 },
        { "times" , (char)215 },
        { "Oslash", (char)216 },
        { "Ugrave", (char)217 },
        { "Uacute", (char)218 },
        { "Ucirc" , (char)219 },
        { "Uuml"  , (char)220 },
        { "Yacute", (char)221 },
        { "THORN" , (char)222 },
        { "szlig" , (char)223 },
        { "agrave", (char)224 },
        { "aacute", (char)225 },
        { "acirc" , (char)226 },
        { "atilde", (char)227 },
        { "auml"  , (char)228 },
        { "aring" , (char)229 },
        { "aelig" , (char)230 },
        { "ccedil", (char)231 },
        { "egrave", (char)232 },
        { "eacute", (char)233 },
        { "ecirc" , (char)234 },
        { "euml"  , (char)235 },
        { "igrave", (char)236 },
        { "iacute", (char)237 },
        { "icirc" , (char)238 },
        { "iuml"  , (char)239 },
        { "eth"   , (char)240 },
        { "ntilde", (char)241 },
        { "ograve", (char)242 },
        { "oacute", (char)243 },
        { "ocirc" , (char)244 },
        { "otilde", (char)245 },
        { "ouml"  , (char)246 },
        { "divide", (char)247 },
        { "oslash", (char)248 },
        { "ugrave", (char)249 },
        { "uacute", (char)250 },
        { "ucirc" , (char)251 },
        { "uuml"  , (char)252 },
        { "yacute", (char)253 },
        { "thorn" , (char)254 },
        { "yuml"  , (char)255 }
    };

    while (isalpha(*s) || isdigit(*s) || *s == '#')
        s++;

    for (i = 0; i < sizeof(refs)/sizeof(refs[0]); i++) {
        if (strncmp(refs[i].name, t, s-t) == 0) {
            c = refs[i].chr;
            break;
        }
    }

    if (*s == ';')
        s++;

    *ref = s;
    return c;
}
