/*
**  MailWindowController.m
**
**  Copyright (c) 2001, 2002
**
**  Author: Ludovic Marcotte <ludovic@Sophos.ca>
**
**  This program is free software; you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation; either version 2 of the License, or
**  (at your option) any later version.
**
**  This program is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with this program; if not, write to the Free Software
**  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
#import "MailWindowController.h"

#import "EditWindowController.h"
#import "ExtendedCell.h"
#import "GNUMail.h"
#import "GNUMail/GNUMailBundle.h"
#import "GNUMailConstants.h"
#import "LabelWidget.h"

#ifndef MACOSX
#import "MailWindow.h"
#endif

#import "FilterManager.h"
#import "MimeType.h"
#import "MimeTypeManager.h"
#import "NSRegEx.h"
#import "NSRegExRange.h"
#import "NSStringExtensions.h"
#import "Utilities.h"

#import <Pantomime/Constants.h>
#import <Pantomime/Folder.h>
#import <Pantomime/IMAPFolder.h>
#import <Pantomime/IMAPStore.h>
#import <Pantomime/InternetAddress.h>
#import <Pantomime/LocalFolder.h>
#import <Pantomime/LocalStore.h>
#import <Pantomime/Message.h>
#import <Pantomime/MimeBodyPart.h>
#import <Pantomime/MimeMultipart.h>

@implementation MailWindowController

//
//
//
- (id) initWithWindowNibName: (NSString *) windowNibName
{
#ifdef MACOSX
  
  self = [super initWithWindowNibName: windowNibName];
  
#else
  MailWindow *mailWindow;
  
  mailWindow = [[MailWindow alloc] initWithContentRect: NSMakeRect(150,100,750,595)
				   styleMask: NSClosableWindowMask|NSTitledWindowMask|
				   NSMiniaturizableWindowMask|NSResizableWindowMask
				   backing: NSBackingStoreRetained
				   defer: NO];

  self = [super initWithWindow: mailWindow];
  
  [mailWindow layoutWindow];
  [mailWindow setDelegate: self];
  [mailWindow setFrame:[Utilities rectForMailWindow] display: NO];

  // We link our outlets
  tableScrollView = [mailWindow tableScrollView];
  textScrollView = [mailWindow textScrollView];

  splitView = [mailWindow splitView];
  tableView = [mailWindow tableView];
  textView = [mailWindow textView];
  
  next = [mailWindow next];
  previous = [mailWindow previous];
  delete = [mailWindow delete];
  mailboxes = [mailWindow mailboxes];
  compose = [mailWindow compose];
  forward = [mailWindow forward];
  reply = [mailWindow reply];
  addresses = [mailWindow reply];
  find = [mailWindow find];

  label = (NSTextField *)[mailWindow label];
  
  statusColumn = [mailWindow statusColumn];
  idColumn = [mailWindow idColumn];
  dateColumn = [mailWindow dateColumn];
  fromColumn = [mailWindow fromColumn];
  subjectColumn = [mailWindow subjectColumn];
  sizeColumn = [mailWindow sizeColumn];

  RELEASE(mailWindow);
#endif

  // We set our window title
  [[self window] setTitle: @""];

  // We must retain our columns under OS X after the outlets are linked (we release them in -dealloc)
#ifdef MACOSX
  RETAIN(statusColumn);
  RETAIN(idColumn);
  RETAIN(dateColumn);
  RETAIN(fromColumn);
  RETAIN(subjectColumn);
  RETAIN(sizeColumn);
#endif

  // We set our custom cell
  [statusColumn setDataCell: AUTORELEASE([[ExtendedCell alloc] init])];

  // We load our accessory views
  [self _loadAccessoryViews];

  // We restore our sorting order
  [self _restoreSortingOrder];

  // We restore our split view knob position
  [self _restoreSplitViewSize];

  // We load the right set of columns
  [self reloadTableColumns];
  
  return self;
}


//
//
//
- (void) dealloc
{
  NSLog(@"MailWindowController: -dealloc");

  RELEASE(folder);
  
  TEST_RELEASE(filterManager);
  TEST_RELEASE(mimeTypeManager);
  TEST_RELEASE(allMessages);
  
  // We release our retained columns under OS X
#ifdef MACOSX
  RETAIN(statusColumn);
  RETAIN(idColumn);
  RETAIN(dateColumn);
  RETAIN(fromColumn);
  RETAIN(subjectColumn);
  RETAIN(sizeColumn);
#endif

  [[NSNotificationCenter defaultCenter]
    removeObserver: self
    name: MimeTypesHaveChanged
    object: nil];

  [[NSNotificationCenter defaultCenter]
    removeObserver: self
    name: FiltersHaveChanged
    object: nil];

  [[NSNotificationCenter defaultCenter]
    removeObserver: self
    name: ReplyToMessageWasSuccessful
    object: nil];

  [super dealloc];
}


//
// action methods
//

- (IBAction) doubleClickedOnTableView: (id) sender
{
  // If we are in the Draft folder, we re-opened the selected mail for editing
  if ( [[[self folder] name] isEqualToString: [[NSUserDefaults standardUserDefaults] 
						objectForKey: @"DRAFTSFOLDERNAME"] ] )
    {
      [[NSApp delegate] restoreDraft: nil];
    }
  // Or.. we just 'reply' to the mail...
  else
    {
      [self replyMessage: nil];
    }
}

- (IBAction) deleteMessage: (id) sender
{
  // If we have no element (or no selection), we return!
  if ([folder count] == 0 || [tableView numberOfSelectedRows] == 0)
    {
      NSBeep();
      return;
    } 
  else
    {
      NSEnumerator *anEnumerator;
      NSNumber *aRow;
      Message *theMessage;
      Flags *theFlags;

      int last_row, nb;
      BOOL firstFlagOfList;
      
      anEnumerator = [tableView selectedRowEnumerator];
      last_row = nb = 0;
      firstFlagOfList = NO;

      while ( (aRow = [anEnumerator nextObject]) )
	{
	  theMessage = [allMessages objectAtIndex: [aRow intValue]];
	  
	  // We set the flag Deleted (or not) to the message
	  theFlags = [theMessage flags];

          if ( nb == 0)
            {
              // This is the first message of the list we want to {un}delete
              // We must save the flag.
              if ( [theFlags contain: DELETED] )
                {
                  [theFlags remove: DELETED];
                  firstFlagOfList = NO;
                }
              else
                {
                  [theFlags add: DELETED];
                  firstFlagOfList = YES;
                }
            }
          else
            {
              if ( (! firstFlagOfList) && [theFlags contain: DELETED] )
                { 
                  [theFlags remove: DELETED];
                }
              else if ( firstFlagOfList && (! [theFlags contain: DELETED]) )
                {
                  [theFlags add: DELETED];
                }
            }
	  
	  last_row = [aRow intValue];
          nb++;
	}
      
      // We now select the row right after the message(s) beeing deleted
      // If there is only one selected message
      // We do this operation IIF sender is the button 'delete'
      if (sender == delete)
	{
	  int count, old_selection;
	  
	  count = [[[self folder] allMessages] count];
	  old_selection = NSNotFound;

	  // If we show the deleted mails, we 'jump' to the next mail
	  // Else, we just try to show the same index again.
	  if ( [[self folder] showDeleted] )
	    {
	      last_row = last_row + 1;
	    }
	  else
	    {
	      // If we were 'deleting' the last row, let's shift
	      // our index by one.
	      if ( last_row >= count )
		{
		  last_row = (count - 1);
		}

	      old_selection = [tableView selectedRow];
	    }
	  
	  // We verify that we aren't off bounds
	  if ( count > last_row 
	       && (nb == 1) )
	    {
	      if (old_selection == last_row)
		{
		  [self tableViewSelectionDidChange: nil];
		}
	      else
		{
		  [tableView selectRow: last_row 
			     byExtendingSelection: NO];
		  [tableView scrollRowToVisible: last_row];
		}
	    }
	  else
	    {
	      // We at least post our notification
	      [[NSNotificationCenter defaultCenter]
		postNotificationName: SelectionOfMessageHasChanged
		object: nil
		userInfo: nil];
	    }
	}
      
      // We always refresh our tableView after a delete operation
      if ( [[self folder] showDeleted] )
	{
	  NSLog(@"Showing deleted messages...");
	  [tableView setNeedsDisplay: YES];  
	}
      else
	{
	  NSLog(@"NOT Showing deleted messages...");
	  [self tableViewShouldReloadData];
	}
    }
}


//
// This method selects the message after the current
// selected message and displays it.
//
- (IBAction) nextMessage: (id) sender
{
  int i = [tableView selectedRow];

  if (i < [folder count] - 1)
    {
      [tableView selectRow: (i+1) byExtendingSelection: NO];
      [tableView scrollRowToVisible: (i+1)];
      [tableView setNeedsDisplay: YES];
    }
  else
    {
      NSBeep();
    }
}


//
//
//
- (IBAction) firstMessage: (id) sender
{
  if ([folder count] > 0)
    {
      [tableView selectRow: 0 byExtendingSelection: NO];
      [tableView scrollRowToVisible: 0];
      [tableView setNeedsDisplay: YES];
    }
}


//
//
//
- (IBAction) lastMessage: (id) sender
{
  if ([folder count] > 0)
    {
      [tableView selectRow: ([folder count] - 1) byExtendingSelection: NO];
      [tableView scrollRowToVisible: ([folder count] - 1)];
      [tableView setNeedsDisplay: YES];
    }
}


//
// This method selects the message before the current
// selected message and displays it.
//
- (IBAction) previousMessage: (id) sender
{
  int i = [tableView selectedRow];
  
  if (i > 0)
    {
      [tableView selectRow: (i-1) byExtendingSelection: NO];
      [tableView scrollRowToVisible: (i-1)];
      [tableView setNeedsDisplay: YES];
    }
  else
    {
      NSBeep();
    }
}


//
//
//
- (IBAction) composeMessage: (id) sender
{
  EditWindowController *controller;
  Message *aMessage;

  // We create an empty message
  aMessage = [[Message alloc] init];
  
  controller = [[EditWindowController alloc] initWithWindowNibName: @"EditWindow"];

  if ( controller )
    {
      [[controller window] setTitle: _(@"New message...")];
      [controller setMessage: aMessage];
      [controller setShowCc: NO];
      
      RELEASE(aMessage);
      
      [controller showWindow: self];
    }
}


//
//
//
- (IBAction) forwardMessage: (id) sender
{
  EditWindowController *controller;
  Message *aMessage;
  
  if ([tableView selectedRow] < 0) 
    {
      NSBeep();
      return;
    }
  
  // We obtain the selected entry in the table for the forward information
  aMessage = [allMessages objectAtIndex: [tableView selectedRow]];

  // We create a forwarded copy of our message and we retain it
  aMessage = [aMessage forward];
  RETAIN(aMessage);
  
  controller = [[EditWindowController alloc] initWithWindowNibName: @"EditWindow"];

  if ( controller )
    {
      [[controller window] setTitle: _(@"Forward a message...")];
      [controller setSignaturePosition: [[NSUserDefaults standardUserDefaults] 
					  integerForKey: @"SIGNATURE_FORWARD_POSITION"] ];
      [controller setMessage: aMessage];
      [controller setShowCc: NO];
      
      RELEASE(aMessage);
      
      [controller showWindow: self];
    }
}


//
//
//
- (IBAction) replyMessage: (id) sender
{
  EditWindowController *editWindowController;
  Message *aMessage;

  BOOL shouldReplyToAll;
  
  if ([tableView selectedRow] < 0) 
    {
      NSBeep();
      return;
    }

  // We obtain the selected entry in the table for the reply informations
  aMessage = [allMessages objectAtIndex: [tableView selectedRow]];
  
  if ([aMessage recipientsCount] > 1)
    {
      int choice;
      
      choice = NSRunAlertPanel(_(@"Reply..."),
			       _(@"Would you like to reply to all recipients?"),
			       _(@"No"),  // default
			       _(@"Yes"), // alternate
			       nil);
  
      if ( choice == NSAlertAlternateReturn )
	{
	  shouldReplyToAll = YES;
	}
      else
	{
	  shouldReplyToAll = NO;
	}
    }
  else
    {
      shouldReplyToAll = NO;
    }
  
  // We create our window controller
  editWindowController = [[EditWindowController alloc] initWithWindowNibName: @"EditWindow"];

  if ( editWindowController )
    {
      [[editWindowController window] setTitle: _(@"Reply to a message...")];
      [editWindowController setSignaturePosition: [[NSUserDefaults standardUserDefaults] integerForKey: @"SIGNATURE_REPLY_POSITION"] ];
      [editWindowController setShowCc: shouldReplyToAll];
      
      // We set the original message
      [editWindowController setUnmodifiedMessage: aMessage];
      
      // We create a replied copy of our message and we retain it
      aMessage = [aMessage replyWithReplyToAll: shouldReplyToAll];
      RETAIN(aMessage);
      [editWindowController setMessage: aMessage];
      RELEASE(aMessage);

      [editWindowController showWindow: self];
    }
}


//
//
//
- (IBAction) replyAllMessage: (id) sender
{
  EditWindowController *editWindowController;
  Message *aMessage;
  
  if ([tableView selectedRow] < 0) 
    {
      NSBeep();
      return;
    }
  
  // We obtain the selected entry in the table for the reply informations
  aMessage = [allMessages objectAtIndex: [tableView selectedRow]];

  // We create our window controller
  editWindowController = [[EditWindowController alloc] initWithWindowNibName: @"EditWindow"];

  if ( editWindowController )
    {
      [[editWindowController window] setTitle: _(@"Reply to a message...")];
      [editWindowController setSignaturePosition: [[NSUserDefaults standardUserDefaults] integerForKey: @"SIGNATURE_REPLY_POSITION"] ];
      [editWindowController setShowCc: YES];

      // We set the original message
      [editWindowController setUnmodifiedMessage: aMessage];
      
      // We create a replied copy of our message and we retain it
      aMessage = [aMessage replyWithReplyToAll: YES];
      RETAIN(aMessage);
      [editWindowController setMessage: aMessage];
      RELEASE(aMessage);
      
      [editWindowController showWindow: self];
    }
}


//
// This method displays a message in the textView.
//
- (void) showMessage: (Message *) theMessage
{
  if ( theMessage )
    {
      Flags *theFlags;
      int i;

      // If the content of the message has neven been parsed before, we do it now!
      if (! [theMessage isInitialized] )
	{
	  [theMessage setInitialized: YES];
	}
      
      // We clear our 'Save Attachment' menu
      [(GNUMail *)[NSApp delegate] removeAllItemsFromMenu: [(GNUMail *)[NSApp delegate] saveAttachmentMenu]];

      // We begin by clearing what we have in our text view
      [[textView textStorage] deleteCharactersInRange:
				NSMakeRange(0, [[textView textStorage] length])];

      // We inform our bundles that the message WILL BE shown in the text view
      for (i = 0; i < [[GNUMail allBundles] count]; i++)
	{
	  id<GNUMailBundle> aBundle;
	  
	  aBundle = [[GNUMail allBundles] objectAtIndex: i];

	  if ( [aBundle respondsToSelector: @selector(messageWillBeDisplayed:inView:)] )
	    {
	      [aBundle messageWillBeDisplayed: theMessage
		       inView: [self textView]];
	    }
	}

      // Then, we add our headers and our content
      [[textView textStorage] appendAttributedString: 
				[Utilities attributedStringFromHeadersForMessage: theMessage
					   showAllHeaders: [self showAllHeaders]] ];
      
      [[textView textStorage] appendAttributedString:
				[self _quoteMessageContentFromAttributedString:			     
					[Utilities formattedAttributedStringFromAttributedString:
						     [Utilities attributedStringFromContentForPart: theMessage
								mimeTypeManager: mimeTypeManager]]] ];
      
      
      // We update the Flags of our message (we add SEEN and remove RECENT)
      theFlags = [theMessage flags];

      // If the message was a new one, let's change the app icon back to GNUMail.tiff
      if ( [theFlags contain: RECENT] )
	{
	  [NSApp setApplicationIconImage: [NSImage imageNamed: @"GNUMail.tiff"]];
	}

      if ( ![theFlags contain: SEEN] ) 
	{ 
	  [theFlags add: SEEN];
	}
      
      [[theMessage flags] remove: RECENT];

      // We ensure that the selected table row is properly refreshed after changing the flags
      [tableView setNeedsDisplayInRect: [tableView rectOfRow: [tableView selectedRow]]];
      
      // We finally highlight the URLs in our message, if we want to
      if ( [[NSUserDefaults standardUserDefaults] objectForKey: @"HIGHLIGHT_URL"] &&
	   [[[NSUserDefaults standardUserDefaults] objectForKey: @"HIGHLIGHT_URL"] intValue] == NSOnState )
	{
	  [self _highlightAndActivateURLs];
	}


      // We inform our bundles that the message HAS BEEN shown in the text view
      for (i = 0; i < [[GNUMail allBundles] count]; i++)
	{
	  id<GNUMailBundle> aBundle;
	  
	  aBundle = [[GNUMail allBundles] objectAtIndex: i];

	  if ( [aBundle respondsToSelector: @selector(messageWasDisplayed:inView:)] )
	    {
	      [aBundle messageWasDisplayed: theMessage
		       inView: [self textView]];
	    }
	}
    }
  else
    {
      NSLog(@"Unable to find the message in the hashtable!");
    }
  
  // We set the flag to hide our all our headers on the next click
  [self setShowAllHeaders: NO];

  // We scroll to the beginning of the message and we remove any previous text selection
  [textView scrollRangeToVisible: NSMakeRange(0,0)];
  [textView setSelectedRange: NSMakeRange(0,0)];
}


//
// This method returns the folder associated to this MailWindow.
//
- (Folder *) folder
{
  return folder;
}


//
// This method sets the folder associated to this MailWindow.
//
- (void) setFolder: (Folder *) theFolder
{
  RETAIN(theFolder);
  RELEASE(folder);
  folder = theFolder;

  /* We now set the window title */
  if ( [folder isKindOfClass: [LocalFolder class]] )
    {
      [[self window] setTitle: [NSString stringWithFormat: _(@"Local - %@"), [folder name]] ];
    }
  else
    {
      IMAPStore *aStore;
      
      aStore = (IMAPStore *) [folder store];
      
      [[self window] setTitle: [NSString stringWithFormat: _(@"IMAP on %@ - %@"), [aStore name], 
					 [folder name]] ];
    }
  
  if ( [folder count] == 0 )
    {
      NSRunInformationalAlertPanel(_(@"No Messages!"),
				   _(@"There are no messages in this Mailbox!"),
				   _(@"OK"),
				   NULL,
				   NULL); 
    }
  else
    {
      int i, count;

      // We reload the data our of table (since it now has some)
      [self tableViewShouldReloadData];
      
      count = [folder count];
      
      // We search for the first message with the RECENT flag and we select it
      for (i = 0; i < count; i++)
	{
	  Message *aMessage;

	  aMessage = [allMessages objectAtIndex: i];

	  if ( [[aMessage flags] contain: RECENT] )
	    {
	      break;
	    }
	}

      if (i == count)
	{
	  if ( isReverseOrder )
	    {
	      i = 0;
	    }
	  else
	    {
	      i--;
	    }
	}

      // We scroll to and select our found row
      [tableView scrollRowToVisible: i];
      [tableView selectRow:i byExtendingSelection:NO];
    }

  // We always update our status label, if we have or not messages in our folder
  [self updateStatusLabel];
}



//
// delegate methods
//

//
//
//
- (int) numberOfRowsInTableView: (NSTableView *)aTableView
{
  return [folder count];
}


//
//
//
- (id)           tableView: (NSTableView *) aTableView
 objectValueForTableColumn: (NSTableColumn *) aTableColumn
                       row: (int) rowIndex
{
  InternetAddress *anInternetAddress;
  Message *aMessage;

  aMessage = [allMessages objectAtIndex: rowIndex];

  // We then provide the real data for our row
  if (aTableColumn == statusColumn)
    {
      return @"";
    }
  else if (aTableColumn == idColumn)
    {
      return [NSString stringWithFormat:@"%d", [aMessage messageNumber]];
    }
  else if (aTableColumn == dateColumn)
    {
      return [[aMessage receivedDate] descriptionWithCalendarFormat:@"%b %d"
				 timeZone: [(NSCalendarDate*)[aMessage receivedDate] timeZone]
				 locale: nil];
    }
  else if (aTableColumn == fromColumn)
    {
      anInternetAddress = [aMessage from];
    
      if ([anInternetAddress personal] == nil || [[anInternetAddress personal] length] == 0)
	{
	  return [anInternetAddress address];
	}
      else
	{
	  return [anInternetAddress personal];
	}
    }
  else if (aTableColumn == subjectColumn)
    {
      return [aMessage subject];
    }
  else if (aTableColumn == sizeColumn) 
    {
      return [NSString stringWithFormat: @"%.1fKB ", ((float)[aMessage size]/(float)1024)];
    }
  else 
    {
      return _(@"< unknown >");
    }

  // Never reached.
  return nil;
}


//
//
//
- (void) tableView: (NSTableView *)aTableView
   willDisplayCell: (id)aCell
    forTableColumn: (NSTableColumn *)aTableColumn
               row: (int)rowIndex
{
  // For our filters
  NSColor *aColor;
  
  // For the rest
  Message *aMessage;
  Flags *theFlags;

  aMessage = [allMessages objectAtIndex: rowIndex];
  theFlags = [aMessage flags]; 

  // We verify for a filter
  aColor = [filterManager colorForMessage: aMessage];
 
  // We if have a special color coming from our filter, we set it for this cell
  if ( aColor )
    {
      [aCell setDrawsBackground: YES];
      [aCell setBackgroundColor: aColor];
    }
  else
    {
      [aCell setDrawsBackground: NO];
    }

  // If it's a new message, we set the cell's text to bold
  if ( [theFlags contain: SEEN] )
    {
#ifdef MACOSX
      [aCell setFont: [NSFont systemFontOfSize: [NSFont smallSystemFontSize]]];
#else
      [aCell setFont: [NSFont systemFontOfSize: 0]];
#endif
    }
  else
    {
#ifdef MACOSX
      [aCell setFont: [NSFont boldSystemFontOfSize: [NSFont smallSystemFontSize]]];
#else
      [aCell setFont: [NSFont boldSystemFontOfSize: 0]];
#endif
    }

  // If it's a deleted message, we set the cell's text to italic
  if ( [theFlags contain: DELETED] )
    {
      [aCell setTextColor: [NSColor darkGrayColor]];
#ifdef MACOX
      [aCell setFont: [[NSFontManager sharedFontManager] convertFont: [NSFont systemFontOfSize: 0]
							 toHaveTrait: NSItalicFontMask]];
#else
      [aCell setFont: [[NSFontManager sharedFontManager] convertFont: [NSFont systemFontOfSize:
										[NSFont smallSystemFontSize]]
							 toHaveTrait: NSItalicFontMask]];
#endif
    }
  else
    {
      [aCell setTextColor: [NSColor blackColor]];
    }

  // We set the right aligment for our last (ie., Size) column.
  if (aTableColumn == sizeColumn)
    {
      [aCell setAlignment: NSRightTextAlignment];
    }
  else
    {
      [aCell setAlignment: NSLeftTextAlignment];
    }

  // We set the image of our status cell
  if ( aTableColumn == statusColumn )
    {
      ExtendedCell *cell;
      
      cell = (ExtendedCell *)[statusColumn dataCell];
      [cell setFlags: [[allMessages objectAtIndex: rowIndex] flags]];
      [cell setIsMimeMessage: ([aMessage mimeVersion] == nil ? NO : YES)];
    }
}


//
//
//
- (void) tableViewSelectionDidChange: (NSNotification *) aNotification
{
  if (_isReloading)
    {
      return;
    }
  
  if ( [tableView numberOfSelectedRows] > 1 )
    {
      [textView setString: @""];
    }
  else 
    {
      // If we have no selection, we simply return.
      if ([tableView selectedRow] < 0) 
	{
	  return;
	}
      else
	{
	  // We obtain our selected message
	  Message *aMessage;
	  
	  aMessage = [allMessages objectAtIndex: [tableView selectedRow]];
	  
	  // We show our message!
	  [self showMessage: aMessage];
	}
    }

  [self updateStatusLabel];

  [[NSNotificationCenter defaultCenter]
    postNotificationName: SelectionOfMessageHasChanged
    object: nil
    userInfo: nil];

}


//
//
//
- (void)   tableView: (NSTableView *)aTableView
 didClickTableColumn: (NSTableColumn *)aTableColumn
{
  NSString *newOrder = [aTableColumn identifier];

  if ( ([newOrder isEqualToString: @"Id"] == NO)
       && ([newOrder isEqualToString: @"Date"] == NO)
       && ([newOrder isEqualToString: @"From"] == NO)
       && ([newOrder isEqualToString: @"Subject"] == NO)
       && ([newOrder isEqualToString: @"Size"] == NO) )
    {
      return;
    }
  
  [aTableView setHighlightedTableColumn: aTableColumn];
  previousOrder = currentOrder;

  if ([currentOrder isEqualToString: newOrder])
    {
      if (isReverseOrder == YES)
	{
	  NSLog(@"non reverse");
	}
      else
	{
	  NSLog(@"reverse");
	}
      isReverseOrder = !isReverseOrder;
    }
  else
    {
      currentOrder = newOrder;
      isReverseOrder = NO;
    }
  
  [[NSUserDefaults standardUserDefaults] setObject: currentOrder
					 forKey: @"SORTINGORDER"];

  [[NSUserDefaults standardUserDefaults] setInteger: isReverseOrder
					 forKey: @"SORTINGSTATE"];

  [self tableViewShouldReloadData];
}

- (void)tableView: (NSTableView *)aTableView
mouseDownInHeaderOfTableColumn: (NSTableColumn *)aTableColumn
{
  NSLog(@"mouse down %@", [aTableColumn identifier]);
}


//
//
//
-  (void) textView: (NSTextView *) aTextView
     clickedOnCell: (id <NSTextAttachmentCell>) attachmentCell
	    inRect: (NSRect) cellFrame
	   atIndex: (unsigned) charIndex
  
{
  NSTextAttachment *attachment;
  NSFileWrapper *filewrapper;
  MimeType *aMimeType;
  
  attachment = [attachmentCell attachment];
  filewrapper = [attachment fileWrapper];
  aMimeType = nil;

  if ( mimeTypeManager )
    {
      aMimeType = [mimeTypeManager mimeTypeForFileExtension:[[filewrapper preferredFilename] pathExtension]];
    }

  
  if ( !aMimeType || [aMimeType action] == 0 )
    {
      NSSavePanel *aSavePanel;
      int aChoice;

      aSavePanel = [NSSavePanel savePanel];
      [aSavePanel setAccessoryView:nil];
      [aSavePanel setRequiredFileType: @""];

      aChoice = [aSavePanel runModalForDirectory: [GNUMail currentWorkingPath]
			    file:[filewrapper preferredFilename]];

      /* if successful, save file under designated name */
      if (aChoice == NSOKButton)
	{
	  if (![filewrapper writeToFile: [aSavePanel filename]
			    atomically: YES
			    updateFilenames: YES] )
	    {
	      NSBeep();
	    }

          [GNUMail setCurrentWorkingPath: [[aSavePanel filename] stringByDeletingLastPathComponent]];
	}
    }
  else
    {
      NSString *aString;

      if (! [[NSFileManager defaultManager] fileExistsAtPath: [aMimeType dataHandlerCommand]] )
	{
	  
	  NSRunAlertPanel(_(@"Error!"),
			  _(@"The external program (%@) for opening this MIME-Type (%@) can't be found."),
			  _(@"OK"),
			  NULL,
			  NULL,
			  [aMimeType dataHandlerCommand], [aMimeType mimeType]);
	  return;
	}
      
      aString = [NSString stringWithFormat:@"%@/%d_%@", NSTemporaryDirectory(), 
			  [[NSProcessInfo processInfo] processIdentifier],
			  [filewrapper preferredFilename]];

      if ( [filewrapper writeToFile: aString
			atomically:YES
			updateFilenames: NO] )
	{
	  NSDictionary *aDictionary;

	  aDictionary = [NSDictionary dictionaryWithObjectsAndKeys: [aMimeType dataHandlerCommand], @"command",
				      [NSArray arrayWithObjects: aString, nil], @"arguments",
				      aString, @"file", nil];
	  
	  
	  [NSThread detachNewThreadSelector: @selector(_launchExternalProgram:)
		    toTarget: self
		    withObject: aDictionary];
	}
      else
	{
	  NSBeep();
	}
    }
}

//
//
//
- (BOOL) textView: (NSTextView *) textView
    clickedOnLink: (id) link 
	  atIndex: (unsigned) charIndex
{
  NSLog(@"Opening %@...", [link description]);
  return [[NSWorkspace sharedWorkspace] openURL: link];
}


//
//
//
- (void) windowWillClose: (NSNotification *) not
{
  NSModalSession aModalSession = 0;
  id panel = nil;
  int i;

  NSLog(@"Clean closing the mailbox...");
  
  // We remove our notification observer
  [[NSNotificationCenter defaultCenter]
    removeObserver: self
    name: ShouldCloseMailbox
    object: nil];

  // We show our 'wait panel'
  if ( [[[NSUserDefaults standardUserDefaults] objectForKey: @"SHOWWAITPANELS"] intValue] == NSOnState )
    {
      panel = NSGetInformationalAlertPanel(_(@"Please wait..."), _(@"Closing the mailbox. Please wait."),
					   nil, nil,nil);
      aModalSession = [NSApp beginModalSessionForWindow: panel];
      [NSApp runModalSession: aModalSession];
    }

  // We save the frame of our window
  [Utilities saveRect: [[self window] frame]
	     forWindowName: @"RECT_MAILWINDOW"];

  // We save the height of our textScrollView
  [[NSUserDefaults standardUserDefaults] setInteger: [textScrollView bounds].size.height
					 forKey: @"TEXTSCROLLVIEW_HEIGHT"];
  

  // We update our last mail window on top
  if ( [GNUMail lastMailWindowOnTop] == [self window] )
    {
      [GNUMail setLastMailWindowOnTop: nil];
    }

  // We update our current super view for bundles (we set it to nil) and we
  // inform our bundles that the viewing view will be removed from the super view
  for (i = 0; i < [[GNUMail allBundles] count]; i++)
    {
      id<GNUMailBundle> aBundle;
      
      aBundle = [[GNUMail allBundles] objectAtIndex: i];
      
      if ( [aBundle hasViewingViewAccessory] )
	{
	  [aBundle setCurrentSuperview: nil];
	  [aBundle viewingViewAccessoryWillBeRemovedFromSuperview: [[self window] contentView]];
	}
    }

  
  // If we must compact the mbox on close, let's do it rigth now.
  if ( [[NSUserDefaults standardUserDefaults] integerForKey: @"COMPACT_MAILBOX_ON_CLOSE"] )
    {
      [folder expunge: NO];
    }
  
  [folder close];

  // We close our 'wait panel'
  if ( [[[NSUserDefaults standardUserDefaults] objectForKey: @"SHOWWAITPANELS"] intValue] == NSOnState )
    {
      [NSApp endModalSession: aModalSession];
      [panel close];
      NSReleaseAlertPanel(panel);
    }

  // We clear our 'Save Attachment' menu
  [(GNUMail *)[NSApp delegate] removeAllItemsFromMenu: [(GNUMail *)[NSApp delegate] saveAttachmentMenu]];

  // We remove our window from our list of opened windows
  [GNUMail removeMailWindow: [self window]];

  AUTORELEASE(self);
}


//
//
//
- (void) windowDidLoad
{
  NSLog(@"MailWindowController: -windowDidLoad");

#ifdef MACOSX
  [mailboxes setTarget: [NSApp delegate]];
  [mailboxes setAction: @selector(showMailboxManager:)];
  
  [addresses setTarget: [NSApp delegate]];
  [addresses setAction: @selector(showAddressBook:)];
  
  [find setTarget: [NSApp delegate]];
  [find setAction: @selector(showFindWindow:)];

  [tableView setDoubleAction: @selector(doubleClickedOnTableView:)];
#endif

  // We add our observer (same as above) for clean closing the Mailbox
  [[NSNotificationCenter defaultCenter]
    addObserver: self
    selector: @selector(close)
    name: ShouldCloseMailbox
    object: nil];

  // We finally add our observer if the mime types and the filters change
  [[NSNotificationCenter defaultCenter]
    addObserver: self
    selector: @selector(loadMimeTypes:)
    name: MimeTypesHaveChanged
    object: nil];

  [[NSNotificationCenter defaultCenter]
    addObserver: self
    selector: @selector(loadFilters:)
    name: FiltersHaveChanged
    object: nil];

  [[NSNotificationCenter defaultCenter]
    addObserver: self
    selector: @selector(updateMessageFlags:)
    name: ReplyToMessageWasSuccessful
    object: nil];
  

   // We load our MIME types, filters and we initialize some values
  [self loadMimeTypes: nil];
  [self loadFilters: nil];

  [self setShowAllHeaders: NO];
  [GNUMail setLastMailWindowOnTop: [self window]];
  
  // We add our window from our list of opened window
  [GNUMail addMailWindow: [self window]];
}


//
//
//
- (void) windowDidBecomeKey: (NSNotification *) aNotification
{
  int i;
  
  // We set the last window on top
  [GNUMail setLastMailWindowOnTop: [self window]];
  
  // We set the current superview of our bundle having providing
  // a viewing accessory view.
  for (i = 0; i < [[GNUMail allBundles] count]; i++)
    {
      id<GNUMailBundle> aBundle;
      
      aBundle = [[GNUMail allBundles] objectAtIndex: i];
      
      if ( [aBundle hasViewingViewAccessory] )
	{
	  [aBundle setCurrentSuperview: [[self window] contentView]];
	}
    }

  [[NSApp delegate] setEnableSaveInDraftsMenuItem: NO];
}


//
//
//
- (Message *) selectedMessage
{
  int index;
  
  index = [tableView selectedRow];
  
  if (index < 0)
    {
      return nil;
    }

  return [allMessages objectAtIndex: index];
}

- (NSArray *) selectedMessages
{
  if ( [tableView numberOfSelectedRows] == 0 )
    {
      NSBeep();
    }
  else
    {
      NSMutableArray *aMutableArray;
      NSEnumerator *anEnumerator;
      NSNumber *aRow;;
      
      aMutableArray = [[NSMutableArray alloc] initWithCapacity: [tableView numberOfSelectedRows]];
      
      anEnumerator = [tableView selectedRowEnumerator];

      while ( (aRow = [anEnumerator nextObject]) )
	{
	  [aMutableArray addObject: [allMessages objectAtIndex: [aRow intValue]] ];
	}
      
      return AUTORELEASE(aMutableArray);
    }
  
  // Reached in case of error.
  return nil;
}


//
//
//
- (void) updateStatusLabel
{
  Message *aMessage;
  NSString *aString;
  
  int i, totalSize, unreadCount, totalUnreadSize, messagesSize;
  
  NSEnumerator *enumerator;
  int numberOfSelectedRows;
  id anObject;
  
  totalSize = unreadCount = totalUnreadSize = 0;

  for (i = 0; i < [folder count]; i++)
    {
      aMessage = [allMessages objectAtIndex: i];
      totalSize += [aMessage size];
      if (![[aMessage flags] contain: SEEN])
	{
	  unreadCount++;
	  totalUnreadSize += [aMessage size];
	}
    }
  
  numberOfSelectedRows = [tableView numberOfSelectedRows];
  messagesSize = 0;
  
  if (numberOfSelectedRows > 0)
    {
      enumerator = [tableView selectedRowEnumerator];
      
      while ((anObject = [enumerator nextObject])) {
	aMessage = [allMessages objectAtIndex: [anObject intValue]];
	messagesSize += [aMessage size];
      }
    }
  
  aString = [NSString stringWithFormat: _(@"%d messages (%dKB) - %d unread (%dKB) - %d selected (%0.1fKB) - %d deleted"),
		      [folder count], 
		      (totalSize/1024), unreadCount, (totalUnreadSize/1024), 
		      numberOfSelectedRows, ((float)messagesSize/(float)1024),
		      [folder numberOfDeletedMessages]];
  
  [label setStringValue: aString];
  [label setNeedsDisplay: YES];
}


//
//
//
- (NSTableView *) tableView
{
  return tableView;
}

//
//
//
- (NSTextView *) textView
{
  return textView;
}


//
//
//
- (BOOL) showAllHeaders
{
  if ( [[NSUserDefaults standardUserDefaults] objectForKey: @"SHOWALLHEADERS"] )
    {
      return ([[[NSUserDefaults standardUserDefaults] objectForKey: @"SHOWALLHEADERS"] intValue] == NSOnState ? YES
	      : showAllHeaders);
    }

  return showAllHeaders;
}


//
//
//
- (void) setShowAllHeaders: (BOOL) aBOOL
{
  showAllHeaders = aBOOL;
}


//
//
//
- (BOOL) showSizes
{
  return showSizes;
}

//
//
//
- (void) setShowSizes: (BOOL) aBOOL
{
  NSRect aRect;

  showSizes = aBOOL;

  aRect = [tableView frame];
  
  if ( showSizes )
    {
      [[[self window] contentView] addSubview: label];
    }
  else
    {
      [label removeFromSuperviewWithoutNeedingDisplay];
    }

  [[[self window] contentView] setNeedsDisplay: YES];
}

//
//
//
- (IBAction) loadFilters : (id) sender
{
  TEST_RELEASE(filterManager);

  // We load our filters
  filterManager = [FilterManager filtersFromDisk];

  if ( filterManager )
    {
      RETAIN(filterManager);
    }

  [tableView setNeedsDisplay: YES];
}
//
//
//
- (IBAction) loadMimeTypes : (id) sender
{
  TEST_RELEASE(mimeTypeManager);

  mimeTypeManager = [MimeTypeManager mimeTypesFromDisk];
  
  if ( mimeTypeManager )
    {
      RETAIN(mimeTypeManager);
    }
}


//
//
//
- (IBAction) updateMessageFlags: (id) sender
{
  if (sender)
    {
      Message *aMessage;
      
      aMessage = [[sender userInfo] objectForKey: @"Message"];

      if (aMessage &&  [[[self folder] allMessages] containsObject: aMessage] )
	{
	  [[aMessage flags] add: ANSWERED];
	  [tableView setNeedsDisplay: YES];
	}
    }
}


//
//
//
- (void) tableViewShouldReloadData
{
  NSArray *oldArray = allMessages;
  SEL sortingSel = NULL;

  if (currentOrder == nil)
    {
      previousOrder = currentOrder = @"Id";
    }
  
  if ( [currentOrder isEqualToString: @"Id"] )
    {
      //NSLog(@"id");
      if (isReverseOrder == YES)
	{
	  sortingSel = @selector(reverseCompareAccordingToNumber:);
	  allMessages = 
	    RETAIN([[folder allMessages] 
		     sortedArrayUsingSelector:
		       @selector(reverseCompareAccordingToNumber:)]);
	}
      else
	{
	  sortingSel = @selector(compareAccordingToNumber:);
	  allMessages = [[NSArray alloc]
			  initWithArray: [folder allMessages]];
	}
    }
  else if ( [currentOrder isEqualToString: @"Date"] )
    {
      //NSLog(@"date");
      if (isReverseOrder == YES)
	{
	  sortingSel = @selector(reverseCompareAccordingToDate:);
	  allMessages = 
	    RETAIN([[folder allMessages] 
		     sortedArrayUsingSelector:
		       @selector(reverseCompareAccordingToDate:)]);
	}
      else
	{
	  sortingSel = @selector(compareAccordingToDate:);
	  allMessages = 
	    RETAIN([[folder allMessages] 
		     sortedArrayUsingSelector:
		       @selector(compareAccordingToDate:)]);
	}
    }
  else if ( [currentOrder isEqualToString: @"From"] )
    {
      //NSLog(@"from");
      if (isReverseOrder == YES)
	{
	  sortingSel = @selector(reverseCompareAccordingToSender:);
	  allMessages = 
	    RETAIN([[folder allMessages] 
		     sortedArrayUsingSelector:
		       @selector(reverseCompareAccordingToSender:)]);
	}
      else
	{
	  sortingSel = @selector(compareAccordingToSender:);
	  allMessages = 
	    RETAIN([[folder allMessages] 
		     sortedArrayUsingSelector:
		       @selector(compareAccordingToSender:)]);
	}
    }
  else if ( [currentOrder isEqualToString: @"Subject"] )
    {
      //NSLog(@"subject");
      if (isReverseOrder == YES)
	{
	  sortingSel = @selector(reverseCompareAccordingToSubject:);
	  allMessages = 
	    RETAIN([[folder allMessages] 
		     sortedArrayUsingSelector:
		       @selector(reverseCompareAccordingToSubject:)]);
	}
      else
	{
	  sortingSel = @selector(compareAccordingToSubject:);
	  allMessages = 
	    RETAIN([[folder allMessages] 
		     sortedArrayUsingSelector:
		       @selector(compareAccordingToSubject:)]);
	}
    }
  else if ( [currentOrder isEqualToString: @"Size"] )
    {
      //NSLog(@"subject");
      if (isReverseOrder == YES)
	{
	  sortingSel = @selector(reverseCompareAccordingToSize:);
	  allMessages = 
	    RETAIN([[folder allMessages] 
		     sortedArrayUsingSelector:
		       @selector(reverseCompareAccordingToSize:)]);
	}
      else
	{
	  sortingSel = @selector(compareAccordingToSize:);
	  allMessages = 
	    RETAIN([[folder allMessages] 
		     sortedArrayUsingSelector:
		       @selector(compareAccordingToSize:)]);
	}
    }
  
  if (oldArray != nil)
    {
      NSArray *sc = [[tableView selectedRowEnumerator] allObjects];
      NSMutableArray *sm;
      int selectedRow = [tableView selectedRow];
      int count = [sc count];
      id message;
      int index;
      int i;
      int newCount = [allMessages count];
      BOOL newSelection = NO;
      NSRange range = NSMakeRange(0, newCount);

      {
	sm = [[NSMutableArray alloc] initWithCapacity: newCount];
	for ( i = 0; i < count; i++)
	  {
	    [sm addObject: 
		  [oldArray objectAtIndex: 
			      [[sc objectAtIndex: i] intValue]]];
	  }
	[sm sortUsingSelector: sortingSel];
      }
      _isReloading = YES,

      
      [tableView deselectAll: self];
      [tableView reloadData];

      for ( i = 0; i < count; i++ )
	{
	  message = [sm objectAtIndex: i];
			      
	  index = [allMessages indexOfObject: message
			       inRange: range];

	  if (index != NSNotFound)
	    {
	      [tableView selectRow: index
			 byExtendingSelection: YES];
	      range = NSMakeRange(index + 1, newCount - index - 1);
	    }
	  else
	    {
	      newSelection = YES;
	    }
	}

      if (selectedRow != -1)
	{
	  message = [oldArray objectAtIndex: selectedRow];
	  index = [allMessages indexOfObject: message];
	  
	  if (index != NSNotFound)
	    {
	      [tableView selectRow: index
			 byExtendingSelection: YES];
	    }
	}
      _isReloading = NO;
      
      if (newSelection == YES)
	[self tableViewSelectionDidChange: nil];

      if ([tableView selectedRow] != -1)
	{
	  [tableView scrollRowToVisible: [tableView selectedRow]];
	}
    }

  //NSLog(@"done");
  TEST_RELEASE(oldArray);

  previousOrder = currentOrder;

  [tableView reloadData];
  
  // We verify if we have at least one selected row, in case we don't, we just 
  // clear our textView
  if ( [tableView numberOfSelectedRows] != 1 )
    {
      [textView setString: @""];
    }
}


//
//
//
- (void) reloadTableColumns
{
  NSArray *shownTableColumns;
  int i;

  shownTableColumns = [[NSUserDefaults standardUserDefaults] objectForKey: @"SHOWNTABLECOLUMNS"];

  // If the value doesn't exist in the user's defaults, we show all table columns.
  if ( !shownTableColumns )
    {
      return;
    }

  RETAIN(statusColumn);
  RETAIN(idColumn);
  RETAIN(dateColumn);
  RETAIN(fromColumn);
  RETAIN(subjectColumn);
  RETAIN(sizeColumn);

  [[self tableView] removeTableColumn: statusColumn];
  [[self tableView] removeTableColumn: idColumn];
  [[self tableView] removeTableColumn: dateColumn];
  [[self tableView] removeTableColumn: fromColumn];
  [[self tableView] removeTableColumn: subjectColumn];
  [[self tableView] removeTableColumn: sizeColumn];

  for (i = 0; i < [shownTableColumns count]; i++)
    {
      NSString *column;

      column = [shownTableColumns objectAtIndex: i];

      if ( [column isEqualToString: @"Status"] )
	{
	  [[self tableView] addTableColumn: statusColumn];
	}
      else if ( [column isEqualToString: @"Id"] )
	{
	  [[self tableView] addTableColumn: idColumn];
	}
      else if ( [column isEqualToString: @"Date"] )
	{
	  [[self tableView] addTableColumn: dateColumn];
	}
      else if ( [column isEqualToString: @"From"] )
	{
	  [[self tableView] addTableColumn: fromColumn];
	}
      else if ( [column isEqualToString: @"Subject"] )
	{
	  [[self tableView] addTableColumn: subjectColumn];
	}
      else if ( [column isEqualToString: @"Size"] )
	{
	  [[self tableView] addTableColumn: sizeColumn];
	}
    }
      
      
  [[self tableView] sizeToFit];

  RELEASE(statusColumn);
  RELEASE(idColumn);
  RELEASE(dateColumn);
  RELEASE(fromColumn);
  RELEASE(subjectColumn);
  RELEASE(sizeColumn);

}

@end


//
// Private interface for MailWindowContrller
//
@implementation MailWindowController (Private)

//
//
//
- (void) _highlightAndActivateURLs
{
  NSRange searchRange, foundRange;
  NSString *aString;

  aString = [textView string];
  searchRange = NSMakeRange(0, [aString length]);

  [[textView textStorage] beginEditing];
  
  do 
    {
      // We assume that all URLs start with http://
      foundRange = [aString rangeOfString: @"http://"
			    options: 0 
			    range: searchRange];
      
      // If we haven't found http://, we try to find https://
      if (! foundRange.length )
	{
	  foundRange = [aString rangeOfString: @"https://"
				options: 0 
				range: searchRange];
	}


      // If we haven't found https://, we try to find ftp://
      if (! foundRange.length )
	{
	  foundRange = [aString rangeOfString: @"ftp://"
				options: 0 
				range: searchRange];
	}

      // If we haven't found ftp://, we try to find file://
      if (! foundRange.length )
	{
	  foundRange = [aString rangeOfString: @"file://"
				options: 0 
				range: searchRange];
	}

      // If we found an URL...
      if ( foundRange.length > 0 ) 
	{ 
	  NSDictionary *linkAttributes;
	  NSRange endOfURLRange;
	  NSURL *anURL;
	  
	  // Restrict the searchRange so that it won't find the same string again
	  searchRange.location = foundRange.location + foundRange.length;
	  searchRange.length = [aString length] - searchRange.location;
	  
	  // We assume the URL ends with whitespace
	  endOfURLRange = [aString rangeOfCharacterFromSet: [NSCharacterSet whitespaceAndNewlineCharacterSet]
				   options: 0 
				   range: searchRange];
	  
	  // The URL could also end at the end of the text.  The next line fixes it in case it does
	  if ( endOfURLRange.length == 0)
	    {
	      endOfURLRange.location= [aString length];
	    }
	  
	  // Set foundRange's length to the length of the URL
	  foundRange.length = endOfURLRange.location - foundRange.location;
	  
	  // We create our URL object from our substring
	  anURL= [NSURL URLWithString: [aString substringWithRange: foundRange]];
	  
	  
	  // Make the link attributes
	  linkAttributes= [NSDictionary dictionaryWithObjectsAndKeys: anURL, NSLinkAttributeName,
					[NSNumber numberWithInt: NSSingleUnderlineStyle], NSUnderlineStyleAttributeName,
					[NSColor blueColor], NSForegroundColorAttributeName,
					NULL];
	  
	  // Finally, apply those attributes to the URL in the text
	  [[textView textStorage] addAttributes: linkAttributes 
				  range: foundRange];
	}
      
    } while ( foundRange.length != 0 );
  
  [[textView textStorage] endEditing];
}


//
//
//
- (void) _launchExternalProgram: (NSDictionary *) theDictionary
{
  NSAutoreleasePool *pool;
  NSTask *aTask;
  
  pool = [[NSAutoreleasePool alloc] init];

  aTask = [[NSTask alloc] init];
  AUTORELEASE(aTask);
  
  [aTask setLaunchPath: [theDictionary objectForKey: @"command"]];
  [aTask setArguments: [theDictionary objectForKey: @"arguments"]];
  [aTask launch];
  [aTask waitUntilExit];
  
  [[NSFileManager defaultManager] removeFileAtPath: [theDictionary objectForKey: @"file"]
				  handler: nil];

  RELEASE(pool);
}


//
//
//
- (void) _loadAccessoryViews
{
  NSRect aRect;
  int i;

  aRect = [find frame];

  for (i = 0; i < [[GNUMail allBundles] count]; i++)
    {
      id<GNUMailBundle> aBundle;

      aBundle = [[GNUMail allBundles] objectAtIndex: i];
      
      if ( [aBundle hasViewingViewAccessory] )
	{
	  id aView;
	  aRect.origin.x += 72;

	  aView = [aBundle viewingViewAccessory];
	  [aView setFrame: aRect];
	  [aView setAutoresizingMask: NSViewMinYMargin];

	  [[[self window] contentView] addSubview: aView];
	}

      // We also set the current superview
      [aBundle setCurrentSuperview: [[self window] contentView]];
    }
}

//
//
//
- (NSMutableAttributedString *) _quoteMessageContentFromAttributedString: (NSAttributedString *) theAttributedString
{
  NSMutableAttributedString *aMutableAttributedString;
  NSDictionary *attributes;
  NSArray *anArray;
  int i;

  // Our quoting attributes for the string
  attributes = [NSDictionary dictionaryWithObjectsAndKeys: [NSColor blueColor], NSForegroundColorAttributeName, nil];

  aMutableAttributedString = [[NSMutableAttributedString alloc] initWithAttributedString: theAttributedString];
  
  // We get the range of all our 'quoted' strings
  anArray = [NSRegEx matchString: [theAttributedString string]
		     withPattern: @"\n[>]+"
		     isCaseSensitive: NO];

  for (i = ([anArray count] - 1); i >= 0; i--)
    {
      NSAttributedString *anAttributedString;
      NSMutableString *aMutableString;
      NSRegExRange *aRegExRange;
      NSRange aRange;
      int j;

      aRegExRange = [anArray objectAtIndex: i];
      aRange = [aRegExRange range];

      aMutableString = [[NSMutableString alloc] initWithString: @"\n"];

      for (j = 0; j < (aRange.length - 1); j++)
	{
	  [aMutableString appendString: @"  |"];
	}

      anAttributedString = [[NSAttributedString alloc] initWithString: aMutableString
						       attributes: attributes];
      RELEASE(aMutableString);
      
      [aMutableAttributedString replaceCharactersInRange: aRange
				withAttributedString: anAttributedString];

      RELEASE(anAttributedString);
    }
  

  return AUTORELEASE(aMutableAttributedString);
}


//
//
//
- (void) _updateMessageSelectionUsingIndex: (int) theIndex
{
  // We show the message
  if (theIndex > ([folder count] - 1) || 
      (theIndex - 1) >= 0)
    {
      theIndex = theIndex - 1;
    }
  else if ( (theIndex - 1) < 0 && 
	    [folder count] > 0)
    { 
      theIndex = 0;
    }
  
  if ([folder count] > 0)
    {
      [tableView selectRow: theIndex 
		 byExtendingSelection: NO];
      
      // We post a notification in case we transferred the first messages (index == 0);
      [self tableViewSelectionDidChange: [NSNotification 
             				   notificationWithName: NSTableViewSelectionDidChangeNotification
      					   object: tableView] ];
    }
  else
    {
      [textView setString: @""];
    }
}

- (void) _restoreSortingOrder
{
  // We get our default sorting order
  if ( [[NSUserDefaults standardUserDefaults] objectForKey: @"SORTINGORDER"] )
    {
      currentOrder = [[NSUserDefaults standardUserDefaults] objectForKey: @"SORTINGORDER"];
      isReverseOrder = [[NSUserDefaults standardUserDefaults] integerForKey: @"SORTINGSTATE"];

      if ( [currentOrder isEqualToString: @"Date"] )
	{
	  [[self tableView] setHighlightedTableColumn: dateColumn];
	}
      else if ( [currentOrder isEqualToString: @"From"] )
	{
	  [[self tableView] setHighlightedTableColumn: fromColumn];
	}
      else if ( [currentOrder isEqualToString: @"Subject"] )
	{
	  [[self tableView] setHighlightedTableColumn: subjectColumn];
	}
      else if ( [currentOrder isEqualToString: @"Size"] )
	{
	  [[self tableView] setHighlightedTableColumn: sizeColumn];
	}
      else
	{
	  [[self tableView] setHighlightedTableColumn: idColumn];
	}
    }
  else
    {
      [[self tableView] setHighlightedTableColumn: idColumn];
    }
}

- (void) _restoreSplitViewSize
{
  if ( [[NSUserDefaults standardUserDefaults] objectForKey: @"TEXTSCROLLVIEW_HEIGHT"] ) 
    {
      NSRect aRect;
      
      aRect = [textScrollView frame];
      aRect.size.height = [[NSUserDefaults standardUserDefaults] integerForKey: @"TEXTSCROLLVIEW_HEIGHT"];
      
      [textScrollView setFrame: aRect];
      
      [splitView adjustSubviews];
      [splitView setNeedsDisplay: YES];
    }
}

@end


//
//
//
@implementation NSWindow (KeyDownSupport)

- (void) keyDown: (NSEvent *) theEvent
{
  NSString *characters;
  unichar character;

  characters = [theEvent characters];
  character = 0;

  if ( [characters length] > 0 )
    {
      character = [characters characterAtIndex: 0];
      
      switch ( character )
        {
        case NSUpArrowFunctionKey:
          [(MailWindowController *)_delegate previousMessage: _delegate];
          break;
        case NSDownArrowFunctionKey:
          [(MailWindowController *)_delegate nextMessage: _delegate];
          break;
        case NSHomeFunctionKey:
        case NSBeginFunctionKey:
          [(MailWindowController *)_delegate firstMessage: _delegate];
          break;
        case NSEndFunctionKey:
          [(MailWindowController *)_delegate lastMessage: _delegate];
          break;
        }
    }
}

@end
