/*
**  GNUMail.m
**
**  Copyright (c) 2001, 2002
**
**  Author: Ludovic Marcotte <ludovic@Sophos.ca>
**
**  This program is free software; you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation; either version 2 of the License, or
**  (at your option) any later version.
**
**  This program is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with this program; if not, write to the Free Software
**  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#import "GNUMail.h"

#import "Address.h"
#import "AddressBookController.h"
#import "BounceWindowController.h"
#import "EditWindowController.h"
#import "Filter.h"
#import "FilterManager.h"
#import "FindWindowController.h"
#import "GNUMail/GNUMailBundle.h"
#import "GNUMailConstants.h"
#import "LocalInboxWindowController.h"
#import "MailboxManagerController.h"
#import "PreferencesWindowController.h"
#import "RAWSourceWindowController.h"
#import "Utilities.h"

#import <Pantomime/Constants.h>
#import <Pantomime/InternetAddress.h>
#import <Pantomime/LocalFolder.h>
#import <Pantomime/LocalStore.h>
#import <Pantomime/Message.h>
#import <Pantomime/URLName.h>

@implementation GNUMail

static NSMutableArray *allMailWindows = nil;
static NSMutableArray *allBundles;

static NSString *currentWorkingPath = nil;

static id lastAddressTakerWindowOnTop = nil;
static id lastMailWindowOnTop = nil;

//
// This method is used to initialize GNUMail.app.
// It verifies if the User's Library directory exist (to store 
// the AddressBook, the MimeTypes, etc) and if not, it creates it.
//
- (id) init
{
  NSFileManager *fileManager;
  BOOL isDir;

  self = [super init];

  fileManager = [NSFileManager defaultManager];

  if ( [fileManager fileExistsAtPath: (NSString *)GNUMailUserLibraryPath()
		    isDirectory: &isDir] )
    {
      if ( ! isDir )
	{
	  NSLog(@"%@ exists but it is a file not a directory.",
		GNUMailUserLibraryPath());
	  exit(1);
	}
    }
  else    {
      if ( ! [fileManager createDirectoryAtPath: (NSString *)GNUMailUserLibraryPath()
			  attributes: nil] )
	{
	  // directory creation failed.  quit.
	  NSLog(@"Could not create directory: %@", GNUMailUserLibraryPath());
	  exit(1);
	}
      else
	{
	  NSLog(@"Created directory: %@", GNUMailUserLibraryPath());
	}
    }
  
  // We register for our notifications
  [[NSNotificationCenter defaultCenter]
    addObserver: self
    selector: @selector(selectionOfMessageHasChanged:)
    name: SelectionOfMessageHasChanged
    object: nil];

  [[NSNotificationCenter defaultCenter]
    addObserver: self
    selector: @selector(updateFilterMenuItems:)
    name: FiltersHaveChanged
    object: nil];

  [[NSNotificationCenter defaultCenter]
    addObserver: self
    selector: @selector(selectionInTextViewHasChanged:)
    name: NSTextViewDidChangeSelectionNotification
    object: nil];

  // We initialize our mutable array containing all opened mail windows
  allMailWindows = [[NSMutableArray alloc] init];

  // We initialize our mutable array containing all our bundles
  allBundles = [[NSMutableArray alloc] init];

  [GNUMail setCurrentWorkingPath: NSHomeDirectory()];

  return self;
}

//
// action methods
//

- (IBAction) bounceMessage: (id) sender
{
  // We first verify if we have at least one transport agent defined
  if (! [[NSUserDefaults standardUserDefaults] objectForKey: @"SENDING"] ||
      ! [(NSArray *)[[NSUserDefaults standardUserDefaults] objectForKey: @"SENDING"] count] )
    {
      NSRunAlertPanel(_(@"Error!"),
		      _(@"You must have at least one transport agent defined.\nSee Preferences -> Sending."),
		      _(@"OK"), // default
		      NULL,     // alternate
		      NULL);
      
      return;
    }

  if ( [GNUMail lastMailWindowOnTop] ) 
    {
      MailWindowController *mailWindowController;
      Message *theMessage;
      
      mailWindowController = (MailWindowController *)[[GNUMail lastMailWindowOnTop] delegate];
      theMessage = [mailWindowController selectedMessage];
      
      if ( theMessage )
	{
	  BounceWindowController *bounceWindowController;
	  
	  bounceWindowController = [[BounceWindowController alloc] initWithWindowNibName: @"BounceWindow"];
	  [bounceWindowController setMessage: theMessage];
	  [bounceWindowController showWindow: self];
	}
      else
	{
	  NSBeep();
	}
    }
  else
    {
      NSBeep();
    }
}


//
//
//
- (IBAction) compactMailbox: (id) sender
{
  if ( [GNUMail lastMailWindowOnTop] ) 
    {
      MailWindowController *mailWindowController;
      int choice;
      
      mailWindowController = (MailWindowController *)[[GNUMail lastMailWindowOnTop] delegate];
      
      choice = NSRunAlertPanel(_(@"Compact..."),
			       _(@"Compacting a mailbox will permanently remove deleted messages.\nDo you want to continue?"),
			       _(@"No"),  // default
			       _(@"Yes"), // alternate
			       NULL);

      if (choice == NSAlertAlternateReturn)
	{
	  [[mailWindowController folder] expunge: NO];
	  [mailWindowController tableViewShouldReloadData];
	}
    }
  else
    {
      NSBeep();
    }
}

//
// This method is used to compose a new message, with an empty content.
//
- (IBAction) composeMessage: (id) sender
{
  EditWindowController *editWindowController;
  Message *aMessage;
  
  // We create an empty message
  aMessage = [[Message alloc] init];
  
  // We create our controller and we show the window
  editWindowController = [[EditWindowController alloc] initWithWindowNibName: @"EditWindow"];

  if ( editWindowController )
    {
      [[editWindowController window] setTitle: _(@"New message...")];
      [editWindowController setMessage: aMessage];
      [editWindowController setShowCc: NO];
            
      [editWindowController showWindow: self];
    }

  RELEASE(aMessage);
}


//
// This method is used to togle the flags of the selected message
// from the last mail window on top to 'undelete' (or delete) if it was
// set to 'deleted' before the call.
//
- (IBAction) deleteOrUndeleteMessage: (id) sender
{  
  if ( [GNUMail lastMailWindowOnTop] ) 
    {
      MailWindowController *mailWindowController;
      
      mailWindowController = (MailWindowController *)[[GNUMail lastMailWindowOnTop] delegate];
      
      // We delete (or undelete) our message
      [mailWindowController deleteMessage: nil];
      
      // We 'swap' our title and our tag
      if ( [deleteOrUndelete tag] == DELETE_MESSAGE )
	{
	  [deleteOrUndelete setTitle: _(@"Undelete")];
	  [deleteOrUndelete setTag: UNDELETE_MESSAGE];
	}
      else
	{
	  [deleteOrUndelete setTitle: _(@"Delete")];
	  [deleteOrUndelete setTag: DELETE_MESSAGE];
	}
    }
  else
    {
      NSBeep();
    }
}


//
//
//
- (IBAction) enterSelectionInFindPanel: (id) sender
{
  if ( [GNUMail lastMailWindowOnTop] ) 
    {
      MailWindowController *mailWindowController;
      NSTextView *aTextView;

      mailWindowController = (MailWindowController *)[[GNUMail lastMailWindowOnTop] delegate];
      aTextView = [mailWindowController textView];
      
      [[[FindWindowController singleInstance] 
	 findField] setStringValue: [[aTextView string] substringWithRange: [aTextView selectedRange]]];
    }
  else
    {
      NSBeep();
    }
}


//
//
//
- (IBAction) findNext: (id) sender
{
  [[FindWindowController singleInstance] nextMessage: nil];
}


//
//
//
- (IBAction) findPrevious: (id) sender
{
  [[FindWindowController singleInstance] previousMessage: nil];
}


// 
// This method is used to forward the selected messsage from the last
// mail window that was on top.
//
- (IBAction) forwardMessage: (id) sender
{
  MailWindowController *mailWindowController;
  
  if ( [GNUMail lastMailWindowOnTop] )
    {
      mailWindowController = (MailWindowController *)[[GNUMail lastMailWindowOnTop] delegate];
      [mailWindowController forwardMessage: nil];
    }
}


//
// This method is used to get the new messages if the LocalInboxWindow
// is currently opened and visible.
//
- (IBAction) getNewMessages: (id) sender
{
  if ( [LocalInboxWindowController localInboxWindowController] && 
       [[[LocalInboxWindowController localInboxWindowController] window] isVisible])
    {
      [[LocalInboxWindowController localInboxWindowController] getNewMessages: self];
    }
  else
    {
      NSBeep();    
    }
}


//
//
//
- (IBAction) markMessageAsReadOrUnread: (id) sender
{
  MailWindowController *mailWindowController;
  
  if ( [GNUMail lastMailWindowOnTop] ) 
    {
      NSEnumerator *anEnumerator;
      Message *aMessage;
      NSNumber *aRow;
      
      mailWindowController = (MailWindowController *)[[GNUMail lastMailWindowOnTop] delegate];
      
      // We mark our message as read (or not)
      anEnumerator = [[mailWindowController tableView] selectedRowEnumerator];

      while ( (aRow = [anEnumerator nextObject]) )
	{
	  aMessage = [[[mailWindowController folder] allMessages] objectAtIndex: [aRow intValue]];
	  
	  // If we must mark all messages as read
	  if ( [markAsReadOrUnread tag] == MARK_AS_READ )
	    {
	      if ( [[aMessage flags] contain: RECENT] )
		{
		  [[aMessage flags] remove: RECENT];
		  [[aMessage flags] add: SEEN];
		} 
	    }
	  // Else, we must mark them all as unread
	  else
	    {
	      if ( [[aMessage flags] contain: SEEN] )
		{
		  [[aMessage flags] remove: SEEN];
		  [[aMessage flags] add: RECENT];
		} 
	    }
	} // while (...)
      
      // We always refresh our tableView after a delete operation
      [[mailWindowController tableView] setNeedsDisplay:YES];  

      // We 'swap' our title and our tag
      if ( [markAsReadOrUnread tag] == MARK_AS_READ )
	{
	  [markAsReadOrUnread setTitle: _(@"Mark as Unread")];
	  [markAsReadOrUnread setTag: MARK_AS_UNREAD];
	}
      else
	{
	  [markAsReadOrUnread setTitle: _(@"Mark as Read")];
	  [markAsReadOrUnread setTag: MARK_AS_READ];
	}
    }
  else
    {
      NSBeep();
    }
}

//
//
//
- (IBAction) print: (id) sender
{
  if ( [GNUMail lastMailWindowOnTop] )
    {
      MailWindowController *mailWindowController;

      mailWindowController = (MailWindowController *)[[GNUMail lastMailWindowOnTop] delegate];
      
      [[NSPrintOperation printOperationWithView: [mailWindowController textView]] runOperation];
    }
  else
    {
      NSBeep();
    }
}

//
// This method is used to reply to the selected message
// to all recipients.
//
- (IBAction) replyAllMessage: (id) sender
{  
  if ( [GNUMail lastMailWindowOnTop] )
    {
      MailWindowController *mailWindowController;

      mailWindowController = (MailWindowController *)[[GNUMail lastMailWindowOnTop] delegate];
      [mailWindowController replyAllMessage: nil];
    }
  else
    {
      NSBeep();
    }
}


//
// This method is used to reply to the selected message.
//
- (IBAction) replyMessage: (id) sender
{
  MailWindowController *mailWindowController;
  
  if ( [GNUMail lastMailWindowOnTop] )
    {
      mailWindowController = (MailWindowController *)[[GNUMail lastMailWindowOnTop] delegate];
      [mailWindowController replyMessage: nil];
    }
}


//
//
//
- (IBAction) restoreDraft: (id) sender
{
  EditWindowController *editWindowController;
  LocalFolder *draftsFolder;
  Message *aMessage;
  id aWindow;
  
  aMessage = nil;
  aWindow = nil;
  
  draftsFolder = (LocalFolder *)[[self localStore] folderForName: [[NSUserDefaults standardUserDefaults] 
								    objectForKey: @"DRAFTSFOLDERNAME"]];
  
  // The folder is opened, we get the selected message
  if (! draftsFolder)
    {
      MailWindowController *aMailWindowController;

      aWindow = [Utilities windowForFolderName: [[NSUserDefaults standardUserDefaults] 
						  objectForKey: @"DRAFTSFOLDERNAME"]];

      aMailWindowController = [aWindow windowController];
      draftsFolder = (LocalFolder *)[aMailWindowController folder];

      if ( [draftsFolder count] > 0 &&
	   [aMailWindowController selectedMessage])
	{
	  aMessage = [aMailWindowController selectedMessage];
	}
      else
	{
	  NSBeep();
	  return;
	}
    }
  else
    {
      // The folder is NOT opened, let's get the last message
      if ( [draftsFolder count] > 0 )
	{
	  aMessage = [draftsFolder messageAtIndex:([draftsFolder count] - 1)];
	  
	  // We MUST initialize our message
	  [aMessage setInitialized: YES];
	}
      else
	{
	  NSBeep();
	  [draftsFolder close];
	  return;
	}
    }
  
  // We create our controller and we show the window if we have a valid message
  if ( aMessage )
    {
      editWindowController = [[EditWindowController alloc] initWithWindowNibName: @"EditWindow"];
      
      if ( editWindowController )
	{
	  [[editWindowController window] setTitle: _(@"New message...")];
	  [editWindowController setMessageFromDraftsFolder: aMessage];
	  [editWindowController setShowCc: NO];
	  
	  [editWindowController showWindow: self];
	}
    }

  // We close the folder, if we need to.
  if (! aWindow )
    {
      [draftsFolder close];
    }
}

//
// This method is used to select all the messages from the last
// mail window that was on top.
//
- (IBAction) selectAllMessages: (id) sender
{  
  if ( [GNUMail lastMailWindowOnTop] )
    {
      MailWindowController *mailWindowController;
      
      mailWindowController = (MailWindowController *)[[GNUMail lastMailWindowOnTop] delegate];
      
      [[mailWindowController tableView] selectAll: sender];
      [[mailWindowController tableView] setNeedsDisplay: YES];
    }
  else
    {
      NSBeep();
    }
}


//
//
//
- (IBAction) saveInDrafts: (id) sender
{
  if ( [GNUMail lastAddressTakerWindowOnTop] )
    {
      EditWindowController *aController;
      LocalFolder *draftsFolder;
      id aWindow;
      
      aController = [GNUMail lastAddressTakerWindowOnTop];
      aWindow = nil;

      draftsFolder = (LocalFolder *)[[self localStore] folderForName: [[NSUserDefaults standardUserDefaults] 
									objectForKey: @"DRAFTSFOLDERNAME"]];
      
      // Then, we try to open our Outbox maibox, if it's already open, we get the window used by this folder
      if (! draftsFolder )
	{
	  aWindow = [Utilities windowForFolderName: [[NSUserDefaults standardUserDefaults] 
						      objectForKey: @"DRAFTSFOLDERNAME"]];
	  draftsFolder = (LocalFolder *)[[aWindow windowController] folder];
	  
	} // if (! draftsFolder )
      
      
      [aController updateMessageContentFromTextView: NO];
      
      // We create a copy of the message in Drafts and we close this folder after
      [draftsFolder appendMessageFromRawSource: [[aController message]
						  dataUsingSendingMode: SEND_TO_FOLDER]];
      
      
      if ( aWindow )
	{
	  [[aWindow windowController] tableViewShouldReloadData];
	}
      else
	{
	  [draftsFolder close];
	}
    }
  else
    {
      NSBeep();
    }
}

//
//
//
- (IBAction) saveAttachment: (id) sender
{
  if ( [GNUMail lastMailWindowOnTop] )
    {
      MailWindowController *mailWindowController;
      NSTextAttachment *attachment;

      attachment = [sender textAttachment];
      
      // We get a reference to our mail window controller
      mailWindowController = (MailWindowController *)[[GNUMail lastMailWindowOnTop] delegate];

      [mailWindowController textView: [mailWindowController textView]
			    clickedOnCell: [attachment attachmentCell]
			    inRect: NSZeroRect
			    atIndex: 0];
    }
  else
    {
      NSBeep();
    }
}

//
// This method is used to save only the textual content of a message
// to the local file system. It skips attachments.
//
- (IBAction) saveTextFromMessage: (id) sender
{
  if ( [GNUMail lastMailWindowOnTop] )
    {
      MailWindowController *mailWindowController;
      NSSavePanel *aSavePanel;
      NSData *aData;
      int aChoice;

      // We get a reference to our mail window controller
      mailWindowController = (MailWindowController *)[[GNUMail lastMailWindowOnTop] delegate];
      
      if ( [mailWindowController selectedMessage] )
	{
	  // We get our content of a message (just the text displayed in mail window)
	  aData = [[[mailWindowController textView] string] 
		    dataUsingEncoding: NSASCIIStringEncoding
		    allowLossyConversion: YES];
	  
	  aSavePanel = [NSSavePanel savePanel];
	  [aSavePanel setAccessoryView: nil];
	  [aSavePanel setRequiredFileType: @""];
	  
	  aChoice = [aSavePanel runModalForDirectory: [GNUMail currentWorkingPath] file: @"Unknown"];
      
	  /* if successful, save file under designated name */
	  if (aChoice == NSOKButton)
	    {
	      if (![aData writeToFile: [aSavePanel filename]
			  atomically: YES] )
		{
		  NSBeep();
		}

              [GNUMail setCurrentWorkingPath: [[aSavePanel filename] stringByDeletingLastPathComponent]];
	    }
	}
      else
	{
	  NSBeep();
	}
    }
  else
    {
      NSBeep();
    }
}


//
//
//
- (IBAction) sortByNumber: (id) sender
{
  if ( [GNUMail lastMailWindowOnTop] ) 
    {
      MailWindowController *mailWindowController;
      
      mailWindowController = (MailWindowController *)[[GNUMail lastMailWindowOnTop] delegate];
      
      [mailWindowController
	tableView: [mailWindowController tableView]
	didClickTableColumn: [[mailWindowController tableView]
			       tableColumnWithIdentifier: @"Id"]];
    }
  else
    {
      NSBeep();
    }
}


//
//
//
- (IBAction) sortByDate: (id) sender
{
  if ( [GNUMail lastMailWindowOnTop] ) 
    {
      MailWindowController *mailWindowController;
      
      mailWindowController = (MailWindowController *)[[GNUMail lastMailWindowOnTop] delegate];
      
      [mailWindowController
	tableView: [mailWindowController tableView]
	didClickTableColumn: [[mailWindowController tableView]
			       tableColumnWithIdentifier: @"Date"]];
    }
  else
    {
      NSBeep();
    }
}


//
//
//
- (IBAction) sortByName: (id) sender
{
  if ( [GNUMail lastMailWindowOnTop] ) 
    {
      MailWindowController *mailWindowController;
      
      mailWindowController = (MailWindowController *)[[GNUMail lastMailWindowOnTop] delegate];
      
      [mailWindowController
	tableView: [mailWindowController tableView]
	didClickTableColumn: [[mailWindowController tableView]
			       tableColumnWithIdentifier: @"From"]];
    }
  else
    {
      NSBeep();
    }
}


//
//
//
- (IBAction) sortBySubject: (id) sender
{
  if ( [GNUMail lastMailWindowOnTop] ) 
    {
      MailWindowController *mailWindowController;
      
      mailWindowController = (MailWindowController *)[[GNUMail lastMailWindowOnTop] delegate];
      
      [mailWindowController
	tableView: [mailWindowController tableView]
	didClickTableColumn: [[mailWindowController tableView]
			       tableColumnWithIdentifier: @"Subject"]];
    }
  else
    {
      NSBeep();
    }
}


//
//
//
- (IBAction) sortBySize: (id) sender
{
  if ( [GNUMail lastMailWindowOnTop] ) 
    {
      MailWindowController *mailWindowController;
      
      mailWindowController = (MailWindowController *)[[GNUMail lastMailWindowOnTop] delegate];
      
      [mailWindowController
	tableView: [mailWindowController tableView]
	didClickTableColumn: [[mailWindowController tableView]
			       tableColumnWithIdentifier: @"Size"]];
    }
  else
    {
      NSBeep();
    }
}



//
// This method is used to show the address book on the screen.
// We use a singleton.
//
- (IBAction) showAddressBook: (id) sender
{
  [[AddressBookController singleInstance] showWindow: self];
}


//
// This method is used to view or hide all headers in the 
// last mail window on top.
//
- (IBAction) showAllHeaders: (id) sender
{ 
  if ( [GNUMail lastMailWindowOnTop] )
    {
      MailWindowController *mailWindowController;
      int row;
      
      mailWindowController = (MailWindowController *)[[GNUMail lastMailWindowOnTop] delegate];
      
      row = [[mailWindowController tableView] selectedRow];

      if (row == -1 || [[mailWindowController tableView] numberOfSelectedRows] > 1)
	{
	  NSBeep();
	  return;
	}

      if ( [sender tag] == SHOW_ALL_HEADERS )
	{
	  [mailWindowController setShowAllHeaders: YES];     
	  [sender setTitle: _(@"Show Filtered Headers")];
	  [sender setTag: HIDE_ALL_HEADERS];
	}
      else
	{
	  [mailWindowController setShowAllHeaders: NO];	      
	  [sender setTitle: _(@"Show All Headers")];
	  [sender setTag: SHOW_ALL_HEADERS];
	}
      
      [menu sizeToFit];
      
      [mailWindowController showMessage: [[[mailWindowController folder] allMessages] objectAtIndex: row] ];
    }
  else
    {
      NSBeep();
    }
}


//
// This method is used to show the find window on the screen.
// We use a singleton.
//
- (IBAction) showFindWindow: (id) sender
{
  [[FindWindowController singleInstance] showWindow: self];
}


//
// This method is used to show the mailbox manager window on the screen.
// We use a singleton.
//
- (IBAction) showMailboxManager: (id) sender
{
  [[MailboxManagerController singleInstance] showWindow: self];
}


//
//
//
- (IBAction) showOrHideDeletedMessages: (id) sender
{
  if ( [GNUMail lastMailWindowOnTop] )
    {
      MailWindowController *mailWindowController;
      
      mailWindowController = (MailWindowController *)[[GNUMail lastMailWindowOnTop] delegate];

      if ( [showOrHideDeleted tag] == SHOW_DELETED_MESSAGES )
	{
	  [self updateShowOrHideDeletedMenuItem: YES];
	  [[mailWindowController folder] setShowDeleted: YES];
	}
      else
	{
	  [self updateShowOrHideDeletedMenuItem: NO];
	  [[mailWindowController folder] setShowDeleted: NO];
	}
      
      [mailWindowController tableViewShouldReloadData];
    }
  else
    {
      NSBeep();
    }
}


//
//
//
- (IBAction) showOrHideSizesForMailWindow: (id) sender
{
  if ( [GNUMail lastMailWindowOnTop] )
    {
      MailWindowController *mailWindowController;
      
      mailWindowController = (MailWindowController *)[[GNUMail lastMailWindowOnTop] delegate];

      if ( [showOrHideSizes tag] == SHOW_SIZES )
	{
	  [showOrHideSizes setTitle: _(@"Hide Sizes")];
	  [showOrHideSizes setTag: HIDE_SIZES];
	  [mailWindowController setShowSizes: YES];
	}
      else
	{
	  [showOrHideSizes setTitle: _(@"Show Sizes")];
	  [showOrHideSizes setTag: SHOW_SIZES];
	  [mailWindowController setShowSizes: NO];
	}
      
      [mailWindowController tableViewShouldReloadData];
    }
  else
    {
      NSBeep();
    }
}


//
// This method is used to show the preferences window on the screen.
// The window is modal.
//
- (IBAction) showPreferencesWindow: (id) sender
{
  NSAutoreleasePool *pool;
  id aWindow;

  pool = [[NSAutoreleasePool alloc] init];
  aWindow = [[PreferencesWindowController singleInstance] window];
  
  if ( aWindow )
    {
      [NSApp runModalForWindow: aWindow];
    }

  RELEASE(pool);
}


//
// This method is used to show the raw source of the selected message
// from the last mail window on top.
//
- (IBAction) showRAWSource: (id) sender
{  
  if ( [GNUMail lastMailWindowOnTop] ) 
    {
      MailWindowController *mailWindowController;
      Message *aMessage;
      
      mailWindowController = (MailWindowController *)[[GNUMail lastMailWindowOnTop] delegate];
      aMessage = [mailWindowController selectedMessage];
      
      if ( aMessage && 
	   [[mailWindowController tableView] numberOfSelectedRows] == 1 )
	{
	  NSString *aString;

	  aString = AUTORELEASE([[NSString alloc] initWithData: [aMessage rawSource]  encoding: NSASCIIStringEncoding]);
	  
	  if ( aString )
	    {
	      RAWSourceWindowController *rawSourceWindowController;

	      rawSourceWindowController = [[RAWSourceWindowController alloc] initWithWindowNibName: @"RAWSourceWindow"];
	      [rawSourceWindowController setRAWSource: aString];
	      [rawSourceWindowController showWindow: self];
	    }
	}
      else
	{
	  NSBeep();
	}
	 
    }
  else
    {
      NSBeep();
    }
}


//
// delegate methods
//

- (BOOL) applicationShouldTerminate: (NSApplication *) sender
{
  // Closes all opened MailWindow and terminates the application
  [[NSNotificationCenter defaultCenter]
    postNotificationName: ShouldCloseMailbox
    object: nil
    userInfo: nil];
  
  // We remove our observer on message selection changes
  [[NSNotificationCenter defaultCenter]
    removeObserver: self
    name: SelectionOfMessageHasChanged
    object: nil];
  
  // We remove our observer on the filter changes
  [[NSNotificationCenter defaultCenter]
    removeObserver: self
    name: FiltersHaveChanged
    object: nil];
    

  // We wait until all our windows are closed
  while ( [allMailWindows count] > 0 )
    {
      [[NSRunLoop currentRunLoop] runMode: NSDefaultRunLoopMode
				  beforeDate: [NSDate distantFuture]];
    }

  // We release our array containing all our mail windows
  RELEASE(allMailWindows);
  allMailWindows = nil;

  // We close our local store
  if ( localStore )
    {
      [localStore close];
      RELEASE(localStore);
    }

  // We release our current working path
  TEST_RELEASE(currentWorkingPath);

  // We release our array containing all our bundles
  RELEASE(allBundles);

  return YES;
}


- (void) applicationWillFinishLaunching: (NSNotification *) not
{
#ifndef MACOSX
  // Local variable
  SEL action = NULL;
#endif
  
  // We begin by setting our NSApp's logo
  [NSApp setApplicationIconImage: [NSImage imageNamed: @"GNUMail.tiff"]];

#ifndef MACOSX
  // We continue by creating our NSMenu
  menu = [[NSMenu alloc] init];
  
  [menu addItemWithTitle:_(@"Info") 
	action: action 
	keyEquivalent: @""];
  [menu addItemWithTitle:_(@"Mailbox") 
	action: action 
	keyEquivalent: @""];
  [menu addItemWithTitle:_(@"Message") 
	action: action 
	keyEquivalent: @""];
  [menu addItemWithTitle:_(@"Compose")
	action: action 
	keyEquivalent: @""];
  [menu addItemWithTitle:_(@"Edit")
	action: action
	keyEquivalent: @"h"];
  [menu addItemWithTitle:_(@"Windows")
	action: action
	keyEquivalent: @"p"];
  [menu addItemWithTitle:_(@"Print")
	action: action
	keyEquivalent: @"p"];
  [menu addItemWithTitle:_(@"Services")
	action: action
	keyEquivalent: @""];
  [menu addItemWithTitle:_(@"Hide")
	action: @selector (hide:)
	keyEquivalent: @"h"];
  [menu addItemWithTitle:_(@"Quit")
	action: @selector(terminate:)
	keyEquivalent: @"q"];
  
  // Our Info menu / submenus
  info = [[NSMenu alloc] init];
  [menu setSubmenu:info forItem:[menu itemWithTitle:_(@"Info")]];
  [info addItemWithTitle:_(@"Info Panel...")
	action: @selector(orderFrontStandardInfoPanel:)   
	keyEquivalent:@""];
  [info addItemWithTitle:_(@"Preferences...")
	action:@selector(showPreferencesWindow:)
	keyEquivalent:@""];
  [info addItemWithTitle:_(@"Help...")
	action: action
	keyEquivalent:@"?"];
  RELEASE(info);

  // Our Mailbox menu / submenus
  mailbox = [[NSMenu alloc] init];
  [menu setSubmenu:mailbox forItem:[menu itemWithTitle:_(@"Mailbox")]];
  [mailbox addItemWithTitle: _(@"Mailboxes...")
	   action:@selector(showMailboxManager:) 
	   keyEquivalent: @"M"];
  [mailbox addItemWithTitle: _(@"Get New Mail")
	   action:@selector(getNewMessages:)
	   keyEquivalent: @"N"];
  [mailbox addItemWithTitle: _(@"Sorting")
	   action: action 
	   keyEquivalent: @""];

  // We create our sorting menu
  sorting = [[NSMenu alloc] init];
  [[mailbox itemAtIndex: 2] setSubmenu: sorting];
  
  [sorting addItemWithTitle: _(@"Sort by Date")
	   action: @selector(sortByDate:) 
	   keyEquivalent: @""];
  [sorting addItemWithTitle: _(@"Sort by Name")
	   action: @selector(sortByName:)
	   keyEquivalent: @"S"];
  [sorting addItemWithTitle: _(@"Sort by Number")
	   action: @selector(sortByNumber:)
	   keyEquivalent: @""];
  [sorting addItemWithTitle: _(@"Sort by Size")
	   action: @selector(sortBySize:)
	   keyEquivalent: @""];
  [sorting addItemWithTitle: _(@"Sort by Subject")
	   action: @selector(sortBySubject:)
	   keyEquivalent: @""];
  RELEASE(sorting);
  
  // We add the rest of our mailbox menu items
  showOrHideDeleted = [[NSMenuItem alloc] init];
  [showOrHideDeleted setTitle: _(@"Hide Deleted")]; // and Show Deleted
  [showOrHideDeleted setAction: @selector(showOrHideDeletedMessages:)];
  [showOrHideDeleted setKeyEquivalent: @""];
  [showOrHideDeleted setTag: HIDE_DELETED_MESSAGES];
  [mailbox addItem: showOrHideDeleted];
  RELEASE(showOrHideDeleted);
	  
  showOrHideSizes = [[NSMenuItem alloc] init];
  [showOrHideSizes setTitle: _(@"Hide Sizes")]; // and Show Sizes
  [showOrHideSizes setAction: @selector(showOrHideSizesForMailWindow:)];
  [showOrHideSizes setKeyEquivalent: @""];
  [showOrHideSizes setTag: HIDE_SIZES];
  [mailbox addItem: showOrHideSizes];
  RELEASE(showOrHideSizes);

  [mailbox addItemWithTitle: _(@"Compact")
	   action: @selector(compactMailbox:)
	   keyEquivalent:@""];
  RELEASE(mailbox);
  
  // Our Message menu / submenus
  message = [[NSMenu alloc] init];
  [menu setSubmenu:message forItem:[menu itemWithTitle:_(@"Message")]];

  [message addItemWithTitle:_(@"Select All Messages")
	   action: @selector(selectAllMessages:)
	   keyEquivalent: @""];
  
  showAllHeaders = [[NSMenuItem alloc] init];
  [showAllHeaders setTitle: _(@"Show All Headers")];
  [showAllHeaders setAction: @selector(showAllHeaders:)];
  [showAllHeaders setKeyEquivalent: @""];
  [message addItem: showAllHeaders];
  RELEASE(showAllHeaders);
  
  [message addItemWithTitle:_(@"Show RAW Source")
	   action:@selector(showRAWSource:)  
	   keyEquivalent: @""];

  [message addItemWithTitle:_(@"Send to Address Book")
	   action: @selector(selectAllMessages:)
	   keyEquivalent: @""];
  
  sendToAddressBook = [[NSMenu alloc] init];
  [message setSubmenu: sendToAddressBook
	   forItem: [message itemWithTitle: _(@"Send to Address Book")]];
  RELEASE(sendToAddressBook);

  [message addItemWithTitle: _(@"Save Attachment")
	   action: action
	   keyEquivalent: @""];
  
  saveAttachment = [[NSMenu alloc] init];
  [message setSubmenu: saveAttachment
	   forItem: [message itemWithTitle: _(@"Save Attachment")]];
  RELEASE(saveAttachment);

  [message addItemWithTitle:_(@"Save Text from Message")
	   action:@selector(saveTextFromMessage:)  
	   keyEquivalent: @""];

  
  markAsReadOrUnread = [[NSMenuItem alloc] init];
  [markAsReadOrUnread setTitle: _(@"Mark as Read")];
  [markAsReadOrUnread setAction: @selector(markMessageAsReadOrUnread:)];
  [markAsReadOrUnread setKeyEquivalent: @""];
  [message addItem: markAsReadOrUnread];
  RELEASE(markAsReadOrUnread);
  

  RELEASE(message);

  deleteOrUndelete = [[NSMenuItem alloc] init];
  [deleteOrUndelete setTitle: _(@"Undelete")];
  [deleteOrUndelete setAction: @selector(deleteOrUndeleteMessage:)];
  [deleteOrUndelete setKeyEquivalent: @"u"];
  [message addItem: deleteOrUndelete];
  RELEASE(deleteOrUndelete);

  [message addItemWithTitle: _(@"Bounce...")
	   action:@selector(bounceMessage:)  
	   keyEquivalent: @""];

  // We create our 'filters' menu item / menu
  filters = [[NSMenu alloc] init];
  [message addItemWithTitle: _(@"Apply Manual Filters")
	   action: action
	   keyEquivalent: @""];
  [[message itemAtIndex: 9] setSubmenu: filters];
  RELEASE(filters);
  
  [[message itemAtIndex: 0] setTag: SHOW_ALL_HEADERS];
  
  // Our Compose menu / submenus
  compose = [[NSMenu alloc] init];
  [menu setSubmenu:compose forItem:[menu itemWithTitle:_(@"Compose")]];
  [compose addItemWithTitle: _(@"New")
	   action:@selector(composeMessage:) 
	   keyEquivalent:@"n"];
  [compose addItemWithTitle: _(@"Reply")
	   action:@selector(replyMessage:)  
	   keyEquivalent:@"R"];
  [compose addItemWithTitle: _(@"Reply All")
	   action:@selector(replyAllMessage:)   
	   keyEquivalent:@"E"];
  [compose addItemWithTitle: _(@"Forward")
	   action:@selector(forwardMessage:)  
	   keyEquivalent:@"W"];
  [compose addItemWithTitle: _(@"Addresses...")
	   action: @selector(showAddressBook:)
	   keyEquivalent:@"A"];
  [compose addItemWithTitle: _(@"Drafts")
	   action: action
	   keyEquivalent:@""];
  
  drafts = [[NSMenu alloc] init];
  
  saveInDrafts = [[NSMenuItem alloc] initWithTitle: _(@"Save in Drafts")
				     action: NULL
				     keyEquivalent: @"s"];
  [drafts addItem: saveInDrafts];
  RELEASE(saveInDrafts);
  
  [drafts addItemWithTitle: _(@"Restore Draft")
	  action: @selector(restoreDraft:)
	  keyEquivalent: @""];
  [[compose itemAtIndex: 5] setSubmenu: drafts];
  RELEASE(drafts);
  RELEASE(compose);
  
  // Our Edit menu / submenus
  edit = [[NSMenu alloc] init];
  [menu setSubmenu: edit forItem: [menu itemWithTitle: _(@"Edit")]];

  [edit addItemWithTitle: _(@"Cut")
	action: @selector(cut:)
	keyEquivalent: @"x"];
  [edit addItemWithTitle: _(@"Copy")
	action: @selector(copy:)
	keyEquivalent: @"c"];
  [edit addItemWithTitle: _(@"Paste")
	action: @selector(paste:)
	keyEquivalent: @"v"];
  [edit addItemWithTitle: _(@"Delete")
	action: @selector(delete:)
	keyEquivalent: @""];
  [edit addItemWithTitle: _(@"Undo")
	action: action
	keyEquivalent: @"z"];

  // Our find submenu and its submenus
  [edit addItemWithTitle: _(@"Find")
  	action: action
  	keyEquivalent: @""];

  [edit addItemWithTitle: _(@"Spelling...")
	action: @selector(checkSpelling:)
	keyEquivalent: @":"];

  [edit addItemWithTitle: _(@"Check Spelling")
	action: @selector(showGuessPanel:)
	keyEquivalent: @";"];

  [edit addItemWithTitle: _(@"Select All")
	action: @selector(selectAll:)
	keyEquivalent: @"a"];
  RELEASE(edit);
  
  find = [[NSMenu alloc] init];
  [[edit itemAtIndex: 5] setSubmenu:find];

  [find addItemWithTitle: _(@"Find Panel...")
	action: @selector(showFindWindow:)
	keyEquivalent: @"f"];
  [find addItemWithTitle: _(@"Find Next")
	action: @selector(findNext:)
	keyEquivalent: @"g"];
  [find addItemWithTitle: _(@"Find Previous")
	action: @selector(findPrevious:)
	keyEquivalent: @"d"];

  enterSelection = [[NSMenuItem alloc] initWithTitle: _(@"Enter Selection")
				       action: action
				       keyEquivalent: @"e"];
  [find addItem: enterSelection];
  RELEASE(enterSelection);

  RELEASE(find);

  // Our Windows menu
  windows = [[NSMenu alloc] init];
  [menu setSubmenu:windows forItem: [menu itemWithTitle:_(@"Windows")]];
  [windows addItemWithTitle: _(@"Arrange")
	   action: @selector(arrangeInFront:)
	   keyEquivalent: @""];
  [windows addItemWithTitle: _(@"Miniaturize")
	   action: @selector(performMiniaturize:)
	       keyEquivalent: @"m"];
  [windows addItemWithTitle: _(@"Close")
	   action: @selector(performClose:)
	   keyEquivalent: @"w"];
  RELEASE(windows);
  
  // Our Print menu / submenus
  print = [[NSMenu alloc] init];
  [menu setSubmenu: print forItem: [menu itemWithTitle: _(@"Print")]];
  [print addItemWithTitle: _(@"Print")
	 action: @selector(print:)
	 keyEquivalent: @"p"];
  [print addItemWithTitle: _(@"Setup Printer")
	 action:action
	 keyEquivalent: @"P"];
  RELEASE(print);

  // Our Services menu
  services = [[NSMenu alloc] init];
  [menu setSubmenu: services forItem: [menu itemWithTitle: _(@"Services")]];
  [NSApp setServicesMenu: services];
  RELEASE(services);

  [NSApp setMainMenu: menu];
  [NSApp setWindowsMenu: windows];

  [menu setTitle: @"GNUMail.app"];
  RELEASE(menu);
#endif
}


//
//
//
- (void) applicationDidFinishLaunching: (NSNotification *) not
{
  // Set up our mailbox locations
  if ( ! [[NSUserDefaults standardUserDefaults] objectForKey: @"LOCALMAILDIR"])
    {
      [[NSUserDefaults standardUserDefaults]
	setObject: [NSHomeDirectory() stringByAppendingPathComponent: @"gnumail"]
	forKey: @"LOCALMAILDIR"];
    }
  
  // Inbox
  if (! [[NSUserDefaults standardUserDefaults] objectForKey: @"INBOXFOLDERNAME"] )
    {
      [[NSUserDefaults standardUserDefaults] setObject: @"Inbox" forKey: @"INBOXFOLDERNAME"];
    }

  // Outbox
  if (! [[NSUserDefaults standardUserDefaults] objectForKey: @"OUTBOXFOLDERNAME"] )
    {
      [[NSUserDefaults standardUserDefaults] setObject: @"Outbox" forKey: @"OUTBOXFOLDERNAME"];
    }
  
  // Trash
  if (! [[NSUserDefaults standardUserDefaults] objectForKey: @"TRASHFOLDERNAME"] )
    {
      [[NSUserDefaults standardUserDefaults] setObject: @"Trash" forKey: @"TRASHFOLDERNAME"];
    }

  // Drafts
  if (! [[NSUserDefaults standardUserDefaults] objectForKey: @"DRAFTSFOLDERNAME"] )
    {
      [[NSUserDefaults standardUserDefaults] setObject: @"Drafts" forKey: @"DRAFTSFOLDERNAME"];
      
      // FIXME
      // In GNUMail.app 1.0.0pre3, we introduced the Drafts folder. If the key doesn't exist
      // that's because the folder doesn't exist either. We new create this folder. This
      // code SHOULD BE REMOVED for 1.0.0.
      [[NSFileManager defaultManager] createFileAtPath:
					[[[NSUserDefaults standardUserDefaults] objectForKey: @"LOCALMAILDIR"]
					  stringByAppendingPathComponent:
					    [[NSUserDefaults standardUserDefaults] 
					      objectForKey: @"DRAFTSFOLDERNAME"]]
				      contents: nil
				      attributes: nil];
    }

  // We remove deprecated preferences values
  [[NSUserDefaults standardUserDefaults] removeObjectForKey: @"SHOWICONS"];

  // We verify if GNUMail.app is using the old Personal preferences. If so, we
  // transparently convert the old preferences to the new model.
  // This code should be removed some time.
  if ( [[NSUserDefaults standardUserDefaults] objectForKey: @"NAME"] )
    {
      NSMutableDictionary *aMutableDictionary;

      aMutableDictionary = [[NSMutableDictionary alloc] init];

      // We add our new 'format' with the previous values
      [aMutableDictionary setObject: [[NSUserDefaults standardUserDefaults] objectForKey: @"NAME"]
			  forKey: @"NAME"];
      
      if ( [[NSUserDefaults standardUserDefaults] objectForKey: @"EMAILADDR"] )
	{
	  [aMutableDictionary setObject: [[NSUserDefaults standardUserDefaults] objectForKey: @"EMAILADDR"]
			      forKey: @"EMAILADDR"];
	}
      
      if ( [[NSUserDefaults standardUserDefaults] objectForKey: @"ORGANIZATION"] )
	{
	  [aMutableDictionary setObject: [[NSUserDefaults standardUserDefaults] objectForKey: @"ORGANIZATION"]
			      forKey: @"ORGANIZATION"];
	}
      
      if ( [[NSUserDefaults standardUserDefaults] objectForKey: @"REPLYTOADDR"] )
	{
	  [aMutableDictionary setObject: [[NSUserDefaults standardUserDefaults] objectForKey: @"REPLYTOADDR"]
			      forKey: @"REPLYTOADDR"];
	}

      // We add our new key
      [[NSUserDefaults standardUserDefaults] setObject: [NSDictionary dictionaryWithObject: aMutableDictionary
								      forKey: @"General"]
					     forKey: @"PERSONAL"];

      // We remove the previous values
      [[NSUserDefaults standardUserDefaults] removeObjectForKey: @"NAME"];
      [[NSUserDefaults standardUserDefaults] removeObjectForKey: @"EMAILADDR"];
      [[NSUserDefaults standardUserDefaults] removeObjectForKey: @"ORGANIZATION"];
      [[NSUserDefaults standardUserDefaults] removeObjectForKey: @"REPLYTOADDR"];
      
      // We synchronize our user defaults
      [[NSUserDefaults standardUserDefaults] synchronize];

      RELEASE(aMutableDictionary);
    }

  // We verify if GNUMail.app is using the old sending options for SMTP/Mailer. If so
  // We take the current settings and we convert them to the new format. When we are done,
  // we remove the old settings.
  if ( [[NSUserDefaults standardUserDefaults] objectForKey: @"DELIVERYMETHOD"] )
    {
      NSMutableDictionary *aMutableDictionary;

      aMutableDictionary = [[NSMutableDictionary alloc] init];
      
      // If we had a SMTP setting...
      if ( [[NSUserDefaults standardUserDefaults] integerForKey:@"DELIVERYMETHOD"] == 1 )
	{
	  [aMutableDictionary setObject: [NSNumber numberWithInt: TRANSPORT_SMTP]
			      forKey: @"TRANSPORT_METHOD"];
	  
	  [aMutableDictionary setObject: @""
			      forKey: @"MAILER_PATH"];
	  
	  [aMutableDictionary setObject: [[NSUserDefaults standardUserDefaults] objectForKey: @"SMTPSERVER"]
			      forKey: @"SMTP_HOST"];
	  
	  [aMutableDictionary setObject: [[NSUserDefaults standardUserDefaults] objectForKey: @"SMTPUSERNAME"]
			      forKey: @"SMTP_USERNAME"];
  
	  [aMutableDictionary setObject: @""
			      forKey: @"SMTP_PASSWORD"];
	}
      else
	{
	  [aMutableDictionary setObject: [NSNumber numberWithInt: TRANSPORT_MAILER]
			      forKey: @"TRANSPORT_METHOD"];
	  
	  [aMutableDictionary setObject: [[NSUserDefaults standardUserDefaults] objectForKey: @"SENDMAILPATH"]
			      forKey: @"MAILER_PATH"];
	  
	  [aMutableDictionary setObject: @""
			      forKey: @"SMTP_HOST"];
	  
	  [aMutableDictionary setObject: @""
			      forKey: @"SMTP_USERNAME"];
	  
	  [aMutableDictionary setObject: @""
			      forKey: @"SMTP_PASSWORD"];
 
	}
      
      // We assign a Generic profile name
      [aMutableDictionary setObject: @"General" forKey: @"PERSONAL_PROFILE"];
      
      [[NSUserDefaults standardUserDefaults] setObject: [NSArray arrayWithObject: aMutableDictionary]
					     forKey: @"SENDING"];

      // We remove our previous values
      [[NSUserDefaults standardUserDefaults] removeObjectForKey: @"DELIVERYMETHOD"];
      [[NSUserDefaults standardUserDefaults] removeObjectForKey: @"SENDMAILPATH"];
      [[NSUserDefaults standardUserDefaults] removeObjectForKey: @"SMTPSERVER"];
      [[NSUserDefaults standardUserDefaults] removeObjectForKey: @"SMTPUSERNAME"];
      
      // We synchronize our user defaults
      [[NSUserDefaults standardUserDefaults] synchronize];

      RELEASE(aMutableDictionary);
    }
  

  // We verify if GNUMail.app has been configured, if not, we suggest the user to do so
  if (! [[NSUserDefaults standardUserDefaults] objectForKey: @"PERSONAL"] )
    {
      NSMutableDictionary *aMutableDictionary;
      int choice;
      BOOL isDir;
      
      choice = NSRunAlertPanel(_(@"Important!"),
			       _(@"No configuration found for GNUMail.app"),
			       _(@"Configure"), // default
			       _(@"Cancel"),    // alternate
			       NULL);
      
      // If the user doesn't want to configure GNUMail.app, we simply quit 
      if (choice == NSAlertAlternateReturn) 
	{
	  [NSApp terminate: self];
	}
      
      // We show our colored icons first - The user could choose to hide them after 
      [[NSUserDefaults standardUserDefaults] setBool:YES forKey:@"SHOWICONS"];
      
      
      // We create the mailboxes directory for GNUMail.app, if we need to!
      if ( ! [[NSFileManager defaultManager]
	       fileExistsAtPath:[[NSUserDefaults standardUserDefaults] objectForKey: @"LOCALMAILDIR"]
	       isDirectory:&isDir] )
	{
	  [[NSFileManager defaultManager] createDirectoryAtPath:
					    [NSHomeDirectory() stringByAppendingPathComponent: @"gnumail"]
					  attributes: nil];
	}
      else
	{
	  // If it's a file
	  if (! isDir )
	    {
	      NSLog(@"%@ is a file and not a directory. Please remove that file before restarting GNUMail.app",
		    [[NSUserDefaults standardUserDefaults] objectForKey: @"LOCALMAILDIR"]);
	      [NSApp terminate: self];
	    }
	}
      
      //
      // We create the following Mailboxes automatically (only if we need to!):
      //
      // - Inbox  : for the newly received messages
      // - Outbox : for the messages that have been sent
      // - Trash  : for the messages we want to delete and transfer locally in IMAP
      // - Drafts : for un-sent messages
      //

      // Inbox
      if (! [[NSFileManager defaultManager] fileExistsAtPath:
					      [[[NSUserDefaults standardUserDefaults] objectForKey: @"LOCALMAILDIR"]
						stringByAppendingPathComponent:
						  [[NSUserDefaults standardUserDefaults] 
						    objectForKey: @"INBOXFOLDERNAME"]]] )
	{
	  [[NSFileManager defaultManager] createFileAtPath:
					    [[[NSUserDefaults standardUserDefaults] objectForKey: @"LOCALMAILDIR"]
					      stringByAppendingPathComponent:
						[[NSUserDefaults standardUserDefaults] 
						  objectForKey: @"INBOXFOLDERNAME"]]
					  contents: nil
					  attributes: nil];
	}
      
      
      // Outbox
      if (! [[NSFileManager defaultManager] fileExistsAtPath:
					      [[[NSUserDefaults standardUserDefaults] objectForKey: @"LOCALMAILDIR"]
						stringByAppendingPathComponent:
						  [[NSUserDefaults standardUserDefaults] 
						    objectForKey: @"OUTBOXFOLDERNAME"]]] )
	{
	  [[NSFileManager defaultManager] createFileAtPath:
					    [[[NSUserDefaults standardUserDefaults] objectForKey: @"LOCALMAILDIR"]
					      stringByAppendingPathComponent:
						[[NSUserDefaults standardUserDefaults] 
						  objectForKey: @"OUTBOXFOLDERNAME"]]
					  contents: nil
					  attributes: nil];
	}
      
      
      // Trash
      if (! [[NSFileManager defaultManager] fileExistsAtPath:
					      [[[NSUserDefaults standardUserDefaults] objectForKey: @"LOCALMAILDIR"]
						stringByAppendingPathComponent:
						  [[NSUserDefaults standardUserDefaults] 
						    objectForKey: @"TRASHFOLDERNAME"]]] )
	{
	  [[NSFileManager defaultManager] createFileAtPath:
					    [[[NSUserDefaults standardUserDefaults] objectForKey: @"LOCALMAILDIR"]
					      stringByAppendingPathComponent:
						[[NSUserDefaults standardUserDefaults] 
						  objectForKey: @"TRASHFOLDERNAME"]]
					  contents: nil
					  attributes: nil];
	}


      // Drafts
      if (! [[NSFileManager defaultManager] fileExistsAtPath:
					      [[[NSUserDefaults standardUserDefaults] objectForKey: @"LOCALMAILDIR"]
						stringByAppendingPathComponent:
						  [[NSUserDefaults standardUserDefaults] 
						    objectForKey: @"DRAFTSFOLDERNAME"]]] )
	{
	  [[NSFileManager defaultManager] createFileAtPath:
					    [[[NSUserDefaults standardUserDefaults] objectForKey: @"LOCALMAILDIR"]
					      stringByAppendingPathComponent:
						[[NSUserDefaults standardUserDefaults] 
						  objectForKey: @"DRAFTSFOLDERNAME"]]
					  contents: nil
					  attributes: nil];
	}
      

      // We create our 'default' personal setting with the new key (PERSONAL)
      aMutableDictionary = [[NSMutableDictionary alloc] init];
      [aMutableDictionary setObject: @"" forKey: @"NAME"];
      [aMutableDictionary setObject: @"" forKey: @"EMAILADDR"];
      [aMutableDictionary setObject: @"" forKey: @"ORGANIZATION"];
      [aMutableDictionary setObject: @"" forKey: @"REPLYTOADDR"];
      
      [[NSUserDefaults standardUserDefaults] setObject: [NSDictionary dictionaryWithObject: aMutableDictionary
								      forKey: @"General"]
					     forKey: @"PERSONAL"];
      
      RELEASE(aMutableDictionary);

      // We create the basic set of hidden headers
      [[NSUserDefaults standardUserDefaults] 
	setObject: [NSArray arrayWithObjects:
			      @"Date", @"From", @"To",
			    @"Cc", @"Subject",nil]
	forKey:@"SHOWNHEADERS"];
      
      [self showPreferencesWindow: nil];
      
    } // if (! [[NSUserDefaults standardUserDefaults] objectForKey: @"PERSONAL"] )
  
  // We initialize our store
  localStore = [[LocalStore alloc] initWithPathToDirectory:
				     [[NSUserDefaults standardUserDefaults] objectForKey: @"LOCALMAILDIR"]];
  
  if (! localStore )
    {
      NSLog(@"Could not open LocalStore.");
      printf("\n\nThat means that GNUMail.app is not properly configured.\n");
      printf("To fix this problem, do the following commands:\n\n");
      printf("%% mkdir ~/gnumail\n");
      printf("%% touch ~/gnumail/Inbox\n");
      printf("%% touch ~/gnumail/Outbox\n");
      printf("%% touch ~/gnumail/Trash\n\n");
      printf("and restart GNUMail.app after that!\n\n");
      
      [NSApp terminate: self];
    }

  // We verify if the SHOWWAITPANELS is present. If not, we initialize it to NSOffState
  if (! [[NSUserDefaults standardUserDefaults] objectForKey: @"SHOWWAITPANELS"] )
    {
      [[NSUserDefaults standardUserDefaults] setInteger: NSOffState
					     forKey: @"SHOWWAITPANELS"];
    }

  // Sync with the defaults file
  [[NSUserDefaults standardUserDefaults] synchronize];

  // We load all our bundles
  [self loadBundles];
  
  // We finally show our Inbox (or any other folders)
  [self showFoldersToOpen];

  // We register our service
  [NSApp setServicesProvider: self];

  // We set up our initial list of filters for the menu
  [self updateFilterMenuItems: nil];
}


//
// methods invoked by notifications
//
- (void) selectionInTextViewHasChanged: (id) sender
{
  NSRange aRange;

  // FIXME - NSOldSelectedCharacterRange isn't declared under OSX
#ifndef MACOSX
  aRange = [[[sender userInfo] objectForKey: NSOldSelectedCharacterRange] rangeValue];
#else
  aRange = [[[sender userInfo] objectForKey: @"NSOldSelectedCharacterRange"] rangeValue];
#endif

  if ( aRange.length )
    {
      [enterSelection setAction: @selector(enterSelectionInFindPanel:)];
    }
  else
    {
      [enterSelection setAction: NULL];
    }
}

- (void) selectionOfMessageHasChanged: (id) sender
{
  MailWindowController *mailWindowController;


  if ( [GNUMail lastMailWindowOnTop] )
    {
      mailWindowController = (MailWindowController *)[[GNUMail lastMailWindowOnTop] delegate];
      
      if ( mailWindowController )
    	{
	  // No matter how the selection has changed, we always remove all items from the
	  // "Send to Address Book" menu.
	  [self removeAllItemsFromMenu: sendToAddressBook];
	  
	  // We we are in a situation with multiple rows selected, we must 
	  // take the first message in our enumerator for flags comparaison.
	  if ( [[mailWindowController tableView] numberOfSelectedRows] > 1 )
	    {
	      NSEnumerator *anEnumerator;
	      Message *aMessage;
	      NSNumber *aRow;
	      
	      anEnumerator = [[mailWindowController tableView] selectedRowEnumerator];
	      aRow = [anEnumerator nextObject];
	      aMessage = [[[mailWindowController folder] allMessages] objectAtIndex: [aRow intValue]];
	      
	      [self updateMenuItemsForMessage: aMessage];
	    }
	  else
	    {
	      Message *aMessage;
	      int i;

	      aMessage = [mailWindowController selectedMessage];
	     
	      [self updateMenuItemsForMessage: aMessage];

	      // We finally build our 'Send to Address Book' menu	      
	      [sendToAddressBook addItemWithTitle: [[aMessage from] unicodeStringValue]
				 action: @selector(addToAddressBook:)
				 keyEquivalent: @""];
	      
	      for (i = 0; i < [aMessage recipientsCount]; i++)
		{
		  [sendToAddressBook addItemWithTitle: [[[aMessage recipients] objectAtIndex: i] unicodeStringValue]
				     action: @selector(addToAddressBook:)
				     keyEquivalent: @""];
		}
	    }
	} // if ( mailWindowController ) ...
    }
  
  // No matter what, we always reset our menu item for showing all headers
  // when this action method is called.
  if ( [showAllHeaders tag] == HIDE_ALL_HEADERS )
    {      
      [showAllHeaders setTitle: @"Show All Headers"];
      [showAllHeaders setTag: SHOW_ALL_HEADERS];
    }
}


//
// access / mutation methods
//
+ (NSArray *) allBundles
{
  return allBundles;
}

+ (NSArray *) allMailWindows
{
  return allMailWindows;
}

- (LocalStore *) localStore
{
  return localStore;
}

+ (NSString *) currentWorkingPath
{
  return currentWorkingPath;
}

+ (void) setCurrentWorkingPath: (NSString *) thePath
{
  RETAIN(thePath);
  RELEASE(currentWorkingPath);
  currentWorkingPath = thePath;
}

+ (id) lastAddressTakerWindowOnTop
{
  return lastAddressTakerWindowOnTop;
}

+ (void) setLastAddressTakerWindowOnTop: (id) aWindow
{
  lastAddressTakerWindowOnTop = aWindow;
}

+ (id) lastMailWindowOnTop
{ 
  return lastMailWindowOnTop;
}

+ (void) setLastMailWindowOnTop: (id) aWindow
{
  lastMailWindowOnTop = aWindow;

  if ( [NSApp delegate] && lastMailWindowOnTop )
    {
      [[NSApp delegate] updateShowOrHideDeletedMenuItem: [[[lastMailWindowOnTop windowController] folder] showDeleted]];
    }
}


- (NSMenu *) saveAttachmentMenu
{
  return saveAttachment;
}

- (void) setEnableSaveInDraftsMenuItem: (BOOL) aBOOL
{
  if ( aBOOL )
    {
      [saveInDrafts setAction: @selector(saveInDrafts:)];
    }
  else
    {
      [saveInDrafts setAction: NULL];
    }
}

//
// other methods
//
- (void) addItemToMenuFromTextAttachment: (NSTextAttachment *) theTextAttachment
{
  NSFileWrapper *aFileWrapper;
  ExtentedMenuItem *menuItem;

  aFileWrapper = [theTextAttachment fileWrapper];
 
  menuItem = [[ExtentedMenuItem alloc] initWithTitle: [aFileWrapper preferredFilename]
				       action: @selector(saveAttachment:)
				       keyEquivalent: @""];
  [menuItem setTextAttachment: theTextAttachment];

  [[self saveAttachmentMenu] addItem: menuItem];

  RELEASE(menuItem);
}


//
//
//
+ (void) addMailWindow: (id) theMailWindow
{
  if (allMailWindows && theMailWindow )
    {
      [allMailWindows addObject: theMailWindow];
    }
}


//
//
//
- (void) addToAddressBook: (id) sender
{
  InternetAddress *anInternetAddress;
  Address *anAddress;

  if (! [sender isKindOfClass: [NSMenuItem class]] )
    {
      return;
    }

  anInternetAddress = [[InternetAddress alloc] initWithString: [sender title]];
  
  if (! anInternetAddress )
    {
      NSLog(@"Could not add %@ to the Address Book.", [sender title]);
      return;
    }
  
  anAddress = [[Address alloc] init];  
  [anAddress setName: [anInternetAddress personal]];
  [anAddress setEmailAddress: [anInternetAddress address]];
  [[AddressBookController singleInstance] addToAddressBook: anAddress];

  RELEASE(anInternetAddress);
  RELEASE(anAddress);
}


//
// Method used to load all bundles in $GNUSTEP_USER_ROOT/Library/GNUMail
//
- (void) loadBundles
{
  NSFileManager *aFileManager;
  NSArray *allFiles;
  int i;


  aFileManager = [NSFileManager defaultManager];
  allFiles = [aFileManager directoryContentsAtPath: GNUMailUserLibraryPath()];

  for (i = 0; i < [allFiles count]; i++)
    {
      NSString *aString;
      
      aString = [allFiles objectAtIndex: i];
      
      // If we found a bundle, let's load it!
      if ( [[aString pathExtension] isEqualToString: @"bundle"] )
	{
	  id<GNUMailBundle> aModule;
	  NSBundle *aBundle;
	  NSString *path;

	  path = [NSString stringWithFormat: @"%@/%@",
			   GNUMailUserLibraryPath(),
			   aString];
	  
	  aBundle = [NSBundle bundleWithPath: path];

	  if ( aBundle )
	    {
	      aModule = [[aBundle principalClass] singleInstance];

	      if ( aModule )
		{
		  [aModule setOwner: self];
		  [allBundles addObject: aModule];
		  NSLog(@"Loaded bundle at path %@", path);
		}
	      else
		{
		  NSLog(@"Failed to initialized bundel at path %@", path);
		}
	    }
	  else
	    {
	      NSLog(@"Error loading bundle at path %@", path);
	    }
	}
    }
}


//
//
//
- (void) removeAllItemsFromMenu: (NSMenu *) theMenu
{
  int i;
  
  for (i = ([theMenu numberOfItems] - 1); i >= 0; i--)
    {
      [theMenu removeItemAtIndex: i];
    }
}


//
//
//
+ (void) removeMailWindow: (id) theMailWindow
{
  if (allMailWindows && theMailWindow )
    {
      [allMailWindows removeObject: theMailWindow];
    }
}


//
// This method is used to show the local inbox window after
// the startup of GNUMail.app or any other folder (for example,
// an IMAP folder).
//
- (void) showFoldersToOpen
{ 
  NSMutableArray *foldersToOpen;
  int i;
  
  if ( [[NSUserDefaults standardUserDefaults] objectForKey: @"FOLDERS_TO_OPEN"] )
    {
      foldersToOpen = [[NSUserDefaults standardUserDefaults] objectForKey: @"FOLDERS_TO_OPEN"];
    }
  else
    {
      foldersToOpen = [NSMutableArray array];
      [foldersToOpen addObject: [NSString stringWithFormat: @"local://%@/%@",
					  [[NSUserDefaults standardUserDefaults] objectForKey: @"LOCALMAILDIR"],
					  [[NSUserDefaults standardUserDefaults] objectForKey: @"INBOXFOLDERNAME"]]];

      [[NSUserDefaults standardUserDefaults] setObject: foldersToOpen
					     forKey: @"FOLDERS_TO_OPEN"];
      [[NSUserDefaults standardUserDefaults] synchronize];
    }
  
  
  for (i = 0; i < [foldersToOpen count]; i++)
    {
      URLName *urlName;

      urlName = [[URLName alloc] initWithString: [foldersToOpen objectAtIndex: i]];

      if ( [[urlName protocol] caseInsensitiveCompare: @"LOCAL"] == NSOrderedSame )
	{
	  [[MailboxManagerController singleInstance] openLocalFolderWithName: [urlName foldername]];
	}
      else if ( [[urlName protocol] caseInsensitiveCompare: @"IMAP"] == NSOrderedSame )
	{
	  [[MailboxManagerController singleInstance] openIMAPFolderWithName: [urlName foldername]
						     serverName: [urlName host]];
	}
      
      RELEASE(urlName);
    }
}


//
//
//
- (void) updateFilterMenuItems: (id) sender
{
  BOOL isDir;
  
  if ( [[NSFileManager defaultManager] fileExistsAtPath: PathToFilters()
				       isDirectory: &isDir] &&
       ! isDir )
    {
      FilterManager *aFilterManager;
      NSMenuItem *aMenuItem;
      int i;
  
      // We first remove all our items in the current menu
      [self removeAllItemsFromMenu: filters];
    
      aFilterManager = [FilterManager filtersFromDisk];
      RETAIN(aFilterManager);
      
      // Our "All" menu item
      aMenuItem = [[NSMenuItem alloc] initWithTitle: _(@"All")
				      action: NULL
				      keyEquivalent: @""];
      [aMenuItem setTag: -1];
      [filters addItem: aMenuItem];
      RELEASE(aMenuItem);

      for (i = 0; i < [[aFilterManager filters] count]; i++)
	{
	  Filter *aFilter;
	  
	  aFilter = [[aFilterManager filters] objectAtIndex: i];
	  
	  aMenuItem = [[NSMenuItem alloc] initWithTitle: [aFilter description]
					  action: NULL
					  keyEquivalent: @""];
	  [aMenuItem setTag: i];
	  [filters addItem: aMenuItem];
	  RELEASE(aMenuItem);
	}
    }
}

//
//
//
- (void) updateMenuItemsForMessage: (Message *) theMessage
{
  // Initial verification, we never know!
  if (! theMessage )
    {
      return;
    }
  
  // We verify if we need to delete or undelete our message and 
  // we set the tag to the right value
  if ( [[theMessage flags] contain: DELETED] )
    {
      [deleteOrUndelete setTitle: _(@"Undelete")];
      [deleteOrUndelete setTag: UNDELETE_MESSAGE];
    }
  else
    {
      [deleteOrUndelete setTitle: _(@"Delete")];
      [deleteOrUndelete setTag: DELETE_MESSAGE];
    }
  
  // We verify if we need to mark the message as read or unread
  if ( [[theMessage flags] contain: SEEN] )
    {
      [markAsReadOrUnread setTitle: _(@"Mark as Unread")];
		  [markAsReadOrUnread setTag: MARK_AS_UNREAD];
    }
  else
    {
      [markAsReadOrUnread setTitle: _(@"Mark as Read")];
      [markAsReadOrUnread setTag: MARK_AS_READ];
    }      
}


//
//
//
- (void) updateShowOrHideDeletedMenuItem: (BOOL) aBOOL
{
  if ( aBOOL )
    {
      [showOrHideDeleted setTitle: _(@"Hide Deleted")];
      [showOrHideDeleted setTag: HIDE_DELETED_MESSAGES];
    }
  else
    {
      [showOrHideDeleted setTitle: _(@"Show Deleted")];
      [showOrHideDeleted setTag: SHOW_DELETED_MESSAGES];
    }
}

//
// services methods
//
- (void) newMessageWithContent: (NSPasteboard *) pboard
	              userData: (NSString *) userData
                         error: (NSString **) error
{
  EditWindowController *editWindowController;
  NSString *aString;
  NSArray *allTypes;
  Message *aMessage;

  allTypes = [pboard types];

  if ( ![allTypes containsObject: NSStringPboardType] )
    {
      *error = @"No string type supplied on pasteboard";
      return;
    }
  
  aString = [pboard stringForType: NSStringPboardType];
  
  if (aString == nil)
    {
      *error = @"No string value supplied on pasteboard";
      return;
    }
  
  // We create a new message with our pasteboard content
  aMessage = [[Message alloc] init];
  [aMessage setContent: aString];

  // We create our controller and we show the window
  editWindowController = [[EditWindowController alloc] initWithWindowNibName: @"EditWindow"];

  if ( editWindowController )
    {
      [[editWindowController window] setTitle: _(@"New message...")];
      [editWindowController setMessage: aMessage];
      [editWindowController setShowCc: NO];
    
      [[editWindowController window] orderFrontRegardless];
    }
  
  RELEASE(aMessage);
}


- (void) newMessageWithRecipient: (NSPasteboard *) pboard
	                userData: (NSString *) userData
                           error: (NSString **) error
{
  EditWindowController *editWindowController;
  InternetAddress *anInternetAddress;
  NSString *aString;
  NSArray *allTypes;
  Message *aMessage;

  allTypes = [pboard types];

  if ( ![allTypes containsObject: NSStringPboardType] )
    {
      *error = @"No string type supplied on pasteboard";
      return;
    }
  
  aString = [pboard stringForType: NSStringPboardType];
  
  if (aString == nil)
    {
      *error = @"No string value supplied on pasteboard";
      return;
    }
  
  // We create a new message and we set the recipient
  aMessage = [[Message alloc] init];
  anInternetAddress = [[InternetAddress alloc] initWithString: aString];
  [anInternetAddress setType: TO];
  [aMessage addToRecipients: anInternetAddress];
  RELEASE(anInternetAddress);

  // We create our controller and we show the window
  editWindowController = [[EditWindowController alloc] initWithWindowNibName: @"EditWindow"];

  if ( editWindowController )
    {
      [[editWindowController window] setTitle: _(@"New message...")];
      [editWindowController setMessage: aMessage];
      [editWindowController setShowCc: NO];
    
      [[editWindowController window] orderFrontRegardless];
    }
  
  RELEASE(aMessage);
}

@end



//
// 
//

@implementation ExtentedMenuItem
{
  NSTextAttachment *textAttachment;
}

- (NSTextAttachment *) textAttachment
{
  return textAttachment;
}

- (void) setTextAttachment: (NSTextAttachment *) theTextAttachment
{
  textAttachment = theTextAttachment;
}

@end


//
// Starting point of GNUMail.app
//
int main(int argc, const char *argv[], char *env[])
{
  NSAutoreleasePool *pool;
  GNUMail *gnumail;

  pool = [[NSAutoreleasePool alloc] init];
  gnumail = [[GNUMail alloc] init];

  [NSApplication sharedApplication];
  [NSApp setDelegate: gnumail];

  NSApplicationMain(argc, argv);

  RELEASE(gnumail);
  RELEASE(pool);

  return 0;
}
