/*
**  ViewingViewController.m
**
**  Copyright (c) 2001, 2002
**
**  Author: Ludovic Marcotte <ludovic@Sophos.ca>
**
**  This program is free software; you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation; either version 2 of the License, or
**  (at your option) any later version.
**
**  This program is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with this program; if not, write to the Free Software
**  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#import "ViewingViewController.h"

#import "GNUMailConstants.h"
#import "NSStringExtensions.h"

#ifndef MACOSX
#import "ViewingView.h"
#endif

static ViewingViewController *singleInstance = nil;

@implementation ViewingViewController

- (id) initWithNibName: (NSString *) theName
{
  self = [super init];

#ifdef MACOSX
  
  if (! [NSBundle loadNibNamed: theName
		  owner: self] )
    {
      NSLog(@"Fatal error occured while loading the ViewingView nib file");
      AUTORELEASE(self);
      return nil;
    }

  RETAIN(view);

#else
  // We link our view
  view = [[ViewingView alloc] initWithParent: self];
  [view layoutView];

  // We link our outlets
  highlightURLs = [view highlightURLs];
  tableView = [view tableView];
  keyField = [view keyField];

  showAllHeaders = [view showAllHeaders];
  hideDeletedMessages = [view hideDeletedMessages];

  matrix = (NSMatrix *)[view matrix];
#endif

  // We initialize our array containing all our headers
  shownHeaders = [[NSMutableArray alloc] init];
    
  // We get our defaults for this panel
  [self initializeFromDefaults];

  return self;
}

- (void) dealloc
{
  singleInstance = nil;
  
  RELEASE(shownHeaders);
  RELEASE(view);

  [super dealloc];
}


//
// delegate methods
//


- (id)           tableView: (NSTableView *) aTableView
 objectValueForTableColumn: (NSTableColumn *) aTableColumn
                       row:(int) rowIndex
{
  return [shownHeaders objectAtIndex: rowIndex];
}


- (void) tableViewSelectionDidChange:(NSNotification *) aNotification
{
  if ([tableView selectedRow] >= 0)
    {
      [keyField setStringValue: [shownHeaders objectAtIndex: [tableView selectedRow]]];
    }
}

- (int) numberOfRowsInTableView: (NSTableView *) aTableView
{
  return [shownHeaders count];
}


- (void) tableView: (NSTableView *) aTableView
    setObjectValue: (id) anObject
    forTableColumn: (NSTableColumn *) aTableColumn
	       row: (int) rowIndex
{
  
  [shownHeaders replaceObjectAtIndex: rowIndex withObject: anObject];
}

//
// action methods
//

- (IBAction) addDefaults: (id) sender
{
  if (! [shownHeaders containsObject: @"Date"] )
    {
      [shownHeaders addObject: @"Date"];
    }
  
  if (! [shownHeaders containsObject: @"From"] )
    {
      [shownHeaders addObject: @"From"];
    }

  if (! [shownHeaders containsObject: @"To"] )
    {
      [shownHeaders addObject: @"To"];
    }

  if (! [shownHeaders containsObject: @"Cc"] )
    {
      [shownHeaders addObject: @"Cc"];
    }
  
  if (! [shownHeaders containsObject: @"Subject"] )
    {
      [shownHeaders addObject: @"Subject"];
    }

  [tableView reloadData];
  [tableView setNeedsDisplay:YES];
}

- (IBAction) addShown: (id) sender
{
  // We verify that the value isn't empty
  if ( [[[keyField stringValue] stringByTrimmingWhiteSpaces] length] > 0 )
    {
      if (! [shownHeaders containsObject: [[keyField stringValue] stringByTrimmingWhiteSpaces] ] )
	{
	  [shownHeaders addObject: [[keyField stringValue] stringByTrimmingWhiteSpaces] ];
	  [keyField setStringValue: @""];
	  [tableView reloadData];
	  [tableView setNeedsDisplay: YES];
	}
      else
	{
	  NSBeep();
	}
    }
  else
    {
      NSBeep();
    }
}

- (IBAction) removeShown: (id) sender
{
  if ([tableView selectedRow] >= 0)
    {
      id obj = [shownHeaders objectAtIndex: [tableView selectedRow]];
      
      if ( obj )
	{
	  [shownHeaders removeObject: obj];
	  [tableView reloadData];
	  [tableView setNeedsDisplay:YES];
	}
    }
  else
    {
      NSBeep();
    }
}


- (IBAction) moveUp: (id) sender
{
  int selectedRow = [tableView selectedRow];
  if (selectedRow <= 0)
    {
      NSBeep();
      return;
    }
  else
    {
      NSString *anHeader;
      
      anHeader = [shownHeaders objectAtIndex: selectedRow];
      [shownHeaders removeObjectAtIndex: selectedRow];
      [shownHeaders insertObject: anHeader atIndex: (selectedRow - 1)];
      [tableView reloadData];
      [tableView selectRow: (selectedRow - 1) byExtendingSelection: NO];
    }
}

- (IBAction) moveDown: (id) sender
{
  int selectedRow = [tableView selectedRow];
  if (selectedRow == ([shownHeaders count] - 1) || selectedRow < 0)
    {   
      NSBeep();
      return;
    }
  else
    {
      NSString *anHeader;

      anHeader = [shownHeaders objectAtIndex: selectedRow];
      [shownHeaders removeObjectAtIndex: selectedRow];
      [shownHeaders insertObject: anHeader atIndex: (selectedRow + 1)];
      [tableView reloadData];
      [tableView selectRow: (selectedRow + 1) byExtendingSelection: NO];
    }
}


//
// access methods
//

- (NSImage *) image
{
  NSBundle *aBundle;
  
  aBundle = [NSBundle bundleForClass: [self class]];
  
  return AUTORELEASE([[NSImage alloc] initWithContentsOfFile:
					[aBundle pathForResource: @"viewing" ofType: @"tiff"]]);
}

- (NSString *) name
{
  return _(@"Viewing");
}

- (NSView *) view
{
  return view;
}

- (BOOL) hasChangesPending
{
  return YES;
}


//
//
//
- (void) initializeFromDefaults
{
  NSArray *anArray, *shownTableColumns;
  int i;

  // Let's begin with our HIDE_DELETED_MESSAGES preference value
  if ( [[NSUserDefaults standardUserDefaults] objectForKey: @"HIDE_DELETED_MESSAGES"] )
    {
      [hideDeletedMessages setState: [[[NSUserDefaults standardUserDefaults] objectForKey: @"HIDE_DELETED_MESSAGES"] intValue] ];
    }

  // We decode our HIGHLIGHT_URL preference value
  if ( [[NSUserDefaults standardUserDefaults] objectForKey: @"HIGHLIGHT_URL"] )
    {
      [highlightURLs setState: [[[NSUserDefaults standardUserDefaults] objectForKey: @"HIGHLIGHT_URL"] intValue] ];
    }

  // We decode our SHOWALLHEADERS value
  if ( [[NSUserDefaults standardUserDefaults] objectForKey: @"SHOWALLHEADERS"] )
    {
      [showAllHeaders setState: [[[NSUserDefaults standardUserDefaults] objectForKey: @"SHOWALLHEADERS"] intValue] ];
    }
  else
    {
      [showAllHeaders setState: NSOffState];
    }

  // We now load the headers we want to show
  anArray = [[NSUserDefaults standardUserDefaults] objectForKey: @"SHOWNHEADERS"];
  
  if ( anArray )
    {
      [shownHeaders addObjectsFromArray: anArray];
      [tableView reloadData];
      [tableView setNeedsDisplay:YES];
    }

  // We now select the right items from our matrix
  shownTableColumns = [[NSUserDefaults standardUserDefaults] objectForKey: @"SHOWNTABLECOLUMNS"];
  
  if ( shownTableColumns )
    {
      for (i = 0; i < [shownTableColumns count]; i++)
	{
	  NSString *column;
	  
	  column = [shownTableColumns objectAtIndex: i];
	  
	  if ( [column isEqualToString: @"Status"] )
	    {
	      [[matrix cellAtRow: 0 column: 0] setState: NSOnState];
	    }
	  else if ( [column isEqualToString: @"Id"] )
	    {
	      [[matrix cellAtRow: 1 column: 0] setState: NSOnState];
	    }
	  else if ( [column isEqualToString: @"Date"] )
	    {
	      [[matrix cellAtRow: 2 column: 0] setState: NSOnState];
	    }
	  else if ( [column isEqualToString: @"From"] )
	    {
	      [[matrix cellAtRow: 0 column: 1] setState: NSOnState];
	    }
	  else if ( [column isEqualToString: @"Subject"] )
	    {
	      [[matrix cellAtRow: 1 column: 1] setState: NSOnState];
	    }
	  else if ( [column isEqualToString: @"Size"] )
	    {
	      [[matrix cellAtRow: 2 column: 1] setState: NSOnState];
	    }
	}
    }
  else
    {
      [matrix selectAll: nil];
    }
}


//
//
//
- (void) saveChanges
{
  NSMutableArray *aMutableArray;

  [[NSUserDefaults standardUserDefaults] setObject: [NSNumber numberWithInt: [hideDeletedMessages state]]
					 forKey: @"HIDE_DELETED_MESSAGES"];
  
  [[NSUserDefaults standardUserDefaults] setObject: [NSNumber numberWithInt: [highlightURLs state]]
					 forKey: @"HIGHLIGHT_URL"];
  
  [[NSUserDefaults standardUserDefaults] setObject: [NSNumber numberWithInt: [showAllHeaders state]]
					 forKey: @"SHOWALLHEADERS"];
  
  [[NSUserDefaults standardUserDefaults] setObject: shownHeaders
					 forKey: @"SHOWNHEADERS"];

  // We update our list of shown headers. The order is important.
  aMutableArray = [[NSMutableArray alloc] init];

  if ( [[matrix cellAtRow: 0 column: 0] state] == NSOnState )
    {
      [aMutableArray addObject: @"Status"];
    }

  if ( [[matrix cellAtRow: 1 column: 0] state] == NSOnState )
    {
      [aMutableArray addObject: @"Id"];
    }

  if ( [[matrix cellAtRow: 2 column: 0] state] == NSOnState )
    {
      [aMutableArray addObject: @"Date"];
    }
  
  if ( [[matrix cellAtRow: 0 column: 1] state] == NSOnState )
    {
      [aMutableArray addObject: @"From"];
    }
  
  if ( [[matrix cellAtRow: 1 column: 1] state] == NSOnState )
    {
      [aMutableArray addObject: @"Subject"];
    }
  
  if ( [[matrix cellAtRow: 2 column: 1] state] == NSOnState )
    {
      [aMutableArray addObject: @"Size"];
    }

  [[NSUserDefaults standardUserDefaults] setObject: aMutableArray
					 forKey: @"SHOWNTABLECOLUMNS"];

  RELEASE(aMutableArray);
}


//
// class methods
//
+ (id) singleInstance
{
  if (! singleInstance)
    {
      singleInstance = [[ViewingViewController alloc] initWithNibName: @"ViewingView"];
    }

  return singleInstance;
}

@end
