/*
**  ReceivingWindowController.m
**
**  Copyright (c) 2001, 2002
**
**  Author: Ludovic Marcotte <ludovic@Sophos.ca>
**
**  This program is free software; you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation; either version 2 of the License, or
**  (at your option) any later version.
**
**  This program is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with this program; if not, write to the Free Software
**  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#import "ReceivingWindowController.h"

#import "GNUMailConstants.h"
#import "IMAPView.h"
#import "POP3View.h"

#ifndef MACOSX
#import "ReceivingWindow.h"
#endif

#import "Utilities.h"

@implementation ReceivingWindowController

- (id) initWithWindowNibName: (NSString *) windowNibName
{
#ifdef MACOSX
  
  self = [super initWithWindowNibName: windowNibName];
  
  imapView = [[IMAPView alloc] init];
  [imapView initUI];

  pop3View = [[POP3View alloc] init];
  [pop3View initUI];

#else
  ReceivingWindow *receivingWindow;

  receivingWindow = [[ReceivingWindow alloc] initWithContentRect: NSMakeRect(300,300,400,280)
					     styleMask: NSTitledWindowMask
					     backing: NSBackingStoreRetained
					     defer: NO];

  [super initWithWindow: receivingWindow];

  [receivingWindow layoutWindow];
  [receivingWindow setDelegate: self];

  // We link our outlets
  tabView = [receivingWindow tabView];
  tabViewItem1 = [receivingWindow tabViewItem1];
  tabViewItem2 = [receivingWindow tabViewItem2];

  imapView = [receivingWindow imapView];
  pop3View = [receivingWindow pop3View];

  serverNameField = [receivingWindow serverNameField];
  serverPortField = [receivingWindow serverPortField];
  usernameField = [receivingWindow usernameField];
  passwordSecureField = [receivingWindow passwordSecureField];
  
  popup = [receivingWindow popup];
  rememberPassword = [receivingWindow rememberPassword];

  RELEASE(receivingWindow);
#endif

  [[self window] setTitle: _(@"Receiving preferences")];

  return self;
}


- (void) dealloc
{
  //NSLog(@"ReceivingWindowController: -dealloc");

#ifdef MACOSX
  RELEASE(imapView);
  RELEASE(pop3View);
#endif

  TEST_RELEASE(serverName);
  
  [super dealloc];
}


//
//
//
- (void) initializeFromDefaults
{
  NSDictionary *allValues;
  NSNumber *serverTypeValue, *portValue;
  
  // We now get all the data
  allValues = [[[NSUserDefaults standardUserDefaults] objectForKey:@"RECEIVING"]
		objectForKey: [self serverName]];
  
  // We decode our type
  serverTypeValue = [allValues objectForKey:@"SERVERTYPE"];
  
  if (serverTypeValue && [serverTypeValue intValue] == IMAP)
    {
      [popup selectItemWithTitle: _(@"IMAP")];
    }
  else
    {
      [popup selectItemWithTitle: _(@"POP3")];
    }
  
  // We decode our port
  portValue =  [allValues objectForKey:@"PORT"];
  
  if ( portValue )
    {
      [serverPortField setIntValue: [portValue intValue]];
    }
  else
    {
      if (serverTypeValue && [serverTypeValue intValue] == IMAP)
	{
	  [serverPortField setIntValue: 143];
	}
      else
	{
	  [serverPortField setIntValue: 110];
	}
    }

  // We refresh our view
  [self setType: nil];
  
  // We decode the rest of the informations. We begin with SERVERNAME
  if ( [self serverName] )
    {
      [serverNameField setStringValue: [self serverName] ];
    }
  else
    {
      [serverNameField setStringValue: @""];
    }

  // USERNAME
  if ( allValues && [allValues objectForKey:@"USERNAME"] )
    {
      [usernameField setStringValue: [allValues objectForKey:@"USERNAME"] ];
    }
  else
    {
      [usernameField setStringValue: @""];
    }

  // We get our password, if we need to!
  if ( allValues && [allValues objectForKey: @"PASSWORD"] )
    {
      NSString *password;

      password = [Utilities decryptPassword: [allValues objectForKey: @"PASSWORD"]
			    withKey: [self serverName]];
      
      if (! password )
	{
	  password = @"";
	}

      [passwordSecureField setStringValue: password];
    }
  else
    {
      [passwordSecureField setStringValue: @""];
    }
  
  // REMEMBERPASSWORD
  if ( allValues && [allValues objectForKey: @"REMEMBERPASSWORD"] )
    {
      [rememberPassword setState: [[allValues objectForKey: @"REMEMBERPASSWORD"] intValue]];
    }
  else
    {
      [rememberPassword setState: NSOffState];
    }

  // POP3 - LEAVEONSERVER
  if ( allValues && [allValues objectForKey:@"LEAVEONSERVER"] )
    {
      [[pop3View leaveOnServer] setState: [[allValues objectForKey: @"LEAVEONSERVER"] intValue]];
    }
  else
    {
      [[pop3View leaveOnServer] setState: NSOnState];
    }
  
  // POP3 - RETAINPERIOD
  if ( allValues && [allValues objectForKey:@"RETAINPERIOD"] )
    {
      [[pop3View daysField] setIntValue: [[allValues objectForKey: @"RETAINPERIOD"] intValue]];
    }
  else
    {
      [[pop3View daysField] setIntValue: 365];
    }
  
  // POP3 - USEAPOP
  if ( allValues && [allValues objectForKey:@"USEAPOP"] )
    {
      [[pop3View useAPOP] setState: [[allValues objectForKey:@"USEAPOP"] intValue]];
    }
  else
    {
      [[pop3View useAPOP] setState: NSOffState];
    }

  // IMAP - TRANSFERTOLOCALTRASH
  if ( allValues && [allValues objectForKey:@"TRANSFERTOLOCALTRASH"] )
    {
      [[imapView transferToLocalTrash] setState: [[allValues objectForKey: @"TRANSFERTOLOCALTRASH"] intValue]];
    }
  else
    {
      [[imapView transferToLocalTrash] setState: NSOffState];
    }
  
  // IMAP - RETRIEVEMINUTES
  if ( allValues && [allValues objectForKey:@"RETRIEVEMINUTES"] )
    {
      [[imapView retrieveMinutesField] setIntValue: [[allValues objectForKey: @"RETRIEVEMINUTES"] intValue]];
    }
  else
    {
      [[imapView retrieveMinutesField] setIntValue: 10];
    }
}

//
// delegate methods
//

- (void) windowWillClose: (NSNotification *) theNotification
{
  AUTORELEASE(self);
}

- (void) windowDidLoad
{
  // We select a default item in our combo
  [popup selectItemWithTitle: _(@"POP3")];

#ifdef MACOSX
  tabViewItem1 = [tabView tabViewItemAtIndex: 0];
  tabViewItem2 = [tabView tabViewItemAtIndex: 1];
#endif
  
  // We refresh our view
  [self setType: nil];
}

//
// action methods
//

- (IBAction) setType: (id) sender
{
  [popup synchronizeTitleAndSelectedItem];

  if ( [[popup titleOfSelectedItem] isEqualToString: _(@"POP3")] )
    {
      [tabViewItem2 setLabel: _(@"POP3")];
      [tabViewItem2 setView: pop3View];
      
      // If the number '143' was on the text field, we set it to 110. Else, we keep that custom port.
      if ( [serverPortField intValue] == 143 )
	{
	  [serverPortField setIntValue: 110];
	}
    }
  else
    {
      [tabViewItem2 setLabel: _(@"IMAP")];
      [tabViewItem2 setView: imapView];

      // If the number '110' was on the text field, we set it to 143. Else, we keep that custom port.
      if ( [serverPortField intValue] == 110 )
	{
	  [serverPortField setIntValue: 143];
	}
    }
  
  [tabView setNeedsDisplay: YES];
}

- (IBAction) okClicked: (id) sender
{
  NSMutableDictionary *allValues, *allServers;
  int value;

  allServers = [[NSMutableDictionary alloc] init];
  
  [allServers addEntriesFromDictionary: [[NSUserDefaults standardUserDefaults] objectForKey: @"RECEIVING"] ];
  
  // We fist verify if the server name has changed
  if ( ! [[serverNameField stringValue] isEqualToString: [self serverName]] )
    {
      // We don't try to remove it if it's not in there!
      if ( [allServers objectForKey: [self serverName]] )
	{
	  [allServers removeObjectForKey: [self serverName]];
	}
      [self setServerName: [serverNameField stringValue]];
    }
  
  // We init our dictionary of values
  allValues = [[NSMutableDictionary alloc] initWithCapacity: 8];

  // We set the type of our server and our port
  if ( [[popup titleOfSelectedItem] isEqualToString: _(@"POP3")] )
    {
      [allValues setObject: [NSNumber numberWithInt: POP3] forKey: @"SERVERTYPE"];
      
      if ([serverPortField intValue] > 0)
	{
	  [allValues setObject: [NSNumber numberWithInt: [serverPortField intValue]]
		     forKey: @"PORT"];
	}
      else
	{
	  [allValues setObject: [NSNumber numberWithInt: 110]
		     forKey: @"PORT"];
	}
    }
  else
    {
      [allValues setObject: [NSNumber numberWithInt: IMAP] forKey: @"SERVERTYPE"];
      
      if ([serverPortField intValue] > 0)
	{
	  [allValues setObject: [NSNumber numberWithInt: [serverPortField intValue]]
		     forKey: @"PORT"];
	}     
      else
	{
	  [allValues setObject: [NSNumber numberWithInt: 143]
		     forKey: @"PORT"];
	}
    }
  
  // We set the rest of the informations
  [allValues setObject: [usernameField stringValue] forKey:@"USERNAME"];

  // We save the password or we remove it from the defaults database
  if ( [rememberPassword state] == NSOnState )
    {
      [allValues setObject:
                   [Utilities encryptPassword: [passwordSecureField stringValue]
                              withKey: [self serverName]]
                 forKey:@"PASSWORD"];
    }
  else
    {
      // We don't try to remove it if it's not in there!
      if ( [allValues objectForKey: @"PASSWORD"] )
	{
	  [allValues removeObjectForKey: @"PASSWORD"];
	}
    }

  [allValues setObject: [NSNumber numberWithInt: [rememberPassword state]]
	     forKey: @"REMEMBERPASSWORD"];

  [allValues setObject: [NSNumber numberWithInt: [[pop3View leaveOnServer] state]] 
	     forKey: @"LEAVEONSERVER"];

  // Our POP3 retain period
  value = [[pop3View daysField] intValue];
  [allValues setObject: [NSNumber numberWithInt: (value <= 0 ? 365 : value)]
	     forKey: @"RETAINPERIOD"];

  [allValues setObject: [NSNumber numberWithInt: [[pop3View useAPOP] state]] 
	     forKey: @"USEAPOP"];

  [allValues setObject: [NSNumber numberWithInt: [[imapView transferToLocalTrash] state]] 
	     forKey: @"TRANSFERTOLOCALTRASH"];
  
  [allValues setObject: [NSNumber numberWithInt: [[imapView retrieveMinutesField] intValue]] 
	     forKey: @"RETRIEVEMINUTES"];
  
  [allServers setObject: allValues forKey: [self serverName]];
  [[NSUserDefaults standardUserDefaults] setObject: allServers forKey: @"RECEIVING"];
  [[NSUserDefaults standardUserDefaults] synchronize];

  RELEASE(allServers);
  RELEASE(allValues);

  [NSApp stopModal];
  [self close];
}


- (IBAction) cancelClicked: (id) sender
{
  [NSApp abortModal];
  [[self window] performClose: nil];
}


//
// access/mutation methods
//

- (NSString *) serverName
{
  return serverName;
}

- (void) setServerName: (NSString *) theName
{
  if (theName)
    {
      RETAIN(theName);
      RELEASE(serverName);
      serverName = theName;
    }
  else
    {
      RELEASE(serverName);
      serverName = nil;
      
      // We select a default item in our combo and we refresh our view
      [popup selectItemWithTitle: _(@"POP3")];
      [serverPortField setIntValue: 110];
      [self setType: nil];
    }
}


@end
