/*
**  ReceivingViewController.m
**
**  Copyright (c) 2001
**
**  Author: Ludovic Marcotte <ludovic@Sophos.ca>
**
**  This program is free software; you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation; either version 2 of the License, or
**  (at your option) any later version.
**
**  This program is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with this program; if not, write to the Free Software
**  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#import "ReceivingViewController.h"

#import "GNUMailConstants.h"
#import "ReceivingWindowController.h"

#ifndef MACOSX
#import "ReceivingView.h"
#endif

static ReceivingViewController *singleInstance = nil;

@implementation ReceivingViewController

- (id) initWithNibName: (NSString *) theName
{
  self = [super init];

#ifdef MACOSX
  
  if (! [NSBundle loadNibNamed: theName
		  owner: self] )
    {
      NSLog(@"Fatal error occured while loading the ReceivingView nib file");
      AUTORELEASE(self);
      return nil;
    }

  RETAIN(view);

  [tableView setTarget: self];
  [tableView setDoubleAction: @selector(edit:)];

#else
  // We link our view
  view = [[ReceivingView alloc] initWithParent: self];
  [view layoutView];

  // We link our outlets
  mailspoolFileField = [view mailspoolFileField];
  minutesField = [view minutesField];
  matrix = [(ReceivingView *)view matrix];
  
  mailspoolFileButton = [view mailspoolFileButton];
  transferFromMailspoolFile = [view transferFromMailspoolFile];

  compactMailboxButton = [view compactMailboxButton];
  showFilterPanelButton = [view showFilterPanelButton];

  tableView = [view tableView];

#endif
    
  // We get our defaults for this panel
  [self initializeFromDefaults];
  [self updateServersList];

  return self;
}

- (void) dealloc
{
  singleInstance = nil;

  RELEASE(allServers);
  RELEASE(view);

  [super dealloc];
}

//
// delegate methods
//

- (int) numberOfRowsInTableView: (NSTableView *)aTableView
{
  return [allServers count];
}

- (id)           tableView: (NSTableView *)aTableView
 objectValueForTableColumn: (NSTableColumn *)aTableColumn
                       row:(int)rowIndex
{
  NSDictionary *allValues;
  NSArray *anArray;
  NSNumber *aNumber;

  anArray = [allServers allKeys];

  allValues = [allServers objectForKey: [anArray objectAtIndex: rowIndex] ];
  
  aNumber = [allValues objectForKey: @"SERVERTYPE"];

  if (! aNumber )
    {
      aNumber = [NSNumber numberWithInt: POP3];
    }
  
  return [NSString stringWithFormat: @"%@ @ %@ [%@]",
		   [allValues objectForKey: @"USERNAME"],
		   [anArray objectAtIndex: rowIndex],
		   ([aNumber intValue] == POP3 ? _(@"POP3") : _(@"IMAP"))];
}


//
// action methods 
//

- (IBAction) add: (id) sender
{
  ReceivingWindowController *receivingWindowController;
  int result;
 
  receivingWindowController = [[ReceivingWindowController alloc] initWithWindowNibName: @"ReceivingWindow"];
  [receivingWindowController setServerName: nil];
  
  result = [NSApp runModalForWindow: [receivingWindowController window]];

  if (result == NSRunStoppedResponse)
    {
      [self updateServersList];
    }
}

- (IBAction) edit: (id) sender
{
  ReceivingWindowController *receivingWindowController;
  NSArray *anArray;
  int result;

  if ([tableView selectedRow] < 0)
    {
      NSBeep();
      return;
    }
  
  anArray = [allServers allKeys];

  receivingWindowController = [[ReceivingWindowController alloc] initWithWindowNibName: @"ReceivingWindow"];
  [receivingWindowController setServerName: [anArray objectAtIndex:[tableView selectedRow]] ];
  [receivingWindowController initializeFromDefaults];

  result = [NSApp runModalForWindow: [receivingWindowController window]];
  
  if (result == NSRunStoppedResponse)
    {
      [self updateServersList];
    }
}

- (IBAction) delete: (id) sender
{
  int choice;
  
  if ( [tableView selectedRow] < 0 ) 
    {
      NSBeep();
      return;
    }
  
  
  choice = NSRunAlertPanel(_(@"Delete..."),
			   _(@"Are you sure you want to delete this receiving server entry?"),
			   _(@"Cancel"), // default
			   _(@"Yes"),    // alternate
			   nil);

  if (choice == NSAlertAlternateReturn)
    {
      // We remove the entry from the user defaults db 
      NSArray *anArray;
      
      anArray = [allServers allKeys];
      [allServers removeObjectForKey: [anArray objectAtIndex: [tableView selectedRow]] ];
      
      // We synchronize with our user defaults
      [[NSUserDefaults standardUserDefaults] setObject: allServers forKey: @"RECEIVING"];
      [[NSUserDefaults standardUserDefaults] synchronize];
      
      [self updateServersList];
      
      if ([allServers count] > 0) 
	{
	  [tableView selectRow:0 byExtendingSelection:NO];
	}
    }
}

- (IBAction) setEnabledFields: (id) sender
{
  if ( [transferFromMailspoolFile state] == NSOnState )
    {
      [mailspoolFileField setEditable: YES];
      [mailspoolFileButton setEnabled: YES];
    }
  else
    {
      [mailspoolFileField setEditable: NO];
      [mailspoolFileButton setEnabled: NO];
    }
}

- (IBAction) setManually: (id) sender
{
  [minutesField setEditable: NO];
}

- (IBAction) setAutomatically: (id) sender
{
  [minutesField setEditable: YES];
}

- (IBAction) getMailspoolFile: (id) sender
{
  NSArray *fileToOpen;
  NSOpenPanel *oPanel;
  NSString *fileName;
  int count, result;
  
  oPanel = [NSOpenPanel openPanel];
  [oPanel setAllowsMultipleSelection:NO];
  result = [oPanel runModalForDirectory:NSHomeDirectory() file:nil types:nil];
  
  if (result == NSOKButton) {
    
    fileToOpen = [oPanel filenames];
    count = [fileToOpen count];
    
    if (count > 0) {
      fileName = [fileToOpen objectAtIndex:0];
      [mailspoolFileField setStringValue:fileName];
    }
  }
}


//
// access methods
//

- (NSImage *) image
{
  NSBundle *aBundle;
  
  aBundle = [NSBundle bundleForClass: [self class]];
  
  return AUTORELEASE([[NSImage alloc] initWithContentsOfFile:
					[aBundle pathForResource: @"MailIcon_retrieve" ofType: @"tiff"]]);
}

- (NSString *) name
{
  return _(@"Receiving");
}

- (NSView *) view
{
  return view;
}

- (BOOL) hasChangesPending
{
  return YES;
}


//
//
//
- (void) initializeFromDefaults
{
  NSString *mailspoolFile;
  int retrieveMethod, retrieveMinutes;
  int shouldTransferFromMailspoolFile;
    
  retrieveMethod = [[NSUserDefaults standardUserDefaults] integerForKey:@"RETRIEVEMETHOD"];
  retrieveMinutes = [[NSUserDefaults standardUserDefaults] integerForKey:@"RETRIEVEMINUTES"];
  
  // RETRIEVEMETHOD
  if ( retrieveMethod )
    {
      [matrix selectCellAtRow: 1 column: 0];
      [minutesField setEditable: YES];
    }
  else
    {
      [matrix selectCellAtRow: 0 column: 0];
      [minutesField setEditable: NO];
    }
  
  // RETRIEVEMINUTES
  if ( retrieveMinutes )
    {
      [minutesField setIntValue: retrieveMinutes];
    }
  else
    {
      [minutesField setStringValue: @"1"];
    }
  
  // MAILSPOOLFILE
  shouldTransferFromMailspoolFile = [[NSUserDefaults standardUserDefaults] 
				      integerForKey: @"TRANSFERFROMMAILSPOOLFILE"];

  if (shouldTransferFromMailspoolFile == NSOnState)
    {
      [transferFromMailspoolFile setState: NSOnState];
      [mailspoolFileField setEditable: YES];
      [mailspoolFileButton setEnabled: YES];
    }
  else
    {
      [transferFromMailspoolFile setState: NSOffState];
      [mailspoolFileField setEditable: NO];
      [mailspoolFileButton setEnabled: NO];
    }

  mailspoolFile = [[NSUserDefaults standardUserDefaults] stringForKey:@"MAILSPOOLFILE"];

  if (mailspoolFile && ([mailspoolFile length] > 0))
    {
      [mailspoolFileField setStringValue: mailspoolFile];
    }
  else
    {
      NSProcessInfo *processInfo;
      
      processInfo = [NSProcessInfo processInfo];
      mailspoolFile = [[processInfo environment] objectForKey: @"MAIL"];

      if ( mailspoolFile )
	{
	  [mailspoolFileField setStringValue: mailspoolFile];
	}
      else
	{
	  [mailspoolFileField setStringValue: @""];
	}
      
    }
  
  // If the user HAS NOT specified one or the MAIL environment variable isn't defined,
  // we try to guess the best path to the mail spool file a user would use.
  if ( [[mailspoolFileField stringValue] isEqualToString: @""] )
    {
      BOOL isDir;
      
      if ( [[NSFileManager defaultManager] fileExistsAtPath: [NSString stringWithFormat: @"/var/mail/%@", 
								       NSUserName()]
					   isDirectory: &isDir] && !isDir )
        {
          [mailspoolFileField setStringValue: [NSString stringWithFormat: @"/var/mail/%@", NSUserName()]];
        }
      else if ( [[NSFileManager defaultManager] fileExistsAtPath: [NSString stringWithFormat: @"/var/spool/mail/%@",
									    NSUserName()]
						isDirectory: &isDir] && ! isDir )
        {
          [mailspoolFileField setStringValue: [NSString stringWithFormat: @"/var/spool/mail/%@", NSUserName()]];
        }
      else if ( [[NSFileManager defaultManager] fileExistsAtPath: [NSString stringWithFormat: @"/usr/spool/mail/%@",
									    NSUserName()]
						isDirectory: &isDir] && ! isDir )
        {
          [mailspoolFileField setStringValue: [NSString stringWithFormat: @"/usr/spool/mail/%@", NSUserName()]];
        }
    }
  
  // COMPACT_ON_CLOSE
  if ( [[NSUserDefaults standardUserDefaults] objectForKey: @"COMPACT_MAILBOX_ON_CLOSE"] )
    {
      [compactMailboxButton setState: [[NSUserDefaults standardUserDefaults] 
					integerForKey: @"COMPACT_MAILBOX_ON_CLOSE"]];
    }
  else
    {
      [compactMailboxButton setState: NSOffState];
    }
  
  // SHOW_FILTER_PANEL
  if ( [[NSUserDefaults standardUserDefaults] objectForKey: @"SHOW_FILTER_PANEL"] )
    {
      [showFilterPanelButton setState: [[NSUserDefaults standardUserDefaults] 
					 integerForKey: @"SHOW_FILTER_PANEL"]];
    }
  else
    {
      [showFilterPanelButton setState: NSOnState];
    }
}


- (void) saveChanges
{
  // We compare the retrieiving method we had and the one we are about the save
  // to know if we must start or stop our timer in the local inbox window controller
  if ([[NSUserDefaults standardUserDefaults] 
	integerForKey:@"RETRIEVEMETHOD"] != [matrix selectedRow])
    {
      [[NSUserDefaults standardUserDefaults] setInteger: [minutesField intValue]
					     forKey: @"RETRIEVEMINUTES"];
      
      if ( [matrix selectedRow] )
	{
	  // We inform our LocalInboxWindowController to start the timer
	  NSLog(@"Starting the timer!");
	  
	  [[NSNotificationCenter defaultCenter]
	    postNotificationName: ShouldStartLocalInboxTimer
	    object: nil
	    userInfo: nil];
	}
      else
	{
	  // We inform our LocalInboxWindowController to stop the timer
	  NSLog(@"Stopping the timer!");

	  [[NSNotificationCenter defaultCenter]
	    postNotificationName: ShouldStopLocalInboxTimer
	    object: nil
	    userInfo: nil];
	}
    } 
  else if (([matrix selectedRow] == 1) 
	   && ([[NSUserDefaults standardUserDefaults] 
		 integerForKey:@"RETRIEVEMINUTES"] != [minutesField intValue]))
    {
      NSLog(@"Restarting the timer!");
      [[NSUserDefaults standardUserDefaults] setInteger: [minutesField intValue]
					     forKey: @"RETRIEVEMINUTES"];

      [[NSNotificationCenter defaultCenter]
	postNotificationName: ShouldStopLocalInboxTimer
	object: nil
	userInfo: nil];
      [[NSNotificationCenter defaultCenter]
	postNotificationName: ShouldStartLocalInboxTimer
	object: nil
	userInfo: nil];
    }
  else
    {  
      [[NSUserDefaults standardUserDefaults] setInteger: [minutesField intValue]
					     forKey: @"RETRIEVEMINUTES"];
    }
  
  [[NSUserDefaults standardUserDefaults] setInteger: [matrix selectedRow]
					 forKey: @"RETRIEVEMETHOD"];

  [[NSUserDefaults standardUserDefaults] setInteger: [transferFromMailspoolFile state]
					 forKey: @"TRANSFERFROMMAILSPOOLFILE"];
  
  [[NSUserDefaults standardUserDefaults] setObject: [mailspoolFileField stringValue] 
					 forKey: @"MAILSPOOLFILE"];
  
  [[NSUserDefaults standardUserDefaults] setInteger: [compactMailboxButton state]
					 forKey: @"COMPACT_MAILBOX_ON_CLOSE"];

  [[NSUserDefaults standardUserDefaults] setInteger: [showFilterPanelButton state]
					 forKey: @"SHOW_FILTER_PANEL"];

  // We inform that our receiving values have changed
  [[NSNotificationCenter defaultCenter]
    postNotificationName: ReceivingValuesHaveChanged
    object: nil
    userInfo: nil];
}


//
// other methods
//
- (void) updateServersList
{
  TEST_RELEASE(allServers);

  allServers = [[NSMutableDictionary alloc] init];

  if ( [[NSUserDefaults standardUserDefaults] objectForKey: @"RECEIVING"] )
    {
      [allServers addEntriesFromDictionary:
        [[NSUserDefaults standardUserDefaults] objectForKey: @"RECEIVING"] ];
    }

  // We synchronize with our user defaults
  [[NSUserDefaults standardUserDefaults] setObject: allServers forKey: @"RECEIVING"];
  [[NSUserDefaults standardUserDefaults] synchronize];
  
  [tableView reloadData];
}


//
// class methods
//
+ (id) singleInstance
{
  if (! singleInstance)
    {
      singleInstance = [[ReceivingViewController alloc] initWithNibName: @"ReceivingView"];
    }

  return singleInstance;
}

@end
