/*
**  MailcapParser.m
**
**  Copyright (c) 2001
**
**  Author: Vincent Ricard <vricard@wanadoo.fr>
**          Ludovic Marcotte <ludovic@Sophos.ca>
**
**  This program is free software; you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation; either version 2 of the License, or
**  (at your option) any later version.
**
**  This program is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with this program; if not, write to the Free Software
**  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#import "MailcapParser.h"

#import "GNUMailConstants.h"
#import "NSStringExtensions.h"

#import <Foundation/NSString.h>
#import <Foundation/NSArray.h>

@implementation MailcapParser

//
//
//
- (id) init
{
  self = [super init];

  return self;
}

//
//
//
- (void) dealloc
{
  [super dealloc];
}


//
//
//
- (NSArray *) parseFile: (NSString *) theMailcapFile
{
  NSString *stringContentOfFile, *aLine;
  NSArray *linesOfMailcapFile;
  NSMutableString *aMutableString;
  NSMutableArray *allMimeTypes;

  BOOL multiline;
  int i, j, countBS;

  // Opening the mailcap file
  stringContentOfFile = [[NSString alloc] initWithContentsOfFile: theMailcapFile];

  if (! [stringContentOfFile length] )
    {
      // Empty file, we return an empty array
      return [NSArray array];
    }

  allMimeTypes = [[NSMutableArray alloc] init];

  // Grammar of a mailcap file (rfc1524, p. 3):
  // Mailcap-File = *Mailcap-Line
  // Mailcap-Line = Comment / Mailcap-Entry
  // Comment = NEWLINE  /  "#" *CHAR NEWLINE
  // NEWLINE = <newline as defined by OS convention>

  // For the moment NEWLINE = '\n'
  linesOfMailcapFile = [stringContentOfFile componentsSeparatedByString: @"\n"];

  aMutableString = [[NSMutableString alloc] initWithString: @""];

  multiline = NO;
  for ( i = 0; i < [linesOfMailcapFile count]; i++ )
    {
      aLine = [linesOfMailcapFile objectAtIndex: i];
      if (! [aLine hasPrefix: @"#"] )
        {
          // It's not a comment
          if ( 0 != [aLine length] )
            {
              // It's not a NEWLINE

              // How many final '\\'
              for ( j = [aLine length], countBS = 0; 0 <= j && [[aLine substringToIndex: j] hasSuffix: @"\\"]; j-- )
                {
                  countBS++;
                }

              if ( countBS & 1 )
                {
                  // Multiline detected
                  multiline = YES;

                  // Append the line without '\\'
                  [aMutableString appendString: [aLine substringToIndex: [aLine length] - 1]];
                }
              else
                {
                  // Parse of the mailcap entry
                  if ( multiline )
                    {
		      MimeType *aMimeType = nil;

                      [aMutableString appendString: aLine];
                      aMimeType = [self parseEntry: aMutableString];

		      if ( aMimeType )
			{
			  [allMimeTypes addObject: aMimeType];
			}

                      // Reset
                      [aMutableString setString: @""];

                      multiline = NO;
                    }
                  else
                    {
		      MimeType *aMimeType = [self parseEntry: aLine];

		      if ( aMimeType )
			{
			  [allMimeTypes addObject: aMimeType];
			}
                    }
                }
            }
        }
    }
  
  RELEASE(stringContentOfFile);
  RELEASE(aMutableString);

  return AUTORELEASE(allMimeTypes);
}

- (MimeType *) parseEntry: (NSString *) theMailcapEntry
{
  NSRange found;
  NSString * typefield, * commandAndOptions;
  unsigned int i, s;
  unichar backSlashChar;
  unichar semiColonChar;
  MimeType *aMimeType;

  // Grammar of mailcap entry (rfc1524, p. 6)
  // Mailcap-Entry = typefield ; view-command
  //                     [";" 1#field]
  // typefield = propertype / implicit-wild
  // propertype = type "/" wildsubtype
  // implicitwild = type
  // wildsubtype = subtype / "*"
  // view-command = mtext
  // mtext = *mchar
  // mchar = schar / qchar
  // schar = * <any CHAR except ";","\", and CTLS>
  // qchar = "\" CHAR ; may quote any char
  // field = flag / namedfield
  // namedfield = fieldname "=" mtext
  // flag = "needsterminal"   ; All these literals are to
  //      / "copiousoutput"   ; be interpreted as
  //      / x-token           ; case-insensitive
  // fieldname =    / "compose"      ;Also all of these
  //                / "composetyped" ;are case-insensitive.
  //                / "print"
  //                / "edit"
  //                / "test"
  //                / "x11-bitmap"
  //                / "textualnewlines"
  //                / "description"
  //                / x-token

  found = [theMailcapEntry rangeOfString: @";"];

  if ( 0 == found.length )
    {
      // Invalid entry: no separator between typefield and view-command
      return nil;
    }
  else if ( 0 == found.location )
    {
      // Invalid entry: the first character is ';'
      return nil;
    }

  aMimeType = [[MimeType alloc] init];

  // We found the typefield
  typefield = [[theMailcapEntry substringWithRange: NSMakeRange(0, found.location)] lowercaseString];
  found = NSMakeRange(found.location + 1, [theMailcapEntry length] - found.location - 1);
  commandAndOptions = [theMailcapEntry substringWithRange: found];

  found = [typefield rangeOfString: @"/"];
  if ( 0 == found.length )
    {
      // No separator, so typefield is an implicit-wild
      if ( ! [self checkTypeField: typefield] )
        {
          // Invalid type
          return nil;
        }

      [aMimeType setMimeType: [NSString stringWithFormat: @"%@/%@", typefield, @"*"]];
    }
  else
    {
      if ( ! [self checkTypeField: [typefield substringWithRange: NSMakeRange(0, found.location)]] )
        {
          // Invalid type
          return nil;
        }
      found = NSMakeRange(found.location + 1, [typefield length] - found.location - 1);
      if ( ! [self checkTypeField: [typefield substringWithRange: found]] )
        {
          // Invalid subtype
          return nil;
        }

      [aMimeType setMimeType: typefield];
    }

  // We look for a ';' without '\'
  backSlashChar = '\\';
  semiColonChar = ';';
  s = 0;
  for ( i = 0; i < [commandAndOptions length]; )
    {
      if ( backSlashChar == [commandAndOptions characterAtIndex: i] )
        {
          // The next character is quoted
          i += 2;
          continue;
        }

      if ( semiColonChar == [commandAndOptions characterAtIndex: i] )
        {
          // We found a separator

	  if ( 0 == s )
	    {
	      // It's the command (not an option)
	      NSString * command = [[commandAndOptions substringWithRange: NSMakeRange(s, i - s)] stringByTrimmingWhiteSpaces];
	      [aMimeType setDataHandlerCommand: command];
            }
          else
            {
	      // It's an option
	      NSString * option = [[commandAndOptions substringWithRange: NSMakeRange(s, i - s)] stringByTrimmingWhiteSpaces];

              // Needsterminal ?
              if ( [option caseInsensitiveCompare: @"needsterminal" ] == NSOrderedSame)
                {
                  [aMimeType setNeedsTerminal: YES];
                }

              // Description ?
              found = [option rangeOfString: @"description=" options: NSCaseInsensitiveSearch];
              if ( (0 == found.location) && (12 == found.length) )
                {
                  found = NSMakeRange(found.length, [option length] - found.length);
                  [aMimeType setDescription: [option substringWithRange: found]];
                }
            }

          s = i + 1;
        }
      i++;
    }

  if ( 0 == s )
    {
      // It's the command (not an option)
      NSString * command = [[commandAndOptions substringWithRange: NSMakeRange(s, i - s)] stringByTrimmingWhiteSpaces];
      [aMimeType setDataHandlerCommand: command];
    }
  else
    {
      // It's an option
      NSString * option = [[commandAndOptions substringWithRange: NSMakeRange(s, i - s)] stringByTrimmingWhiteSpaces];

      // Needsterminal ?
      if ( NSOrderedSame == [option caseInsensitiveCompare: @"needsterminal" ] )
	{
	  [aMimeType setNeedsTerminal: YES];
	}

      // Description ?
      found = [option rangeOfString: @"description=" options: NSCaseInsensitiveSearch];
      if ( (0 == found.location) && (12 == found.length) )
        {
          found = NSMakeRange(found.length, [option length] - found.length);
          [aMimeType setDescription: [option substringWithRange: found]];
        }
    }

  return AUTORELEASE(aMimeType);
}


//
//
//
- (BOOL) checkTypeField: (NSString *) theTypeField
{
  NSCharacterSet *CTLsSet;
  NSCharacterSet *otherSet;
  NSRange aRange;

  // rfc1521
  // token = 1*<any (ASCII) CHAR except SPACE, CTLs,
  //                   or tspecials>
  // tspecials :=  "(" / ")" / "<" / ">" / "@"
  //                /  "," / ";" / ":" / "\" / <">
  //                /  "/" / "[" / "]" / "?" / "="

  // CTLs and DEL
  CTLsSet = [NSCharacterSet controlCharacterSet];

  // SPACE and tspecials
  otherSet = [NSCharacterSet characterSetWithCharactersInString: @" ()<>@,;:\\\"/[]?="];

  aRange = [theTypeField rangeOfCharacterFromSet: CTLsSet];
  if ( 0 != aRange.length )
    {
      // forbidden character detected
      return NO;
    }
  aRange = [theTypeField rangeOfCharacterFromSet: otherSet];
  if ( 0 != aRange.length )
    {
      // forbidden character detected
      return NO;
    }

  return YES;
}

@end
