/*
**  FilterEditorWindowController.m
**
**  Copyright (c) 2001
**
**  Author: Ludovic Marcotte <ludovic@Sophos.ca>
**
**  This program is free software; you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation; either version 2 of the License, or
**  (at your option) any later version.
**
**  This program is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with this program; if not, write to the Free Software
**  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#import "FilterEditorWindowController.h"

#import "../../Filter.h"
#import "FilterHeaderEditorWindowController.h"
#import "FilterManager.h"
#import "FilterMessageWindowController.h"
#import "GNUMail.h"
#import "GNUMailConstants.h"

#import <Pantomime/LocalStore.h>

#ifndef MACOSX
#import "FilterEditorWindow.h"
#endif


@implementation FilterEditorWindowController

//
//
//
- (id) initWithWindowNibName: (NSString *) windowNibName
{
#ifdef MACOSX
  
  self = [super initWithWindowNibName: windowNibName];
  
#else
  FilterEditorWindow *filterEditorWindow;
  
  filterEditorWindow = [[FilterEditorWindow alloc] initWithContentRect: NSMakeRect(300,300,500,345)
						   styleMask: NSTitledWindowMask
						   backing: NSBackingStoreRetained
						   defer: NO];
  
  self = [super initWithWindow: filterEditorWindow];
  
  [filterEditorWindow layoutWindow];
  [filterEditorWindow setDelegate: self];

  // We link our outlets
  descriptionField = [filterEditorWindow descriptionField];
  activeButton = [filterEditorWindow activeButton];
  criteriaSourcePopUpButton = [filterEditorWindow criteriaSourcePopUpButton];
  criteriaFindOperationPopUpButton = [filterEditorWindow criteriaFindOperationPopUpButton];
  criteriaStringField = [filterEditorWindow criteriaStringField];
  matrix = [filterEditorWindow matrix];
  actionColorPopUpButton = [filterEditorWindow actionColorPopUpButton]; 
  actionFolderNamePopUpButton = [filterEditorWindow actionFolderNamePopUpButton];
  actionEMailStringPopUpButton = [filterEditorWindow actionEMailStringPopUpButton];
  actionEMailStringField = [filterEditorWindow actionEMailStringField];
  actionEMailStringButton = [filterEditorWindow actionEMailStringButton];

  RELEASE(filterEditorWindow);
#endif

  [[self window] setTitle: _(@"Add a filter")];
  [self _addItemsToFolderNamePopUpButton];
  
  return self;
}


- (void) dealloc
{
  NSLog(@"FilterEditorWindowController: -dealloc");
  
  [super dealloc];
}

//
// delegate methods
//

- (void) windowWillClose: (NSNotification *) theNotification
{
  //NSLog(@"FilterEditorWindowController: -windowWillClose");
  
  RELEASE(filterManager);
  TEST_RELEASE(filter);

  AUTORELEASE(self);
}

- (void) windowDidLoad
{
  [super windowDidLoad];
}


//
// action methods
//
- (IBAction) okClicked: (id) sender
{
  Filter *aFilter;

  // We first synchronize all our popups
  [criteriaSourcePopUpButton synchronizeTitleAndSelectedItem];
  [criteriaFindOperationPopUpButton synchronizeTitleAndSelectedItem];
  [actionColorPopUpButton synchronizeTitleAndSelectedItem];
  [actionFolderNamePopUpButton synchronizeTitleAndSelectedItem];
  [actionEMailStringPopUpButton synchronizeTitleAndSelectedItem];

  if ( [self filter] )
    {
      aFilter = [self filter];
    }
  else
    {
      aFilter = [[Filter alloc] init];
      
      [aFilter setDescription: [descriptionField stringValue]];
      
      [filterManager addFilter: aFilter];
      RELEASE(aFilter);
    }
  
  // Description
  [aFilter setDescription: [descriptionField stringValue]];
  [aFilter setIsActive: ([activeButton state] == NSOnState ? YES : NO)];
  
  // Criteria
  [aFilter setCriteriaSource: ([criteriaSourcePopUpButton indexOfSelectedItem] + 1)];
  [aFilter setCriteriaFindOperation: ([criteriaFindOperationPopUpButton indexOfSelectedItem] + 1)];
  [aFilter setCriteriaString: [criteriaStringField stringValue]];
  
  // Action
  [aFilter setAction: ([matrix selectedRow] + 1)];

  switch ( [actionColorPopUpButton indexOfSelectedItem] )
    {
    case 0:
      [aFilter setActionColor: [NSColor blueColor]];
      break;
      
    case 1:
      [aFilter setActionColor: [NSColor brownColor]];
      break;

    case 2:
      [aFilter setActionColor: [NSColor cyanColor]];
      break;
     
    case 3:
      [aFilter setActionColor: [NSColor darkGrayColor]];
      break;
 
    case 4:
      [aFilter setActionColor: [NSColor greenColor]];
      break;

    case 5:
      [aFilter setActionColor: [NSColor magentaColor]];
      break;
      
    case 6:
      [aFilter setActionColor: [NSColor orangeColor]];
      break;
      
    case 7:
      [aFilter setActionColor: [NSColor purpleColor]];
      break;
      
    case 8:
      [aFilter setActionColor: [NSColor redColor]];
      break;

    case 9:
      [aFilter setActionColor: [NSColor whiteColor]];
      break;

    case 10:
    default:
      [aFilter setActionColor: [NSColor yellowColor]];
      break;
    }
  
  [aFilter setActionFolderName: [actionFolderNamePopUpButton titleOfSelectedItem]];
  [aFilter setActionEMailOperation: ([actionEMailStringPopUpButton indexOfSelectedItem] + 1)];
  [aFilter setActionEMailString: [actionEMailStringField stringValue]];

  [NSApp stopModal];
  [self close];
}


- (IBAction) cancelClicked: (id) sender
{
  [NSApp abortModal];
  [self close];
}

- (IBAction) selectionHasChanged : (id) sender
{
  NSLog(@"Selection has changed...");
}

- (IBAction) setMessage: (id) sender
{
  FilterMessageWindowController *aFilterMessageWindowController;

  aFilterMessageWindowController = [[FilterMessageWindowController alloc]
				     initWithWindowNibName: @"FilterMessageWindow"];

  [aFilterMessageWindowController setMessageString: [[self filter] actionMessageString]];
  [NSApp runModalForWindow: [aFilterMessageWindowController window]];
  
  [[self filter] setActionMessageString: [aFilterMessageWindowController messageString]];
  RELEASE(aFilterMessageWindowController);
}


- (IBAction) criteriaSourceSelectionHasChanged: (id) sender
{
  [criteriaSourcePopUpButton synchronizeTitleAndSelectedItem];

  // We verify if it's the last item (Expert...) that has been selected
  if ( [criteriaSourcePopUpButton indexOfSelectedItem] == ([criteriaSourcePopUpButton numberOfItems] - 1) )
    {
      FilterHeaderEditorWindowController *filterHeaderEditorWindowController;
      int result;

      filterHeaderEditorWindowController = [[FilterHeaderEditorWindowController alloc]
					     initWithWindowNibName: @"FilterHeaderEditorWindow"];

      [filterHeaderEditorWindowController setHeaders: [filter criteriaHeaders]];
      result = [NSApp runModalForWindow: [filterHeaderEditorWindowController window]];
    
      // If "OK" was clicked, we must update our array of headers
      if ( result == NSRunStoppedResponse )
	{
	  [filter setCriteriaHeaders: [NSArray arrayWithArray: [filterHeaderEditorWindowController allHeaders]]];
	}

      RELEASE(filterHeaderEditorWindowController);
    }
} 

//
// access/mutation methods
//

- (FilterManager *) filterManager
{
  return filterManager;
}

- (void) setFilterManager: (FilterManager *) theFilterManager
{
  RETAIN(theFilterManager);
  RELEASE(filterManager);
  filterManager = theFilterManager;
}

- (Filter *) filter
{
  return filter;
}

- (void) setFilter: (Filter *) theFilter
{
  if ( theFilter )
    {
      RETAIN(theFilter);
      RELEASE(filter);
      filter = theFilter;

      // Description
      [descriptionField setStringValue: [filter description]];
      [activeButton setState: ([filter isActive] ? NSOnState : NSOffState)];
      
      // Criteria
      [criteriaSourcePopUpButton selectItemAtIndex: ([filter criteriaSource] - 1)];
      [criteriaFindOperationPopUpButton selectItemAtIndex: ([filter criteriaFindOperation] - 1)];
      [criteriaStringField setStringValue:  [filter criteriaString]];
      
      // Action
      [matrix selectCellAtRow: ([filter action] - 1) column: 0];
      
      if ( [[filter actionColor] isEqual: [NSColor blueColor]] )
	{
	  [actionColorPopUpButton selectItemAtIndex: 0];
	}
      else if ( [[filter actionColor] isEqual: [NSColor brownColor]] )
	{
	  [actionColorPopUpButton selectItemAtIndex: 1];
	}
      else if ( [[filter actionColor] isEqual: [NSColor cyanColor]] )
	{
	  [actionColorPopUpButton selectItemAtIndex: 2];
	}
      else if ( [[filter actionColor] isEqual: [NSColor darkGrayColor]] )
	{
	  [actionColorPopUpButton selectItemAtIndex: 3];
	}
      else if ( [[filter actionColor] isEqual: [NSColor greenColor]] )
	{
	  [actionColorPopUpButton selectItemAtIndex: 4];
	}
      else if ( [[filter actionColor] isEqual: [NSColor magentaColor]] )
	{
	  [actionColorPopUpButton selectItemAtIndex: 5];
	}
      else if ( [[filter actionColor] isEqual: [NSColor orangeColor]] )
	{
	  [actionColorPopUpButton selectItemAtIndex: 6];
	}
      else if ( [[filter actionColor] isEqual: [NSColor purpleColor]] )
	{
	  [actionColorPopUpButton selectItemAtIndex: 7];
	}
      else if ( [[filter actionColor] isEqual: [NSColor redColor]] )
	{
	  [actionColorPopUpButton selectItemAtIndex: 8];
	}
      else if ( [[filter actionColor] isEqual: [NSColor whiteColor]] )
	{
	  [actionColorPopUpButton selectItemAtIndex: 9];
	}
      else
	{
	  [actionColorPopUpButton selectItemAtIndex: 10];
	}
      
      // We try to select our folder, if it has been deleted (or renamed), we select Inbox.
      if ( [actionFolderNamePopUpButton indexOfItemWithTitle: [filter actionFolderName]] < 0 )
	{
	  [actionFolderNamePopUpButton selectItemWithTitle: [[NSUserDefaults standardUserDefaults] 
							      objectForKey: @"INBOXFOLDERNAME"]];
	}
      else
	{
	  [actionFolderNamePopUpButton selectItemWithTitle: [filter actionFolderName]];
	}
      [actionEMailStringPopUpButton selectItemAtIndex: ([filter actionEMailOperation] - 1)];
      [actionEMailStringField setStringValue: [filter actionEMailString]];
    }
  else
    {
      RELEASE(filter);
      filter = nil;
    }
}

//
// other methods
//
- (void) _addItemsToFolderNamePopUpButton
{   
  [actionFolderNamePopUpButton removeAllItems];
  [actionFolderNamePopUpButton addItemsWithTitles: [[[[(GNUMail *)[NSApp delegate] localStore]
						       folderEnumerator] allObjects]
						     sortedArrayUsingSelector: 
						       @selector(caseInsensitiveCompare:)] ];

  // We verify if we have at least one item, if not, we add "Inbox"
  if ( [actionFolderNamePopUpButton numberOfItems] == 0 )
    {
      [actionFolderNamePopUpButton addItemWithTitle: @"Inbox"];
    }
}

@end
