#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it 
# and/or modify it under the terms of the GNU General Public 
# License as published by the Free Software Foundation; either 
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be 
# useful, but WITHOUT ANY WARRANTY; without even the implied 
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2000-2005 Free Software Foundation
#
#
# FILE:
# GFTree.py
#
# DESCRIPTION:
"""
Logical tree support
"""
# NOTES:
#

from GFValue import GFValue
from GFObj import GFObj
from gnue.common import events
from gnue.forms import GFDisplayHandler
from GFBlock import GFBlock
import string
import copy

#
# internal tree node representation
#
class TreeNode:
  def __init__(self, parent=None):
    self.desc     = "n/a"    # description -- will appear in the tree view
    self.id       = 0        # database ID -- may be used to find the selected value in the DB
    self.parent   = parent   # parent node
    self.children = []       # children list (to spawn a subtree)
    self.columns  = []       # values of the columns
    self.leaf     = None     # GFLeaf reference
    self.srcinfo  = None     # reference to info struct in GFTree._infoMap
    self.recno    = 0        # record number in the block
    self._ui_obj = None      # this one will be set to the TreeItem obj by the UI

  def getUiObj (self):
    return self._ui_obj

  def setUiObj (self, obj=None):
    self._ui_obj = obj

#
# will hold information on the various sources (GFBlocks)
# and column namings.
#
class NodeInfo:
  def __init__(self):
    self.block = None      # the block we represent
    self.fld_id = ""       # ID field (primary key)
    self.fld_desc = ""     # Description field (Human readable)
    self.fld_query = ""    # Query field
    self.pat_query = None  # Query pattern

#
# GFTree
#
class GFTree (GFObj):
  def __init__(self, parent=None):
    GFObj.__init__(self, parent, 'GFTree')

    # Default attributes (these may be replaced by parser)
    self.Char__height = int(gConfigForms('widgetHeight'))

    # Runtime Variables
    self._inits = [self.initialize]

    self._treeRoot = TreeNode() # here's our internal tree representation
    self._leafMap = {}          # block information for other tables (leaves)
    self._colList = []          # column information for main tree

    self._infoMap = {}          # saves information on tree/leaves sources/fields/blocks

    self._incompleteNodes = []  # nodes that don't have columns yet set (but their kids have)

    self._loadList = []         # root items we have to load kids for (i.e. on IdleEvent)

    # note on columns:
    # the ListView right next to the tree will hold
    # be updated OnSelect to contain all children of
    # the current selection. (alternative: update it
    # to give the feel of an Qt-ListView
    # (multi-column tree-view)

    # note on leaves:
    # leaves should have an attribute "big" or "don't show in tree"
    # or whatever. it should prevent the items of a leafnode
    # from being loaded into the tree and instead only be shown
    # in the ListView (which might be implemented to have a decent
    # caching). 'normal' tree items (no leafnodes) are assumed to be
    # "small" (loadable in one single step, no caching needed...)

    # operations on big leaves (like computation/loading of several
    # columns) should be done in idleEvents.

    # still need to specify decent column info

    # trigger stuff
    self._validTriggers = {
		'ON-SELECTED':      'On-Selected',
		'POST-KIDSCHANGED': 'Post-KidsChanged'
	}

    self._triggerGlobal = 1
    self._triggerFunctions = {
			'setNodeCol':        { 'function': self._trigSetNodeCol },
			'getIncompleteNode': { 'function': self._trigGetIncompleteNode },
                        'popIncompleteNode': { 'function': self._trigPopIncompleteNode }
		}
   
    #self._triggerProperties = {'name': {'get': self.getName} }

  #
  # builds _infoMap structure
  #
  def _buildInfoMap (self):
    ## tree nodes info
    i = NodeInfo()
    i.block     = self._blk
    i.fld_id    = self.fld_id
    i.fld_desc  = self.fld_desc
    i.fld_query = self.fld_parentid
    i.pat_query = ''   # this only goes for the root node! FIXME!
    self._infoMap[self.name] = i

    ## leaf nodes info
    for leaf in self._leafMap.values():
      info = NodeInfo()
      info.block     = leaf._blk
      info.fld_id    = leaf.fld_id
      info.fld_desc  = leaf.fld_desc
      info.fld_query = leaf.fld_id
      info.pat_query = '*'
      self._infoMap[leaf.name] = info

  #
  # returns the currently changing (i.e. no-column-values-set-yet) node
  #
  def _trigGetIncompleteNode (self):
    if not self._incompleteNodes:
      return None
    else:
      return self._incompleteNodes[0]

  #
  # pops the next incomplete node from the incomplete-list (-> node is "complete" now)
  #
  def _trigPopIncompleteNode (self):
    self._incompleteNodes.pop (0)
  
  #
  # sets the column text for the current node
  #
  def _trigSetNodeCol (self, col, val):
    pass

  def _buildObject(self):
    if not hasattr(self, 'rows') and hasattr(self,'visibleCount'):
      self.rows = self.visibleCount
      del self.visibleCount
    return GFObj._buildObject(self)

  #
  # Routines called during a phaseInit
  #
  def initialize(self):
    if hasattr (self, 'block'):
      self._form = form = self.findParentOfType ('GFForm')
      self._orig_blk = form._logic._blockMap[self.block]
      self._blk = self._orig_blk
      
      # bad attempt to make a copy of a block... :-(
      #self._blk = copy.copy (self._orig_blk)
      #self._blk.initialize()

    self.evtCtrl = self._form._instance.eventController
    self.evtCtrl.registerEventListeners ({
		'uitreeItemSelected':  self._onItemSelected,
		'uitreeItemExpanded':  self._onItemExpanded,
		'uitreeItemCollapsed': self._onItemCollapsed
	})
  
  #
  # OnExpand handler. params: node
  #
  def _onItemExpanded (self, event):
    pass

  #
  # OnCollapse handler. params: node (surprise :-)
  #
  def _onItemCollapsed (self, event):
    pass

  #
  # OnSelect handler. params: cursel, lastsel
  #
  def _onItemSelected (self, event):
    try:
      blk = event.cursel.parent.leaf._blk
      blk.jumpRecord (event.cursel.recno)
      blk = event.lastsel.parent.leaf._blk
    except AttributeError:
      pass

    self.processTrigger ('ON-SELECTED')

  #
  # build tree from DB
  #
  def buildTree (self):

    self._buildInfoMap()

    self._treeRoot.id      = 0
    self._treeRoot.desc    = "kick me!"
    self._treeRoot.srcinfo = self._infoMap[self.name]

    #self.__populate_tree (self._treeRoot)

    self.__pop_query (self._treeRoot, 0)
    self.__pop_load_init (self._treeRoot)
    self.__pop_load_part (self._treeRoot, -1)
    self.__pop_load_finish (self._treeRoot)

    return self._treeRoot

  #
  # wrapper function for __populate_base
  # to populate tree nodes
  #
  def __populate_tree (self, rootnode):
    self.__populate_base (rootnode, self._blk,
                          self.fld_id, self.fld_desc,
                          self.fld_parentid, rootnode.id)

  # wrapper function for __populate_base
  # to populate leaf nodes
  def __populate_leaf (self, rootnode):
    leaf = rootnode.leaf
    self.__populate_base (rootnode, leaf._blk,
                          leaf.fld_id, leaf.fld_desc,
                          leaf.fld_id, '*')
 
  #
  # starts query for items to populate a (sub)tree with.
  # parameters:
  #   rootnode:  the root node
  #   pat_query: select pattern
  #
  def __pop_query (self, rootnode, val_query): 
    blk = rootnode.srcinfo.block
    fld_query = blk._fieldMap[rootnode.srcinfo.fld_query]
    #print "query for ", rootnode.srcinfo.fld_query, "=", val_query
    blk.initQuery()
    fld_query.setValue (val_query)
    blk.processQuery()

  #
  # initializes load phase for tree items
  # parameters:
  #   rootnode: root node to add items to
  #
  def __pop_load_init (self, rootnode):
    rootnode.srcinfo.block.firstRecord()
    self._loadList.append (rootnode)

  #
  # loads given number of tree items (call __pop_load_init first!)
  # nr: maximum number of items to read
  # returns the number of items actually loaded
  #
  def __pop_load_part (self, rootnode, nr):
    blk      = rootnode.srcinfo.block
    fld_id   = blk._fieldMap[rootnode.srcinfo.fld_id]
    fld_desc = blk._fieldMap[rootnode.srcinfo.fld_desc]

    total   = nr
    do_loop = not blk._resultSet.isLastRecord()

    while nr != 0 and do_loop:
      ## main item stuff
      node_data = TreeNode (rootnode)
      node_data.id       = fld_id.getValue()
      node_data.desc     = fld_desc.getValue()
      node_data.children = []
      node_data.recno    = blk._currentRecord
      node_data.columns  = []
      node_data.leaf     = None
      node_data.srcinfo  = self._infoMap[self.name] # goes for tree nodes

      ## column stuff (pretty clumsy... still waiting for enlightenment)
      if blk == self._blk:        # if this is tree-node (!= leaf-node)...
        leaf_tbl = blk._fieldMap[self.fld_tblhint].getValue()
        if leaf_tbl:
          node_data.leaf    = self._leafMap[leaf_tbl]
          node_data.srcinfo = self._infoMap[node_data.leaf.name]
        for ci in self._colList:  # add columns to the tree item
          node_data.columns.append (blk._fieldMap[ci.field].getValue())
      else:
        #node_data.srcinfo = self._infoMap[node_data.leaf.name]
        for ci in self._colList:  # add column to a leaf
          colsrc = rootnode.leaf._colsrcMap[ci.name]
          node_data.columns.append (blk._fieldMap[colsrc.field].getValue())

      rootnode.children.append (node_data)
      
      ## while-loop stuff
      nr = nr - 1
      do_loop = not blk._resultSet.isLastRecord()
      blk.nextRecord()

    # while-loop ends.

    self.evtCtrl.dispatchEvent (events.Event ('gftreeAppendKids',
                            root=rootnode, kids=rootnode.children,
                            _form=self._form))

    return total-nr

  #
  # ends load phase for tree items
  #
  def __pop_load_finish (self, rootnode):
    self._loadList.remove (rootnode)

    ## populate sub-trees/leaves
    if rootnode.srcinfo.block == self._blk:    # if this is _not_ a leaf-list
      for n in rootnode.children:
        pat = '*'
        if n.leaf == None: pat = n.id
        self.__pop_query (n, pat)
        self.__pop_load_init (n)
        self.__pop_load_part (n, -1)

    # trigger KidsChanged -- that should fill in column values for tree nodes
    if rootnode != self._treeRoot:
      self._incompleteNodes.append (rootnode)
      self.processTrigger ('POST-KIDSCHANGED')

  #
  # aborts load phase for tree items
  #
  def __pop_load_cancel (self, rootnode, blk):
    pass

  #
  # functions populates list subnode 'rootnode' with
  # items from DB with items in the DB that have parent id 'rootid'
  #
  # (or populates with items from another table that match the
  #  val_query in fld_query => leafnodes)
  #
  # fld_id:    ID field (primary key). will be used to jump to records (??)
  # fld_desc:  human readable information field
  # fld_query: "use it or not"-decision information field
  # val_query: "use it or nor"-decision query string
  # blk:       block to use
  # rootnode:  root node to insert items into (see above for type)
  #
  def __populate_base (self, rootnode, blk, fld_id, fld_desc, fld_query, val_query):

    blk.initQuery()
    blk._fieldMap[fld_query].setValue (val_query)
    blk.processQuery()
   
    blk.firstRecord()
    go = not blk._resultSet.isLastRecord()
    while go == True:
      item = blk
      node_data = TreeNode (rootnode)
      node_data.desc     = item._fieldMap[fld_desc].getValue()
      node_data.id       = item._fieldMap[fld_id].getValue()
      node_data.parent   = rootnode
      node_data.children = []
      node_data.columns  = []
      node_data.recno    = blk._currentRecord
      node_data.leaf     = None

      # if this is a leafnode (main tree item which
      # spawns a sub-tree from another table)
      if blk == self._blk:
        leaf_tbl = item._fieldMap[self.fld_tblhint].getValue()
        if leaf_tbl:
          node_data.leaf = self._leafMap[leaf_tbl]
        
        # load column information for tree-nodes and leaf-nodes
        for ci in self._colList:
          node_data.columns.append (item._fieldMap[ci.field].getValue())

      else:

        # load columns for leaf-ends
	#for ci in rootnode.leaf._colsrcList:
        #  node_data.columns.append (item._fieldMap[ci.field].getValue())
        for ci in self._colList:
          colsrc = rootnode.leaf._colsrcMap[ci.name]
          node_data.columns.append (item._fieldMap[colsrc.field].getValue())

      rootnode.children.append (node_data)

      go = not blk._resultSet.isLastRecord()
      if go:
         blk.nextRecord()

    # end 'while'
    
    self.evtCtrl.dispatchEvent (events.Event ('gftreeAppendKids',
                                root=rootnode, kids=rootnode.children,
                                _form=self._form))

    ## populate sub-trees/leaves
    if blk == self._blk:
      for n in rootnode.children:
        if n.leaf == None:
          self.__populate_tree (n)
        else:
          self.__populate_leaf (n)

    # trigger KidsChanged -- that should fill in column values for tree nodes
    if rootnode != self._treeRoot:
      self._incompleteNodes.append (rootnode)
      self.processTrigger ('POST-KIDSCHANGED')

  #
  # removes the children of a node (and sends rm event to ui)
  # 
  def __remove_kids (self, rootnode):
    for item in rootnode.children:
      if item.children:
        self.__remove_kids (item)
    self.evtCtrl.dispatchEvent (events.Event ('gftreeRemoveKids',
                                root=rootnode, _form=self._form))
    for item in rootnode.children:
      rootnode.children.remove (item)
