/*  GnomeKiss - A KiSS viewer for the GNOME desktop
    Copyright (C) 2000-2001  Nick Lamb <njl195@zepler.org.uk>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <ctype.h>
#include <gnome.h>
#include "kiss.h"

/* Concrete instantiation */
GSList *cells;
guchar *global_start, *global_end;

#define NIBBLE(x) { if (j & 1) x = (*(tmp++)) & 15; else x = (*tmp) >> 4; }

static int cell_load(KissCell *cell, char *filename) {
  FILE *fp;
  guchar header[32], *buffer, *tmp;
  guchar k;
  int bpp, i, j;

  fp= open_any(filename, "r");
  if (fp == NULL) return 1;

  fread(header, 4, 1, fp);

  if (strncmp(header, "KiSS", 4)) {
    bpp= 4;
    cell->width= header[0] + (256 * header[1]);
    cell->height= header[2] + (256 * header[3]);
    cell->offx= 0;
    cell->offy= 0;
  } else { /* New-style image file, read full header */
    fread(header + 4, 28, 1, fp);
    bpp= header[5];
    cell->width= header[8] + (256 * header[9]);
    cell->height= header[10] + (256 * header[11]);
    cell->offx= header[12] + (256 * header[13]);
    cell->offy= header[14] + (256 * header[15]);
  }

  for (i= 0; i < SETS; ++i) {
    cell->x[i]= cell->offx;
    cell->y[i]= cell->offy;
  }

  if (bpp == 32) {
    cell->index= NULL;
    cell->argb= g_new(guchar, cell->width * cell->height * 4);
  } else {
    cell->index= g_new(guchar, cell->width * cell->height);
    cell->argb= NULL;
  }


  cell->events= g_new0(KissCellEvents, 1);

  switch (bpp) {
  case 4:
    buffer= g_new(guchar, (cell->width + 1) / 2);
    for (i= 0; i < cell->height; ++i) {
      fread(buffer, (cell->width + 1) / 2, 1, fp);
      for (j= 0, tmp= buffer; j < cell->width; ++j) {
        NIBBLE(k);
        cell->index[i * cell->width + j] = k;
      }
    }
    g_free(buffer);
    break;
  case 8:
    for (i= 0; i < cell->height; ++i) {
      buffer= cell->index + (i * cell->width);
      fread(buffer, cell->width, 1, fp);
    }
    break;
  case 32:
    buffer= g_new(guchar, cell->width * 4);
    for (i= 0; i < cell->height; ++i) {
      fread(buffer, cell->width, 4, fp);
      for (j= 0; j < cell->width; ++j) {
        cell->argb[4 * (i * cell->width + j)]= buffer[j*4 + 3];
        cell->argb[4 * (i * cell->width + j) + 1]= buffer[j*4 + 2];
        cell->argb[4 * (i * cell->width + j) + 2]= buffer[j*4 + 1];
        cell->argb[4 * (i * cell->width + j) + 3]= buffer[j*4 ];
      }
    }
    g_free(buffer);
    break;
  default:
    log_error(_("cell \"%s\" has unsupported color depth"), filename);
    /* leaves cell->index allocated but garbage filled */
    break;
  }

  fclose(fp);
  cell->mapped= 1; /* mapped */
  cell->duplicate= 0; /* first instance */

  /* Some configs say (640,480) and then load an 800x600 background CEL */
  if (config.width < cell->width + cell->offx) {
    config.width= cell->width + cell->offx;
    config.row_stride= config.width * 3;
    log_warning(_("cell \"%s\" is too wide for the playfield"), filename);
  }
  if (config.height < cell->height + cell->offy) {
    config.height= cell->height + cell->offy;
    log_warning(_("cell \"%s\" is too high for the playfield"), filename);
  }

  cell->name= g_strdup(filename);

  return 0;
}

void object_move(KissObject *object, int x, int y) {
  guchar *start, *end;

  start= config.rgb_buf + (config.row_stride * object->y[view]);
  if (start < global_start) global_start= start;
  end= start + (config.row_stride * object->height);
  if (end > global_end) global_end= end;

  object_set_location(object, x, y);

  start= config.rgb_buf + (config.row_stride * object->y[view]);
  if (start < global_start) global_start= start;
  end= start + (config.row_stride * object->height);
  if (end > global_end) global_end= end;

  if (global_lock != KISS_PENDING && area != NULL) {
    global_lock= KISS_PENDING;
    gtk_widget_queue_draw(area);
  }

  check_collisions(object, 0);
}

void object_set_location(KissObject *object, int x, int y) {
  GSList *list;
  KissCell *cell;

  if (x + object->width > config.width)
    x= config.width - object->width;
  if (y + object->height > config.height)
    y= config.height - object->height;
  if (x < 0) x= 0;
  if (y < 0) y= 0;

  for (list= object->cells; list != NULL; list= g_slist_next(list)) {
    cell= (KissCell *) list->data;
    cell->x[view] = x + cell->offx;
    cell->y[view] = y + cell->offy;
  }
  object->x[view]= x; object->y[view]= y;
}

void cell_render(gpointer data, gpointer user_data) {
  guchar alpha;
  KissCell *cell;
  guchar *begin;
  guchar *source, *src, *dest, *dst, *pal;
  guchar red, green, blue;
  unsigned int row, col;
  
  cell= (KissCell *) data;
  if (!cell->mapped || !cell->alpha || !(cell->visible & (1 << view)))
    return;

  if (cell->argb && cell->alpha == 255) {
    begin= config.rgb_buf;
    src= source= cell->argb;
    dst= dest= config.rgb_buf + (cell->y[view] * config.row_stride)
                              + cell->x[view] * 3;

    for (row= 0; row < cell->height; ++row) {

      if (dest < global_start) {
        src= source+= cell->width * 4;
        dst= dest+= config.row_stride;
        continue;
      }
      if (dest >= global_end) {
        return;
      }

      for (col= 0; col < cell->width; ++col) {
        if (*src == 0) {
          dst+= 3;
          src+= 4;
        } else if (*src == 255) {
          src++;
          *(dst++)= *(src++);
          *(dst++)= *(src++);
          *(dst++)= *(src++);
        } else {
          alpha = *(src++);
          *dst= ((*(src++) * alpha) + (*dst * (255 - alpha))) / 255; dst++;
          *dst= ((*(src++) * alpha) + (*dst * (255 - alpha))) / 255; dst++;
          *dst= ((*(src++) * alpha) + (*dst * (255 - alpha))) / 255; dst++;
        }
      }

      src= source+= cell->width * 4;
      dst= dest+= config.row_stride;
    }

  } else if (cell->argb) {
    begin= config.rgb_buf;
    src= source= cell->argb;
    dst= dest= config.rgb_buf + (cell->y[view] * config.row_stride)
                              + cell->x[view] * 3;

    for (row= 0; row < cell->height; ++row) {

      if (dest < global_start) {
        src= source+= cell->width * 4;
        dst= dest+= config.row_stride;
        continue;
      }
      if (dest >= global_end) {
        return;
      }

      for (col= 0; col < cell->width; ++col) {
        if (*src == 0) {
          dst+= 3;
          src+= 4;
        } else {
          alpha = (*(src++) * cell->alpha) / 255;
          *dst= ((*(src++) * alpha) + (*dst * (255 - alpha))) / 255; dst++;
          *dst= ((*(src++) * alpha) + (*dst * (255 - alpha))) / 255; dst++;
          *dst= ((*(src++) * alpha) + (*dst * (255 - alpha))) / 255; dst++;
        }
      }

      src= source+= cell->width * 4;
      dst= dest+= config.row_stride;
    }

  } else if (cell->index && cell->alpha == 255) {
    pal= palettes[cell->palette * COLS + config.pal_set[view]];
    begin= config.rgb_buf;
    source= cell->index;
    /* src is abused terribly in this code fragment */
    dst= dest= config.rgb_buf + (cell->y[view] * config.row_stride)
                              + cell->x[view] * 3;

    for (row= 0; row < cell->height; ++row) {

      if (dest < global_start) {
        source+= cell->width;
        dst= dest+= config.row_stride;
        continue;
      }
      if (dest >= global_end) {
        return;
      }

      for (col= 0; col < cell->width; ++col) {
        if (source[col] != 0) {
          src= &pal[source[col] *3];
          dst[0]= src[0];
          dst[1]= src[1];
          dst[2]= src[2];
        }
        dst+= 3;
      }

      source+= cell->width;
      dst= dest+= config.row_stride;
    }
  } else if (cell->index) {
    alpha= cell->alpha;
    pal= palettes[cell->palette * COLS + config.pal_set[view]];
    begin= config.rgb_buf;
    src= source= cell->index;
    dst= dest= config.rgb_buf + (cell->y[view] * config.row_stride)
                              + cell->x[view] * 3;

    for (row= 0; row < cell->height; ++row) {

      if (dest < global_start) {
        source+= cell->width;
        dst= dest+= config.row_stride;
        continue;
      }
      if (dest >= global_end) {
        return;
      }

      for (col= 0; col < cell->width; ++col) {
        if (source[col] != 0) {
          red = pal[source[col]*3 ];
          green = pal[source[col]*3 +1];
          blue = pal[source[col]*3 +2];
          dst[0]= ((red * alpha) + (dst[0] * (255 - alpha))) / 255;
          dst[1]= ((green * alpha) + (dst[1] * (255 - alpha))) / 255;
          dst[2]= ((blue * alpha) + (dst[2] * (255 - alpha))) / 255;
        }
        dst+= 3;
      }

      source+= cell->width;
      dst= dest+= config.row_stride;
    }
  }

}

/* NB Returns a pointer to the CELL */
KissCell *cell_intersect(KissCell *cell, int x, int y) {

  x-= cell->x[view];
  y-= cell->y[view];

  if (!cell->mapped || !(cell->visible & (1 << view)))
    return NULL;

  if (cell->ghosted)
    return NULL;
  
  if (x < 0 || x >= cell->width)
    return NULL;

  if (y < 0 || y >= cell->height)
    return NULL;

  if (cell->index && cell->index[x + y * cell->width] == 0)
    return NULL;

  if (cell->argb && cell->argb[(x + y * cell->width) * 4] == 0)
    return NULL;

  return cell;
}

KissCell *cell_find(gchar *name) {
  GSList *list;
  KissCell *current, *match= NULL;
  int count= 0;

  for (list= cells; list != NULL; list= g_slist_next(list)) {
    current= (KissCell *) list->data;
    if (!g_strcasecmp(current->name, name)) {
      match= current;
      ++count;
    }
  }

  if (count > 1) {
    log_warning(_("cell name \"%s\" is ambiguous"), name);
  }

  return match;
}

KissObject *object_id(guint id) {
  if (id >= OBJECTS || config.objects[id].cells == NULL) {
    log_error(_("no object #%u defined in layout"), id);
    return NULL;
  } else {
    return &(config.objects[id]);
  }
}

KissObject *object_find(gchar *ref) {
  return object_id(numeric(ref));
}

KissCell *cell_new(guint objid, guint pal, gchar *filename) {
  KissCell *cell, *original = NULL;
  GSList *list;

  if (objid > OBJECTS) {
    log_warning(_("object #%u exceeds object limit"), objid);
    return NULL;
  }

  cell= g_new0(KissCell, 1);
  cell->object= &(config.objects[objid]);

  /* Check if a similarly named cell was already found */
  for (list= cells; list != NULL; list= g_slist_next(list)) {
    if (!g_strcasecmp(((KissCell *) list->data)->name, filename)) {
      original= (KissCell *) list->data;
      break;
    }
  }

  /* If so copy the shareable data across */
  if (original) {
    cell->name= original->name;
    cell->index= original->index;
    cell->argb= original->argb;
    cell->width= original->width;
    cell->height= original->height;
    cell->offx= original->offx;
    cell->offy= original->offy;
    cell->mapped= original->mapped;
    cell->events= original->events;
    cell->duplicate= 1; /* just a copy */
  } else if (cell_load(cell, filename)) {
    g_free(cell);
    return NULL;
  }

  cell->palette= pal;

  /* Add to object and global display lists */
  cells = g_slist_prepend(cells, cell);
  cell->object->cells = g_slist_prepend(cell->object->cells, cell);
  if (cell->width + cell->offx > cell->object->width)
    cell->object->width = cell->width + cell->offx;
  if (cell->height + cell->offy > cell->object->height)
    cell->object->height = cell->height + cell->offy;

  return cell;
}

void cell_delete(KissCell *cell) {
  if (!cell->duplicate) {
    g_free(cell->name);
    if (cell->index) g_free(cell->index);
    if (cell->argb) g_free(cell->argb);

    FREE_ACTION_LIST(cell->events->catch);
    FREE_ACTION_LIST(cell->events->drop);
    FREE_ACTION_LIST(cell->events->fixcatch);
    FREE_ACTION_LIST(cell->events->fixdrop);
    FREE_ACTION_LIST(cell->events->press);
    FREE_ACTION_LIST(cell->events->release);
    g_free(cell->events);
  }

  g_free(cell);
}

