/*
 * GNoise
 *
 * Copyright (C) 1999-2001 Dwight Engen
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: display_cache.c,v 1.5 2001/09/04 23:05:04 dengen Exp $
 *
 */

#include <errno.h>
#include <gtk/gtk.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/types.h>
#include "snd_buf.h"
#include "display_cache.h"

static gint8	dcache_allocate (snd_buf_t *sb);
static gboolean	dcache_create   (snd_buf_t *sb);



static gint8
dcache_allocate(snd_buf_t *sb)
{
    int i;

    sb->dcache->zoom_level = -1;

    /* determine first cacheable zoom level */
    for(i = 0; i < ZOOM_LEVELS; i++)
    {
	guint32 zoom = 1<<i;
	if (zoom < ZOOM_CACHE)
	{
	    log("DCACHE ALLOC", "zoom level 1:%-7d generated\n", zoom);
	    continue;
	}

	/* remember first cached zoom level */
	if (zoom == ZOOM_CACHE)
	    sb->dcache->zoom_level = i;

	/* allocate memory for cache entries */
	sb->dcache->size[i] = sb->info.samples/zoom*sb->info.channels*2
			      + 2 * sb->info.channels;
	sb->dcache->data[i] = realloc(sb->dcache->data[i], sb->dcache->size[i]);
	if (sb->dcache->data[i] == NULL)
	{
	    log("DCACHE ALLOC", "unable to allocate display cache memory\n");
	    return(-1);
	}

	log("DCACHE ALLOC", "zoom level 1:%-7d cache entries/bytes %d\n",
	    zoom, sb->dcache->size[i]);
    }
    return(sb->dcache->zoom_level);
}



static gboolean
dcache_create(snd_buf_t *sb)
{
    if (dcache_allocate(sb) >= 0)
    {
	if (dcache_update(sb, 0, sb->info.samples * sb->info.channels, TRUE))
	    return(TRUE);
	else
	{
	    dcache_free(sb);
	}
    }
    return(FALSE);
}



gboolean
dcache_resize(snd_buf_t *sb)
{
    /* resize the cache when the file size changed due to edits */
    if (sb->info.samples != sb->dcache->samples)
    {
	log("DCACHE RESIZE", "resize  to samples:%d (old:%d)\n",
	    sb->info.samples, sb->dcache->samples);
	if (dcache_allocate(sb) < 0)
	{
	    log("DCACHE", "unable to resize!\n");
	    return(FALSE);
	}
    }
    return(TRUE);
}



void *
dcache_attach(void *thread_arg)
{
    snd_buf_t *sb = (snd_buf_t *)thread_arg;
    gboolean rc = TRUE;

    if (!dcache_load(sb))
    {
	if (dcache_create(sb) && dcache_save(sb))
	    ;
	else
	    rc = FALSE;
    }

    queue_cmd(CMD_REAP_LOAD_THREAD, NULL);
    return((void *)rc);
}



/*
 * FUNC: update display cache data
 *   IN: wave set
 *	 the region to generate/update the display cache for
 *	 if the user should be able to cancel the operation
 *  OUT: TRUE on success, FALSE if user pressed Cancel
 *	 or we couldn't finish
 * NOTE: calculates display cache for all channels, should probably do an
 *	 optimized version for when only one channel is modified. runs as
 *	 non gui thread.
 */
gboolean
dcache_update(snd_buf_t *sb,
	      guint32 data_start,
	      guint32 data_end,
	      gboolean allow_cancel)
{
    guint8 i,j;
    guint32 data_indx;
    guint8 smp[MAX_CHANNELS];
    guint8 smp_min[MAX_CHANNELS];
    guint8 smp_max[MAX_CHANNELS];
    gint old_percent = 0;		/* last percent complete set */
    gint new_percent;			/* current percent complete */
    gint channel;

    status("Generating display cache...");

    if (allow_cancel)
    {
	queue_cmd(CMD_WIDGET_SHOW, "Cancel");
	progress_cancel = FALSE;
    }
    queue_cmd(CMD_WIDGET_SHOW, "Progressbar");


    /* initialize min/max to silence for initial block */
    for(channel = 0; channel < sb->info.channels; channel++)
    {
	smp_min[channel] = 127;
	smp_max[channel] = 127;
    }

    /* make starting and ending sample mod the lowest cacheable
     * zoom_level
     */
    if (data_start % ZOOM_CACHE)
    {
	data_start -= (data_start % ZOOM_CACHE);
    }
    if (data_end % ZOOM_CACHE)
    {
	data_end +=
	    ((data_end + ZOOM_CACHE) % ZOOM_CACHE);

	if (data_end > sb->info.samples * sb->info.channels)
	    data_end = sb->info.samples * sb->info.channels;
    }

    for(data_indx = data_start;
	data_indx < data_end && !progress_cancel;
	data_indx+=sb->info.channels)
    {
	/* adjust min/max for this block */
	for(channel = 0; channel < sb->info.channels; channel++)
	{
	    switch(sb->info.sample_bits)
	    {
		case 8:
		    smp[channel] = ((gint8 *)sb->data)[data_indx + channel]+255;
		    break;

		case 16:
		    smp[channel] = ((((gint16 *)sb->data)[data_indx + channel]+32768) * 255)/65536;
		    break;
	    }

	    smp_min[channel] = MIN(smp[channel], smp_min[channel]);
	    smp_max[channel] = MAX(smp[channel], smp_max[channel]);
	}

	/* see if we've gone far enough to generate a cache entry
	 * for the lowest cacheable zoom level
	 */
	if (((data_indx/sb->info.channels % ZOOM_CACHE) == 0) &&
	    (data_indx != data_start))
	{
	    //log("DCACHE UPDT", "hit on data_indx %d\n", data_indx);
	    //for (i = ws->zoom_level; i < ZOOM_LEVELS; i++)
	    for (i = sb->dcache->zoom_level; i < ZOOM_LEVELS; i++)
	    {
		gint cache_indx;

		cache_indx = ((data_indx/sb->info.channels / (1<<i)) - 1) * sb->info.channels;

		if ((cache_indx < 0) || 	// don't fill until slot is full
		    (data_indx/sb->info.channels % (1<<i) != 0))	// not until both of the previous level's preceding blocks are full
		    continue;

		cache_indx *= 2;		// 2 entries per slot, min & max
		//log("DCACHE UPDT", "ZL:%d CI:%d\n", i, cache_indx);

		/* fill in lowest zoom level directly */
		if (i == sb->dcache->zoom_level)
		{
		    /* the additional arithmetic is to make positive values
		     * appear above 0 amplitude
		     */
		    for(channel = 0; channel < sb->info.channels; channel++)
		    {
			sb->dcache->data[i][cache_indx+  channel*2] = 255-smp_max[channel];
			sb->dcache->data[i][cache_indx+1+channel*2] = 255-smp_min[channel];

			// initialize min/max to silence for next block
			smp_min[channel] = 127;
			smp_max[channel] = 127;
		    }
		} else {


#		    define ENTRIES_PER_BLOCK (sb->info.channels * 2)
#		    define MIN_INDX1 (cache_indx * 2)
#		    define MIN_INDX2 ((MIN_INDX1) + (ENTRIES_PER_BLOCK))
#		    define MAX_INDX1 ((MIN_INDX1) + 1)
#		    define MAX_INDX2 ((MIN_INDX2) + 1)

		    // generate this level from the level above's last 2

		    for (j = 0; j < sb->info.channels; j++)
		    {
			// min from previous
			sb->dcache->data[i][cache_indx+j*2] = MIN(
			    sb->dcache->data[i-1][MIN_INDX1+j*2],
			    sb->dcache->data[i-1][MIN_INDX2+j*2]);
			
			// max from previous
			sb->dcache->data[i][cache_indx+j*2+1] = MAX(
			    sb->dcache->data[i-1][MAX_INDX1+j*2],
			    sb->dcache->data[i-1][MAX_INDX2+j*2]);

#			if 0
			log("DCACHE UPDT", "indx %d:%d min %d max %d\n",
			    i, cache_indx, ws->dcache->data[i][cache_indx],
			    ws->dcache->data[i][cache_indx+1]);
#			endif
		    }
		}
	    }
	}

	new_percent = (data_indx / (data_end / 100));
	if(new_percent != old_percent)
	{
	    gfloat prog;

	    old_percent = new_percent;
	    prog = (gfloat)new_percent;
	    queue_cmd(CMD_PROGRESS_BAR_UPDATE, &prog);
	}

    }

    queue_cmd(CMD_WIDGET_HIDE, "Progressbar");
    if (allow_cancel)
	queue_cmd(CMD_WIDGET_HIDE, "Cancel");

    sb->dcache->dirty = TRUE;
    return(!progress_cancel);
}


void
dcache_free(snd_buf_t *sb)
{
    guint8 i;

    status("Freeing display cache...");

    for(i = 0; i < ZOOM_LEVELS; i++)
    {
	if (sb->dcache->data[i])
	{
	    free(sb->dcache->data[i]);
	    sb->dcache->data[i] = NULL;
	}
    }
}


gboolean
dcache_load(snd_buf_t *sb)
{
    int			dfd;
    gchar		*df_dir;
    gchar		*df_base;
    char		df_name[NAME_MAX];
    dcache_header_t	dh;
    uint		dcache_level_size;
    int			i,k;
    int			cache_zoom_level;
    struct stat		wav_stat;
    struct stat		df_stat;


    status("Loading display cache...");

    df_dir  = g_dirname(sb->file);
    df_base = g_basename(sb->file);
    sprintf(df_name, "%s/.%s.dcache", df_dir, df_base);
    g_free(df_dir);

    dfd = open(df_name, O_RDONLY);
    if (dfd < 0)
    {
	goto err1;
    }

    /* ensure that display cache is uptodate with file */
    if ((fstat(sb->fd, &wav_stat) < 0) || (fstat(dfd, &df_stat) < 0))
    {
	log("DCACHE LOAD", "unable to stat wav or dcache %s\n",
	    strerror(errno));
	goto err2;
    }

    if (wav_stat.st_mtime > df_stat.st_mtime)
    {
	log("DCACHE LOAD", "display cache older than file, regenerating\n");
	goto err2;
    }

    cache_zoom_level = dcache_allocate(sb);
    if (cache_zoom_level < 0)
	goto err2;
    read(dfd, &dh, sizeof(dh));

    /* verify header */
    if (memcmp(&dh.id, "DCACHE", 6) != 0)
    {
	log("DCACHE LOAD", "Not a valid cache file %s\n", df_name);
	goto err3;
    }

    /* verify saved file has same zoom level as we've allocated */
    if (dh.zoom_level != cache_zoom_level)
    {
	log("DCACHE LOAD", "cache file not same zoom threshold as this version of gnoise");
	goto err3;
    }

    log("DCACHE LOAD", "loading %s display cache\n", dh.version);

    /* load lowest level (the only one saved) */
    read(dfd, &dcache_level_size, sizeof(dcache_level_size));
    log("DCACHE LOAD", "loading    level 1:%-7d size:%d\n", 1<<dh.zoom_level, dcache_level_size);
    read(dfd, sb->dcache->data[dh.zoom_level], dcache_level_size);

    /* generate remaining levels from the previous levels */
    for (i = cache_zoom_level+1; i < ZOOM_LEVELS; i++)
    {
	gint cache_indx;

	log("DCACHE LOAD", "generating level 1:%d\n", 1<<i);
	for(cache_indx = 0;
	    cache_indx < sb->dcache->size[i];
	    cache_indx += sb->info.channels * 2)
	{
#		    define ENTRIES_PER_BLOCK (sb->info.channels * 2)
#		    define MIN_INDX1 (cache_indx * 2)
#		    define MIN_INDX2 ((MIN_INDX1) + (ENTRIES_PER_BLOCK))
#		    define MAX_INDX1 ((MIN_INDX1) + 1)
#		    define MAX_INDX2 ((MIN_INDX2) + 1)

		/* generate this level from the level above's last 2 */
		for (k = 0; k < sb->info.channels; k++)
		{

		/* min from previous */
		sb->dcache->data[i][cache_indx+k*2] = MIN(
		    sb->dcache->data[i-1][MIN_INDX1+k*2],
		    sb->dcache->data[i-1][MIN_INDX2+k*2]);
		
		/* max from previous */
		sb->dcache->data[i][cache_indx+k*2+1] = MAX(
		    sb->dcache->data[i-1][MAX_INDX1+k*2],
		    sb->dcache->data[i-1][MAX_INDX2+k*2]);
		}
	} /* for cache_indx */
    } /* for zoom_level */
    log("DCACHE LOAD", "done\n");

    close(dfd);
    sb->dcache->dirty = FALSE;
    return(TRUE);

err3:
    dcache_free(sb);
err2:
    close(dfd);
err1:
    return(FALSE);
}



/*
 * FUNC: save display cache to disk
 *   IN: wave set
 *  OUT: TRUE on success, FALSE on failure
 */
gboolean
dcache_save(snd_buf_t *sb)
{
    FILE *dfp;
    gchar *df_dir;
    gchar *df_base;
    char df_name[NAME_MAX];
    dcache_header_t dh;
    int i;

    if (!prefs.dcache_save)
	return TRUE;
    status("Saving display cache...");
    memset(&dh, 0, sizeof(dh));
    memcpy(&dh.id, "DCACHE", 6);
    sprintf(dh.version, "v1.0");

    // find dcache levels
    for (i=0; i < ZOOM_LEVELS; i++)
    {
	if ((1<<i) == ZOOM_CACHE)
	{
	    dh.zoom_level = i;
	    break;
	}
    }

    df_dir  = g_dirname(sb->file);
    df_base = g_basename(sb->file);
    sprintf(df_name, "%s/.%s.dcache", df_dir, df_base);
    g_free(df_dir);
    dfp = fopen(df_name, "wb");
    if (dfp != NULL)
    {
	fwrite(&dh, sizeof(dh), 1, dfp);

	if (sb->dcache->data[i] != NULL)
	{
	    log("DCACHE SAVE", "writing zoom level 1:%-7d size:%d\n",
		1<<i, sb->dcache->size[i]);
	    fwrite(&sb->dcache->size[i], sizeof(sb->dcache->size[i]), 1, dfp);
	    fwrite(sb->dcache->data[i], sb->dcache->size[i], 1, dfp);
	    fclose(dfp);
	    return TRUE;
	} else {
	    log("DCACHE SAVE", "null DC for level %d!\n", 1<<i);
	    return(FALSE);
	}
    } else {
	log("DCACHE SAVE", "unable to save display cache to file %s %s\n",
	    df_name, strerror(errno));
	return(FALSE);
    }
    sb->dcache->dirty = FALSE;
    log("DCACHE SAVE", "saved display cache to file %s\n", df_base);
}


/*
 * In memory display cache layout, assuming 1:256 is first cacheable zoom level
 *
 * mono format:
 *
 *   cache line 8             cache line 9               cache line 10
 * 0  min for 0-255    |  
 * 1  max for 0-255    |     0 min for 0-511    |
 *                           |                  |
 * 2  min for 256-511  |     1 max for 0-511    |  \
 * 3  max for 256-511  |                            \___| min for 0-1024 |
 *
 * 4  min for 512-767  |
 * 5  max for 512-767  |     2 min for 512-1024 |
 *                           |                  |
 * 6  min for 768-1024 |     3 max for 512-1024 |
 * 7  max for 768-1024 |
 *
 * 8  min for 512-767  |                             
 * 9  max for 512-767  |     4 min for 512-1024 |   
 *                           |                  |
 * 10 min for 768-1024 |     5 max for 512-1024 |
 * 11 max for 768-1024 |
 *
 * 12 min for 512-767  |                             
 * 13 max for 512-767  |     6 min for 512-1024 |   
 *                           |                  |
 * 14 min for 768-1024 |     7 max for 512-1024 |
 * 15 max for 768-1024 |
 *
 *
 *
 * stereo format:
 *
 *   cache line 8             cache line 9               cache line 10
 * 0 lmin for 0-255    |                         
 * | lmax for 0-255    |       
 * | rmin for 0-255    |\     
 * | rmax for 0-255    | \   0 lmin for 0-511    |
 *                        >--| lmax for 0-511    |
 * 4 lmin for 256-511  | /   | rmin for 0-511    |
 * | lmax for 256-511  |/    | rmax for 0-511    |                    
 * | rmin for 256-511  |
 * | rmax for 256-511  |
 *
 * 8 lmin for 512-767  |     
 * | lmax for 512-767  |     
 * | rmin for 512-767  |\         
 * | rmax for 512-767  | \   4 lmin for 512-1024 |
 *                        >--| lmax for 512-1024 |
 * 12 lmin for 768-1024| /   | rmin for 512-1024 |
 * | lmax for 768-1024 |/    | rmax for 512-1024 |
 * | rmin for 768-1024 |
 * | rmax for 768-1024 |
 *
 * 16 lmin for 512-767 |     
 * | lmax for 512-767  |     
 * | rmin for 512-767  |\         
 * | rmax for 512-767  | \   8 lmin for 512-1024 |
 *                        >--| lmax for 512-1024 |
 * 20 lmin for 768-1024| /   | rmin for 512-1024 |
 * | lmax for 768-1024 |/    | rmax for 512-1024 |
 * | rmin for 768-1024 |
 * | rmax for 768-1024 |
 *
 * 24 lmin for 512-767 |     
 * | lmax for 512-767  |     
 * | rmin for 512-767  |\         
 * | rmax for 512-767  | \   12 lmin for 512-1024 |
 *                        >--| lmax for 512-1024 |
 * | lmin for 768-1024 | /   | rmin for 512-1024 |
 * | lmax for 768-1024 |/    | rmax for 512-1024 |
 * | rmin for 768-1024 |
 * | rmax for 768-1024 |
 */
