------------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--                               L A Y O U T                                --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                            $Revision: 1.12 $
--                                                                          --
--            Copyright (C) 2001 Free Software Foundation, Inc.             --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT;  see file COPYING.  If not, write --
-- to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, --
-- MA 02111-1307, USA.                                                      --
--                                                                          --
-- GNAT was originally developed  by the GNAT team at  New York University. --
-- It is now maintained by Ada Core Technologies Inc (http://www.gnat.com). --
--                                                                          --
------------------------------------------------------------------------------

with Atree;    use Atree;
with Debug;    use Debug;
with Einfo;    use Einfo;
with Errout;   use Errout;
with Sem_Ch13; use Sem_Ch13;
with Sem_Eval; use Sem_Eval;
with Sem_Util; use Sem_Util;
with Sinfo;    use Sinfo;
with Snames;   use Snames;
with Targparm; use Targparm;
with Ttypes;   use Ttypes;
with Uintp;    use Uintp;

package body Layout is

   ------------------------
   -- Local Declarations --
   ------------------------

   SSU : constant Int := Ttypes.System_Storage_Unit;

   -----------------------
   -- Local Subprograms --
   -----------------------

   procedure Adjust_Esize_Alignment (E : Entity_Id);
   --  E is the entity for a type or object. This procedure checks that the
   --  size and alignment are compatible, and if not either gives an error
   --  message if they cannot be adjusted or else adjusts them appropriately.

   ----------------------------
   -- Adjust_Esize_Alignment --
   ----------------------------

   procedure Adjust_Esize_Alignment (E : Entity_Id) is
      Abits     : Int;
      Esize_Set : Boolean;

   begin
      --  Nothing to do if size unknown

      if Unknown_Esize (E) then
         return;
      end if;

      --  Determine if size is constrained by an attribute definition clause
      --  which must be obeyed. If so, we cannot increase the size in this
      --  routine.

      --  For a type, the issue is whether an object size clause has been
      --  set. A normal size clause constrains only the value size (RM_Size)

      if Is_Type (E) then
         Esize_Set := Has_Object_Size_Clause (E);

      --  For an object, the issue is whether a size clause is present

      else
         Esize_Set := Has_Size_Clause (E);
      end if;

      --  If size is known it must be a multiple of the byte size

      if Esize (E) mod SSU /= 0 then

         --  If not, and size specified, then give error

         if Esize_Set then
            Error_Msg_NE
              ("size for& not a multiple of byte size", Size_Clause (E), E);
            return;

         --  Otherwise bump up size to a byte boundary

         else
            Set_Esize (E, (Esize (E) + SSU - 1) / SSU * SSU);
         end if;
      end if;

      --  Now we have the size set, it must be a multiple of the alignment
      --  nothing more we can do here if the alignment is unknown here.

      if Unknown_Alignment (E) then
         return;
      end if;

      --  At this point both the Esize and Alignment are known, so we need
      --  to make sure they are consistent.

      Abits := UI_To_Int (Alignment (E)) * SSU;

      if Esize (E) mod Abits = 0 then
         return;
      end if;

      --  Here we have a situation where the Esize is not a multiple of
      --  the alignment. We must either increase Esize or reduce the
      --  alignment to correct this situation.

      --  The case in which we can decrease the alignment is where the
      --  alignment was not set by an alignment clause, and the type in
      --  question is a discrete type, where is is definitely safe to
      --  reduce the alignment. For example:

      --    t : integer range 1 .. 2;
      --    for t'size use 8;

      --  In this situation, the initial alignment of t is 4, copied from
      --  the Integer base type, but it is safe to reduce it to 1 at this
      --  stage, since we will only be loading a single byte.

      if Is_Discrete_Type (Etype (E))
        and then not Has_Alignment_Clause (E)
      then
         loop
            Abits := Abits / 2;
            exit when Esize (E) mod Abits = 0;
         end loop;

         Init_Alignment (E, Abits / SSU);
         return;
      end if;

      --  Now the only possible approach left is to increase the Esize
      --  but we can't do that if the size was set by a specific clause.

      if Esize_Set then
         Error_Msg_NE
           ("size for& is not a multiple of alignment",
            Size_Clause (E), E);

      --  Otherwise we can indeed increase the size to a multiple of alignment

      else
         Set_Esize (E, ((Esize (E) + (Abits - 1)) / Abits) * Abits);
      end if;
   end Adjust_Esize_Alignment;

   -------------------
   -- Layout_Object --
   -------------------

   procedure Layout_Object (E : Entity_Id) is
      T : constant Entity_Id := Etype (E);

   begin
      --  Nothing to do if backend does layout

      if Backend_Layout or Debug_Flag_FF then
         return;
      end if;

      --  Set size if not set for object and known for type. Use the
      --  RM_Size if that is known for the type and Esize is not.

      if Unknown_Esize (E) then
         if Known_Esize (T) then
            Set_Esize (E, Esize (T));

         elsif Known_RM_Size (T) then
            Set_Esize (E, RM_Size (T));
         end if;
      end if;

      --  Set alignment from type if unknown and type alignment known

      if Unknown_Alignment (E) and then Known_Alignment (T) then
         Set_Alignment (E, Alignment (T));
      end if;

      --  Make sure size and alignment are consistent

      Adjust_Esize_Alignment (E);

      --  Final adjustment, if we don't know the alignment, and the Esize
      --  was not set by an explicit Object_Size attribute clause, then
      --  we reset the Esize to unknown, since we really don't know it.

      if Unknown_Alignment (E)
        and then not Has_Size_Clause (E)
      then
         Set_Esize (E, Uint_0);
      end if;
   end Layout_Object;

   -----------------
   -- Layout_Type --
   -----------------

   procedure Layout_Type (E : Entity_Id) is
   begin
      --  For string literal types, for now, kill the size always, this
      --  is because gigi does not like or need the size to be set ???

      if Ekind (E) = E_String_Literal_Subtype then
         Set_Esize (E, Uint_0);
         Set_RM_Size (E, Uint_0);
         return;
      end if;

      --  For access types, set the size.  This is normally system address
      --  size, except for fat pointers (unconstrained array access types),
      --  where the size is two times the address size, to accomodate the
      --  two pointers that are required for a fat pointer (data and
      --  template). Note that E_Access_Protected_Subprogram_Type is not
      --  an access type for this purpose since it is not a pointer but is
      --  equivalent to a record. For access subtypes, copy the size from
      --  the base type since Gigi represents them the same way.

      if Is_Access_Type (E) then

         --  If Esize already set (e.g. by a size clause), then nothing
         --  further to be done here.

         if Known_Esize (E) then
            null;

         --  Access to subprogram is a strange beast, and we let the
         --  backend figure out what is needed (it may be some kind
         --  of fat pointer, including the static link for example.

         elsif Ekind (E) = E_Access_Protected_Subprogram_Type then
            null;

         --  For access subtypes, copy the size information from base type

         elsif Ekind (E) = E_Access_Subtype then
            Set_Size_Info (E, Base_Type (E));
            Set_RM_Size   (E, RM_Size (Base_Type (E)));

         --  For other access types, we use either address size, or, if
         --  a fat pointer is used (pointer-to-unconstrained array case),
         --  twice the address size to accomodate a fat pointer.

         else
            declare
               Desig : Entity_Id := Designated_Type (E);

            begin
               if Is_Private_Type (Desig)
                 and then Present (Full_View (Desig))
               then
                  Desig := Full_View (Desig);
               end if;

               if (Is_Array_Type (Desig)
                 and then not Is_Constrained (Desig)
                 and then not Has_Completion_In_Body (Desig)
                 and then not Debug_Flag_6)
               then
                  Init_Size (E, 2 * System_Address_Size);

                  --  Check for bad convention set

                  if Convention (E) = Convention_C
                       or else
                     Convention (E) = Convention_CPP
                  then
                     Error_Msg_N
                       ("?this access type does not " &
                        "correspond to C pointer", E);
                  end if;

               else
                  Init_Size (E, System_Address_Size);
               end if;
            end;
         end if;

      --  Scalar types: set size and alignment

      elsif Is_Scalar_Type (E) then

         --  For discrete types, the RM_Size and Esize must be set
         --  already, since this is part of the earlier processing
         --  and the front end is always required to layout the
         --  sizes of such types (since they are available as static
         --  attributes). All we do is to check that this rule is
         --  indeed obeyed!

         if Is_Discrete_Type (E) then

            --  If the RM_Size is not set, then here is where we set it.

            --  Note: an RM_Size of zero looks like not set here, but this
            --  is a rare case, and we can simply reset it without any harm.

            if not Known_RM_Size (E) then
               Set_Discrete_RM_Size (E);
            end if;

            --  If Esize for a discrete type is not set then set it

            if not Known_Esize (E) then
               declare
                  S : Int := 8;

               begin
                  loop
                     --  If size is big enough, set it and exit

                     if S >= RM_Size (E) then
                        Init_Esize (E, S);
                        exit;

                     --  If the RM_Size is greater than 64 (happens only
                     --  when strange values are specified by the user,
                     --  then Esize is simply a copy of RM_Size, it will
                     --  be further refined later on)

                     elsif S = 64 then
                        Set_Esize (E, RM_Size (E));
                        exit;

                     --  Otherwise double possible size and keep trying

                     else
                        S := S * 2;
                     end if;
                  end loop;
               end;
            end if;

         --  For non-discrete sclar types, if the RM_Size is not set,
         --  then set it now to a copy of the Esize if the Esize is set.

         else
            if Known_Esize (E) and then Unknown_RM_Size (E) then
               Set_RM_Size (E, Esize (E));
            end if;
         end if;

         --  For scalar types, we calculate the alignment as the largest power
         --  of two multiple of System.Storage_Unit that does not exceed either
         --  the actual size of the type, or the maximum required alignment

         --  However, skip this for packed array types, since those will be
         --  handled in the back end entirely. Also skip this if the alignment
         --  has been explicitly set

         if not Is_Packed_Array_Type (E)
           and then not Has_Alignment_Clause (E)
         then
            declare
               S : constant Int :=
                     UI_To_Int (Esize (E)) / SSU;
               A : Int;

            begin
               A := 1;
               while 2 * A <= Ttypes.Maximum_Alignment
                  and then 2 * A <= S
               loop
                  A := 2 * A;
               end loop;

               --  Now we think we should set the alignment to A, but we
               --  skip this if an alignment is already set to a value
               --  greater than A (happens for derived types).

               --  However, if the alignment is known and too small it
               --  must be increased, this happens in a case like:

               --     type R is new Character;
               --     for R'Size use 16;

               --  Here the alignment inherited from Character is 1, but
               --  it must be increased to 2 to reflect the increased size.

               if Unknown_Alignment (E) or else Alignment (E) < A then
                  Init_Alignment (E, A);
               end if;
            end;
         end if;

      --  Non-primitive types

      else
         --  If RM_Size is known, set Esize if not known

         if Known_RM_Size (E) and then Unknown_Esize (E) then

            --  If the alignment is known, we bump the Esize up to the
            --  next alignment boundary if it is not already on one.

            if Known_Alignment (E) then
               declare
                  A : constant Uint := Alignment_In_Bits (E);
                  S : constant Uint := RM_Size (E);

               begin
                  Set_Esize (E, (S * A + A - 1) / A);
               end;
            end if;

         --  If Esize is set, and RM_Size is not, RM_Size is copied from
         --  Esize at least for now this seems reasonable, and is in any
         --  case needed for compatibility with old versions of gigi.
         --  look to be unknown.

         elsif Known_Esize (E) and then Unknown_RM_Size (E) then
            Set_RM_Size (E, Esize (E));
         end if;

         --  For array base types, set component size if object size of
         --  the component type is known and is a small power of 2 (8,
         --  16, 32, 64), since this is what will always be used.

         if Ekind (E) = E_Array_Type
           and then Unknown_Component_Size (E)
         then
            declare
               CT : constant Entity_Id := Component_Type (E);

            begin
               --  For some reasons, access types can cause trouble,
               --  So let's just do this for discrete types ???

               if Present (CT)
                 and then Is_Discrete_Type (CT)
                 and then Known_Static_Esize (CT)
               then
                  declare
                     S : constant Uint := Esize (CT);

                  begin
                     if S = 8  or else
                        S = 16 or else
                        S = 32 or else
                        S = 64
                     then
                        Set_Component_Size (E, Esize (CT));
                     end if;
                  end;
               end if;
            end;
         end if;
      end if;
   end Layout_Type;

   --------------------------
   -- Set_Discrete_RM_Size --
   --------------------------

   procedure Set_Discrete_RM_Size (Def_Id : Entity_Id) is
      FST : constant Entity_Id := First_Subtype (Def_Id);

   begin
      --  All discrete types except for the base types in standard
      --  are constrained, so indicate this by setting Is_Constrained.

      Set_Is_Constrained (Def_Id);

      --  We set generic types to have an unknown size, since the
      --  representation of a generic type is irrelevant, in view
      --  of the fact that they have nothing to do with code.

      if Is_Generic_Type (Root_Type (FST)) then
         Set_RM_Size (Def_Id, Uint_0);

      --  If the subtype statically matches the first subtype, then
      --  it is required to have exactly the same layout. This is
      --  required by aliasing considerations.

      elsif Def_Id /= FST and then
        Subtypes_Statically_Match (Def_Id, FST)
      then
         Set_RM_Size   (Def_Id, RM_Size (FST));
         Set_Size_Info (Def_Id, FST);

      --  In all other cases the RM_Size is set to the minimum size.
      --  Note that this routine is never called for subtypes for which
      --  the RM_Size is set explicitly by an attribute clause.

      else
         Set_RM_Size (Def_Id, UI_From_Int (Minimum_Size (Def_Id)));
      end if;
   end Set_Discrete_RM_Size;

end Layout;
