------------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--                     S Y S T E M . T R A C E B A C K                      --
--                       (H P / U X  V E R S I O N )                        --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                            $Revision: 1.1 $
--                                                                          --
--           Copyright (C) 1999-2000 Ada Core Technologies, Inc.            --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT;  see file COPYING.  If not, write --
-- to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, --
-- MA 02111-1307, USA.                                                      --
--                                                                          --
-- As a special exception,  if other files  instantiate  generics from this --
-- unit, or you link  this unit with other files  to produce an executable, --
-- this  unit  does not  by itself cause  the resulting  executable  to  be --
-- covered  by the  GNU  General  Public  License.  This exception does not --
-- however invalidate  any other reasons why  the executable file  might be --
-- covered by the  GNU Public License.                                      --
--                                                                          --
-- GNAT was originally developed  by the GNAT team at  New York University. --
-- It is now maintained by Ada Core Technologies Inc (http://www.gnat.com). --
--                                                                          --
------------------------------------------------------------------------------

package body System.Traceback is

   pragma Linker_Options ("/usr/lib/libcl.a");
   --  Needed for definitions of U_*

   type UWREC is record
      cur_frsize : Address;  --  Frame size of current routine.
      cursp      : Address;  --  The current value of stack pointer.
      currls     : Address;  --  PC-space of the calling routine.
      currlo     : Address;  --  PC-offset of the calling routine.
      curdp      : Address;  --  Data Pointer of the current routine.
      toprp      : Address;  --  Initial value of RP.
      topmrp     : Address;  --  Initial value of MRP.
      topsr0     : Address;  --  Initial value of sr0.
      topsr4     : Address;  --  Initial value of sr4.
      r3         : Address;  --  Initial value of gr3.
      cur_r19    : Address;  --  GR19 value of the calling routine.
   end record;

   type PREVREC is record
      prev_frsize : Address;  --  frame size of calling routine.
      prevSP      : Address;  --  SP of calling routine.
      prevRLS     : Address;  --  PC_space of calling routine's caller.
      prevRLO     : Address;  --  PC_offset of calling routine's caller.
      prevDP      : Address;  --  DP of calling routine.
      udescr0     : Address;  --  low word of calling routine's unwind desc.
      udescr1     : Address;  --  high word of calling routine's unwind desc.
      ustart      : Address;  --  start of the unwind region.
      uw_index    : Address;  --  index into the unwind table.
      uend        : Address;  --  end of the unwind region.
      prev_r19    : Address;  --  GR19 value of the caller's caller.
   end record;

   type MState (Discr : Boolean := False) is record
      case Discr is
         when False => Current  : UWREC;
         when True  => Previous : PREVREC;
      end case;
   end record;
   pragma Unchecked_Union (MState);

   type MState_Ptr is access all MState;

   procedure U_get_frame_info (Frame : MState_Ptr);
   pragma Import (C, U_get_frame_info, "U_get_frame_info");

   function U_get_previous_frame
     (current_frame  : MState_Ptr;
      previous_frame : MState_Ptr)
      return Integer;
   pragma Import (C, U_get_previous_frame, "U_get_previous_frame");

   ----------------
   -- Call_Chain --
   ----------------

   procedure Call_Chain
     (Traceback : System.Address;
      Max_Len   : Natural;
      Len       : out Natural;
      Exclude_Min,
      Exclude_Max : System.Address := System.Null_Address)
   is
      type Tracebacks_Array is array (1 .. Max_Len) of System.Address;
      pragma Suppress_Initialization (Tracebacks_Array);

      Asm_Call_Size : constant := 3;
      --  Code_Loc returned by currlo is the return point but here we
      --  want to return the call point. Under HP-UX a call
      --  asm instruction takes 3 bytes. So we must remove this value from
      --  currlo to have the call point.

      Frame       : aliased MState;
      Code        : System.Address;
      J           : Natural := 1;
      Pop_Success : Boolean;
      Trace       : Tracebacks_Array;
      for Trace'Address use Traceback;

      ---------------
      -- Pop_Frame --
      ---------------

      function Pop_Frame (Frame : MState_Ptr) return Boolean;

      function Pop_Frame (Frame : MState_Ptr) return Boolean is
         Tmp_Frame : aliased MState;

      begin
         if U_get_previous_frame (Frame, Tmp_Frame'Unchecked_Access) /= 0 then
            return False;
         end if;

         Frame.Current.cur_frsize := Tmp_Frame.Previous.prev_frsize;
         Frame.Current.cursp      := Tmp_Frame.Previous.prevSP;
         Frame.Current.currls     := Tmp_Frame.Previous.prevRLS;
         Frame.Current.currlo     := Tmp_Frame.Previous.prevRLO and not (3);
         Frame.Current.curdp      := Tmp_Frame.Previous.prevDP;
         return True;
      end Pop_Frame;

   begin
      U_get_frame_info (Frame'Unchecked_Access);
      Frame.Current.topsr0  := 0;
      Frame.Current.topsr4  := 0;
      Pop_Success := Pop_Frame (Frame'Unchecked_Access);

      loop
         Code := Frame.Current.currlo;

         exit when not Pop_Success or else Code = 0 or else J = Max_Len + 1;

         Code := Code - Asm_Call_Size;

         if Code < Exclude_Min or else Code > Exclude_Max then
            Trace (J) := Code;
            J := J + 1;
         end if;

         Pop_Success := Pop_Frame (Frame'Unchecked_Access);
      end loop;

      Len := J - 1;
   end Call_Chain;

   ------------------
   -- C_Call_Chain --
   ------------------

   function C_Call_Chain
     (Traceback   : System.Address;
      Max_Len     : Natural) return Natural
   is
      Val : Natural;
   begin
      Call_Chain (Traceback, Max_Len, Val);
      return Val;
   end C_Call_Chain;

end System.Traceback;
