#include <unistd.h>
#include <netinet/in.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <fcntl.h>

#include <signal.h>

#include <ctype.h>

#include <time.h>

#include <string.h>

#include "gnapster.h"
#include "upload.h"
#include "chan.h"
#include "xtext.h"

#include "debug.h"

#undef read_file
#undef read_file_with_n

extern GnapsterMain *gmain;
extern UserInfo user_info;
extern GList *hooks;

extern GdkPixmap *speedgreen, *speedyellow, *speedred, *speedgray;
extern GdkBitmap *speedgreenb, *speedyellowb, *speedredb, *speedgrayb;

GList *log_hooks = NULL;

GCompletion *jcomp = NULL;

#define FREE_SETUP(x,y) x *y; if (!data) return; y = data

FreeTable gnapster_free[] = {
  /* Type               Function           Size */
   { N_SEARCH,		search_free,       sizeof(Search) },
   { TRANSFER,	        transfer_free,     sizeof(Transfer) },
   
   { SERVER,		server_free,       sizeof(Server) },
   { N_USER,		user_free,         sizeof(User) },
   { N_CHAN_INFO,       channel_free,      sizeof(ChannelInfo) },
   { N_RESUME,		resume_free,       sizeof(Resume) },
   
   { W_PROP,		property_free,     sizeof(WidgetProperty) },
   
   { HOOK,              hook_free,         sizeof(ThemeHook) },
   { THEME,             theme_free,        sizeof(Theme) },
   
   { FILE_ENTRY,        file_entry_free,   sizeof(FileEntry) },

   { SEARCH_TAB,        search_tab_free,   sizeof(SearchTab) },
   { BROWSE_TAB,        browse_tab_free,   sizeof(BrowseTab) },
   { DOWNLOAD_TAB,      download_tab_free, sizeof(DownloadTab) },
   { UPLOAD_TAB,        upload_tab_free,   sizeof(UploadTab) },
   { CONSOLE_TAB,       console_tab_free,  sizeof(ConsoleTab) },
   { MOTD_TAB,          motd_tab_free,     sizeof(MOTDTab) },
   
   { STAB,              stab_free,         sizeof(STab) },
   
   { CONN_INFO,         conn_info_free,    sizeof(ConnInfo) },
   
   { GMAIN,             gmain_free,        sizeof(GnapsterMain) },

   { CBPACK,            cbpack_free,       sizeof(CBPack) },
   
   { ACCOUNT,           account_free,      sizeof(Account) },
   
   { SHARE,             share_free,        sizeof(ShareData) },
   
   { CONNECTION,        connection_free,   sizeof(Connection) },
   
   { -1,		NULL,             -1 },
};

void tint_color(GdkColor *c, int r, int g, int b) {
   c->red = (c->red * r) >> 8;
   c->green = (c->green * g) >> 8;
   c->blue = (c->blue * b) >> 8;
}

/* determines a valid output method */
FILE *get_file_handle() {
   char *tty;
   FILE *f;
   
   tty = ttyname(fileno(stdin));
   f = (tty) ? stderr : fopen("./debug.out", "w");
   
   return f ? f : stdout;
}

void close_file_handle(FILE *f) {
   if (f == stdin || f == stdout || f == stderr)
     return;
   
   fclose(f);
}

/* taken from the Epplet lib */
void setup_sigchld() {
   struct sigaction sa;
   
   sa.sa_handler = handle_child;
   sa.sa_flags = SA_RESTART;
   
   sigemptyset(&sa.sa_mask);
   
   sigaction(SIGCHLD, &sa, (struct sigaction *)0);
}

void handle_child(int num) {
   int status;
   pid_t pid;
   
   while((pid = waitpid(-1, &status, WNOHANG)) > 0);
}

void bg_exec(char *cmd) {
   int cpid;
   
   fprintf(stderr, "attempting to exec %s\n", cmd);
   
/*   ocmd = next_arg(cmd, &cmd);
   while((cmdlist[i++] = next_arg(cmd, &cmd)));
   cmdlist[i++] = NULL;*/
   
   cpid = fork();
   
   if (cpid == 0) {
      execl("/bin/sh", "sh", "-c", cmd, NULL);
      
      _exit(0);
   }
}

void set_conn_info(STab *stab, char *ip, unsigned short port, char *server, Account *acnt) {
   d_assert(ip != NULL);
   d_assert(port > 0);
   d_assert(server != NULL);
   
   /* if we have already had a connection on this stab, we need to be sure
    * we remove it from the menu */
   remove_server_menu();
   
   d_free(stab->ci->ip);
   d_free(stab->ci->server);
   j_free(ACCOUNT, stab->ci->account);
   
   stab->ci->ip = d_strdup(ip);
   stab->ci->port = port;
   stab->ci->server = d_strdup(server);
   stab->ci->account = acnt ? acnt : get_account(NULL);
   
   append_server_menu();
   
   /* backwards compat */
   j_config_set_string("/gnapster/Options/server", ip);
   j_config_set_int("/gnapster/Options/server_port", port);
   j_config_sync();
}

void unshare_files(STab *stab) {
   napster_send(stab->ci->sock, CMDS_REMOVEALLFILES, NULL);
   
   gtk_label_set_text(GTK_LABEL(gmain->ut->label), "Sharing is disabled");
   hook_text_insert(stab, CURR, SYSTEM, "general_message", "%s",
		    "Unsharing all files...");
   
   stab->ci->state &= ~SHARING_MASK;
}

void transfer_init(Transfer *t) {
   t->sock = -1;
   t->sock_input = -1;
   
   t->update_tag = -1;
   
   t->accept_timeout = -1;
   t->start_timeout = -1;
   
   t->timeout_tag = -1;
}

void rem_trail(char *s, char t) {
   int x;
   
   if (!s)
     return;

   x = strlen(s);
   
   while(s[--x] == t)
     s[x] = 0;
}

char *translate_args(char *args, int argcount) {
   char *s, *arg, *adup;
   
   s = d_strdup("");
   
   /* preserve the original data */
   adup = d_strdup(args);
   args = adup;
   
   for(; argcount; argcount--) {
      if (argcount > 1)
	arg = next_arg(args, &args);
      else
	arg = last_arg(args, &args);
      
      d_strexp(&s, "%s%s\4", s, arg);
   }
   
   d_free(adup);
   
   return s;
}

void get_xfer_stats(STab *stab, unsigned long *d, unsigned long *u, unsigned long *td, unsigned long *tu, float *d_kps, float *u_kps, float *td_kps, float *tu_kps) {
   int tdx = 0, tux = 0;
   time_t ttime;
   
   *d = *u = 0;
   *td = *tu = 0;
   *d_kps = *u_kps = 0.0;
   
   ttime = time(NULL);
   ttime -= gmain->clistart;
   
   /* total incoming/outgoing divided by total client time, server
    * independant */
   *td_kps = (((float)gmain->in * 1024.0) / (float)ttime);
   *tu_kps = (((float)gmain->out * 1024.0) / (float)ttime);
   
   if (stab) {
      *d = stab->ci->download_size;
      *u = stab->ci->upload_size;
      *td = stab->ci->download_xfers;
      *tu = stab->ci->upload_xfers;
      *d_kps = stab->ci->download_kps;
      *u_kps = stab->ci->upload_kps;
   } else {
      FORSTABS(*d += stab->ci->download_size;
	       *u += stab->ci->upload_size;
	       *td += stab->ci->download_xfers;
	       *tu += stab->ci->upload_xfers;
	       *d_kps += stab->ci->download_kps;
	       *u_kps += stab->ci->upload_kps;
	       if (stab->ci->download_xfers) tdx++;
	       if (stab->ci->upload_xfers) tux++);
      
      if (tdx)
	*d_kps /= tdx;
      if (tux)
	*u_kps /= tux;
   }
}

void set_tog_widgets(GtkItemFactory *ifact) {
   GtkWidget *w;

#ifdef HAVE_GNOME
   GtkWidget *shell;
   int pos;
   
   shell = gnome_app_find_menu_pos(GNOME_APP(gmain->window)->menubar,
			       "Servers/Show Server Tabs",
			       &pos);
   if (!pos)
     return;
   
   w = g_list_nth_data(GTK_MENU_SHELL(shell)->children, pos - 1);
#else
   w = gtk_item_factory_get_widget(GTK_ITEM_FACTORY(ifact),
				   "/Servers/Show Server Tabs");
#endif /* HAVE_GNOME */
   
   gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(w),
				  user_info.conf[SHOW_STABS]);
}

void char_list_free(GList *list) {
   
   for(; list; list=list->next)
     if (list->data)
       d_free(list->data);
   
   g_list_free(list);
}

int sort_get_data(GtkCList *clist, const void *p1, const void *p2, char **t1, char **t2) {
   GtkCListRow *r1, *r2;
   int column;

   column = clist->sort_column;
   
   r1 = (GtkCListRow *) p1;
   r2 = (GtkCListRow *) p2;
   
   if (t1)
     *t1 = GTK_CELL_TEXT(r1->cell[column])->text;
   
   if (t2)
     *t2 = GTK_CELL_TEXT(r2->cell[column])->text;
   
   return column;
}

int int_sort(GtkCList *clist, const void *p1, const void *p2) {
   char *t1, *t2;
   int i1, i2;
   
   sort_get_data(clist, p1, p2, &t1, &t2);
   
   convert(t1, "%i", &i1);
   convert(t2, "%i", &i2);
   
   if (i1 <= i2) 
     return 1;
   
   if (i1 > i2)
     return -1;
   
   return 0;
}

int string_sort(GtkCList *clist, const void *p1, const void *p2) {
   char *t1, *t2;
   
   sort_get_data(clist, p1, p2, &t1, &t2);
   
   return j_strcasecmp(t1, t2);
}

void *get_sort_function(char c) {
   if (c == 'i')
     return int_sort;
   
   if (c == 's')
     return string_sort;
   
   return NULL;
}

void column_sort(GtkWidget *w, int column, void *data) {
   char *ptrn;
   int ptrn_len;
   GtkCList *clist;
   
   d_assert(data != NULL);
   
   ptrn = data;
   ptrn_len = strlen(ptrn);
   
   if (column >= ptrn_len || column < 0)
     return;
   
   if (GTK_IS_CTREE(w))
     clist = GTK_CLIST(GTK_CTREE(w));
   else
     clist = GTK_CLIST(w);
   
   if (column == clist->sort_column) {
      if (clist->sort_type == GTK_SORT_ASCENDING)
	clist->sort_type = GTK_SORT_DESCENDING;
      else
	clist->sort_type = GTK_SORT_ASCENDING;
   } else
     gtk_clist_set_compare_func(clist, get_sort_function(ptrn[column]));

   gtk_clist_set_sort_column(clist, column);
   
   if (GTK_IS_CTREE(w))
     gtk_ctree_sort_node(GTK_CTREE(w), NULL);
   else if (GTK_IS_CLIST(w))
     gtk_clist_sort(clist);
}

int j_strlen(char *s) {
   if (!s)
     return 0;
   
   return strlen(s);
}

int j_strcmp(char *s1, char *s2) {
   if (!s1 && !s2)
     return -1;
   
   if (!s1 && s2)
     return -1;
   
   if (s1 && !s2)
     return 1;
   
   return (strcmp(s1, s2));
}

int j_strcasecmp(char *s1, char *s2) {
   if (!s1 && !s2)
     return -1;
   
   if (!s1 && s2)
     return -1;
   
   if (s1 && !s2)
     return 1;
   
   return (strcasecmp(s1, s2));
}

Account *get_account(char *user) {
   FILE *f;
   GList *felem, *ptr;
   FileEntry *fent;
   Account *acnt = NULL;
   char *dptr, *u, *p;
   int primary;
   
   f = open_local_path("accounts", NULL, "r");
   if (!f)
     return NULL;
   
   felem = read_file(DEBUG f);
   
   fclose(f);
   
   for(ptr=felem; ptr; ptr=ptr->next) {
      fent = ptr->data;
      if (!fent)
	continue;
      
      dptr = fent->lptr;
      
      u = next_arg_full(dptr, &dptr, ':');
      p = next_arg(dptr, &dptr);
      convert(next_arg(dptr, &dptr), "%i", &primary);
      
      NA_ERR_HANDLE_BREAK();
      
      if (user && strcmp(user, u))
	continue;
      if (!user && !primary)
	continue;
      
      acnt = d_new(ACCOUNT);
      
      acnt->user = d_strdup(u);
      acnt->pass = d_strdup(p);
      acnt->primary = primary;
      
      break;
   }
   
   read_file_free(felem);
   
   return acnt;
}

int get_user_info() {
   user_info.download_dir = d_config_get_string("/gnapster/User/dl_dir");
   user_info.upload_dir = d_config_get_string("/gnapster/User/ul_dir");
   user_info.connection = j_config_get_int("/gnapster/User/connection");
   
   return (user_info.download_dir && user_info.upload_dir && user_info.connection >= 0);
}

char *str_trans(char *s, char o, char n) {
   char *ptr;
   
   for(ptr=s; *ptr; ptr++)
     if (*ptr == o)
       *ptr = n;
   
   return s;
}

char *complete_data(void *data) {
   return all_lower(data);
}

void j_complete_free() {
   if (!jcomp)
     return;
   
   g_completion_free(jcomp);
   
   jcomp = NULL;
}

char *j_complete(GList *list, GList **ret, char *s) {
   GCompletion *comp;
   GList *c;
   char *pfx;
   
   if (!list || !s || !*s)
     return NULL;
   
   s = d_strdup(all_lower(s));
   
   comp = g_completion_new(complete_data);
   
   g_completion_add_items(comp, list);
   
   c = g_completion_complete(comp, s, &pfx);
   if (pfx)
     d_hunk_add(pfx, strlen(pfx) + 1);

   d_free(s);
   
   if (!c || !pfx) { /* if c is NULL, pfx will be too, so dont have to free */
      g_completion_free(comp);
      
      return NULL;
   }

   /* if there is more than one, fill ret */
   if (ret)
     *ret = (g_list_length(c) > 1) ? c : NULL;
   
   d_free(pfx);
   
   g_list_free(list);
   
   jcomp = comp;
   
   return c->data;
}

GList *j_comp_items(STab *stab, GtkCList *clist) {
   GList *ptr, *ret = NULL;
   User *dptr;
   
   for(ptr=clist->row_list; ptr; ptr=ptr->next) {
       dptr = GTK_CLIST_ROW(ptr)->data;
      if (!dptr)
	break;
      
      /* don't add yourself */
      if (dptr->user && strcmp(dptr->user, stab->ci->account->user))
	ret = g_list_append(ret, dptr->user);
   }
   
   return ret;
}

GtkWidget *find_prop(STab *stab, int key) {
   GList *ptr;
   GtkWidget *w;
   WidgetProperty *prop;
   
   for(ptr=stab->pwidgets; ptr; ptr=ptr->next) {
      w = ptr->data;
      if (!w)
	continue;
      
      prop = gtk_object_get_data(GTK_OBJECT(w), "prop");
      if (!prop)
	continue;
      
      if (prop->key == key)
	return w;
   }
   
   return NULL;
}

void get_conn_pixmap(int c, GdkPixmap **p, GdkBitmap **b) {
   switch(c) {
    case 1: case 2: case 3: case 4:
      *p = speedred;
      *b = speedredb;
      break;
    case 5: case 6:
      *p = speedyellow;
      *b = speedyellowb;
      break;
    case 7: case 8: case 9: case 10:
      *p = speedgreen;
      *b = speedgreenb;
      break;
    default:
      *p = speedgray;
      *b = speedgrayb;
      break;
   }
}

void pending_upload(STab *stab, char *user, char *file) {
   Transfer *t;
   
   t = d_new(TRANSFER);
   transfer_init(t);
   
   t->user = d_strdup(user);
   t->file = d_strdup(file);
   
   stab->ut->p_up = g_list_prepend(stab->ut->p_up, t);
}

void remove_pending_upload(Transfer *upload) {
   GList *iter_ptr;
   Transfer *pending;
   
   ITER_LIST(upload->st->ut->p_up) {
      LIST_DATA(pending);
      
      if (!j_strcmp(pending->user, upload->user) &&
	  !j_strcmp(pending->file, upload->file)) {
	 LIST_REMOVE(upload->st->ut->p_up, pending);	 
	 j_free(TRANSFER, pending);
	 
	 break;
      }
   }
}

int is_upload(STab *stab, char *user, char *file) {
   GList *iter_ptr;
   Transfer *pending;
   
   ITER_LIST(stab->ut->p_up) {
      LIST_DATA(pending);

      if (!j_strcmp(pending->user, user) &&
	  !j_strcmp(pending->file, file))
	return 1;
   }   
   
   return 0;
}

int total_conn() {
   int i = 0;
  
   FORSTABS(if (connected(stab->ci)) i++);
   
   return i;
}

GtkWidget *find_stab_vbox(GtkWidget *w) {
   GtkWidget *vbox, *ptr;
   
   vbox = NULL;
   
   for(ptr=w; ptr; ptr=ptr->parent) {
      if (ptr == gmain->srv_notebook)
	return vbox;
      
      vbox = ptr;
   }
   
   return NULL;
}

void conn_info_init(ConnInfo *ci) {
   ci->sock = -1;   
}

char *true_id(char *id) {
   char *pos;
   
   if ((pos = strchr(id, '-')))
     *pos = 0;
   
   return d_strdup(id);
}

void j_error(char *func, ...) {
   va_list args;
   char *arg, *prev;
   
   if (!strcmp(func, "fopen"))
     return;
   
   fprintf(stderr, "%s(", func);
   
   arg = prev = NULL;
   
   va_start(args, func);
   for(;;) {
      arg = va_arg(args, char *);
      if (!arg)
	break;
      
      fprintf(stderr, "%s%s", (prev) ? "," : "", arg);
      
      prev = arg;
   }
   va_end(args);
   
   fprintf(stderr, "): %s\n", strerror(errno));
}

void print_bits(int x) {
   int i;
   
   for(i=0; i < 32; i++) {
      printf("%i", (x & (1 << (31 - i))) ? 1 : 0);
      if (i % 4 == 3)
	printf(" ");
   }
   printf("\n");
}

/* can accept both \ and / directory paths */
char *trunc_file(char *path) {
   char *file;
   
   if (!path)
     return NULL;

   file = strrchr(path, '\\');
   if (!file && (!(file = strrchr(path, '/'))))
     return path;
   
   /* push past the '\' and return */
   return (file + 1);
}

GList *remove_nick(GList *nick_history, char *user) {
   GList *ptr, *link, *ret;
   char *dptr;
   
   link = NULL;
   
   for(ptr=nick_history; ptr; ptr=ptr->next) {
      dptr = ptr->data;
      if (!dptr)
	continue;
      
      if (!strcasecmp(dptr, user))
	link = ptr;
      
      if (link)
	break;
   }
   
   if (link) {
      d_free(link->data);
      link->data = NULL;
      
      ret = g_list_remove_link(nick_history, link);
      return ret;
   }
   
   return nick_history;
}

STab *get_stab(int pn) {
   if (pn < 0)
     return NULL;
   
   return g_list_nth_data(ttbl, pn);
}

void load_stabs() {
   FILE *f;
   STab *stab;
   GList *felem, *ptr;
   FileEntry *fent;
   char *ip, *dptr, *server, *username;
   unsigned short port;
   int count = 0;
   
   f = open_stabs("r");
   if (!f)
     return;
   
   felem = read_file(DEBUG f);
   
   fclose(f);
   
   for(ptr=felem; ptr; ptr=ptr->next) {
      fent = ptr->data;
      if (!fent)
	continue;
      
      dptr = fent->lptr;
      
      ip = next_arg_full(dptr, &dptr, ':');
      convert(next_arg(dptr, &dptr), "%hu", &port);
      server = next_arg(dptr, &dptr);
      username = next_arg(dptr, &dptr);
      
      NA_ERR_HANDLE_BREAK();
      
      stab = (count) ? append_server_tab() : ((STab *)(ttbl->data));
      
      stab->ci->ip = d_strdup(ip);
      stab->ci->port = port;
      stab->ci->server = d_strdup(server);
      stab->ci->account = get_account(username);
      if (!stab->ci->account) {
	 j_error_dialog("Failed to locate the account associated with this server!");
	 fprintf(stderr, "error opening user account %s, falling back to primary account\n", username);
	 stab->ci->account = get_account(NULL);
	 if (!stab->ci->account) {
	    fprintf(stderr, "failure to open default account!  dieing!\n");
	    return;
	 }
      }
      
      connect_cb(stab);
      
      count++;
   }
   
   read_file_free(felem);
}

STab *get_current_stab() {
   int pn;
   STab *stab;
   
   if (!ttbl)
     return NULL;
   
   pn = gtk_notebook_get_current_page(GTK_NOTEBOOK(gmain->srv_notebook));
   if (pn < 4)
     return ttbl->data;
   
   stab = get_stab(pn - 4);
   
   return stab;
}

ConnInfo *get_conn_info(int pn) {
   STab *stab;
   
   stab = g_list_nth_data(ttbl, pn);
   if (!stab)
     return NULL;
   
   return (stab->ci);
}

int cancels(ConnInfo *ci) {
   return ((ci->state & CANCELS_MASK) >> 8);
}

int cancels_inc(ConnInfo *ci, int inc) {
   unsigned char c;
   
   c = (ci->state & CANCELS_MASK) >> 8;
   
   c += inc;
   
   ci->state &= ~CANCELS_MASK;
   ci->state |= (c << 8);
   
   return c;
}

int debug(ConnInfo *ci) {
   return (ci->state & DEBUG_MASK);
}

int connected(ConnInfo *ci) {
   return (ci->state & CONNECTED_MASK);
}

int connecting(ConnInfo *ci) {
   return (ci->state & CONNECTING_MASK);
}

int searching(ConnInfo *ci) {
   return (ci->state & SEARCHING_MASK);
}

int browsing(ConnInfo *ci) {
   return (ci->state & BROWSING_MASK);
}

int sharing(ConnInfo *ci) {
   return (ci->state & SHARING_MASK);
}

int safe_exit(ConnInfo *ci) {
   return (ci->state & SAFE_EXIT_MASK);
}

char *global_path(char *subdir, char *file) {
   char *path;
   
   if (file)
     d_msprintf(&path, "%s/%s/%s", CONFIG_PATH, subdir, file);
   else
     d_msprintf(&path, "%s/%s", CONFIG_PATH, subdir);
   
   return path;
}

char *local_path(char *subdir, char *file) {
   char *path;
   
   if (file)
     d_msprintf(&path, "%s/.gnapster/%s/%s", getenv("HOME"), subdir, file);
   else
     d_msprintf(&path, "%s/.gnapster/%s", getenv("HOME"), subdir);
   
   return path;
}

int is_comment(char *text) {
   char *ptr;
   
   ptr = text;
   while(*ptr == ' ')
     ptr++;
  
   return (*ptr == '#');
}

char *hack_n(char *s) {
   return hack_chr(s, '\n');
}

char *hack_chr(char *s, char c) {
   char *ptr;
   
   if (!s)
     return NULL;
   
   ptr = s;
   
   while(*ptr) {
      if (*ptr == c) {
	*ptr = 0;
	break;
      }
      
      ptr++;
   }
   
   return s;
}

void read_file_free(GList *list) {
   GList *ptr;
   FileEntry *fent;
   
   for(ptr=list; ptr; ptr=ptr->next) {
      fent = ptr->data;
      if (!fent)
	continue;
      
      j_free(FILE_ENTRY, fent);
      ptr->data = NULL;
   }
   
   g_list_free(list);
}

GList *read_file_full(const char *file, int line, FILE *f, int allow_n) {
   char buf[4096];
   GList *list = NULL;
   FileEntry *fent;
   
   for(;;) {
      fgets(buf, sizeof(buf) - 1, f);
      if (feof(f)) 
	break;
      
      if (!buf[0])
	continue;
      
      if (allow_n || buf[0] != '\n') {
	 fent = d_new(FILE_ENTRY);
	 
	 /* this is actually setup like this so next arg can be used on
	  * fent->lptr, and keep fent->line at the right memory address */
	 fent->line = d_strdup(buf);
	 fent->lptr = fent->line;
	 
	 list = g_list_append(list, fent);
      }
   }

   return list;
}

GList *read_file(const char *file, int line, FILE *f) {
   GList *list;
   
   list = read_file_full(file, line, f, 0);
   
   return list;
}

GList *read_file_with_n(const char *file, int line, FILE *f) {
   GList *list;
   
   list = read_file_full(file, line, f, 1);
   
   return list;
}

size_t struct_size(int t) {
   int i;
   
   for(i=0; gnapster_free[i].func; i++) 
     if (gnapster_free[i].type == t)
       return gnapster_free[i].size;
   
   return -1;
}

int get_type(void *data) {
   int t = -1;

   if (data)
     t = ((Type *)data)->type;

   return t;
}

void set_type(void *data, int t) {
   if (t < 0 || !data)
     return;
   
   ((Type *)data)->type = t;
}

char *all_lower(char *s) {
   int x;
   static char ret[1024 * 6];
   char *ptr;
   
   if (!s)
     return NULL;

   x = strlen(s);
   if (x <= 0)
     return NULL;

   if (x >= sizeof(ret))
     return s;
   
   for(ptr=ret; *s; s++)
     *ptr++ = tolower(*s);
   
   *ptr++ = 0;

   return ret;
}

void conv_file(char *s) {
   char *ptr;

   if (!user_info.conf[NAME_FIX])
     return;

   ptr = strrchr(s, '/');
   if (!ptr)
     return;
   while(*ptr) {
      if (*ptr == ' ') *ptr = '_';
      ptr++;
   }
}

int file_exists(char *file) {
   struct stat st;
   int x;

   x = stat(file, &st);

   return (x >= 0);
}

char *next_arg_full_max_length(char *s, char **next, char end, int max) {
   char *ptr, *str;
   
   na_err = (!s || !(*s));
   if (na_err)
     return NULL;

   ptr = s;
   
   str = (end == '"') ? 
     ++ptr : ptr;
   
   while(*str && *str != end && *str != '\n' && max)
     str++, max--;
   
   if (*str) {
      *str++ = 0;
      if (end == '"') str++;
   }

   if (next)
     *next = str;

   return ptr;
}

char *next_arg_full(char *s, char **next, char end) {
   return next_arg_full_max_length(s, next, end, -1);
}

char *next_arg_first(char *s, char **next, char **first) {
   *first = NULL;

   if (!s || !*s) {
      na_err = 1;
      return NULL;
   }

   *first = s;

   return next_arg(s, next);
}

char *next_arg(char *s, char **next) {
   char cmd;

   if (!s || !*s) {
      na_err = 1;
      return NULL;
   }

   cmd = (*s == '"') ? '"' : ' ';

   return next_arg_full(s, next, cmd);
}

char *last_arg(char *s, char **next) {
   return next_arg_full(s, next, 0);
}

void convert(char *s, char *type, void *c) {
   char *ptr;
   
   if (!s)
     return;
   
   ptr = s;
   
   while(*ptr == '0' && *(ptr + 1) != 0)
     ptr++;

   if (ptr)
     sscanf(ptr, type, c);
}
      
long my_atol(const char *nptr) {
   if (!nptr) return -1;

   return atol(nptr);
}

int my_atoi(const char *nptr) {
   if (!nptr) return -1;
   
   return atoi(nptr);
}

char *simplify_size(unsigned long int size) {
   float new_size;
   int div_count;
   static char buf[25] = { 0 };
   char lbl, div_table[4] = { 'b', 'k', 'M', ' ' };
   
   new_size = size;
   div_count = 0;
   
   while(new_size >= 1024) {
      new_size /= 1024;
      div_count++;
   }
   
   lbl = div_table[div_count];

   g_snprintf(buf, sizeof(buf), "%.02f%c", new_size, lbl);
   
   return buf;
}

void mp3_list_free(STab *stab) {
   GList *ptr, *child, *cptr;
   ShareData *shr;
   
   if (!stab->ut->shared)
     return;
   
   for(ptr=stab->ut->shared; ptr; ptr=ptr->next) {
      child = ptr->data;
      if (!child)
	continue;
      
      if (child->data)
	d_free(child->data);
      
      for(cptr=child->next; cptr; cptr=cptr->next) {
	 shr = cptr->data;
	 if (shr)
	   j_free(SHARE, shr);
      }
      
      g_list_free(child);
   }
   
   g_list_free(stab->ut->shared);
   stab->ut->shared = NULL;
}

void connection_free(char *file, int line, void *data) {
   FREE_SETUP(Connection, c);

   D_FREE(file, line, c);
}

void search_tab_free(char *file, int line, void *data) {
   FREE_SETUP(SearchTab, st);
   
   D_FREE(file, line, st);
}

void browse_tab_free(char *file, int line, void *data) {
   FREE_SETUP(BrowseTab, bt);
   
   D_FREE(file, line, bt);
}

void download_tab_free(char *file, int line, void *data) {
   FREE_SETUP(DownloadTab, dt);
   
   D_FREE(file, line, dt);
}

void upload_tab_free(char *file, int line, void *data) {
   FREE_SETUP(UploadTab, ut);
   
   D_FREE(file, line, ut);
}

void console_tab_free(char *file, int line, void *data) {
   FREE_SETUP(ConsoleTab, ct);
   
   D_FREE(file, line, ct);
}

void motd_tab_free(char *file, int line, void *data) {
   FREE_SETUP(MOTDTab, mt);
   
   D_FREE(file, line, mt);
}

void stab_free(char *file, int line, void *data) {
   FREE_SETUP(STab, stab);
   
   D_FREE(file, line, stab);
}

void cbpack_free(char *file, int line, void *data) {
   FREE_SETUP(CBPack, cb);
   
   D_FREE(file, line, cb->file);
   D_FREE(file, line, cb->func_name);
   
   D_FREE(file, line, cb);
}

void account_free(char *file, int line, void *data) {
   FREE_SETUP(Account, acnt);
   
   D_FREE(file, line, acnt->user);
   D_FREE(file, line, acnt->pass);
   
   D_FREE(file, line, acnt);
}

void conn_info_free(char *file, int line, void *data) {
   FREE_SETUP(ConnInfo, ci);
   
   D_FREE(file, line, ci->status);
   D_FREE(file, line, ci->ip);
   D_FREE(file, line, ci->server);
   D_FREE(file, line, ci->cp);
   
   j_free(ACCOUNT, ci->account);
   
   D_FREE(file, line, ci);
}

void gmain_free(char *file, int line, void *data) {
   FREE_SETUP(GnapsterMain, gmain);
   
   D_FREE(file, line, gmain);
}

void file_entry_free(char *file, int line, void *data) {
   FREE_SETUP(FileEntry, fent);
   
   D_FREE(file, line, fent->line);
   D_FREE(file, line, fent);
}

void theme_free(char *file, int line, void *data) {
   FREE_SETUP(Theme, theme);
   
   D_FREE(file, line, theme->file);
   D_FREE(file, line, theme->name);
   D_FREE(file, line, theme->author);
   D_FREE(file, line, theme);
}

void hook_free(char *file, int line, void *data) {
   FREE_SETUP(ThemeHook, hook);
   
   D_FREE(file, line, hook->name);
   D_FREE(file, line, hook->data);
   D_FREE(file, line, hook);
}

void property_free(char *file, int line, void *data) {
   FREE_SETUP(WidgetProperty, prop);
   
   D_FREE(file, line, prop->conf);
   g_list_free(prop->sensitive);
   D_FREE(file, line, prop);
}

void transfer_free(char *file, int line, void *data) {
   FREE_SETUP(Transfer, t);
   
   D_FREE(file, line, t->file);
   D_FREE(file, line, t->header);
   D_FREE(file, line, t->id);
   D_FREE(file, line, t->user);
   D_FREE(file, line, t->fpath);
   D_FREE(file, line, t->resume_path);
   D_FREE(file, line, t);
}

void share_free(char *file, int line, void *data) {
   FREE_SETUP(ShareData, shr);
   
   D_FREE(file, line, shr->filename);
   D_FREE(file, line, shr->checksum);
   D_FREE(file, line, shr);
}

void search_free(char *file, int line, void *data) {
   FREE_SETUP(Search, search);
   
   D_FREE(file, line, search->file);
   D_FREE(file, line, search->user);
   D_FREE(file, line, search->id);
   D_FREE(file, line, search);
}

void server_free(char *file, int line, void *data) {
   FREE_SETUP(Server, server);
   
   D_FREE(file, line, server->ip);
   D_FREE(file, line, server->desc);
   D_FREE(file, line, server->network);
   D_FREE(file, line, server);
}

void user_free(char *file, int line, void *data) {
   FREE_SETUP(User, user);
   
   D_FREE(file, line, user->user);
   D_FREE(file, line, user);
}

void channel_free(char *file, int line, void *data) {
   FREE_SETUP(Channel, channel);
   
   D_FREE(file, line, channel->chan);
   D_FREE(file, line, channel->topic);
   D_FREE(file, line, channel);
}

void resume_free(char *file, int line, void *data) {
   FREE_SETUP(Resume, resume);
   
   D_FREE(file, line, resume->file);
   D_FREE(file, line, resume->checksum);
   D_FREE(file, line, resume);
}

void j_dbg_free(char *file, int line, int type, void *data) {
   FreeTable *ptr;
   
   for(ptr=gnapster_free; ptr->func; ptr++) {
      if (ptr->type == type) {
	 /* handle the free */
	 (ptr->func) (file, line, data);
	 return;
      }
   }
   
/*   printf("%s:%i: *** memory type %i unknown!\n", file, line, type);*/
   D_FREE(file, line, data);
}

void gnapster_destroy(gpointer data) {
   int type;
   
   d_assert(data != NULL);
   
   type = get_type(data);
/*   if (type == N_SEARCH)
     return;*/
   
   j_dbg_free(__FILE__, __LINE__, type, data);
}

int same_color(GdkColor col1, GdkColor col2) {
   if (col1.red == col2.red &&
       col1.green == col2.green &&
       col1.blue == col2.blue)
     return 1;
   
   return 0;
}

void unlink_log() {
   char *path;
   
   path = local_path("logs", "messages");

   d_free(path);
}

int copy_file(char *opath, char *npath) {
   char buf[2048];
   int o_fd, n_fd;
   int n;
   
   o_fd = open(opath, O_RDONLY);
   if (o_fd < 0)
     return o_fd;

   n_fd = open(npath, O_CREAT | O_WRONLY | O_TRUNC, 0644);
   if (n_fd < 0) {
      close(o_fd);
      return n_fd;
   }
   
   while((n = read(o_fd, buf, sizeof(buf))) > 0)
     write(n_fd, buf, n);
   
   close(o_fd);
   close(n_fd);
   
   return 1;
}

/* this function copies your ~/.gnome/gnapster file to 
 * ~/.gnapster/gnapster.conf if you do not have GNOME enabled 
 * 
 * it also checks to make sure that ~/.gnapter/gnapster.conf doesn't
 * already exist */
void relocate_config(char *home) {
#ifndef HAVE_GNOME
   char *orig_path, *new_path;
   
   d_msprintf(&orig_path, "%s/.gnome/gnapster", home);
   
   if (!file_exists(orig_path)) {
      d_free(orig_path);
      
      return;
   }
   
   new_path = local_path("gnapster.conf", NULL);
   
   if (file_exists(new_path)) {
      d_free(orig_path);
      d_free(new_path);
      
      return;
   }
   
   copy_file(orig_path, new_path);
   
   d_free(orig_path);
   d_free(new_path);
#endif /* !HAVE_GNOME */
}

void create_priv_dir() {
   FILE *aj_f;
   struct stat st;
   char *home, *path;
   
   home = getenv("HOME");
   
   relocate_config(home);
   
   path = local_path("", NULL);
   if (stat(path, &st) < 0)
     if (mkdir(path, 0755) < 0)
       fprintf(stderr, "Could not create ~/.gnapster directory...functionality may be limited\n");
   
   d_free(path);
   
   path = local_path("logs", NULL);
   if (stat(path, &st) < 0)
     if (mkdir(path, 0755) < 0)
       fprintf(stderr, "Could not create ~/.gnapster/logs directory...\n");
   
   d_free(path);
   
   path = local_path("themes", NULL);
   if (stat(path, &st) < 0)
     if (mkdir(path, 0755) < 0)
       fprintf(stderr, "Could not create ~/.gnapster/themes directory...\n");
   
   d_free(path);
   
   if (!(aj_f = open_autojoin("r"))) {
      FILE *f;
      
      f = open_autojoin("w");
      if (!f)
	return;

      fprintf(stderr, "autojoin not found, creating defaults...\n");

      fprintf(f, "gnapster\n");
      
      fclose(f);
   } else
     fclose(aj_f);
}

FILE *open_local_path(char *dir, char *file, char *flags) {
   FILE *f;
   char *conf;
   
   conf = local_path(dir, file);
   
   f = fopen(conf, flags);
   
   d_free(conf);
   
   return f;
}

FILE *open_log(char *file, char *flags) {
   FILE *f;
   char *conf;
   
   conf = local_path("logs", file);
   
   f = fopen(conf, flags);
   
   d_free(conf);
   
   return f;
}

void log_add_hooks(char *place, ...) {
   va_list args;
   char *hook;
   
   va_start(args, place);
   for(;;) {
      hook = va_arg(args, char *);
      if (!hook)
	break;
      
      log_hooks = g_list_append(log_hooks, d_strdup(hook));
   }
   va_end(args);
}

int hook_log(char *hook) {
   GList *ptr;
   char *dptr;
   
   if (!hook)
     return 0;
   
   for(ptr=log_hooks; ptr; ptr=ptr->next) {
      dptr = ptr->data;
      if (!dptr)
	continue;
      
      if (!strcmp(hook, dptr))
	return 1;
   }
   
   return 0;
}

int match_date(char *path, FILE *f) {
   char buf[11];
   FILE *new_file;
   struct tm *tm;
   char *dptr;
   time_t timep;
   int month, day, year, fd, n;
   
   timep = time(NULL);
   tm = localtime((time_t *)&timep);
   
   if (!file_exists(path)) {
      new_file = fopen(path, "w");
      fprintf(new_file, "%02i/%02i/%04i\n\n",
	      tm->tm_mon + 1, tm->tm_mday, 1900 + tm->tm_year);
      fclose(new_file);
   }
   
   fd = open(path, O_RDONLY);
   if (fd < 0)
     return 0;

   n = read(fd, buf, sizeof(buf) - 1);
   if (n < 0) {
      close(fd);
      return 0;
   }
   
   buf[n] = 0;
   
   dptr = buf;
   convert(next_arg_full(dptr, &dptr, '/'), "%i", &month);
   convert(next_arg_full(dptr, &dptr, '/'), "%i", &day);
   convert(next_arg(dptr, &dptr), "%i", &year);
   
   close(fd);
   
   if (na_err) {
      new_file = fopen(path, "w");
      fprintf(new_file, "%02i/%02i/%04i\n\n",
	      tm->tm_mon + 1, tm->tm_mday, 1900 + tm->tm_year);
      fclose(new_file);
   }
   
   NA_ERR_HANDLE(1);

   /* was it created today? */
   if (tm->tm_mday == day && (tm->tm_mon + 1) == month && (1900 + tm->tm_year) == year)
     return 1;
   
   return 0;
}

void move_file(char *path, FILE **f) {
   char *npath;
   
   d_msprintf(&npath, "%s.old", path);
   
   rename(path, npath);
   
   d_free(npath);
}

void rotate_logs(char *path, FILE **f) {
   if (!f)
     return;

   if (!match_date(path, *f))
     move_file(path, f);
   
   *f = fopen(path, "a");

   d_free(path);
}

void console_log(char *hook, char *text) {
   FILE *f;
   char *stripped;
   
   if (!user_info.conf[LOGGING] || !hook || !text)
     return;
   
   if (!hook_log(hook))
     return;
   
   rotate_logs(local_path("logs", "messages"), &f);
   
   if (!f)
     return;
   
   stripped = gtk_xtext_strip_color(text, strlen(text), NULL, NULL);
   if (stripped) {
      d_memory_hunk_append(DEBUG stripped, strlen(stripped));
      
      /* \n is already trailing */
      fprintf(f, "%s", stripped);
   }
   
   d_free(stripped);
   
   fclose(f);
}
