-------------------------------------------------------------------------------
--
--  Filename        : $Source: /cvsroot/gnade/gnade/contrib/gsql/datasource_handler.adb,v $
--  Description     : This package handles all ODBC data sources
--  Author          : Michael Erdmann
--  Created         : 8.8.2001
--  Last Modified By: $Author: merdmann $
--  Last Modified On: $Date: 2002/06/10 06:58:41 $
--  Status          : $State: Exp $
--
--  Copyright (C) 2000 Michael Erdmann
--                                                                           --
--  GNADE is free software;  you can redistribute it  and/or modify it under --
--  terms of the  GNU General Public License as published  by the Free Soft- --
--  ware  Foundation;  either version 2,  or (at your option) any later ver- --
--  sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
--  OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
--  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
--  for  more details.  You should have  received  a copy of the GNU General --
--  Public License  distributed with GNAT;  see file COPYING.  If not, write --
--  to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, --
--  MA 02111-1307, USA.                                                      --
--                                                                           --
--  As a special exception,  if other files  instantiate  generics from this --
--  unit, or you link  this unit with other files  to produce an executable, --
--  this  unit  does not  by itself cause  the resulting  executable  to  be --
--  covered  by the  GNU  General  Public  License.  This exception does not --
--  however invalidate  any other reasons why  the executable file  might be --
--  covered by the  GNU Public License.                                      --
--                                                                           --
--  Author: Michael Erdmann <michael.erdmann@snafu.de>                       --
--                                                                           --
--  GNADE is implemented to work with GNAT, the GNU Ada compiler.            --
--                                                                           --
-------------------------------------------------------------------------------

with Ada.Strings.Unbounded;    use Ada.Strings.Unbounded;
with Ada.Text_Io;              use Ada.Text_IO;
with Ada.Exceptions;           use Ada.Exceptions;
with Ada.Strings;              use Ada.Strings;
with Ada.Strings.Fixed;        use Ada.Strings.Fixed;

-- GSQL Support packages
with GUI_Preferences;          use GUI_Preferences;
with Contents_Manager.GUI;     use Contents_Manager.GUI;
with GUI_Logging;              use GUI_Logging;
with Gui_Result;               use GUI_Result;
with Gui_Window;               use Gui_Window;

with Contents_Manager;         use Contents_Manager;
with Preferences;              use Preferences;
with ODBC;                     use ODBC;

package body Datasource_Handler is

   Version : constant String := "$Id: datasource_handler.adb,v 1.15 2002/06/10 06:58:41 merdmann Exp $" ;

   Datasource_Section : Element_ID := Null_Element_ID;

   ------------------
   -- Data Sources --
   ------------------
   CurrentDB : Handle := null;

   ---------------------------
   -- Load_All_Datasources  --
   ---------------------------
   procedure Load_All_Datasources is
      S    : Source_Information;
      Db   : ODBC.Object;
      Item : Handle := null;
      Id   : Element_ID;
   begin
      Initialize(Db);
      S := Data_Source( Db, First => True );

      while S /= Null_Source_Information loop

         Item := new Object;
         Item.Self := Item;

         Initialize( Item.Server );
         Data_Source( Item.Server, To_String(S.Server) );

         Id := Add_Element(
            Parent  => Datasource_Section,
            Title   => To_String( S.Description & "[" & S.Server & "]" ),
            Content => Contents_Handler.Handle(Item));

         S := Data_Source(Db);
      end loop;

      Finalize(Db);
   end Load_All_Datasources;

   ----------------
   -- Initialize --
   ----------------
   procedure Initialize is
      Item : Handle := new Object;
   begin
      Item.Datasource_Root := True;

      Datasource_Section := Add_Element(
         Title => "Data Sources", Content => Contents_Handler.Handle(Item) );
      Load_All_Datasources;
   end Initialize;

   --------------
   -- Finalize --
   --------------
   procedure Finalize is
   begin
      null;
   end Finalize;

   --------------
   -- Selected --
   --------------
   procedure Selected(
      This : in out Object ) is
      -- connect to the selected data source
      P    : String_Array_Access;
   begin
      if This.Datasource_Root then
         return;
      end if;

      if CurrentDB /= null then
         Disconnect( CurrentDB.Server );
      end if;

      Authorization(
         This.Server,
         Login    => Preference(Prf_Login),
         Password => Preference(Prf_Password )
      );

      Log("Connecting to data source");
      Connect( This.Server );
      CurrentDB := This.Self;
      Log("ok");
      GUI_Logging.Active_Database( Key( This ) );

   exception
      when The_Error : others =>
         GUI_Logging.Error( Exception_Name( The_Error ) );
         GUI_Logging.Error( Exception_Message( The_Error ) );
         Display_DB_Info;

         Error( "Connect failed");
         GUI_Logging.Active_Database( "no database connected" );
         CurrentDB := null;
   end Selected;

   -----------
   -- Value --
   -----------
   procedure Value(
      This : in Object;
      Text : out Text_Buffer;
      Last : out Positive ) is
   begin
      null;
   end Value;

   ---------------------
   -- Display_DB_Info --
   ---------------------
   procedure Display_DB_Info is
      DB_Error: Unbounded_String;
      Message : Unbounded_String;
   begin
      if CurrentDB /= null then
         Get_Error( CurrentDB.Server, DB_Error, Message );
         if DB_Error /= Null_Unbounded_String then
            Error( To_String(Db_Error) );
            Error( To_String(Message) );
         end if;
      end if;
   end Display_DB_Info;

   ----------------------
   -- Current_Database --
   ----------------------
   function Current_Database return Handle is
   begin
      return CurrentDB;
   end Current_Database;

   -------------
   -- Execute --
   -------------
   procedure Execute(
      Cmd    : in String ) is
      -- execute a query and display the result
      Q      : ODBC.Object renames CurrentDB.Server;
      Result : Result_Record := Null_Result_Record;
      Rows   : Natural := 0;
      Error  : Unbounded_String := Null_Unbounded_String;
      Msg    : Unbounded_String := Null_Unbounded_String;
   begin
      ODBC.Query( Q, Cmd );

      if Columns(Q) > 0 then
         Add_Result( Q, To_Unbounded_String("Result..") );
      end if;

      Display_DB_Info;

   exception
      when Error : others =>
         GUI_Logging.Error( Exception_Name (Error) );
         GUI_Logging.Error( Exception_Message (Error) );
   end Execute;

   ----------------------------
   -- Datasource_Description --
   ----------------------------
   function Datasource_Description return String is
      Info : ODBC.Source_Information;
   begin
      if Current_Database = Not_Attached then
         return   "- No Database connected ";
      end if;

      Info := Data_Source( Current_Database.Server );
      return
         To_String( Info.Server ) &
         "[" & To_String( Info.Description ) & "]";
   end Datasource_Description;

   ----------
   -- Save --
   ----------
   procedure Save(
      Item  : in out Object;
      Value : in Text_Buffer;
      Name  : in String := "" ) is
      -- no save operation on a data source
   begin
      Log("No function for data sources");
   end Save;

   ----------
   -- Load --
   ----------
   procedure Load(
      Item  : in out Object;
      Value : out Text_Buffer;
      Last  : out Natural ) is
      -- no load operation on a data source
   begin
      Log("No function for data sources");
   end Load;

   -----------------
   -- Open_Editor --
   -----------------
   procedure Open_Editor(
      This : in out Object ) is
      -- no editor to open for data sources
   begin
      Log("No function for data sources");
   end Open_Editor;

   ------------------
   -- Close_Editor --
   ------------------
   procedure Close_Editor(
      This : in out Object ) is
   begin
      Log("No function for data sources");
   end Close_Editor;

   -------------
   -- Execute --
   -------------
   procedure Execute(
      This : in out Object ) is
      Name : Unbounded_String := Key( This );
   begin
      Log("No function for data sources");
   end Execute;

end Datasource_Handler;

