------------------------------------------------------------------------------
--  Thin Ada95 binding to OCI (Oracle Call Interface)                    --
--  Copyright (C) 2000-2004 Dmitriy Anisimkov.                              --
--  License agreement and authors contact information are in file oci.ads   --
------------------------------------------------------------------------------

--  $Id: oci-thick-date.adb,v 1.6 2004/08/23 13:41:32 vagul Exp $

with OCI.Lib;
with OCI.Thread;
with Interfaces.C;
with OCI.Environments;

package body OCI.Thick.Date is

   use Lib;

   ---------
   -- "<" --
   ---------

   function "<" (Left, Right : OCIDate) return Boolean is
   begin
      return Compare (Left, Right) = -1;
   end "<";

   ----------
   -- "<=" --
   ----------

   function "<=" (Left, Right : OCIDate) return Boolean is
   begin
      return Compare (Left, Right) /= 1;
   end "<=";

   ---------
   -- ">" --
   ---------

   function ">" (Left, Right : OCIDate) return Boolean is
   begin
      return Compare (Left, Right) = 1;
   end ">";

   ----------
   -- "=>" --
   ----------

   function ">=" (Left, Right : OCIDate) return Boolean is
   begin
      return Compare (Left, Right) /= -1;
   end ">=";

   -------------
   -- Compare --
   -------------

   function Compare (Left, Right : OCIDate) return Compare_Result is
      Result : aliased SWord;
      Rc :  SWord := OCIDateCompare
        (Err => Thread.Error,
         Date1 => Left,
         Date2 => Right,
         Result => Result'Access);
   begin
      Check_Error (Rc);
      return Compare_Result (Result);
   end Compare;

   -----------
   -- Equal --
   -----------

   function Equal (Left, Right : OCIDate) return Boolean is
   begin
      return Compare (Left, Right) = 0;
   end Equal;

   -------------
   -- SysDate --
   -------------

   function SysDate return OCIDate is
      Result : aliased OCIDate;
      Rc :  SWord := OCIDateSysDate (Err => Thread.Error,
         Date => Result'Access);
   begin
      Check_Error (Rc);
      return Result;
   end SysDate;

   ------------
   -- To_Ada --
   ------------

   function To_Ada (DT : in OCIDate) return Ada.Calendar.Time is
   begin
      return Ada.Calendar.Time_Of
         (Year    => Integer (DT.OCIDateYYYY),
          Month   => Integer (DT.OCIDateMM),
          Day     => Integer (DT.OCIDateDD),
          Seconds => Duration
                       (Natural (DT.OCIDateTime.OCITimeSS)
                        + 60 * (Natural (DT.OCIDateTime.OCITimeMI)
                                + 60 * Natural (DT.OCIDateTime.OCITimeHH))));
   end To_Ada;

   -------------
   -- To_Date --
   -------------

   function To_Date (Item : String; Format : String) return OCIDate is
      Result : aliased OCIDate;
      Rc : SWord := OCIDateFromText
              (Err          => Thread.Error,
               Date_Str     => C.To_C (Item),
               d_Str_Length => Item'Length,
               Fmt          => C.To_C (Format),
               Fmt_Length   => Format'Length,
               Lang_Name    => CStr.Null_Ptr,
               Lang_Length  => 0,
               Date         => Result'Access);
   begin
      Check_Error (Rc);
      return Result;
   end To_Date;

   ------------
   -- To_OCI --
   ------------

   function To_OCI (DT : in Ada.Calendar.Time) return OCIDate is
      use Ada.Calendar;

      Year    : Year_Number;
      Month   : Month_Number;
      Day     : Day_Number;
      Seconds : Day_Duration;
      Secs    : Natural;
   begin
      Split (DT, Year, Month, Day, Seconds);

      if Seconds = 0.0 then
         Secs := 0;
      else
         Secs := Natural (Seconds - 0.5);
      end if;

      return (OCIDateYYYY => Sb2 (Year),
              OCIDateMM   => Ub1 (Month),
              OCIDateDD   => Ub1 (Day),
              OCIDateTime => (OCITimeHH => Ub1 (Secs / 3600),
                              OCITimeMI => Ub1 (Secs / 60 rem 60),
                              OCITimeSS => Ub1 (Secs rem 60)));
   end To_OCI;

   ---------------
   -- To_String --
   ---------------

   function To_String (From : OCIDate; Format : String) return String is
      use Interfaces.C;
      Buff : aliased Text := (0 .. Format'Length + 64 => C.nul);
      Len  : aliased Ub4 := Buff'Length - 1;
      Rc   : SWord := OCIDateToText
        (Err         => Thread.Error,
         Date        => From,
         Fmt         => C.To_C (Format),
         Fmt_Length  => Format'Length,
         Lang_Name   => CStr.Null_Ptr,
         Lang_Length => 0,
         Buf_Size    => Len'Access,
         Buf         => CStr.To_Chars_Ptr (Buff'Unchecked_Access));
   begin
      Check_Error (Rc);
      return C.To_Ada (Buff, False) (1 .. Integer (Len));
   end To_String;

begin
   OCI.Environments.Set_Create_Mode_Flag (OCI_OBJECT);
end OCI.Thick.Date;
