-------------------------------------------------------------------------------
--                                                                           --
--  Filename        : $Source: /cvsroot/gnade/gnade/dbi/adbc/gnu-db-adbc-driver-mysql.adb,v $
--  Description     : Ada Database Object - Driver for MySQL                 --
--  Author          : Michael Erdmann                                        --
--  Created         : 18.1.2002                                              --
--  Last Modified By: $Author: merdmann $
--  Last Modified On: $Date: 2002/03/21 05:45:04 $
--  Status          : $State: Exp $
--                                                                           --
--  Copyright (C) 2002 Michael Erdmann                                       --
--                                                                           --
--  GNADE is free software;  you can redistribute it  and/or modify it under --
--  terms of the  GNU General Public License as published  by the Free Soft- --
--  ware  Foundation;  either version 2,  or (at your option) any later ver- --
--  sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
--  OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
--  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
--  for  more details.  You should have  received  a copy of the GNU General --
--  Public License  distributed with GNAT;  see file COPYING.  If not, write --
--  to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, --
--  MA 02111-1307, USA.                                                      --
--                                                                           --
--  As a special exception,  if other files  instantiate  generics from this --
--  unit, or you link  this unit with other files  to produce an executable, --
--  this  unit  does not  by itself cause  the resulting  executable  to  be --
--  covered  by the  GNU  General  Public  License.  This exception does not --
--  however invalidate  any other reasons why  the executable file  might be --
--  covered by the  GNU Public License.                                      --
--                                                                           --
--  This software is implemented to work with GNAT, the GNU Ada compiler.    --
--                                                                           --
--  Functional Description                                                   --
--  ======================                                                   --
--  This is the driver for MySQL bindings. The driver implements the         --
--  following classes:                                                       --
--                                                                           --
--  ADBC.Statement                                                           --
--  ADBC.Row                                                                 --
--  ADBC.Resultset                                                           --
--                                                                           --
--  Additionally it implements the ADBC.Driver interface.                    --
--                                                                           --
--  Restrictions                                                             --
--  ============                                                             --
--  Only Linux                                                               --
--                                                                           --
--  Contact                                                                  --
--  =======                                                                  --
--  Error reports shall be handled via http://gnade.sourceforge.net          --
--  Features and ideas via: gnade-develop@lists.sourceforge.net              --
--                                                                           --
--  Author contact:                                                          --
--               purl:/net/michael.erdmann                                   --
--                                                                           --
-------------------------------------------------------------------------------
--* Ada
with System;                                use System;
with System.Storage_Elements;               use System.Storage_Elements;
with System.Address_To_Access_Conversions;

with Ada.Exceptions;                        use Ada.Exceptions;
with Ada.Tags;                              use Ada.Tags;
with Ada.Characters.Latin_1;                use Ada.Characters;
with Ada.Calendar;                          use Ada.Calendar;
with Ada.Strings.Fixed;                     use Ada.Strings.Fixed;
with Ada.Strings.Unbounded;                 use Ada.Strings.Unbounded;
with Ada.Unchecked_Conversion;

with Ada.Text_IO;                           use Ada.Text_IO;

with Interfaces.C.Strings;                  use Interfaces.C.Strings;
with Interfaces.C;                          use Interfaces.C;
use  Interfaces;

with Unchecked_Deallocation;

with GNU.DB.ADBC.Statement;                 use GNU.DB.ADBC.Statement;
with GNU.DB.ADBC.Resultset;                 use GNU.DB.ADBC.Resultset;
with GNU.DB.ADBC.Connection;                use GNU.DB.ADBC.Connection;
with GNU.DB.ADBC.Hostvariable;              use GNU.DB.ADBC.Hostvariable;
with GNU.DB.ADBC.Hostvariable.Types;        use GNU.DB.ADBC.Hostvariable.Types;

with GNU.DB.ADBC.Row;                       use GNU.DB.ADBC.Row;

package body GNU.DB.ADBC.Driver.MySQL is

   pragma Linker_Options ("-lmysqlclient");

   Version : constant  String :=
      "$Id: gnu-db-adbc-driver-mysql.adb,v 1.4 2002/03/21 05:45:04 merdmann Exp $";

   --- ******************************************************************** ---
   --- ***  L O W    L E V E L    I N T E R F A C E   T O    M Y S Q L  *** ---
   --- ******************************************************************** ---

   --- Interface to the MYSQL Library. Some of the data types are simply
   --- Addresses because the internals of these structured do not have
   --- to be known here.

   subtype MYSQL_SERVER is  System.Address;
   subtype MYSQL_ROW    is  System.Address;
   subtype MYSQL_RES    is  System.Address;
   subtype MYSQL_FNO    is  C.unsigned;

   type MYSQL_RES_Record is                                     -- mysql.h:89
        record
            row_count    : c.unsigned;                          -- mysql.h:90
            field_count  : c.unsigned;                          -- mysql.h:91
            current_field: c.unsigned;                          -- mysql.h:91
            fields       : Address;                             -- mysql.h:92
            data         : Address;                             -- mysql.h:93
            data_cursor  : Address;                             -- mysql.h:94
            field_alloc  : Address;                             -- mysql.h:95
            row          : MYSQL_ROW;                           -- mysql.h:96
            current_row  : MYSQL_ROW;                           -- mysql.h:97
            lengths      : Address;                             -- mysql.h:98
            handle       : Address;                             -- mysql.h:99
            eof          : C.char;                              -- mysql.h:100
        end record;

   pragma Convention(C, MYSQL_RES_Record );

   function mysql_connect( mysql : MYSQL_SERVER;
                           host  : Strings.chars_ptr;
                           user  : Strings.chars_ptr;
                           passwd: Strings.chars_ptr)
                                   return MYSQL_SERVER;
   pragma Import (C, mysql_connect, "mysql_connect");

   procedure mysql_close(sock: MYSQL_SERVER);
   pragma Import (C, mysql_close, "mysql_close");

   function mysql_select_db(mysql: MYSQL_SERVER; db : Strings.chars_ptr) return c.int;
   pragma Import (C, mysql_select_db, "mysql_select_db");

   function mysql_query(mysql: MYSQL_SERVER; q : Strings.chars_ptr) return c.int;
   pragma Import (C, mysql_query, "mysql_query");

   function mysql_fetch_row(mysql: MYSQL_RES) return MYSQL_ROW;
   pragma Import (C, mysql_fetch_row, "mysql_fetch_row");

   function mysql_field_seek(mysql : MYSQL_RES; offset: MYSQL_FNO) return MYSQL_FNO;
   pragma Import (C, mysql_field_seek, "mysql_field_seek");    -- mysql.h:166

   type MYSQL_field_types is (                                  -- mysql_com.h:64
        FIELD_TYPE_DECIMAL,                                     -- mysql_com.h:64
        FIELD_TYPE_TINY,                                        -- mysql_com.h:64
        FIELD_TYPE_SHORT,                                       -- mysql_com.h:65
        FIELD_TYPE_LONG,                                        -- mysql_com.h:65
        FIELD_TYPE_FLOAT,                                       -- mysql_com.h:66
        FIELD_TYPE_DOUBLE,                                      -- mysql_com.h:66
        FIELD_TYPE_NULL,                                        -- mysql_com.h:67
        FIELD_TYPE_TIMESTAMP,                                   -- mysql_com.h:67
        FIELD_TYPE_LONGLONG,                                    -- mysql_com.h:68
        FIELD_TYPE_INT24,                                       -- mysql_com.h:68
        FIELD_TYPE_DATE,                                        -- mysql_com.h:69
        FIELD_TYPE_TIME,                                        -- mysql_com.h:69
        FIELD_TYPE_DATETIME,                                    -- mysql_com.h:70
        FIELD_TYPE_YEAR,                                        -- mysql_com.h:70
        FIELD_TYPE_NEWDATE,                                     -- mysql_com.h:71
        FIELD_TYPE_ENUM,                                        -- mysql_com.h:72
        FIELD_TYPE_SET,                                         -- mysql_com.h:73
        FIELD_TYPE_TINY_BLOB,                                   -- mysql_com.h:74
        FIELD_TYPE_MEDIUM_BLOB,                                 -- mysql_com.h:75
        FIELD_TYPE_LONG_BLOB,                                   -- mysql_com.h:76
        FIELD_TYPE_BLOB,                                        -- mysql_com.h:77
        FIELD_TYPE_VAR_STRING,                                  -- mysql_com.h:78
        FIELD_TYPE_STRING                                       -- mysql_com.h:80
    );
    for MYSQL_field_types use (                             -- mysql_com.h:64
        FIELD_TYPE_DECIMAL => 0,                                -- mysql_com.h:64
        FIELD_TYPE_TINY => 1,                                   -- mysql_com.h:64
        FIELD_TYPE_SHORT => 2,                                  -- mysql_com.h:65
        FIELD_TYPE_LONG => 3,                                   -- mysql_com.h:65
        FIELD_TYPE_FLOAT => 4,                                  -- mysql_com.h:66
        FIELD_TYPE_DOUBLE => 5,                                 -- mysql_com.h:66
        FIELD_TYPE_NULL => 6,                                   -- mysql_com.h:67
        FIELD_TYPE_TIMESTAMP => 7,                              -- mysql_com.h:67
        FIELD_TYPE_LONGLONG => 8,                               -- mysql_com.h:68
        FIELD_TYPE_INT24 => 9,                                  -- mysql_com.h:68
        FIELD_TYPE_DATE => 10,                                  -- mysql_com.h:69
        FIELD_TYPE_TIME => 11,                                  -- mysql_com.h:69
        FIELD_TYPE_DATETIME => 12,                              -- mysql_com.h:70
        FIELD_TYPE_YEAR => 13,                                  -- mysql_com.h:70
        FIELD_TYPE_NEWDATE => 14,                               -- mysql_com.h:71
        FIELD_TYPE_ENUM => 247,                                 -- mysql_com.h:72
        FIELD_TYPE_SET => 248,                                  -- mysql_com.h:73
        FIELD_TYPE_TINY_BLOB => 249,                            -- mysql_com.h:74
        FIELD_TYPE_MEDIUM_BLOB => 250,                          -- mysql_com.h:75
        FIELD_TYPE_LONG_BLOB => 251,                            -- mysql_com.h:76
        FIELD_TYPE_BLOB => 252,                                 -- mysql_com.h:77
        FIELD_TYPE_VAR_STRING => 253,                           -- mysql_com.h:78
        FIELD_TYPE_STRING => 254                                -- mysql_com.h:80
    );
   for MYSQL_field_types'size use 32;

   type st_mysql_field is                                       -- mysql.h:39
        record
            name      : Strings.chars_ptr;                      -- mysql.h:40
            table     : Strings.chars_ptr;                      -- mysql.h:41
            def       : Strings.chars_ptr;                      -- mysql.h:42
            c_type    : MYSQL_field_types;                      -- mysql.h:43
            length    : c.unsigned;                             -- mysql.h:44
            max_length: c.unsigned;                             -- mysql.h:45
            flags     : c.unsigned;                             -- mysql.h:46
            decimals  : c.unsigned;                             -- mysql.h:47
        end record;

   pragma Convention(C, st_mysql_field);          -- mysql.h:39
   type MYSQL_Field is access all st_mysql_field;

    type enum_mysql_status is (                                 -- mysql.h:67
        MYSQL_STATUS_READY,                                     -- mysql.h:67
        MYSQL_STATUS_GET_RESULT,                                -- mysql.h:67
        MYSQL_STATUS_USE_RESULT                                 -- mysql.h:68
    );
    for enum_mysql_status'size use 32;                          -- mysql.h:67

    type struct_st_net is record
            fd          : c.int;                               -- mysql_com.h:54
            fcntl       : c.int;                               -- mysql_com.h:55
            buff        : Strings.chars_ptr;                   -- mysql_com.h:56
            buff_end    : Strings.chars_ptr;                   -- mysql_com.h:56
            write_pos   : Strings.chars_ptr;                   -- mysql_com.h:56
            last_error  : C.char_Array(0..199);                -- mysql_com.h:57
            last_errno  : c.unsigned;                          -- mysql_com.h:58
            max_packet  : c.unsigned;                          -- mysql_com.h:58
            timeout     : c.unsigned;                          -- mysql_com.h:58
            pkt_nr      : c.unsigned;                          -- mysql_com.h:58
            error       : c.signed_char;                       -- mysql_com.h:59
            return_errno: c.signed_char;                       -- mysql_com.h:59
        end record;

    pragma Convention(C,  struct_st_net);                      -- mysql_com.h:53
    subtype NET is struct_st_net;

    type A_MYSQL_FIELD_T is access all st_mysql_field;         -- mysql.h:82

    type st_mem_root is                                        -- mysql.h:18
        record
            free         : Address;                             -- mysql.h:19
            used         : Address;                             -- mysql.h:20
            min_malloc   : c.unsigned;                          -- mysql.h:21
            block_size   : c.unsigned;                          -- mysql.h:22
            error_handler: Address     ;                        -- mysql.h:23
        end record;

    pragma Convention(C, st_mem_root);                          -- mysql.h:18
    subtype MEM_ROOT is st_mem_root;                            -- mysql.h:24


    type st_mysql is  record
            The_net            : NET;                           -- mysql.h:71
            host               : Strings.chars_ptr;             -- mysql.h:72
            user               : Strings.chars_ptr;                       -- mysql.h:72
            passwd             : Strings.chars_ptr;                       -- mysql.h:72
            unix_socket        : Strings.chars_ptr;                       -- mysql.h:72
            server_version     : Strings.chars_ptr;                       -- mysql.h:72
            host_info          : Strings.chars_ptr;                       -- mysql.h:72
            info               : Strings.chars_ptr;                       -- mysql.h:73
            db                 : Strings.chars_ptr;                       -- mysql.h:73
            port               : c.unsigned;                -- mysql.h:74
            client_flag        : c.unsigned;                -- mysql.h:74
            server_capabilities: c.unsigned;                -- mysql.h:74
            protocol_version   : c.unsigned;                -- mysql.h:75
            field_count        : c.unsigned;                -- mysql.h:76
            thread_id          : c.unsigned;                -- mysql.h:77
            affected_rows      : c.unsigned;                -- mysql.h:78
            insert_id          : c.unsigned;                -- mysql.h:79
            extra_info         : c.unsigned;                -- mysql.h:80
            status             : enum_mysql_status;             -- mysql.h:81
            fields             : A_MYSQL_FIELD_T;               -- mysql.h:82
            field_alloc        : MEM_ROOT;                      -- mysql.h:83
            free_me            : c.signed_char;                 -- mysql.h:84
            reconnect          : c.signed_char;                       -- mysql.h:85
        end record;

   pragma Convention(C, St_mysql);


               -- mysql.h:89
   function mysql_init(
      mysql :  MYSQL_SERVER ) return MYSQL_SERVER;
   pragma Import (C, mysql_init, "mysql_init");

               -- mysql.h:89
   function mysql_fetch_field(
      handle: MYSQL_RES) return MYSQL_FIELD;
   pragma Import (C, mysql_fetch_field, "mysql_fetch_field");

   function mysql_store_result(
      mysql: MYSQL_SERVER) return MYSQL_RES; -- mysql.h:161
   pragma Import (C, mysql_store_result, "mysql_store_result");

   function mysql_use_result(
      mysql: MYSQL_SERVER) return MYSQL_RES;                   -- mysql.h:162
   pragma Import (C, mysql_use_result, "mysql_use_result");

   procedure mysql_free_result(
      result: MYSQL_RES);                                      -- mysql.h:163
   pragma Import (C, mysql_free_result, "mysql_free_result");

   -----------------
   -- Get_Address --
   -----------------
   function Get_Address(
      Row : in MYSQL_ROW;
      Pos : in Positive ) return Address is
      -- Retrieve the address from an array fo addresses

      Result : array( Positive ) of Address ;
      for Result'Address use Row;
   begin
      return Result( Pos );
   end Get_Address;

   -----------------
   -- Copy_String --
   -----------------
   procedure Copy_String (
      Addr   : in Address;
      Value  : out String ) is
      -- Copy a string from the memory address. This code is not nice but
      -- is works.
      byte   : Character;

      Src    : String( 1..Value'Length );
      for Src'Address use Addr;

      J      : Integer := Value'First;
   begin
      Value := (others => ' ');
      for I in Value'Range loop
         Byte := Src(I);
         exit when Byte = Latin_1.NUL;

         Value(J) := Byte;
         J := J + 1;
      end loop;
   end Copy_String;

   --------------
   -- Num_Rows --
   --------------
   function NumRows(
      result : in Address  ) return Integer is
      --  Return the number of rows found during a selection
      Res    : MYSQL_RES_Record;
      for Res'Address use Result;
   begin
      if result = Null_Address then
         return 0;
      end if;

      return Integer(Res.Row_Count);
   end NumRows;

   ------------------
   -- Num_Affected --
   ------------------
   function Num_Affected(
      Server  : in Address  ) return Integer is
      -- return the number of affected rows
      package P is new Address_To_Access_Conversions( Object => St_Mysql );
      use P;
   begin
      return Integer(To_Pointer(Server).Affected_Rows);
   end Num_Affected;

   -----------------
   -- Object_Data --
   -----------------
   type Object_Data is record
         Self               : Driver.Handle;
         Connection_Handle  : MYSQL_Server     := Null_Address;
      end record;

   --- ****************************************************************** ---
   --- *****       S T A T E M E N T     H A N D L I N G           ****** ---
   --- ****************************************************************** ---

   ------------------
   -- My_Statement --
   ------------------
   type My_Statement is new Statement.Object with record
         null;
      end record;

   type My_Stmt_Access is access all My_Statement;

   Statement_Table : array( Statement_ID ) of My_Stmt_Access;

   ----------------------
   -- Create_Statement --
   ----------------------
   function Create_Statement(
      This   : in Object;
      Con    : in Connection_ID ) return Statement_ID is
      -- this does more or less nothing, because the interface
      -- of mysql does not requiere any explicit prepare actions
      Result : My_Stmt_Access := new My_Statement( Con );
      Id     : Statement_ID := Statement.Allocate( Statement.Handle(Result) );
   begin
      Statement_Table(Id) := Result;

      return Id;
   end Create_Statement;

   ----------------------
   -- Delete_Statement --
   ----------------------
   procedure Delete_Statement(
      This : in Object;
      Stmt : in Statement_ID ) is

      procedure Free is
         new Unchecked_Deallocation( My_Statement, My_Stmt_Access);
   begin
      pragma Debug (Put_Line("DRIVER: delete statement " & Statement_ID'Image(Stmt) ));
      Free( Statement_Table(Stmt) );
   end Delete_Statement;

   --- ****************************************************************** ---
   --- ****               R O W    H A N D L I N G                   **** ---
   --- ****************************************************************** ---
   type My_Row is new Row.Object with record
         RowHandle     : MYSQL_ROW  := Null_Address;
         Set           : Resultset_ID;
      end record;

   ---------
   -- Get --
   ---------
   procedure Get(
      Row    : in My_Row;
      Pos    : in Positive;
      Result : out String;
      Length : out Natural );

   type My_Row_Access is access My_Row;

   --- ****************************************************************** ---
   --- *****       R E S U L T S E T     H A N D L I N G           ****** ---
   --- ****************************************************************** ---

   type Field_Info_Array is
      array( Positive range 1..Max_Number_Of_Attributes ) of St_MySql_Field;

   --------------------
   -- My_Resultset --
   --------------------
   type My_Resultset is new Resultset.Object with record
         ResultHandle  : MYSQL_RES;
         Affected_Rows : Natural          := 0;

         -- here we store information about the attrviutes of a resultset
         -- which is not needed in the result set clas.
         Next_Field    : Positive         := Field_Info_Array'First;
         Fieldinfo     : Field_Info_Array;
      end record;

   type My_Resultset_Access is access all My_Resultset;

   Result_Table : array( Resultset_ID ) of My_Resultset_Access;

   -----------
   -- Fetch --
   -----------
   procedure Fetch(
      This   : in out My_Resultset;
      Result : out Row.Handle;
      Mode   : in Fetch_Mode_Type := Next ) is
      Row    : My_Row_Access;
   begin
      Row := new My_Row( This.Next_Field  );
      Row.Set := ID(This);

      Row.RowHandle := Mysql_Fetch_Row( This.Resulthandle );
      if Row.RowHandle = Null_Address then
         raise End_Of_Resultset;
      end if;

      Result := GNU.DB.ADBC.Row.Handle(Row);
   end Fetch;

   ----------------------
   -- Create_Resultset --
   ----------------------
   function Create_Resultset(
      This   : in Object;
      Stmt   : in Statement_ID ) return Resultset_ID is
      Result : My_Resultset_Access := new My_Resultset(Stmt) ;
      Id     : Resultset_ID;
   begin
      pragma Debug(
         Put_Line("DRIVER: create_resultset for " & Statement_ID'Image(Stmt) )
      );
      Result.Resulthandle := Null_Address;

      Id := Resultset.Allocate( Resultset.Handle(Result) );
      Result_Table(Id) := Result;

      return Id;
   end Create_Resultset;

   ----------------------
   -- Delete_Resultset --
   ----------------------
   procedure Delete_Resultset(
      This   : in Object;
      Result : in Resultset_ID ) is

      procedure Free is
         new Unchecked_Deallocation( My_Resultset, My_Resultset_Access);
   begin
      Free( Result_Table(Result) );

      pragma Debug(Put_Line("DRIVER: delete resultset " & Resultset_ID'Image(Result)));
   end Delete_Resultset;

   ---------
   -- Get --
   ---------
   procedure Get(
      Row    : in My_Row;
      Pos    : in Positive;
      Result : out String;
      Length : out Natural ) is
      -- get a field from the resultset. The format is expected to be
      -- a string.
      Set    : My_Resultset renames
               My_Resultset_Access( Resultset.Get_Handle( Row.Set ) ).all;
      P      : Address;
   begin
      Length := Natural( Set.FieldInfo(Pos).Length );
      P := Get_Address( Row.RowHandle, Pos);
      if P /= Null_Address then
         Copy_String( P, Result(1..Length));
      else
         Length := 0;
      end if;
   end Get;

   --- ****************************************************************** ---
   --- *****       C O N N E C T I O N   H A N D L I N G           ****** ---
   --- ****************************************************************** ---

   -------------
   -- Connect --
   -------------
   procedure Connect(
      This     : in out Object;
      User     : in String;
      Password : in String;
      Database : in String) is
      -- connecting to an actual database server
      Data     : Object_Data_Access renames This.Data;
      RC       : C.Int := 0;
   begin
      if Data = null then
         Data := new Object_Data;
      end if;

      Data.Connection_Handle := MYSQL_Connect( Null_Address,
                 New_String( "localhost" ),
                 New_String( User ),
                 New_String( Password ) );

      if Data.Connection_Handle = Null_Address then
         raise Connection_Failure;
      end if;

      RC :=  mysql_select_db( Data.Connection_Handle, New_String(Database) );
      if not ( RC = 0 ) then
         raise Wrong_Database_Name;
      end if;
      pragma Debug( Put_Line("DRIVER:[MySQL] connect to database complete"));
   end Connect;

   ----------------
   -- Disconnect --
   ----------------
   procedure Disconnect(
      This : in out Object ) is
      -- diconnect from the data base
      Data : Object_Data_Access renames This.Data;
   begin
      mysql_close( Data.Connection_Handle );
      pragma Debug(Put_Line("DRIVER: disconnect"));
   end Disconnect;

   --- ****************************************************************** ---
   --- *****       C O N N E C T I O N   H A N D L I N G           ****** ---
   --- ****************************************************************** ---

   -------------
   -- Prepare --
   -------------
   procedure Prepare(
      This : in out Object;
      Stmt : in Statement_ID ) is
      -- there is no functionaltiy for MySQL required since it does
      -- not support the preperation of statement.
      St   : My_Statement renames My_Stmt_Access( Get_Handle(Stmt) ).all;
   begin
      null;
      pragma Debug( Put_Line("DRIVER: prepare does nothing") );
   end Prepare;

   ------------
   -- Expand --
   ------------
   function Expand(
      This : in Object;
      -- This procedure returns the SQL ASCII representation of the host
      -- variable for an SQL statement.
      -- In case of ODBC this procedure returns allways a '?' character
      -- and the Bind method below does the trick of connecting variables
      -- with the data base.
      -- In all other cases the ascii string will be returned.
      V    : in Hostvariable.Handle ) return String is
   begin
      if V.all'Tag = SQL_String'Tag then
         return '"' & Value( SQL_String(V.all) ) & '"';
      elsif V.all'Tag = SQL_Integer'Tag then
         return Integer'Image( Value(SQL_Integer(V.all)) );
      else
         return "?";
      end if;
   end Expand;

   ----------
   -- Bind --
   ----------
   procedure Bind_Host_Variable(
      This : in Object;
      Stmt : in Statement_ID;
      V    : in Hostvariable.Handle) is
      -- This procedure will be called before the statement is executed. The
      -- main application is the allocation of defered buffers as for example
      -- used by the ODBC interface.
      -- Since MySQL does not use defered buffers this procedure does
      -- more or less nothing, but it provides the skeleton.
      -- The mapping between ADBC and DBCS specific representation is done at
      -- this point.

      procedure Bind_SQL_String(
         S : in String ) is
      begin
         pragma Debug(
            Put_Line("DRIVER: Bind SQL_String does nothing, value='" & S & "'" )
         );
         null;
      end Bind_SQL_String;

      procedure Bind_SQL_Integer(
         S : in Integer ) is
      begin
         pragma Debug(
            Put_Line("DRIVER: Bind SQL_Integer does nothing, value =" & Integer'Image(S))
         );
         null;
      end Bind_SQL_Integer;

   begin
      if V.all'Tag = SQL_String'Tag then
         Bind_SQL_String( Value( SQL_String(V.all) ) );
      elsif V.all'Tag = SQL_Integer'Tag then
         Bind_SQL_Integer( Value( SQL_Integer(V.all) ) );
      end if;
   end Bind_Host_Variable;

   --------------------
   -- Get_Host_Value --
   --------------------
   procedure Get_Host_Value(
      This : in Object;
      Stmt : in Statement_ID;
      V    : in Hostvariable.Handle) is
      -- get the contents of a host variable from the database
   begin
      pragma Debug(Put_Line("DRIVER:Get_Value does nothing"));
      null;
   end Get_Host_Value;

   -------------
   -- Execute --
   -------------
   procedure Execute_Statement(
      This       : in Object;
      Result     : in Resultset_ID;
      Stmt       : in Statement_ID ) is
      -- execute a statement:
      --   S.1 - Resolve unbounde variables
      --   S.2 - Execute the actuial query
      --   S.3 - Fetch the field information
      --   S.4 - Fetch the number of affected rows if the query has
      --         not created any result set.
      Data       : Object_Data_Access renames This.Data;
      Set        : My_Resultset renames
                      My_Resultset_Access( Resultset.Get_Handle( Result ) ).all;

      Q          : constant String := Query(Stmt);
      RES_Address: MYSQL_RES;
      RC         : C.int    := 0;
      Field      : MYSQL_Field;
   begin
      RC := mysql_query( Data.Connection_Handle, New_String(Q));
      if RC /= 0  then
         raise Empty_Resultset;
      end if;

      -- fetch the result set                                       *** S.3 ***
      RES_Address := mysql_store_result( Data.Connection_Handle );
      if RES_Address /= Null_Address then
         if NumRows( RES_Address ) < 1 then
            raise Empty_Resultset;
         end if;
         Number_Of_Records( Result, NumRows(RES_Address) );
         Set.ResultHandle := RES_Address;

         Field := mysql_fetch_field( Set.ResultHandle  );
         while Field /= null loop
            Attribute( Result, Value( Field.Name ) );
            Set.Fieldinfo( Set.Next_Field ) := Field.all;

            Set.Next_Field := Set.Next_Field + 1;
            exit when not ( Set.Next_Field in Set.Fieldinfo'Range );

            Field := mysql_fetch_field( Set.ResultHandle );
         end loop;
      else                                                      -- *** S.4 ***
         Set.Affected_Rows := Num_Affected( Data.Connection_Handle );
      end if;
   end Execute_Statement;

   ------------
   -- Create --
   ------------
   type Mysql_Access is access all Object;

   function Create return Driver.Handle is
      Db   : Mysql_Access := new Object;
      Data : Object_Data_Access := new Object_Data;
   begin
      Data.Self := Driver.Handle(Db);
      Db.Data := Data;
      return Driver.Handle(Db);
   end Create;

end GNU.DB.ADBC.Driver.MySQL;

