-------------------------------------------------------------------------------
--                                                                           --
--  Filename        : $Source: /cvsroot/gnade/gnade/tools/imp.adb,v $
--  Description     : Import a file into the data base                       --
--  Author          : Michael Erdmann                                        --
--  Created         : 5.4.2003                                               --
--  Last Modified By: $Author: merdmann $                                    --
--  Last Modified On: $Date: 2003/05/10 16:24:03 $                           --
--  Status          : $State: Exp $                                          --
--                                                                           --
--  Copyright (C) 2000 - 2003 Michael Erdmann                                --
--                                                                           --
--  GNADE is free software;  you can redistribute it  and/or modify it under --
--  terms of the  GNU General Public License as published  by the Free Soft- --
--  ware  Foundation;  either version 2,  or (at your option) any later ver- --
--  sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
--  OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
--  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
--  for  more details.  You should have  received  a copy of the GNU General --
--  Public License  distributed with GNAT;  see file COPYING.  If not, write --
--  to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, --
--  MA 02111-1307, USA.                                                      --
--                                                                           --
--  As a special exception,  if other files  instantiate  generics from this --
--  unit, or you link  this unit with other files  to produce an executable, --
--  this  unit  does not  by itself cause  the resulting  executable  to  be --
--  covered  by the  GNU  General  Public  License.  This exception does not --
--  however invalidate  any other reasons why  the executable file  might be --
--  covered by the  GNU Public License.                                      --
--                                                                           --
--  Author: Michael Erdmann <michael.erdmann@snafu.de>                       --
--                                                                           --
--  GNADE is implemented to work with GNAT, the GNU Ada compiler.            --
--                                                                           --
--  Functional Description                                                   --
--  ======================                                                   --
--  This program allows to fill a table with data stored in an export file   --
--  created by the exp program.                                              --
--  Since this programm touches only the named field in the export file, it  --
--  is possible to insert the contents of multiple fields in different steps.--
--                                                                           --
--                                                                           --
--  Restrictions                                                             --
--  ============                                                             --
--  None                                                                     --
--                                                                           --
--  References                                                               --
--  ==========                                                               --
--  None                                                                     --
--                                                                           --
-------------------------------------------------------------------------------
with Ada.Text_IO;               use Ada.Text_IO;
with Ada.Streams.Stream_IO;     use Ada.Streams;
with Ada.Command_Line;          use Ada.Command_Line;
with Ada.Strings.Unbounded;     use Ada.Strings.Unbounded;
with Import;                    use Import;
with Util;                      use Util;
with Tools;		        use Tools;
with Files;                     use Files;

procedure Imp is
   Version        : constant String := "$Id: imp.adb,v 1.6 2003/05/10 16:24:03 merdmann Exp $";

   Login          : Unbounded_String;
   Pwd            : Unbounded_String;
   Source         : Unbounded_String;
   Argc           : Natural := 1;
   Quiet          : Boolean := False;
   Table          : Unbounded_String;
   File_Name      : Unbounded_String;
   Nbr_Of_Records : Natural := 0;

   Import_File    : Files.Object;


   SQL_Insert : constant Unbounded_String := To_Unbounded_String("insert into ");

   package Files is new Util.Bucket( Bucket_Type => Unbounded_String ); 

   ----------
   -- Help --
   ----------
   procedure Help is
   begin
      Error("usage : " );
      Error("    imp [-u <login> [ -l <passwd> ]] source table file(s)" );
      Error("");
      Error("    -u <login>  - Login to be used to access the data source.");
      Error("    -l <passwd> - Password to bu used for login.");
      Error("    source      - Name of an ODBC data source.");
      Error("    table       - SQL Table Name (upper/lower case are relevant.");
      Error("    file(s)     - A list of files created by the exp tool.");
      Error("");
   end Help;

   ------------
   -- Import --
   ------------
   procedure Execute_Import(
      Name   : in Unbounded_String; 
      T      : in Unbounded_String ) is 
      -- export the contents of a table into the given output stream  
      Data   : String_Array_Access := null;
      Row    : Natural := 1;
   begin    
      Open( Import_File, To_String( Name ) );
      Import.Prepare( To_String(Table), Fields(Import_File).all );

      Data := Read( Import_File );
      while Data /= Null_String_Array loop    
         Nbr_Of_Records := Nbr_Of_Records + 1;
         Import.Execute( Data.all ); 
	      
	 Data := Read( Import_File );
         Row := Row + 1;
      end loop;

   exception      
      when File_Not_Existing =>
      	 raise;

      when Others =>
         Error( "Exception in row : " & Natural'Image(Row) );
         raise; 
   end Execute_Import;

begin 
   -- Handle the command line arguments
   while Argc in 1..Argument_Count loop
      declare
         Arg : constant String := Argument( Argc );
      begin
         if Arg = "-u" then
            Argc := Argc + 1;
            Login := To_Unbounded_String( Argument( Argc ) );
         elsif Arg = "-p" then
            Argc := Argc + 1;
            Pwd := To_Unbounded_String( Argument( Argc ) );
         elsif Arg = "-h" then
            Help;
            return;
	 elsif Arg = "-v" then
	    Opt_Verbose := True;
         elsif Arg = "-q" then
            Util.Quiet := True;
	    Opt_Verbose  := False;
         else
            if Source = Null_Unbounded_String then
               Source := To_Unbounded_String( Arg );
            elsif Table = Null_Unbounded_String then 
               Table := To_Unbounded_String( Arg );
            else 
               Files.Add( To_Unbounded_String( Arg ) );
            end if;
         end if;
         Argc := Argc + 1;
      end;
   end loop;
 
   Report("GNADE Table import, Version " & Tools.Version);
   Report("Copyright (C) 2003, Michael Erdmann (michael.erdmann@snafu.de)");
   Report("");

   if Source = Null_Unbounded_String then
      Error("no data source has been specified");
      Error("");
      Help;
      Set_Exit_Status(Failure);
      return;
   end if;

   Import.Initialize( To_String( Source ), To_String( Login ), To_String( Pwd ) );

   if Files.End_Of_Bucket then
      File_Name := Table & ".exp";
   else 
      File_Name := Files.Get( First => true );
   end if;

   loop 
      Initialize( Import_File );
      Report("Importing file " & To_String( File_Name ) );

      Execute_Import( File_Name, Table );

      Finalize( Import_File );

      exit when Files.End_Of_Bucket;
      File_Name := Files.Get;
   end loop;
 
   Import.Finalize;

   Report("Number of imported record(s): " & Natural'Image(Nbr_Of_Records) );
   Set_Exit_Status( Success );
end Imp;

