#include <stdlib.h>
#include <unistd.h>
#include <stdio.h>
#include <ctype.h>

#include <gnome.h>

#include "complex.h"
#include "filter.h"
#include "fft.h"
#include "mfsk.h"
#include "viterbi27.h"
#include "viterbi.h"
#include "macro.h"
#include "prefs.h"
#include "varicode.h"
#include "interleave.h"

static int graydecodetab[16] = {
	0,  1,  3,  2,
	6,  7,  5,  4,
	12, 13, 15, 14,
	10, 11, 9,  8
};

static void recvbit(struct mfskrx *m, int bit)
{
	int c;

	m->datashreg = (m->datashreg << 1) | !!bit;

	/* search for "001" */
	if ((m->datashreg & 7) == 1) {
		/* the "1" belongs to the next symbol */
		c = varidec(m->datashreg >> 1);

		if (c != -1)
			show_rx_char(c);

		/* we already received this */
		m->datashreg = 1;
	}
}

static inline unsigned char clamp(gfloat in)
{
	if (in > 255.0)
		in = 255.0;
	if (in < 0.0)
		in = 0.0;

	return (unsigned char) floor(in);
}

static void recvsymbol(struct mfskrx *m, complex *bins)
{
	float tone, sum, met, b[4];
	unsigned char symbols[4];
	int i, j;

	/* gray decode and form soft decision samples */
	sum = b[0] = b[1] = b[2] = b[3] = 0.0;
	for (i = 0; i < 16; i++) {
		j = graydecodetab[i];

		tone = cmod(bins[i + BaseTone]);

		b[0] += (j & 8) ? tone : -tone;
		b[1] += (j & 4) ? tone : -tone;
		b[2] += (j & 2) ? tone : -tone;
		b[3] += (j & 1) ? tone : -tone;

		sum += tone;
	}

	/* shift to range 0...255 */
	symbols[0] = clamp(128.0 + (b[0] / sum * 128.0));
	symbols[1] = clamp(128.0 + (b[1] / sum * 128.0));
	symbols[2] = clamp(128.0 + (b[2] / sum * 128.0));
	symbols[3] = clamp(128.0 + (b[3] / sum * 128.0));

	deinterleave(symbols);

	viterbi27(m->viterbi, symbols);

	if (m->viterbi->datalen > 0) {
		met = (m->viterbi->metric - m->prevmetric) / 20.0;
		m->prevmetric = m->viterbi->metric;

		if (met > 0) {
			met -= m->averagemetric;
			m->averagemetric += met / 8.0;
			update_metric_dial(m->averagemetric);
		}

		if (SQUELCHon && m->averagemetric < prefs.sqval)
			return;

		for (i = 0; i < m->viterbi->datalen; i++) {
			recvbit(m, m->viterbi->data[i] & 128);
			recvbit(m, m->viterbi->data[i] & 64);
			recvbit(m, m->viterbi->data[i] & 32);
			recvbit(m, m->viterbi->data[i] & 16);
			recvbit(m, m->viterbi->data[i] & 8);
			recvbit(m, m->viterbi->data[i] & 4);
			recvbit(m, m->viterbi->data[i] & 2);
			recvbit(m, m->viterbi->data[i] & 1);
		}
	}
}

static complex mixer(struct mfskrx *m, complex in)
{
	complex z;
	float f;

	f = m->freq - (float) BaseTone * SampleRate / SymbolLen;

	z.re = cos(m->phaseacc);
	z.im = sin(m->phaseacc);

	z = cmul(z, in);

	m->phaseacc -= 2.0 * M_PI * f / SampleRate;

	if (m->phaseacc > M_PI)
		m->phaseacc -= 2.0 * M_PI;
	if (m->phaseacc < M_PI)
		m->phaseacc += 2.0 * M_PI;

	return z;
}

static int decodesymbol(complex *in)
{
	int i, symbol;
	float x, max;

	in += BaseTone;

	symbol = 0;
	max = 0.0;
	
	for (i = 0; i < 16; i++) {
		x = cmod(in[i]);

		if (x > max) {
			max = x;
			symbol = i;
		}
	}

	return symbol;
}

struct mfskrx *initrx(void)
{
	struct mfskrx *m;

	if ((m = calloc(1, sizeof(struct mfskrx))) == NULL)
		return NULL;

	if ((m->hilbert = init_filter(0.05, 0.45)) == NULL)
		return NULL;

	if ((m->slfft = init_slfft(SymbolLen, BaseTone, BaseTone + NumTones)) == NULL)
		return NULL;

	if ((m->viterbi = init_viterbi27()) == NULL)
		return NULL;

	init_deinterleaver();

	return m;
}

static void synchronize(struct mfskrx *m)
{
	int i, j, syn = -1;
	float val, max = 0.0;

	if (m->currsymbol == m->prev1symbol)
		return;
	if (m->prev1symbol == m->prev2symbol)
		return;

	for (i = 0; i < PipeLen; i++) {
		j = (i + m->pipeptr) % PipeLen;
		val = cmod(m->pipe[j].vector[m->prev1symbol]);
		if (val > max) {
			syn = i;
			max = val;
		}
	}

	m->syncaverage += (syn - SymbolLen) / 16.0;
	m->symboltime = ((unsigned int) m->syncaverage) % SymbolLen;
}

static void afc(struct mfskrx *m)
{
	complex z;
	float x;

	if (AFCon == FALSE)
		return;

	if (m->currsymbol != m->prev1symbol)
		return;

	z = ccor(m->prev1vector, m->currvector);
	x = carg(z) / SymbolLen / (2.0 * M_PI / SampleRate);

	if (x > -ToneSpacing / 2.0 &&  x < ToneSpacing / 2.0)
		trx_set_freq(m->freq + (x / 8.0));
}

void rxprocess(struct mfskrx *m, float *buf, int len)
{
	complex z, *bins;
	int i;

	while (len-- > 0) {
		/* create analytic signal... */
		z.re = z.im = *buf++;
		z = filter(m->hilbert, z);

		/* ...so it can be shifted in frequency */
		z = mixer(m, z);

		/* feed the real part to the sliding FFT */
		bins = slfft(m->slfft, z.re);

		/* copy current vector to the pipe */
		for (i = 0; i < NumTones; i++)
			m->pipe[m->pipeptr].vector[i] = bins[i + BaseTone];

		if (m->counter > 0) {
			m->counter--;
		} else if (m->symbolphase == m->symboltime) {
			m->counter = SymbolLen / 2;

			m->currsymbol = decodesymbol(bins);
			m->currvector = bins[m->currsymbol + BaseTone];

			/* decode symbol */
			recvsymbol(m, bins);

			/* draw the scope */
			bitshape_draw(m);

			/* symbol sync */
			synchronize(m);

			/* frequency tracking */
			afc(m);

			m->prev2symbol = m->prev1symbol;
			m->prev2vector = m->prev1vector;
			m->prev1symbol = m->currsymbol;
			m->prev1vector = m->currvector;
		}

		m->pipeptr = (m->pipeptr + 1) % PipeLen;
		m->symbolphase = (m->symbolphase + 1) % SymbolLen;
	}
}
