/*
 * Gmail. A Gnome email client.
 * Copyright (C) 1999-2000 Wayne Schuller
 *
 * main.c - gmail main functions.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include "main.h"

/* Global variables. */
_GmailApp_	*GmailApp;		/* Global vars live here. */
DTimer *timer;

/* Exported functions. */
gboolean DestroyRightPane(void);
void gmail_init(void);
void broken_cb (GtkWidget *widget, void *data);
void save_msg (GtkWidget *widget, void *data);
void set_ok (GtkWidget *widget, gboolean *dialog_result);
void about_cb (GtkWidget *widget, void *data);
void copy_cb (GtkWidget *widget, void *data);
void cut_cb (GtkWidget *widget, void *data);
void paste_cb (GtkWidget *widget, void *data);
void select_all_cb (GtkWidget *widget, void *data);
void entry_drag_data_received (GtkWidget *widget, GdkDragContext *context, gint x, gint y, GtkSelectionData *data, guint info, guint time);
void online_help (GtkWidget *widget, gchar *name);
void clear_cb (GtkWidget *widget, void *data);

/* Local functions. */
static void cache_help (GtkWidget *widget, void *data);
static void quit_cb (GtkWidget *widget, void *data);
static void NoCodeDialog(void);
static void export_to_mbox (GtkWidget *widget, gchar *type);

/* External functions. */
/* setup.c */
extern void setup_druid(void);

/* mysql.c */
extern	gboolean open_db(MYSQL *mysql);
extern	gboolean check_db(MYSQL *mysql);
extern	Message * load_vfolder_msg(gint id);
extern	gint message_count (void);
extern void vfolder_display_rightpane_by_mailbox(Mailbox *mailbox);
extern 	void free_message(Message *message);
extern gboolean vfolder_contains_unread(Mailbox *mailbox);

/* settings.c */
extern GtkWidget * read_settings(void);
extern void          open_preferences (void);
extern gboolean 	check_config_version(void);
extern void rebuild_all_mi_values (GtkWidget *widget, void *data);
void set_mailmenu_options(GtkOptionMenu *mailmenu);

/* mblist.c */
extern void delete_msg (GtkWidget *widget, gpointer data);
extern void next_prev_cb (GtkWidget *widget, gboolean next);
extern void next_prev_unread_cb (GtkWidget *widget, gboolean next);
extern void search_cb (GtkWidget *widget, void *data);

/* compose.c */
extern void     compose_cb (GtkWidget *widget, gchar *type);
extern gchar * strip_email(gchar *to);

/* checkmail.c */
extern void     checkmail_cb (GtkWidget *widget, void *data);

/* matching.c */
extern void recreate_matched_index (GtkWidget *widget, void *data);

/* print.c */
extern void print_cb (GtkWidget *widget, void *data);
extern void print_preview_cb (GtkWidget *widget, void *data);

/* outgoing.c */
void send_queued_cb (GtkWidget *widget, gpointer *data);

/* Menu Items */
static GnomeUIInfo edit_menu [] = {
	GNOMEUIINFO_MENU_COPY_ITEM(copy_cb, NULL),
	/* FIXME  GNOMEUIINFO_MENU_SELECT_ALL_ITEM(select_all_cb, NULL), */
  { GNOME_APP_UI_ITEM, N_("Search"), N_("Search messages for requested text"), search_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_SEARCH, 'f', GDK_CONTROL_MASK, NULL },
	GNOMEUIINFO_END
};

static GnomeUIInfo adv_menu [] = {
	{ GNOME_APP_UI_ITEM, N_("Matched Index Online Help"), N_("Matched Index Online Help"), cache_help, NULL, NULL, GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_PIXMAP_HELP, 0, 0, NULL },
	{ GNOME_APP_UI_ITEM, N_("Recreate Matched Index"), N_("Rebuild index design based on current vfolders (can be slow)."), recreate_matched_index, NULL, NULL,
	GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_CONVERT, 0, 0, NULL },
	{ GNOME_APP_UI_ITEM, N_("Rebuild All MI Values"), N_("Scan every message to rebuild the matched-index values."), rebuild_all_mi_values, NULL, NULL, 
	GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_REFRESH, 0, 0, NULL },
        GNOMEUIINFO_END
};

static GnomeUIInfo help_menu [] = {
	GNOMEUIINFO_HELP ("gmail"),
        GNOMEUIINFO_ITEM_STOCK (N_("About Gmail"), N_("About Gmail"), about_cb, GNOME_STOCK_MENU_ABOUT),
        GNOMEUIINFO_END
};

static GnomeUIInfo settings_menu[] =
{
  GNOMEUIINFO_MENU_PREFERENCES_ITEM(open_preferences, NULL),
  GNOMEUIINFO_SUBTREE_STOCK (N_("Advanced"), &adv_menu, GNOME_STOCK_MENU_EXEC),
  GNOMEUIINFO_END
};

static GnomeUIInfo file_menu[] =
{
  { GNOME_APP_UI_ITEM, N_("_Check Mail"), N_("Download new messages from POP3 server"), checkmail_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_MAIL_RCV, 'M', GDK_CONTROL_MASK, NULL },

  { GNOME_APP_UI_ITEM, N_("_Send Queued Mail"), N_("Send all the locally queued messages"), send_queued_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_MAIL_SND, 0, 0, NULL },

  { GNOME_APP_UI_ITEM, "_New", N_("Compose a new email message"), compose_cb, "new", NULL, GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_NEW, 'n', GDK_CONTROL_MASK, NULL },

	GNOMEUIINFO_MENU_SAVE_ITEM(save_msg, NULL),

  { GNOME_APP_UI_ITEM, N_("Export All"), N_("Export all messages to an mbox file."), export_to_mbox, "all", NULL, GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_SAVE, 0, 0, NULL },

  GNOMEUIINFO_SEPARATOR,
 
	GNOMEUIINFO_MENU_PRINT_ITEM(print_cb, NULL),
  GNOMEUIINFO_ITEM (N_("Print preview..."), N_("Preview data to be printed"), print_preview_cb, NULL),


  GNOMEUIINFO_SEPARATOR,

  GNOMEUIINFO_MENU_EXIT_ITEM(quit_cb, NULL),

  GNOMEUIINFO_END
};
/* The menu definitions: File/Exit and Help/About are mandatory */
static GnomeUIInfo main_menu [] = {
        GNOMEUIINFO_MENU_FILE_TREE(file_menu),
        GNOMEUIINFO_MENU_EDIT_TREE(edit_menu),
        GNOMEUIINFO_MENU_SETTINGS_TREE(settings_menu),
        GNOMEUIINFO_SUBTREE (N_("Help"), &help_menu),
        GNOMEUIINFO_END
};
/* The toolbar */
static GnomeUIInfo toolbar_info[] = {
    { GNOME_APP_UI_ITEM, N_("Check"), N_("Download new messages from POP3 server"), checkmail_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_PIXMAP_MAIL_RCV, 0, 0, NULL },
    { GNOME_APP_UI_ITEM, N_("Search"), N_("Search messages for requested text."), search_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_PIXMAP_SEARCH, 'f', GDK_CONTROL_MASK, NULL },
    { GNOME_APP_UI_ITEM, N_("Delete"), N_("Delete current selected message"), delete_msg, NULL, NULL,
      GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_PIXMAP_TRASH, 0, 0, NULL },
    { GNOME_APP_UI_ITEM, N_("Compose"), N_("Compose a new message"), compose_cb, "new", NULL,
      GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_PIXMAP_MAIL_NEW, 0, 0, NULL },
    { GNOME_APP_UI_ITEM, N_("Reply"), N_("Reply to the message"), compose_cb, "reply", NULL,
      GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_PIXMAP_MAIL_RPL, 0, 0, NULL },
    { GNOME_APP_UI_ITEM, N_("Reply to all"), N_("Reply to all recipients of the message"), compose_cb, "replytoall", NULL,
      GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_PIXMAP_MAIL_RPL, 0, 0, NULL },
    { GNOME_APP_UI_ITEM, N_("Forward"), N_("Forward the message"), compose_cb, "forward", NULL,
      GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_PIXMAP_MAIL_FWD, 0, 0, NULL },
    { GNOME_APP_UI_ITEM, N_("Prev Msg"), N_("Prev Msg"), next_prev_cb, GINT_TO_POINTER(FALSE), NULL,
      GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_PIXMAP_BACK, 0, 0, NULL },
    { GNOME_APP_UI_ITEM, N_("Next Msg"), N_("Next Msg"), next_prev_cb, GINT_TO_POINTER(TRUE), NULL,
      GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_PIXMAP_FORWARD, 0, 0, NULL },
    { GNOME_APP_UI_ITEM, N_("Next UMsg"), N_("Next Unread Msg"), next_prev_unread_cb, GINT_TO_POINTER(TRUE), NULL,
      GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_PIXMAP_FORWARD, 0, 0, NULL },
    { GNOME_APP_UI_ITEM, N_("Print"), N_("Print"), print_cb, NULL, NULL,
      GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_PIXMAP_PRINT, 0, 0, NULL },
    { GNOME_APP_UI_ITEM, N_("Quit"), N_("Quit"), quit_cb, NULL, NULL,
      GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_PIXMAP_QUIT, 0, 0, NULL },
    GNOMEUIINFO_END
};

int
main (int argc, char *argv[])
{
	/* i8n stuff. */
	bindtextdomain(PACKAGE, GNOMELOCALEDIR);
	textdomain(PACKAGE);

	gnome_init (PACKAGE, VERSION, argc, argv);

	gnome_window_icon_set_default_from_file(GMAIL_ICONDIR"/gmail_icon.png");

	glade_gnome_init();

	gmail_init(); /* Do the normal gmail start. */

	gtk_main (); 
	
	return 0;
}


/* gmail_init - setup the gmail basics. */
void
gmail_init(void)
{
	GtkWidget *vbox, *hbox;
	GtkTooltips *tooltips;
	static const char *soundtrigger[] = {"gmail", "startup", NULL};
	GtkCTreeNode *node;
	gchar *last_vfolder = NULL, *msg = NULL;

	/* GmailApp contains all global vars. */
	GmailApp = NULL;
	GmailApp = g_malloc(sizeof(_GmailApp_));

	g_assert(GmailApp != NULL);

	/* Very simply check to see if they have configured gmail before. */
	if (gnome_config_get_string("gmail/UserInfo/Realname") == NULL) {
		setup_druid(); 
		return;
		}

	/* Make sure our database versions match. */
	if (!check_config_version()) return;

	GmailApp->app = gnome_app_new (PACKAGE, PACKAGE);

	gtk_widget_set_usize(GTK_WIDGET(GmailApp->app), 800, 600);

	/* gtk_window_set_default_size(GTK_WINDOW(GmailApp->app), 750, 700); 
	gtk_window_set_policy(GTK_WINDOW(GmailApp->app), FALSE, TRUE, FALSE);  */
	/* gtk_widget_set_usize(GTK_WIDGET(GmailApp->app), gnome_config_get_int_with_default("/gmail/Geometry/width=850", NULL), gnome_config_get_int_with_default("/gmail/Geometry/height=650", NULL));  */

	gnome_app_create_menus (GNOME_APP (GmailApp->app), main_menu);
	gnome_app_create_toolbar (GNOME_APP (GmailApp->app), toolbar_info); 

	gtk_signal_connect (GTK_OBJECT (GmailApp->app), "delete_event",
			    GTK_SIGNAL_FUNC (quit_cb),
			    NULL);

	/* General drop not in use right now. 
  	gtk_drag_dest_set (GmailApp->app, GTK_DEST_DEFAULT_ALL,
		     email_field_drop_types, ELEMENTS (email_field_drop_types),
		     GDK_ACTION_COPY | GDK_ACTION_MOVE); 

  	gtk_signal_connect( GTK_OBJECT(GmailApp->app), "drag_data_received",
		      GTK_SIGNAL_FUNC( label_drag_data_received), NULL); */


	vbox = gtk_vbox_new(FALSE, 5);
	gnome_app_set_contents(GNOME_APP (GmailApp->app), vbox);

	/* FIXME: Set up the bottom toolbar.
	 * It should be the gnome appbar then our option menu.
	 */
	hbox = gtk_hbox_new(FALSE, 5);

	GmailApp->appbar = gnome_appbar_new(FALSE, TRUE, GNOME_PREFERENCES_NEVER);
	gtk_box_pack_start(GTK_BOX (hbox), GmailApp->appbar, TRUE, TRUE, 0);

	gnome_app_install_appbar_menu_hints(GNOME_APPBAR(GmailApp->appbar), main_menu); 

	GmailApp->mailmenu = gtk_option_menu_new();
	set_mailmenu_options(GTK_OPTION_MENU(GmailApp->mailmenu));
	tooltips = gtk_tooltips_new();
	gtk_object_ref(GTK_OBJECT(tooltips));
	gtk_tooltips_set_tip (tooltips, GmailApp->mailmenu, "Click to choose a different mail setup", NULL);

	gtk_box_pack_end(GTK_BOX (hbox), GmailApp->mailmenu, FALSE, FALSE, 0);

	gtk_box_pack_end(GTK_BOX (vbox), hbox, FALSE, FALSE, 0);
	gtk_widget_show_all(hbox);

	/* Set up the horizontal pane. */
	GmailApp->hpaned = gtk_hpaned_new();
 	gtk_paned_set_handle_size (GTK_PANED(GmailApp->hpaned), 10);
	gtk_paned_set_gutter_size (GTK_PANED(GmailApp->hpaned), 15);
	gtk_paned_set_position(GTK_PANED(GmailApp->hpaned), gnome_config_get_int_with_default("/gmail/Geometry/hpaned=150", NULL));
	gtk_box_pack_start(GTK_BOX (vbox), GmailApp->hpaned, TRUE, TRUE, 0);

	/* 
	 * If it's three pane view chuck in the extra right pane. 
	 */
	if (!TwoPaneView()) {
		GmailApp->rightpane = gtk_vpaned_new();
  		gtk_paned_set_handle_size (GTK_PANED(GmailApp->rightpane), 10);
  		gtk_paned_set_gutter_size (GTK_PANED(GmailApp->rightpane), 15);
		gtk_paned_set_position(GTK_PANED(GmailApp->rightpane), gnome_config_get_int_with_default("/gmail/Geometry/rightpane=200", NULL));
		gtk_paned_add2 (GTK_PANED(GmailApp->hpaned), GmailApp->rightpane);
	}

	/* Create a scrolled window to hold the leftpane clist. */
	GmailApp->leftpane = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (GmailApp->leftpane), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);    
	gtk_container_set_border_width (GTK_CONTAINER (GmailApp->leftpane), 5);


	/* Read mailbox settings into leftpane clist. */
	GmailApp->mblist = read_settings(); 
	gtk_scrolled_window_add_with_viewport (GTK_SCROLLED_WINDOW (GmailApp->leftpane), GmailApp->mblist);
	gtk_widget_show(GmailApp->mblist);

	gtk_paned_add1 (GTK_PANED(GmailApp->hpaned), GmailApp->leftpane);
	gtk_widget_show_all (GmailApp->app); /* Lets see it! */
	
	if (!open_db(&GmailApp->mysql)) {
		g_warning(_("Gmail FATAL ERROR: Failed to open database."));
		return;	
		}

	/* Open the second database connection here also. 
	 * We use it for rebuilding matched values and other matching stuff.
	 */

	if (!open_db(&GmailApp->mysql2)) { 
		g_warning(_("Gmail FATAL ERROR: Failed to open second database connection."));
		return; 
		}

	last_vfolder = gnome_config_get_string_with_default("/gmail/UserInfo/LastVfolder=Inbox", NULL);

	GmailApp->node = NULL; /* Reset our global. */

	gtk_widget_show_all (GmailApp->mblist); 
	gtk_widget_show_all (GmailApp->hpaned); 

	/* Loop through the ctree structure bolding vfolders with unread.
	 */
	node = gtk_ctree_node_nth(GTK_CTREE(GmailApp->mblist), 0);
	for (; node != NULL; node = GTK_CTREE_NODE_NEXT(node)) {
		Mailbox *mailbox = NULL;

		/* Mark vfolders bold if they are unread.  */
		mailbox = gtk_ctree_node_get_row_data(GTK_CTREE (GmailApp->mblist), GTK_CTREE_NODE (node));

		if (vfolder_contains_unread(mailbox))
			gtk_ctree_node_set_cell_style(GTK_CTREE (GmailApp->mblist), GTK_CTREE_NODE(node), 0, GmailApp->boldstyle);

		/* Display the last opened vfolder when we find it. */
		if (last_vfolder != NULL && mailbox != NULL && mailbox->name != NULL && strcmp(last_vfolder, mailbox->name) == 0) {
			vfolder_display_rightpane_by_mailbox(mailbox);
			GmailApp->node = node;
			gtk_ctree_select(GTK_CTREE(GmailApp->mblist), GmailApp->node);
		}
	}  

	/* If nothing was selected display the first vfolder. */
	if (GmailApp->node == NULL) {	
		Mailbox *mailbox = NULL;

		node = gtk_ctree_node_nth(GTK_CTREE(GmailApp->mblist), 0);
		mailbox = gtk_ctree_node_get_row_data(GTK_CTREE (GmailApp->mblist), GTK_CTREE_NODE (node));
		vfolder_display_rightpane_by_mailbox(mailbox);
		GmailApp->node = node;
		gtk_ctree_select(GTK_CTREE(GmailApp->mblist), GmailApp->node);
	}


	gnome_triggers_vdo("", NULL, soundtrigger); /* Play sound */

	/* Send welcome message to the status bar. */
	msg = g_strdup_printf(_("Welcome to Gmail. (%i email messages in the database)"), message_count());
	gnome_appbar_set_status (GNOME_APPBAR (GmailApp->appbar), msg);
	g_free(msg);
}


/* Callback functions */

void
broken_cb (GtkWidget *widget, void *data)
{
	g_print (_("This feature not yet implemented...\n"));
	NoCodeDialog(); 
	return;
}

/* Quit callback.
 * There are some variables which we save to the config file at quit time.
 * Ideally, none of these would be here, but we would catch them all with
 * signals and apply them when appropriate.
 */
static void
quit_cb (GtkWidget *widget, void *data)
{
	static const char *soundtrigger[] = {"gmail", "shutdown", NULL};
	Mailbox *mailbox = gtk_ctree_node_get_row_data(GTK_CTREE (GmailApp->mblist), GTK_CTREE_NODE (GmailApp->node));

#if 0
	GdkWindow *toplevel;
	gint width, height, root_x, root_y;
#endif

	/* Lets record the pane settings. 
	 * FIXME: catch the signal where this is moved and save it then.
	 */
	gnome_config_set_int ("/gmail/Geometry/hpaned", GTK_PANED(GmailApp->hpaned)->child1_size);
	if (!TwoPaneView()) {
		gnome_config_set_int ("/gmail/Geometry/rightpane", GTK_PANED(GmailApp->rightpane)->child1_size);
		}

	gnome_config_set_string("/gmail/UserInfo/LastVfolder", mailbox->name);

#if 0
	/* Attempt to save the current window size. */
	/* Gedit used as a reference here. */
	toplevel = gdk_window_get_toplevel (GTK_WINDOW (GmailApp->app));
	gdk_window_get_root_origin (toplevel, &root_x, &root_y);
		/* We don't want to save the size of a maximized window,
		   so chek the left margin. This will not work if there is
	 	   a pannel in left, but there is no other way that I know
		   of knowing if a window is maximzed or not */
	if (root_x != 0) {
		gdk_window_get_size (GTK_WINDOW (GmailApp->app), &width, &height);
		gnome_config_set_int ("/gmail/Geometry/width", (gint) width);
		gnome_config_set_int ("/gmail/Geometry/height", (gint) height);
		}
#endif

	gnome_config_sync();

	/* FIXME: Free all the data stored in the leftpane clist. */

	/* Finally close the database connection! */
	mysql_close(&GmailApp->mysql);

	g_free(GmailApp->sortfield);
	g_free(GmailApp->sortorder);

	g_free(GmailApp);

	gnome_triggers_vdo("", NULL, soundtrigger); /* Play sound */

	/* Remove any temporary attachment files left remaining. */
	system("rm .gmail_* 2>/dev/null");

	gtk_main_quit ();
	return;
}

void
about_cb (GtkWidget *widget, void *data)
{
	GtkWidget *about;
	GtkWidget *href;
	GtkWidget *hbox;
	
	const gchar *authors[] = {
		"Wayne Schuller (k_wayne@linuxpower.org)",
		NULL
	};
	
	about = gnome_about_new ( PACKAGE, VERSION,
				  "(C) 1999-2000 Wayne Schuller",
				  authors,
				  _("An experimental sql based vfolder email system.\nReleased under the terms of the GNU Public License.\nSupport the Free Software Foundation: http://www.gnu.org"), NULL);
	href = gnome_href_new ("http://gmail.linuxpower.org",
             _("Gmail web site"));	
	hbox = gtk_hbox_new(TRUE, 0);
	gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(about)->vbox), hbox, FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(hbox), href, FALSE, FALSE, 0);
	gtk_widget_show_all(about);
}



static void NoCodeDialog (void) {
	GtkWidget *dialog;

	dialog = gnome_message_box_new(_("This code not implemented yet.\nSend patches to k_wayne@linuxpower.org"), GNOME_MESSAGE_BOX_ERROR, _("Ok"), 0);
  gtk_window_set_position(GTK_WINDOW(dialog), GTK_WIN_POS_CENTER);
	gtk_window_set_modal(GTK_WINDOW (dialog), TRUE);
	gnome_dialog_set_parent(GNOME_DIALOG(dialog), GTK_WINDOW(GmailApp->app));
	gtk_widget_show(dialog);
}

/* This function is called whenever we want to clear whatever is in the right
 * pane.
 * It also provides some code reuse.
 * It is only necessary when in TwoPane view mode. Thus this function
 * assumes this is the case!
 * It returns TRUE if the right pane was successfully destroyed. It returns
 * FALSE if something important couldn't be destroyed, in which case the
 * calling function should bail out.
 * FIXME: it should send dialog boxes, not just stdout messages.
 */
gboolean
DestroyRightPane (void) {
	GtkBin *bin;

	/* We MUST be in the TwoPane view. */
	if (!TwoPaneView()) { return(FALSE); }

	/* Easy case, if another message is up, destroy it and replace.. 
	 * This logic is flawed. 
   * Do we need to check which bit is 'realized' (right term?).
   */

	if (GTK_IS_SCROLLED_WINDOW (GmailApp->msglistpane)) {

		/* Hard case, rightlist vfolder is up. Destroy carefully, let the 
		 * rightlist survive so next/prev buttons work.
		 * FIXME: don't we need to ref the rightlist here? See GGAD book.
	   */

		bin = GTK_BIN(GmailApp->msglistpane);
		if (GTK_IS_CLIST(bin->child)) gtk_container_remove(GTK_CONTAINER(GmailApp->msglistpane), bin->child);
		gtk_widget_destroy(GmailApp->msglistpane); 
		GmailApp->msglistpane = NULL;

		} 

		if (GTK_IS_WIDGET(GmailApp->msgpane)) {
			/* Easy case. Just destroy the msg pane. */
			gtk_widget_destroy(GmailApp->msgpane); 
			GmailApp->msgpane = NULL;
		}

	gnome_appbar_clear_stack(GNOME_APPBAR (GmailApp->appbar));
 
	return(TRUE);
}


/* export_to_mbox - Export the vfolder/s to a mbox text file.
 * This function is useful if the user wants an mbox of their mail.
 * This is helpful if the database format has a big change, you can
 * export all your files to an mbox, put them in your mail spool, and
 * pop them back into a new database.
 * FIXME: There is also a problem with outgoing messages, as they don't really
 * have any headers. Should we just not export them?
 * 
 */
static void
export_to_mbox (GtkWidget *widget, gchar *type)
{
	MYSQL_RES *res;
	MYSQL_RES *res2;
	MYSQL_ROW row;
	MYSQL_ROW row2;
	gchar *query, *date;
	FILE *fd;
	gchar *filename;
	GtkWidget *dialog;
	GtkWidget *label;
	gint current, total, i;

	/* FIXME: Create popup dialog with a progress bar?. */
	dialog = gnome_dialog_new(_("Gmail Warning:"),
                        GNOME_STOCK_BUTTON_OK,
                        GNOME_STOCK_BUTTON_CANCEL,
                        NULL);
 	if (GmailApp->app)
		gnome_dialog_set_parent(GNOME_DIALOG(dialog), GTK_WINDOW(GmailApp->app));

	label = gtk_label_new(_("Gmail will attempt to export all your messages to an MBOX format file.\nIt will output them to a file called 'export.mbox' (file chooser coming soon).\nThe export code is not completely reliable yet, use with caution.\nClick OK to continue."));
	gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox),label, FALSE,FALSE,0);
	gtk_widget_show_all(dialog);
	/*set up an infinite loop*/
	for(;;) {
		i = gnome_dialog_run(GNOME_DIALOG(dialog));
		if	(i == 0) {
			gnome_dialog_close(GNOME_DIALOG(dialog));
			break;
		} else if(i == 1) {

			gnome_dialog_close(GNOME_DIALOG(dialog));
			return;
			break;
		} else if(i < 0) {

			/*the user closed the dialog from the window
       manager*/
			return;
			break;
		}
 	}

	if (strcmp(type, "all") == 0) {
			/* g_print("Export all messages to mbox.\n"); */

			/* Create query. */
			query = g_strdup_printf("SELECT headers,message,fromfield,date FROM display,details WHERE direction = \"Incoming\"");

		} else if (strcmp(type, "vfolder") == 0) {
			g_print("Export vfolder to mbox.\n");
			broken_cb(widget, NULL);
			return;

			/* Find vfolder name. */

			/* Create query. */


		} else {
			g_warning(_("export_to_mbox passed bad data."));
			return;
		}

	/* Ask user to give a filename. */
	filename = g_strdup("export.mbox");	/* FIXME: hack */

	/* Open the file. */
	fd = fopen(filename, "a"); /* Open for appending. */
	if (fd == NULL) {
		g_warning(_("Couldn't open file %s"), filename);
		return;
		}

	/* Open Database. */
	if (!check_db(&GmailApp->mysql)) { return; };

	/* Run query fetching messages. */
	if (mysql_query(&GmailApp->mysql, query) != 0) {
		g_print(_("Query failed: %s\n"), mysql_error(&GmailApp->mysql));
		return;
		} 

	if (!check_db(&GmailApp->mysql2)) { return; };

	/* Print query results into the file.. 
	 * For some reason, qpopper strips out the envelope header line:
	 * eg: From blah@hotmail.com 10 Dec 1993 
	 * I'm assuming that all pop3 servers are doing this.
	 * So the hard part here is adding a RFC822 envelope header with the date.
	 *
	 * FIXME: Add this first envelope header line in correctly.
	 */
	res = mysql_store_result(&GmailApp->mysql);

	/* Set the progress bar. */
	total = mysql_num_rows(res);
	current = 0;
	gnome_appbar_set_status (GNOME_APPBAR (GmailApp->appbar), _("Exporting gmail messages to mbox format."));
	gnome_appbar_set_progress (GNOME_APPBAR(GmailApp->appbar), (gfloat) current/total);
	while (g_main_iteration(FALSE));

	while ((row = mysql_fetch_row(res))) {
		gchar *newquery;
		gchar *time, *month, *day;
		gchar *from;


		newquery = g_strdup_printf("SELECT DAYNAME('%s'), MONTHNAME('%s'), DAYOFMONTH('%s'), YEAR('%s')", row[3], row[3], row[3], row[3]);
	if (mysql_query(&GmailApp->mysql2, newquery) != 0) {
		g_print(_("Query failed: %s\n"), mysql_error(&GmailApp->mysql));
		return;
		} 
		res2 = mysql_use_result(&GmailApp->mysql2);
		row2 = mysql_fetch_row(res2);

		/* RFC822 says just use first 3 digits from day/month name. */
		day = g_strndup(row2[0], 3);
		month = g_strndup(row2[1], 3);

		time = strstr(row[3], " ");
		time++;

		/* Create the envelope header. */
		date = g_strdup_printf("%s %s %s %s %s", day, month, row2[2], time, row2[3]);

		from = strip_email(row[2]);

		/* We need to be careful with newlines here.
     * We don't need to give \r\n's.. the pop servers do that.
		 */
		fprintf(fd, "From %s %s\n", from, date);
		fprintf(fd, "%s%s\n", row[0], row[1]); 

		g_free(date);
		g_free(day);
		g_free(month);
		g_free(from);
		mysql_free_result(res2); 

		current++;
		gnome_appbar_set_progress (GNOME_APPBAR(GmailApp->appbar), (gfloat) current/total);
	while (g_main_iteration(FALSE));
		}
		
	/* Close the file. */
	fclose(fd);

	/* Close the database. */
	mysql_free_result(res); 

	/* Report to the user. */
	gnome_appbar_set_status (GNOME_APPBAR (GmailApp->appbar), _("Export of all incoming messages to export.mbox file complete."));
	dialog = gnome_dialog_new (
                 "Export to Mbox",
                 GNOME_STOCK_BUTTON_OK,
                 NULL);
	if (GmailApp->app)
             gnome_dialog_set_parent (GNOME_DIALOG (dialog), GTK_WINDOW (GmailApp->app));
	label = gtk_label_new (_("Export of all incoming messages to export.mbox file is complete."));
	gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (dialog)->vbox), label, TRUE, TRUE, 0);

  gtk_signal_connect_object(GTK_OBJECT(dialog), "clicked", GTK_SIGNAL_FUNC (gtk_widget_destroy), GTK_OBJECT (dialog));

	gtk_widget_show_all(dialog);

	/* Free memory. */
	g_free(query);
}



/* Use drag'n'drop from gnomecard to compose a message. 
 * This code assumes the callback is for an entry widget.
 */
void  
entry_drag_data_received  (GtkWidget          *widget,
			    GdkDragContext     *context,
			    gint                x,
			    gint                y,
			    GtkSelectionData   *data,
			    guint               info,
			    guint               time)
{
	g_return_if_fail(GTK_IS_ENTRY(widget));

	if (strlen (gtk_entry_get_text (GTK_ENTRY(widget))) == 0) {
		gtk_entry_set_text (GTK_ENTRY(widget), data->data);
		return;
	} else {
		gtk_entry_append_text (GTK_ENTRY(widget),", ");
		gtk_entry_append_text (GTK_ENTRY(widget), data->data);
	}
}

/* Save email message to a file.
 * Gnumeric equivalent code was used as a reference here.
 */
void
save_msg (GtkWidget *widget, void *data)
{
	FILE *fd;
	GtkFileSelection *fsel;
	gboolean accepted = FALSE;
	gchar *title;
	GList	*sel = NULL;
	gint id = -1, pos = -1;

	/* Get the mysql id of the selected message. */
	sel = GTK_CLIST(GmailApp->rightlist)->selection;
	if (sel != NULL) pos = (gint) sel->data;
	if (pos < 0) return;
	id = (gint) gtk_clist_get_row_data(GTK_CLIST(GmailApp->rightlist), pos);	
	if (id < 0) return;

	title = g_strdup_printf(_("Save msg %i as"), id);
	fsel = GTK_FILE_SELECTION (gtk_file_selection_new (title));
	g_free(title);

  gtk_window_set_modal (GTK_WINDOW (fsel), TRUE);

	/* Connect the signals for Ok and Cancel */
  gtk_signal_connect (GTK_OBJECT (fsel->ok_button), "clicked",
          GTK_SIGNAL_FUNC (set_ok), &accepted);
  gtk_signal_connect(GTK_OBJECT (fsel->cancel_button), "clicked",
          GTK_SIGNAL_FUNC (gtk_main_quit), NULL);

	gtk_widget_show(GTK_WIDGET(fsel));
	gtk_grab_add (GTK_WIDGET (fsel));
	gtk_main(); 

	if (accepted) {
		char *name = gtk_file_selection_get_filename (fsel);
		Message *msg;

		g_print(_("Saving msg to file %s\n"), name);

		/* Load message. */
		msg = load_vfolder_msg(id);
		if (msg == NULL) return;

		/* Open the file. */
		fd = fopen(name, "a"); /* Open for appending. */
		if (fd == NULL) {
			g_warning(_("Couldn't open file %s"), name);
			return;
			}

			fprintf(fd, "%s\n%s\n", msg->headers->str, msg->message->str);

			fclose(fd);
			free_message(msg); /* Free the memory */
			
	}

	gtk_widget_destroy (GTK_WIDGET (fsel));

	return;
}

/* This function taken from similiar code in gnumeric. */
void
set_ok (GtkWidget *widget, gboolean *dialog_result)
{
  *dialog_result = TRUE;
  gtk_main_quit ();
}

/* Call the Gnome online help for the caching page. */
static void
cache_help (GtkWidget *widget, void *data) {
	static GnomeHelpMenuEntry help_ref = {PACKAGE, "gmail/caching.html" };
	gnome_help_display(0, &help_ref);
}

void
online_help (GtkWidget *widget, gchar *name) {
	GnomeHelpMenuEntry help_ref = {PACKAGE, name };
	gnome_help_display(0, &help_ref);
}

void
copy_cb (GtkWidget *widget, void *data) 
{
	GladeXML *xml; 
	GtkWidget *text;

	/* See if we can get the xml info from the widget. */
	xml = glade_get_widget_tree(GTK_WIDGET (widget));

	/* If not just get it from the msg pane. */
	if (xml == NULL) 
		xml = glade_get_widget_tree(GTK_WIDGET (GmailApp->msgpane));

	if (xml == NULL) return;

	text = glade_xml_get_widget(xml, "textmsg"); 

	gtk_editable_copy_clipboard(GTK_EDITABLE(text));
}

void
cut_cb (GtkWidget *widget, void *data) 
{
	GladeXML *xml; 
	GtkWidget *text;

	/* See if we can get the xml info from the widget. */
	xml = glade_get_widget_tree(GTK_WIDGET (widget));

	/* If not just get it from the msg pane. */
	if (xml == NULL) 
		xml = glade_get_widget_tree(GTK_WIDGET (GmailApp->msgpane));

	if (xml == NULL) return;

	text = glade_xml_get_widget(xml, "textmsg"); 

	gtk_editable_cut_clipboard(GTK_EDITABLE(text));
}

void
paste_cb (GtkWidget *widget, void *data) 
{
	GladeXML *xml; 
	GtkWidget *text;

	/* See if we can get the xml info from the widget. */
	xml = glade_get_widget_tree(GTK_WIDGET (widget));

	/* If not just get it from the msg pane. */
	if (xml == NULL) 
		xml = glade_get_widget_tree(GTK_WIDGET (GmailApp->msgpane));

	if (xml == NULL) return;

	text = glade_xml_get_widget(xml, "textmsg"); 

	gtk_editable_paste_clipboard(GTK_EDITABLE(text));
}

void
clear_cb (GtkWidget *widget, void *data) 
{
	GladeXML *xml; 
	GtkWidget *text;

	/* See if we can get the xml info from the widget. */
	xml = glade_get_widget_tree(GTK_WIDGET (widget));

	/* If not just get it from the msg pane. */
	if (xml == NULL) 
		xml = glade_get_widget_tree(GTK_WIDGET (GmailApp->msgpane));

	if (xml == NULL) return;

	text = glade_xml_get_widget(xml, "textmsg"); 

	/* FIXME: put in the relevent clear function here. */
}

/* select all the text in a message.
 * This works from a composed message.
 * FIXME: make it work in the main window on whatever the curren msg is.
 */
void
select_all_cb (GtkWidget *widget, void *data) 
{
	GladeXML *xml; 
	GtkWidget *text;

	/* See if we can get the xml info from the widget. */
	xml = glade_get_widget_tree(GTK_WIDGET (widget));

	/* If not just get it from the msg pane. */
	if (xml == NULL) 
		xml = glade_get_widget_tree(GTK_WIDGET (GmailApp->msgpane));

	if (xml == NULL) return;

	text = glade_xml_get_widget(xml, "textmsg"); 

	gtk_editable_select_region(GTK_EDITABLE(text), 0, -1);
}
