/*  
 * Gmail. A Gnome email client.
 * Copyright (C) 1999-2000 Wayne Schuller
 *
 * settings.c - Read and parse gmail settings file..etc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include "main.h"

GtkWidget * read_settings(void);
gboolean check_config_version(void);
void update_font_styles(GtkStyle *style);
void version_warning_clicked (GnomeDialog * dialog, gint button_number, gpointer data);
void open_preferences (void);
void vfolder_properties (GtkWidget *widget, void *data);
void save_all_vfolder_data(GtkCTree *ctree);
MailSetup *load_msetup(gchar *name);
MailSetup *load_default_msetup(void);
void free_msetup(MailSetup *msetup);
void mi_warning(GtkWidget *widget, void *data);
void rebuild_vfolder_mi_values (GtkWidget *widget, void *data);
void rebuild_all_mi_values (GtkWidget *widget, void *data);
void query_drag_data_received (GtkWidget *widget, GdkDragContext *context, gint x, gint y, GtkSelectionData *data, guint info, guint time);
void set_mailmenu_options(GtkOptionMenu *mailmenu);
void change_default_msetup(GtkWidget *widget, gchar *name);
void mail_setup_modify(GtkWidget *widget, gchar *modify);
void mail_setup_make_copy(GtkWidget *widget, void *data);
void mail_setup_set_default(GtkWidget *widget, void *data);
void mail_setup_delete(GtkWidget *widget, void *data);

/* notebook pages */
GtkWidget *create_database_page (gboolean full);

/* Local static functions. */
static void apply_prefs (GnomePropertyBox *property_box, gint page_num);
static void dialog_help_cb (GnomePropertyBox *property_box, gint page_num);
static void apply_vfolder_prop_cb (GnomePropertyBox *property_box, gint page_num, GladeXML *xml);
static void property_box_changed(GtkWidget *widget, GdkEventButton *event, GtkWidget *prop_box);
static void toggle_property_box_changed(GtkWidget *widget, GtkWidget *prop_box);
static void apply_mail_setup (GnomePropertyBox *property_box, gint page_num, gboolean new);
static void msetup_clist_selected (GtkWidget *clist, GdkEventButton *event, gpointer data);
static void save_single_vfolder_data(Mailbox *mailbox);
static void vfolder_tree_move (GtkCTree *ctree, GList *node, GList *new_parent, GList *new_sibling, gpointer user_data);
static void mail_setup_help_cb (GnomePropertyBox *property_box, gint page_num);


/* Externally called functions. */
extern void	broken_cb (GtkWidget *widget, void *data);
extern gboolean	DestroyRightPane(void);
extern void     gmail_init(void);
extern void recreate_matched_index (GtkWidget *widget, void *data);
extern void rebuild_mi_values (gchar *query);
extern void vfolder_selected (GtkCTree *ctree, GdkEventButton *event, gpointer data);
extern void button_press (GtkWidget *widget, GdkEventButton *event, gpointer data);
extern gchar * strip_email(gchar *to);

extern _GmailApp_	*GmailApp;

/* Standard DnD email type. */
extern GtkTargetEntry email_field_drop_types[];

/* 
 * Read the settings. 
 * We use the leftpane clist as our data-structure. (This is naughty(TM))
 * Basically the vfolder name is in the clist, and and we attach a Mailbox
 * data structure to each row.
 */
GtkWidget *
read_settings (void) {
	GtkWidget *clist;
	gchar *titles[2] = { "Vfolders"};
	void *iter;
	GtkCTreeNode *node = NULL;

	/* Create default dummy values. */
	GmailApp->rightlist = NULL;
	GmailApp->msglistpane = NULL;
	GmailApp->msgpane = NULL;
	GmailApp->style = NULL;
	GmailApp->boldstyle = NULL;
	GmailApp->search = FALSE;

	GmailApp->sortfield = gnome_config_get_string("/gmail/UserInfo/sortfield");
	if (GmailApp->sortfield == NULL)
		GmailApp->sortfield = g_strdup("date"); 
	GmailApp->sortorder = gnome_config_get_string("/gmail/UserInfo/sortorder");
	if (GmailApp->sortorder == NULL)
		GmailApp->sortorder = g_strdup("DESC");

	/* FIXME: Do we need this assert?
	We are lost without these settings. 
	g_assert(GmailApp->hostname != NULL);
	g_assert(GmailApp->username != NULL);
	g_assert(GmailApp->dbname != NULL);
	*/

	balsa_icons_init();

	/* Create the left ctree holding the vfolders. */
	clist = gtk_ctree_new_with_titles (1, 0, titles);

	gtk_signal_connect(GTK_OBJECT(clist), "button_press_event",
                       GTK_SIGNAL_FUNC(vfolder_selected),
                       NULL);

	/* We want to be able to reorder with dnd. */
	gtk_ctree_set_reorderable(GTK_CTREE(clist), TRUE);

	/* Use _after so we don't save the info till the reorder is done. */
	gtk_signal_connect_after(GTK_OBJECT(clist), "tree_move",
                       GTK_SIGNAL_FUNC(vfolder_tree_move),
                       NULL);

	/* Loop through mailboxes. */
	iter = gnome_config_init_iterator("/gmail/vfolders");

	while (TRUE) {
		Mailbox	*mailbox;
		gchar *text[1];
		gchar *name = NULL;
		gchar *query = NULL;
		gchar *config, *email;
		GdkPixmap *pixmap = NULL;
		GdkBitmap *bitmap = NULL;

		iter = gnome_config_iterator_next(iter, &name, &query);
		if (iter == NULL) break;

		text[0] = g_strdup(name);

		/* Work out what icon to show. */
		if (strcmp(name, "Inbox") == 0) {
			pixmap = balsa_icon_get_pixmap(BALSA_ICON_INBOX);
			bitmap = balsa_icon_get_bitmap(BALSA_ICON_INBOX);
		} else if (strcmp(name, "Outbox") == 0) {
			pixmap = balsa_icon_get_pixmap(BALSA_ICON_OUTBOX);
			bitmap = balsa_icon_get_bitmap(BALSA_ICON_OUTBOX);
		} else if (strcmp(name, "Unread") == 0) {
			pixmap = balsa_icon_get_pixmap(BALSA_ICON_TRAY_FULL);
			bitmap = balsa_icon_get_bitmap(BALSA_ICON_TRAY_FULL);
		} else {
			pixmap = balsa_icon_get_pixmap(BALSA_ICON_TRAY_EMPTY);
			bitmap = balsa_icon_get_bitmap(BALSA_ICON_TRAY_EMPTY);

			/* Example of using gnome stock icons.
			gnomepixmap = gnome_stock_new_with_icon(GNOME_STOCK_MENU_MAIL);
			pixmap = GNOME_PIXMAP(gnomepixmap)->pixmap;
			bitmap = GNOME_PIXMAP(gnomepixmap)->mask;
			*/
		}

		/* Put it in the ctree. */
		node = gtk_ctree_insert_node (GTK_CTREE (clist), NULL, GTK_CTREE_NODE(node), text, 4, pixmap, bitmap, pixmap, bitmap, TRUE, FALSE); 
		g_free(text[0]);  

		/* Add stock icon to the clist vfolder name. 
		 * Experimental.
		gnomepixmap = gnome_stock_new_with_icon(GNOME_STOCK_MENU_MAIL);
		gtk_clist_set_pixtext(GTK_CLIST (clist), row, 0, name, 3, GNOME_PIXMAP(gnomepixmap)->pixmap, GNOME_PIXMAP(gnomepixmap)->mask);  
		 */

		config = g_strdup_printf("/gmail/vfolder_email_default/%s", name);	
		email = gnome_config_get_string(config);

		/* Create new mailbox.   */
		mailbox = g_malloc (sizeof(Mailbox)); 
		/* Do we need to strdup the next two? */
		mailbox->name = g_strdup(name);
		mailbox->query = g_strdup(query);
		mailbox->email = g_strdup(email);

		gtk_ctree_node_set_row_data(GTK_CTREE (clist), GTK_CTREE_NODE(node), mailbox);
		}


	/* This call must appear below the above 4 clist commands. Weird. */
	update_font_styles(clist->style);
	gtk_widget_set_style (clist, GmailApp->style);      

	/* Remove deprecated config settings. */

	/* POP3 servers deprecated from version 0.6.0 and earlier. */
	if (gnome_config_has_section("/gmail/POP3Servers")) {
		gchar *val;

		g_print("Removing deprecated pop3 section in ~/.gnome/gmail\n");

		/* Copy over settings into the new style format (mailsetup: default). */
		gnome_config_set_string("/gmail/mailsetups/default", "default");

		val = gnome_config_get_string("/gmail/POP3Servers/username");
		gnome_config_set_string("/gmail/mailsetup_default/username", val);
		g_free(val);
		val = gnome_config_get_string("/gmail/POP3Servers/password");
		gnome_config_private_set_string("/gmail/mailsetup_default/password", val);
		g_free(val);
		val = gnome_config_get_string("/gmail/POP3Servers/servername");
		gnome_config_set_string("/gmail/mailsetup_default/servername", val);
		g_free(val);
		gnome_config_set_bool("/gmail/mailsetup_default/leavemailonserver", TRUE);
		val = gnome_config_get_string("/gmail/UserInfo/smtp");
		gnome_config_set_string("/gmail/mailsetup_default/smtp", val);
		g_free(val);


		gnome_config_clean_section("/gmail/POP3Servers");	
		gnome_config_clean_key("/gmail/UserInfo/smtp");
		gnome_config_sync();	
		}

	/* mailbox deprecated from version 0.6.0 and earlier. */
	if (gnome_config_has_section("/gmail/mailbox")) {
		g_print("Removing deprecated mailboxes section in ~/.gnome/gmail\n");
		gnome_config_clean_section("/gmail/mailbox");	
		gnome_config_sync();	
		}

	GmailApp->vfolders_clean = TRUE; /* Reset this flag. */

	return clist;
}


/* Create the database stuff. We also call this from the gnome druid, so we
 * have a special 'full' option to decide whether we want to see all the 
 * settings or just the basics. In the druid we full=FALSE so we just get
 * the basics.
 */
GtkWidget *
create_database_page (gboolean full) {
  	GtkWidget *vbox;
  	GtkWidget *frame;
  	GtkWidget *label;
  	GtkWidget *table;

  	vbox = gtk_vbox_new (FALSE, 0);
  	gtk_container_set_border_width (GTK_CONTAINER (vbox), 10);

  	frame = gtk_frame_new ("Mysql Database Info");
  	gtk_box_pack_start (GTK_BOX (vbox), frame, FALSE, FALSE, 5);

  	table = gtk_table_new (4, 7, FALSE);
  	gtk_container_add (GTK_CONTAINER (frame), table);

	/* FIXME: this dumb label doesn't use the width of the table. :( 
	labelintro = gtk_label_new("Vfolders require a mysql server. It's good. These basic settings should have been done for you by the gnome-druid. See the README");

  	gtk_table_attach_defaults (GTK_TABLE (table), labelintro, 0, 3, 0, 1);
	gtk_label_set_justify(GTK_LABEL (labelintro), GTK_JUSTIFY_LEFT);
	gtk_label_set_line_wrap (GTK_LABEL (labelintro), TRUE); */

  	label = gtk_label_new ("Mysql server hostname:");
  	gtk_table_attach (GTK_TABLE (table), label, 0, 1, 1, 2,
		    GTK_FILL, GTK_FILL, 10, 10);
	gtk_label_set_justify(GTK_LABEL (label), GTK_JUSTIFY_RIGHT);

  	GmailApp->db_host_entry = gtk_entry_new ();
  	gtk_table_attach (GTK_TABLE (table), GmailApp->db_host_entry, 1, 3, 1, 2,
		    GTK_EXPAND | GTK_FILL, GTK_FILL, 0, 10);

  label = gtk_label_new ("Mysql user name:");
  gtk_table_attach (GTK_TABLE (table), label, 0, 1, 2, 3,
		    GTK_FILL, GTK_FILL, 10, 10);
  GmailApp->db_user_entry = gtk_entry_new ();
  gtk_table_attach (GTK_TABLE (table), GmailApp->db_user_entry, 1, 3, 2, 3,
		    GTK_EXPAND | GTK_FILL, GTK_FILL, 0, 10);

  label = gtk_label_new ("Mysql user password:");
  gtk_table_attach (GTK_TABLE (table), label, 0, 1, 3, 4,
		    GTK_FILL, GTK_FILL, 10, 10);
  GmailApp->db_pass_entry = gtk_entry_new ();
	gtk_entry_set_visibility(GTK_ENTRY(GmailApp->db_pass_entry), FALSE);
  gtk_table_attach (GTK_TABLE (table), GmailApp->db_pass_entry, 1, 3, 3, 4,
		    GTK_EXPAND | GTK_FILL, GTK_FILL, 0, 10);

  label = gtk_label_new ("Mysql database name:");
  gtk_table_attach (GTK_TABLE (table), label, 0, 1, 4, 5,
		    GTK_FILL, GTK_FILL, 10, 10);
  GmailApp->db_name_entry = gtk_entry_new ();
  gtk_table_attach (GTK_TABLE (table), GmailApp->db_name_entry, 1, 3, 4, 5,
		    GTK_EXPAND | GTK_FILL, GTK_FILL, 0, 10);

  return vbox;
}


gboolean
check_config_version () {
	gint version;

	version = gnome_config_get_int("/gmail/UserInfo/DatabaseVersion");
	if (version == DATABASE_VERSION) {
		return(TRUE);
	} else {
		GtkWidget *dialog, *label;
		gchar *msg;

		dialog = gnome_dialog_new("Gmail Warning", GNOME_STOCK_BUTTON_OK, GNOME_STOCK_BUTTON_CANCEL, NULL);
		gtk_window_set_modal(GTK_WINDOW (dialog), TRUE);

		msg = g_strdup_printf("Warning: Your database version number is %i, but this version of Gmail requires version %i.\nYou should probably delete your ~/.gnome/gmail and setup a new database.\nIf you want to attempt to transfer your database, read README.export_import\nYou can click ok here to continue running gmail, but things will probably go wrong.\nOtherwise click on cancel to give up.", version, DATABASE_VERSION);
		label = gtk_label_new (msg);
		gtk_box_pack_start (GTK_BOX (GNOME_DIALOG(dialog)->vbox), label, FALSE, FALSE, 5);
		gtk_widget_show_all(GNOME_DIALOG(dialog)->vbox);

		switch(gnome_dialog_run_and_close(GNOME_DIALOG (dialog))) {
    			case 0: return(TRUE);
			break;
	
 		   	case 1: return(FALSE);
			break;

 	   		default: break;
		}
	}
	
	return(FALSE);
}

void
version_warning_clicked (GnomeDialog * dialog, gint button_number, gpointer data)
{
	gtk_widget_destroy(GTK_WIDGET(dialog));

	switch (button_number) {
		case 0: /* OK button */
		gmail_init(); /* Normal Gmail start */
		break;
		case 1: /* Cancel button */
		gtk_main_quit ();
		return;
		break;
		}
}

/* update_font_styles - set the global font variables based on gnome config
 * settings.
 * Called in read_settings at start of program and also when the fonts
 * are updated.
 * FIXME: It is passed the clist in use in order to be able to copy the
 * style. I need to do this to obey gtk themes. There must be a better way
 * to just get a copy of the current gtk theme style.
 * FIXME: we need to free the data from gnome_config_get_string
 */
void
update_font_styles(GtkStyle *style)
{
	/* copy the current style. */
	GmailApp->style = gtk_style_copy(style);  
	if (gnome_config_get_string("/gmail/fonts/mainfont") != NULL) {
	gdk_font_unref(GmailApp->style->font);
	GmailApp->style->font = gdk_font_load (gnome_config_get_string("/gmail/fonts/mainfont")); 
		}

	/* copy the current style. */
	GmailApp->boldstyle = gtk_style_copy(style); 
	if (gnome_config_get_string("/gmail/fonts/boldfont") != NULL) {
		gdk_font_unref(GmailApp->boldstyle->font);
		GmailApp->boldstyle->font = gdk_font_load (gnome_config_get_string("/gmail/fonts/boldfont"));
		}
}

/* vfolder_properties - open the properties window for the vfolder. */
void
vfolder_properties (GtkWidget *widget, void *data)
{
	Mailbox	*mailbox = NULL;
	GladeXML *xml;
	gchar *glade_src;
	GtkWidget *vfolder_prop, *wid;
	gint row = -1;

	/* Get the vfolder data. */
	mailbox = gtk_ctree_node_get_row_data(GTK_CTREE (GmailApp->mblist), GTK_CTREE_NODE (GmailApp->node));

	g_return_if_fail(mailbox != NULL);

	/* g_print("Row: %i, Name: %s, Query: %s\n", row, mailbox->name, mailbox->query); */

	/* Load the glade file. */
	glade_src = g_strdup_printf("%s/vfolder_properties.glade", GMAIL_GLADEDIR);
	xml = glade_xml_new(glade_src, "vfolder_properties");
	g_free(glade_src);

	if (!xml) {
		g_warning("Unable to load xml interface. Are the gmail glade files installed onto your system?\n");
		exit(1);
		}

	glade_xml_signal_autoconnect(xml);

	vfolder_prop = glade_xml_get_widget(xml, "vfolder_properties");

	wid = glade_xml_get_widget(xml, "name"); 
	gtk_entry_set_text(GTK_ENTRY(wid), mailbox->name);
	gtk_object_set_data(GTK_OBJECT(wid), "row", GINT_TO_POINTER(row));

	gtk_signal_connect(GTK_OBJECT(wid), "key-press-event",
                       GTK_SIGNAL_FUNC(property_box_changed),
                       GNOME_PROPERTY_BOX(vfolder_prop));  

	wid = glade_xml_get_widget(xml, "defaultemail"); 
	if (mailbox->email) gtk_entry_set_text(GTK_ENTRY(wid), mailbox->email);

	gtk_signal_connect(GTK_OBJECT(wid), "key-press-event",
                       GTK_SIGNAL_FUNC(property_box_changed),
                       GNOME_PROPERTY_BOX(vfolder_prop));  

	wid = glade_xml_get_widget(xml, "query"); 
	gtk_text_insert (GTK_TEXT (wid), NULL, NULL, NULL, mailbox->query, -1);

	/* Setup the query text as a dnd target. */
	gtk_drag_dest_set (wid, GTK_DEST_DEFAULT_ALL,
		     email_field_drop_types, 1,
		     GDK_ACTION_COPY); 
	gtk_signal_connect(GTK_OBJECT(wid), "drag_data_received",
		      GTK_SIGNAL_FUNC(query_drag_data_received), NULL);


	gtk_signal_connect(GTK_OBJECT(wid), "key-press-event",
                       GTK_SIGNAL_FUNC(property_box_changed),
                       GNOME_PROPERTY_BOX(vfolder_prop));  

	glade_src = g_strdup_printf("Vfolder Properties - %s", mailbox->name);
	gtk_window_set_title(GTK_WINDOW (vfolder_prop), glade_src);
	g_free(glade_src);

	gtk_signal_connect(GTK_OBJECT(vfolder_prop), "help",
                       GTK_SIGNAL_FUNC(dialog_help_cb),
                       NULL);

	gtk_signal_connect(GTK_OBJECT(vfolder_prop), "apply",
                       GTK_SIGNAL_FUNC(apply_vfolder_prop_cb),
                       xml);

	gtk_signal_connect(GTK_OBJECT(vfolder_prop), "destroy",
                       GTK_SIGNAL_FUNC(mi_warning), NULL);

	
	if (GmailApp->app)
		gnome_dialog_set_parent (GNOME_DIALOG (vfolder_prop), GTK_WINDOW (GmailApp->app));

	gtk_widget_show(vfolder_prop);
}

static void
dialog_help_cb (GnomePropertyBox *property_box, gint page_num)
{
	static GnomeHelpMenuEntry help_ref = {PACKAGE, "gmail/vfolder.html" };

	gnome_help_display(0, &help_ref);
}

static void
apply_vfolder_prop_cb (GnomePropertyBox *property_box, gint page_num, GladeXML *xml)
{
	gchar *name, *query, *email;
	GtkWidget *wid;
	Mailbox	*mailbox = NULL;
	gint row;
	GdkPixmap *pixmap = NULL;
	GdkBitmap *bitmap = NULL;
	guint8 spacing;
	
	if (page_num != -1) return;  /* Only apply once. */

	/* Get the vfolder name */
	wid = glade_xml_get_widget(xml, "name"); 
	row = (gint) gtk_object_get_data(GTK_OBJECT(wid), "row");
	name = gtk_entry_get_text(GTK_ENTRY(wid));

	/* Apply the default email address. */
	wid = glade_xml_get_widget(xml, "defaultemail"); 
	email = gtk_entry_get_text(GTK_ENTRY(wid));

	/* Get the query */
	wid = glade_xml_get_widget(xml, "query"); 
	query = gtk_editable_get_chars(GTK_EDITABLE (wid), 0, -1); 

	/* Get all the node values. */
	gtk_ctree_get_node_info(GTK_CTREE (GmailApp->mblist), GTK_CTREE_NODE (GmailApp->node), NULL, &spacing, &pixmap, &bitmap, NULL, NULL, NULL, NULL); 
	gtk_ctree_node_set_pixtext(GTK_CTREE (GmailApp->mblist), GTK_CTREE_NODE (GmailApp->node), 0, name, spacing, pixmap, bitmap); 

	/* Get a ptr to the old mailbox structure. */
	mailbox = gtk_ctree_node_get_row_data(GTK_CTREE (GmailApp->mblist), GTK_CTREE_NODE (GmailApp->node));

	/* Update the mailbox struct with the new values. */
	g_free(mailbox->name);
	mailbox->name = g_strdup(name);
	g_free(mailbox->email);
	mailbox->email = g_strdup(email);
	g_free(mailbox->query);
	mailbox->query = g_strdup(query);

	/* Write the entire clist to the config file */
	save_all_vfolder_data(GTK_CTREE(GmailApp->mblist)); 
	
	/* g_print("apply! %s, %s, %s \n", name, email, query); */
}

/* save_all_vfolder_data.
 * Save all our vfolder ctree data to the gnome config settings.
 * Overwrites all the old gnome_config data.
 */
void
save_all_vfolder_data(GtkCTree *ctree)
{
	Mailbox	*mailbox = NULL;
	GtkCTreeNode *node;

	/* Clear the gnome config settings. 
	 * FIXME: Is this too dangerous? If we crash after this we lost all the
	 * vfolder data. :(
	 */
	gnome_config_clean_section("/gmail/vfolders");
	gnome_config_clean_section("/gmail/vfolder_email_default");
	gnome_config_sync();

	/* Iterate over the vfolder ctree */
	node = gtk_ctree_node_nth(GTK_CTREE(ctree), 0);
	while (node != NULL) {
		mailbox = gtk_ctree_node_get_row_data(GTK_CTREE (ctree), GTK_CTREE_NODE (node));
		save_single_vfolder_data(mailbox);
		node = GTK_CTREE_NODE_NEXT(node);
	}

	GmailApp->vfolders_clean = FALSE; /* unclean = been changed. */
}

/* save_single_vfolder_data
 * Saves the particular single vfolder to the gnome config settings.
 */
static void 
save_single_vfolder_data(Mailbox *mailbox)
{
	gchar *config;

	g_return_if_fail(mailbox != NULL);

	/* Save the query. */
	config = g_strdup_printf("/gmail/vfolders/%s", mailbox->name);
	gnome_config_set_string(config, mailbox->query);
	g_free(config);

	/* Save the default email address. */
	config = g_strdup_printf("/gmail/vfolder_email_default/%s", mailbox->name);
	if (mailbox->email != NULL) gnome_config_set_string(config, mailbox->email);
	g_free(config);

	gnome_config_sync();
}

/* If the vfolders have been modified and the user has caching turned on
 * then we give a warning window to update the matched index.
 */
void mi_warning(GtkWidget *widget, void *data)
{
	static GnomeHelpMenuEntry help_ref = {PACKAGE, "gmail/caching.html" };
	GtkWidget *dialog, *label;

	/* Warn/Report to the user about caching stuff.  */
	if (GmailApp->vfolders_clean) return; /* Nothing changed. */
	if (!cache_vfolders()) return; /* Not using caching, no warning needed. */

	dialog = gnome_dialog_new ( "Gmail Popup", GNOME_STOCK_PIXMAP_INDEX, GNOME_STOCK_BUTTON_OK, GNOME_STOCK_BUTTON_HELP, NULL);
	label = gtk_label_new ("Vfolder settings modified.\nBecause you have vfolder caching turned on, it is suggested that\nyou run Settings->Advanced->Recreate Matched Index.\nTo do this immediately, click on 'Index', otherwise click 'OK' to continue.\nClick on 'Help' if you do not understand this message."); 

	if (GmailApp->app)
		gnome_dialog_set_parent (GNOME_DIALOG (dialog), GTK_WINDOW (GmailApp->app));
	gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (dialog)->vbox), label, TRUE, TRUE, 0);	
	gtk_widget_show_all(GNOME_DIALOG(dialog)->vbox);

	switch(gnome_dialog_run_and_close(GNOME_DIALOG (dialog))) {
    		case 0: recreate_matched_index(NULL, NULL);
			break;
	
		case 1: break;

		case 2: gnome_help_display(0, &help_ref);

		default: break;
	}

	GmailApp->vfolders_clean = TRUE;

}

/* The next cluster of functions are simply shortcuts for marking a propbox
 * as changing. I don't know if there is an easier way to do this.
 * I wasn't able to get the signal callbacks to call the gnome-libs function
 * directly.
 * So basically they are wrappers for gnome_property_box_changed.
 * There are two different types based on the different callback arguments
 * for different widgets.
 */
static void
property_box_changed(GtkWidget *widget, GdkEventButton *event, GtkWidget *prop_box)
{
	gnome_property_box_changed(GNOME_PROPERTY_BOX(prop_box)); 
}

static void
toggle_property_box_changed(GtkWidget *widget, GtkWidget *prop_box)
{
	gnome_property_box_changed(GNOME_PROPERTY_BOX(prop_box)); 
}


/* Callback for rebuilding the mi values for all messages. */
void rebuild_all_mi_values (GtkWidget *widget, void *data) 
{
	gchar *query = NULL;
	GtkWidget *dialog, *label;
	gint i, rw;

	query = g_strdup_printf("SELECT (id) FROM display ORDER BY id DESC");

	/* Send warning dialog. */
	dialog = gnome_dialog_new("Gmail Warning:",
                        GNOME_STOCK_BUTTON_OK,
                        GNOME_STOCK_BUTTON_CANCEL,
                        NULL);
 	if (GmailApp->app)
		gnome_dialog_set_parent(GNOME_DIALOG(dialog), GTK_WINDOW(GmailApp->app));

	label = gtk_label_new("Rebuilding the MI values is important for fast gmail caching.\nDon't run this if you haven't updated your matched-index or\nif you haven't tested that all your vfolders work.\nYou should have an up to date matched-index (MI) before running this.\nClick OK to continue.");
	gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox),label, FALSE,FALSE,0);
	gtk_widget_show_all(dialog);

	/* set up an infinite loop */
	for(;;) {
		i = gnome_dialog_run(GNOME_DIALOG(dialog));
		if	(i == 0) {
			gnome_dialog_close(GNOME_DIALOG(dialog));
			break;
		} else if (i == 1) {

			gnome_dialog_close(GNOME_DIALOG(dialog));
			return;
			break;
		} else if (i < 0) {

			/*the user closed the dialog from the window
       manager*/
			return;
			break;
		}
 	}


	/* Loop through each vfolder clearing them as containing no unread
	 * messages. The ones that really do have unread messages will get 
	 * marked as such when we process every message.
	 */
	for (rw = 0; rw != GTK_CLIST (GmailApp->mblist)->rows; rw++) {
		gtk_clist_set_cell_style(GTK_CLIST (GmailApp->mblist), rw, 0, GmailApp->style);
	}

	rebuild_mi_values(query);
	
	g_free(query);
}

/* Callback for rebuilding mi values for a particular vfolder. */
void rebuild_vfolder_mi_values (GtkWidget *widget, void *data)
{
	Mailbox	*mailbox;
	gchar *query;

	mailbox = gtk_ctree_node_get_row_data(GTK_CTREE (GmailApp->mblist), GTK_CTREE_NODE (GmailApp->node));

	g_return_if_fail(mailbox != NULL);

	g_print("Rebuilding MI values for vfolder: %s.\n", mailbox->name);

	/* It's ok to use the cache values here. */
	if (cache_vfolders()) 
		query = g_strdup_printf("SELECT id FROM display WHERE FIND_IN_SET('%s', matched)", mailbox->name);
	else
		query = g_strdup_printf("SELECT display.id FROM display,details WHERE display.id = details.id AND (%s)", mailbox->query);

	rebuild_mi_values(query);
}

/* Open the preferences window.
 * Put in the settings for the user to see and modify.
 */
void 
open_preferences ()
{
	GladeXML *xml;
	gchar *glade_src;
	GtkWidget *wid;
	void *iter;
	static GnomeHelpMenuEntry help_ref = {PACKAGE, "gmail/prefs" };
	GtkAdjustment *adj;

	/* If prefs are open don't do anything. 
	if (GmailApp->prefs) return; */

	/* Load the glade file. */
	glade_src = g_strdup_printf("%s/preferences.glade", GMAIL_GLADEDIR);
	xml = glade_xml_new(glade_src, "propertybox1");
	g_free(glade_src);

	if (!xml) {
		g_warning("Unable to load xml interface. Are the gmail glade files installed onto your system?\n");
		exit(1);
		}

	glade_xml_signal_autoconnect(xml);

	GmailApp->prefs = glade_xml_get_widget(xml, "propertybox1");

	gtk_signal_connect(GTK_OBJECT(GmailApp->prefs), "apply",
                       GTK_SIGNAL_FUNC(apply_prefs), NULL);

	gtk_signal_connect(GTK_OBJECT(GmailApp->prefs), "help",
                       GTK_SIGNAL_FUNC(gnome_help_pbox_display),
                       &help_ref);

	gtk_signal_connect_object (GTK_OBJECT (GmailApp->prefs), "destroy", GTK_SIGNAL_FUNC (gtk_object_unref), GTK_OBJECT (xml));

	/* Identity Tab */
	wid = glade_xml_get_widget(xml, "name"); 
	gtk_entry_set_text (GTK_ENTRY (wid), gnome_config_get_string("/gmail/UserInfo/RealName"));
	gtk_signal_connect(GTK_OBJECT(wid), "key-press-event",
                       GTK_SIGNAL_FUNC(property_box_changed),
                       GNOME_PROPERTY_BOX(GmailApp->prefs));  

	wid = glade_xml_get_widget(xml, "email"); 
	gtk_entry_set_text (GTK_ENTRY (wid), gnome_config_get_string("/gmail/UserInfo/EmailAddress"));
	gtk_signal_connect(GTK_OBJECT(wid), "key-press-event",
                       GTK_SIGNAL_FUNC(property_box_changed),
                       GNOME_PROPERTY_BOX(GmailApp->prefs));  

	wid = glade_xml_get_widget(xml, "sig"); 
	gtk_entry_set_text (GTK_ENTRY (wid), gnome_config_get_string("/gmail/UserInfo/Signature"));
	gtk_signal_connect(GTK_OBJECT(wid), "changed",
                       GTK_SIGNAL_FUNC(toggle_property_box_changed),
                       GNOME_PROPERTY_BOX(GmailApp->prefs));  

	wid = glade_xml_get_widget(xml, "ab"); 
	gtk_entry_set_text (GTK_ENTRY (wid), gnome_config_get_string("/gmail/UserInfo/AddressBook=gnomecard"));
	gtk_signal_connect(GTK_OBJECT(wid), "key-press-event",
                       GTK_SIGNAL_FUNC(property_box_changed),
                       GNOME_PROPERTY_BOX(GmailApp->prefs));  


	/* Database Tab */
	wid = glade_xml_get_widget(xml, "sqlserver"); 
	gtk_entry_set_text (GTK_ENTRY (wid), gnome_config_get_string("/gmail/MysqlServer/hostname"));
	gtk_signal_connect(GTK_OBJECT(wid), "key-press-event",
                       GTK_SIGNAL_FUNC(property_box_changed),
                       GNOME_PROPERTY_BOX(GmailApp->prefs));  

	wid = glade_xml_get_widget(xml, "sqluser"); 
	gtk_entry_set_text (GTK_ENTRY (wid), gnome_config_get_string("/gmail/MysqlServer/username"));
	gtk_signal_connect(GTK_OBJECT(wid), "key-press-event",
                       GTK_SIGNAL_FUNC(property_box_changed),
                       GNOME_PROPERTY_BOX(GmailApp->prefs));  

	wid = glade_xml_get_widget(xml, "sqlpass"); 
	gtk_entry_set_text (GTK_ENTRY (wid), gnome_config_private_get_string("/gmail/MysqlServer/password"));
	gtk_signal_connect(GTK_OBJECT(wid), "key-press-event",
                       GTK_SIGNAL_FUNC(property_box_changed),
                       GNOME_PROPERTY_BOX(GmailApp->prefs));  

	wid = glade_xml_get_widget(xml, "sqldb"); 
	gtk_entry_set_text (GTK_ENTRY (wid), gnome_config_get_string("/gmail/MysqlServer/dbname"));
	gtk_signal_connect(GTK_OBJECT(wid), "key-press-event",
                       GTK_SIGNAL_FUNC(property_box_changed),
                       GNOME_PROPERTY_BOX(GmailApp->prefs));  

	/* Set vfolder cache toggle. */
	wid = glade_xml_get_widget(xml, "cache"); 
	if (gnome_config_get_bool("/gmail/UserInfo/CacheVfolders") == FALSE) 
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON (wid), FALSE);
	else 
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON (wid), TRUE);
	gtk_signal_connect(GTK_OBJECT(wid), "toggled",
                       GTK_SIGNAL_FUNC(toggle_property_box_changed),
                       GNOME_PROPERTY_BOX(GmailApp->prefs));  

	/* Mail Servers tab. */
	GmailApp->msetup_clist = glade_xml_get_widget(xml, "clist1"); 

	/* Loop through each gnome_config mail setup, creating a clist entry. */	
	iter = gnome_config_init_iterator("/gmail/mailsetups");

	while (TRUE) {
		MailSetup *msetup;
		gchar *text[2];
		gchar *name, *value;

		iter = gnome_config_iterator_next(iter, &name, &value);
		if (iter == NULL) break;

		/* Put it in the clist. */
		text[0] = g_strdup(value);
		text[1] = g_strdup(name);
		gtk_clist_prepend (GTK_CLIST (GmailApp->msetup_clist), text); 
		g_free(text[0]); 
		g_free(text[1]); 

		/* Create new mailsetup.   */
		msetup = load_msetup(name);

		/* Attach mailsetup as clist row data. */
		gtk_clist_set_row_data(GTK_CLIST (GmailApp->msetup_clist), 0, msetup);
		}

	gtk_signal_connect(GTK_OBJECT(GmailApp->msetup_clist), "button_press_event", GTK_SIGNAL_FUNC(msetup_clist_selected), NULL);

	/* Appearances tab. */
	wid = glade_xml_get_widget(xml, "fontpicker1"); 
	gnome_font_picker_set_font_name(GNOME_FONT_PICKER(wid), gnome_config_get_string("/gmail/fonts/mainfont"));
	gnome_font_picker_fi_set_use_font_in_label(GNOME_FONT_PICKER (wid), TRUE, 15);
	gtk_signal_connect(GTK_OBJECT(wid), "clicked",
                       GTK_SIGNAL_FUNC(toggle_property_box_changed),
                       GNOME_PROPERTY_BOX(GmailApp->prefs));  

	wid = glade_xml_get_widget(xml, "fontpicker2"); 
	gnome_font_picker_set_font_name(GNOME_FONT_PICKER(wid), gnome_config_get_string("/gmail/fonts/boldfont"));
	gnome_font_picker_fi_set_use_font_in_label(GNOME_FONT_PICKER (wid), TRUE, 15);
	gtk_signal_connect(GTK_OBJECT(wid), "clicked",
                       GTK_SIGNAL_FUNC(toggle_property_box_changed),
                       GNOME_PROPERTY_BOX(GmailApp->prefs));  

	
	/* Set twopane or three pane mode. (default is three pane). */
	if (TwoPaneView()) 
		wid = glade_xml_get_widget(xml, "twopaneview"); 
	else
		wid = glade_xml_get_widget(xml, "threepaneview"); 
 	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (wid), TRUE); 

	gtk_signal_connect(GTK_OBJECT(wid), "toggled",
                       GTK_SIGNAL_FUNC(toggle_property_box_changed),
                       GNOME_PROPERTY_BOX(GmailApp->prefs));  

#if 0
	wid = glade_xml_get_widget(xml, "window_width"); 
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(wid), gnome_config_get_int_with_default("/gmail/Geometry/width=850", NULL));
	gtk_signal_connect(GTK_OBJECT(wid), "value_changed",
                       GTK_SIGNAL_FUNC(toggle_property_box_changed),
                       GNOME_PROPERTY_BOX(GmailApp->prefs));  

	wid = glade_xml_get_widget(xml, "window_height"); 
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(wid), gnome_config_get_int_with_default("/gmail/Geometry/height=650", NULL));
	gtk_signal_connect(GTK_OBJECT(wid), "value_changed",
                       GTK_SIGNAL_FUNC(toggle_property_box_changed),
                       GNOME_PROPERTY_BOX(GmailApp->prefs));  
#endif

	wid = glade_xml_get_widget(xml, "openfirst"); 
	if (gnome_config_get_bool("/gmail/UserInfo/display_first_message") == TRUE) {
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON (wid), TRUE);
	} else {
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON (wid), FALSE);
	}
	gtk_signal_connect(GTK_OBJECT(wid), "toggled",
                       GTK_SIGNAL_FUNC(toggle_property_box_changed),
                       GNOME_PROPERTY_BOX(GmailApp->prefs));  

	wid = glade_xml_get_widget(xml, "follow_sort"); 
	if (gnome_config_get_bool("/gmail/UserInfo/follow_sort") == TRUE) {
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON (wid), TRUE);
	} else {
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON (wid), FALSE);
	}
	gtk_signal_connect(GTK_OBJECT(wid), "toggled",
                       GTK_SIGNAL_FUNC(toggle_property_box_changed),
                       GNOME_PROPERTY_BOX(GmailApp->prefs));  

	wid = glade_xml_get_widget(xml, "wordwrap"); 
	if (gnome_config_get_bool("/gmail/UserInfo/wrap_text") == TRUE) {
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON (wid), TRUE);

		/* FIXME: Unghost the spin button. */
	} else {
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON (wid), FALSE);

		/* FIXME: Ghost the spinbutton. */
	}

	gtk_signal_connect(GTK_OBJECT(wid), "toggled",
                       GTK_SIGNAL_FUNC(toggle_property_box_changed),
                       GNOME_PROPERTY_BOX(GmailApp->prefs));  

	/* Set the wrap length. */
	wid = glade_xml_get_widget(xml, "ww_len"); 
	gtk_spin_button_set_value (GTK_SPIN_BUTTON(wid), gnome_config_get_int("/gmail/UserInfo/wrap_len"));

	adj = gtk_spin_button_get_adjustment(GTK_SPIN_BUTTON(wid));

	gtk_signal_connect(GTK_OBJECT(adj), "value_changed",
                       GTK_SIGNAL_FUNC(toggle_property_box_changed),
                       GNOME_PROPERTY_BOX(GmailApp->prefs));  


	wid = glade_xml_get_widget(xml, "fullfrom"); 
	if (gnome_config_get_bool("/gmail/UserInfo/fullfrom=FALSE") == TRUE) {
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON (wid), TRUE);
	} else {
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON (wid), FALSE);
	}

	gtk_signal_connect(GTK_OBJECT(wid), "toggled",
                       GTK_SIGNAL_FUNC(toggle_property_box_changed),
                       GNOME_PROPERTY_BOX(GmailApp->prefs));  

	/* Do we need the below line? */
	/* gtk_signal_connect_object (GTK_OBJECT (GmailApp->prefs), "destroy", GTK_SIGNAL_FUNC (gtk_object_unref), GTK_OBJECT (xml)); */

 	if (GmailApp->app)
		gnome_dialog_set_parent (GNOME_DIALOG (GmailApp->prefs), GTK_WINDOW (GmailApp->app));  

	gtk_widget_show_all(GmailApp->prefs);
}

/* Save all the settings that are in the prefs to the gnome_config file. */
static void	
apply_prefs (GnomePropertyBox *property_box, gint page_num)
{
	GladeXML *xml;
	GtkWidget *wid;
	gint row;
	void *iter;

	if (page_num != -1) return;  /* Only apply once. */

	xml = glade_get_widget_tree(GTK_WIDGET(property_box));

	/* Identity Tab */
	wid = glade_xml_get_widget(xml, "name"); 
	gnome_config_set_string("/gmail/UserInfo/RealName", gtk_entry_get_text(GTK_ENTRY(wid)));
	wid = glade_xml_get_widget(xml, "email"); 
	gnome_config_set_string("/gmail/UserInfo/EmailAddress", gtk_entry_get_text(GTK_ENTRY(wid)));
	wid = glade_xml_get_widget(xml, "sig"); 
	gnome_config_set_string("/gmail/UserInfo/Signature", gtk_entry_get_text(GTK_ENTRY(wid)));
	wid = glade_xml_get_widget(xml, "ab"); 
	gnome_config_set_string("/gmail/UserInfo/AddressBook", gtk_entry_get_text(GTK_ENTRY(wid)));

	/* Database Tab */
	wid = glade_xml_get_widget(xml, "sqlserver"); 
	gnome_config_set_string("/gmail/MysqlServer/hostname", gtk_entry_get_text(GTK_ENTRY(wid)));
	wid = glade_xml_get_widget(xml, "sqluser"); 
	gnome_config_set_string("/gmail/MysqlServer/username", gtk_entry_get_text(GTK_ENTRY(wid)));
	wid = glade_xml_get_widget(xml, "sqlpass"); 
	gnome_config_private_set_string("/gmail/MysqlServer/password", gtk_entry_get_text(GTK_ENTRY(wid)));
	wid = glade_xml_get_widget(xml, "sqldb"); 
	gnome_config_set_string("/gmail/MysqlServer/dbname", gtk_entry_get_text(GTK_ENTRY(wid)));
	wid = glade_xml_get_widget(xml, "cache"); 
	if (GTK_TOGGLE_BUTTON (wid)->active) {
		gnome_config_set_string("/gmail/UserInfo/CacheVfolders", "TRUE");
	} else {
		gnome_config_set_string("/gmail/UserInfo/CacheVfolders", "FALSE");
		}

	/* Mail Setups Tab. */
	/* Clean out the current mailsetups. */
	iter = gnome_config_init_iterator("/gmail/mailsetups");
	while (TRUE) {
		gchar *name, *value, *cfg;
		iter = gnome_config_iterator_next(iter, &name, &value);
		if (iter == NULL) break;
		cfg = g_strdup_printf("/gmail/mailsetup_%s", name);	
		gnome_config_clean_section(cfg);	
		gnome_config_private_clean_section(cfg);	
		g_free(cfg);
		}
	gnome_config_clean_section("/gmail/mailsetups");	

	for (row = 0; row != GTK_CLIST (GmailApp->msetup_clist)->rows; row++) {
		gchar *name, *def, *cfg;
		MailSetup *msetup;

		/* Set the /gmail/mailsetups entry. */
		gtk_clist_get_text(GTK_CLIST(GmailApp->msetup_clist), row, 0, &def);
		gtk_clist_get_text(GTK_CLIST(GmailApp->msetup_clist), row, 1, &name);
		cfg = g_strdup_printf("/gmail/mailsetups/%s", name);
		gnome_config_set_string(cfg, def);
		g_free(cfg);

		/* Set the specific stuff. */
		msetup = gtk_clist_get_row_data(GTK_CLIST (GmailApp->msetup_clist), row);

		cfg = g_strdup_printf("/gmail/mailsetup_%s/username", name);	
		gnome_config_set_string(cfg, msetup->popuser); 
		g_free(cfg);
		cfg = g_strdup_printf("/gmail/mailsetup_%s/password", name);	
		gnome_config_private_set_string(cfg, msetup->poppass); 
		g_free(cfg);
		cfg = g_strdup_printf("/gmail/mailsetup_%s/servername", name);	
		gnome_config_set_string(cfg, msetup->popserver); 
		g_free(cfg);
		cfg = g_strdup_printf("/gmail/mailsetup_%s/smtp", name);	
		gnome_config_set_string(cfg, msetup->smtpserver); 
		g_free(cfg);
		cfg = g_strdup_printf("/gmail/mailsetup_%s/leavemailonserver", name);	
		gnome_config_set_bool(cfg, msetup->leave_on_server); 
		g_free(cfg);

		/* Should we free the row data here? what if the clist isn't destroyed? 
		free_msetup(msetup); */
		}

	/* Reset the option menu to the new settings. */
	set_mailmenu_options(GTK_OPTION_MENU(GmailApp->mailmenu));

	/* Appearance Tab  */
	wid = glade_xml_get_widget(xml, "fontpicker1"); 
	gnome_config_set_string("/gmail/fonts/mainfont", gnome_font_picker_get_font_name(GNOME_FONT_PICKER(wid)));
	wid = glade_xml_get_widget(xml, "fontpicker2"); 
	gnome_config_set_string("/gmail/fonts/boldfont", gnome_font_picker_get_font_name(GNOME_FONT_PICKER(wid)));

	/* Update the font styles over the current default. */
	update_font_styles(GmailApp->style);  

	/* FIXME: clear/delete/rebuild the clists to get new styles. */

	/* Apply two or three pane view. 
	 * Check if they are changing the view type. If so we have some extra
	 * work to do.
	 */
	wid = glade_xml_get_widget(xml, "twopaneview"); 
	if (GTK_TOGGLE_BUTTON (wid)->active)   {
		if (!TwoPaneView()) {
			/* Need to change from three pane to two pane view. */

			/* Just destroy the rightpane. */
			gtk_widget_destroy(GmailApp->rightpane);
			GmailApp->rightpane = NULL;			
			GmailApp->msgpane = NULL;			
			GmailApp->msglistpane = NULL;

			/* FIXME: for some reason we coredump without destroying righlist. */
			gtk_widget_destroy(GmailApp->rightlist);
			GmailApp->rightlist = NULL;

			}
		gnome_config_set_bool("/gmail/UserInfo/TwoPaneView", TRUE);
		} else {
		if (TwoPaneView()) {
			/* Need to change from two pane to three pane view. */
			DestroyRightPane();

			GmailApp->rightpane = gtk_vpaned_new();
			gtk_paned_set_handle_size (GTK_PANED(GmailApp->rightpane), 10);
  			gtk_paned_set_gutter_size (GTK_PANED(GmailApp->rightpane), 15);
			gtk_paned_set_position(GTK_PANED(GmailApp->rightpane), 150);

			/* What happens if you add2 over something else? */
			gtk_paned_add2 (GTK_PANED(GmailApp->hpaned), GmailApp->rightpane);

			/* Kill the rightlist so it gets recreated later. */
			gtk_widget_destroy(GmailApp->rightlist);
			GmailApp->rightlist = NULL;


			gtk_widget_show_all (GmailApp->hpaned);
			}
		gnome_config_set_bool("/gmail/UserInfo/TwoPaneView", FALSE);
		}

#if 0
	wid = glade_xml_get_widget(xml, "window_width"); 
	gnome_config_set_int("/gmail/Geometry/width", gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(wid)));
	wid = glade_xml_get_widget(xml, "window_height"); 
	gnome_config_set_int("/gmail/Geometry/height", gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(wid)));
	gtk_widget_set_usize(GTK_WIDGET(GmailApp->app), gnome_config_get_int_with_default("/gmail/Geometry/width=850", NULL), gnome_config_get_int_with_default("/gmail/Geometry/height=650", NULL));
#endif

	wid = glade_xml_get_widget(xml, "openfirst"); 
	gnome_config_set_bool("/gmail/UserInfo/display_first_message", GTK_TOGGLE_BUTTON (wid)->active);

	wid = glade_xml_get_widget(xml, "follow_sort"); 
	gnome_config_set_bool("/gmail/UserInfo/follow_sort", GTK_TOGGLE_BUTTON (wid)->active);

	wid = glade_xml_get_widget(xml, "wordwrap"); 
	gnome_config_set_bool("/gmail/UserInfo/wrap_text", GTK_TOGGLE_BUTTON (wid)->active);
	wid = glade_xml_get_widget(xml, "ww_len"); 
	gnome_config_set_int("/gmail/UserInfo/wrap_len", gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(wid)));

	wid = glade_xml_get_widget(xml, "fullfrom"); 
	gnome_config_set_bool("/gmail/UserInfo/fullfrom", GTK_TOGGLE_BUTTON (wid)->active);

	/* Save the data. */
	gnome_config_sync();

	/* Close the MySQL data connection, so that the connection can be
	 * reloaded if anything has changed.
	 */
	mysql_close(&GmailApp->mysql);

	/* FIXEM: Unref the xml data. 
	gtk_object_unref(GTK_OBJECT(xml)); */
}

/* Create or modify a new mail setup.
 *
 *
 */
void
mail_setup_modify(GtkWidget *widget, gchar *modify) 
{
	GladeXML *xml;
	GtkWidget *wid, *props;
	gchar *glade_src, *text;
	MailSetup *msetup;
	gboolean new = FALSE;
	GList	*sel = NULL;
	gint pos = -1;

	/* Get the position of current selected mail setup. */
	sel = GTK_CLIST(GmailApp->msetup_clist)->selection;
	if (sel != NULL) pos = (gint) sel->data;

	if (strcmp(modify, "TRUE") == 0) new = FALSE; else new = TRUE;

	/* Can't modify if nothing is selected. */
	if (!new && pos == -1) return;

	msetup = gtk_clist_get_row_data(GTK_CLIST (GmailApp->msetup_clist), pos);
	
	/* Load the glade file. */
	glade_src = g_strdup_printf("%s/preferences.glade", GMAIL_GLADEDIR);
	xml = glade_xml_new(glade_src, "mailproperties");
	g_free(glade_src);

	if (!xml) {
		g_warning("Unable to load xml interface. Are the gmail glade files installed onto your system?\n");
		exit(1);
		}

	glade_xml_signal_autoconnect(xml);

	props = glade_xml_get_widget(xml, "mailproperties");

	/* Attach which row of the mailsetup clist is assoicated with this dialog. */
	gtk_object_set_data(GTK_OBJECT(props), "row", GINT_TO_POINTER(pos));

	gtk_widget_show(props);

	/* Setup signals. */
	wid = glade_xml_get_widget(xml, "setupname"); 
	if (!new) {
		/* Get the setup name from the prefs clist. */
		gtk_clist_get_text(GTK_CLIST(GmailApp->msetup_clist), pos, 1, &text);
		gtk_entry_set_text (GTK_ENTRY (wid), text);
		}

	gtk_signal_connect(GTK_OBJECT(wid), "key-press-event",
                       GTK_SIGNAL_FUNC(property_box_changed),
                       GNOME_PROPERTY_BOX(props));  

	wid = glade_xml_get_widget(xml, "user"); 
	if (!new) {
		gtk_entry_set_text (GTK_ENTRY (wid), msetup->popuser);
		}
	gtk_signal_connect(GTK_OBJECT(wid), "key-press-event",
                       GTK_SIGNAL_FUNC(property_box_changed),
                       GNOME_PROPERTY_BOX(props));  

	wid = glade_xml_get_widget(xml, "pass"); 
	if (!new) {
		gtk_entry_set_text (GTK_ENTRY (wid), msetup->poppass);
		}
	gtk_signal_connect(GTK_OBJECT(wid), "key-press-event",
                       GTK_SIGNAL_FUNC(property_box_changed),
                       GNOME_PROPERTY_BOX(props));  

	wid = glade_xml_get_widget(xml, "pop3"); 
	if (!new) {
		gtk_entry_set_text (GTK_ENTRY (wid), msetup->popserver);
		}
	gtk_signal_connect(GTK_OBJECT(wid), "key-press-event",
                       GTK_SIGNAL_FUNC(property_box_changed),
                       GNOME_PROPERTY_BOX(props));  

	wid = glade_xml_get_widget(xml, "smtp"); 
	if (!new) {
		gtk_entry_set_text (GTK_ENTRY (wid), msetup->smtpserver);
		}
	gtk_signal_connect(GTK_OBJECT(wid), "key-press-event",
                       GTK_SIGNAL_FUNC(property_box_changed),
                       GNOME_PROPERTY_BOX(props));  

	wid = glade_xml_get_widget(xml, "leavemail"); 
	if (!new) {
		if (msetup->leave_on_server) 
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON (wid), TRUE); else
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON (wid), FALSE);
		}
	gtk_signal_connect(GTK_OBJECT(wid), "toggled",
                       GTK_SIGNAL_FUNC(toggle_property_box_changed),
                       GNOME_PROPERTY_BOX(props));  

	/* Catch the apply signal. */
	gtk_signal_connect(GTK_OBJECT(props), "apply",
                       GTK_SIGNAL_FUNC(apply_mail_setup), GINT_TO_POINTER(new));

	gtk_signal_connect(GTK_OBJECT(props), "help",
                       GTK_SIGNAL_FUNC(mail_setup_help_cb), NULL);

	gtk_signal_connect_object (GTK_OBJECT (props), "destroy", GTK_SIGNAL_FUNC (gtk_object_unref), GTK_OBJECT (xml));

}

void
mail_setup_make_copy(GtkWidget *widget, void *data) 
{
	gchar *name, *newname[2];
	gint row;
	MailSetup *msetup, *msetup2;
	GList	*sel = NULL;
	gint pos = -1;

	/* Get the position of current selected mail setup. */
	sel = GTK_CLIST(GmailApp->msetup_clist)->selection;
	if (sel != NULL) pos = (gint) sel->data;
	if (pos == -1) return;

	/* Copy the clist row */
	msetup = gtk_clist_get_row_data(GTK_CLIST (GmailApp->msetup_clist), pos);
	gtk_clist_get_text(GTK_CLIST(GmailApp->msetup_clist), pos, 1, &name);
	newname[0] = g_strdup("alt");
	newname[1] = g_strdup_printf("Copy of %s", name);
	row = gtk_clist_append(GTK_CLIST(GmailApp->msetup_clist), newname);
	g_free(newname[0]);
	g_free(newname[1]);

	/* Copy over the clist data to the new row. */
	msetup2 = g_malloc(sizeof(MailSetup));
	msetup2->popuser = g_strdup(msetup->popuser);
	msetup2->poppass = g_strdup(msetup->poppass);
	msetup2->popserver = g_strdup(msetup->popserver);
	msetup2->smtpserver = g_strdup(msetup->smtpserver);
	msetup2->leave_on_server = msetup->leave_on_server;
	gtk_clist_set_row_data(GTK_CLIST (GmailApp->msetup_clist), row, msetup2);

	gnome_property_box_changed(GNOME_PROPERTY_BOX(GmailApp->prefs)); 

}

void
mail_setup_set_default(GtkWidget *widget, void *data) 
{
	gint row;
	GList	*sel = NULL;
	gint pos = -1;

	/* Get the position of current selected mail setup. */
	sel = GTK_CLIST(GmailApp->msetup_clist)->selection;
	if (sel != NULL) pos = (gint) sel->data;
	if (pos == -1) return;

	/* Loop through the clist, marking everything except the selected 
	 * row as alt. Selected row is the default.
	 */
	for (row = 0; row != GTK_CLIST (GmailApp->msetup_clist)->rows; row++) {

		if (row == pos) 
			gtk_clist_set_text(GTK_CLIST (GmailApp->msetup_clist), row, 0, "default"); 
			else
			gtk_clist_set_text(GTK_CLIST (GmailApp->msetup_clist), row, 0, "alt"); 

		}

	gnome_property_box_changed(GNOME_PROPERTY_BOX(GmailApp->prefs)); 
}

void
mail_setup_delete(GtkWidget *widget, void *data) 
{
	MailSetup *msetup;
	GList	*sel = NULL;
	gint pos = -1;

	/* Get the position of current selected mail setup. */
	sel = GTK_CLIST(GmailApp->msetup_clist)->selection;
	if (sel != NULL) pos = (gint) sel->data;
	if (pos == -1) return;

	/* Check if this setup is the default. If so, set row 0 to be default. */

	/* Free the data attached to the row. */
	msetup = gtk_clist_get_row_data(GTK_CLIST (GmailApp->msetup_clist), pos);
	free_msetup(msetup);

	/* Remove the row. */
	gtk_clist_remove(GTK_CLIST(GmailApp->msetup_clist), pos);
	
	gnome_property_box_changed(GNOME_PROPERTY_BOX(GmailApp->prefs)); 
}

/* Put the mail setup settings into the prefs clist. 
 *
 */
static void
apply_mail_setup (GnomePropertyBox *property_box, gint page_num, gboolean new) 
{
	MailSetup *msetup;
	GladeXML *xml;
	gchar *name, *text[2], *def = NULL;	
	GtkWidget *wid;
	gint row, nrow;

	if (page_num != -1) return;  /* Only apply once. */

	row = (gint) gtk_object_get_data(GTK_OBJECT(property_box), "row");

	xml = glade_get_widget_tree(GTK_WIDGET(property_box));

	if (xml == NULL) g_warning("Failed to get xml data for mail setup apply.\n");

	/* Create a MailSetup structure. */
	msetup = g_malloc(sizeof(MailSetup));

	wid = glade_xml_get_widget(xml, "setupname"); 
	name = gtk_entry_get_text(GTK_ENTRY(wid));

	wid = glade_xml_get_widget(xml, "user"); 
	msetup->popuser = g_strdup(gtk_entry_get_text(GTK_ENTRY(wid))); 

	wid = glade_xml_get_widget(xml, "pass"); 
	msetup->poppass = g_strdup(gtk_entry_get_text(GTK_ENTRY(wid)));

	wid = glade_xml_get_widget(xml, "pop3"); 
	msetup->popserver = g_strdup(gtk_entry_get_text(GTK_ENTRY(wid)));

	wid = glade_xml_get_widget(xml, "smtp"); 
	msetup->smtpserver = g_strdup(gtk_entry_get_text(GTK_ENTRY(wid)));

	wid = glade_xml_get_widget(xml, "leavemail"); 
	if (GTK_TOGGLE_BUTTON (wid)->active) 
		msetup->leave_on_server = TRUE; else 
		msetup->leave_on_server = FALSE; 


	if (!new) {
		gtk_clist_get_text(GTK_CLIST(GmailApp->msetup_clist), row, 0, &def);

		/* Create a row in the clist for this setup. */
		text[0] = g_strdup(def);
		text[1] = g_strdup(name);

		/* It would probably be nicer to just update the row. */
		gtk_clist_remove(GTK_CLIST(GmailApp->msetup_clist), row);
		gtk_clist_insert (GTK_CLIST (GmailApp->msetup_clist), row, text); 
		g_free(text[0]); 
		g_free(text[1]);  

		/* Set the mailsetup structure in the clist row data. */
		gtk_clist_set_row_data(GTK_CLIST (GmailApp->msetup_clist), row, msetup); 
		} else {
		/* Create a row in the clist for this setup. */
		text[0] = g_strdup("alt");
		text[1] = g_strdup(name);
		nrow = gtk_clist_append (GTK_CLIST (GmailApp->msetup_clist), text); 
		g_free(text[0]); 
		g_free(text[1]);  

		/* Set the mailsetup structure in the clist row data. */
		gtk_clist_set_row_data(GTK_CLIST (GmailApp->msetup_clist), nrow, msetup); 
		}


	gnome_property_box_changed(GNOME_PROPERTY_BOX(GmailApp->prefs));  
}

/* Remember which row in the mail setup prefs is selected */
static void
msetup_clist_selected (GtkWidget *clist, GdkEventButton *event, gpointer data)
{
	gint row, column;

	gtk_clist_get_selection_info (GTK_CLIST (clist), event->x, event->y, &row, &column); /* Find our current row and column. */


	if (event != NULL && event->button == 1 && event->type == GDK_2BUTTON_PRESS) {
		gtk_clist_select_row(GTK_CLIST(clist), row, column);
		mail_setup_set_default(clist, data);
	} else if (event->button == 2) {
		gtk_clist_select_row(GTK_CLIST(clist), row, column);
		mail_setup_modify(GmailApp->msetup_clist, "TRUE");
		}
}

void free_msetup(MailSetup *msetup)
{
	if (msetup == NULL) return;

	if (msetup->popuser != NULL) g_free(msetup->popuser);
	if (msetup->poppass != NULL) g_free(msetup->poppass);
	if (msetup->popserver != NULL) g_free(msetup->popserver);
	if (msetup->smtpserver != NULL) g_free(msetup->smtpserver);

	g_free(msetup);

}

/* Return a mailsetup struct with the default mail-setup.
 * 
 * If we can't find one we return NULL. This isn't very nice.
 */
MailSetup *load_default_msetup() 
{
	MailSetup *msetup;
	gchar *name = NULL, *nm;
	void *iter;
	gboolean found = FALSE;

	/* Find the default msetup by looping through the values. */
	iter = gnome_config_init_iterator("/gmail/mailsetups");
	while (TRUE) {
		gchar *value;

		iter = gnome_config_iterator_next(iter, &nm, &value);
		if (iter == NULL) break;
	
		if (strcmp(value, "default") == 0) {
			found = TRUE;
			break;
			}
	}

	if (found == FALSE) 
		return(NULL);
	

	name = g_strdup(nm);

	msetup = load_msetup(name);

	g_free(name);
	return(msetup);
}

/* Load a mailsetup into the structure, from the given mailsetup name. */
MailSetup *load_msetup(gchar *name)
{
	MailSetup *msetup;
	gchar *config;

	/* Creat the msetup structure and put in values. */
	msetup = g_malloc (sizeof(MailSetup)); 

	msetup->name = g_strdup(name);
	config = g_strdup_printf("/gmail/mailsetup_%s/username", name);	
	msetup->popuser = gnome_config_get_string(config); 
	g_free(config);
	config = g_strdup_printf("/gmail/mailsetup_%s/password", name);	
	msetup->poppass = gnome_config_private_get_string(config); 
	g_free(config);
	config = g_strdup_printf("/gmail/mailsetup_%s/servername", name);	
	msetup->popserver = gnome_config_get_string(config); 
	g_free(config);
	config = g_strdup_printf("/gmail/mailsetup_%s/smtp", name);	
	msetup->smtpserver = gnome_config_get_string(config); 
	g_free(config);
	config = g_strdup_printf("/gmail/mailsetup_%s/leavemailonserver", name);	
	msetup->leave_on_server = gnome_config_get_bool(config); 

	return(msetup);
}

static void
mail_setup_help_cb (GnomePropertyBox *property_box, gint page_num)
{
	static GnomeHelpMenuEntry help_ref = {PACKAGE, "gmail/prefs-2.html" };

	gnome_help_display(0, &help_ref);
}

/* Call when the tree_move signal is emitted on the vfolder ctree. */
static void vfolder_tree_move (GtkCTree *ctree, GList *node, GList *new_parent, GList *new_sibling, gpointer user_data) {

	/* Write the entire ctree to the config file */
	save_all_vfolder_data(ctree); 
}

/* 
 * Handle drag and drop of an email into the query.
 *
 * Basically we will that if the query isn't empty, then the current
 * query is bracketed, or will simply work with another AND statement added.
 * So stuff with OR in them should be bracketed.
 *
 * We add: AND LCASE(headers) LIKE "%dnd_email%"
 * And then we put brackets around the whole thing for safety.
 *
 */
void  
query_drag_data_received   (GtkWidget          *widget,
			    GdkDragContext     *context,
			    gint                x,
			    gint                y,
			    GtkSelectionData   *data,
			    guint               info,
			    guint               time)
{
	gchar *query, *email;
	GtkWidget *vfolder_prop;
	GladeXML *xml;

	g_return_if_fail(GTK_IS_TEXT(widget));

	/* Strip the email address from the full name. 
	 * FIXME: Use gmime to parse this data.
	 */
	email = strip_email(data->data);

	/* Append the new email address to the end of the query. */
	/* query = g_strdup_printf(" || LCASE(headers) LIKE \"%%%s%%\"", email); */
	query = g_strdup_printf("%s", email);
	gtk_text_insert (GTK_TEXT (widget), NULL, NULL, NULL, query, -1);

	/* Insert the open bracket at the start. 
	gtk_text_set_point(GTK_TEXT(widget), 0);
	gtk_text_insert (GTK_TEXT (widget), NULL, NULL, NULL, "(", 1); */

	/* Set the point back to the end, in case they add more. */
	gtk_text_set_point(GTK_TEXT(widget), gtk_text_get_length(GTK_TEXT(widget)));

	g_free(query);
	g_free(email);

	/* Set the apply button on the vfolder property box. */
	xml = glade_get_widget_tree(GTK_WIDGET(widget));
	vfolder_prop = glade_xml_get_widget(xml, "vfolder_properties");
	gnome_property_box_changed(GNOME_PROPERTY_BOX(vfolder_prop)); 
	gtk_object_unref(GTK_OBJECT(xml));
}

/* Clear the option menu in the bottom right hand corner and reload
 * with the available mail preferences.
 */
void 
set_mailmenu_options(GtkOptionMenu *mailmenu)
{
	void *iter = NULL;
	GtkWidget *menu_item, *menu;
	gint n = 0, def = 0;
	gboolean default_found = FALSE;

	g_return_if_fail(mailmenu != NULL);

	gtk_option_menu_remove_menu(mailmenu);

	/* Loop through the available mail-prefs. */
	iter = gnome_config_init_iterator("/gmail/mailsetups");

	menu = gtk_menu_new();

	while (TRUE) {
		gchar *name = NULL, *value = NULL;
		gchar *name_data;

		iter = gnome_config_iterator_next(iter, &name, &value);
		if (iter == NULL) break;

		menu_item = gtk_menu_item_new_with_label(name);

		name_data = g_strdup(name);

		gtk_signal_connect_after(GTK_OBJECT (menu_item), "activate", (GtkSignalFunc) change_default_msetup, name_data);

		gtk_menu_append (GTK_MENU (menu), menu_item);

		/* If this is the default then set it active. */
		if (strcmp(value, "default") == 0)  {
			def = n;
			default_found = TRUE;
			}
		gtk_widget_show(menu_item);
		n++;
		}

	/* FIXME: If default is not found, then fix gnome_config and set it to be the last item we had. */

	gtk_option_menu_set_menu (GTK_OPTION_MENU (mailmenu), menu);

	/* Make the default the one shown. */
	gtk_option_menu_set_history (GTK_OPTION_MENU (mailmenu), def);
	
	
}

/* Called when the user activates the mail-setup option menu.
 * We change the default mail setup to be the one they selected.
 * 
 */
void change_default_msetup(GtkWidget *widget, gchar *name)
{
	gchar *cfg;
	MailSetup *msetup;

	/* Find the old default and clear it. */
	msetup = load_default_msetup();
	cfg = g_strdup_printf("/gmail/mailsetups/%s", msetup->name);
	gnome_config_set_string(cfg, "alt");
	g_free(cfg);
	free_msetup(msetup);

	/* Set the new name to be the default. */
	cfg = g_strdup_printf("/gmail/mailsetups/%s", name);
	gnome_config_set_string(cfg, "default");
	g_free(cfg);

	gnome_config_sync();

	/* g_print("Default is now %s\n", name); */
}
