/*
 * Gmail. A Gnome email client.
 * Copyright (C) 1999 Wayne Schuller
 *
 * message-display.c - contains functions for displaying vfolders + messages.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include "main.h"

/* Shared functions. */
void delete_msg (GtkWidget *widget, gpointer data);
void next_prev_cb (GtkWidget *widget, gboolean next);
void next_prev_unread_cb (GtkWidget *widget, gboolean next);
void vfolder_selected (GtkCTree *ctree, GdkEventButton *event, gpointer data);
void button_press (GtkWidget *widget, GdkEventButton *event, gpointer data);
void address_drag_data_get (GtkWidget *widget, GdkDragContext *context, GtkSelectionData *selection_data, guint info, guint32 time);
void clist_drag_data_get (GtkWidget *widget, GdkDragContext *context, GtkSelectionData *selection_data, guint info, guint32 time);
void vfolder_display_rightpane_by_mailbox(Mailbox *mailbox);
void search_cb (GtkWidget *widget, void *data);
void search_apply (GtkWidget *widget, void *data);
void search_ok (GtkWidget *widget, void *data);
void search_close (GtkWidget *widget, void *data);

/* Local static functions. */
static void display_message(Message *msg, gboolean forced);
static GtkWidget * create_msg_pane(void);
static void columnclicked_cb (GtkCList * clist, gint column);
static void mark_msg_read(GtkWidget *widget, gboolean data);
static void mark_msg_unread(GtkWidget *widget, gboolean data);
static void make_vfolder_copy (GtkWidget *widget, void *data);
static void make_vfolder_new (GtkWidget *widget, void *data);
static void delete_vfolder (GtkWidget *widget, void *data);
static Mailbox * copy_mailbox(Mailbox *mailbox);
static void rightlist_clist_create(void);
static void columnresize_cb(GtkCList *clist, gint column, gint width, gpointer user_data);
static void msg_clicked_cb (GtkWidget *clist, gint row, gint column, GdkEventButton *event, gpointer data);
static gint attachment_button_press_event (GtkWidget *widget, GdkEventButton *event, gpointer data);
static void save_attach (GtkWidget *widget, gint row);
static gint exec_attach (GtkWidget *clist, gint row, gint column, GdkEventButton *event, gpointer data);
static void rightlist_popup (GtkWidget *widget, GdkEventButton *event, gpointer data);
static void ctx_open_msg (GtkWidget *widget, gpointer data);

extern _GmailApp_ *GmailApp;
extern  gboolean vfolder_msg_unread(gint id);
extern Message * load_vfolder_msg(gint id);
extern gboolean check_db(MYSQL *mysql);
extern gboolean DestroyRightPane(void);
extern void broken_cb (GtkWidget *widget, void *data);
extern void set_ok (GtkWidget *widget, gboolean *dialog_result);
extern void free_message(Message *message);
extern void mark_readstatus(gint msg_id, gchar *val);
extern gboolean vfolder_contains_unread(Mailbox *mailbox);
extern void rebuild_vfolder_mi_values (GtkWidget *widget, void *data);
extern void vfolder_properties (GtkWidget *widget, void *data);
extern void save_all_vfolder_data(GtkCTree *ctree);
extern void mi_warning(GtkWidget *widget, void *data);
extern void recreate_matched_index (GtkWidget *widget, void *data);
extern void mark_as_matched(gint id, GString *vfolders, gboolean overwrite); 
extern void mark_vfolder_unread_status_by_name(gchar *vfolder, gboolean unread_status);

/* main.c */
extern void save_msg (GtkWidget *widget, void *data);

static GnomeUIInfo vfolder_popup_menu [] = {
  { GNOME_APP_UI_ITEM, N_("Vfolder options:"), N_("Vfolder options"), NULL, NULL, NULL, GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_MAIL, 0, 0, NULL },
  GNOMEUIINFO_SEPARATOR,
  { GNOME_APP_UI_ITEM, N_("_Properties"), N_("_Vfolder properties."), vfolder_properties, NULL, NULL, GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_PROP, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("_Add New Vfolder"), N_("_Add New Vfolder"), make_vfolder_new, NULL, NULL, GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_NEW, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("_Make Copy"), N_("_Make Copy"), make_vfolder_copy, NULL, NULL, GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_COPY, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("_Delete"), N_("_Delete vfolder."), delete_vfolder, NULL, NULL, GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_TRASH, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("_Rebuild MI Values"), N_("_Rebuild Matched Index Values"), rebuild_vfolder_mi_values, NULL, NULL, GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_REFRESH, 0, 0, NULL },
  GNOMEUIINFO_END
};

static GnomeUIInfo rightlist_popup_menu [] = {
  { GNOME_APP_UI_ITEM, N_("Message options:"), N_("Message options"), NULL, NULL, NULL, GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_MAIL, 0, 0, NULL },
  GNOMEUIINFO_SEPARATOR,
  { GNOME_APP_UI_ITEM, N_("_Open"), N_("_Open message."), ctx_open_msg, NULL, NULL, GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_OPEN, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("_Save"), N_("_Save message."), save_msg, NULL, NULL, GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_SAVE, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("_Delete"), N_("_Delete message."), delete_msg, NULL, NULL, GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_TRASH, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Mark _Unread"), N_("Mark messages _Unread"), mark_msg_unread, NULL, NULL, GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_FORWARD, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Mark _Read"), N_("Mark messages _Read"), mark_msg_read, NULL, NULL, GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_BACK, 0, 0, NULL },
  GNOMEUIINFO_END
};

static GnomeUIInfo mime_popup_menu [] = {
  { GNOME_APP_UI_ITEM, N_("Save Attachment"), N_("_Save attachment"), save_attach, NULL, NULL, GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_SAVE, 0, 0, NULL },
  GNOMEUIINFO_END
};

/* Standard DnD types */
extern GtkTargetEntry email_field_drop_types[];

/* vfolder_selected
 * This is a callback for mouse clicks on the vfolder ctree.
 * It handles both left and right clicks.
 *
 */
void vfolder_selected (GtkCTree *ctree, GdkEventButton *event, gpointer data)
{
	Mailbox *mailbox;	
	gint row, column;
	GtkCTreeNode *node;

	if (event == NULL) return;

	gtk_clist_get_selection_info (GTK_CLIST (ctree), event->x, event->y, &row, &column); /* Find our current row and column. */
	node = gtk_ctree_node_nth (ctree, row); /* Find the node we are at. */

	if (node == NULL) return; /* No node no code. :) */

	GmailApp->node = node;

	/* Check for double click. */	
	if (event->button == 1 && event->type == GDK_2BUTTON_PRESS) {
		static const char *soundtrigger[] = {"gtk-events", "activate", NULL};
		gnome_triggers_vdo("", NULL, soundtrigger); /* Play sound */
		
		/* Get the vfolder data. */
		mailbox = gtk_ctree_node_get_row_data(GTK_CTREE (ctree), GTK_CTREE_NODE (node));

		vfolder_display_rightpane_by_mailbox(mailbox);

	} else if (event->button == 3) {
		gint result;
		GtkWidget *menu;

		/* Select the node we are at. */
		gtk_ctree_select(ctree, node);

		/* Open the popup menu. */
		menu = gnome_popup_menu_new (vfolder_popup_menu);
		result = gnome_popup_menu_do_popup_modal (menu, NULL, NULL, event, ctree);
		gtk_widget_destroy (menu);
		/* g_print("User chose %i\n", result);  */

	} else if (event->button == 2) {

		/* Middle mouse brings up properties. */	
		vfolder_properties(NULL, NULL);
	}
}

/* Handle rightclick popup menus (for vfolder clist only). */
void
button_press (GtkWidget *widget, GdkEventButton *event, gpointer data)
{
	GtkWidget *menu;
	gint result;
	gint row, col;

	if (event->button != 3) return;

	gtk_clist_get_selection_info(GTK_CLIST(widget), event->x, event->y, &row, &col);
	gtk_clist_select_row(GTK_CLIST(widget), row, col);

	menu = gnome_popup_menu_new (vfolder_popup_menu);
	result = gnome_popup_menu_do_popup_modal (menu, NULL, NULL, event, widget);
	gtk_widget_destroy (menu);
	/* g_print("User chose %i\n", result); */
}

/* This code actually displays a mail message in the relevent pane.
 * It display the msg that is passed to it.
 * It also processes MIME attachments.
 * 
 * It is up to the calling function to free the memory used by the msg.
 *
 * At the moment we have a global variable called GmailApp->msgpane.
 * We assume the message always goes in here. If it doesn't exist we 
 * call create_msg_pane.
 * We rely on libglade to give us the widgets we need, so this code
 * will generalise easy to have multiple message windows.
 *
 * If forced is true it means the message is being displayed without the user
 * manually doing it. If so we don't mark the message as read or unbold it.
 * 
 * FIXME: Are there mem leaks in here?
 */
static void 
display_message(Message *msg, gboolean forced) 
{
	gchar *title = NULL, *status = NULL;
	GladeXML *xml; 
	GtkWidget *wid;
	GMimeMessage *message = NULL;
	gboolean is_html;
	gchar *body;
	GString *fulltext;
	GList	*sel = NULL;
	gint pos = -1;
	gboolean mime = FALSE;
	Mailbox *mailbox = NULL;

	/* Get the position of the selected message. */
	sel = GTK_CLIST(GmailApp->rightlist)->selection;
	if (sel != NULL) pos = (gint) sel->data;
	if (pos < 0) return;

	g_return_if_fail(msg != NULL);
	g_return_if_fail(msg->headers != NULL);
	g_return_if_fail(msg->message != NULL);

	if (TwoPaneView()) {

		/* FIXME: Reuse the msg pane in this mode if we have it... */
		if (!GmailApp->msgpane) {
			DestroyRightPane();
			GmailApp->msgpane = create_msg_pane();
			gtk_paned_add2 (GTK_PANED(GmailApp->hpaned), GmailApp->msgpane); 
			}

	} else {
		/* We are in the infamous THREE PANE VIEW */
		/* Create the msg pane if it doesn't exist. */
		if (!GTK_IS_WIDGET(GmailApp->msgpane)) {
			GmailApp->msgpane = create_msg_pane();
			gtk_paned_add2 (GTK_PANED(GmailApp->rightpane), GmailApp->msgpane); 
			} 
	}
	if (GmailApp->msgpane == NULL) g_print("NULL msg pane! ...\n");

	/* Attach the msg id as data to the message pane. */
	gtk_object_set_data(GTK_OBJECT(GmailApp->msgpane), "id", GINT_TO_POINTER(msg->id));
	
	/* Put the headers and the body to make the full text message. */
	fulltext = g_string_new(msg->headers->str);
	g_string_append(fulltext, msg->message->str);

	/* If g_mime can parse it ok we will consider it a mime message.
	 * Otherwise we assume plain text stuff.
	 */
	message = g_mime_parser_construct_message(fulltext->str, fulltext->len, TRUE);

	if (message != NULL) mime = TRUE;

	if (mime) 
		body = g_mime_message_get_body(message, TRUE, &is_html);  
	else
		body = g_strdup(msg->message->str);

	/* g_print("Parts: %i\n", g_list_length(message->mime_part->children));  */

	xml = glade_get_widget_tree(GTK_WIDGET(GmailApp->msgpane));

	if (xml == NULL) g_print("xml is NULL! ...\n");

	wid = glade_xml_get_widget(xml, "from"); 
	if (wid != NULL) {
	gtk_label_set_text(GTK_LABEL(wid), msg->from);
	gtk_widget_set_style (wid, GmailApp->boldstyle); }

	/* Set up dnd source for email drags.  */
	wid = glade_xml_get_widget(xml, "from_eventbox"); 
	gtk_drag_source_set (wid, GDK_BUTTON1_MASK, email_field_drop_types, 1, GDK_ACTION_COPY | GDK_ACTION_DEFAULT); 
	gtk_signal_connect(GTK_OBJECT(wid), "drag_data_get", GTK_SIGNAL_FUNC(address_drag_data_get), NULL);  
	wid = glade_xml_get_widget(xml, "to_eventbox"); 
	gtk_drag_source_set (wid, GDK_BUTTON1_MASK, email_field_drop_types, 1, GDK_ACTION_COPY | GDK_ACTION_DEFAULT); 
	gtk_signal_connect(GTK_OBJECT(wid), "drag_data_get", GTK_SIGNAL_FUNC(address_drag_data_get), NULL);  
	wid = glade_xml_get_widget(xml, "cc_eventbox"); 
	gtk_drag_source_set (wid, GDK_BUTTON1_MASK, email_field_drop_types, 1, GDK_ACTION_COPY | GDK_ACTION_DEFAULT); 
	gtk_signal_connect(GTK_OBJECT(wid), "drag_data_get", GTK_SIGNAL_FUNC(address_drag_data_get), NULL);  

	wid = glade_xml_get_widget(xml, "to"); 
	if (wid != NULL) {
	gtk_label_set_text(GTK_LABEL(wid), msg->to);
	gtk_widget_set_style (wid, GmailApp->style);  }

	wid = glade_xml_get_widget(xml, "subject"); 
	if (wid != NULL) {
	gtk_label_set_text(GTK_LABEL(wid), msg->subject);
	gtk_widget_set_style (wid, GmailApp->boldstyle);  }

	wid = glade_xml_get_widget(xml, "date"); 
	if (wid != NULL) {
	gtk_label_set_text(GTK_LABEL(wid), msg->date);
	gtk_widget_set_style (wid, GmailApp->style); }

	/* FIXME: only make this appear when there is a cc field. */
	wid = glade_xml_get_widget(xml, "cc"); 
	if (wid != NULL) {
	gtk_label_set_text(GTK_LABEL(wid), msg->cc);
	gtk_widget_set_style (wid, GmailApp->style);  }

	/* Freeze, clear and insert the text message. */

	wid = glade_xml_get_widget(xml, "textmsg"); 
	gtk_text_freeze(GTK_TEXT (wid));
	gtk_widget_set_style (wid, GmailApp->style); 
	gtk_text_backward_delete(GTK_TEXT (wid), gtk_text_get_length(GTK_TEXT (wid)));
	/* Insert message into the textmsg box. */
	gtk_text_insert (GTK_TEXT (wid), NULL, NULL, NULL, body, -1);
	gtk_text_thaw(GTK_TEXT (wid));

	/* Freeze, clear and insert the text headers. */
	wid = glade_xml_get_widget(xml, "textheaders"); 
	gtk_text_freeze(GTK_TEXT (wid));
	gtk_widget_set_style (wid, GmailApp->style); 
	gtk_text_backward_delete(GTK_TEXT (wid), gtk_text_get_length(GTK_TEXT (wid)));
	/* Insert message into the textmsg box. */
	gtk_text_insert (GTK_TEXT (wid), NULL, NULL, NULL, fulltext->str, -1);
	gtk_text_thaw(GTK_TEXT (wid));

	gtk_widget_show_all (GmailApp->msgpane);

	wid = glade_xml_get_widget(xml, "mimeclist"); 

	/* Signal for the left double click.  */
	gtk_signal_connect(GTK_OBJECT(wid), "select_row",
                       GTK_SIGNAL_FUNC(exec_attach), NULL); 

	/* Signal for the right click.  */
	gtk_signal_connect (GTK_OBJECT (wid), "button_press_event",
          GTK_SIGNAL_FUNC (attachment_button_press_event), NULL); 

	/* Even if there is no MIME in this message, we clear the MIME
	 * clist otherwise the user will see old values.
	 */
	wid = glade_xml_get_widget(xml, "mimelabel"); 
	gtk_label_set_text(GTK_LABEL(wid), "No MIME");
	wid = glade_xml_get_widget(xml, "mimeclist"); 
	gtk_clist_clear(GTK_CLIST(wid));

	if (mime) {
		gint i = 0, len = 0;
		GList *pos;
		GMimePart *part;
		GtkWidget *label = NULL;

		if (message->mime_part != NULL && message->mime_part->children != NULL)
		for (pos = message->mime_part->children; pos != NULL; pos = pos->next, i++) {
			gchar *datum[5];		/* temporary storage area for clist */

			part = pos->data; 
			if (part == NULL) break;
				
			/* g_print("part %i\n", i); */

			datum[0] = g_strdup_printf("%i", i);
			datum[1] = g_strdup_printf("%i", len);
			datum[2] = g_strdup_printf("%s", g_mime_part_encoding_to_string(g_mime_part_get_encoding(part))); 
			datum[3] = g_strdup_printf("%s", g_mime_content_type_to_string(g_mime_part_get_content_type(part)));

			/* If disposition exists, look for filename. */
			if (part->disposition)
			datum[4] = g_strdup(g_mime_part_get_filename(part));
			else
			datum[4] = g_strdup("");
	
			gtk_clist_append (GTK_CLIST (wid), datum);

			if (datum[0] != NULL) g_free(datum[0]);
			if (datum[1] != NULL) g_free(datum[1]);
			if (datum[2] != NULL) g_free(datum[2]);
			if (datum[3] != NULL) g_free(datum[3]);
			if (datum[4] != NULL) g_free(datum[4]);
			}	

	/* Set the notebook tab to show the number of MIME parts. */
	label = glade_xml_get_widget(xml, "mimelabel"); 

	switch (i) {
		case 0: title = g_strdup_printf("No MIME");
						break;
		case 1: title = g_strdup_printf("MIME (1 part)");
						break;
		default: title = g_strdup_printf("MIME (%i parts)", i);
						break;
		}
	gtk_label_set_text(GTK_LABEL(label), title);
	g_free(title);
	}


	/*
	 * Set the status bar message with the msg database id.
	 */
	title = g_strdup_printf("Msg id %i", msg->id);
	gnome_appbar_set_status (GNOME_APPBAR (GmailApp->appbar), title);
	g_free(title);

	/* Mark the message as read, now that we have displayed it. */
	gtk_clist_get_text(GTK_CLIST(GmailApp->rightlist), pos, 0, &status);
	if (!forced && (strcmp(status, "Unread") == 0)) {

		/* If the message is incoming, mark it as read. */
		/* We don't mark outgoing messages as read.. they are 'sent'. */
 		if (strcmp(msg->direction, "Incoming") == 0) { mark_readstatus(msg->id, "Read"); }

		/* If we are viewing an unread message, and it is unforced
		 * then we have to unbold the message in the clist. 
	 	 */
		gtk_clist_set_cell_style(GTK_CLIST (GmailApp->rightlist), pos, 0, GmailApp->style);
		gtk_clist_set_cell_style(GTK_CLIST (GmailApp->rightlist), pos, 1, GmailApp->style);
		gtk_clist_set_cell_style(GTK_CLIST (GmailApp->rightlist), pos, 2, GmailApp->style);
		gtk_clist_set_cell_style(GTK_CLIST (GmailApp->rightlist), pos, 3, GmailApp->style);

		/* For incoming msg's, set the clist read status to 'read'. */
		if (strcmp(msg->direction, "Incoming") == 0)  gtk_clist_set_text(GTK_CLIST (GmailApp->rightlist), pos, 0, "Read");

		while (g_main_iteration(FALSE));  /* Screen update. */

		/* Now that we have unbolded a message, we check if that
		 * vfolder has any unread messages. This chews some CPU
		 * but allows us to unbold the vfolder as soon as the as the
		 * unread message is read.
		 * (we also bold the vfolder if this is not the case, though
		 * theoretically it will already be bold, but we set it
		 * just to be sure.)
		 * FIXME: if caching us the Matched-Index to check the other
		 * vfolders this message belongs to as well.
		 */
		mailbox = gtk_ctree_node_get_row_data(GTK_CTREE (GmailApp->mblist), GTK_CTREE_NODE (GmailApp->node));
		if (vfolder_contains_unread(mailbox))
			gtk_ctree_node_set_cell_style(GTK_CTREE (GmailApp->mblist), GTK_CTREE_NODE(GmailApp->node), 0, GmailApp->boldstyle);
		else
			gtk_ctree_node_set_cell_style(GTK_CTREE (GmailApp->mblist), GTK_CTREE_NODE(GmailApp->node), 0, GmailApp->style);
	
	}

	g_string_free(fulltext, TRUE);
	g_free(body);

	/* Free the message data. */
	if (message) g_mime_message_destroy(message);

	return;
}

/* Create the widgets for the msg_pane. 
 * This function is called by display_message.
 * We now don't rely on any global variables, but purely on libglade.
 * This is coded to survive even if some widgets are missing, just
 * to allow users to easily alter the glade file without needing
 * to hack the code.
 */
static GtkWidget * create_msg_pane(void)
{
	GtkWidget *label;
	GladeXML *xml;
	GtkWidget *glademsg;
	gchar *glade_src;

	/* Load the glade file. */
	glade_src = g_strdup_printf("%s/message.glade", GMAIL_GLADEDIR);
	xml = glade_xml_new(glade_src, "frame1");
	g_free(glade_src);

	if (!xml) {
		g_warning(_("Unable to load xml interface. Are the gmail glade files installed onto your system?\n"));
		exit(1);
		}

	glade_xml_signal_autoconnect(xml);

	/* Grab our glade widgets. */
	glademsg = glade_xml_get_widget(xml, "frame1");

	/* FIXME: life would be easier if I knew how to set styles globally. */
	label = glade_xml_get_widget(xml, "fromlabel");
	if (label) gtk_widget_set_style (label, GmailApp->style); 

	label = glade_xml_get_widget(xml, "tolabel");
	if (label) gtk_widget_set_style (label, GmailApp->style); 

	label = glade_xml_get_widget(xml, "subjectlabel");
	if (label) gtk_widget_set_style (label, GmailApp->style); 

	label = glade_xml_get_widget(xml, "datelabel");
	if (label) gtk_widget_set_style (label, GmailApp->style); 

	label = glade_xml_get_widget(xml, "cclabel");
	if (label) gtk_widget_set_style (label, GmailApp->style); 

	return(glademsg); /* Return the msg frame. */
}


/* This function called when a user clicks on the 'Delete' toolbar button. 
 * Currently msg deletion is only supported for the mysql messages.
 * FIXME: Should we give them a warning before deleting a message?
 * The warning could contain some headers of the email, just to be sure.
 * FIXME: Go through every folder in the matched index and check
 * whether it contains unread or not. If not we can unbold it.
 */
void			
delete_msg (GtkWidget *widget, gpointer data)
{
	gchar *query;
	gint id;
	static const char *soundtrigger[] = {"gmail", "delete", NULL};
	Message *mesg = NULL;
	GList	*sel = NULL,*tempsel=NULL;
	gint pos = -1,finished=0;

	if (GTK_CLIST(GmailApp->rightlist)->rows == 0) return;
	/* Get the position of the selected message. */
	sel = GTK_CLIST(GmailApp->rightlist)->selection;
	/* Loop through all selected messages */
	while(!finished){
		if (sel != NULL) pos = (gint) sel->data;
		if (sel->next==NULL)
			finished=1;
		else 
			tempsel=sel->next;
		if (pos < 0) return;	/* Nothing to delete. */

		/* Get the id of the msg from the row data. */
		id = GPOINTER_TO_INT(gtk_clist_get_row_data(GTK_CLIST (GmailApp->rightlist), pos));

		/* Open Database. */
		if (!check_db(&GmailApp->mysql)) { return; };

		query = g_strdup_printf("DELETE FROM display WHERE id = %i", id);

		if (mysql_query(&GmailApp->mysql, query) != 0) {
			/* FIXME: send a dialog if the delete failed. */
			g_print("Query failed: %s\n", mysql_error(&GmailApp->mysql));
			return;
		} 

		g_free(query);

		query = g_strdup_printf("DELETE FROM details WHERE id = %i", id);
		if (mysql_query(&GmailApp->mysql, query) != 0) {
			/* FIXME: send a dialog if the delete failed. */
			g_print("Query failed: %s\n", mysql_error(&GmailApp->mysql));
			return;
		} 

		g_free(query);

		/* delete the row from the clist. */
		if (GTK_IS_CLIST(GmailApp->rightlist))
			gtk_clist_remove(GTK_CLIST(GmailApp->rightlist), pos);	

		gnome_triggers_vdo("", NULL, soundtrigger); /* Play sound */

		/* If there are no more messages, return. */
		if (GTK_CLIST(GmailApp->rightlist)->rows == 0) return;
		if(!finished)
			sel=tempsel;
	}
	/* If we just deleted the last msg, the next msg is the one before it.*/
	if (pos >= GTK_CLIST(GmailApp->rightlist)->rows)
		pos = GTK_CLIST(GmailApp->rightlist)->rows - 1;

	/* select the next msg. then we can delete stacks in a row. */
	gtk_clist_select_row(GTK_CLIST (GmailApp->rightlist), pos, 0);

	/* Get the id of the next msg. */
	id = GPOINTER_TO_INT(gtk_clist_get_row_data(GTK_CLIST (GmailApp->rightlist), pos));

	/* If we are in three pane mode display the email. */
	if (!TwoPaneView()) {
		mesg = load_vfolder_msg(id);
		if (mesg != NULL) { display_message(mesg, TRUE); }
		free_message(mesg); /* Free the memory */
	}

}

/* 
 * Scroll to the next or previous message depending on which button is pressed.
 */
void			
next_prev_cb (GtkWidget *widget, gboolean next) 
{
	gint id, pos = -1;
	Message *msg = NULL;
	GtkWidget *clist = GmailApp->rightlist;
	GList	*sel = NULL;
	/* GdkEventButton *event; unused. we want to check for shift to be able to implement a special 'search for same author' feature. */

	if (!GTK_IS_CLIST(clist)) return;
	if (GTK_CLIST(GmailApp->rightlist)->rows == 0) return;

	/* Get the position of current selected message. */
	sel = GTK_CLIST(GmailApp->rightlist)->selection;
	if (sel != NULL) pos = (gint) sel->data;

	/* Check boundaries. */
	if (!next) if (pos == 0) return;
	if (next) if (pos >= GTK_CLIST(clist)->rows - 1) return;

	/* Deselect the current row so we don't end up with both this
	 * and the next/previous one selected */
	gtk_clist_unselect_row(GTK_CLIST (clist), pos, 0);
	/* Skip forward or back. */
	if (next) pos++; else pos--;

	/* change the selection as well. */
	gtk_clist_select_row(GTK_CLIST (clist), pos, 0);
	gtk_clist_moveto(GTK_CLIST (clist), pos , 0, 0.5, 0);

	/* Load and display the new message. */
	id = GPOINTER_TO_INT(gtk_clist_get_row_data(GTK_CLIST (clist), pos));
	msg = load_vfolder_msg(id);
	if (msg == NULL) return;
	display_message(msg, FALSE); 
	free_message(msg); /* Free the memory */
}

/* 
 * Scroll to the next or previous unread message depending on which button is pressed.
 */
void			
next_prev_unread_cb (GtkWidget *widget, gboolean next) 
{
	gint id, pos = -1;
	Message *msg = NULL;
	GtkWidget *clist = GmailApp->rightlist;
	GList	*sel = NULL;
	int rows;
	int found=0,started;
	
	if (!GTK_IS_CLIST(clist)) return;
	if (GTK_CLIST(GmailApp->rightlist)->rows == 0) return;

	/* Get the position of current selected message. */
	sel = GTK_CLIST(GmailApp->rightlist)->selection;
	if (sel != NULL) pos = (gint) sel->data;
	rows=GTK_CLIST(clist)->rows;
	started=pos;
	/* Loop until found an unread message or end of folder hit */
	while(!found){		
		
		/* Check boundaries, and loop if necassary */
		if (!next){
			if (pos == 0)
				pos=rows;
			if (pos == started+1)
				return;
		}
		if (next){
			if (pos >= rows-1)
				pos=-1;
			if (pos == started-1)
				return;
		}
		/* Skip forward or back. */
		if (next) pos++; else pos--;
		
		/* Load and display the new message. */
		id = GPOINTER_TO_INT(gtk_clist_get_row_data(GTK_CLIST (clist), pos));
		if (vfolder_msg_unread(id)){
			/* change the selection as well. */
			msg = load_vfolder_msg(id);
			if (msg == NULL) return;
			gtk_clist_unselect_row(GTK_CLIST (clist), started, 0);
			gtk_clist_select_row(GTK_CLIST (clist), pos, 0);
			gtk_clist_moveto(GTK_CLIST (clist), pos , 0, 0.5, 0);
			display_message(msg, FALSE); 
			free_message(msg); /* Free the memory */
			found=1;
		}
	}
}

/* Sort a column.
 * This sorts the column by modifying the SQL query to order by the field
 * that the user is clicking on.
 * It's a hack.
 * Nevertheless there are two main advantages:
 * 1. Order is remembered between sessions.
 * 2. The database does the job of ordering fields and not us.
 * 
 */
static void
columnclicked_cb (GtkCList * clist, gint column) 
{
	gchar *field = NULL;
	gint pos = -1;
	Mailbox *mailbox = NULL;

	mailbox = gtk_ctree_node_get_row_data(GTK_CTREE (GmailApp->mblist), GTK_CTREE_NODE (GmailApp->node));

	g_return_if_fail(mailbox != NULL);

	/* Find which column was clicked. */
	/* FIXME: these won't be hardcoded forever. */
	switch (column) {
		case 0:
			field = g_strdup("readstatus");
			break;
		case 1:
			field = g_strdup("fromfield");
			break;
		case 2:
			field = g_strdup("subject");
			break;
		case 3:
			field = g_strdup("date");
			break;
		default:
			return;
		}

	if (strcmp(GmailApp->sortfield, field) == 0) {
		if (strcmp(GmailApp->sortorder, "ASC") == 0) {
			g_free(GmailApp->sortorder);
			GmailApp->sortorder = g_strdup("DESC");
		} else {
			g_free(GmailApp->sortorder);
			GmailApp->sortorder = g_strdup("ASC");
		}
	} else {
		g_free(GmailApp->sortfield);
		GmailApp->sortfield = g_strdup(field); 
		GmailApp->sortorder = g_strdup("DESC");
	}

	vfolder_display_rightpane_by_mailbox(mailbox);

	/* Reselect the row that was previously highlighted, if any. 
	 * This code is optional for the user.
	 */
	if (gnome_config_get_bool("/gmail/UserInfo/follow_sort")) {
		pos = gtk_clist_find_row_from_data(GTK_CLIST(GmailApp->rightlist), mailbox);
		gtk_clist_select_row(GTK_CLIST(GmailApp->rightlist), pos, 0);
		gtk_clist_moveto(GTK_CLIST (GmailApp->rightlist), pos, 0, 0.5, 0); 
		}

	/* Record to the config settings which field was sorted. */
	gnome_config_set_string("/gmail/UserInfo/sortfield", GmailApp->sortfield);
	gnome_config_set_string("/gmail/UserInfo/sortorder", GmailApp->sortorder);
	gnome_config_sync();

	g_free(field);
}

/* Called when a user clicks on the attachment clist.
 *
 */
static gint 
attachment_button_press_event (GtkWidget *widget, GdkEventButton *event, gpointer data)
{
	GtkWidget *menu;
	gint result;
	gint row, col;

	g_return_val_if_fail(event != NULL, FALSE);
	if (event->button != 3) return(FALSE);

	/* Trying to stop this stupid thing appearing at the wrong time.  */
	gtk_signal_emit_stop_by_name(GTK_OBJECT(widget), "button_press_event");

	gtk_clist_get_selection_info(GTK_CLIST(widget), event->x, event->y, &row, &col);

	if (row < 0 || row >= GTK_CLIST(widget)->rows) return(FALSE);

	gtk_clist_select_row(GTK_CLIST(widget), row, col);

	menu = gnome_popup_menu_new (mime_popup_menu);
	result = gnome_popup_menu_do_popup_modal (menu, NULL, NULL, event, GINT_TO_POINTER(row));
	gtk_widget_destroy (menu);

	return(TRUE);
}

/* Save attachment to a file.
 * 
 */
static void
save_attach (GtkWidget *widget, gint row)
{
	FILE *fd;
	GtkFileSelection *fsel;
	gboolean accepted = FALSE;
	gint id;
	gchar *title;
	GMimeMessage *message;
	Message *msg = NULL;
	GMimePart *part;
	GList *pos;

	id = (gint) gtk_object_get_data(GTK_OBJECT(GmailApp->msgpane), "id");

	if (id < 0) return;

	/* Load the message. */
	msg = load_vfolder_msg(id); if (msg == NULL) return;

	/* Put the whole message back together in the headers gstring. */
	msg->headers = g_string_append(msg->headers, msg->message->str);

	/* Get GMime to parse it. */
	message = g_mime_parser_construct_message(msg->headers->str, msg->headers->len, FALSE);

	/* Find the mime part that is selected. */
	pos = g_list_nth(message->mime_part->children, row);
	part = pos->data;

	/* Create the file selection widget. */
	title = g_strdup_printf(N_("Save attachment as"));
	fsel = GTK_FILE_SELECTION (gtk_file_selection_new (title));
	g_free(title);

	/* Suggest the file name if it is in the content disposition. */
	gtk_file_selection_set_filename(GTK_FILE_SELECTION(fsel), g_mime_part_get_filename(part));

	gtk_window_set_modal (GTK_WINDOW (fsel), TRUE);

	/* Connect the signals for Ok and Cancel */
	gtk_signal_connect (GTK_OBJECT (fsel->ok_button), "clicked",
          GTK_SIGNAL_FUNC (set_ok), &accepted);

	/* FIXME: Make the cancel button work. */
	gtk_signal_connect_object(GTK_OBJECT (fsel->cancel_button), "clicked",
          GTK_SIGNAL_FUNC (gtk_widget_destroy), GTK_OBJECT(fsel)); 

	gtk_widget_show(GTK_WIDGET(fsel));
	gtk_grab_add (GTK_WIDGET (fsel));
	gtk_main(); 

	if (accepted) {
		char *name = NULL;
		gchar *attach;
		gint len;

		/* Decode the attachment. */
		attach = (gchar *) g_mime_part_get_content(part, &len); 

		name = gtk_file_selection_get_filename (fsel);
		g_print(_("Saving msg to file %s (Length: %i)\n"), name, len);

		/* Open and write the file. */
		if ((fd = fopen(name, "wb"))) {
			fwrite(attach, sizeof(guchar), len, fd);
			fclose(fd);
		} else {
			/* FIXME: Give GUI response. */
			fprintf(stderr, _("Couldn't open %s\n"), name);
		}

		/* Free memory. */
		gtk_widget_destroy (GTK_WIDGET (fsel));

	}

	g_mime_message_destroy(message);
	free_message(msg); /* Free the memory */


	return;
}

/* Called when user double clicks on an attachment.
 * Saves the attachment to a temporary file, then executes it.
 * 
 */
static gint
exec_attach (GtkWidget *clist, gint row, gint column, GdkEventButton *event, gpointer data)
{
	FILE *fd;
	gint id;
	GMimeMessage *message;
	Message *msg = NULL;
	GMimePart *part = NULL;
	GList *pos;
	const char *mime_type = NULL;
	const char *mime_prog = NULL;
	char **argv;
	int argc;
	gint len;
	gchar *fname, *name, *attach = NULL;

	/* g_return_val_if_fail(event != NULL, TRUE); */

	/* If they have double clicked an attachment run it. */
	if (event != NULL && event->button == 1 && event->type == GDK_2BUTTON_PRESS) {

	/* For unknown reasons the signal gets called to many times so
	 * we stop the emission here and now. 
	 */
	gtk_signal_emit_stop_by_name(GTK_OBJECT(clist), "select_row");

	id = (gint) gtk_object_get_data(GTK_OBJECT(GmailApp->msgpane), "id");

	if (id < 0) return(TRUE);

	/* Load the message. */
	msg = load_vfolder_msg(id); if (msg == NULL) return(TRUE);

	/* Put the whole message back together in the headers gstring. */
	msg->headers = g_string_append(msg->headers, msg->message->str);

	/* Get GMime to parse it. */
	message = g_mime_parser_construct_message(msg->headers->str, msg->headers->len, FALSE);

	/* Find the mime part that is selected. */
	pos = g_list_nth(message->mime_part->children, row);
	part = pos->data;

	/* Get the suggested filename. */
	name = g_strdup(g_mime_part_get_filename(part));
	if (name == NULL) name = g_strdup("blah");
	fname = g_strdup_printf(".gmail_%s", name);

	/* Decode the attachment. */
	attach = (gchar *) g_mime_part_get_content(part, &len); 

	/* g_print("len is %i\n", len); */

	/* Open and write the file. 
	 * It gets deleted when the user quits gmail.
	 */
	if ((fd = fopen(fname, "wb"))) {
		fwrite(attach, 1, len, fd);
		fclose(fd);
	} else {
		/* FIXME: Give GUI response. */
		fprintf(stderr, _("Couldn't open %s\n"), fname);
	}


	if (!((mime_type = gnome_mime_type (fname))))
		return(TRUE);
	/* else if (!((mime_type = gnome_mime_type_from_magic (fname))))
	return(TRUE); */

	g_print("mime_type: %s,", mime_type);

	if (!((mime_prog = gnome_mime_program (mime_type))))
	return(TRUE);

	g_print(" mime_prog: %s.\n", mime_prog);

	if (!strstr (mime_prog, "%f"))
	return(TRUE);

	argv = g_strsplit (mime_prog, " ", -1);

	/* Loop through the arguments and replace the %f with the filename.
	 * This doesn't match some mime type setups.
	 * eg: gnome-moz-remote file:%f
	 * The above fails as it has no space around the %f.
	 */
	for (argc=0; argv[argc]; argc++) {
		if (!strcmp(argv[argc], "%f")) {
      			g_free (argv[argc]);
      			argv[argc] = g_strdup (fname);
    		}
	}

	/* g_print("Running %s %s\n", argv[0], argv[1]); */
	gnome_execute_async (NULL, argc, argv);  
	g_strfreev (argv);

	g_free(name);
	g_free(fname);

	}
	return(FALSE);
}

static void make_vfolder_copy (GtkWidget *widget, void *data)
{
	Mailbox	*mailbox, *mailbox2;
	gchar *name[1];
	GtkCTreeNode *node = NULL;
	GdkPixmap *pixmap;
	GdkBitmap *bitmap;

	mailbox = gtk_ctree_node_get_row_data(GTK_CTREE (GmailApp->mblist), GTK_CTREE_NODE (GmailApp->node));

	/* Create new mailbox.  */
	mailbox2 = copy_mailbox(mailbox);

	pixmap = balsa_icon_get_pixmap(BALSA_ICON_TRAY_EMPTY);
	bitmap = balsa_icon_get_bitmap(BALSA_ICON_TRAY_EMPTY);

	name[0] = g_strdup(mailbox2->name);
	node = gtk_ctree_insert_node (GTK_CTREE (GmailApp->mblist), NULL, GTK_CTREE_NODE(node), name, 4, pixmap, bitmap, pixmap, bitmap, TRUE, FALSE); 
	g_free(name[0]); 

	gtk_ctree_node_set_row_data(GTK_CTREE (GmailApp->mblist), GTK_CTREE_NODE(node), mailbox2);

	/* Write the entire clist to the config file */
	save_all_vfolder_data(GTK_CTREE(GmailApp->mblist)); 

	mi_warning(NULL, NULL);
}

/* copy_mailbox - Make a copy of a mailbox structure. */
static Mailbox * copy_mailbox(Mailbox *mailbox)
{
	Mailbox *mailbox2;

	g_return_val_if_fail(mailbox != NULL, NULL);

	mailbox2 = g_malloc (sizeof(Mailbox)); 
	mailbox2->name = g_strdup_printf("copy of %s", mailbox->name);	
	mailbox2->query = g_strdup(mailbox->query);
	mailbox2->email = g_strdup(mailbox->email);

	return(mailbox2);
}

/* Create a new empty vfolder. */
static void make_vfolder_new (GtkWidget *widget, void *data)
{
	Mailbox	*mailbox;
	gchar *name[1];
	GtkCTreeNode *node = NULL;
	GdkPixmap *pixmap;
	GdkBitmap *bitmap;

	pixmap = balsa_icon_get_pixmap(BALSA_ICON_TRAY_EMPTY);
	bitmap = balsa_icon_get_bitmap(BALSA_ICON_TRAY_EMPTY);

	mailbox = g_malloc (sizeof(Mailbox)); 
	mailbox->name = g_strdup(_("New Vfolder"));
	mailbox->query = g_strdup("");
	mailbox->email = g_strdup("");

	name[0] = g_strdup(mailbox->name);
	node = gtk_ctree_insert_node (GTK_CTREE (GmailApp->mblist), NULL, GTK_CTREE_NODE(node), name, 4, pixmap, bitmap, pixmap, bitmap, TRUE, FALSE); 
	g_free(name[0]); 

	gtk_ctree_node_set_row_data(GTK_CTREE (GmailApp->mblist), GTK_CTREE_NODE(node), mailbox);

	/* Write the entire clist to the config file */
	save_all_vfolder_data(GTK_CTREE(GmailApp->mblist));

	/* Open the properties for the new vfolder. */
	GmailApp->node = node;
	vfolder_properties(NULL, NULL);
}

/* Delete a vfolder.
 * FIXME: free the mailbox data.
 */
static void delete_vfolder (GtkWidget *widget, void *data)
{
	gtk_ctree_remove_node(GTK_CTREE (GmailApp->mblist), GTK_CTREE_NODE (GmailApp->node));
	GmailApp->node = NULL; /* nothing currently selected. */

	/* Write the entire clist to the config file */
	save_all_vfolder_data(GTK_CTREE(GmailApp->mblist)); 
}

void 
search_cb (GtkWidget *widget, void *data)
{
	GladeXML *xml;
	gchar *glade_src;
	GtkWidget *wid;

	if (GmailApp->search) return;

	/* Load the glade file. */
	glade_src = g_strdup_printf("%s/search.glade", GMAIL_GLADEDIR);
	xml = glade_xml_new(glade_src, "search");
	g_free(glade_src);

	if (!xml) {
		g_warning(_("Unable to load xml interface. Are the gmail glade files installed onto your system?\n"));
		exit(1);
		}

	glade_xml_signal_autoconnect(xml);

	GmailApp->search = glade_xml_get_widget(xml, "search");
	gnome_dialog_set_default(GNOME_DIALOG(GmailApp->search), 1);

	gnome_dialog_button_connect(GNOME_DIALOG(GmailApp->search), 0, search_ok, NULL);
	gnome_dialog_button_connect(GNOME_DIALOG(GmailApp->search), 1, search_apply, NULL);
	gnome_dialog_button_connect(GNOME_DIALOG(GmailApp->search), 2, search_close, NULL);
	gtk_widget_show(GmailApp->search);

	/* if (GmailApp->app)
             gnome_dialog_set_parent (GNOME_DIALOG (GmailApp->search), GTK_WINDOW (GmailApp->app));*/

	wid = glade_xml_get_widget(xml, "text"); 

	gtk_signal_connect(GTK_OBJECT(wid), "activate",
                       GTK_SIGNAL_FUNC(search_apply), wid); 

	/* Let them begin typing in the entry */
	gtk_widget_grab_focus(GTK_WIDGET(wid));
}

void search_apply (GtkWidget *widget, void *data)
{
	GladeXML *xml;
	GtkWidget *wid;
	gchar *text;
	Mailbox *mailbox, *vfolder;
	GString *query = g_string_new(NULL);

	xml = glade_get_widget_tree(GTK_WIDGET(widget));
	wid = glade_xml_get_widget(xml, "text"); 
	text = gtk_entry_get_text(GTK_ENTRY(wid));

	/* Don't do anything if no text is entered. */
	if (strlen(text) == 0) return;

	/* Create a fake vfolder that we display. */
	mailbox = g_malloc (sizeof(Mailbox)); 
	mailbox->name = g_strdup("_Search"); /* Reserved name */
	mailbox->email = g_strdup("");

	/* Create the query part. */
	/* FIXME: have buttons for different options, and cat them together.
	 * eg: incoming, outgoing, this vfolder only, headers, body.
	 */

	wid = glade_xml_get_widget(xml, "current"); 
	if (GTK_TOGGLE_BUTTON (wid)->active) {
		/* Get info about currently selected vfolder. */
		vfolder = gtk_ctree_node_get_row_data(GTK_CTREE (GmailApp->mblist), GTK_CTREE_NODE (GmailApp->node));

		/* Use caching if it is turned on. */
		if (cache_vfolders()) 
			g_string_sprintfa(query, "(FIND_IN_SET('%s', matched)) AND ", vfolder->name);
		 else 
			g_string_sprintfa(query, "(%s) AND ", vfolder->query);
	}

	/* Add search for our text string. */
	g_string_sprintfa(query, "((details.headers LIKE \"%%%s%%\") OR (details.message LIKE \"%%%s%%\"))", text, text);

	mailbox->query = g_strdup(query->str);

	g_string_free(query, TRUE);

	vfolder_display_rightpane_by_mailbox(mailbox);

	/* FIXME: Free the mailbox data. */
}

void search_ok (GtkWidget *widget, void *data) 
{
	/* Do the search. */
	search_apply(widget, data);

	/* Now close the widget. */
	search_close(widget, data);

}

void search_close (GtkWidget *widget, void *data) 
{
	GladeXML *xml;
	GtkWidget *wid;

	xml = glade_get_widget_tree(GTK_WIDGET(widget));
	wid = glade_xml_get_widget(xml, "search"); 

	gnome_dialog_close(GNOME_DIALOG(wid));

	GmailApp->search = FALSE;
}

/*
 * Called when the user is dragging a gmail email address from the message
 * box.
 *
 * FIXME: generalise to work for any label in the message box.
 */
void
address_drag_data_get  (GtkWidget        *widget,
			GdkDragContext   *context,
         		GtkSelectionData *selection_data,
         		guint            info,
         		guint32          time)
{
	char *data = NULL; 

	/* Find the label from the widget data. */
	if (GTK_IS_LABEL(GTK_BIN(widget)->child))
		data = g_strdup(GTK_LABEL(GTK_BIN(widget)->child)->label);
	else
		return;	

	gtk_selection_data_set (selection_data, selection_data->target, 8, data, strlen (data)+1); 

	g_free(data);
}

/* For drag and drop from the msg clist.
 * Grabs the authors name from the clist.
 */
void
clist_drag_data_get    (GtkWidget        *widget,
			GdkDragContext   *context,
         		GtkSelectionData *selection_data,
         		guint            info,
         		guint32          time)
{
	GList	*sel = NULL;
	gint pos = -1, id;
	char *data = NULL; 
	Message *msg;

	g_return_if_fail(GTK_IS_CLIST(widget));

	/* Get the position of the selected message. */
	sel = GTK_CLIST(GmailApp->rightlist)->selection;
	if (sel != NULL) pos = (gint) sel->data; else return;

	/* Get the mysql id of the selected message. */
	id = GPOINTER_TO_INT(gtk_clist_get_row_data(GTK_CLIST(widget), pos));

	/* Load and display the image. */
	msg = load_vfolder_msg(id);

	gtk_selection_data_set (selection_data, selection_data->target, 8, msg->from, strlen(msg->from)+1); 

	g_free(data);
	free_message(msg);
}


/* vfolder_display_rightpane_by_mailbox 
 *
 * This function does alot of different things, and should probably be
 * refactored with a better design.
 *
 * Basically it does the following:
 * 
 * 	- Decide what query to run (based on whether we are caching or not)
 *	- Load the query in the mailbox.
 *	- Display it in the right clist.
 * 	- Updated the matched index for each message (if not caching).
 *	- If a cached vfolder returns no results, check if it is because 
 *      they are running vfolder not in the MI yet.
 *
 * Probably the first step in refactoring is to write a seperate function to
 * handle search queries. Make a function to handle any common code between
 * the search and vfolder cases.
 *
 */
void 
vfolder_display_rightpane_by_mailbox(Mailbox *mailbox)
{
	MYSQL_RES *res;
	MYSQL_ROW mysqlrow;
	gint	mid;
	gboolean unread_found = FALSE;
	gint unread_count = 0;
	gchar *query = NULL, *newquery = NULL;
	gchar *msg = NULL;
	Message *mesg = NULL;
	gboolean search = FALSE;

	g_return_if_fail(mailbox != NULL);
	g_return_if_fail(mailbox->query != NULL);
	g_return_if_fail(mailbox->name != NULL);

	query = mailbox->query;

	/* g_print("query is %s\n", query); */

	/* Check that query is valid. */
	if (query == NULL || strlen(query) < 2) {
		GtkWidget *dialog;
		gchar *msg;
		msg = g_strdup_printf (_("Vfolder contains empty query.\n"));
		dialog = gnome_message_box_new(msg, GNOME_MESSAGE_BOX_ERROR, "doh!", 0);
		gtk_window_set_modal(GTK_WINDOW (dialog), TRUE);
 		gtk_window_set_position(GTK_WINDOW(dialog), GTK_WIN_POS_CENTER);
 		gtk_widget_show(dialog);
		g_free(msg);	
		g_warning("vfolder_display_rightpane passed bad data.\n");
		return;
		}

	/* Open Database. */
	if (!check_db(&GmailApp->mysql)) { return; };

	/* Is this a search query? */
	if (strcmp(mailbox->name, "_Search") == 0) search = TRUE;

	/* 
	 * Create the query to call to generate the vfolder.
	 * We run the query normally if:
	 * 	- Caching is turned off.
	 * 	- If it is the Unread vfolder
	 * 	- If it is a search query.
	 *
	 * If we are caching, just search on the matched index.
	 *
	 * NOTE: if we're not caching we update the matched index on the fly.
	 * FIXME: This function should be refactored into the part that calls
	 * and displays a query and the part that does the MI specific stuff.
	 * There are too many flags and crap.
	 */	
	if (search || (strcmp(mailbox->name, "Unread") == 0) || !cache_vfolders()) 
		newquery = g_strdup_printf("SELECT readstatus,fromfield,subject,date,display.id FROM display,details WHERE display.id = details.id AND (%s) ORDER BY %s %s", query, GmailApp->sortfield, GmailApp->sortorder);
	else
		newquery = g_strdup_printf("SELECT readstatus,fromfield,subject,date,id FROM display WHERE FIND_IN_SET('%s', matched) ORDER BY %s %s", mailbox->name, GmailApp->sortfield, GmailApp->sortorder);

	/* g_print("newquery: %s\n", newquery); */

	if (mysql_query(&GmailApp->mysql, newquery) != 0) {
		GtkWidget *dialog;
		gchar *msg;
		msg = g_strdup_printf (_("MYSQL Query failed:\n%s\n"), mysql_error(&GmailApp->mysql));
		dialog = gnome_message_box_new(msg, GNOME_MESSAGE_BOX_ERROR, "doh!", 0);
		gtk_window_set_modal(GTK_WINDOW (dialog), TRUE);
 		gtk_window_set_position(GTK_WINDOW(dialog), GTK_WIN_POS_CENTER);
 		gtk_widget_show(dialog);
		g_free(msg);	
		g_print("Failed query was: %s\n", newquery);
		g_free(newquery);
		return;
		} 

	g_free(newquery);

	res = mysql_use_result(&GmailApp->mysql);

	/* If we already have the clist , just clear it. 
	 * Reuse this widget to avoid flicker.
	 */
	if (GTK_IS_CLIST(GmailApp->rightlist)) {
		gtk_clist_freeze(GTK_CLIST(GmailApp->rightlist));
		gtk_clist_clear(GTK_CLIST(GmailApp->rightlist));
		} else {
		rightlist_clist_create(); /* Create the rightlist. */
		gtk_clist_freeze(GTK_CLIST(GmailApp->rightlist));
		}

	/* Make it so you can select multiple messages at once */
	gtk_clist_set_selection_mode(GTK_CLIST(GmailApp->rightlist),GTK_SELECTION_EXTENDED);
	
	/* Ok here is where life gets tricky. In 2 pane mode, we always 
	 * keep a copy of the vfolder rightlist around so the prev and next 
	 * buttons work while the clist isn't even open (eg: whilst viewing 
	 * messages).
	 * So we need to recreate the scrolled window and put the old rightlist
	 * back inside.
	 * 3 pane mode is alot easier!
	 */
	if (TwoPaneView() && !GTK_IS_SCROLLED_WINDOW (GmailApp->msglistpane)) {
		gtk_widget_destroy(GmailApp->msgpane); 
		GmailApp->msgpane = NULL;
  		GmailApp->msglistpane = gtk_scrolled_window_new (NULL, NULL);
		gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (GmailApp->msglistpane),
						GTK_POLICY_AUTOMATIC, 
						GTK_POLICY_AUTOMATIC);    
		gtk_paned_add2 (GTK_PANED(GmailApp->hpaned), GmailApp->msglistpane);
		gtk_container_add(GTK_CONTAINER(GmailApp->msglistpane), GmailApp->rightlist);
		}

	gnome_appbar_set_status (GNOME_APPBAR (GmailApp->appbar), _("Updating clist."));
	gnome_appbar_refresh(GNOME_APPBAR (GmailApp->appbar));

	/* Add each row in the result as a clist item. 
	 * This is called once per message in the vfolder.
	 *
	 * We could use the strip_name function here, but we are trying
	 * to minimise function calls in this loop.
	 *
	 * FIXME: Should do optimisations here. Too many strdups.
	 */
	while ((mysqlrow = mysql_fetch_row(res))) {
		gchar *datum[4];
		gint id;
		GList *ias = internet_address_parse_string(mysqlrow[1]);
		InternetAddress *ia = NULL;

		if (ias) ia = ias->data;

		datum[0] = mysqlrow[0];

		/* If they want to see just the persons name, and the 
		 * gmime internet_address is parsed properly,
		 * then give them the name, or the ia email address.
		 * Otherwise just paste the raw field we have.
		 */
		if (gnome_config_get_bool("/gmail/UserInfo/fullfrom=FALSE") == FALSE && ia) {
			if (ia->name) datum[1] = ia->name; /* We have a name. */
			else if (ia->value.addr) datum[1] = ia->value.addr;
			else datum[1] = mysqlrow[1];
		} else {
			/* Either they want the full field or the internet
			 * address parsing failed.
			 */
			datum[1] = mysqlrow[1];
		}

		datum[2] = mysqlrow[2];
		datum[3] = mysqlrow[3];

		/* We use prepend as it is apparently O(1). */
		gtk_clist_prepend (GTK_CLIST (GmailApp->rightlist), datum);

		/* Destroy the ia AFTER adding the data to the clist. */
		if (ia) internet_address_destroy(ia);

		/* Set unread messages to look a bit different. 
		 * This code used to check for:
	  	 * if (strncmp(datum[0], "Unread", 6) == 0) {	
		 * But we know it is either 'Unread' or 'Read', so
		 * we can simply check the first char, saving a function call.
		 */
	  	 if (datum[0][0] == 'U') {	
			gtk_clist_set_cell_style(GTK_CLIST (GmailApp->rightlist), 0, 0, GmailApp->boldstyle);
			gtk_clist_set_cell_style(GTK_CLIST (GmailApp->rightlist), 0, 1, GmailApp->boldstyle);
			gtk_clist_set_cell_style(GTK_CLIST (GmailApp->rightlist), 0, 2, GmailApp->boldstyle);
			gtk_clist_set_cell_style(GTK_CLIST (GmailApp->rightlist), 0, 3, GmailApp->boldstyle);
			unread_found = TRUE;
			unread_count++;
			} 

		id = (gint) atoi(mysqlrow[4]);

		/* Store the mysql id of the email as the row data. */
		gtk_clist_set_row_data_full(GTK_CLIST (GmailApp->rightlist), 0, GINT_TO_POINTER(id), NULL);

		/* When we are not caching, we also update the matched field
		 * on the run. 
		 * This is to make the inbox work.
		 * This gives a small speed hit, but if they want speed
		 * they should be caching.
		 */
		if (!search && !cache_vfolders()) {
			GString *vname = g_string_new(mailbox->name);	
			if (vname != NULL && atoi(mysqlrow[4]) > 0)
			 	mark_as_matched(atoi(mysqlrow[4]), vname, FALSE);  
			g_string_free(vname, TRUE);
			}

		}
	gtk_clist_thaw(GTK_CLIST(GmailApp->rightlist));

	mysql_free_result(res); 

	/* Update vfolders unread status. */
	if (!search) mark_vfolder_unread_status_by_name(mailbox->name, unread_found);

	gtk_window_set_title(GTK_WINDOW (GmailApp->app), _("Gmail - Mysql vfolder"));
	/* Highlight the first message, if it exists. */
	if (GTK_CLIST(GmailApp->rightlist)->rows > 0) {
		gtk_clist_select_row(GTK_CLIST(GmailApp->rightlist), 0, 0); 
		mid = GPOINTER_TO_INT(gtk_clist_get_row_data(GTK_CLIST (GmailApp->rightlist), 0));

		/* If we are in three pane mode select the first email. */
		if (!TwoPaneView() && gnome_config_get_bool_with_default("/gmail/UserInfo/display_first_message=false", NULL)) {
			mesg = load_vfolder_msg(mid);
			if (mesg != NULL) display_message(mesg, TRUE); 
			free_message(mesg); /* Free the memory */
		}
	}

	/* Display info about the vfolder in the appbar. */
	msg = g_strdup_printf(_("Vfolder '%s' contains %i messages (%i unread)."), mailbox->name, GTK_CLIST(GmailApp->rightlist)->rows, unread_count);
	gnome_appbar_set_status (GNOME_APPBAR (GmailApp->appbar), msg);
	g_free(msg);

	gtk_widget_show_all(GmailApp->msglistpane);

	/* FIXME: If there were no messages in the vfolder, it may be because
	 * we are in caching mode, and that vfolder hasn't been put into 
	 * the matched index. Check for this, and if so tell the user and offer
	 * to rebuild the matched index.
	 */
	if (!search && GTK_CLIST(GmailApp->rightlist)->rows == 0 && cache_vfolders()) {
		gboolean mi_current = FALSE;

		/* Open Database. */
		if (!check_db(&GmailApp->mysql)) { return; };

		/* Check if vfolder name is in the matched index. */
		newquery = g_strdup_printf("SHOW COLUMNS FROM display LIKE 'matched'");

		if (mysql_query(&GmailApp->mysql, newquery) != 0) {
			msg = g_strdup_printf ("MYSQL Query failed:\n%s\n", mysql_error(&GmailApp->mysql));
			g_warning("%s\nFailed query was: %s\n", msg, newquery);
			g_free(newquery);
			g_free(msg);	
			return;
		} 

		g_free(newquery);

		res = mysql_use_result(&GmailApp->mysql);

		while ((mysqlrow = mysql_fetch_row(res))) {
			/* If we find the name in the SET description, then MI is up to date. */
			if (strstr(mysqlrow[1], mailbox->name) != NULL)
				mi_current = TRUE;
			}	

		mysql_free_result(res); 

		/* FIXME: this function locks up with the ctree on. */
		/* Warn the user if vfolder name wasn't in the MI. */
		if (FALSE && !mi_current) {
			GtkWidget *dialog, *label;
			gint button;
			static GnomeHelpMenuEntry help_ref = {PACKAGE, "gmail/caching.html" };

			dialog = gnome_dialog_new("Gmail Warning", GNOME_STOCK_PIXMAP_INDEX, GNOME_STOCK_BUTTON_OK, GNOME_STOCK_BUTTON_HELP, NULL);

			/* gtk_window_set_modal(GTK_WINDOW (dialog), TRUE); */

			label = gtk_label_new("The reason that this vfolder contains no messages is because you are in caching mode,\nbut this vfolder is not in your 'matched index'.\nTo recreate your MI, click on 'Index' or click 'Ok' to simply continue.");
			gtk_box_pack_start (GTK_BOX (GNOME_DIALOG(dialog)->vbox), label, FALSE, FALSE, 5);

			gtk_widget_show_all(GNOME_DIALOG(dialog)->vbox);
			gnome_dialog_set_close(GNOME_DIALOG(dialog), TRUE);

			switch(gnome_dialog_run_and_close(GNOME_DIALOG(dialog))) {
 		 		case -1: break;

				case 0: recreate_matched_index(NULL, NULL);
		     	       		break;
	
 		 		case 1: break;

 		   		case 2: gnome_help_display(0, &help_ref);
					break;

 		   		default: break;
			}
		}
	}
}

static void rightlist_clist_create(void)
{
	gint width;
	gchar *titles[4] = { _("Status"), _("From"), _("Subject"), _("Date")};
	GtkWidget *gnomepixmap = NULL;
	GtkWidget *pixtitle;

	GmailApp->rightlist = gtk_clist_new(4);

	gtk_clist_set_shadow_type(GTK_CLIST(GmailApp->rightlist), GTK_SHADOW_ETCHED_IN); 

	/* Get a stock icon in the clist titles.   */
	gnomepixmap = gnome_stock_new_with_icon(GNOME_STOCK_MENU_EXEC); 
	pixtitle = gnome_pixmap_button((GtkWidget *) gnomepixmap, titles[0]); 
	gtk_button_set_relief(GTK_BUTTON(pixtitle), GTK_RELIEF_NONE);
	gtk_clist_set_column_widget(GTK_CLIST (GmailApp->rightlist), 0, pixtitle);

	/* Callback for clicking on a column title. */
	gtk_signal_connect(GTK_OBJECT(pixtitle), "clicked",
                       GTK_SIGNAL_FUNC(columnclicked_cb), GINT_TO_POINTER(0));

	gnomepixmap = gnome_stock_new_with_icon(GNOME_STOCK_MENU_HOME); 
	pixtitle = gnome_pixmap_button((GtkWidget *) gnomepixmap, titles[1]); 
	gtk_button_set_relief(GTK_BUTTON(pixtitle), GTK_RELIEF_NONE);
	gtk_clist_set_column_widget(GTK_CLIST (GmailApp->rightlist), 1, pixtitle);

	/* Callback for clicking on a column title. */
	gtk_signal_connect(GTK_OBJECT(pixtitle), "clicked",
                       GTK_SIGNAL_FUNC(columnclicked_cb), GINT_TO_POINTER(1));

	gnomepixmap = gnome_stock_new_with_icon(GNOME_STOCK_MENU_BOOK_OPEN); 
	pixtitle = gnome_pixmap_button((GtkWidget *) gnomepixmap, titles[2]); 
	gtk_button_set_relief(GTK_BUTTON(pixtitle), GTK_RELIEF_NONE);
	gtk_clist_set_column_widget(GTK_CLIST (GmailApp->rightlist), 2, pixtitle);

	/* Callback for clicking on a column title. */
	gtk_signal_connect(GTK_OBJECT(pixtitle), "clicked",
                       GTK_SIGNAL_FUNC(columnclicked_cb), GINT_TO_POINTER(2));

	gnomepixmap = gnome_stock_new_with_icon(GNOME_STOCK_MENU_TIMER); 
	pixtitle = gnome_pixmap_button((GtkWidget *) gnomepixmap, titles[3]); 
	gtk_button_set_relief(GTK_BUTTON(pixtitle), GTK_RELIEF_NONE);
	gtk_clist_set_column_widget(GTK_CLIST (GmailApp->rightlist), 3, pixtitle);

	/* Callback for clicking on a column title. */
	gtk_signal_connect(GTK_OBJECT(pixtitle), "clicked",
                       GTK_SIGNAL_FUNC(columnclicked_cb), GINT_TO_POINTER(3));

	gtk_clist_column_titles_show(GTK_CLIST (GmailApp->rightlist));

	width = gnome_config_get_int("/gmail/UserInfo/rightlist0"); 
	if (width <= 0) width = 50;
	gtk_clist_set_column_width (GTK_CLIST (GmailApp->rightlist), 0, width);
	width = gnome_config_get_int("/gmail/UserInfo/rightlist1"); 
	if (width <= 0) width = 120;
	gtk_clist_set_column_width (GTK_CLIST (GmailApp->rightlist), 1, width);
	width = gnome_config_get_int("/gmail/UserInfo/rightlist2"); 
	if (width <= 0) width = 120;
	gtk_clist_set_column_width (GTK_CLIST (GmailApp->rightlist), 2, width);
	width = gnome_config_get_int("/gmail/UserInfo/rightlist3"); 
	if (width <= 0) width = 50;
	gtk_clist_set_column_width (GTK_CLIST (GmailApp->rightlist), 3, width);
	gtk_clist_set_selection_mode (GTK_CLIST (GmailApp->rightlist), GTK_SELECTION_SINGLE);
	gtk_clist_column_titles_active (GTK_CLIST (GmailApp->rightlist));
	gtk_widget_set_style (GmailApp->rightlist, GmailApp->style); 
	gtk_widget_ref(GmailApp->rightlist);

	/* Callback for clicking on a row. */
	gtk_signal_connect(GTK_OBJECT(GmailApp->rightlist), "select_row",
                       GTK_SIGNAL_FUNC(msg_clicked_cb),
                       NULL);

	/* Callback for clicking on a column title. */
	gtk_signal_connect(GTK_OBJECT(GmailApp->rightlist), "click_column",
                       GTK_SIGNAL_FUNC(columnclicked_cb),
                       NULL);

	/* Callback for resizing a column. */
	gtk_signal_connect(GTK_OBJECT(GmailApp->rightlist), "resize_column",
                       GTK_SIGNAL_FUNC(columnresize_cb),
                       NULL);

	/* callback for right click menu. */
	gtk_signal_connect (GTK_OBJECT (GmailApp->rightlist), "button_press_event",
       	 GTK_SIGNAL_FUNC (rightlist_popup), NULL);


	/* Setup dnd source for the clist. */
	gtk_drag_source_set (GmailApp->rightlist, GDK_BUTTON1_MASK, email_field_drop_types, 1, GDK_ACTION_COPY | GDK_ACTION_DEFAULT); 
	gtk_signal_connect(GTK_OBJECT(GmailApp->rightlist), "drag_data_get", GTK_SIGNAL_FUNC(clist_drag_data_get), NULL);  

	/* Create the scrolled window to store it in. */
	GmailApp->msglistpane = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (GmailApp->msglistpane),
					GTK_POLICY_AUTOMATIC, 
					GTK_POLICY_AUTOMATIC);    
	if (TwoPaneView())
		gtk_paned_add2 (GTK_PANED(GmailApp->hpaned), GmailApp->msglistpane);
	else 
		gtk_paned_add1 (GTK_PANED(GmailApp->rightpane), GmailApp->msglistpane);

	/* Add it to our msglistpane scrolled window container. */
	gtk_container_add(GTK_CONTAINER(GmailApp->msglistpane), GmailApp->rightlist);

} /* Creation of clist */

/* Remeber the column resizes by setting gnome_config. */
static void
columnresize_cb  (GtkCList *clist, gint column, gint width, gpointer user_data) {
	gchar *config = NULL;

	if (GmailApp->rightlist != NULL && clist == GTK_CLIST(GmailApp->rightlist)) {
		config = g_strdup_printf("/gmail/UserInfo/rightlist%i", column);
		gnome_config_set_int(config, width);
	} 
	/* g_print("Resize %s by %i\n", config, width); */
	gnome_config_sync();	/* Write info to config file */
	g_free(config);
}


/* msg_clicked - Open a vfolder message in the right pane.
 * This function is called when the user double clicks on a message
 * in a vfolder message list in the right pane.
 * It gets the message id from the clist row data.
 *
 */
static void
msg_clicked_cb (GtkWidget *clist, gint row, gint column, GdkEventButton *event, gpointer data)
{
	gchar *title;
	Message *msg = NULL;
	gint id;
	GList *ias;
	InternetAddress *ia = NULL;
	gchar *name;
	
	id = GPOINTER_TO_INT(gtk_clist_get_row_data(GTK_CLIST(GmailApp->rightlist), row));

	/* Load the msg details so we can set the title, display it etc. */
	msg = load_vfolder_msg(id);

	if (msg == NULL) return;

	/*
	 * Set the status bar message with the msg database id.
	 */
	title = g_strdup_printf("Msg id %i", msg->id);
	gnome_appbar_set_status (GNOME_APPBAR (GmailApp->appbar), title);
	g_free(title);

	/* Get the name of the sender. */
	ias = internet_address_parse_string(msg->from);
	if (ias && ias->data) {
		ia = ias->data;
		if (ia && ia->name)
			name = g_strdup(ia->name);
		else 
			name = g_strdup(ia->value.addr);
		} else {
		name = g_strdup(msg->from);
	}
	

	/*
	 * Set the window title with from/subject. 
	 */
	title = g_strdup_printf(_("Gmail - From: %s, Subject: %s"), name, msg->subject);
	gtk_window_set_title(GTK_WINDOW (GmailApp->app), title);
	g_free(title);
	internet_address_destroy(ia);


	/* Display message */
	if (msg != NULL && event != NULL && event->button == 1) {
		if (event->type == GDK_2BUTTON_PRESS || !TwoPaneView()) {
				display_message(msg, FALSE);
				}
		}

	free_message(msg); /* Free the memory */
}

static void
rightlist_popup (GtkWidget *widget, GdkEventButton *event, gpointer data)
{
	gint result;
	gint row, col;

	if (event->button != 3 || event->type != GDK_BUTTON_PRESS) return;

	gtk_clist_get_selection_info(GTK_CLIST(widget), event->x, event->y, &row, &col);
	gtk_clist_select_row(GTK_CLIST(widget), row, col);

  GmailApp->rightlist_popup = gnome_popup_menu_new (rightlist_popup_menu);
	result = gnome_popup_menu_do_popup_modal (GmailApp->rightlist_popup, NULL, NULL, event, widget);
  gtk_widget_destroy (GmailApp->rightlist_popup);
	/* g_print("User chose %i\n", result); */
}

/* 
 * Open a message from the rightclick context menu.
 */
static void ctx_open_msg (GtkWidget *widget, gpointer data)
{
	Message *msg = NULL;
	GList	*sel = NULL;
	gint pos = -1, *id;

	/* Get the position of the selected message. */
	sel = GTK_CLIST(GmailApp->rightlist)->selection;
	if (sel != NULL) pos = (gint) sel->data;

	/* Get the mysql id of the selected message. */
	id = (gint *) gtk_clist_get_row_data(GTK_CLIST(GmailApp->rightlist), pos);	

	/* Load and display the image. */
	msg = load_vfolder_msg((gint) *id);

	if (msg != NULL)  display_message(msg, FALSE); 
	free_message(msg); /* Free the memory */
}


static void mark_msg_read(GtkWidget *widget, gboolean data)
{
	gint id;
	GList	*sel = NULL,*tempsel=NULL;
	gint pos = -1,finished=0;
	gchar *stat_string;

	stat_string = g_strdup("Read");
	
	if (GTK_CLIST(GmailApp->rightlist)->rows == 0) return;
	/* Get the position of the selected message. */
	sel = GTK_CLIST(GmailApp->rightlist)->selection;
	/* Loop through all selected messages */
	while(!finished){
		if (sel != NULL) pos = (gint) sel->data;
		if (sel->next==NULL)
			finished=1;
		else 
			tempsel=sel->next;
		if (pos < 0) return;	/* Nothing to mark. */

		/* Get the id of the msg from the row data. */
		id = GPOINTER_TO_INT(gtk_clist_get_row_data(GTK_CLIST (GmailApp->rightlist), pos));

		mark_readstatus(id, stat_string);

		/* Mark as read in the clist. */
		if (GTK_IS_CLIST(GmailApp->rightlist)) {
			gtk_clist_set_text(GTK_CLIST(GmailApp->rightlist), pos,0,stat_string);
			gtk_clist_set_cell_style(GTK_CLIST (GmailApp->rightlist), pos, 0, GmailApp->style);
			gtk_clist_set_cell_style(GTK_CLIST (GmailApp->rightlist), pos, 1, GmailApp->style);
			gtk_clist_set_cell_style(GTK_CLIST (GmailApp->rightlist), pos, 2, GmailApp->style);
			gtk_clist_set_cell_style(GTK_CLIST (GmailApp->rightlist), pos, 3, GmailApp->style);
		}

		if(!finished)
			sel=tempsel;
	}
}

static void mark_msg_unread(GtkWidget *widget, gboolean data)
{
	gint id;
	GList	*sel = NULL,*tempsel=NULL;
	gint pos = -1,finished=0;
	gchar *stat_string;
	/*  if(data)
    	stat_string=g_strdup("Unread");
    else
    	stat_string=g_strdup("Read");
    */
	stat_string=g_strdup("Unread");
	
	if (GTK_CLIST(GmailApp->rightlist)->rows == 0) return;
	/* Get the position of the selected message. */
	sel = GTK_CLIST(GmailApp->rightlist)->selection;
	/* Loop through all selected messages */
	while(!finished){
		if (sel != NULL) pos = (gint) sel->data;
		if (sel->next==NULL)
			finished=1;
		else 
			tempsel=sel->next;
		if (pos < 0) return;	/* Nothing to mark. */

		/* Get the id of the msg from the row data. */
		id = GPOINTER_TO_INT(gtk_clist_get_row_data(GTK_CLIST (GmailApp->rightlist), pos));
		mark_readstatus(id, stat_string);

		/* Mark as read in the clist. */
		if (GTK_IS_CLIST(GmailApp->rightlist)) {
			gtk_clist_set_text(GTK_CLIST(GmailApp->rightlist), pos, 0, stat_string);
			gtk_clist_set_cell_style(GTK_CLIST (GmailApp->rightlist), pos, 0, GmailApp->boldstyle);
			gtk_clist_set_cell_style(GTK_CLIST (GmailApp->rightlist), pos, 1, GmailApp->boldstyle);
			gtk_clist_set_cell_style(GTK_CLIST (GmailApp->rightlist), pos, 2, GmailApp->boldstyle);
			gtk_clist_set_cell_style(GTK_CLIST (GmailApp->rightlist), pos, 3, GmailApp->boldstyle);
		}

		if(!finished)
			sel=tempsel;
	}
}
