/*
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * See the COPYING file for license information.
 *
 * Guillaume Chazarain <booh@altern.org>
 */

/*************************
 * The MODELVIEW matrix. *
 *************************/

#include "gliv.h"

#include <string.h>             /* memcpy() */
#include <math.h>               /* cos(), sin(), ... */
#include <GL/gl.h>

#ifndef HAVE_HYPOT
#define hypot(x, y) (sqrt((x) * (x) + (y) * (y)))
#endif

extern rt_struct *rt;
extern options_struct *options;
extern gliv_image *current_image;

/*
 * OpenGL uses a transposed matrix, we use a 'normal' one,
 * we transpose it just before glLoadMatrix().
 *
 * Value:         Index:
 * c1  s1  0  x | 0   1   2   3
 * s2  c2  0  y | 4   5   6   7
 * 0   0   1  0 | 8   9   10  11 \ constant so
 * 0   0   0  1 | 12  13  14  15 / unused.
 */
static gdouble matrix[8] = {
    /* We only use two rows. */
    1.0, 0.0, 0.0, 0.0,
    0.0, 1.0, 0.0, 0.0
};

/* Used to know if the OpenGL matrix and this one are in sync. */
static gboolean matrix_changed = TRUE;

G_GNUC_PURE gdouble get_matrix_zoom(void)
{
    /*
     * c1 == zoom*cos, s1 == zoom*sin
     * cos^2 + sin^2 == 1 => c1^2 + s1^2 == zoom^2
     */
    return hypot(matrix[0], matrix[1]);
}

/* To be displayed in the status bar. */
G_GNUC_PURE gdouble get_matrix_angle(void)
{
    gdouble cosinus, angle;

    cosinus = matrix[0] / get_matrix_zoom();
    angle = acos(cosinus);

    if (matrix[1] < 0)
        /* Negative sinus => negative angle. */
        angle *= -1.0;

    return angle;
}

/* OpenGL coordinates to window coordinates. */
static void point_coord(gdouble x, gdouble y, gdouble * res_x, gdouble * res_y)
{
    /* OpenGL coordinates through the modelview matrix. */
    *res_x = matrix[0] * x + matrix[1] * y + matrix[3];
    *res_y = matrix[4] * x + matrix[5] * y + matrix[7];

    /* And now through the projection matrix. */
    *res_x += rt->wid_size->width / 2.0;
    *res_y += rt->wid_size->height / 2.0;
}

/*** Input, output. ***/

void write_gl_matrix(void)
{
    /* *INDENT-OFF* */
    static gdouble transposed[16] = {
	1.0, 0.0, 0.0, 0.0,
	0.0, 1.0, 0.0, 0.0,
	0.0, 0.0, 1.0, 0.0,
	0.0, 0.0, 0.0, 1.0
    };
    /* *INDENT-ON* */

    if (matrix_changed == TRUE) {
        transposed[0] = matrix[0];      /* c1 */
        transposed[5] = matrix[5];      /* c2 */
        transposed[4] = matrix[1];      /* s1 */
        transposed[1] = matrix[4];      /* s2 */
        transposed[12] = matrix[3];     /* x */
        transposed[13] = matrix[7];     /* y */

        glLoadMatrixd(transposed);
        matrix_changed = FALSE;
    }
}

/* In paramaters dest and src, NULL is the current matrix. */
void matrix_cpy(gdouble * dest, gdouble * src)
{
    if (dest == NULL) {
        dest = matrix;
        matrix_changed = TRUE;
    } else
        /* src == NULL */
        src = matrix;

    memcpy(dest, src, 8 * sizeof(gdouble));
}

/*** Informations gathering about the matrix. ***/

/*
 * Position of the (x, y) point with respect to the line that
 * goes through (x0, y0) and (x1, y1).
 *       < 0 : point is below
 *       > 0 : point is above
 */
G_GNUC_CONST static gdouble point_and_line(gdouble x, gdouble y,
                                           gdouble x0, gdouble y0,
                                           gdouble x1, gdouble y1)
{
    gdouble line_y;

    /*
     * We cannot have (x0 == x1) because we check first
     * if the rotation is a multiple of Pi/2.
     */
    line_y = (y0 * (x - x1) + y1 * (x0 - x)) / (x0 - x1);

    return line_y - y;
}

#ifdef TEST_TILER
#define MIN_X (rt->wid_size->width / 4.0)
#define MAX_X (rt->wid_size->width * 3.0 / 4.0)
#define MIN_Y (rt->wid_size->height / 4.0)
#define MAX_Y (rt->wid_size->height * 3.0 / 4.0)
#else
#define MIN_X 0.0
#define MAX_X ((gdouble) rt->wid_size->width)
#define MIN_Y 0.0
#define MAX_Y ((gdouble) rt->wid_size->height)
#endif

/* Check each vertex of the tile. */
G_GNUC_PURE static gboolean point_is_in_window(gdouble x, gdouble y)
{
    return (x >= MIN_X && x <= MAX_X && y >= MIN_Y && y <= MAX_Y);
}

#define CHECK_POINT_IN_WINDOW(pt_x, pt_y) \
    if (point_is_in_window(pt_x, pt_y) == TRUE) \
        return TRUE

/* tmp is declared in matrix_tile_visible(). */
#define DOUBLE_XCHG(a, b) \
    do {                  \
        tmp = a;          \
        a = b;            \
        b = tmp;          \
    } while (0)

#define SORT_PT(x0, y0, x1, y1) \
    if (x0 > x1) {              \
        DOUBLE_XCHG(x0, x1);    \
        DOUBLE_XCHG(y0, y1);    \
    }

G_GNUC_PURE static gboolean angle_is_right(void)
{
    gdouble tmp, modulo;

    modulo = modf(get_matrix_angle() * 2.0 / PI, &tmp);

    return DOUBLE_EQUAL(modulo, 0.0);
}

G_GNUC_PURE gboolean matrix_tile_visible(tile_dim * tile)
{
    gdouble x0, y0, x1, y1;
    gdouble x2, y2, x3, y3;
    gdouble min_tile_x, max_tile_x, min_tile_y, max_tile_y;
    gdouble tmp;

    /* Compute the tile position in the window. */
    point_coord(tile->x0, tile->y0, &x0, &y0);
    point_coord(tile->x1, tile->y0, &x1, &y1);
    point_coord(tile->x1, tile->y1, &x2, &y2);
    point_coord(tile->x0, tile->y1, &x3, &y3);

    /* Sort the coordinates. */
    min_tile_x = MIN(MIN(x0, x1), MIN(x2, x3));
    max_tile_x = MAX(MAX(x0, x1), MAX(x2, x3));
    min_tile_y = MIN(MIN(y0, y1), MIN(y2, y3));
    max_tile_y = MAX(MAX(y0, y1), MAX(y2, y3));

    if (min_tile_x >= MAX_X || max_tile_x <= MIN_X ||
        min_tile_y >= MAX_Y || max_tile_y <= MIN_Y)
        /* The tile is obviously out of the window. */
        return FALSE;

    if (angle_is_right() == TRUE)
        /* The rotation is a multiple of Pi/2, the above check is enough. */
        return TRUE;

    /* If one of the point is in the window, we know we can return TRUE. */

    CHECK_POINT_IN_WINDOW(x0, y0);
    CHECK_POINT_IN_WINDOW(x1, y1);
    CHECK_POINT_IN_WINDOW(x2, y2);
    CHECK_POINT_IN_WINDOW(x3, y3);

    /*
     * We check the window corners against the tile edges (it's a rectangle).
     * The tile can be in one of these configurations, but we'll always the
     * first one by switching (x1, y1) with (x3, y3) if needed.
     * Therefore we want x0 <= x3 <= x1 <= x2 or x0 <= x1 <= x2 <= x2
     * and (xi, yi) to keep linked.
     *
     *                   (x1, y1)                             (x3, y3)
     *                     jp                                 jp
     *                 j    p                            j    p
     *             j         p                       j         p
     * (x0, y0) j              p         (x0, y0) j              p
     *                                                              
     *                       j  (x2, y2)                       j  (x2, y2)
     *                   j                                 j
     *               j                                 j
     *            (x3, y3)                             (x1, y1)
     */

    /* First, make x0 the minimum. */

    SORT_PT(x0, y0, x1, y1);
    SORT_PT(x0, y0, x2, y2);
    SORT_PT(x0, y0, x3, y3);

    /* Then x3. */

    SORT_PT(x3, y3, x1, y1);
    SORT_PT(x3, y3, x2, y2);

    /* Finally x1 and x2. */

    SORT_PT(x1, y1, x2, y2);

    if (y3 < y1) {
        /* Second configuration -> first one. */
        DOUBLE_XCHG(x1, x3);
        DOUBLE_XCHG(y1, y3);
    }

    /* Then the checks. */
    return
        (point_and_line(MIN_X, MIN_Y, x2, y2, x3, y3) > 0) &&
        (point_and_line(MAX_X, MIN_Y, x0, y0, x3, y3) > 0) &&
        (point_and_line(MAX_X, MAX_Y, x0, y0, x1, y1) < 0) &&
        (point_and_line(MIN_X, MAX_Y, x1, y1, x2, y2) < 0);
}

G_GNUC_PURE gboolean is_matrix_symmetry(void)
{
    /* c1 == c2 => rotation, c1 == -c2 => symmetry. */
    return (DOUBLE_EQUAL(matrix[0], matrix[5]) == FALSE);
}

G_GNUC_PURE gboolean get_matrix_has_changed(void)
{
    return matrix_changed;
}

G_GNUC_PURE gboolean is_filtering_needed(void)
{
    return (DOUBLE_EQUAL(get_matrix_zoom(), 1.0) == FALSE) ||
        (angle_is_right() == FALSE);
}

#define MIN_PTR(a, b) ((*(a) < *(b)) ? (a) : (b))
#define MAX_PTR(a, b) ((*(a) > *(b)) ? (a) : (b))

/* *a becomes the minimum and *b the maximum. */
G_GNUC_PURE static void min_max(gdouble * a, gdouble * b,
                                gdouble * c, gdouble * d)
{
    gdouble tmp, *ptr;

    tmp = *a;
    ptr = MIN_PTR(MIN_PTR(a, b), MIN_PTR(c, d));
    if (ptr != a) {
        *a = *ptr;
        *ptr = tmp;
    }

    tmp = *b;
    ptr = MAX_PTR(MAX_PTR(b, c), d);
    if (ptr != b) {
        *b = *ptr;
        *ptr = tmp;
    }
}

/*** Operations on the matrix. ***/

/* Returns FALSE if the image is already maximised. */
gboolean matrix_set_max_zoom(gint width, gint height, gboolean do_it)
{
    /*
     * The bounding box of the current image :
     *
     * (x0, y0)===(x1, y1)
     *    ||         ||
     *    ||         ||
     * (x3, y3)===(x2, y2)
     *
     * The image may be rotated, so the bounding box too.
     */

    gdouble x0, y0, x1, y1, x2, y2, x3, y3;
    gdouble half_w, half_h, zoom;

    if ((do_it == FALSE) && (DOUBLE_EQUAL(matrix[3], 0.0) == FALSE ||
                             DOUBLE_EQUAL(matrix[7], 0.0) == FALSE))
        /* Image not centered. */
        return TRUE;

    if (width == -1)
        width = rt->wid_size->width;

    if (height == -1)
        height = rt->wid_size->height;

    if ((options->maximize == FALSE &&
         (current_image->width < width && current_image->height < height)) ||
        (options->scaledown == FALSE &&
         (current_image->width > width || current_image->height > height)))
        return TRUE;


    half_w = current_image->width / 2.0;
    half_h = current_image->height / 2.0;

    point_coord(-half_w, -half_h, &x0, &y0);
    point_coord(half_w, -half_h, &x1, &y1);
    point_coord(half_w, half_h, &x2, &y2);
    point_coord(-half_w, half_h, &x3, &y3);

    min_max(&x0, &x1, &x2, &x3);
    min_max(&y0, &y1, &y2, &y3);

    /* Now x0, y0 are the minima and x1, y1 the maxima. */
    zoom = MIN(width / (x1 - x0), height / (y1 - y0));

    if (DOUBLE_EQUAL(zoom, 1.0) == FALSE ||
        DOUBLE_EQUAL(matrix[3], 0.0) == FALSE ||
        DOUBLE_EQUAL(matrix[7], 0.0) == FALSE) {

        if (do_it == TRUE) {
            matrix_zoom(zoom, 0.0, 0.0);
            matrix[3] = matrix[7] = 0.0;
        }
        return TRUE;
    }

    return FALSE;
}

void matrix_reset(void)
{
    guint i;

    for (i = 0; i < 8; i++)
        matrix[i] = (i % 5 == 0) ? 1.0 : 0.0;

    matrix_changed = TRUE;
}

/*
 * Rotation:         Product:
 * cos   sin  0  0 | c1*cos+s2*sin   s1*cos+c2*sin   0  x*cos+y*sin
 * -sin  cos  0  0 | -c1*sin+s2*cos  -s1*sin+c2*cos  0  -x*sin+y*cos
 * 0     0    1  0 | 0               0               1  0
 * 0     0    0  1 | 0               0               0  1
 */
void matrix_rotate(gdouble angle)
{
    gdouble cosinus, sinus;
    gdouble c1, s1, c2, s2, x, y;
    gboolean zoom;

    /* Do we maximize after rotating ? */
    zoom = (options->maximize || options->scaledown) &&
        (matrix_set_max_zoom(-1, -1, FALSE) == FALSE);

    cosinus = cos(angle);
    sinus = sin(angle);

    c1 = matrix[0];
    c2 = matrix[5];
    s1 = matrix[1];
    s2 = matrix[4];
    x = matrix[3];
    y = matrix[7];

    matrix[0] = c1 * cosinus + s2 * sinus;
    matrix[1] = s1 * cosinus + c2 * sinus;
    matrix[4] = -c1 * sinus + s2 * cosinus;
    matrix[5] = -s1 * sinus + c2 * cosinus;

    matrix[3] = x * cosinus + y * sinus;
    matrix[7] = -x * sinus + y * cosinus;

    if (zoom == TRUE)
        matrix_set_max_zoom(-1, -1, TRUE);
    else
        matrix_changed = TRUE;
}

void matrix_move(gdouble x, gdouble y)
{
    matrix[3] += x;
    matrix[7] += y;

    matrix_changed = TRUE;
}

/* (x, y) : zoom center. */
void matrix_zoom(gdouble ratio, gdouble x, gdouble y)
{
    gdouble offset_x, offset_y;

    offset_x = rt->wid_size->width / 2.0 - x;
    offset_y = rt->wid_size->height / 2.0 - y;

    matrix_move(offset_x, offset_y);

    matrix[0] *= ratio;
    matrix[1] *= ratio;
    matrix[3] *= ratio;
    matrix[4] *= ratio;
    matrix[5] *= ratio;
    matrix[7] *= ratio;

    matrix_move(-offset_x, -offset_y);
}

static void flip(guchar id)
{
    /* Flip either the x or y row. */
    matrix[id] *= -1.0;
    matrix[id + 1] *= -1.0;
    matrix[id + 3] *= -1.0;

    matrix_changed = TRUE;
}

void matrix_flip_h(void)
{
    /* Flip x. */
    flip(4);
}

void matrix_flip_v(void)
{
    /* Flip y. */
    flip(0);
}
