/*
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * See the COPYING file for license information.
 *
 * Guillaume Chazarain <booh@altern.org>
 */

/*******************************
 * File extensions management. *
 *******************************/

#include "gliv.h"

#ifdef HAVE_BSEARCH
#include <stdlib.h>             /* bsearch() */
#else
#include "../lib/bsearch.h"
#endif

#include <string.h>             /* strrchr() */

extern options_struct *options;

typedef struct {
    const gchar *ext;
    const loader loader;
} format;

/* *INDENT-OFF* */
static format supported_extensions[] = {
    {"avs",   LOADER_MAGICK_ONE},
    {"bie",   LOADER_MAGICK_MANY},
    {"bmp",   LOADER_PIXBUF},
    {"cgm",   LOADER_MAGICK_ONE},
    {"cmyk",  LOADER_MAGICK_ONE},
    {"dcx",   LOADER_MAGICK_MANY},
    {"dib",   LOADER_MAGICK_ONE},
    {"dicom", LOADER_MAGICK_ONE},
    {"epdf",  LOADER_MAGICK_ONE},
    {"epi",   LOADER_MAGICK_ONE},
    {"eps",   LOADER_MAGICK_ONE},
    {"eps2",  LOADER_MAGICK_ONE},
    {"epsf",  LOADER_MAGICK_ONE},
    {"epsi",  LOADER_MAGICK_ONE},
    {"ept",   LOADER_MAGICK_ONE},
    {"fax",   LOADER_MAGICK_MANY},
    {"fig",   LOADER_MAGICK_ONE},
    {"fits",  LOADER_MAGICK_ONE},
    {"fpx",   LOADER_MAGICK_ONE},
    {"gif",   LOADER_PIXBUF},
    {"hdf",   LOADER_MAGICK_MANY},
    {"hpgl",  LOADER_MAGICK_ONE},
    {"ico",   LOADER_PIXBUF},
    {"jbig",  LOADER_MAGICK_MANY},
    {"jif",   LOADER_PIXBUF},
    {"jpeg",  LOADER_PIXBUF},
    {"jpg",   LOADER_PIXBUF},
    {"miff",  LOADER_MAGICK_MANY},
    {"mng",   LOADER_MAGICK_ONE},
    {"mtv",   LOADER_MAGICK_MANY},
    {"p7",    LOADER_MAGICK_ONE},
    {"pbm",   LOADER_MAGICK_MANY},
    {"pcd",   LOADER_MAGICK_ONE},
    {"pcl",   LOADER_MAGICK_ONE},
    {"pcx",   LOADER_MAGICK_ONE},
    {"pdf",   LOADER_MAGICK_MANY},
    {"pgm",   LOADER_MAGICK_MANY},
    {"pix",   LOADER_MAGICK_ONE},
    {"png",   LOADER_PIXBUF},
    {"pnm",   LOADER_PIXBUF},
    {"ppm",   LOADER_PIXBUF},
    {"ps",    LOADER_MAGICK_MANY},
    {"ps2",   LOADER_MAGICK_MANY},
    {"psd",   LOADER_MAGICK_ONE},
    {"rad",   LOADER_MAGICK_ONE},
    {"ras",   LOADER_PIXBUF},
    {"rgb",   LOADER_MAGICK_ONE},
    {"rla",   LOADER_MAGICK_ONE},
    {"rle",   LOADER_MAGICK_ONE},
    {"sgi",   LOADER_MAGICK_MANY},
    {"sun",   LOADER_MAGICK_MANY},
    {"tga",   LOADER_PIXBUF_OR_MAGICK_MANY},
    {"tif",   LOADER_PIXBUF},
    {"tiff",  LOADER_PIXBUF},
    {"tim",   LOADER_MAGICK_ONE},
    {"viff",  LOADER_MAGICK_MANY},
    {"wbmp",  LOADER_PIXBUF_OR_NOTHING},
    {"xbm",   LOADER_PIXBUF},
    {"xpm",   LOADER_PIXBUF},
    {"xwd",   LOADER_MAGICK_ONE},
    {"yuv",   LOADER_MAGICK_MANY}
};
/* *INDENT-ON* */

#define NB_FORMATS (sizeof(supported_extensions)/sizeof(format))

/* Used by bsearch. */
G_GNUC_PURE static gint compar(gconstpointer key, gconstpointer ptr)
{
    return strcmp((gchar *) key, ((format *) ptr)->ext);
}

static loader loader_by_filename(gchar * filename)
{
    format *res;
    gchar *tmp, *ext;

    ext = strrchr(filename, '.');
    if (ext == NULL) {
        return LOADER_NONE;
    }

    ext = g_strdup(ext + 1);

    /* Downcase the extension so that we can do case sensitive strcmp. */
    for (tmp = ext; *tmp != '\0'; tmp++)
        if (*tmp >= 'A' && *tmp <= 'Z')
            *tmp += 'a' - 'A';

    res =
        bsearch(ext, supported_extensions, NB_FORMATS, sizeof(format), compar);

    g_free(ext);

    if (res == NULL) {
        g_printerr(_("%s : unknown extension\n"), filename);
        return LOADER_NONE;
    }
#ifndef MAGICK_CONVERT
    if (res->loader == LOADER_MAGICK_ONE || res->loader == LOADER_MAGICK_MANY) {
        g_printerr(_("%s : ImageMagick support is disabled\n"), filename);

        return LOADER_NONE;
    }
#endif

    return res->loader;
}

GdkPixbuf *file_to_pixbuf(gchar * filename)
{
    loader load;
    GdkPixbuf *im;

    load = loader_by_filename(filename);
    switch (load) {
    case LOADER_PIXBUF:
        im = GDK_PIXBUF_LOAD(filename);
        break;

#ifdef MAGICK_CONVERT
    case LOADER_MAGICK_ONE:
        im = load_with_imagemagick(filename, FALSE);
        break;

    case LOADER_MAGICK_MANY:
        im = load_with_imagemagick(filename, TRUE);
        break;
#endif

    default:
        /* case LOADER_NONE: */
        im = NULL;
    }

    if ((im == NULL) && (options->force == TRUE)) {
        im = GDK_PIXBUF_LOAD(filename);
#ifdef MAGICK_CONVERT
        if (im == NULL)
            im = load_with_imagemagick(filename, TRUE);
#endif
    }

    if (im == NULL)
        g_printerr(_("Cannot load %s\n"), filename);

    return im;
}

/* This is checked before adding a file to the list. */
gboolean extension_is_ok(gchar * file)
{
    return options->force == TRUE || loader_by_filename(file) != LOADER_NONE;
}
