/*
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * See the COPYING file for license information.
 *
 * Guillaume Chazarain <booh@altern.org>
 */

/*************************
 * The toplevel widgets. *
 *************************/

#include "gliv.h"

extern rt_struct *rt;
extern options_struct *options;
extern gliv_image *current_image;
extern GtkWidget *gl_widget;
extern GtkMenuBar *menu_bar;

static GtkVBox *widgets;

static GtkWindow *main_window;
static GtkWindow *fs_window;

/* The main status bar and its subdivisions. */
static GtkHBox *status_bar;
static GtkEntry *entry_ident;
static GtkEntry *entry_size;
static GtkEntry *entry_state;

#ifdef GTK2
void show_dialog(GtkWidget * widget, gchar * name, gboolean unused)
{
    if (options->fullscreen == TRUE)
        gtk_window_set_position(GTK_WINDOW(widget), GTK_WIN_POS_MOUSE);

    if (name != NULL)
        gtk_window_set_title(GTK_WINDOW(widget), locale_to_utf8(name));

    gtk_widget_show_all(widget);
}
#else
/* Give back the grab to the fullscreen window when a dialog is closed. */
static void restore_grab(void)
{
    if (options->fullscreen == TRUE)
        gdk_keyboard_grab(GTK_WIDGET(fs_window)->window, TRUE,
                          GDK_CURRENT_TIME);
    else
        gdk_keyboard_ungrab(GDK_CURRENT_TIME);
}

void show_dialog(GtkWidget * widget, gchar * name, gboolean grab)
{
    if (options->fullscreen == TRUE) {
        gtk_widget_realize(widget);
        gdk_window_set_override_redirect(widget->window, TRUE);
        gtk_window_set_position(GTK_WINDOW(widget), GTK_WIN_POS_MOUSE);
    } else
        gtk_window_set_title(GTK_WINDOW(widget), name);

    gtk_widget_show_all(widget);

    if (options->fullscreen == TRUE && grab == TRUE) {
        gtk_signal_connect(GTK_OBJECT(widget), "destroy",
                           G_CALLBACK(restore_grab), NULL);

        gdk_keyboard_grab(widget->window, FALSE, GDK_CURRENT_TIME);
    }

}
#endif

static GtkEntry *add_entry(gboolean grow)
{
    GtkEntry *entry;

    entry = GTK_ENTRY(gtk_entry_new());
    gtk_editable_set_editable(GTK_EDITABLE(entry), FALSE);
    gtk_box_pack_start(GTK_BOX(status_bar), GTK_WIDGET(entry), grow, TRUE, 0);

    /* We don't want the blinking cursor. */
    GTK_WIDGET_UNSET_FLAGS(GTK_WIDGET(entry), GTK_CAN_FOCUS);

    return entry;
}

static void create_status_bar(void)
{
    status_bar = GTK_HBOX(gtk_hbox_new(FALSE, 0));

    entry_ident = add_entry(TRUE);
    entry_size = add_entry(FALSE);
    entry_state = add_entry(FALSE);
}

void set_loading_entry(gchar * filename)
{
    static guchar count = 0;
    static gchar *old_ident;
    gchar *new_ident;

    /* 
     * When loading the first image, the window is
     * not mapped, so we don't change the entry.
     */
    if (count < 2) {
        count++;
        return;
    }

    if (filename == NULL) {
        gtk_entry_set_text(entry_ident, old_ident);
        g_free(old_ident);
    } else {
        old_ident = gtk_editable_get_chars(GTK_EDITABLE(entry_ident), 0, -1);

        new_ident = g_strdup_printf("%s %s: %s",
                                    old_ident, _("loading"), filename);

        gtk_entry_set_text(entry_ident, locale_to_utf8(new_ident));
        g_free(new_ident);
    }

    if (options->status_bar == TRUE) {
        /* Make sure the widget is updated now. */
        gtk_widget_queue_draw(GTK_WIDGET(entry_ident));
        gdk_window_process_all_updates();
    }
}

static void toggle_widget(GtkWidget * widget, gboolean * flag)
{
    guint16 new_width, new_height;

    new_width = GTK_WIDGET(main_window)->allocation.width;
    new_height = GTK_WIDGET(main_window)->allocation.height;

    if (*flag == FALSE) {
        gtk_widget_show_all(widget);
        new_height += widget->requisition.height;
    } else {
        new_height -= widget->requisition.height;
        gtk_widget_hide(widget);
    }

    *flag ^= TRUE;

    if (options->fullscreen == FALSE)
        gtk_window_resize(main_window, new_width, new_height);
}

void toggle_status_bar(void)
{
    toggle_widget(GTK_WIDGET(status_bar), &options->status_bar);
}

void toggle_menu_bar(void)
{
    toggle_widget(GTK_WIDGET(menu_bar), &options->menu_bar);
}

void update_status_bar(void)
{
    gchar *size_str, *state_str;
    const gchar *sym;
    gdouble zoom, angle;

    if (current_image->ident != NULL) {
        /* Filename and dimensions status, only the first time. */
        gtk_entry_set_text(entry_ident, locale_to_utf8(current_image->ident));
        gtk_editable_set_position(GTK_EDITABLE(entry_ident), 0);

        size_str = g_strdup_printf("%dx%d",
                                   current_image->width, current_image->height);
        gtk_entry_set_text(entry_size, size_str);
        g_free(size_str);
    }

    /* The state */

    zoom = get_matrix_zoom() * 100.0;
    angle = get_matrix_angle() * 180.0 / PI;
    sym = (is_matrix_symmetry() == TRUE) ? " /" : "";

    /* We don't want -0.000 */
    if (DOUBLE_EQUAL(angle, 0.0) == TRUE)
        angle = 0.0;

    state_str = g_strdup_printf(_("%.3lf%% (%.3lf deg%s)"), zoom, angle, sym);
    gtk_entry_set_text(entry_state, locale_to_utf8(state_str));

    g_free(state_str);
}

#ifdef GTK2
static gboolean first_map(void)
#else
static void main_realize(void)
#endif
{
    if (options->menu_bar == FALSE)
        gtk_widget_hide(GTK_WIDGET(menu_bar));

    if (options->status_bar == FALSE)
        gtk_widget_hide(GTK_WIDGET(status_bar));

#ifdef GTK2
    gtk_window_move(fs_window, 0, 0);

    return TRUE;
#endif
}

void create_windows(void)
{
    GtkWindow *first_window;
    GtkAccelGroup *accel_group;

    /* The main window : seen when not in fullscreen. */
    main_window = GTK_WINDOW(gtk_window_new(GTK_WINDOW_TOPLEVEL));
    gtk_window_set_title(main_window, "GLiv");
#ifndef GTK2
    gtk_signal_connect(GTK_OBJECT(main_window), "realize",
                       GTK_SIGNAL_FUNC(main_realize), NULL);
#endif
    install_callbacks(main_window);

    /* The window seen in fullscreen mode. */
#ifdef GTK2
    fs_window = GTK_WINDOW(gtk_window_new(GTK_WINDOW_TOPLEVEL));
#else
    fs_window = GTK_WINDOW(gtk_window_new(GTK_WINDOW_POPUP));
    gtk_window_set_position(fs_window, GTK_WIN_POS_CENTER);
#endif
    gtk_window_set_title(fs_window, locale_to_utf8(_("GLiv in fullscreen")));

    gtk_window_set_default_size(fs_window, rt->scr_width, rt->scr_height);
    gtk_window_set_decorated(fs_window, FALSE);
    install_callbacks(GTK_OBJECT(fs_window));

    /* The widgets. */
    create_gl_widget();
    create_status_bar();
    accel_group = create_menus();

    /* Bind keyboard accelerators to windows. */
    gtk_window_add_accel_group(main_window, accel_group);
    gtk_window_add_accel_group(fs_window, accel_group);

    /* Collection. */
    widgets = GTK_VBOX(gtk_vbox_new(FALSE, 0));
    gtk_box_pack_start(GTK_BOX(widgets), GTK_WIDGET(menu_bar), FALSE, FALSE, 0);
    gtk_box_pack_start_defaults(GTK_BOX(widgets), gl_widget);
    gtk_box_pack_end(GTK_BOX(widgets), GTK_WIDGET(status_bar), FALSE, 0, 0);

    if (options->fullscreen == FALSE)
        first_window = main_window;
    else {
        first_window = fs_window;
#ifndef GTK2
        gtk_widget_realize(GTK_WIDGET(main_window));
        gtk_widget_hide(GTK_WIDGET(main_window));
#endif
    }

    gtk_container_add(GTK_CONTAINER(first_window), GTK_WIDGET(widgets));
#ifdef GTK2
    g_signal_connect(first_window, "map", G_CALLBACK(first_map), NULL);
#endif
    gtk_widget_show_all(GTK_WIDGET(first_window));
}

void goto_window(void)
{
    guint16 new_width, new_height;

    new_width = current_image->width;
    new_height = current_image->height;

    if (options->menu_bar == TRUE) {
        new_height += GTK_WIDGET(menu_bar)->requisition.height;
        if (new_width < GTK_WIDGET(menu_bar)->requisition.width)
            new_width = GTK_WIDGET(menu_bar)->requisition.width;
    }

    if (options->status_bar == TRUE) {
        new_height += GTK_WIDGET(status_bar)->requisition.height;
        if (new_width < GTK_WIDGET(status_bar)->requisition.width)
            new_width = GTK_WIDGET(status_bar)->requisition.width;
    }

    if (new_width >= rt->scr_width || new_height >= rt->scr_height) {
        new_width = 3 * rt->scr_width / 4;
        new_height = 3 * rt->scr_height / 4;
    }
#ifndef GTK2
    gtk_widget_set(gl_widget, "GtkWidget::width", current_image->width, NULL);
    gtk_widget_set(gl_widget, "GtkWidget::height", current_image->height, NULL);
    gtk_widget_set_usize(gl_widget, 0, 0);
#endif

    gtk_window_resize(main_window, new_width, new_height);
}

#ifndef GTK2
static void reparent(GtkWidget * new_parent)
{
    /* Be sure all widgets are displayed. */

    if (options->status_bar == TRUE && GTK_WIDGET_REALIZED(status_bar) == FALSE)
        gtk_container_foreach(GTK_CONTAINER(status_bar),
                              G_CALLBACK(gtk_widget_realize), NULL);

    gtk_widget_show_all(GTK_WIDGET(status_bar));

    gtk_widget_show_all(GTK_WIDGET(menu_bar));

    /* Then reparent them. */
    gtk_widget_reparent(GTK_WIDGET(widgets), new_parent);

    /* Finally, hide widget previously hidden. */

    if (options->status_bar == FALSE)
        gtk_widget_hide(GTK_WIDGET(status_bar));

    if (options->menu_bar == FALSE)
        gtk_widget_hide(GTK_WIDGET(menu_bar));
}
#endif

void toggle_fullscreen(gboolean enable)
{
    static gint previous_width, previous_height;
    static gliv_image *previous_image = NULL;
    GtkWindow *new, *old;

    if (options->fullscreen == FALSE) {
        previous_width = GTK_WIDGET(main_window)->allocation.width;
        previous_height = GTK_WIDGET(main_window)->allocation.height;
        previous_image = current_image;
    }

    if (enable == TRUE) {
        /* Go to fullscreen mode. */
        new = fs_window;
        old = main_window;
    } else {
        /* Go to window mode. */
        new = main_window;
        old = fs_window;
    }

    gtk_widget_show_all(GTK_WIDGET(new));
    gtk_widget_hide(GTK_WIDGET(old));

#ifdef GTK2
    gtk_widget_reparent(GTK_WIDGET(widgets), GTK_WIDGET(new));
#else
    reparent(GTK_WIDGET(new));
    intercept_events(enable);
#endif

    if (enable == FALSE) {
        if (previous_image == current_image)
            gtk_window_resize(main_window, previous_width, previous_height);
        else
            goto_window();
    }

    update_bg_color(FALSE);

    options->fullscreen = enable;
    schedule_hide_cursor();
}
